"use strict";
var path = require('path');
var ClassData_1 = require('../src/ClassData');
var jvm_1 = require('../src/jvm');
var classpath_1 = require('../src/classpath');
var util_1 = require('../src/util');
var jvmObject, classpath;
/**
 * Implementation of Levenshtein distance.
 * @url http://en.wikibooks.org/wiki/Algorithm_Implementation/Strings/Levenshtein_distance#JavaScript
 */
function getEditDistance(a, b) {
    var matrix = [], i, j;
    if (a.length == 0)
        return b.length;
    if (b.length == 0)
        return a.length;
    // increment along the first column of each row
    for (i = 0; i <= b.length; i++) {
        matrix[i] = [i];
    }
    // increment each column in the first row
    for (j = 0; j <= a.length; j++) {
        matrix[0][j] = j;
    }
    // Fill in the rest of the matrix
    for (i = 1; i <= b.length; i++) {
        for (j = 1; j <= a.length; j++) {
            if (b.charAt(i - 1) == a.charAt(j - 1)) {
                matrix[i][j] = matrix[i - 1][j - 1];
            }
            else {
                matrix[i][j] = Math.min(matrix[i - 1][j - 1] + 1, // substitution
                Math.min(matrix[i][j - 1] + 1, // insertion
                matrix[i - 1][j] + 1)); // deletion
            }
        }
    }
    return matrix[b.length][a.length];
}
/**
 * Loads the given classname, parses the class file, and returns the method
 * signatures of all of the native methods in the class.
 * className should be in pkg/Name format, not pkg.Name.
 */
function getNativeSigs(className) {
    var rv = [], i;
    for (i = 0; i < classpath.length; i++) {
        var klassData = classpath[i].tryLoadClassSync(className);
        if (klassData !== null) {
            var klass = new ClassData_1.ReferenceClassData(klassData);
            var methods = klass.getMethods();
            methods.forEach(function (m) {
                if (m.accessFlags.isNative()) {
                    rv.push(m.signature);
                }
            });
        }
    }
    return rv;
}
/**
 * What it says on the tin. Returns an array of class names (in foo/bar/Baz
 * format) that have native methods implemented in Doppio.
 */
function getClassesWithImplementedNatives() {
    var nativeImpls = jvmObject.getNatives(), methods = {};
    Object.keys(nativeImpls).forEach(function (clsName) {
        methods[clsName] = Object.keys(nativeImpls[clsName]);
    });
    return methods;
}
/**
 * Given two arrays, returns any items in the first array that are not in the
 * second array.
 */
function missingFromArray(items, arr) {
    var rv = [], i;
    for (i = 0; i < items.length; i++) {
        if (arr.indexOf(items[i]) === -1) {
            rv.push(items[i]);
        }
    }
    return rv;
}
/**
 * Given a set of strings and a set of candidates, returns a map that maps
 * each string to the candidate with the shortest edit distance.
 */
function getSimilar(strs, candidates) {
    var minEditDistance, bestCandidate, i, j, str, tmp, rv = {};
    for (i = 0; i < strs.length; i++) {
        minEditDistance = 999999999999999;
        str = strs[i];
        for (j = 0; j < candidates.length; j++) {
            tmp = getEditDistance(str, candidates[j]);
            if (tmp < minEditDistance) {
                minEditDistance = tmp;
                bestCandidate = candidates[j];
            }
        }
        if (bestCandidate != null) {
            rv[str] = bestCandidate;
        }
        else {
            rv[str] = "";
        }
    }
    return rv;
}
function formatKlass(title, data) {
    var left, output = [];
    // Determine the width of each column by finding the longest data item.
    output.push(title);
    for (left in data) {
        output.push("\n\tMissing: " + left);
        output.push("\n\tMatch? : " + data[left] + "\n");
    }
    return output.join('');
}
/**
 * Prints the result.
 */
function printResult(result) {
    var klassName, printed = false;
    for (klassName in result) {
        printed = true;
        if (result.hasOwnProperty(klassName)) {
            console.log(formatKlass(klassName, result[klassName]) + "\n");
        }
    }
    if (!printed) {
        console.log("All native method implementations match native methods in your class files.");
    }
}
/**
 * Main function.
 */
function main() {
    var implNatives = getClassesWithImplementedNatives(), klassNatives, klassImplNatives, klassName, missingNatives, similarMap = {};
    for (klassName in implNatives) {
        if (implNatives.hasOwnProperty(klassName)) {
            klassNatives = getNativeSigs(klassName);
            klassImplNatives = implNatives[klassName];
            // See if any implemented natives are not in natives.
            missingNatives = missingFromArray(klassImplNatives, klassNatives);
            if (missingNatives.length > 0) {
                // Find candidates for the missing items.
                similarMap[klassName] = getSimilar(missingNatives, klassNatives);
            }
        }
    }
    printResult(similarMap);
}
var JAVA_HOME = path.resolve(__dirname, '../vendor/java_home');
var DOPPIO_HOME = path.resolve(__dirname, '..');
var opts = util_1.merge(jvm_1["default"].getDefaultOptions(DOPPIO_HOME), {
    nativeClasspath: [path.resolve(__dirname, '../src/natives')]
});
new jvm_1["default"](opts, function (err, _jvmObject) {
    if (err) {
        throw err;
    }
    jvmObject = _jvmObject;
    classpath_1.ClasspathFactory(JAVA_HOME, opts.bootstrapClasspath, function (items) {
        classpath = items;
        main();
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmluZF9pbnZhbGlkX25hdGl2ZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9jb25zb2xlL2ZpbmRfaW52YWxpZF9uYXRpdmVzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFTQSxJQUFZLElBQUksV0FBTSxNQUFNLENBQUMsQ0FBQTtBQUU3QiwwQkFBaUMsa0JBQWtCLENBQUMsQ0FBQTtBQUNwRCxvQkFBZ0IsWUFBWSxDQUFDLENBQUE7QUFHN0IsMEJBQStDLGtCQUFrQixDQUFDLENBQUE7QUFHbEUscUJBQW9CLGFBQWEsQ0FBQyxDQUFBO0FBRWxDLElBQUksU0FBYyxFQUNkLFNBQTJCLENBQUM7QUFFaEM7OztHQUdHO0FBQ0gseUJBQXlCLENBQVMsRUFBRSxDQUFTO0lBQzNDLElBQUksTUFBTSxHQUFlLEVBQUUsRUFBRSxDQUFTLEVBQUUsQ0FBUyxDQUFDO0lBQ2xELEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxDQUFDO1FBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUM7SUFDbkMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUM7UUFBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQztJQUVuQywrQ0FBK0M7SUFDL0MsR0FBRyxDQUFBLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQzlCLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2xCLENBQUM7SUFFRCx5Q0FBeUM7SUFDekMsR0FBRyxDQUFBLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQzlCLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDbkIsQ0FBQztJQUVELGlDQUFpQztJQUNqQyxHQUFHLENBQUEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDOUIsR0FBRyxDQUFBLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQzlCLEVBQUUsQ0FBQSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDbEMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxDQUFDLEdBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2xDLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsZUFBZTtnQkFDckMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxZQUFZO2dCQUNoQyxNQUFNLENBQUMsQ0FBQyxHQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXO1lBQ3BFLENBQUM7UUFDSCxDQUFDO0lBQ0gsQ0FBQztJQUVELE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNwQyxDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILHVCQUF1QixTQUFpQjtJQUN0QyxJQUFJLEVBQUUsR0FBYSxFQUFFLEVBQUUsQ0FBUyxDQUFDO0lBQ2pDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUN0QyxJQUFJLFNBQVMsR0FBRyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDekQsRUFBRSxDQUFDLENBQUMsU0FBUyxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDdkIsSUFBSSxLQUFLLEdBQUcsSUFBSSw4QkFBa0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUM5QyxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDakMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxVQUFDLENBQVM7Z0JBQ3hCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDO29CQUM3QixFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQztnQkFDdkIsQ0FBQztZQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztJQUNILENBQUM7SUFDRCxNQUFNLENBQUMsRUFBRSxDQUFDO0FBQ1osQ0FBQztBQUVEOzs7R0FHRztBQUNIO0lBQ0UsSUFBSSxXQUFXLEdBQTJELFNBQVMsQ0FBQyxVQUFVLEVBQUUsRUFDNUYsT0FBTyxHQUFrQyxFQUFFLENBQUM7SUFFaEQsTUFBTSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUMsVUFBQyxPQUFlO1FBQy9DLE9BQU8sQ0FBQyxPQUFPLENBQUMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELENBQUMsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLE9BQU8sQ0FBQztBQUNqQixDQUFDO0FBRUQ7OztHQUdHO0FBQ0gsMEJBQTBCLEtBQWUsRUFBRSxHQUFhO0lBQ3RELElBQUksRUFBRSxHQUFhLEVBQUUsRUFBRSxDQUFTLENBQUM7SUFDakMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQ2xDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEIsQ0FBQztJQUNILENBQUM7SUFDRCxNQUFNLENBQUMsRUFBRSxDQUFDO0FBQ1osQ0FBQztBQUVEOzs7R0FHRztBQUNILG9CQUFvQixJQUFjLEVBQUUsVUFBb0I7SUFDdEQsSUFBSSxlQUF1QixFQUFFLGFBQXFCLEVBQUUsQ0FBUyxFQUFFLENBQVMsRUFDcEUsR0FBVyxFQUFFLEdBQVcsRUFBRSxFQUFFLEdBQTRCLEVBQUUsQ0FBQztJQUMvRCxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDakMsZUFBZSxHQUFHLGVBQWUsQ0FBQztRQUNsQyxHQUFHLEdBQUcsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2QsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ3ZDLEdBQUcsR0FBRyxlQUFlLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsR0FBRyxlQUFlLENBQUMsQ0FBQyxDQUFDO2dCQUMxQixlQUFlLEdBQUcsR0FBRyxDQUFDO2dCQUN0QixhQUFhLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLENBQUM7UUFDSCxDQUFDO1FBRUQsRUFBRSxDQUFDLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDMUIsRUFBRSxDQUFDLEdBQUcsQ0FBQyxHQUFHLGFBQWEsQ0FBQztRQUMxQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixFQUFFLENBQUMsR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2YsQ0FBQztJQUNILENBQUM7SUFDRCxNQUFNLENBQUMsRUFBRSxDQUFDO0FBQ1osQ0FBQztBQUVELHFCQUFxQixLQUFhLEVBQUUsSUFBOEI7SUFDaEUsSUFBSSxJQUFZLEVBQUUsTUFBTSxHQUFhLEVBQUUsQ0FBQztJQUN4Qyx1RUFBdUU7SUFDdkUsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNuQixHQUFHLENBQUMsQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNsQixNQUFNLENBQUMsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsQ0FBQztRQUNwQyxNQUFNLENBQUMsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQ3pCLENBQUM7QUFFRDs7R0FFRztBQUNILHFCQUFxQixNQUFzRDtJQUN6RSxJQUFJLFNBQWlCLEVBQUUsT0FBTyxHQUFZLEtBQUssQ0FBQztJQUNoRCxHQUFHLENBQUMsQ0FBQyxTQUFTLElBQUksTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN6QixPQUFPLEdBQUcsSUFBSSxDQUFDO1FBQ2YsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDckMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO1FBQ2hFLENBQUM7SUFDSCxDQUFDO0lBRUQsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ2IsT0FBTyxDQUFDLEdBQUcsQ0FBQyw2RUFBNkUsQ0FBQyxDQUFDO0lBQzdGLENBQUM7QUFDSCxDQUFDO0FBRUQ7O0dBRUc7QUFDSDtJQUNFLElBQUksV0FBVyxHQUFvQyxnQ0FBZ0MsRUFBRSxFQUNqRixZQUFzQixFQUFFLGdCQUEwQixFQUFFLFNBQWlCLEVBQ3JFLGNBQXdCLEVBQ3hCLFVBQVUsR0FBbUQsRUFBRSxDQUFDO0lBQ3BFLEdBQUcsQ0FBQyxDQUFDLFNBQVMsSUFBSSxXQUFXLENBQUMsQ0FBQyxDQUFDO1FBQzlCLEVBQUUsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzFDLFlBQVksR0FBRyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDeEMsZ0JBQWdCLEdBQUcsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBRTFDLHFEQUFxRDtZQUNyRCxjQUFjLEdBQUcsZ0JBQWdCLENBQUMsZ0JBQWdCLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDbEUsRUFBRSxDQUFDLENBQUMsY0FBYyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUM5Qix5Q0FBeUM7Z0JBQ3pDLFVBQVUsQ0FBQyxTQUFTLENBQUMsR0FBRyxVQUFVLENBQUMsY0FBYyxFQUFFLFlBQVksQ0FBQyxDQUFDO1lBQ25FLENBQUM7UUFDSCxDQUFDO0lBQ0gsQ0FBQztJQUNELFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQztBQUMxQixDQUFDO0FBRUQsSUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUscUJBQXFCLENBQUMsQ0FBQztBQUNqRSxJQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUNsRCxJQUFJLElBQUksR0FBcUIsWUFBSyxDQUFDLGdCQUFHLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLEVBQUU7SUFDckUsZUFBZSxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztDQUM3RCxDQUFDLENBQUM7QUFFSCxJQUFJLGdCQUFHLENBQUMsSUFBSSxFQUFFLFVBQVMsR0FBUSxFQUFFLFVBQWU7SUFDOUMsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUNSLE1BQU0sR0FBRyxDQUFDO0lBQ1osQ0FBQztJQUNELFNBQVMsR0FBRyxVQUFVLENBQUM7SUFDdkIsNEJBQWdCLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxVQUFDLEtBQUs7UUFDekQsU0FBUyxHQUFHLEtBQUssQ0FBQztRQUNsQixJQUFJLEVBQUUsQ0FBQztJQUNULENBQUMsQ0FBQyxDQUFBO0FBQ0osQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENoZWNrcyB3aGljaCBEb3BwaW8gTmF0aXZlIE1ldGhvZHMgYXJlIGFjdHVhbGx5IHVzZWQgaW4gdGhlIHZlcnNpb24gb2YgdGhlXG4gKiBKYXZhIENsYXNzIExpYnJhcnkgdGhhdCBEb3BwaW8gaXMgdXNpbmcuXG4gKlxuICogUHJvcG9zZXMgc2ltaWxhciBjYW5kaWRhdGVzIGZvciB0aGVzZSBtaXNzaW5nIG5hdGl2ZSBtZXRob2RzLCBzaW5jZVxuICogb2NjYXNpb25hbGx5IG1ldGhvZHMgYXJlIHJlbmFtZWQuXG4gKiBAdG9kbyBFeHBvc2UgQ0xJIGZvciBzZWxlY3RpbmcgdGhlIGNsYXNzcGF0aC4gTm8gYXJncyA9IGRlZmF1bHQuXG4gKi9cbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBvcyBmcm9tICdvcyc7XG5pbXBvcnQge1JlZmVyZW5jZUNsYXNzRGF0YX0gZnJvbSAnLi4vc3JjL0NsYXNzRGF0YSc7XG5pbXBvcnQgSlZNIGZyb20gJy4uL3NyYy9qdm0nO1xuaW1wb3J0IHtNZXRob2R9IGZyb20gJy4uL3NyYy9tZXRob2RzJztcbmltcG9ydCAqIGFzIEpWTVR5cGVzIGZyb20gJy4uL2luY2x1ZGVzL0pWTVR5cGVzJztcbmltcG9ydCB7SUNsYXNzcGF0aEl0ZW0sIENsYXNzcGF0aEZhY3Rvcnl9IGZyb20gJy4uL3NyYy9jbGFzc3BhdGgnO1xuaW1wb3J0ICogYXMgSkRLSW5mbyBmcm9tICcuLi92ZW5kb3IvamF2YV9ob21lL2pkay5qc29uJztcbmltcG9ydCB7SlZNT3B0aW9uc30gZnJvbSAnLi4vc3JjL2ludGVyZmFjZXMnO1xuaW1wb3J0IHttZXJnZX0gZnJvbSAnLi4vc3JjL3V0aWwnO1xuXG52YXIganZtT2JqZWN0OiBKVk0sXG4gICAgY2xhc3NwYXRoOiBJQ2xhc3NwYXRoSXRlbVtdO1xuXG4vKipcbiAqIEltcGxlbWVudGF0aW9uIG9mIExldmVuc2h0ZWluIGRpc3RhbmNlLlxuICogQHVybCBodHRwOi8vZW4ud2lraWJvb2tzLm9yZy93aWtpL0FsZ29yaXRobV9JbXBsZW1lbnRhdGlvbi9TdHJpbmdzL0xldmVuc2h0ZWluX2Rpc3RhbmNlI0phdmFTY3JpcHRcbiAqL1xuZnVuY3Rpb24gZ2V0RWRpdERpc3RhbmNlKGE6IHN0cmluZywgYjogc3RyaW5nKTogbnVtYmVyIHtcbiAgdmFyIG1hdHJpeDogbnVtYmVyW11bXSA9IFtdLCBpOiBudW1iZXIsIGo6IG51bWJlcjtcbiAgaWYgKGEubGVuZ3RoID09IDApIHJldHVybiBiLmxlbmd0aDtcbiAgaWYgKGIubGVuZ3RoID09IDApIHJldHVybiBhLmxlbmd0aDtcblxuICAvLyBpbmNyZW1lbnQgYWxvbmcgdGhlIGZpcnN0IGNvbHVtbiBvZiBlYWNoIHJvd1xuICBmb3IoaSA9IDA7IGkgPD0gYi5sZW5ndGg7IGkrKykge1xuICAgIG1hdHJpeFtpXSA9IFtpXTtcbiAgfVxuXG4gIC8vIGluY3JlbWVudCBlYWNoIGNvbHVtbiBpbiB0aGUgZmlyc3Qgcm93XG4gIGZvcihqID0gMDsgaiA8PSBhLmxlbmd0aDsgaisrKSB7XG4gICAgbWF0cml4WzBdW2pdID0gajtcbiAgfVxuXG4gIC8vIEZpbGwgaW4gdGhlIHJlc3Qgb2YgdGhlIG1hdHJpeFxuICBmb3IoaSA9IDE7IGkgPD0gYi5sZW5ndGg7IGkrKykge1xuICAgIGZvcihqID0gMTsgaiA8PSBhLmxlbmd0aDsgaisrKSB7XG4gICAgICBpZihiLmNoYXJBdChpLTEpID09IGEuY2hhckF0KGotMSkpIHtcbiAgICAgICAgbWF0cml4W2ldW2pdID0gbWF0cml4W2ktMV1bai0xXTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIG1hdHJpeFtpXVtqXSA9IE1hdGgubWluKG1hdHJpeFtpLTFdW2otMV0gKyAxLCAvLyBzdWJzdGl0dXRpb25cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgTWF0aC5taW4obWF0cml4W2ldW2otMV0gKyAxLCAvLyBpbnNlcnRpb25cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgbWF0cml4W2ktMV1bal0gKyAxKSk7IC8vIGRlbGV0aW9uXG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIG1hdHJpeFtiLmxlbmd0aF1bYS5sZW5ndGhdO1xufVxuXG4vKipcbiAqIExvYWRzIHRoZSBnaXZlbiBjbGFzc25hbWUsIHBhcnNlcyB0aGUgY2xhc3MgZmlsZSwgYW5kIHJldHVybnMgdGhlIG1ldGhvZFxuICogc2lnbmF0dXJlcyBvZiBhbGwgb2YgdGhlIG5hdGl2ZSBtZXRob2RzIGluIHRoZSBjbGFzcy5cbiAqIGNsYXNzTmFtZSBzaG91bGQgYmUgaW4gcGtnL05hbWUgZm9ybWF0LCBub3QgcGtnLk5hbWUuXG4gKi9cbmZ1bmN0aW9uIGdldE5hdGl2ZVNpZ3MoY2xhc3NOYW1lOiBzdHJpbmcpOiBzdHJpbmdbXSB7XG4gIHZhciBydjogc3RyaW5nW10gPSBbXSwgaTogbnVtYmVyO1xuICBmb3IgKGkgPSAwOyBpIDwgY2xhc3NwYXRoLmxlbmd0aDsgaSsrKSB7XG4gICAgbGV0IGtsYXNzRGF0YSA9IGNsYXNzcGF0aFtpXS50cnlMb2FkQ2xhc3NTeW5jKGNsYXNzTmFtZSk7XG4gICAgaWYgKGtsYXNzRGF0YSAhPT0gbnVsbCkge1xuICAgICAgbGV0IGtsYXNzID0gbmV3IFJlZmVyZW5jZUNsYXNzRGF0YShrbGFzc0RhdGEpO1xuICAgICAgbGV0IG1ldGhvZHMgPSBrbGFzcy5nZXRNZXRob2RzKCk7XG4gICAgICBtZXRob2RzLmZvckVhY2goKG06IE1ldGhvZCkgPT4ge1xuICAgICAgICBpZiAobS5hY2Nlc3NGbGFncy5pc05hdGl2ZSgpKSB7XG4gICAgICAgICAgcnYucHVzaChtLnNpZ25hdHVyZSk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcnY7XG59XG5cbi8qKlxuICogV2hhdCBpdCBzYXlzIG9uIHRoZSB0aW4uIFJldHVybnMgYW4gYXJyYXkgb2YgY2xhc3MgbmFtZXMgKGluIGZvby9iYXIvQmF6XG4gKiBmb3JtYXQpIHRoYXQgaGF2ZSBuYXRpdmUgbWV0aG9kcyBpbXBsZW1lbnRlZCBpbiBEb3BwaW8uXG4gKi9cbmZ1bmN0aW9uIGdldENsYXNzZXNXaXRoSW1wbGVtZW50ZWROYXRpdmVzKCk6IHtbY2xzTmFtZTogc3RyaW5nXTogc3RyaW5nW119IHtcbiAgdmFyIG5hdGl2ZUltcGxzOiB7IFtjbHNOYW1lOiBzdHJpbmddOiB7IFttZXRoU2lnOiBzdHJpbmddOiBGdW5jdGlvbiB9IH0gPSBqdm1PYmplY3QuZ2V0TmF0aXZlcygpLFxuICAgICAgbWV0aG9kczoge1twa2dOYW1lOiBzdHJpbmddOiBzdHJpbmdbXX0gPSB7fTtcblxuICBPYmplY3Qua2V5cyhuYXRpdmVJbXBscykuZm9yRWFjaCgoY2xzTmFtZTogc3RyaW5nKSA9PiB7XG4gICAgbWV0aG9kc1tjbHNOYW1lXSA9IE9iamVjdC5rZXlzKG5hdGl2ZUltcGxzW2Nsc05hbWVdKTtcbiAgfSk7XG5cbiAgcmV0dXJuIG1ldGhvZHM7XG59XG5cbi8qKlxuICogR2l2ZW4gdHdvIGFycmF5cywgcmV0dXJucyBhbnkgaXRlbXMgaW4gdGhlIGZpcnN0IGFycmF5IHRoYXQgYXJlIG5vdCBpbiB0aGVcbiAqIHNlY29uZCBhcnJheS5cbiAqL1xuZnVuY3Rpb24gbWlzc2luZ0Zyb21BcnJheShpdGVtczogc3RyaW5nW10sIGFycjogc3RyaW5nW10pOiBzdHJpbmdbXSB7XG4gIHZhciBydjogc3RyaW5nW10gPSBbXSwgaTogbnVtYmVyO1xuICBmb3IgKGkgPSAwOyBpIDwgaXRlbXMubGVuZ3RoOyBpKyspIHtcbiAgICBpZiAoYXJyLmluZGV4T2YoaXRlbXNbaV0pID09PSAtMSkge1xuICAgICAgcnYucHVzaChpdGVtc1tpXSk7XG4gICAgfVxuICB9XG4gIHJldHVybiBydjtcbn1cblxuLyoqXG4gKiBHaXZlbiBhIHNldCBvZiBzdHJpbmdzIGFuZCBhIHNldCBvZiBjYW5kaWRhdGVzLCByZXR1cm5zIGEgbWFwIHRoYXQgbWFwc1xuICogZWFjaCBzdHJpbmcgdG8gdGhlIGNhbmRpZGF0ZSB3aXRoIHRoZSBzaG9ydGVzdCBlZGl0IGRpc3RhbmNlLlxuICovXG5mdW5jdGlvbiBnZXRTaW1pbGFyKHN0cnM6IHN0cmluZ1tdLCBjYW5kaWRhdGVzOiBzdHJpbmdbXSk6IHtbc2lnOiBzdHJpbmddOiBzdHJpbmd9IHtcbiAgdmFyIG1pbkVkaXREaXN0YW5jZTogbnVtYmVyLCBiZXN0Q2FuZGlkYXRlOiBzdHJpbmcsIGk6IG51bWJlciwgajogbnVtYmVyLFxuICAgICAgc3RyOiBzdHJpbmcsIHRtcDogbnVtYmVyLCBydjoge1tzaWc6IHN0cmluZ106IHN0cmluZ30gPSB7fTtcbiAgZm9yIChpID0gMDsgaSA8IHN0cnMubGVuZ3RoOyBpKyspIHtcbiAgICBtaW5FZGl0RGlzdGFuY2UgPSA5OTk5OTk5OTk5OTk5OTk7XG4gICAgc3RyID0gc3Ryc1tpXTtcbiAgICBmb3IgKGogPSAwOyBqIDwgY2FuZGlkYXRlcy5sZW5ndGg7IGorKykge1xuICAgICAgdG1wID0gZ2V0RWRpdERpc3RhbmNlKHN0ciwgY2FuZGlkYXRlc1tqXSk7XG4gICAgICBpZiAodG1wIDwgbWluRWRpdERpc3RhbmNlKSB7XG4gICAgICAgIG1pbkVkaXREaXN0YW5jZSA9IHRtcDtcbiAgICAgICAgYmVzdENhbmRpZGF0ZSA9IGNhbmRpZGF0ZXNbal07XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKGJlc3RDYW5kaWRhdGUgIT0gbnVsbCkge1xuICAgICAgcnZbc3RyXSA9IGJlc3RDYW5kaWRhdGU7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJ2W3N0cl0gPSBcIlwiO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcnY7XG59XG5cbmZ1bmN0aW9uIGZvcm1hdEtsYXNzKHRpdGxlOiBzdHJpbmcsIGRhdGE6IHtbbGVmdDogc3RyaW5nXTogc3RyaW5nfSk6IHN0cmluZyB7XG4gIHZhciBsZWZ0OiBzdHJpbmcsIG91dHB1dDogc3RyaW5nW10gPSBbXTtcbiAgLy8gRGV0ZXJtaW5lIHRoZSB3aWR0aCBvZiBlYWNoIGNvbHVtbiBieSBmaW5kaW5nIHRoZSBsb25nZXN0IGRhdGEgaXRlbS5cbiAgb3V0cHV0LnB1c2godGl0bGUpO1xuICBmb3IgKGxlZnQgaW4gZGF0YSkge1xuICAgIG91dHB1dC5wdXNoKFwiXFxuXFx0TWlzc2luZzogXCIgKyBsZWZ0KTtcbiAgICBvdXRwdXQucHVzaChcIlxcblxcdE1hdGNoPyA6IFwiICsgZGF0YVtsZWZ0XSArIFwiXFxuXCIpO1xuICB9XG5cbiAgcmV0dXJuIG91dHB1dC5qb2luKCcnKTtcbn1cblxuLyoqXG4gKiBQcmludHMgdGhlIHJlc3VsdC5cbiAqL1xuZnVuY3Rpb24gcHJpbnRSZXN1bHQocmVzdWx0OiB7W2tsYXNzTmFtZTogc3RyaW5nXToge1tzaWc6IHN0cmluZ106IHN0cmluZ319KTogdm9pZCB7XG4gIHZhciBrbGFzc05hbWU6IHN0cmluZywgcHJpbnRlZDogYm9vbGVhbiA9IGZhbHNlO1xuICBmb3IgKGtsYXNzTmFtZSBpbiByZXN1bHQpIHtcbiAgICBwcmludGVkID0gdHJ1ZTtcbiAgICBpZiAocmVzdWx0Lmhhc093blByb3BlcnR5KGtsYXNzTmFtZSkpIHtcbiAgICAgIGNvbnNvbGUubG9nKGZvcm1hdEtsYXNzKGtsYXNzTmFtZSwgcmVzdWx0W2tsYXNzTmFtZV0pICsgXCJcXG5cIik7XG4gICAgfVxuICB9XG5cbiAgaWYgKCFwcmludGVkKSB7XG4gICAgY29uc29sZS5sb2coXCJBbGwgbmF0aXZlIG1ldGhvZCBpbXBsZW1lbnRhdGlvbnMgbWF0Y2ggbmF0aXZlIG1ldGhvZHMgaW4geW91ciBjbGFzcyBmaWxlcy5cIik7XG4gIH1cbn1cblxuLyoqXG4gKiBNYWluIGZ1bmN0aW9uLlxuICovXG5mdW5jdGlvbiBtYWluKCkge1xuICB2YXIgaW1wbE5hdGl2ZXM6IHsgW3BrZ05hbWU6IHN0cmluZ106IHN0cmluZ1tdIH0gPSBnZXRDbGFzc2VzV2l0aEltcGxlbWVudGVkTmF0aXZlcygpLFxuICAgICAga2xhc3NOYXRpdmVzOiBzdHJpbmdbXSwga2xhc3NJbXBsTmF0aXZlczogc3RyaW5nW10sIGtsYXNzTmFtZTogc3RyaW5nLFxuICAgICAgbWlzc2luZ05hdGl2ZXM6IHN0cmluZ1tdLFxuICAgICAgc2ltaWxhck1hcDoge1trbGFzc05hbWU6IHN0cmluZ106IHtbc2lnOiBzdHJpbmddOiBzdHJpbmd9fSA9IHt9O1xuICBmb3IgKGtsYXNzTmFtZSBpbiBpbXBsTmF0aXZlcykge1xuICAgIGlmIChpbXBsTmF0aXZlcy5oYXNPd25Qcm9wZXJ0eShrbGFzc05hbWUpKSB7XG4gICAgICBrbGFzc05hdGl2ZXMgPSBnZXROYXRpdmVTaWdzKGtsYXNzTmFtZSk7XG4gICAgICBrbGFzc0ltcGxOYXRpdmVzID0gaW1wbE5hdGl2ZXNba2xhc3NOYW1lXTtcblxuICAgICAgLy8gU2VlIGlmIGFueSBpbXBsZW1lbnRlZCBuYXRpdmVzIGFyZSBub3QgaW4gbmF0aXZlcy5cbiAgICAgIG1pc3NpbmdOYXRpdmVzID0gbWlzc2luZ0Zyb21BcnJheShrbGFzc0ltcGxOYXRpdmVzLCBrbGFzc05hdGl2ZXMpO1xuICAgICAgaWYgKG1pc3NpbmdOYXRpdmVzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgLy8gRmluZCBjYW5kaWRhdGVzIGZvciB0aGUgbWlzc2luZyBpdGVtcy5cbiAgICAgICAgc2ltaWxhck1hcFtrbGFzc05hbWVdID0gZ2V0U2ltaWxhcihtaXNzaW5nTmF0aXZlcywga2xhc3NOYXRpdmVzKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbiAgcHJpbnRSZXN1bHQoc2ltaWxhck1hcCk7XG59XG5cbmNvbnN0IEpBVkFfSE9NRSA9IHBhdGgucmVzb2x2ZShfX2Rpcm5hbWUsICcuLi92ZW5kb3IvamF2YV9ob21lJyk7XG5jb25zdCBET1BQSU9fSE9NRSA9IHBhdGgucmVzb2x2ZShfX2Rpcm5hbWUsICcuLicpO1xubGV0IG9wdHM6IEpWTU9wdGlvbnMgPSA8YW55PiBtZXJnZShKVk0uZ2V0RGVmYXVsdE9wdGlvbnMoRE9QUElPX0hPTUUpLCB7XG4gIG5hdGl2ZUNsYXNzcGF0aDogW3BhdGgucmVzb2x2ZShfX2Rpcm5hbWUsICcuLi9zcmMvbmF0aXZlcycpXVxufSk7XG5cbm5ldyBKVk0ob3B0cywgZnVuY3Rpb24oZXJyOiBhbnksIF9qdm1PYmplY3Q6IEpWTSkge1xuICBpZiAoZXJyKSB7XG4gICAgdGhyb3cgZXJyO1xuICB9XG4gIGp2bU9iamVjdCA9IF9qdm1PYmplY3Q7XG4gIENsYXNzcGF0aEZhY3RvcnkoSkFWQV9IT01FLCBvcHRzLmJvb3RzdHJhcENsYXNzcGF0aCwgKGl0ZW1zKSA9PiB7XG4gICAgY2xhc3NwYXRoID0gaXRlbXM7XG4gICAgbWFpbigpO1xuICB9KVxufSk7XG4iXX0=