"use strict";
var gLong_1 = require('./gLong');
var assert_1 = require('./assert');
/**
 * A ByteStream, implemented using a NodeBuffer.
 */
var ByteStream = (function () {
    function ByteStream(buffer) {
        this.buffer = buffer;
        this._index = 0;
    }
    /**
     * Returns the current read index, and increments the index by the indicated
     * amount.
     */
    ByteStream.prototype.incIndex = function (inc) {
        var readIndex = this._index;
        this._index += inc;
        return readIndex;
    };
    ByteStream.prototype.rewind = function () {
        this._index = 0;
    };
    ByteStream.prototype.seek = function (idx) {
        assert_1["default"](idx >= 0 && idx < this.buffer.length, "Invalid seek position.");
        this._index = idx;
    };
    ByteStream.prototype.pos = function () {
        return this._index;
    };
    ByteStream.prototype.skip = function (bytesCount) {
        this._index += bytesCount;
    };
    ByteStream.prototype.hasBytes = function () {
        return this._index < this.buffer.length;
    };
    ByteStream.prototype.getFloat = function () {
        return this.buffer.readFloatBE(this.incIndex(4));
    };
    ByteStream.prototype.getDouble = function () {
        return this.buffer.readDoubleBE(this.incIndex(8));
    };
    ByteStream.prototype.getUint = function (byteCount) {
        switch (byteCount) {
            case 1:
                return this.getUint8();
            case 2:
                return this.getUint16();
            case 4:
                return this.getUint32();
            default:
                throw new Error("Invalid byte count for getUint: " + byteCount);
        }
    };
    ByteStream.prototype.getInt = function (byteCount) {
        switch (byteCount) {
            case 1:
                return this.getInt8();
            case 2:
                return this.getInt16();
            case 4:
                return this.getInt32();
            default:
                throw new Error("Invalid byte count for getUint: " + byteCount);
        }
    };
    ByteStream.prototype.getUint8 = function () {
        return this.buffer[this.incIndex(1)];
    };
    ByteStream.prototype.getUint16 = function () {
        return this.buffer.readUInt16BE(this.incIndex(2));
    };
    ByteStream.prototype.getUint32 = function () {
        return this.buffer.readUInt32BE(this.incIndex(4));
    };
    ByteStream.prototype.getInt8 = function () {
        return this.buffer.readInt8(this.incIndex(1));
    };
    ByteStream.prototype.getInt16 = function () {
        return this.buffer.readInt16BE(this.incIndex(2));
    };
    ByteStream.prototype.getInt32 = function () {
        return this.buffer.readInt32BE(this.incIndex(4));
    };
    ByteStream.prototype.getInt64 = function () {
        var high = this.getUint32();
        var low = this.getUint32();
        return gLong_1["default"].fromBits(low, high);
    };
    ByteStream.prototype.read = function (bytesCount) {
        var rv = this.buffer.slice(this._index, this._index + bytesCount);
        this._index += bytesCount;
        return rv;
    };
    ByteStream.prototype.peek = function () {
        return this.buffer[this._index];
    };
    ByteStream.prototype.size = function () {
        return this.buffer.length - this._index;
    };
    ByteStream.prototype.slice = function (len) {
        var arr = new ByteStream(this.buffer.slice(this._index, this._index + len));
        this._index += len;
        return arr;
    };
    ByteStream.prototype.getBuffer = function () {
        return this.buffer;
    };
    return ByteStream;
}());
exports.__esModule = true;
exports["default"] = ByteStream;
//# sourceMappingURL=data:application/json;base64,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