"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var util_1 = require('./util');
var util = require('./util');
var ByteStream_1 = require('./ByteStream');
var ConstantPool_1 = require('./ConstantPool');
var attributes_1 = require('./attributes');
var threading_1 = require('./threading');
var logging = require('./logging');
var methods_1 = require('./methods');
var ClassLoader_1 = require('./ClassLoader');
var enums_1 = require('./enums');
var ClassLock_1 = require('./ClassLock');
var assert_1 = require('./assert');
var gLong_1 = require('./gLong');
var StringOutputStream_1 = require('./StringOutputStream');
var Monitor_1 = require('./Monitor');
var trace = logging.trace;
var debug = logging.debug;
var global_1 = require('./global');
if (typeof RELEASE === 'undefined')
    global_1["default"].RELEASE = false;
/**
 * Auto-incrementing reference number. Uniquely identifies each object allocated
 * by the JVM. Started at 1 because we use 0 to identify NULL.
 */
var ref = 1;
function getRef() {
    return ref++;
}
/**
 * Defines special JVM-injected fields. The map stores the TypeScript type of
 * the field and the default value for the field, which will be assigned in the
 * JavaScript constructor for the class.
 */
var injectedFields = {
    'Ljava/lang/invoke/MemberName;': {
        vmtarget: ["(thread: JVMThread, descriptor: string, args: any[], cb?: (e?: JVMTypes.java_lang_Throwable, rv?: any) => void) => void", "null"],
        vmindex: ["number", "-1"]
    },
    'Ljava/lang/Object;': {
        'ref': ["number", "getRef()"],
        '$monitor': ["Monitor", "null"]
    },
    'Ljava/net/PlainSocketImpl;': {
        '$is_shutdown': ['boolean', 'false'],
        '$ws': ['Interfaces.IWebsock', 'null']
    },
    'Ljava/lang/Class;': {
        '$cls': ['ClassData', 'null'],
        'signers': ['JVMTypes.java_lang_Object[]', 'null']
    },
    'Ljava/lang/ClassLoader;': {
        '$loader': ['ClassLoader', 'new CustomClassLoader(thread.getBsCl(), this);']
    },
    'Ljava/lang/Thread;': {
        // Note: Need to handle initial case when thread is NULL.
        '$thread': ['JVMThread', 'thread ? new thread.constructor(thread.getJVM(), thread.getThreadPool(), this) : null']
    }
};
/**
 * Defines special JVM-injected method. The map stores the TypeScript type
 * signature of the method and the JavaScript body of the method, keyed on the
 * method's name. These are all instance methods (e.g. non-static).
 */
var injectedMethods = {
    'Ljava/lang/Object;': {
        'getClass': ["(): ClassData", "function() { return this.constructor.cls }"],
        'getMonitor': ["(): Monitor", "function() {\n  if (this.$monitor === null) {\n    this.$monitor = new Monitor();\n  }\n  return this.$monitor;\n}"]
    },
    'Ljava/lang/String;': {
        'toString': ["(): string", "function() { return util.chars2jsStr(this['java/lang/String/value']); }"]
    },
    'Ljava/lang/Byte;': {
        'unbox': ["(): number", "function() { return this['java/lang/Byte/value']; }"]
    },
    'Ljava/lang/Character;': {
        'unbox': ["(): number", "function() { return this['java/lang/Character/value']; }"]
    },
    'Ljava/lang/Double;': {
        'unbox': ["(): number", "function() { return this['java/lang/Double/value']; }"]
    },
    'Ljava/lang/Float;': {
        'unbox': ["(): number", "function() { return this['java/lang/Float/value']; }"]
    },
    'Ljava/lang/Integer;': {
        'unbox': ["(): number", "function() { return this['java/lang/Integer/value']; }"]
    },
    'Ljava/lang/Long;': {
        'unbox': ["(): Long", "function() { return this['java/lang/Long/value']; }"]
    },
    'Ljava/lang/Short;': {
        'unbox': ["(): number", "function() { return this['java/lang/Short/value']; }"]
    },
    'Ljava/lang/Boolean;': {
        'unbox': ["(): number", "function() { return this['java/lang/Boolean/value']; }"]
    },
    // To catch any errors. Should never actually happen; Voids don't show up in arg lists.
    'Ljava/lang/Void;': {
        'unbox': ["(): number", "function() { throw new Error(\"Cannot unbox a Void type.\"); }"]
    },
    'Ljava/lang/invoke/MethodType;': {
        'toString': ["(): string", "function() { return \"(\" + this['java/lang/invoke/MethodType/ptypes'].array.map(function (type) { return type.$cls.getInternalName(); }).join(\"\") + \")\" + this['java/lang/invoke/MethodType/rtype'].$cls.getInternalName(); }"]
    }
};
/**
 * Same as injected methods, but these are static.
 */
var injectedStaticMethods = {
    'Ljava/lang/Byte;': {
        'box': ["(val: number): java_lang_Byte", "function(val) { var rv = new this(null); rv['java/lang/Byte/value'] = val; return rv; }"]
    },
    'Ljava/lang/Character;': {
        'box': ["(val: number): java_lang_Character", "function(val) { var rv = new this(null); rv['java/lang/Character/value'] = val; return rv; }"]
    },
    'Ljava/lang/Double;': {
        'box': ["(val: number): java_lang_Double", "function(val) { var rv = new this(null); rv['java/lang/Double/value'] = val; return rv; }"]
    },
    'Ljava/lang/Float;': {
        'box': ["(val: number): java_lang_Float", "function(val) { var rv = new this(null); rv['java/lang/Float/value'] = val; return rv; }"]
    },
    'Ljava/lang/Integer;': {
        'box': ["(val: number): java_lang_Integer", "function(val) { var rv = new this(null); rv['java/lang/Integer/value'] = val; return rv; }"]
    },
    'Ljava/lang/Long;': {
        'box': ["(val: Long): java_lang_Long", "function(val) { var rv = new this(null); rv['java/lang/Long/value'] = val; return rv; }"]
    },
    'Ljava/lang/Short;': {
        'box': ["(val: number): java_lang_Short", "function(val) { var rv = new this(null); rv['java/lang/Short/value'] = val; return rv; }"]
    },
    'Ljava/lang/Boolean;': {
        'box': ["(val: number): java_lang_Boolean", "function(val) { var rv = new this(null); rv['java/lang/Boolean/value'] = val; return rv; }"]
    },
    'Ljava/lang/Void;': {
        'box': ["(): java_lang_Void", "function() { return new this(null); }"]
    }
};
/**
 * Extends a JVM class by making its prototype a blank instantiation of an
 * object with the super class's prototype as its prototype. Inspired from
 * TypeScript's __extend function.
 */
function extendClass(cls, superCls) {
    function __() { this.constructor = cls; }
    __.prototype = superCls.prototype;
    cls.prototype = new __();
}
/**
 * Represents a single class in the JVM.
 */
var ClassData = (function () {
    /**
     * Responsible for setting up all of the fields that are guaranteed to be
     * present on any ClassData object.
     */
    function ClassData(loader) {
        this.accessFlags = null;
        /**
         * We make this private to *enforce* call sites to use our getter functions.
         * The actual state of this class depends on the state of its parents, and
         * parents do not inform their children when they change state.
         */
        this.state = enums_1.ClassState.LOADED;
        this.jco = null;
        this.superClass = null;
        this.loader = loader;
    }
    /**
     * Get the external form of this class's name (e.g. java.lang.String).
     */
    ClassData.prototype.getExternalName = function () {
        return util_1.ext_classname(this.className);
    };
    /**
     * Get the internal form of this class's name (e.g. Ljava/lang/String;).
     */
    ClassData.prototype.getInternalName = function () {
        return this.className;
    };
    /**
     * Get the name of the package that this class belongs to (e.g. java.lang).
     */
    ClassData.prototype.getPackageName = function () {
        var extName = this.getExternalName(), i;
        // Find the index of the last '.' in the name.
        for (i = extName.length - 1; i >= 0 && extName[i] !== '.'; i--) { }
        if (i >= 0) {
            return extName.slice(0, i);
        }
        else {
            return "";
        }
    };
    /**
     * Returns the ClassLoader object of the classloader that initialized this
     * class. Returns null for the default classloader.
     */
    ClassData.prototype.getLoader = function () {
        return this.loader;
    };
    /**
     * Get the class's super class, which is always going to be a reference
     * class.
     */
    ClassData.prototype.getSuperClass = function () {
        return this.superClass;
    };
    /**
     * Get all of the interfaces that the class implements.
     */
    ClassData.prototype.getInterfaces = function () {
        return [];
    };
    /**
     * Get all of the injected fields for this class. The value for each field
     * in the returned map is its type.
     */
    ClassData.prototype.getInjectedFields = function () {
        var rv = {};
        if (injectedFields[this.getInternalName()] !== undefined) {
            var fields = injectedFields[this.getInternalName()];
            Object.keys(fields).forEach(function (fieldName) {
                rv[fieldName] = fields[fieldName][0];
            });
        }
        return rv;
    };
    /**
     * Get all of the injected methods for this class. The value for each method
     * in the returned map is its type.
     */
    ClassData.prototype.getInjectedMethods = function () {
        var rv = {}, lookupName = this.getInternalName();
        // All array classes share the same injected methods.
        if (lookupName[0] === '[') {
            lookupName = '[';
        }
        if (injectedMethods[lookupName] !== undefined) {
            var methods = injectedMethods[lookupName];
            Object.keys(methods).forEach(function (methodName) {
                rv[methodName] = methods[methodName][0];
            });
        }
        return rv;
    };
    /**
     * Get all of the injected static methods for this class. The value for each
     * method in the returned map is its type.
     */
    ClassData.prototype.getInjectedStaticMethods = function () {
        var rv = {}, lookupName = this.getInternalName();
        // All array classes share the same injected methods.
        if (lookupName[0] === '[') {
            lookupName = '[';
        }
        if (injectedStaticMethods[lookupName] !== undefined) {
            var methods = injectedStaticMethods[lookupName];
            Object.keys(methods).forEach(function (methodName) {
                rv[methodName] = methods[methodName][0];
            });
        }
        return rv;
    };
    /**
     * Get a java.lang.Class object corresponding to this class.
     */
    ClassData.prototype.getClassObject = function (thread) {
        if (this.jco === null) {
            this.jco = new (thread.getBsCl().getResolvedClass('Ljava/lang/Class;').getConstructor(thread))(thread);
            this.jco.$cls = this;
            this.jco['java/lang/Class/classLoader'] = this.getLoader().getLoaderObject();
        }
        return this.jco;
    };
    /**
     * Get the protection domain of this class.
     * This value is NULL for all but reference classes loaded by app classloaders.
     */
    ClassData.prototype.getProtectionDomain = function () {
        return null;
    };
    /**
     * Retrieves the method defined in this particular class by the given method
     * signature *without* invoking method lookup.
     * @param methodSignature The method's full signature, e.g. <clinit>()V
     */
    ClassData.prototype.getMethod = function (methodSignature) {
        return null;
    };
    /**
     * Retrieve all of the methods defined on this class.
     */
    ClassData.prototype.getMethods = function () {
        return [];
    };
    /**
     * Retrieve the set of fields defined on this class.
     */
    ClassData.prototype.getFields = function () {
        return [];
    };
    /**
     * Set the state of this particular class to LOADED/RESOLVED/INITIALIZED.
     */
    ClassData.prototype.setState = function (state) {
        this.state = state;
    };
    /**
     * Gets the current state of this class.
     */
    ClassData.prototype.getState = function () {
        if (this.state === enums_1.ClassState.RESOLVED && this.getMethod('<clinit>()V') === null) {
            // We can promote to INITIALIZED if this class has no static initialization
            // logic, and its parent class is initialized.
            var scls = this.getSuperClass();
            if (scls !== null && scls.getState() === enums_1.ClassState.INITIALIZED) {
                this.state = enums_1.ClassState.INITIALIZED;
            }
        }
        return this.state;
    };
    /**
     * Check if the class is initialized.
     * @param thread The thread that is performing the check. If initialization
     *   is in progress on that thread, then the class is, for all intents and
     *   purposes, initialized.
     */
    ClassData.prototype.isInitialized = function (thread) {
        return this.getState() === enums_1.ClassState.INITIALIZED;
    };
    // Convenience function.
    ClassData.prototype.isResolved = function () { return this.getState() !== enums_1.ClassState.LOADED; };
    ClassData.prototype.isSubinterface = function (target) {
        return false;
    };
    ClassData.prototype.isSubclass = function (target) {
        if (this === target) {
            return true;
        }
        if (this.getSuperClass() === null) {
            return false;
        }
        return this.getSuperClass().isSubclass(target);
    };
    ClassData.prototype.resolve = function (thread, cb, explicit) {
        if (explicit === void 0) { explicit = true; }
        throw new Error("Unimplemented.");
    };
    ClassData.prototype.initialize = function (thread, cb, explicit) {
        if (explicit === void 0) { explicit = true; }
        throw new Error("Unimplemented.");
    };
    ClassData.prototype.outputInjectedMethods = function (jsClassName, outputStream) {
        var lookupName = this.getInternalName();
        if (lookupName[0] === '[') {
            lookupName = '[';
        }
        if (injectedMethods[lookupName] !== undefined) {
            var methods = injectedMethods[lookupName];
            Object.keys(methods).forEach(function (methodName) {
                outputStream.write("  " + jsClassName + ".prototype." + methodName + " = " + methods[methodName][1] + ";\n");
            });
        }
        if (injectedStaticMethods[lookupName] !== undefined) {
            var staticMethods = injectedStaticMethods[lookupName];
            Object.keys(staticMethods).forEach(function (methodName) {
                outputStream.write("  " + jsClassName + "." + methodName + " = " + staticMethods[methodName][1] + ";\n");
            });
        }
    };
    return ClassData;
}());
exports.ClassData = ClassData;
var PrimitiveClassData = (function (_super) {
    __extends(PrimitiveClassData, _super);
    function PrimitiveClassData(className, loader) {
        _super.call(this, loader);
        this.className = className;
        // PrimitiveClassData objects are ABSTRACT, FINAL, and PUBLIC.
        this.accessFlags = new util_1.Flags(0x411);
        this.setState(enums_1.ClassState.INITIALIZED);
    }
    /**
     * Returns a boolean indicating if this class is an instance of the target class.
     * "target" is a ClassData object.
     * The ClassData objects do not need to be initialized; just loaded.
     */
    PrimitiveClassData.prototype.isCastable = function (target) {
        return this.className === target.getInternalName();
    };
    /**
     * Returns the internal class name for the corresponding boxed type.
     */
    PrimitiveClassData.prototype.boxClassName = function () {
        return util_1.boxClassName(this.className);
    };
    /**
     * Returns a boxed version of the given primitive.
     */
    PrimitiveClassData.prototype.createWrapperObject = function (thread, value) {
        var boxName = this.boxClassName();
        var boxCls = thread.getBsCl().getInitializedClass(thread, boxName);
        // these are all initialized in preinit (for the BSCL, at least)
        var boxCons = boxCls.getConstructor(thread);
        var wrapped = new boxCons(thread);
        if (boxName !== 'V') {
            // XXX: all primitive wrappers store their value in a private static final field named 'value'
            wrapped[util_1.descriptor2typestr(boxName) + '/value'] = value;
            assert_1["default"](typeof value === "number" || typeof value === "boolean" || typeof value.low_ === "number", "Invalid primitive value: " + value);
        }
        return wrapped;
    };
    PrimitiveClassData.prototype.tryToResolve = function () {
        return true;
    };
    PrimitiveClassData.prototype.tryToInitialize = function () {
        return true;
    };
    /**
     * Primitive classes are already resolved.
     */
    PrimitiveClassData.prototype.resolve = function (thread, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        setImmediate(function () { return cb(_this); });
    };
    return PrimitiveClassData;
}(ClassData));
exports.PrimitiveClassData = PrimitiveClassData;
var ArrayClassData = (function (_super) {
    __extends(ArrayClassData, _super);
    function ArrayClassData(componentType, loader) {
        _super.call(this, loader);
        this._constructor = null;
        this.className = "[" + componentType;
        // ArrayClassData objects are ABSTRACT, FINAL, and PUBLIC.
        this.accessFlags = new util_1.Flags(0x411);
        this.componentClassName = componentType;
    }
    /**
     * Looks up a method with the given signature. Returns null if no method
     * found.
     */
    ArrayClassData.prototype.methodLookup = function (signature) {
        return this.superClass.methodLookup(signature);
    };
    ArrayClassData.prototype.fieldLookup = function (name) {
        return this.superClass.fieldLookup(name);
    };
    /**
     * Resolve the class.
     */
    ArrayClassData.prototype.resolve = function (thread, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        if (this.isResolved()) {
            // Short circuit.
            setImmediate(function () { return cb(_this); });
            return;
        }
        util_1.asyncForEach(["Ljava/lang/Object;", this.componentClassName], function (cls, nextItem) {
            _this.loader.resolveClass(thread, cls, function (cdata) {
                if (cdata !== null) {
                    nextItem();
                }
                else {
                    nextItem("Failed.");
                }
            });
        }, function (err) {
            if (!err) {
                _this.setResolved(_this.loader.getResolvedClass("Ljava/lang/Object;"), _this.loader.getResolvedClass(_this.componentClassName));
                cb(_this);
            }
            else {
                cb(null);
            }
        });
    };
    ArrayClassData.prototype.getComponentClass = function () {
        return this.componentClass;
    };
    /**
     * Resolved and initialized are the same for array types.
     */
    ArrayClassData.prototype.setResolved = function (super_class_cdata, component_class_cdata) {
        this.superClass = super_class_cdata;
        this.componentClass = component_class_cdata;
        this.setState(enums_1.ClassState.INITIALIZED);
    };
    ArrayClassData.prototype.tryToResolve = function () {
        var loader = this.loader, superClassCdata = loader.getResolvedClass("Ljava/lang/Object;"), componentClassCdata = loader.getResolvedClass(this.componentClassName);
        if (superClassCdata === null || componentClassCdata === null) {
            return false;
        }
        else {
            this.setResolved(superClassCdata, componentClassCdata);
            return true;
        }
    };
    ArrayClassData.prototype.tryToInitialize = function () {
        // Arrays are initialized once resolved.
        return this.tryToResolve();
    };
    /**
     * Returns a boolean indicating if this class is an instance of the target class.
     * "target" is a ClassData object.
     * The ClassData objects do not need to be initialized; just loaded.
     * See §2.6.7 for casting rules.
     */
    ArrayClassData.prototype.isCastable = function (target) {
        if (!(target instanceof ArrayClassData)) {
            if (target instanceof PrimitiveClassData) {
                return false;
            }
            // Must be a reference type.
            if (target.accessFlags.isInterface()) {
                // Interface reference type
                var type = target.getInternalName();
                return type === 'Ljava/lang/Cloneable;' || type === 'Ljava/io/Serializable;';
            }
            // Non-interface reference type
            return target.getInternalName() === 'Ljava/lang/Object;';
        }
        // We are both array types, so it only matters if my component type can be
        // cast to its component type.
        return this.getComponentClass().isCastable(target.getComponentClass());
    };
    ArrayClassData.prototype.initialize = function (thread, cb, explicit) {
        if (explicit === void 0) { explicit = true; }
        this.resolve(thread, cb, explicit);
    };
    /**
     * Get the array constructor for this particular JVM array class.
     * Uses typed arrays when available for primitive arrays.
     */
    ArrayClassData.prototype.getJSArrayConstructor = function () {
        if (!util_1.typedArraysSupported) {
            return 'Array';
        }
        switch (this.componentClassName) {
            case 'B':
                return 'Int8Array';
            case 'C':
                return 'Uint16Array';
            case 'S':
                return 'Int16Array';
            case 'I':
                return 'Int32Array';
            case 'F':
                return 'Float32Array';
            case 'D':
                return 'Float64Array';
            default:
                return 'Array';
        }
    };
    /**
     * Get the initial value placed into each array element.
     */
    ArrayClassData.prototype.getJSDefaultArrayElement = function () {
        switch (this.componentClassName[0]) {
            case '[':
                return "new (cls.getComponentClass().getConstructor())(thread, otherLengths)";
            case 'L':
                return "null";
            case 'J':
                return "gLongZero";
            default:
                return "0";
        }
    };
    /**
     * Creates a specialized `slice` method that creates a shallow slice of this
     * array. Specialized to the type of array (JS or Typed).
     */
    ArrayClassData.prototype._getSliceMethod = function () {
        var output = new StringOutputStream_1["default"](), jsArrCons = this.getJSArrayConstructor();
        output.write("function(start, end) {\n    var newObj = new this.constructor(null, 0);\n");
        if (jsArrCons === 'Array') {
            output.write("    newObj.array = this.array.slice(start, end);\n");
        }
        else {
            var elementSize;
            switch (jsArrCons) {
                case 'Int8Array':
                    elementSize = 1;
                    break;
                case 'Int16Array':
                case 'Uint16Array':
                    elementSize = 2;
                    break;
                case 'Int32Array':
                case 'Float32Array':
                    elementSize = 4;
                    break;
                case 'Float64Array':
                    elementSize = 8;
                    break;
                default:
                    assert_1["default"](false, "Illegal array type returned??");
            }
            // Although ArrayBuffer.slice has an optional end argument, we need to
            // multiply it if it does exist.
            output.write("    if (end === undefined) end = this.array.length;\n      " + (elementSize > 1 ? "start *= " + elementSize + ";\nend *= " + elementSize + ";" : '') + "\n      newObj.array = new " + jsArrCons + "(this.array.buffer.slice(start, end));\n");
        }
        output.write("    return newObj;\n  }");
        return output.flush();
    };
    ArrayClassData.prototype._constructConstructor = function (thread) {
        assert_1["default"](this._constructor === null, "Tried to construct constructor twice for " + this.getExternalName() + "!");
        var outputStream = new StringOutputStream_1["default"](), jsClassName = util_1.jvmName2JSName(this.getInternalName());
        // Arguments: extendClass, cls, superCls, gLongZero, thread
        outputStream.write("extendClass(" + jsClassName + ", superCls.getConstructor(thread));\n  function " + jsClassName + "(thread, lengths) {\n");
        this.superClass.outputInjectedFields(outputStream);
        // Initialize array.
        if (this.componentClassName[0] !== '[') {
            // Array elements are a non-array type.
            outputStream.write("    this.array = new " + this.getJSArrayConstructor() + "(lengths);\n");
            if (this.getJSArrayConstructor() === 'Array') {
                // TypedArrays are already initialized to 0, so this check skips array
                // initialization in that case.
                outputStream.write("    for (var i = 0; i < lengths; i++) {\n      this.array[i] = " + this.getJSDefaultArrayElement() + ";\n    }\n");
            }
        }
        else {
            // Multi-dimensional array.
            outputStream.write("    if (typeof lengths === 'number') {\n        this.array = new " + this.getJSArrayConstructor() + "(lengths);\n        for (var i = 0; i < length; i++) {\n          this.array[i] = null;\n        }\n      } else {\n        var length = lengths[0], otherLengths = lengths.length > 2 ? lengths.slice(1) : lengths[1];\n        this.array = new " + this.getJSArrayConstructor() + "(length);\n        for (var i = 0; i < length; i++) {\n          this.array[i] = " + this.getJSDefaultArrayElement() + ";\n        }\n      }\n");
        }
        outputStream.write("  }\n\n  " + jsClassName + ".prototype.slice = " + this._getSliceMethod() + ";\n  " + jsClassName + ".cls = cls;\n");
        this.outputInjectedMethods(jsClassName, outputStream);
        outputStream.write("\n  return " + jsClassName + ";");
        // All arrays extend java/lang/Object
        // function _create(extendClass, cls, superCls, gLongZero, thread) {
        var fcn = new Function("extendClass", "cls", "superCls", "gLongZero", "thread", "getRef", "util", outputStream.flush());
        return fcn(extendClass, this, this.superClass, gLong_1["default"].ZERO, thread, getRef, util);
    };
    ArrayClassData.prototype.getConstructor = function (thread) {
        assert_1["default"](this.isResolved(), "Tried to get constructor for class " + this.getInternalName() + " before it was resolved.");
        if (this._constructor === null) {
            this._constructor = this._constructConstructor(thread);
        }
        return this._constructor;
    };
    return ArrayClassData;
}(ClassData));
exports.ArrayClassData = ArrayClassData;
/**
 * Represents a "reference" Class -- that is, a class that neither represents a
 * primitive nor an array.
 */
var ReferenceClassData = (function (_super) {
    __extends(ReferenceClassData, _super);
    function ReferenceClassData(buffer, protectionDomain, loader, cpPatches) {
        _super.call(this, loader);
        this.interfaceClasses = null;
        this.superClassRef = null;
        /**
         * Initialization lock.
         */
        this.initLock = new ClassLock_1["default"]();
        /**
         * Stores the JavaScript constructor for this class.
         */
        this._constructor = null;
        /**
         * Virtual field table
         */
        this._fieldLookup = {};
        /**
         * All fields in object instantiations. Fields from super classes are in front
         * of fields from this class. A field's offset in the array is its *vmindex*.
         */
        this._objectFields = [];
        /**
         * All static fields in this particular class. The field's offset in this
         * array is its *vmindex*.
         */
        this._staticFields = [];
        /**
         * Virtual method table, keyed by method signature. Unlike _vmTable,
         * _methodLookup contains static methods and constructors, too.
         */
        this._methodLookup = {};
        /**
         * Virtual method table, keyed by vmindex. Methods originally defined by
         * super classes are in front of methods defined in this class. Overriding
         * methods are placed into the vmindex of the originating method.
         */
        this._vmTable = [];
        /**
         * Default method implementations that this class did *not* inherit, but are
         * still invocable in the class via their full name (e.g. through an
         * invokespecial bytecode).
         */
        this._uninheritedDefaultMethods = [];
        this._protectionDomain = protectionDomain ? protectionDomain : null;
        var byteStream = new ByteStream_1["default"](buffer), i = 0;
        if ((byteStream.getUint32()) !== 0xCAFEBABE) {
            throw new Error("Magic number invalid");
        }
        this.minorVersion = byteStream.getUint16();
        this.majorVersion = byteStream.getUint16();
        if (!(45 <= this.majorVersion && this.majorVersion <= 52)) {
            throw new Error("Major version invalid");
        }
        this.constantPool = new ConstantPool_1.ConstantPool();
        this.constantPool.parse(byteStream, cpPatches);
        // bitmask for {public,final,super,interface,abstract} class modifier
        this.accessFlags = new util_1.Flags(byteStream.getUint16());
        this.className = this.constantPool.get(byteStream.getUint16()).name;
        // super reference is 0 when there's no super (basically just java.lang.Object)
        var superRef = byteStream.getUint16();
        if (superRef !== 0) {
            this.superClassRef = this.constantPool.get(superRef);
        }
        // direct interfaces of this class
        var isize = byteStream.getUint16();
        this.interfaceRefs = new Array(isize);
        for (i = 0; i < isize; ++i) {
            this.interfaceRefs[i] = this.constantPool.get(byteStream.getUint16());
        }
        // fields of this class
        var numFields = byteStream.getUint16();
        this.fields = new Array(numFields);
        for (i = 0; i < numFields; ++i) {
            this.fields[i] = new methods_1.Field(this, this.constantPool, i, byteStream);
        }
        // class methods
        var numMethods = byteStream.getUint16();
        this.methods = new Array(numMethods);
        for (i = 0; i < numMethods; i++) {
            var m = new methods_1.Method(this, this.constantPool, i, byteStream);
            this.methods[i] = m;
        }
        // class attributes
        this.attrs = attributes_1.makeAttributes(byteStream, this.constantPool);
        if (byteStream.hasBytes()) {
            throw "Leftover bytes in classfile: " + byteStream;
        }
    }
    ReferenceClassData.prototype.getSuperClassReference = function () {
        return this.superClassRef;
    };
    ReferenceClassData.prototype.getInterfaceClassReferences = function () {
        return this.interfaceRefs.slice(0);
    };
    /**
     * Retrieve the set of interfaces that this class implements.
     * DO NOT MUTATE!
     */
    ReferenceClassData.prototype.getInterfaces = function () {
        return this.interfaceClasses;
    };
    /**
     * The set of fields that this class has.
     * DO NOT MUTATE!
     */
    ReferenceClassData.prototype.getFields = function () {
        return this.fields;
    };
    /**
     * Get the Virtual Method table for this class.
     */
    ReferenceClassData.prototype.getVMTable = function () {
        return this._vmTable;
    };
    /**
     * Returns the VM index for the given method. Returns -1 if not present in the
     * virtual method table (e.g. is static or a constructor).
     */
    ReferenceClassData.prototype.getVMIndexForMethod = function (m) {
        // Use M's signature, as we might override the method and use a different
        // method object in the table for its vmindex.
        return this._vmTable.indexOf(this.methodLookup(m.signature));
    };
    /**
     * Returns the method corresponding to the given VMIndex.
     */
    ReferenceClassData.prototype.getMethodFromVMIndex = function (i) {
        if (this._vmTable[i] !== undefined) {
            return this._vmTable[i];
        }
        return null;
    };
    /**
     * Get the VM index for the given field
     * NOTE: A static and non-static field can have the same vmindex! Caller must
     * be able to differentiate between static and non-static behavior!
     */
    ReferenceClassData.prototype.getVMIndexForField = function (f) {
        if (f.accessFlags.isStatic()) {
            assert_1["default"](f.cls === this, "Looks like we actually need to support static field lookups!");
            return this._staticFields.indexOf(f);
        }
        else {
            return this._objectFields.indexOf(f);
        }
    };
    ReferenceClassData.prototype.getStaticFieldFromVMIndex = function (index) {
        var f = this._staticFields[index];
        if (f !== undefined) {
            return f;
        }
        return null;
    };
    ReferenceClassData.prototype.getObjectFieldFromVMIndex = function (index) {
        var f = this._objectFields[index];
        if (f !== undefined) {
            return f;
        }
        return null;
    };
    /**
     * Get a field from its "slot". A "slot" is just the field's index in its
     * defining class's field array.
     */
    ReferenceClassData.prototype.getFieldFromSlot = function (slot) {
        return this.fields[slot];
    };
    /**
     * Get a method from its "slot". A "slot" is just the method's index in its
     * defining class's method array.
     */
    ReferenceClassData.prototype.getMethodFromSlot = function (slot) {
        return this.methods[slot];
    };
    /**
     * Retrieve a method with the given signature from this particular class.
     * Does not search superclasses / interfaces.
     */
    ReferenceClassData.prototype.getMethod = function (sig) {
        var m = this._methodLookup[sig];
        if (m.cls === this) {
            return m;
        }
        return null;
    };
    ReferenceClassData.prototype.getSpecificMethod = function (definingCls, sig) {
        if (this.getInternalName() === definingCls) {
            return this.getMethod(sig);
        }
        var searchClasses = this.interfaceClasses.slice(0), m;
        if (this.superClass) {
            searchClasses.push(this.superClass);
        }
        for (var i = 0; i < searchClasses.length; i++) {
            if (null !== (m = searchClasses[i].getSpecificMethod(definingCls, sig))) {
                return m;
            }
        }
        return null;
    };
    /**
     * Get the methods belonging to this particular class.
     * DO NOT MUTATE!
     */
    ReferenceClassData.prototype.getMethods = function () {
        return this.methods;
    };
    /**
     * Get the set of default methods that are invocable on this object, but were
     * not inherited in the virtual method table.
     * DO NOT MUTATE!
     */
    ReferenceClassData.prototype.getUninheritedDefaultMethods = function () {
        return this._uninheritedDefaultMethods;
    };
    ReferenceClassData.prototype.getProtectionDomain = function () {
        return this._protectionDomain;
    };
    /**
     * Resolves this class's virtual method table according to the JVM
     * specification:
     * http://docs.oracle.com/javase/specs/jvms/se8/html/jvms-5.html#jvms-5.4.3.3
     */
    ReferenceClassData.prototype._resolveMethods = function () {
        var _this = this;
        if (this.superClass !== null) {
            // Start off with my parents' method table.
            this._vmTable = this._vmTable.concat(this.superClass._vmTable);
            Object.keys(this.superClass._methodLookup).forEach(function (m) {
                _this._methodLookup[m] = _this.superClass._methodLookup[m];
            });
        }
        // My methods override my super class'.
        this.methods.forEach(function (m) {
            var superM = _this._methodLookup[m.signature];
            if (!m.accessFlags.isStatic() && m.name !== "<init>") {
                // Only non-static non-constructor methods are placed into the virtual
                // method table.
                if (superM === undefined) {
                    // New vmindex.
                    _this._vmTable.push(m);
                }
                else {
                    // Old vmindex. Inherit the super class method's vmindex.
                    _this._vmTable[_this._vmTable.indexOf(superM)] = m;
                }
            }
            _this._methodLookup[m.signature] = m;
        });
        // Root out any miranda / default / static interface methods. Only install
        // them if there are no alternatives already in the table.
        this.interfaceClasses.forEach(function (iface) {
            Object.keys(iface._methodLookup).forEach(function (ifaceMethodSig) {
                var ifaceM = iface._methodLookup[ifaceMethodSig];
                if (_this._methodLookup[ifaceMethodSig] === undefined) {
                    if (!ifaceM.accessFlags.isStatic()) {
                        // New vmindex.
                        _this._vmTable.push(ifaceM);
                    }
                    _this._methodLookup[ifaceMethodSig] = ifaceM;
                }
                else if (ifaceM.isDefault()) {
                    // Default method; uninherited, but still callable via full signature.
                    _this._uninheritedDefaultMethods.push(ifaceM);
                }
            });
        });
    };
    /**
     * Resolves all of the fields for this class according to the JVM
     * specification:
     * http://docs.oracle.com/javase/specs/jvms/se8/html/jvms-5.html#jvms-5.4.3.2
     */
    ReferenceClassData.prototype._resolveFields = function () {
        var _this = this;
        if (this.superClass !== null) {
            // Start off w/ my parent class' fields.
            this._objectFields = this._objectFields.concat(this.superClass._objectFields);
            Object.keys(this.superClass._fieldLookup).forEach(function (f) {
                _this._fieldLookup[f] = _this.superClass._fieldLookup[f];
            });
        }
        // Superinterface fields trump superclass fields.
        this.interfaceClasses.forEach(function (iface) {
            Object.keys(iface._fieldLookup).forEach(function (ifaceFieldName) {
                var ifaceF = iface._fieldLookup[ifaceFieldName];
                assert_1["default"](ifaceF.accessFlags.isStatic(), "Interface fields must be static.");
                _this._fieldLookup[ifaceFieldName] = ifaceF;
            });
        });
        // My fields override all other fields.
        this.fields.forEach(function (f) {
            _this._fieldLookup[f.name] = f;
            if (f.accessFlags.isStatic()) {
                _this._staticFields.push(f);
            }
            else {
                _this._objectFields.push(f);
            }
        });
    };
    /**
     * Looks up a method with the given signature. Returns null if no method
     * found.
     */
    ReferenceClassData.prototype.methodLookup = function (signature) {
        var m = this._methodLookup[signature];
        if (m !== undefined) {
            return m;
        }
        else {
            return null;
        }
    };
    /**
     * Performs method lookup, and includes signature polymorphic results if
     * the method is signature polymorphic.
     */
    ReferenceClassData.prototype.signaturePolymorphicAwareMethodLookup = function (signature) {
        var m;
        if (null !== (m = this.methodLookup(signature))) {
            return m;
        }
        else if (this.className === 'Ljava/lang/invoke/MethodHandle;') {
            // Check if this is a signature polymorphic method.
            // From S2.9:
            // A method is signature polymorphic if and only if all of the following conditions hold :
            // * It is declared in the java.lang.invoke.MethodHandle class.
            // * It has a single formal parameter of type Object[].
            // * It has a return type of Object.
            // * It has the ACC_VARARGS and ACC_NATIVE flags set.
            var polySig = signature.slice(0, signature.indexOf('(')) + "([Ljava/lang/Object;)Ljava/lang/Object;", m = this._methodLookup[polySig];
            if (m !== undefined && m.accessFlags.isNative() && m.accessFlags.isVarArgs() && m.cls === this) {
                return m;
            }
        }
        else if (this.superClass !== null) {
            return this.superClass.signaturePolymorphicAwareMethodLookup(signature);
        }
        return null;
    };
    /**
     * Looks up a field with the given name. Returns null if no method found.
     */
    ReferenceClassData.prototype.fieldLookup = function (name) {
        var f = this._fieldLookup[name];
        if (f !== undefined) {
            return f;
        }
        else {
            return null;
        }
    };
    ReferenceClassData.prototype.getAttribute = function (name) {
        var attrs = this.attrs;
        for (var i = 0; i < attrs.length; i++) {
            var attr = attrs[i];
            if (attr.getName() === name) {
                return attr;
            }
        }
        return null;
    };
    ReferenceClassData.prototype.getAttributes = function (name) {
        var attrs = this.attrs;
        var results = [];
        for (var i = 0; i < attrs.length; i++) {
            var attr = attrs[i];
            if (attr.getName() === name) {
                results.push(attr);
            }
        }
        return results;
    };
    /**
     * Get the bootstrap method information for an InvokeDynamic opcode.
     */
    ReferenceClassData.prototype.getBootstrapMethod = function (idx) {
        var bms = this.getAttribute('BootstrapMethods');
        return bms.bootstrapMethods[idx];
    };
    /**
     * Returns the initial value for a given static field in the class. Should
     * only be called when the constructor is created.
     */
    ReferenceClassData.prototype._getInitialStaticFieldValue = function (thread, name) {
        var f = this.fieldLookup(name);
        if (f !== null && f.accessFlags.isStatic()) {
            var cva = f.getAttribute('ConstantValue');
            if (cva !== null) {
                switch (cva.value.getType()) {
                    case enums_1.ConstantPoolItemType.STRING:
                        var stringCPI = cva.value;
                        if (stringCPI.value === null) {
                            stringCPI.value = thread.getJVM().internString(stringCPI.stringValue);
                        }
                        return stringCPI.value;
                    default:
                        // TODO: Type better.
                        return cva.value.value;
                }
            }
            else {
                return util_1.initialValue(f.rawDescriptor);
            }
        }
        assert_1["default"](false, "Tried to construct a static field value that " + (f !== null ? "isn't static" : "doesn't exist") + ": " + (f !== null ? f.cls.getInternalName() : this.getInternalName()) + " " + name);
    };
    ReferenceClassData.prototype.setResolved = function (superClazz, interfaceClazzes) {
        this.superClass = superClazz;
        trace("Class " + this.getInternalName() + " is now resolved.");
        this.interfaceClasses = interfaceClazzes;
        // TODO: Assert we are not already resolved or initialized?
        this._resolveMethods();
        this._resolveFields();
        this.setState(enums_1.ClassState.RESOLVED);
    };
    ReferenceClassData.prototype.tryToResolve = function () {
        if (this.getState() === enums_1.ClassState.LOADED) {
            // Need to grab the super class, and interfaces.
            var loader = this.loader, toResolve = this.superClassRef !== null ? this.interfaceRefs.concat(this.superClassRef) : this.interfaceRefs, allGood = true, resolvedItems = [], i, item;
            for (i = 0; i < toResolve.length; i++) {
                item = toResolve[i];
                if (item.tryResolve(loader)) {
                    resolvedItems.push(item.cls);
                }
                else {
                    return false;
                }
            }
            // It worked!
            this.setResolved(this.superClassRef !== null ? resolvedItems.pop() : null, resolvedItems);
        }
        return true;
    };
    /**
     * Attempt to synchronously initialize. This is possible if there is no
     * static initializer, and the parent classes are properly initialized.
     */
    ReferenceClassData.prototype.tryToInitialize = function () {
        if (this.getState() === enums_1.ClassState.INITIALIZED) {
            // Already initialized.
            return true;
        }
        if (this.getState() === enums_1.ClassState.RESOLVED || this.tryToResolve()) {
            // Ensure parent is initialized.
            if (this.superClass !== null && !this.superClass.tryToInitialize()) {
                // Parent failed to initialize.
                return false;
            }
            // Check if this class has a static initializer.
            var clinit = this.getMethod('<clinit>()V');
            if (clinit !== null) {
                // Nope; this class needs to do the full initialization song-and-dance.
                return false;
            }
            else {
                // No static initializer! This class is initialized!
                this.setState(enums_1.ClassState.INITIALIZED);
                return true;
            }
        }
        // This class is not resolved.
        return false;
    };
    /**
     * Returns a boolean indicating if this class is an instance of the target class.
     * "target" is a ClassData object.
     * The ClassData objects do not need to be initialized; just loaded.
     * See §2.6.7 for casting rules.
     * @todo Determine this statically to make this a constant time operation.
     */
    ReferenceClassData.prototype.isCastable = function (target) {
        if (!(target instanceof ReferenceClassData)) {
            return false;
        }
        if (this.accessFlags.isInterface()) {
            // We are both interfaces
            if (target.accessFlags.isInterface()) {
                return this.isSubinterface(target);
            }
            // Only I am an interface
            if (!target.accessFlags.isInterface()) {
                return target.getInternalName() === 'Ljava/lang/Object;';
            }
        }
        else {
            // I am a regular class, target is an interface
            if (target.accessFlags.isInterface()) {
                return this.isSubinterface(target);
            }
            // We are both regular classes
            return this.isSubclass(target);
        }
    };
    /**
     * Returns 'true' if I implement the target interface.
     */
    ReferenceClassData.prototype.isSubinterface = function (target) {
        if (this.className === target.getInternalName()) {
            return true;
        }
        var ifaces = this.getInterfaces();
        for (var i = 0; i < ifaces.length; i++) {
            var superIface = ifaces[i];
            if (superIface.isSubinterface(target)) {
                return true;
            }
        }
        if (this.getSuperClass() == null) {
            return false;
        }
        return this.getSuperClass().isSubinterface(target);
    };
    /**
     * Asynchronously *initializes* the class and its super classes.
     * Throws a Java exception on the thread if initialization fails.
     * @param thread The thread that is performing the initialization.
     * @param cb Callback to invoke when completed. Contains a reference to the
     *   class if it succeeds, or NULL if a failure occurs.
     * @param [explicit] Defaults to true. If true, this class is being
     *   *explicitly* initialized by a user. If false, the JVM is implicitly
     *   initializing the class.
     */
    ReferenceClassData.prototype.initialize = function (thread, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        if (this.isResolved()) {
            if (this.isInitialized(thread)) {
                // Nothing to do! Either resolution failed and an exception has already
                // been thrown, cdata is already initialized, or the current thread is
                // initializing the class.
                setImmediate(function () {
                    cb(_this);
                });
            }
            else if (this.initLock.tryLock(thread, cb)) {
                // Initialize the super class, and then this class.
                if (this.superClass != null) {
                    this.superClass.initialize(thread, function (cdata) {
                        if (cdata == null) {
                            // Nothing to do. Initializing the super class failed.
                            _this.initLock.unlock(null);
                        }
                        else {
                            // Initialize myself.
                            _this._initialize(thread, function (cdata) {
                                _this.initLock.unlock(cdata);
                            });
                        }
                    }, explicit);
                }
                else {
                    // java/lang/Object's parent is NULL.
                    // Continue initializing this class.
                    this._initialize(thread, function (cdata) {
                        _this.initLock.unlock(cdata);
                    });
                }
            }
        }
        else {
            // Resolve first, then initialize.
            this.resolve(thread, function (cdata) {
                if (cdata !== null) {
                    _this.initialize(thread, cb, explicit);
                }
                else {
                    // Else: An exception was thrown.
                    cb(cdata);
                }
            }, explicit);
        }
    };
    /**
     * Helper function. Initializes this class alone. Assumes super class is
     * already initialized.
     */
    ReferenceClassData.prototype._initialize = function (thread, cb) {
        var _this = this;
        var cons = this.getConstructor(thread);
        if (cons['<clinit>()V'] !== undefined) {
            debug("T" + thread.getRef() + " Running static initialization for class " + this.className + "...");
            cons['<clinit>()V'](thread, null, function (e) {
                if (e) {
                    debug("Initialization of class " + _this.className + " failed.");
                    _this.setState(enums_1.ClassState.RESOLVED);
                    /**
                     * "The class or interface initialization method must have completed
                     *  abruptly by throwing some exception E. If the class of E is not
                     *  Error or one of its subclasses, then create a new instance of the
                     *  class ExceptionInInitializerError with E as the argument, and use
                     *  this object in place of E."
                     * @url http://docs.oracle.com/javase/specs/jvms/se7/html/jvms-5.html#jvms-5.5
                     */
                    if (e.getClass().isCastable(thread.getBsCl().getResolvedClass('Ljava/lang/Error;'))) {
                        // 'e' is 'Error or one of its subclasses'.
                        thread.throwException(e);
                        cb(null);
                    }
                    else {
                        // Wrap the error.
                        thread.getBsCl().initializeClass(thread, 'Ljava/lang/ExceptionInInitializerError;', function (cdata) {
                            if (cdata == null) {
                                // Exceptional failure right here: *We failed to construct ExceptionInInitializerError*!
                                // initializeClass will throw an exception on our behalf;
                                // nothing to do.
                                cb(null);
                            }
                            else {
                                // Construct the object!
                                var eCons = cdata.getConstructor(thread), e2 = new eCons(thread);
                                // Construct the ExceptionInInitializerError!
                                e2["<init>(Ljava/lang/Throwable;)V"](thread, [e], function (e) {
                                    // Throw the newly-constructed error!
                                    thread.throwException(e2);
                                    cb(null);
                                });
                            }
                        });
                    }
                }
                else {
                    _this.setState(enums_1.ClassState.INITIALIZED);
                    debug("Initialization of class " + _this.className + " succeeded.");
                    // Normal case! Initialization succeeded.
                    cb(_this);
                }
            });
        }
        else {
            // Class doesn't have a static initializer.
            this.setState(enums_1.ClassState.INITIALIZED);
            cb(this);
        }
    };
    /**
     * A reference class can be treated as initialized in a thread if that thread
     * is in the process of initializing it.
     */
    ReferenceClassData.prototype.isInitialized = function (thread) {
        return this.getState() === enums_1.ClassState.INITIALIZED || this.initLock.getOwner() === thread;
    };
    /**
     * Resolve the class.
     */
    ReferenceClassData.prototype.resolve = function (thread, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        var toResolve = this.interfaceRefs.slice(0);
        if (this.superClassRef !== null) {
            toResolve.push(this.superClassRef);
        }
        toResolve = toResolve.filter(function (item) { return !item.isResolved(); });
        util_1.asyncForEach(toResolve, function (clsRef, nextItem) {
            clsRef.resolve(thread, _this.loader, _this, function (status) {
                if (!status) {
                    nextItem("Failed.");
                }
                else {
                    nextItem();
                }
            }, explicit);
        }, function (err) {
            if (!err) {
                _this.setResolved(_this.superClassRef !== null ? _this.superClassRef.cls : null, _this.interfaceRefs.map(function (ref) { return ref.cls; }));
                cb(_this);
            }
            else {
                cb(null);
            }
        });
    };
    /**
     * Find Miranda and default interface methods in this class. These
     * methods manifest as new vmindices in the virtual method table compared with
     * the superclass, and are not defined in this class itself.
     */
    ReferenceClassData.prototype.getMirandaAndDefaultMethods = function () {
        var _this = this;
        var superClsMethodTable = this.superClass !== null ? this.superClass.getVMTable() : [];
        return this.getVMTable().slice(superClsMethodTable.length).filter(function (method) { return method.cls !== _this; });
    };
    ReferenceClassData.prototype.outputInjectedFields = function (outputStream) {
        if (this.superClass !== null) {
            this.superClass.outputInjectedFields(outputStream);
        }
        var injected = injectedFields[this.getInternalName()];
        if (injected !== undefined) {
            Object.keys(injected).forEach(function (fieldName) {
                outputStream.write("this." + fieldName + " = " + injected[fieldName][1] + ";\n");
            });
        }
    };
    ReferenceClassData.prototype._constructConstructor = function (thread) {
        assert_1["default"](this._constructor === null, "Attempted to construct constructor twice for class " + this.getExternalName() + "!");
        var jsClassName = util_1.jvmName2JSName(this.getInternalName()), outputStream = new StringOutputStream_1["default"]();
        // Expects args: extendClass, cls, InternalStackFrame, NativeStackFrame, BytecodeStackFrame, gLongZero, ClassLoader, Monitor, thread
        outputStream.write("if (cls.superClass !== null) {\n    extendClass(" + jsClassName + ", cls.superClass.getConstructor(thread));\n  }\n  function " + jsClassName + "(thread) {\n");
        // Injected fields.
        this.outputInjectedFields(outputStream);
        // Output instance field assignments.
        this._objectFields.forEach(function (f) { return f.outputJavaScriptField(jsClassName, outputStream); });
        outputStream.write("  }\n  " + jsClassName + ".cls = cls;\n");
        // Injected methods.
        this.outputInjectedMethods(jsClassName, outputStream);
        // Static fields.
        this._staticFields.forEach(function (f) { return f.outputJavaScriptField(jsClassName, outputStream); });
        // Static and instance methods.
        this.getMethods().forEach(function (m) { return m.outputJavaScriptFunction(jsClassName, outputStream); });
        // Miranda and default interface methods.
        this.getMirandaAndDefaultMethods().forEach(function (m) { return m.outputJavaScriptFunction(jsClassName, outputStream); });
        // Uninherited default methods.
        this.getUninheritedDefaultMethods().forEach(function (m) { return m.outputJavaScriptFunction(jsClassName, outputStream, true); });
        outputStream.write("  return " + jsClassName + ";");
        var evalText = outputStream.flush();
        // NOTE: Thread will be null during system bootstrapping.
        if (!RELEASE && thread !== null && thread.getJVM().shouldDumpCompiledCode()) {
            thread.getJVM().dumpObjectDefinition(this, evalText);
        }
        var fcn = new Function("extendClass", "cls", "InternalStackFrame", "NativeStackFrame", "BytecodeStackFrame", "gLongZero", "CustomClassLoader", "Monitor", "thread", "getRef", "util", evalText);
        return fcn(extendClass, this, threading_1.InternalStackFrame, threading_1.NativeStackFrame, threading_1.BytecodeStackFrame, gLong_1["default"].ZERO, ClassLoader_1.CustomClassLoader, Monitor_1["default"], thread, getRef, util);
    };
    ReferenceClassData.prototype.getConstructor = function (thread) {
        if (this._constructor == null) {
            assert_1["default"](this.isResolved(), "Cannot construct " + this.getInternalName() + "'s constructor until it is resolved.");
            this._constructor = this._constructConstructor(thread);
        }
        return this._constructor;
    };
    return ReferenceClassData;
}(ClassData));
exports.ReferenceClassData = ReferenceClassData;
//# sourceMappingURL=data:application/json;base64,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