"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var ClassData_1 = require('./ClassData');
var ClassLock_1 = require('./ClassLock');
var classpath_1 = require('./classpath');
var enums_1 = require('./enums');
var util_1 = require('./util');
var logging = require('./logging');
var assert_1 = require('./assert');
var debug = logging.debug;
var error = logging.error;
/**
 * Used to lock classes for loading.
 */
var ClassLocks = (function () {
    function ClassLocks() {
        /**
         * typrStr => array of callbacks to trigger when operation completes.
         */
        this.locks = {};
    }
    /**
     * Checks if the lock for the given class is already taken. If not, it takes
     * the lock. If it is taken, we enqueue the callback.
     * NOTE: For convenience, will handle triggering the owner's callback as well.
     */
    ClassLocks.prototype.tryLock = function (typeStr, thread, cb) {
        if (typeof this.locks[typeStr] === 'undefined') {
            this.locks[typeStr] = new ClassLock_1["default"]();
        }
        return this.locks[typeStr].tryLock(thread, cb);
    };
    /**
     * Releases the lock on the given string.
     */
    ClassLocks.prototype.unlock = function (typeStr, cdata) {
        this.locks[typeStr].unlock(cdata);
        // No need for this lock to remain.
        delete this.locks[typeStr];
    };
    /**
     * Returns the owning thread of a given lock. Returns null if the specified
     * type string is not locked.
     */
    ClassLocks.prototype.getOwner = function (typeStr) {
        if (this.locks[typeStr]) {
            return this.locks[typeStr].getOwner();
        }
        return null;
    };
    return ClassLocks;
}());
/**
 * Base classloader class. Contains common class resolution and instantiation
 * logic.
 */
var ClassLoader = (function () {
    /**
     * @param bootstrap The JVM's bootstrap classloader. ClassLoaders use it
     *   to retrieve primitive types.
     */
    function ClassLoader(bootstrap) {
        this.bootstrap = bootstrap;
        /**
         * Stores loaded *reference* and *array* classes.
         */
        this.loadedClasses = {};
        /**
         * Stores callbacks that are waiting for another thread to finish loading
         * the specified class.
         */
        this.loadClassLocks = new ClassLocks();
    }
    /**
     * Retrieve a listing of classes that are loaded in this class loader.
     */
    ClassLoader.prototype.getLoadedClassNames = function () {
        return Object.keys(this.loadedClasses);
    };
    /**
     * Adds the specified class to the classloader. As opposed to defineClass,
     * which defines a new class from bytes with the classloader.
     *
     * What's the difference?
     * * Classes created with defineClass are defined by this classloader.
     * * Classes added with addClass may have been defined by a different
     *   classloader. This happens when a custom class loader's loadClass
     *   function proxies classloading to a different classloader.
     *
     * @param typeStr The type string of the class.
     * @param classData The class data object representing the class.
     */
    ClassLoader.prototype.addClass = function (typeStr, classData) {
        // If the class is already added, ensure it is the same class we are adding again.
        assert_1["default"](this.loadedClasses[typeStr] != null ? this.loadedClasses[typeStr] === classData : true);
        this.loadedClasses[typeStr] = classData;
    };
    /**
     * No-frills. Get the class if it's defined in the class loader, no matter
     * what shape it is in.
     *
     * Should only be used internally by ClassLoader subclasses.
     */
    ClassLoader.prototype.getClass = function (typeStr) {
        return this.loadedClasses[typeStr];
    };
    /**
     * Defines a new class with the class loader from an array of bytes.
     * @param thread The thread that is currently in control when this class is
     *   being defined. An exception may be thrown if there is an issue parsing
     *   the class file.
     * @param typeStr The type string of the class (e.g. "Ljava/lang/Object;")
     * @param data The data associated with the class as a binary blob.
     * @param protectionDomain The protection domain for the class (can be NULL).
     * @return The defined class, or null if there was an issue.
     */
    ClassLoader.prototype.defineClass = function (thread, typeStr, data, protectionDomain) {
        try {
            var classData = new ClassData_1.ReferenceClassData(data, protectionDomain, this);
            this.addClass(typeStr, classData);
            if (this instanceof BootstrapClassLoader) {
                debug("[BOOTSTRAP] Defining class " + typeStr);
            }
            else {
                debug("[CUSTOM] Defining class " + typeStr);
            }
            return classData;
        }
        catch (e) {
            if (thread === null) {
                // This will only happen when we're loading java/lang/Thread for
                // the very first time.
                error("JVM initialization failed: " + e);
                error(e.stack);
            }
            else {
                thread.throwNewException('Ljava/lang/ClassFormatError;', e);
            }
            return null;
        }
    };
    /**
     * Defines a new array class with this loader.
     */
    ClassLoader.prototype.defineArrayClass = function (typeStr) {
        assert_1["default"](this.getLoadedClass(util_1.get_component_type(typeStr)) != null);
        var arrayClass = new ClassData_1.ArrayClassData(util_1.get_component_type(typeStr), this);
        this.addClass(typeStr, arrayClass);
        return arrayClass;
    };
    /**
     * Attempts to retrieve the given loaded class.
     * @param typeStr The name of the class.
     * @return Returns the loaded class, or null if no such class is currently
     *   loaded.
     */
    ClassLoader.prototype.getLoadedClass = function (typeStr) {
        var cls = this.loadedClasses[typeStr];
        if (cls != null) {
            return cls;
        }
        else {
            if (util_1.is_primitive_type(typeStr)) {
                // Primitive classes must be fetched from the bootstrap classloader.
                return this.bootstrap.getPrimitiveClass(typeStr);
            }
            else if (util_1.is_array_type(typeStr)) {
                // We might be able to load this array class synchronously.
                // Component class must be loaded. And we must define the array class
                // with the component class's loader.
                var component = this.getLoadedClass(util_1.get_component_type(typeStr));
                if (component != null) {
                    var componentCl = component.getLoader();
                    if (componentCl === this) {
                        // We're responsible for defining the array class.
                        return this.defineArrayClass(typeStr);
                    }
                    else {
                        // Delegate to the other loader, then add the class to our loaded
                        // roster.
                        cls = componentCl.getLoadedClass(typeStr);
                        this.addClass(typeStr, cls);
                        return cls;
                    }
                }
            }
            return null;
        }
    };
    /**
     * Attempts to retrieve the given resolved class.
     * @param typeStr The name of the class.
     * @return Returns the class if it is both loaded and resolved. Returns null
     *   if this is not the case.
     */
    ClassLoader.prototype.getResolvedClass = function (typeStr) {
        var cls = this.getLoadedClass(typeStr);
        if (cls !== null) {
            if (cls.isResolved() || cls.tryToResolve()) {
                return cls;
            }
            else {
                return null;
            }
        }
        else {
            return null;
        }
    };
    /**
     * Attempts to retrieve the given initialized class.
     * @param typeStr The name of the class.
     * @return Returns the class if it is initialized. Returns null if this is
     *   not the case.
     */
    ClassLoader.prototype.getInitializedClass = function (thread, typeStr) {
        var cls = this.getLoadedClass(typeStr);
        if (cls !== null) {
            if (cls.isInitialized(thread) || cls.tryToInitialize()) {
                return cls;
            }
            else {
                return null;
            }
        }
        else {
            return cls;
        }
    };
    /**
     * Asynchronously loads the given class.
     */
    ClassLoader.prototype.loadClass = function (thread, typeStr, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        // See if we can grab this synchronously first.
        var cdata = this.getLoadedClass(typeStr);
        if (cdata) {
            setImmediate(function () {
                cb(cdata);
            });
        }
        else {
            // Check the loadClass lock for this class.
            if (this.loadClassLocks.tryLock(typeStr, thread, cb)) {
                // Async it is!
                if (util_1.is_reference_type(typeStr)) {
                    this._loadClass(thread, typeStr, function (cdata) {
                        _this.loadClassLocks.unlock(typeStr, cdata);
                    }, explicit);
                }
                else {
                    // Array
                    this.loadClass(thread, util_1.get_component_type(typeStr), function (cdata) {
                        if (cdata != null) {
                            // Synchronously will work now.
                            _this.loadClassLocks.unlock(typeStr, _this.getLoadedClass(typeStr));
                        }
                    }, explicit);
                }
            }
        }
    };
    /**
     * Convenience function: Resolve many classes. Calls cb with null should
     * an error occur.
     */
    ClassLoader.prototype.resolveClasses = function (thread, typeStrs, cb) {
        var _this = this;
        var classes = {};
        util_1.asyncForEach(typeStrs, function (typeStr, next_item) {
            _this.resolveClass(thread, typeStr, function (cdata) {
                if (cdata === null) {
                    next_item("Error resolving class: " + typeStr);
                }
                else {
                    classes[typeStr] = cdata;
                    next_item();
                }
            });
        }, function (err) {
            if (err) {
                cb(null);
            }
            else {
                cb(classes);
            }
        });
    };
    /**
     * Asynchronously *resolves* the given class by loading the class and
     * resolving its super class, interfaces, and/or component classes.
     */
    ClassLoader.prototype.resolveClass = function (thread, typeStr, cb, explicit) {
        if (explicit === void 0) { explicit = true; }
        this.loadClass(thread, typeStr, function (cdata) {
            if (cdata === null || cdata.isResolved()) {
                // Nothing to do! Either cdata is null, an exception triggered, and we
                // failed, or cdata is already resolved.
                setImmediate(function () { cb(cdata); });
            }
            else {
                cdata.resolve(thread, cb, explicit);
            }
        }, explicit);
    };
    /**
     * Asynchronously *initializes* the given class and its super classes.
     */
    ClassLoader.prototype.initializeClass = function (thread, typeStr, cb, explicit) {
        if (explicit === void 0) { explicit = true; }
        // Get the resolved class.
        this.resolveClass(thread, typeStr, function (cdata) {
            if (cdata === null || cdata.isInitialized(thread)) {
                // Nothing to do! Either resolution failed and an exception has already
                // been thrown, cdata is already initialized, or the current thread is
                // initializing the class.
                setImmediate(function () {
                    cb(cdata);
                });
            }
            else {
                assert_1["default"](util_1.is_reference_type(typeStr));
                cdata.initialize(thread, cb, explicit);
            }
        }, explicit);
    };
    /**
     * Throws the appropriate exception/error for a class not being found.
     * If loading was implicitly triggered by the JVM, we call NoClassDefFoundError.
     * If the program explicitly called loadClass, then we throw the ClassNotFoundException.
     */
    ClassLoader.prototype.throwClassNotFoundException = function (thread, typeStr, explicit) {
        thread.throwNewException(explicit ? 'Ljava/lang/ClassNotFoundException;' : 'Ljava/lang/NoClassDefFoundError;', "Cannot load class: " + util_1.ext_classname(typeStr));
    };
    return ClassLoader;
}());
exports.ClassLoader = ClassLoader;
/**
 * The JVM's bootstrap class loader. Loads classes directly from files on the
 * file system.
 */
var BootstrapClassLoader = (function (_super) {
    __extends(BootstrapClassLoader, _super);
    /**
     * Constructs the bootstrap classloader with the given classpath.
     * @param classPath The classpath, where the *first* item is the *last*
     *   classpath searched. Meaning, the classPath[0] should be the bootstrap
     *   class path.
     * @param extractionPath The path where jar files should be extracted.
     * @param cb Called once all of the classpath items have been checked.
     *   Passes an error if one occurs.
     */
    function BootstrapClassLoader(javaHome, classpath, cb) {
        var _this = this;
        // The correct way to do this would be super(this), but we cannot reference this before calling super()
        _super.call(this, null);
        this.bootstrap = this;
        this.classpath = null;
        this.loadedPackages = {};
        classpath_1.ClasspathFactory(javaHome, classpath, function (items) {
            _this.classpath = items.reverse();
            cb();
        });
    }
    /**
     * Registers that a given class has successfully been loaded from the specified
     * classpath item.
     */
    BootstrapClassLoader.prototype._registerLoadedClass = function (clsType, cpItem) {
        var pkgName = clsType.slice(0, clsType.lastIndexOf('/')), itemLoader = this.loadedPackages[pkgName];
        if (!itemLoader) {
            this.loadedPackages[pkgName] = [cpItem];
        }
        else if (itemLoader[0] !== cpItem && itemLoader.indexOf(cpItem) === -1) {
            // Common case optimization: Simply check the first array element.
            itemLoader.push(cpItem);
        }
    };
    /**
     * Returns a listing of tuples containing:
     * * The package name (e.g. java/lang)
     * * Classpath locations where classes in the package were loaded.
     */
    BootstrapClassLoader.prototype.getPackages = function () {
        var _this = this;
        return Object.keys(this.loadedPackages).map(function (pkgName) {
            return [pkgName, _this.loadedPackages[pkgName].map(function (item) { return item.getPath(); })];
        });
    };
    /**
     * Retrieves or defines the specified primitive class.
     */
    BootstrapClassLoader.prototype.getPrimitiveClass = function (typeStr) {
        var cdata = this.getClass(typeStr);
        if (cdata == null) {
            cdata = new ClassData_1.PrimitiveClassData(typeStr, this);
            this.addClass(typeStr, cdata);
        }
        return cdata;
    };
    /**
     * Asynchronously load the given class from the classpath.
     *
     * SHOULD ONLY BE INVOKED INTERNALLY BY THE CLASSLOADER.
     */
    BootstrapClassLoader.prototype._loadClass = function (thread, typeStr, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        debug("[BOOTSTRAP] Loading class " + typeStr);
        // This method is only valid for reference types!
        assert_1["default"](util_1.is_reference_type(typeStr));
        // Search the class path for the class.
        var clsFilePath = util_1.descriptor2typestr(typeStr), cPathLen = this.classpath.length, toSearch = [], clsData;
        searchLoop: for (var i = 0; i < cPathLen; i++) {
            var item = this.classpath[i];
            switch (item.hasClass(clsFilePath)) {
                case enums_1.TriState.INDETERMINATE:
                    toSearch.push(item);
                    break;
                case enums_1.TriState.TRUE:
                    // Break out of the loop; TRUE paths are guaranteed to have the class.
                    toSearch.push(item);
                    break searchLoop;
            }
        }
        util_1.asyncFind(toSearch, function (pItem, callback) {
            pItem.loadClass(clsFilePath, function (err, data) {
                if (err) {
                    callback(false);
                }
                else {
                    clsData = data;
                    callback(true);
                }
            });
        }, function (pItem) {
            if (pItem) {
                var cls = _this.defineClass(thread, typeStr, clsData, null);
                if (cls !== null) {
                    _this._registerLoadedClass(clsFilePath, pItem);
                }
                cb(cls);
            }
            else {
                // No such class.
                debug("Could not find class " + typeStr);
                _this.throwClassNotFoundException(thread, typeStr, explicit);
                cb(null);
            }
        });
    };
    /**
     * Returns a listing of reference classes loaded in the bootstrap loader.
     */
    BootstrapClassLoader.prototype.getLoadedClassFiles = function () {
        var loadedClasses = this.getLoadedClassNames();
        return loadedClasses.filter(function (clsName) { return util_1.is_reference_type(clsName); });
    };
    /**
     * Returns the JVM object corresponding to this ClassLoader.
     * @todo Represent the bootstrap by something other than 'null'.
     * @todo These should be one-in-the-same.
     */
    BootstrapClassLoader.prototype.getLoaderObject = function () {
        return null;
    };
    /**
     * Returns the current classpath.
     */
    BootstrapClassLoader.prototype.getClassPath = function () {
        var cpLen = this.classpath.length, cpStrings = new Array(cpLen);
        for (var i = 0; i < cpLen; i++) {
            // Reverse it so it is the expected order (last item is first search target)
            cpStrings[i] = this.classpath[cpLen - i - 1].getPath();
        }
        return cpStrings;
    };
    /**
     * Returns the classpath item objects in the classpath.
     */
    BootstrapClassLoader.prototype.getClassPathItems = function () {
        return this.classpath.slice(0);
    };
    return BootstrapClassLoader;
}(ClassLoader));
exports.BootstrapClassLoader = BootstrapClassLoader;
/**
 * A Custom ClassLoader. Loads classes by calling loadClass on the user-defined
 * loader.
 */
var CustomClassLoader = (function (_super) {
    __extends(CustomClassLoader, _super);
    function CustomClassLoader(bootstrap, loaderObj) {
        _super.call(this, bootstrap);
        this.loaderObj = loaderObj;
    }
    /**
     * Asynchronously load the given class from the classpath. Calls the
     * classloader's loadClass method.
     *
     * SHOULD ONLY BE INVOKED BY THE CLASS LOADER.
     *
     * @param thread The thread that triggered the loading.
     * @param typeStr The type string of the class.
     * @param cb The callback that will be called with the loaded class. It will
     *   be passed a null if there is an error -- which also indicates that it
     *   threw an exception on the JVM thread.
     * @param explicit 'True' if loadClass was explicitly invoked by the program,
     *   false otherwise. This changes the exception/error that we throw.
     */
    CustomClassLoader.prototype._loadClass = function (thread, typeStr, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        debug("[CUSTOM] Loading class " + typeStr);
        // This method is only valid for reference types!
        assert_1["default"](util_1.is_reference_type(typeStr));
        // Invoke the custom class loader.
        this.loaderObj['loadClass(Ljava/lang/String;)Ljava/lang/Class;'](thread, [util_1.initString(this.bootstrap, util_1.ext_classname(typeStr))], function (e, jco) {
            if (e) {
                // Exception! There was an issue defining the class.
                _this.throwClassNotFoundException(thread, typeStr, explicit);
                cb(null);
            }
            else {
                // Add the class returned by loadClass, in case the classloader
                // proxied loading to another classloader.
                var cls = jco.$cls;
                _this.addClass(typeStr, cls);
                cb(cls);
            }
        });
    };
    /**
     * Returns the JVM object corresponding to this ClassLoader.
     * @todo These should be one-in-the-same.
     */
    CustomClassLoader.prototype.getLoaderObject = function () {
        return this.loaderObj;
    };
    return CustomClassLoader;
}(ClassLoader));
exports.CustomClassLoader = CustomClassLoader;
//# sourceMappingURL=data:application/json;base64,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