"use strict";
var util = require('./util');
var enums_1 = require('./enums');
var assert_1 = require('./assert');
/**
 * Stores all of the constant pool classes, keyed on their enum value.
 */
var CP_CLASSES = {};
// #region Tier 0
/**
 * Represents a constant UTF-8 string.
 * ```
 * CONSTANT_Utf8_info {
 *   u1 tag;
 *   u2 length;
 *   u1 bytes[length];
 * }
 * ```
 */
var ConstUTF8 = (function () {
    function ConstUTF8(rawBytes) {
        this.value = this.bytes2str(rawBytes);
    }
    /**
     * Parse Java's pseudo-UTF-8 strings into valid UTF-16 codepoints (spec 4.4.7)
     * Note that Java uses UTF-16 internally by default for string representation,
     * and the pseudo-UTF-8 strings are *only* used for serialization purposes.
     * Thus, there is no reason for other parts of the code to call this routine!
     * TODO: To avoid copying, create a character array for this data.
     * http://docs.oracle.com/javase/specs/jvms/se8/html/jvms-4.html#jvms-4.4.7
     */
    ConstUTF8.prototype.bytes2str = function (bytes) {
        var y, z, v, w, x, charCode, idx = 0, rv = '';
        while (idx < bytes.length) {
            x = bytes[idx++] & 0xff;
            // While the standard specifies that surrogate pairs should be handled, it seems like
            // they are by default with the three byte format. See parsing code here:
            // http://hg.openjdk.java.net/jdk8u/jdk8u-dev/jdk/file/3623f1b29b58/src/share/classes/java/io/DataInputStream.java#l618
            // One UTF-16 character.
            if (x <= 0x7f) {
                // One character, one byte.
                charCode = x;
            }
            else if (x <= 0xdf) {
                // One character, two bytes.
                y = bytes[idx++];
                charCode = ((x & 0x1f) << 6) + (y & 0x3f);
            }
            else {
                // One character, three bytes.
                y = bytes[idx++];
                z = bytes[idx++];
                charCode = ((x & 0xf) << 12) + ((y & 0x3f) << 6) + (z & 0x3f);
            }
            rv += String.fromCharCode(charCode);
        }
        return rv;
    };
    ConstUTF8.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.UTF8;
    };
    ConstUTF8.prototype.getConstant = function (thread) { return this.value; };
    ConstUTF8.prototype.isResolved = function () { return true; };
    ConstUTF8.fromBytes = function (byteStream, constantPool) {
        var strlen = byteStream.getUint16();
        return new this(byteStream.read(strlen));
    };
    ConstUTF8.size = 1;
    // Variable-size.
    ConstUTF8.infoByteSize = 0;
    return ConstUTF8;
}());
exports.ConstUTF8 = ConstUTF8;
CP_CLASSES[enums_1.ConstantPoolItemType.UTF8] = ConstUTF8;
/**
 * Represents a constant 32-bit integer.
 * ```
 * CONSTANT_Integer_info {
 *   u1 tag;
 *   u4 bytes;
 * }
 * ```
 */
var ConstInt32 = (function () {
    function ConstInt32(value) {
        this.value = value;
    }
    ConstInt32.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.INTEGER;
    };
    ConstInt32.prototype.getConstant = function (thread) { return this.value; };
    ConstInt32.prototype.isResolved = function () { return true; };
    ConstInt32.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getInt32());
    };
    ConstInt32.size = 1;
    ConstInt32.infoByteSize = 4;
    return ConstInt32;
}());
exports.ConstInt32 = ConstInt32;
CP_CLASSES[enums_1.ConstantPoolItemType.INTEGER] = ConstInt32;
/**
 * Represents a constant 32-bit floating point number.
 * ```
 * CONSTANT_Float_info {
 *   u1 tag;
 *   u4 bytes;
 * }
 * ```
 */
var ConstFloat = (function () {
    function ConstFloat(value) {
        this.value = value;
    }
    ConstFloat.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.FLOAT;
    };
    ConstFloat.prototype.getConstant = function (thread) { return this.value; };
    ConstFloat.prototype.isResolved = function () { return true; };
    ConstFloat.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getFloat());
    };
    ConstFloat.size = 1;
    ConstFloat.infoByteSize = 4;
    return ConstFloat;
}());
exports.ConstFloat = ConstFloat;
CP_CLASSES[enums_1.ConstantPoolItemType.FLOAT] = ConstFloat;
/**
 * Represents a constant 64-bit integer.
 * ```
 * CONSTANT_Long_info {
 *   u1 tag;
 *   u4 high_bytes;
 *   u4 low_bytes;
 * }
 * ```
 */
var ConstLong = (function () {
    function ConstLong(value) {
        this.value = value;
    }
    ConstLong.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.LONG;
    };
    ConstLong.prototype.getConstant = function (thread) { return this.value; };
    ConstLong.prototype.isResolved = function () { return true; };
    ConstLong.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getInt64());
    };
    ConstLong.size = 2;
    ConstLong.infoByteSize = 8;
    return ConstLong;
}());
exports.ConstLong = ConstLong;
CP_CLASSES[enums_1.ConstantPoolItemType.LONG] = ConstLong;
/**
 * Represents a constant 64-bit floating point number.
 * ```
 * CONSTANT_Double_info {
 *   u1 tag;
 *   u4 high_bytes;
 *   u4 low_bytes;
 * }
 * ```
 */
var ConstDouble = (function () {
    function ConstDouble(value) {
        this.value = value;
    }
    ConstDouble.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.DOUBLE;
    };
    ConstDouble.prototype.getConstant = function (thread) { return this.value; };
    ConstDouble.prototype.isResolved = function () { return true; };
    ConstDouble.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getDouble());
    };
    ConstDouble.size = 2;
    ConstDouble.infoByteSize = 8;
    return ConstDouble;
}());
exports.ConstDouble = ConstDouble;
CP_CLASSES[enums_1.ConstantPoolItemType.DOUBLE] = ConstDouble;
// #endregion
// #region Tier 1
/**
 * Represents a class or interface.
 * ```
 * CONSTANT_Class_info {
 *   u1 tag;
 *   u2 name_index;
 * }
 * ```
 * @todo Have a classloader-local cache of class reference objects.
 */
var ClassReference = (function () {
    function ClassReference(name) {
        /**
         * The resolved class reference.
         */
        this.cls = null;
        /**
         * The JavaScript constructor for the referenced class.
         */
        this.clsConstructor = null;
        /**
         * The array class for the resolved class reference.
         */
        this.arrayClass = null;
        /**
         * The JavaScript constructor for the array class.
         */
        this.arrayClassConstructor = null;
        this.name = name;
    }
    /**
     * Attempt to synchronously resolve.
     */
    ClassReference.prototype.tryResolve = function (loader) {
        if (this.cls === null) {
            this.cls = loader.getResolvedClass(this.name);
        }
        return this.cls !== null;
    };
    /**
     * Resolves the class reference by resolving the class. Does not run
     * class initialization.
     */
    ClassReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        // Because of Java 8 anonymous classes, THIS CHECK IS REQUIRED FOR CORRECTNESS.
        // (ClassLoaders do not know about anonymous classes, hence they are
        //  'anonymous')
        // (Anonymous classes are an 'Unsafe' feature, and are not part of the standard,
        //  but they are employed for lambdas and such.)
        // NOTE: Thread is 'null' during JVM bootstrapping.
        if (thread !== null) {
            var currentMethod = thread.currentMethod();
            // The stack might be empty during resolution, which occurs during JVM bootup.
            if (currentMethod !== null && this.name === currentMethod.cls.getInternalName()) {
                this.setResolved(thread, thread.currentMethod().cls);
                return cb(true);
            }
        }
        loader.resolveClass(thread, this.name, function (cdata) {
            _this.setResolved(thread, cdata);
            cb(cdata !== null);
        }, explicit);
    };
    ClassReference.prototype.setResolved = function (thread, cls) {
        this.cls = cls;
        if (cls !== null) {
            this.clsConstructor = cls.getConstructor(thread);
        }
    };
    ClassReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.CLASS;
    };
    ClassReference.prototype.getConstant = function (thread) { return this.cls.getClassObject(thread); };
    ClassReference.prototype.isResolved = function () { return this.cls !== null; };
    ClassReference.fromBytes = function (byteStream, constantPool) {
        var nameIndex = byteStream.getUint16(), cpItem = constantPool.get(nameIndex);
        assert_1["default"](cpItem.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool ClassReference type != UTF8');
        // The ConstantPool stores class names without the L...; descriptor stuff
        return new this(util.typestr2descriptor(cpItem.value));
    };
    ClassReference.size = 1;
    ClassReference.infoByteSize = 2;
    return ClassReference;
}());
exports.ClassReference = ClassReference;
CP_CLASSES[enums_1.ConstantPoolItemType.CLASS] = ClassReference;
/**
 * Represents a field or method without indicating which class or interface
 * type it belongs to.
 * ```
 * CONSTANT_NameAndType_info {
 *   u1 tag;
 *   u2 name_index;
 *   u2 descriptor_index;
 * }
 * ```
 */
var NameAndTypeInfo = (function () {
    function NameAndTypeInfo(name, descriptor) {
        this.name = name;
        this.descriptor = descriptor;
    }
    NameAndTypeInfo.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.NAME_AND_TYPE;
    };
    NameAndTypeInfo.prototype.isResolved = function () { return true; };
    NameAndTypeInfo.fromBytes = function (byteStream, constantPool) {
        var nameIndex = byteStream.getUint16(), descriptorIndex = byteStream.getUint16(), nameConst = constantPool.get(nameIndex), descriptorConst = constantPool.get(descriptorIndex);
        assert_1["default"](nameConst.getType() === enums_1.ConstantPoolItemType.UTF8 &&
            descriptorConst.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool NameAndTypeInfo types != UTF8');
        return new this(nameConst.value, descriptorConst.value);
    };
    NameAndTypeInfo.size = 1;
    NameAndTypeInfo.infoByteSize = 4;
    return NameAndTypeInfo;
}());
exports.NameAndTypeInfo = NameAndTypeInfo;
CP_CLASSES[enums_1.ConstantPoolItemType.NAME_AND_TYPE] = NameAndTypeInfo;
/**
 * Represents constant objects of the type java.lang.String.
 * ```
 * CONSTANT_String_info {
 *   u1 tag;
 *   u2 string_index;
 * }
 * ```
 */
var ConstString = (function () {
    function ConstString(stringValue) {
        this.value = null;
        this.stringValue = stringValue;
    }
    ConstString.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.STRING;
    };
    ConstString.prototype.resolve = function (thread, loader, caller, cb) {
        this.value = thread.getJVM().internString(this.stringValue);
        setImmediate(function () { return cb(true); });
    };
    ConstString.prototype.getConstant = function (thread) { return this.value; };
    ConstString.prototype.isResolved = function () { return this.value !== null; };
    ConstString.fromBytes = function (byteStream, constantPool) {
        var stringIndex = byteStream.getUint16(), utf8Info = constantPool.get(stringIndex);
        assert_1["default"](utf8Info.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool ConstString type != UTF8');
        return new this(utf8Info.value);
    };
    ConstString.size = 1;
    ConstString.infoByteSize = 2;
    return ConstString;
}());
exports.ConstString = ConstString;
CP_CLASSES[enums_1.ConstantPoolItemType.STRING] = ConstString;
/**
 * Represents a given method type.
 * ```
 * CONSTANT_MethodType_info {
 *   u1 tag;
 *   u2 descriptor_index;
 * }
 * ```
 */
var MethodType = (function () {
    function MethodType(descriptor) {
        this.methodType = null;
        this.descriptor = descriptor;
    }
    MethodType.prototype.resolve = function (thread, cl, caller, cb) {
        var _this = this;
        util.createMethodType(thread, cl, this.descriptor, function (e, type) {
            if (e) {
                thread.throwException(e);
                cb(false);
            }
            else {
                _this.methodType = type;
                cb(true);
            }
        });
    };
    MethodType.prototype.getConstant = function (thread) { return this.methodType; };
    MethodType.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.METHOD_TYPE;
    };
    MethodType.prototype.isResolved = function () { return this.methodType !== null; };
    MethodType.fromBytes = function (byteStream, constantPool) {
        var descriptorIndex = byteStream.getUint16(), utf8Info = constantPool.get(descriptorIndex);
        assert_1["default"](utf8Info.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool MethodType type != UTF8');
        return new this(utf8Info.value);
    };
    MethodType.size = 1;
    MethodType.infoByteSize = 2;
    return MethodType;
}());
exports.MethodType = MethodType;
CP_CLASSES[enums_1.ConstantPoolItemType.METHOD_TYPE] = MethodType;
// #endregion
// #region Tier 2
/**
 * Represents a particular method.
 * ```
 * CONSTANT_Methodref_info {
 *   u1 tag;
 *   u2 class_index;
 *   u2 name_and_type_index;
 * }
 * ```
 */
var MethodReference = (function () {
    function MethodReference(classInfo, nameAndTypeInfo) {
        this.method = null;
        /**
         * The signature of the method, including the owning class.
         * e.g. bar/Baz/foo(IJ)V
         */
        this.fullSignature = null;
        this.paramWordSize = -1;
        /**
         * Contains a reference to the MemberName object for the method that invokes
         * the desired function.
         */
        this.memberName = null;
        /**
         * Contains an object that needs to be pushed onto the stack before invoking
         * memberName.
         */
        this.appendix = null;
        /**
         * The JavaScript constructor for the class that the method belongs to.
         */
        this.jsConstructor = null;
        this.classInfo = classInfo;
        this.nameAndTypeInfo = nameAndTypeInfo;
        this.signature = this.nameAndTypeInfo.name + this.nameAndTypeInfo.descriptor;
    }
    MethodReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.METHODREF;
    };
    /**
     * Checks the method referenced by this constant pool item in the specified
     * bytecode context.
     * Returns null if an error occurs.
     * - Throws a NoSuchFieldError if missing.
     * - Throws an IllegalAccessError if field is inaccessible.
     * - Throws an IncompatibleClassChangeError if the field is an incorrect type
     *   for the field access.
     */
    MethodReference.prototype.hasAccess = function (thread, frame, isStatic) {
        var method = this.method, accessingCls = frame.method.cls;
        if (method.accessFlags.isStatic() !== isStatic) {
            thread.throwNewException('Ljava/lang/IncompatibleClassChangeError;', "Method " + method.name + " from class " + method.cls.getExternalName() + " is " + (isStatic ? 'not ' : '') + "static.");
            frame.returnToThreadLoop = true;
            return false;
        }
        else if (!util.checkAccess(accessingCls, method.cls, method.accessFlags)) {
            thread.throwNewException('Ljava/lang/IllegalAccessError;', accessingCls.getExternalName() + " cannot access " + method.cls.getExternalName() + "." + method.name);
            frame.returnToThreadLoop = true;
            return false;
        }
        return true;
    };
    MethodReference.prototype.resolveMemberName = function (method, thread, cl, caller, cb) {
        var _this = this;
        var memberHandleNatives = thread.getBsCl().getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;').getConstructor(thread), appendix = new (thread.getBsCl().getInitializedClass(thread, '[Ljava/lang/Object;').getConstructor(thread))(thread, 1);
        util.createMethodType(thread, cl, this.nameAndTypeInfo.descriptor, function (e, type) {
            if (e) {
                thread.throwException(e);
                cb(false);
            }
            else {
                /* MemberName linkMethod( int refKind, Class<?> defc,
                   String name, Object type,
                   Object[] appendixResult) */
                memberHandleNatives['java/lang/invoke/MethodHandleNatives/linkMethod(Ljava/lang/Class;ILjava/lang/Class;Ljava/lang/String;Ljava/lang/Object;[Ljava/lang/Object;)Ljava/lang/invoke/MemberName;'](thread, 
                // Class callerClass
                [caller.getClassObject(thread),
                    // int refKind
                    enums_1.MethodHandleReferenceKind.INVOKEVIRTUAL,
                    // Class defc
                    _this.classInfo.cls.getClassObject(thread),
                    // String name
                    thread.getJVM().internString(_this.nameAndTypeInfo.name),
                    // Object type, Object[] appendixResult
                    type, appendix], function (e, rv) {
                    if (e !== null) {
                        thread.throwException(e);
                        cb(false);
                    }
                    else {
                        _this.appendix = appendix.array[0];
                        _this.memberName = rv;
                        cb(true);
                    }
                });
            }
        });
    };
    MethodReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        if (!this.classInfo.isResolved()) {
            this.classInfo.resolve(thread, loader, caller, function (status) {
                if (!status) {
                    cb(false);
                }
                else {
                    _this.resolve(thread, loader, caller, cb, explicit);
                }
            }, explicit);
        }
        else {
            var cls = this.classInfo.cls, method = cls.methodLookup(this.signature);
            if (method === null) {
                if (util.is_reference_type(cls.getInternalName())) {
                    // Signature polymorphic lookup.
                    method = cls.signaturePolymorphicAwareMethodLookup(this.signature);
                    if (method !== null && (method.name === 'invoke' || method.name === 'invokeExact')) {
                        // In order to completely resolve the signature polymorphic function,
                        // we need to resolve its MemberName object and Appendix.
                        return this.resolveMemberName(method, thread, loader, caller, function (status) {
                            if (status === true) {
                                _this.setResolved(thread, method);
                            }
                            else {
                                thread.throwNewException('Ljava/lang/NoSuchMethodError;', "Method " + _this.signature + " does not exist in class " + _this.classInfo.cls.getExternalName() + ".");
                            }
                            cb(status);
                        });
                    }
                }
            }
            if (method !== null) {
                this.setResolved(thread, method);
                cb(true);
            }
            else {
                thread.throwNewException('Ljava/lang/NoSuchMethodError;', "Method " + this.signature + " does not exist in class " + this.classInfo.cls.getExternalName() + ".");
                cb(false);
            }
        }
    };
    MethodReference.prototype.setResolved = function (thread, method) {
        this.method = method;
        this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        this.fullSignature = this.method.fullSignature;
        this.jsConstructor = this.method.cls.getConstructor(thread);
    };
    MethodReference.prototype.isResolved = function () { return this.method !== null; };
    MethodReference.prototype.getParamWordSize = function () {
        if (this.paramWordSize === -1) {
            this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        }
        return this.paramWordSize;
    };
    MethodReference.fromBytes = function (byteStream, constantPool) {
        var classIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), classInfo = constantPool.get(classIndex), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1["default"](classInfo.getType() === enums_1.ConstantPoolItemType.CLASS &&
            nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool MethodReference types mismatch');
        return new this(classInfo, nameAndTypeInfo);
    };
    MethodReference.size = 1;
    MethodReference.infoByteSize = 4;
    return MethodReference;
}());
exports.MethodReference = MethodReference;
CP_CLASSES[enums_1.ConstantPoolItemType.METHODREF] = MethodReference;
/**
 * Represents a particular interface method.
 * ```
 * CONSTANT_InterfaceMethodref_info {
 *   u1 tag;
 *   u2 class_index;
 *   u2 name_and_type_index;
 * }
 * ```
 */
var InterfaceMethodReference = (function () {
    function InterfaceMethodReference(classInfo, nameAndTypeInfo) {
        /**
         * The signature of the method, including the owning class.
         * e.g. bar/Baz/foo(IJ)V
         */
        this.fullSignature = null;
        this.method = null;
        this.paramWordSize = -1;
        this.jsConstructor = null;
        this.classInfo = classInfo;
        this.nameAndTypeInfo = nameAndTypeInfo;
        this.signature = this.nameAndTypeInfo.name + this.nameAndTypeInfo.descriptor;
    }
    InterfaceMethodReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.INTERFACE_METHODREF;
    };
    /**
     * Checks the method referenced by this constant pool item in the specified
     * bytecode context.
     * Returns null if an error occurs.
     * - Throws a NoSuchFieldError if missing.
     * - Throws an IllegalAccessError if field is inaccessible.
     * - Throws an IncompatibleClassChangeError if the field is an incorrect type
     *   for the field access.
     */
    InterfaceMethodReference.prototype.hasAccess = function (thread, frame, isStatic) {
        var method = this.method, accessingCls = frame.method.cls;
        if (method.accessFlags.isStatic() !== isStatic) {
            thread.throwNewException('Ljava/lang/IncompatibleClassChangeError;', "Method " + method.name + " from class " + method.cls.getExternalName() + " is " + (isStatic ? 'not ' : '') + "static.");
            frame.returnToThreadLoop = true;
            return false;
        }
        else if (!util.checkAccess(accessingCls, method.cls, method.accessFlags)) {
            thread.throwNewException('Ljava/lang/IllegalAccessError;', accessingCls.getExternalName() + " cannot access " + method.cls.getExternalName() + "." + method.name);
            frame.returnToThreadLoop = true;
            return false;
        }
        return true;
    };
    InterfaceMethodReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        if (!this.classInfo.isResolved()) {
            this.classInfo.resolve(thread, loader, caller, function (status) {
                if (!status) {
                    cb(false);
                }
                else {
                    _this.resolve(thread, loader, caller, cb, explicit);
                }
            }, explicit);
        }
        else {
            var cls = this.classInfo.cls, method = cls.methodLookup(this.signature);
            this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
            if (method !== null) {
                this.setResolved(thread, method);
                cb(true);
            }
            else {
                thread.throwNewException('Ljava/lang/NoSuchMethodError;', "Method " + this.signature + " does not exist in class " + this.classInfo.cls.getExternalName() + ".");
                cb(false);
            }
        }
    };
    InterfaceMethodReference.prototype.setResolved = function (thread, method) {
        this.method = method;
        this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        this.fullSignature = this.method.fullSignature;
        this.jsConstructor = this.method.cls.getConstructor(thread);
    };
    InterfaceMethodReference.prototype.getParamWordSize = function () {
        if (this.paramWordSize === -1) {
            this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        }
        return this.paramWordSize;
    };
    InterfaceMethodReference.prototype.isResolved = function () { return this.method !== null; };
    InterfaceMethodReference.fromBytes = function (byteStream, constantPool) {
        var classIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), classInfo = constantPool.get(classIndex), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1["default"](classInfo.getType() === enums_1.ConstantPoolItemType.CLASS &&
            nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool InterfaceMethodReference types mismatch');
        return new this(classInfo, nameAndTypeInfo);
    };
    InterfaceMethodReference.size = 1;
    InterfaceMethodReference.infoByteSize = 4;
    return InterfaceMethodReference;
}());
exports.InterfaceMethodReference = InterfaceMethodReference;
CP_CLASSES[enums_1.ConstantPoolItemType.INTERFACE_METHODREF] = InterfaceMethodReference;
/**
 * Represents a particular field.
 * ```
 * CONSTANT_Fieldref_info {
 *   u1 tag;
 *   u2 class_index;
 *   u2 name_and_type_index;
 * }
 * ```
 */
var FieldReference = (function () {
    function FieldReference(classInfo, nameAndTypeInfo) {
        this.field = null;
        /**
         * The full name of the field, including the owning class.
         * e.g. java/lang/String/value
         */
        this.fullFieldName = null;
        /**
         * The constructor for the field owner. Used for static fields.
         */
        this.fieldOwnerConstructor = null;
        this.classInfo = classInfo;
        this.nameAndTypeInfo = nameAndTypeInfo;
    }
    FieldReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.FIELDREF;
    };
    /**
     * Checks the field referenced by this constant pool item in the specified
     * bytecode context.
     * Returns null if an error occurs.
     * - Throws a NoSuchFieldError if missing.
     * - Throws an IllegalAccessError if field is inaccessible.
     * - Throws an IncompatibleClassChangeError if the field is an incorrect type
     *   for the field access.
     */
    FieldReference.prototype.hasAccess = function (thread, frame, isStatic) {
        var field = this.field, accessingCls = frame.method.cls;
        if (field.accessFlags.isStatic() !== isStatic) {
            thread.throwNewException('Ljava/lang/IncompatibleClassChangeError;', "Field " + name + " from class " + field.cls.getExternalName() + " is " + (isStatic ? 'not ' : '') + "static.");
            frame.returnToThreadLoop = true;
            return false;
        }
        else if (!util.checkAccess(accessingCls, field.cls, field.accessFlags)) {
            thread.throwNewException('Ljava/lang/IllegalAccessError;', accessingCls.getExternalName() + " cannot access " + field.cls.getExternalName() + "." + name);
            frame.returnToThreadLoop = true;
            return false;
        }
        return true;
    };
    FieldReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        if (!this.classInfo.isResolved()) {
            this.classInfo.resolve(thread, loader, caller, function (status) {
                if (!status) {
                    cb(false);
                }
                else {
                    _this.resolve(thread, loader, caller, cb, explicit);
                }
            }, explicit);
        }
        else {
            var cls = this.classInfo.cls, field = cls.fieldLookup(this.nameAndTypeInfo.name);
            if (field !== null) {
                this.fullFieldName = util.descriptor2typestr(field.cls.getInternalName()) + "/" + field.name;
                this.field = field;
                cb(true);
            }
            else {
                thread.throwNewException('Ljava/lang/NoSuchFieldError;', "Field " + this.nameAndTypeInfo.name + " does not exist in class " + this.classInfo.cls.getExternalName() + ".");
                cb(false);
            }
        }
    };
    FieldReference.prototype.isResolved = function () { return this.field !== null; };
    FieldReference.fromBytes = function (byteStream, constantPool) {
        var classIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), classInfo = constantPool.get(classIndex), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1["default"](classInfo.getType() === enums_1.ConstantPoolItemType.CLASS &&
            nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool FieldReference types mismatch');
        return new this(classInfo, nameAndTypeInfo);
    };
    FieldReference.size = 1;
    FieldReference.infoByteSize = 4;
    return FieldReference;
}());
exports.FieldReference = FieldReference;
CP_CLASSES[enums_1.ConstantPoolItemType.FIELDREF] = FieldReference;
/**
 * Used by an invokedynamic instruction to specify a bootstrap method,
 * the dynamic invocation name, the argument and return types of the call,
 * and optionally, a sequence of additional constants called static arguments
 * to the bootstrap method.
 * ```
 * CONSTANT_InvokeDynamic_info {
 *   u1 tag;
 *   u2 bootstrap_method_attr_index;
 *   u2 name_and_type_index;
 * }
 * ```
 */
var InvokeDynamic = (function () {
    function InvokeDynamic(bootstrapMethodAttrIndex, nameAndTypeInfo) {
        /**
         * Once a CallSite is defined for a particular lexical occurrence of
         * InvokeDynamic, the CallSite will be reused for each future execution
         * of that particular occurrence.
         *
         * We store the CallSite objects here for future retrieval, along with an
         * optional 'appendix' argument.
         */
        this.callSiteObjects = {};
        /**
         * A MethodType object corresponding to this InvokeDynamic call's
         * method descriptor.
         */
        this.methodType = null;
        this.bootstrapMethodAttrIndex = bootstrapMethodAttrIndex;
        this.nameAndTypeInfo = nameAndTypeInfo;
        this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
    }
    InvokeDynamic.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.INVOKE_DYNAMIC;
    };
    InvokeDynamic.prototype.isResolved = function () { return this.methodType !== null; };
    InvokeDynamic.prototype.resolve = function (thread, loader, caller, cb) {
        var _this = this;
        util.createMethodType(thread, loader, this.nameAndTypeInfo.descriptor, function (e, rv) {
            if (e) {
                thread.throwException(e);
                cb(false);
            }
            else {
                _this.methodType = rv;
                cb(true);
            }
        });
    };
    InvokeDynamic.prototype.getCallSiteObject = function (pc) {
        var cso = this.callSiteObjects[pc];
        if (cso) {
            return cso;
        }
        else {
            return null;
        }
    };
    InvokeDynamic.prototype.constructCallSiteObject = function (thread, cl, clazz, pc, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        /**
         * A call site specifier gives a symbolic reference to a method handle which
         * is to serve as the bootstrap method for a dynamic call site (§4.7.23).
         * The method handle is resolved to obtain a reference to an instance of
         * java.lang.invoke.MethodHandle (§5.4.3.5).
         */
        var bootstrapMethod = clazz.getBootstrapMethod(this.bootstrapMethodAttrIndex), unresolvedItems = bootstrapMethod[1].concat(bootstrapMethod[0], this).filter(function (item) { return !item.isResolved(); });
        if (unresolvedItems.length > 0) {
            // Resolve all needed constant pool items (including this one).
            return util.asyncForEach(unresolvedItems, function (cpItem, nextItem) {
                cpItem.resolve(thread, cl, clazz, function (status) {
                    if (!status) {
                        nextItem("Failed.");
                    }
                    else {
                        nextItem();
                    }
                }, explicit);
            }, function (err) {
                if (err) {
                    cb(false);
                }
                else {
                    // Rerun. This time, all items are resolved.
                    _this.constructCallSiteObject(thread, cl, clazz, pc, cb, explicit);
                }
            });
        }
        /**
         * A call site specifier gives zero or more static arguments, which
         * communicate application-specific metadata to the bootstrap method. Any
         * static arguments which are symbolic references to classes, method
         * handles, or method types are resolved, as if by invocation of the ldc
         * instruction (§ldc), to obtain references to Class objects,
         * java.lang.invoke.MethodHandle objects, and java.lang.invoke.MethodType
         * objects respectively. Any static arguments that are string literals are
         * used to obtain references to String objects.
         */
        function getArguments() {
            var cpItems = bootstrapMethod[1], i, cpItem, rvObj = new (thread.getBsCl().getInitializedClass(thread, '[Ljava/lang/Object;').getConstructor(thread))(thread, cpItems.length), rv = rvObj.array;
            for (i = 0; i < cpItems.length; i++) {
                cpItem = cpItems[i];
                switch (cpItem.getType()) {
                    case enums_1.ConstantPoolItemType.CLASS:
                        rv[i] = cpItem.cls.getClassObject(thread);
                        break;
                    case enums_1.ConstantPoolItemType.METHOD_HANDLE:
                        rv[i] = cpItem.methodHandle;
                        break;
                    case enums_1.ConstantPoolItemType.METHOD_TYPE:
                        rv[i] = cpItem.methodType;
                        break;
                    case enums_1.ConstantPoolItemType.STRING:
                        rv[i] = cpItem.value;
                        break;
                    case enums_1.ConstantPoolItemType.UTF8:
                        rv[i] = thread.getJVM().internString(cpItem.value);
                        break;
                    case enums_1.ConstantPoolItemType.INTEGER:
                        rv[i] = cl.getInitializedClass(thread, 'I').createWrapperObject(thread, cpItem.value);
                        break;
                    case enums_1.ConstantPoolItemType.LONG:
                        rv[i] = cl.getInitializedClass(thread, 'J').createWrapperObject(thread, cpItem.value);
                        break;
                    case enums_1.ConstantPoolItemType.FLOAT:
                        rv[i] = cl.getInitializedClass(thread, 'F').createWrapperObject(thread, cpItem.value);
                        break;
                    case enums_1.ConstantPoolItemType.DOUBLE:
                        rv[i] = cl.getInitializedClass(thread, 'D').createWrapperObject(thread, cpItem.value);
                        break;
                    default:
                        assert_1["default"](false, "Invalid CPItem for static args: " + enums_1.ConstantPoolItemType[cpItem.getType()]);
                        break;
                }
            }
            assert_1["default"]((function () {
                var status = true;
                cpItems.forEach(function (cpItem, i) {
                    if (rv[i] === undefined) {
                        console.log("Undefined item at arg " + i + ": " + enums_1.ConstantPoolItemType[cpItem.getType()]);
                        status = false;
                    }
                    else if (rv[i] === null) {
                        console.log("Null item at arg " + i + ": " + enums_1.ConstantPoolItemType[cpItem.getType()]);
                        status = false;
                    }
                });
                return status;
            })(), "Arguments cannot be undefined or null.");
            return rvObj;
        }
        /**
         * A call site specifier gives a method descriptor, TD. A reference to an
         * instance of java.lang.invoke.MethodType is obtained as if by resolution
         * of a symbolic reference to a method type with the same parameter and
         * return types as TD (§5.4.3.5).
         *
         * Do what all OpenJDK-based JVMs do: Call
         * MethodHandleNatives.linkCallSite with:
         * - The class w/ the invokedynamic instruction
         * - The bootstrap method
         * - The name string from the nameAndTypeInfo
         * - The methodType object from the nameAndTypeInfo
         * - The static arguments from the bootstrap method.
         * - A 1-length appendix box.
         */
        var methodName = thread.getJVM().internString(this.nameAndTypeInfo.name), appendixArr = new (cl.getInitializedClass(thread, '[Ljava/lang/Object;').getConstructor(thread))(thread, 1), staticArgs = getArguments(), mhn = cl.getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;').getConstructor(thread);
        mhn['java/lang/invoke/MethodHandleNatives/linkCallSite(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;[Ljava/lang/Object;)Ljava/lang/invoke/MemberName;'](thread, [clazz.getClassObject(thread), bootstrapMethod[0].methodHandle, methodName, this.methodType, staticArgs, appendixArr], function (e, rv) {
            if (e) {
                thread.throwException(e);
                cb(false);
            }
            else {
                _this.setResolved(pc, [rv, appendixArr.array[0]]);
                cb(true);
            }
        });
    };
    InvokeDynamic.prototype.setResolved = function (pc, cso) {
        // Prevent resolution races. It's OK to create multiple CSOs, but only one
        // should ever be used!
        if (this.callSiteObjects[pc] === undefined) {
            this.callSiteObjects[pc] = cso;
        }
    };
    InvokeDynamic.fromBytes = function (byteStream, constantPool) {
        var bootstrapMethodAttrIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1["default"](nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool InvokeDynamic types mismatch');
        return new this(bootstrapMethodAttrIndex, nameAndTypeInfo);
    };
    InvokeDynamic.size = 1;
    InvokeDynamic.infoByteSize = 4;
    return InvokeDynamic;
}());
exports.InvokeDynamic = InvokeDynamic;
CP_CLASSES[enums_1.ConstantPoolItemType.INVOKE_DYNAMIC] = InvokeDynamic;
/**
 * Represents a given method handle.
 * ```
 * CONSTANT_MethodHandle_info {
 *   u1 tag;
 *   u1 reference_kind;
 *   u2 reference_index;
 * }
 * ```
 */
var MethodHandle = (function () {
    function MethodHandle(reference, referenceType) {
        /**
         * The resolved MethodHandle object.
         */
        this.methodHandle = null;
        this.reference = reference;
        this.referenceType = referenceType;
    }
    MethodHandle.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.METHOD_HANDLE;
    };
    MethodHandle.prototype.isResolved = function () { return this.methodHandle !== null; };
    MethodHandle.prototype.getConstant = function (thread) { return this.methodHandle; };
    /**
     * Asynchronously constructs a JVM-visible MethodHandle object for this
     * MethodHandle.
     *
     * Requires producing the following, and passing it to a MethodHandle
     * constructor:
     * * [java.lang.Class] The defining class.
     * * [java.lang.String] The name of the field/method/etc.
     * * [java.lang.invoke.MethodType | java.lang.Class] The type of the field OR,
     *   if a method, the type of the method descriptor.
     *
     * If needed, this function will resolve needed classes.
     */
    MethodHandle.prototype.resolve = function (thread, cl, caller, cb, explicit) {
        var _this = this;
        if (!this.reference.isResolved()) {
            return this.reference.resolve(thread, cl, caller, function (status) {
                if (!status) {
                    cb(false);
                }
                else {
                    _this.resolve(thread, cl, caller, cb, explicit);
                }
            }, explicit);
        }
        this.constructMethodHandleType(thread, cl, function (type) {
            if (type === null) {
                cb(false);
            }
            else {
                var methodHandleNatives = cl.getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;').getConstructor(thread);
                methodHandleNatives['linkMethodHandleConstant(Ljava/lang/Class;ILjava/lang/Class;Ljava/lang/String;Ljava/lang/Object;)Ljava/lang/invoke/MethodHandle;'](thread, [caller.getClassObject(thread), _this.referenceType, _this.getDefiningClassObj(thread), thread.getJVM().internString(_this.reference.nameAndTypeInfo.name), type], function (e, methodHandle) {
                    if (e) {
                        thread.throwException(e);
                        cb(false);
                    }
                    else {
                        _this.methodHandle = methodHandle;
                        cb(true);
                    }
                });
            }
        });
    };
    MethodHandle.prototype.getDefiningClassObj = function (thread) {
        if (this.reference.getType() === enums_1.ConstantPoolItemType.FIELDREF) {
            return this.reference.field.cls.getClassObject(thread);
        }
        else {
            return this.reference.method.cls.getClassObject(thread);
        }
    };
    MethodHandle.prototype.constructMethodHandleType = function (thread, cl, cb) {
        if (this.reference.getType() === enums_1.ConstantPoolItemType.FIELDREF) {
            var resolveObj = this.reference.nameAndTypeInfo.descriptor;
            cl.resolveClass(thread, resolveObj, function (cdata) {
                if (cdata !== null) {
                    cb(cdata.getClassObject(thread));
                }
                else {
                    cb(null);
                }
            });
        }
        else {
            util.createMethodType(thread, cl, this.reference.nameAndTypeInfo.descriptor, function (e, rv) {
                if (e) {
                    thread.throwException(e);
                    cb(null);
                }
                else {
                    cb(rv);
                }
            });
        }
    };
    MethodHandle.fromBytes = function (byteStream, constantPool) {
        var referenceKind = byteStream.getUint8(), referenceIndex = byteStream.getUint16(), reference = constantPool.get(referenceIndex);
        assert_1["default"](0 < referenceKind && referenceKind < 10, 'ConstantPool MethodHandle invalid referenceKind: ' + referenceKind);
        // Sanity check.
        assert_1["default"]((function () {
            switch (referenceKind) {
                case enums_1.MethodHandleReferenceKind.GETFIELD:
                case enums_1.MethodHandleReferenceKind.GETSTATIC:
                case enums_1.MethodHandleReferenceKind.PUTFIELD:
                case enums_1.MethodHandleReferenceKind.PUTSTATIC:
                    return reference.getType() === enums_1.ConstantPoolItemType.FIELDREF;
                case enums_1.MethodHandleReferenceKind.INVOKEINTERFACE:
                    return reference.getType() === enums_1.ConstantPoolItemType.INTERFACE_METHODREF
                        && reference.nameAndTypeInfo.name[0] !== '<';
                case enums_1.MethodHandleReferenceKind.INVOKEVIRTUAL:
                case enums_1.MethodHandleReferenceKind.INVOKESTATIC:
                case enums_1.MethodHandleReferenceKind.INVOKESPECIAL:
                    // NOTE: Spec says METHODREF, but I've found instances where
                    // INVOKESPECIAL is used on an INTERFACE_METHODREF.
                    return (reference.getType() === enums_1.ConstantPoolItemType.METHODREF
                        || reference.getType() === enums_1.ConstantPoolItemType.INTERFACE_METHODREF)
                        && reference.nameAndTypeInfo.name[0] !== '<';
                case enums_1.MethodHandleReferenceKind.NEWINVOKESPECIAL:
                    return reference.getType() === enums_1.ConstantPoolItemType.METHODREF
                        && reference.nameAndTypeInfo.name === '<init>';
            }
            return true;
        })(), "Invalid constant pool reference for method handle reference type: " + enums_1.MethodHandleReferenceKind[referenceKind]);
        return new this(reference, referenceKind);
    };
    MethodHandle.size = 1;
    MethodHandle.infoByteSize = 3;
    return MethodHandle;
}());
exports.MethodHandle = MethodHandle;
CP_CLASSES[enums_1.ConstantPoolItemType.METHOD_HANDLE] = MethodHandle;
// #endregion
/**
 * Constant pool type *resolution tiers*. Value is the tier, key is the
 * constant pool type.
 * Tier 0 has no references to other constant pool items, and can be resolved
 * first.
 * Tier 1 refers to tier 0 items.
 * Tier n refers to tier n-1 items and below.
 * Initialized in the given fashion to give the JS engine a tasty type hint.
 */
var CONSTANT_POOL_TIER = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
// Populate CONSTANT_POOL_TIER. Put into a closure to avoid scope pollution.
(function (tierInfos) {
    tierInfos.forEach(function (tierInfo, index) {
        tierInfo.forEach(function (type) {
            CONSTANT_POOL_TIER[type] = index;
        });
    });
})([
    // Tier 0
    [
        enums_1.ConstantPoolItemType.UTF8,
        enums_1.ConstantPoolItemType.INTEGER,
        enums_1.ConstantPoolItemType.FLOAT,
        enums_1.ConstantPoolItemType.LONG,
        enums_1.ConstantPoolItemType.DOUBLE
    ],
    // Tier 1
    [
        enums_1.ConstantPoolItemType.CLASS,
        enums_1.ConstantPoolItemType.STRING,
        enums_1.ConstantPoolItemType.NAME_AND_TYPE,
        enums_1.ConstantPoolItemType.METHOD_TYPE
    ],
    // Tier 2
    [
        enums_1.ConstantPoolItemType.FIELDREF,
        enums_1.ConstantPoolItemType.METHODREF,
        enums_1.ConstantPoolItemType.INTERFACE_METHODREF,
        enums_1.ConstantPoolItemType.INVOKE_DYNAMIC
    ],
    // Tier 3
    [
        enums_1.ConstantPoolItemType.METHOD_HANDLE
    ]
]);
/**
 * Represents a constant pool for a particular class.
 */
var ConstantPool = (function () {
    function ConstantPool() {
    }
    ConstantPool.prototype.parse = function (byteStream, cpPatches) {
        var _this = this;
        if (cpPatches === void 0) { cpPatches = null; }
        var cpCount = byteStream.getUint16(), 
        // First key is the tier.
        deferredQueue = [[], [], []], 
        // The ending offset of the constant pool items.
        endIdx = 0, idx = 1, 
        // Tag of the currently-being-processed item.
        tag = 0, 
        // Offset of the currently-being-processed item.
        itemOffset = 0, 
        // Tier of the currently-being-processed item.
        itemTier = 0;
        this.constantPool = new Array(cpCount);
        // Scan for tier info.
        while (idx < cpCount) {
            itemOffset = byteStream.pos();
            tag = byteStream.getUint8();
            assert_1["default"](CP_CLASSES[tag] !== null && CP_CLASSES[tag] !== undefined, 'Unknown ConstantPool tag: ' + tag);
            itemTier = CONSTANT_POOL_TIER[tag];
            if (itemTier > 0) {
                deferredQueue[itemTier - 1].push({ offset: itemOffset, index: idx });
                byteStream.skip(CP_CLASSES[tag].infoByteSize);
            }
            else {
                this.constantPool[idx] = CP_CLASSES[tag].fromBytes(byteStream, this);
            }
            idx += CP_CLASSES[tag].size;
        }
        endIdx = byteStream.pos();
        // Process tiers.
        deferredQueue.forEach(function (deferredItems) {
            deferredItems.forEach(function (item) {
                byteStream.seek(item.offset);
                tag = byteStream.getUint8();
                _this.constantPool[item.index] = CP_CLASSES[tag].fromBytes(byteStream, _this);
                if (cpPatches !== null && cpPatches.array[item.index] !== null && cpPatches.array[item.index] !== undefined) {
                    /*
                     * For each CP entry, the corresponding CP patch must either be null or have
                     * the format that matches its tag:
                     *
                     * * Integer, Long, Float, Double: the corresponding wrapper object type from java.lang
                     * * Utf8: a string (must have suitable syntax if used as signature or name)
                     * * Class: any java.lang.Class object
                     * * String: any object (not just a java.lang.String)
                     * * InterfaceMethodRef: (NYI) a method handle to invoke on that call site's arguments
                     */
                    var patchObj = cpPatches.array[item.index];
                    switch (patchObj.getClass().getInternalName()) {
                        case 'Ljava/lang/Integer;':
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.INTEGER);
                            _this.constantPool[item.index].value = patchObj['java/lang/Integer/value'];
                            break;
                        case 'Ljava/lang/Long;':
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.LONG);
                            _this.constantPool[item.index].value = patchObj['java/lang/Long/value'];
                            break;
                        case 'Ljava/lang/Float;':
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.FLOAT);
                            _this.constantPool[item.index].value = patchObj['java/lang/Float/value'];
                            break;
                        case 'Ljava/lang/Double;':
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.DOUBLE);
                            _this.constantPool[item.index].value = patchObj['java/lang/Double/value'];
                            break;
                        case 'Ljava/lang/String;':
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.UTF8);
                            _this.constantPool[item.index].value = patchObj.toString();
                            break;
                        case 'Ljava/lang/Class;':
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.CLASS);
                            _this.constantPool[item.index].name = patchObj.$cls.getInternalName();
                            _this.constantPool[item.index].cls = patchObj.$cls;
                            break;
                        default:
                            assert_1["default"](tag === enums_1.ConstantPoolItemType.STRING);
                            _this.constantPool[item.index].stringValue = "";
                            // XXX: Not actually a string, but the JVM does this.
                            _this.constantPool[item.index].value = patchObj;
                            break;
                    }
                }
            });
        });
        // Return to the correct offset, at the end of the CP data.
        byteStream.seek(endIdx);
        return byteStream;
    };
    ConstantPool.prototype.get = function (idx) {
        assert_1["default"](this.constantPool[idx] !== undefined, "Invalid ConstantPool reference.");
        return this.constantPool[idx];
    };
    ConstantPool.prototype.each = function (fn) {
        this.constantPool.forEach(function (item, idx) {
            if (item !== undefined) {
                fn(idx, item);
            }
        });
    };
    return ConstantPool;
}());
exports.ConstantPool = ConstantPool;
/// Resolved forms of constant pool items.
//# sourceMappingURL=data:application/json;base64,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