"use strict";
var enums_1 = require('./enums');
var assert_1 = require('./assert');
/**
 * Represents a JVM monitor.
 */
var Monitor = (function () {
    function Monitor() {
        /**
         * The owner of the monitor.
         */
        this.owner = null;
        /**
         * Number of times that the current owner has locked this monitor.
         */
        this.count = 0;
        /**
         * JVM threads that are waiting for the current owner to relinquish the
         * monitor.
         */
        this.blocked = {};
        /**
         * Queue of JVM threads that are waiting for a JVM thread to notify them.
         */
        this.waiting = {};
    }
    /**
     * Attempts to acquire the monitor.
     *
     * Thread transitions:
     * * RUNNABLE => BLOCKED [If fails to acquire lock]
     *
     * @param thread The thread that is trying to acquire the monitor.
     * @param cb If this method returns false, then this callback will be
     *   triggered once the thread becomes owner of the monitor. At that time,
     *   the thread will be in the RUNNABLE state.
     * @return True if successfull, false if not. If not successful, the thread
     *   becomes BLOCKED, and the input callback will be triggered once the
     *   thread owns the monitor and is RUNNABLE.
     */
    Monitor.prototype.enter = function (thread, cb) {
        if (this.owner === thread) {
            this.count++;
            return true;
        }
        else {
            return this.contendForLock(thread, 1, enums_1.ThreadStatus.BLOCKED, cb);
        }
    };
    /**
     * Generic version of Monitor.enter for contending for the lock.
     *
     * Thread transitions:
     * * RUNNABLE => UNINTERRUPTIBLY_BLOCKED [If fails to acquire lock]
     * * RUNNABLE => BLOCKED [If fails to acquire lock]
     *
     * @param thread The thread contending for the lock.
     * @param count The lock count to use once the thread owns the lock.
     * @param blockStatus The ThreadStatus to use should the thread need to
     *   contend for the lock (either BLOCKED or UNINTERRUPTIBLY_BLOCKED).
     * @param cb The callback to call once the thread becomes owner of the lock.
     * @return True if the thread immediately acquired the lock, false if the
     *   thread is now blocked on the lock.
     */
    Monitor.prototype.contendForLock = function (thread, count, blockStatus, cb) {
        var owner = this.owner;
        assert_1["default"](owner != thread, "Thread attempting to contend for lock it already owns!");
        if (owner === null) {
            assert_1["default"](this.count === 0);
            this.owner = thread;
            this.count = count;
            return true;
        }
        else {
            /**
             * "If another thread already owns the monitor associated with objectref,
             *  the thread blocks until the monitor's entry count is zero, then tries
             *  again to gain ownership."
             * @from http://docs.oracle.com/javase/specs/jvms/se7/html/jvms-6.html#jvms-6.5.monitorenter
             */
            this.blocked[thread.getRef()] = { thread: thread, cb: cb, count: count };
            thread.setStatus(blockStatus, this);
            return false;
        }
    };
    /**
     * Exits the monitor. Handles notifying the waiting threads if the lock
     * becomes available.
     *
     * Thread transitions:
     * * *NONE* on the argument thread.
     * * A *BLOCKED* thread may be scheduled if the owner gives up the monitor.
     *
     * @param thread The thread that is exiting the monitor.
     * @return True if exit succeeded, false if an exception occurred.
     */
    Monitor.prototype.exit = function (thread) {
        var owner = this.owner;
        if (owner === thread) {
            if (--this.count === 0) {
                this.owner = null;
                this.appointNewOwner();
            }
        }
        else {
            /**
             * "If the thread that executes monitorexit is not the owner of the
             *  monitor associated with the instance referenced by objectref,
             *  monitorexit throws an IllegalMonitorStateException."
             * @from http://docs.oracle.com/javase/specs/jvms/se7/html/jvms-6.html#jvms-6.5.monitorexit
             */
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', "Cannot exit a monitor that you do not own.");
        }
        return owner === thread;
    };
    /**
     * Chooses one of the blocked threads to become the monitor's owner.
     */
    Monitor.prototype.appointNewOwner = function () {
        var blockedThreadRefs = Object.keys(this.blocked);
        if (blockedThreadRefs.length > 0) {
            // Unblock a random thread.
            var unblockedRef = blockedThreadRefs[Math.floor(Math.random() * blockedThreadRefs.length)], 
            // XXX: Typing hack. Key must be a number.
            unblocked = this.blocked[unblockedRef];
            this.unblock(unblocked.thread, false);
        }
    };
    /**
     * "Causes the current thread to wait until another thread invokes the
     *  notify() method or the notifyAll() method for this object, or some other
     *  thread interrupts the current thread, or a certain amount of real time
     *  has elapsed.
     *
     *  This method causes the current thread (call it T) to place itself in the
     *  wait set for this object and then to relinquish any and all
     *  synchronization claims on this object."
     *
     * We coalesce all possible wait configurations into this one function.
     * @from http://docs.oracle.com/javase/7/docs/api/java/lang/Object.html#wait(long, int)
     * @param thread The thread that wants to wait on this monitor.
     * @param cb The callback triggered once the thread wakes up.
     * @param timeoutMs? An optional timeout that specifies how long the thread
     *   should wait, in milliseconds. If this value is 0, then we ignore it.
     * @param timeoutNs? An optional timeout that specifies how long the thread
     *   should wait, in nanosecond precision (currently ignored).
     * @todo Use high-precision timers in browsers that support it.
     * @return True if the wait succeeded, false if it triggered an exception.
     */
    Monitor.prototype.wait = function (thread, cb, timeoutMs, timeoutNs) {
        var _this = this;
        if (this.getOwner() === thread) {
            // INVARIANT: Thread shouldn't currently be blocked on a monitor.
            assert_1["default"](thread.getStatus() !== enums_1.ThreadStatus.BLOCKED);
            this.waiting[thread.getRef()] = {
                thread: thread,
                cb: cb,
                count: this.count,
                isTimed: timeoutMs != null && timeoutMs !== 0
            };
            // Revoke ownership.
            this.owner = null;
            this.count = 0;
            if (timeoutMs != null && timeoutMs !== 0) {
                // Scheduler a timer that wakes up the thread.
                // XXX: Casting to 'number', since NodeJS typings specify a Timer.
                this.waiting[thread.getRef()].timer = setTimeout(function () {
                    _this.unwait(thread, true);
                }, timeoutMs);
                thread.setStatus(enums_1.ThreadStatus.TIMED_WAITING, this);
            }
            else {
                thread.setStatus(enums_1.ThreadStatus.WAITING, this);
            }
            // Select a new owner.
            this.appointNewOwner();
            return true;
        }
        else {
            /**
             * "The current thread must own this object's monitor"
             */
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', "Cannot wait on an object that you do not own.");
            return false;
        }
    };
    /**
     * Removes the specified thread from the waiting set, and makes it compete
     * for the monitor lock. Once it acquires the lock, we restore its lock
     * count prior to triggering the wait callback.
     *
     * If the thread is interrupted, the wait callback is *not* triggered.
     *
     * @param thread The thread to remove.
     * @param fromTimer Indicates if this function call was triggered from a
     *   timer event.
     * @param [interrupting] If true, then we are *interrupting* the wait. Do not
     *   trigger the wait callback.
     * @param [unwaitCb] If interrupting is true, then this callback is triggered
     *   once the thread reacquires the lock.
     */
    Monitor.prototype.unwait = function (thread, fromTimer, interrupting, unwaitCb) {
        if (interrupting === void 0) { interrupting = false; }
        if (unwaitCb === void 0) { unwaitCb = null; }
        // Step 1: Remove the thread from the waiting set.
        var waitEntry = this.waiting[thread.getRef()], 
        // Interrupting a previously-waiting thread before it acquires a lock
        // makes no semantic sense, as the thread is currently suspended in a
        // synchronized block that requires ownership of the monitor.
        blockStatus = enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED, blockCb = function () {
            // Thread is RUNNABLE before we trigger the callback.
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            if (interrupting) {
                unwaitCb();
            }
            else {
                waitEntry.cb(fromTimer);
            }
        };
        assert_1["default"](waitEntry != null);
        delete this.waiting[thread.getRef()];
        // Step 2: Remove the timer if the timer did not trigger this event.
        if (thread.getStatus() === enums_1.ThreadStatus.TIMED_WAITING && !fromTimer) {
            var timerId = waitEntry.timer;
            assert_1["default"](timerId != null);
            clearTimeout(timerId);
        }
        // Step 3: Acquire the monitor [ASYNC]
        if (this.contendForLock(thread, waitEntry.count, blockStatus, blockCb)) {
            // Success! Trigger the blockCb anyway. If 'contendForLock' returns false,
            // it will trigger blockCb once the thread acquires the lock.
            blockCb();
        }
    };
    /**
     * Removes the specified thread from being blocked on the monitor so it can
     * re-compete for ownership.
     * @param [interrupting] If true, we are interrupting the monitor block. The
     *   thread should not acquire the lock, and the block callback should not
     *   be triggered.
     */
    Monitor.prototype.unblock = function (thread, interrupting) {
        if (interrupting === void 0) { interrupting = false; }
        var blockEntry = this.blocked[thread.getRef()];
        // Cannot interrupt an uninterruptibly blocked thread.
        assert_1["default"](interrupting ? thread.getStatus() === enums_1.ThreadStatus.BLOCKED : true);
        if (blockEntry != null) {
            delete this.blocked[thread.getRef()];
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            if (!interrupting) {
                // No one else can own the monitor.
                assert_1["default"](this.owner == null && this.count === 0, "T" + thread.getRef() + ": We're not interrupting a block, but someone else owns the monitor?! Owned by " + (this.owner == null ? "[no one]" : "" + this.owner.getRef()) + " Count: " + this.count);
                // Assign this thread as the monitor owner.
                this.owner = thread;
                this.count = blockEntry.count;
                // Trigger the callback.
                blockEntry.cb();
            }
        }
    };
    /**
     * Notifies a single waiting thread.
     * @param thread The notifying thread. *MUST* be the owner.
     */
    Monitor.prototype.notify = function (thread) {
        if (this.owner === thread) {
            var waitingRefs = Object.keys(this.waiting);
            if (waitingRefs.length > 0) {
                // Notify a random thread.
                this.unwait(this.waiting[waitingRefs[Math.floor(Math.random() * waitingRefs.length)]].thread, false);
            }
        }
        else {
            /**
             * "Throws IllegalMonitorStateException if the current thread is not the
             *  owner of this object's monitor."
             * @from http://docs.oracle.com/javase/7/docs/api/java/lang/Object.html#notify()
             */
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', "Cannot notify on a monitor that you do not own.");
        }
    };
    /**
     * Notifies all waiting threads.
     * @param thread The notifying thread. *MUST* be the owner.
     */
    Monitor.prototype.notifyAll = function (thread) {
        if (this.owner === thread) {
            var waitingRefs = Object.keys(this.waiting), i;
            // Notify each thread.
            for (i = 0; i < waitingRefs.length; i++) {
                this.unwait(this.waiting[waitingRefs[i]].thread, false);
            }
        }
        else {
            /**
             * "Throws IllegalMonitorStateException if the current thread is not the
             *  owner of this object's monitor."
             * @from http://docs.oracle.com/javase/7/docs/api/java/lang/Object.html#notifyAll()
             */
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', "Cannot notifyAll on a monitor that you do not own.");
        }
    };
    /**
     * @return The owner of the monitor.
     */
    Monitor.prototype.getOwner = function () {
        return this.owner;
    };
    Monitor.prototype.isWaiting = function (thread) {
        // Waiting, but *not* timed waiting.
        return this.waiting[thread.getRef()] != null && !this.waiting[thread.getRef()].isTimed;
    };
    Monitor.prototype.isTimedWaiting = function (thread) {
        // Timed waiting, *not* waiting.
        return this.waiting[thread.getRef()] != null && this.waiting[thread.getRef()].isTimed;
    };
    Monitor.prototype.isBlocked = function (thread) {
        // Blocked.
        return this.blocked[thread.getRef()] != null;
    };
    return Monitor;
}());
exports.__esModule = true;
exports["default"] = Monitor;
//# sourceMappingURL=data:application/json;base64,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