"use strict";
var util_1 = require('./util');
var enums_1 = require('./enums');
var assert_1 = require('./assert');
var ExceptionHandler = (function () {
    function ExceptionHandler(startPC, endPC, handlerPC, catchType) {
        this.startPC = startPC;
        this.endPC = endPC;
        this.handlerPC = handlerPC;
        this.catchType = catchType;
    }
    ExceptionHandler.prototype.getName = function () {
        return 'ExceptionHandler';
    };
    ExceptionHandler.parse = function (bytesArray, constantPool) {
        var startPC = bytesArray.getUint16(), endPC = bytesArray.getUint16(), handlerPC = bytesArray.getUint16(), cti = bytesArray.getUint16(), catchType = cti === 0 ? "<any>" : constantPool.get(cti).name;
        return new this(startPC, endPC, handlerPC, catchType);
    };
    return ExceptionHandler;
}());
exports.ExceptionHandler = ExceptionHandler;
var Code = (function () {
    function Code(maxStack, maxLocals, exceptionHandlers, attrs, code) {
        this.maxStack = maxStack;
        this.maxLocals = maxLocals;
        this.exceptionHandlers = exceptionHandlers;
        this.attrs = attrs;
        this.code = code;
    }
    Code.prototype.getName = function () {
        return 'Code';
    };
    Code.prototype.getMaxStack = function () {
        return this.maxStack;
    };
    Code.parse = function (byteStream, constantPool) {
        var maxStack = byteStream.getUint16(), maxLocals = byteStream.getUint16(), codeLen = byteStream.getUint32();
        if (codeLen === 0) {
            throw "Error parsing code: Code length is zero";
        }
        var code = byteStream.slice(codeLen).getBuffer(), exceptLen = byteStream.getUint16(), exceptionHandlers = [];
        for (var i = 0; i < exceptLen; i++) {
            exceptionHandlers.push(ExceptionHandler.parse(byteStream, constantPool));
        }
        // yes, there are even attrs on attrs. BWOM... BWOM...
        var attrs = makeAttributes(byteStream, constantPool);
        return new this(maxStack, maxLocals, exceptionHandlers, attrs, code);
    };
    Code.prototype.getCode = function () {
        return this.code;
    };
    Code.prototype.getAttribute = function (name) {
        for (var i = 0; i < this.attrs.length; i++) {
            var attr = this.attrs[i];
            if (attr.getName() === name) {
                return attr;
            }
        }
        return null;
    };
    return Code;
}());
exports.Code = Code;
var LineNumberTable = (function () {
    function LineNumberTable(entries) {
        this.entries = entries;
    }
    LineNumberTable.prototype.getName = function () {
        return 'LineNumberTable';
    };
    /**
     * Returns the relevant source code line number for the specified program
     * counter.
     */
    LineNumberTable.prototype.getLineNumber = function (pc) {
        var j, lineNumber = -1;
        // get the last line number before the stack frame's pc
        for (j = 0; j < this.entries.length; j++) {
            var entry = this.entries[j];
            if (entry.startPC <= pc) {
                lineNumber = entry.lineNumber;
            }
            else {
                // Further entries are past the PC.
                break;
            }
        }
        return lineNumber;
    };
    LineNumberTable.parse = function (byteStream, constantPool) {
        var entries = [];
        var lntLen = byteStream.getUint16();
        for (var i = 0; i < lntLen; i++) {
            var spc = byteStream.getUint16();
            var ln = byteStream.getUint16();
            entries.push({
                'startPC': spc,
                'lineNumber': ln
            });
        }
        return new this(entries);
    };
    return LineNumberTable;
}());
exports.LineNumberTable = LineNumberTable;
var SourceFile = (function () {
    function SourceFile(filename) {
        this.filename = filename;
    }
    SourceFile.prototype.getName = function () {
        return 'SourceFile';
    };
    SourceFile.parse = function (byteStream, constantPool) {
        return new this(constantPool.get(byteStream.getUint16()).value);
    };
    return SourceFile;
}());
exports.SourceFile = SourceFile;
var StackMapTable = (function () {
    function StackMapTable(entries) {
        this.entries = entries;
    }
    StackMapTable.prototype.getName = function () {
        return 'StackMapTable';
    };
    StackMapTable.parse = function (byteStream, constantPool) {
        var numEntries = byteStream.getUint16(), entries = [];
        for (var i = 0; i < numEntries; i++) {
            entries.push(this.parseEntry(byteStream, constantPool));
        }
        return new this(entries);
    };
    StackMapTable.parseEntry = function (byteStream, constantPool) {
        var frameType = byteStream.getUint8(), locals, offsetDelta, i;
        if (frameType < 64) {
            return {
                type: enums_1.StackMapTableEntryType.SAME_FRAME,
                offsetDelta: frameType
            };
        }
        else if (frameType < 128) {
            return {
                type: enums_1.StackMapTableEntryType.SAME_LOCALS_1_STACK_ITEM_FRAME,
                offsetDelta: frameType - 64,
                stack: [this.parseVerificationTypeInfo(byteStream, constantPool)]
            };
        }
        else if (frameType < 247) {
        }
        else if (frameType === 247) {
            return {
                type: enums_1.StackMapTableEntryType.SAME_LOCALS_1_STACK_ITEM_FRAME_EXTENDED,
                offsetDelta: byteStream.getUint16(),
                stack: [this.parseVerificationTypeInfo(byteStream, constantPool)]
            };
        }
        else if (frameType < 251) {
            return {
                type: enums_1.StackMapTableEntryType.CHOP_FRAME,
                offsetDelta: byteStream.getUint16(),
                k: 251 - frameType
            };
        }
        else if (frameType === 251) {
            return {
                type: enums_1.StackMapTableEntryType.SAME_FRAME_EXTENDED,
                offsetDelta: byteStream.getUint16()
            };
        }
        else if (frameType < 255) {
            offsetDelta = byteStream.getUint16();
            locals = [];
            for (i = 0; i < frameType - 251; i++) {
                locals.push(this.parseVerificationTypeInfo(byteStream, constantPool));
            }
            return {
                type: enums_1.StackMapTableEntryType.APPEND_FRAME,
                offsetDelta: offsetDelta,
                locals: locals
            };
        }
        else if (frameType === 255) {
            offsetDelta = byteStream.getUint16();
            var numLocals = byteStream.getUint16();
            locals = [];
            for (i = 0; i < numLocals; i++) {
                locals.push(this.parseVerificationTypeInfo(byteStream, constantPool));
            }
            var numStackItems = byteStream.getUint16();
            var stack = [];
            for (i = 0; i < numStackItems; i++) {
                stack.push(this.parseVerificationTypeInfo(byteStream, constantPool));
            }
            return {
                type: enums_1.StackMapTableEntryType.FULL_FRAME,
                offsetDelta: offsetDelta,
                numLocals: numLocals,
                locals: locals,
                numStackItems: numStackItems,
                stack: stack
            };
        }
    };
    StackMapTable.parseVerificationTypeInfo = function (byteStream, constantPool) {
        var tag = byteStream.getUint8();
        if (tag === 7) {
            var cls = constantPool.get(byteStream.getUint16()).name;
            return 'class ' + (/\w/.test(cls[0]) ? util_1.descriptor2typestr(cls) : "\"" + cls + "\"");
        }
        else if (tag === 8) {
            return 'uninitialized ' + byteStream.getUint16();
        }
        else {
            var tagToType = ['bogus', 'int', 'float', 'double', 'long', 'null', 'this', 'object', 'uninitialized'];
            return tagToType[tag];
        }
    };
    return StackMapTable;
}());
exports.StackMapTable = StackMapTable;
var LocalVariableTable = (function () {
    function LocalVariableTable(entries) {
        this.entries = entries;
    }
    LocalVariableTable.prototype.getName = function () {
        return 'LocalVariableTable';
    };
    LocalVariableTable.parse = function (byteStream, constantPool) {
        var numEntries = byteStream.getUint16(), entries = [];
        for (var i = 0; i < numEntries; i++) {
            entries.push(this.parseEntries(byteStream, constantPool));
        }
        return new this(entries);
    };
    LocalVariableTable.parseEntries = function (bytes_array, constant_pool) {
        return {
            startPC: bytes_array.getUint16(),
            length: bytes_array.getUint16(),
            name: constant_pool.get(bytes_array.getUint16()).value,
            descriptor: constant_pool.get(bytes_array.getUint16()).value,
            ref: bytes_array.getUint16()
        };
    };
    return LocalVariableTable;
}());
exports.LocalVariableTable = LocalVariableTable;
var LocalVariableTypeTable = (function () {
    function LocalVariableTypeTable(entries) {
        this.entries = entries;
    }
    LocalVariableTypeTable.prototype.getName = function () {
        return 'LocalVariableTypeTable';
    };
    LocalVariableTypeTable.parse = function (byteStream, constantPool) {
        var numEntries = byteStream.getUint16(), i, entries = [];
        for (i = 0; i < numEntries; i++) {
            entries.push(this.parseTableEntry(byteStream, constantPool));
        }
        return new this(entries);
    };
    LocalVariableTypeTable.parseTableEntry = function (byteStream, constantPool) {
        return {
            startPC: byteStream.getUint16(),
            length: byteStream.getUint16(),
            name: constantPool.get(byteStream.getUint16()).value,
            signature: constantPool.get(byteStream.getUint16()).value,
            index: byteStream.getUint16()
        };
    };
    return LocalVariableTypeTable;
}());
exports.LocalVariableTypeTable = LocalVariableTypeTable;
var Exceptions = (function () {
    function Exceptions(exceptions) {
        this.exceptions = exceptions;
    }
    Exceptions.prototype.getName = function () {
        return 'Exceptions';
    };
    Exceptions.parse = function (byteStream, constantPool) {
        var numExceptions = byteStream.getUint16();
        var excRefs = [];
        for (var i = 0; i < numExceptions; i++) {
            excRefs.push(byteStream.getUint16());
        }
        return new this(excRefs.map(function (ref) { return constantPool.get(ref).name; }));
    };
    return Exceptions;
}());
exports.Exceptions = Exceptions;
var InnerClasses = (function () {
    function InnerClasses(classes) {
        this.classes = classes;
    }
    InnerClasses.prototype.getName = function () {
        return 'InnerClasses';
    };
    InnerClasses.parse = function (bytes_array, constant_pool) {
        var numClasses = bytes_array.getUint16(), classes = [];
        for (var i = 0; i < numClasses; i++) {
            classes.push(this.parseClass(bytes_array, constant_pool));
        }
        return new this(classes);
    };
    InnerClasses.parseClass = function (byteStream, constantPool) {
        return {
            innerInfoIndex: byteStream.getUint16(),
            outerInfoIndex: byteStream.getUint16(),
            innerNameIndex: byteStream.getUint16(),
            innerAccessFlags: byteStream.getUint16()
        };
    };
    return InnerClasses;
}());
exports.InnerClasses = InnerClasses;
var ConstantValue = (function () {
    function ConstantValue(value) {
        this.value = value;
    }
    ConstantValue.prototype.getName = function () {
        return 'ConstantValue';
    };
    ConstantValue.parse = function (bytes_array, constant_pool) {
        var ref = bytes_array.getUint16();
        return new this(constant_pool.get(ref));
    };
    return ConstantValue;
}());
exports.ConstantValue = ConstantValue;
var Synthetic = (function () {
    function Synthetic() {
    }
    Synthetic.prototype.getName = function () {
        return 'Synthetic';
    };
    Synthetic.parse = function (byteStream, constantPool) {
        return new this();
    };
    return Synthetic;
}());
exports.Synthetic = Synthetic;
var Deprecated = (function () {
    function Deprecated() {
    }
    Deprecated.prototype.getName = function () {
        return 'Deprecated';
    };
    Deprecated.parse = function (byteStream, constantPool) {
        return new this();
    };
    return Deprecated;
}());
exports.Deprecated = Deprecated;
var Signature = (function () {
    function Signature(sig) {
        this.sig = sig;
    }
    Signature.prototype.getName = function () {
        return 'Signature';
    };
    Signature.parse = function (byteStream, constantPool) {
        return new this(constantPool.get(byteStream.getUint16()).value);
    };
    return Signature;
}());
exports.Signature = Signature;
var RuntimeVisibleAnnotations = (function () {
    function RuntimeVisibleAnnotations(rawBytes, isHidden, isCallerSensitive, isCompiled) {
        this.rawBytes = rawBytes;
        this.isHidden = isHidden;
        this.isCallerSensitive = isCallerSensitive;
        this.isCompiled = isCompiled;
    }
    RuntimeVisibleAnnotations.prototype.getName = function () {
        return 'RuntimeVisibleAnnotations';
    };
    RuntimeVisibleAnnotations.parse = function (byteStream, constantPool, attrLen) {
        // No need to parse; OpenJDK parses these from within Java code from
        // the raw bytes.
        // ...but we need to look for the 'Hidden' annotation, which specifies if
        // the method should be omitted from stack frames.
        // And the 'compiled' annotation, which specifies if the method was
        // compiled.
        // And the 'CallerSensitive' annotation, which specifies that the function's
        // behavior differs depending on the caller.
        /**
         * Skip the current RuntimeVisibleAnnotation.
         */
        function skipAnnotation() {
            byteStream.skip(2); // type index
            var numValuePairs = byteStream.getUint16(), i;
            for (i = 0; i < numValuePairs; i++) {
                byteStream.skip(2); // element name index
                skipElementValue();
            }
        }
        /**
         * Skip this particular element value.
         */
        function skipElementValue() {
            var tag = String.fromCharCode(byteStream.getUint8());
            switch (tag) {
                case 'e':
                    // Fall-through.
                    byteStream.skip(2);
                case 'Z':
                case 'B':
                case 'C':
                case 'S':
                case 'I':
                case 'F':
                case 'J':
                case 'D':
                case 's':
                case 'c':
                    byteStream.skip(2);
                    break;
                case '@':
                    skipAnnotation();
                    break;
                case '[':
                    var numValues = byteStream.getUint16(), i;
                    for (i = 0; i < numValues; i++) {
                        skipElementValue();
                    }
                    break;
            }
        }
        var rawBytes = byteStream.read(attrLen), isHidden = false, isCompiled = false, isCallerSensitive = false;
        byteStream.seek(byteStream.pos() - rawBytes.length);
        var numAttributes = byteStream.getUint16(), i;
        for (i = 0; i < numAttributes; i++) {
            var typeName = constantPool.get(byteStream.getUint16());
            // Rewind.
            byteStream.seek(byteStream.pos() - 2);
            skipAnnotation();
            switch (typeName.value) {
                case 'Ljava/lang/invoke/LambdaForm$Hidden;':
                    isHidden = true;
                    break;
                case 'Lsig/sun/reflect/CallerSensitive;':
                    isCallerSensitive = true;
                    break;
                case 'Lsig/java/lang/invoke/LambdaForm$Compiled':
                    isCompiled = true;
                    break;
            }
        }
        return new this(rawBytes, isHidden, isCallerSensitive, isCompiled);
    };
    return RuntimeVisibleAnnotations;
}());
exports.RuntimeVisibleAnnotations = RuntimeVisibleAnnotations;
var AnnotationDefault = (function () {
    function AnnotationDefault(rawBytes) {
        this.rawBytes = rawBytes;
    }
    AnnotationDefault.prototype.getName = function () {
        return 'AnnotationDefault';
    };
    AnnotationDefault.parse = function (byteStream, constantPool, attrLen) {
        return new this(byteStream.read(attrLen));
    };
    return AnnotationDefault;
}());
exports.AnnotationDefault = AnnotationDefault;
var EnclosingMethod = (function () {
    function EnclosingMethod(encClass, encMethod) {
        this.encClass = encClass;
        this.encMethod = encMethod;
    }
    EnclosingMethod.prototype.getName = function () {
        return 'EnclosingMethod';
    };
    EnclosingMethod.parse = function (byteStream, constantPool) {
        var encClass = constantPool.get(byteStream.getUint16()), methodRef = byteStream.getUint16(), encMethod = null;
        if (methodRef > 0) {
            encMethod = constantPool.get(methodRef);
            assert_1["default"](encMethod.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, "Enclosing method must be a name and type info.");
        }
        return new this(encClass, encMethod);
    };
    return EnclosingMethod;
}());
exports.EnclosingMethod = EnclosingMethod;
var BootstrapMethods = (function () {
    function BootstrapMethods(bootstrapMethods) {
        this.bootstrapMethods = bootstrapMethods;
    }
    BootstrapMethods.prototype.getName = function () {
        return 'BootstrapMethods';
    };
    BootstrapMethods.parse = function (byteStream, constantPool) {
        var numBootstrapMethods = byteStream.getUint16(), bootstrapMethods = [];
        for (var i = 0; i < numBootstrapMethods; i++) {
            var methodHandle = constantPool.get(byteStream.getUint16());
            var numArgs = byteStream.getUint16();
            var args = [];
            for (var j = 0; j < numArgs; j++) {
                args.push(constantPool.get(byteStream.getUint16()));
            }
            bootstrapMethods.push([methodHandle, args]);
        }
        return new this(bootstrapMethods);
    };
    return BootstrapMethods;
}());
exports.BootstrapMethods = BootstrapMethods;
var RuntimeVisibleParameterAnnotations = (function () {
    function RuntimeVisibleParameterAnnotations(rawBytes) {
        this.rawBytes = rawBytes;
    }
    RuntimeVisibleParameterAnnotations.prototype.getName = function () {
        return 'RuntimeVisibleParameterAnnotations';
    };
    RuntimeVisibleParameterAnnotations.parse = function (byteStream, constantPool, attrLen) {
        return new this(byteStream.read(attrLen));
    };
    return RuntimeVisibleParameterAnnotations;
}());
exports.RuntimeVisibleParameterAnnotations = RuntimeVisibleParameterAnnotations;
function makeAttributes(byteStream, constantPool) {
    var attrTypes = {
        'Code': Code,
        'LineNumberTable': LineNumberTable,
        'SourceFile': SourceFile,
        'StackMapTable': StackMapTable,
        'LocalVariableTable': LocalVariableTable,
        'LocalVariableTypeTable': LocalVariableTypeTable,
        'ConstantValue': ConstantValue,
        'Exceptions': Exceptions,
        'InnerClasses': InnerClasses,
        'Synthetic': Synthetic,
        'Deprecated': Deprecated,
        'Signature': Signature,
        'RuntimeVisibleAnnotations': RuntimeVisibleAnnotations,
        'AnnotationDefault': AnnotationDefault,
        'EnclosingMethod': EnclosingMethod,
        'BootstrapMethods': BootstrapMethods,
        'RuntimeVisibleParameterAnnotations': RuntimeVisibleParameterAnnotations
    };
    var numAttrs = byteStream.getUint16();
    var attrs = [];
    for (var i = 0; i < numAttrs; i++) {
        var name = constantPool.get(byteStream.getUint16()).value;
        var attrLen = byteStream.getUint32();
        if (attrTypes[name] != null) {
            var oldLen = byteStream.size();
            var attr = attrTypes[name].parse(byteStream, constantPool, attrLen, name);
            var newLen = byteStream.size();
            assert_1["default"]((oldLen - newLen) <= attrLen, "A parsed attribute read beyond its data! " + name);
            if (oldLen - newLen !== attrLen) {
                byteStream.skip(attrLen - oldLen + newLen);
            }
            attrs.push(attr);
        }
        else {
            // we must silently ignore other attrs
            byteStream.skip(attrLen);
        }
    }
    return attrs;
}
exports.makeAttributes = makeAttributes;
//# sourceMappingURL=data:application/json;base64,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