"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var enums_1 = require('./enums');
var assert_1 = require('./assert');
var fs = require('fs');
var BrowserFS = require('browserfs');
var bfsPath = BrowserFS.BFSRequire('path');
var nodePath = require('path');
var util_1 = require('./util');
var BFSFS = BrowserFS.BFSRequire('fs');
var ZipFS = BrowserFS.FileSystem.ZipFS;
function win2nix(p) {
    return p.replace(/\\/g, '/');
}
/**
 * Represents a JAR file on the classpath.
 */
var AbstractClasspathJar = (function () {
    function AbstractClasspathJar(path) {
        this._fs = new BFSFS.FS();
        /**
         * Was the JAR file successfully read?
         * - TRUE: JAR file is read and mounted in this._fs.
         * - FALSE: JAR file could not be read.
         * - INDETERMINATE: We have yet to try reading this JAR file.
         */
        this._jarRead = enums_1.TriState.INDETERMINATE;
        this._path = path;
    }
    AbstractClasspathJar.prototype.getPath = function () { return this._path; };
    AbstractClasspathJar.prototype.loadJar = function (cb) {
        var _this = this;
        if (this._jarRead !== enums_1.TriState.TRUE) {
            fs.readFile(this._path, function (e, data) {
                if (e) {
                    _this._jarRead = enums_1.TriState.FALSE;
                    cb(e);
                }
                else {
                    try {
                        ZipFS.computeIndex(data, function (index) {
                            try {
                                _this._fs.initialize(new ZipFS(index, bfsPath.basename(_this._path)));
                                _this._jarRead = enums_1.TriState.TRUE;
                                cb();
                            }
                            catch (e) {
                                _this._jarRead = enums_1.TriState.FALSE;
                                cb(e);
                            }
                        });
                    }
                    catch (e) {
                        _this._jarRead = enums_1.TriState.FALSE;
                        cb(e);
                    }
                }
            });
        }
        else {
            setImmediate(function () { return cb(_this._jarRead === enums_1.TriState.TRUE ? null : new Error("Failed to load JAR file.")); });
        }
    };
    AbstractClasspathJar.prototype.tryLoadClassSync = function (type) {
        if (this._jarRead === enums_1.TriState.TRUE) {
            if (this.hasClass(type) !== enums_1.TriState.FALSE) {
                try {
                    // NOTE: Path must be absolute, otherwise BrowserFS
                    // will try to use process.cwd().
                    return this._fs.readFileSync("/" + type + ".class");
                }
                catch (e) {
                    return null;
                }
            }
            else {
                return null;
            }
        }
        else {
            // Must go the async route.
            return null;
        }
    };
    /**
     * Wrap an operation that depends on the jar being loaded.
     */
    AbstractClasspathJar.prototype._wrapOp = function (op, failCb) {
        var _this = this;
        switch (this._jarRead) {
            case enums_1.TriState.TRUE:
                op();
                break;
            case enums_1.TriState.FALSE:
                setImmediate(function () { return failCb(new Error("Unable to load JAR file.")); });
                break;
            default:
                this.loadJar(function () {
                    _this._wrapOp(op, failCb);
                });
                break;
        }
    };
    /**
     * Wrap a synchronous operation that depends on the jar being loaded.
     * Returns null if the jar isn't loaded, or if the operation fails.
     */
    AbstractClasspathJar.prototype._wrapSyncOp = function (op) {
        if (this._jarRead === enums_1.TriState.TRUE) {
            try {
                return op();
            }
            catch (e) {
                return null;
            }
        }
        else {
            return null;
        }
    };
    AbstractClasspathJar.prototype.loadClass = function (type, cb) {
        var _this = this;
        this._wrapOp(function () {
            // Path must be absolute to avoid relative path issues.
            _this._fs.readFile("/" + type + ".class", cb);
        }, cb);
    };
    AbstractClasspathJar.prototype.statResource = function (p, cb) {
        var _this = this;
        this._wrapOp(function () {
            _this._fs.stat(p, cb);
        }, cb);
    };
    AbstractClasspathJar.prototype.readdir = function (p, cb) {
        var _this = this;
        this._wrapOp(function () {
            _this._fs.readdir(win2nix(p), cb);
        }, cb);
    };
    AbstractClasspathJar.prototype.tryReaddirSync = function (p) {
        var _this = this;
        return this._wrapSyncOp(function () {
            return _this._fs.readdirSync(win2nix(p));
        });
    };
    AbstractClasspathJar.prototype.tryStatSync = function (p) {
        var _this = this;
        return this._wrapSyncOp(function () {
            return _this._fs.statSync(win2nix(p));
        });
    };
    AbstractClasspathJar.prototype.getFS = function () {
        return this._fs.getRootFS();
    };
    return AbstractClasspathJar;
}());
exports.AbstractClasspathJar = AbstractClasspathJar;
/**
 * A JAR item on the classpath that is not in the meta index.
 */
var UnindexedClasspathJar = (function (_super) {
    __extends(UnindexedClasspathJar, _super);
    function UnindexedClasspathJar(p) {
        _super.call(this, p);
        // Contains the list of classes accessible from this classpath item.
        this._classList = null;
    }
    UnindexedClasspathJar.prototype.hasClass = function (type) {
        if (this._jarRead === enums_1.TriState.FALSE) {
            return enums_1.TriState.FALSE;
        }
        else {
            return this._hasClass(type);
        }
    };
    UnindexedClasspathJar.prototype._hasClass = function (type) {
        if (this._classList) {
            return this._classList[type] ? enums_1.TriState.TRUE : enums_1.TriState.FALSE;
        }
        return enums_1.TriState.INDETERMINATE;
    };
    /**
     * Initialize this item on the classpath with the given classlist.
     * @param classes List of classes in pkg/path/Name format.
     */
    UnindexedClasspathJar.prototype.initializeWithClasslist = function (classes) {
        assert_1["default"](this._classList === null, "Initializing a classpath item twice!");
        this._classList = {};
        var len = classes.length;
        for (var i = 0; i < len; i++) {
            this._classList[classes[i]] = true;
        }
    };
    UnindexedClasspathJar.prototype.initialize = function (cb) {
        var _this = this;
        this.loadJar(function (err) {
            if (err) {
                cb();
            }
            else {
                var pathStack = ['/'];
                var classlist = [];
                var fs_1 = _this._fs;
                while (pathStack.length > 0) {
                    var p = pathStack.pop();
                    try {
                        var stat = fs_1.statSync(p);
                        if (stat.isDirectory()) {
                            var listing = fs_1.readdirSync(p);
                            for (var i = 0; i < listing.length; i++) {
                                pathStack.push(bfsPath.join(p, listing[i]));
                            }
                        }
                        else if (bfsPath.extname(p) === '.class') {
                            // Cut off initial / from absolute path.
                            classlist.push(p.slice(1, p.length - 6));
                        }
                    }
                    catch (e) {
                    }
                }
                _this.initializeWithClasslist(classlist);
                cb();
            }
        });
    };
    return UnindexedClasspathJar;
}(AbstractClasspathJar));
exports.UnindexedClasspathJar = UnindexedClasspathJar;
/**
 * A JAR file on the classpath that is in the meta-index.
 */
var IndexedClasspathJar = (function (_super) {
    __extends(IndexedClasspathJar, _super);
    function IndexedClasspathJar(metaIndex, p) {
        _super.call(this, p);
        this._metaIndex = metaIndex;
        this._metaName = bfsPath.basename(p);
    }
    IndexedClasspathJar.prototype.initialize = function (cb) {
        setImmediate(function () { return cb(); });
    };
    IndexedClasspathJar.prototype.hasClass = function (type) {
        if (this._jarRead === enums_1.TriState.FALSE) {
            return enums_1.TriState.FALSE;
        }
        else {
            var pkgComponents = type.split('/');
            var search = this._metaIndex;
            // Pop off class name.
            pkgComponents.pop();
            for (var i = 0; i < pkgComponents.length; i++) {
                var item = search[pkgComponents[i]];
                if (!item) {
                    // item === undefined or false.
                    return enums_1.TriState.FALSE;
                }
                else if (item === true) {
                    return enums_1.TriState.INDETERMINATE;
                }
                else {
                    // Must be an object.
                    search = item;
                }
            }
            // Assume meta-index is complete.
            return enums_1.TriState.FALSE;
        }
    };
    return IndexedClasspathJar;
}(AbstractClasspathJar));
exports.IndexedClasspathJar = IndexedClasspathJar;
/**
 * Represents a folder on the classpath.
 */
var ClasspathFolder = (function () {
    function ClasspathFolder(path) {
        this._path = path;
    }
    ClasspathFolder.prototype.getPath = function () { return this._path; };
    ClasspathFolder.prototype.hasClass = function (type) {
        return enums_1.TriState.INDETERMINATE;
    };
    ClasspathFolder.prototype.initialize = function (cb) {
        // NOP.
        setImmediate(cb);
    };
    ClasspathFolder.prototype.tryLoadClassSync = function (type) {
        try {
            return fs.readFileSync(nodePath.resolve(this._path, type + ".class"));
        }
        catch (e) {
            return null;
        }
    };
    ClasspathFolder.prototype.loadClass = function (type, cb) {
        fs.readFile(nodePath.resolve(this._path, type + ".class"), cb);
    };
    ClasspathFolder.prototype.statResource = function (p, cb) {
        fs.stat(nodePath.resolve(this._path, p), cb);
    };
    ClasspathFolder.prototype.readdir = function (p, cb) {
        fs.readdir(nodePath.resolve(this._path, p), cb);
    };
    ClasspathFolder.prototype.tryReaddirSync = function (p) {
        try {
            return fs.readdirSync(nodePath.resolve(this._path, p));
        }
        catch (e) {
            return null;
        }
    };
    ClasspathFolder.prototype.tryStatSync = function (p) {
        try {
            return fs.statSync(nodePath.resolve(this._path, p));
        }
        catch (e) {
            return null;
        }
    };
    return ClasspathFolder;
}());
exports.ClasspathFolder = ClasspathFolder;
/**
 * Represents a classpath item that cannot be found.
 */
var ClasspathNotFound = (function () {
    function ClasspathNotFound(path) {
        this._path = path;
    }
    ClasspathNotFound.prototype.getPath = function () { return this._path; };
    ClasspathNotFound.prototype.hasClass = function (type) { return enums_1.TriState.FALSE; };
    ClasspathNotFound.prototype.initialize = function (cb) { setImmediate(cb); };
    ClasspathNotFound.prototype.initializeWithClasslist = function (classlist) { };
    ClasspathNotFound.prototype.tryLoadClassSync = function (type) { return null; };
    ClasspathNotFound.prototype._notFoundError = function (cb) { setImmediate(function () { return cb(new Error("Class cannot be found.")); }); };
    ClasspathNotFound.prototype.loadClass = function (type, cb) { this._notFoundError(cb); };
    ClasspathNotFound.prototype.statResource = function (p, cb) { this._notFoundError(cb); };
    ClasspathNotFound.prototype.readdir = function (p, cb) { this._notFoundError(cb); };
    ClasspathNotFound.prototype.tryReaddirSync = function (p) { return null; };
    ClasspathNotFound.prototype.tryStatSync = function (p) { return null; };
    return ClasspathNotFound;
}());
exports.ClasspathNotFound = ClasspathNotFound;
/**
 * Parse the meta index into a lookup table from package name (with slashes) to JAR file.
 * Returns a tuple of JAR files in the meta index and the meta index.
 */
function parseMetaIndex(metaIndex) {
    var lines = metaIndex.split("\n");
    var rv = {};
    var currentJar = null;
    for (var i = 0; i < lines.length; i++) {
        var line = lines[i];
        if (line.length > 0) {
            switch (line[0]) {
                case '%':
                case '@':
                    // Comment or resource-only JAR file.
                    continue;
                case '!':
                case '#':
                    // JAR file w/ classes.
                    // Skip symbol and space.
                    var jarName = line.slice(2);
                    rv[jarName] = currentJar = {};
                    break;
                default:
                    // Package name. If it ends with /, then it's shared
                    // amongst multiple JAR files.
                    // We don't treat those separately, though, so standardize it.
                    if (line[line.length - 1] === '/') {
                        line = line.slice(0, line.length - 1);
                    }
                    var pkgComponents = line.split('/');
                    var current = currentJar;
                    var i_1 = void 0;
                    for (i_1 = 0; i_1 < pkgComponents.length - 1; i_1++) {
                        var cmp = pkgComponents[i_1], next = current[cmp];
                        if (!next) {
                            current = current[cmp] = {};
                        }
                        else {
                            // Invariant: You can't list a package and its subpackages
                            // for same jar file. Thus, current[cmp] cannot be a boolean.
                            current = current[cmp];
                        }
                    }
                    current[pkgComponents[i_1]] = true;
                    break;
            }
        }
    }
    return rv;
}
/**
 * Given a list of paths (which may or may not exist), produces a list of
 * classpath objects.
 */
function ClasspathFactory(javaHomePath, paths, cb) {
    var classpathItems = new Array(paths.length), i = 0;
    fs.readFile(nodePath.join(javaHomePath, 'lib', 'meta-index'), function (err, data) {
        var metaIndex = {};
        if (!err) {
            metaIndex = parseMetaIndex(data.toString());
        }
        util_1.asyncForEach(paths, function (p, nextItem) {
            var pRelToHome = nodePath.relative(javaHomePath + "/lib", p);
            fs.stat(p, function (err, stats) {
                var cpItem;
                if (err) {
                    cpItem = new ClasspathNotFound(p);
                }
                else if (stats.isDirectory()) {
                    cpItem = new ClasspathFolder(p);
                }
                else {
                    if (metaIndex[pRelToHome]) {
                        cpItem = new IndexedClasspathJar(metaIndex[pRelToHome], p);
                    }
                    else {
                        cpItem = new UnindexedClasspathJar(p);
                    }
                }
                classpathItems[i++] = cpItem;
                cpItem.initialize(nextItem);
            });
        }, function (e) {
            cb(classpathItems);
        });
    });
}
exports.ClasspathFactory = ClasspathFactory;
//# sourceMappingURL=data:application/json;base64,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