"use strict";
/***
This is modified from part of jsdifflib v1.0. <http://snowtide.com/jsdifflib>

Copyright (c) 2007, Snowtide Informatics Systems, Inc.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this
        list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
    * Neither the name of the Snowtide Informatics Systems nor the names of its
        contributors may be used to endorse or promote products derived from this
        software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
***/
/* Original author: Chas Emerick <cemerick@snowtide.com> */
function text_diff(a_lines, b_lines, context) {
    return (new SequenceMatcher(a_lines, b_lines)).text_diff(context);
}
exports.text_diff = text_diff;
// comparison function for sorting lists of numeric tuples
function __ntuplecomp(a, b) {
    var mlen = Math.max(a.length, b.length);
    for (var i = 0; i < mlen; i++) {
        if (a[i] < b[i])
            return -1;
        if (a[i] > b[i])
            return 1;
    }
    return a.length == b.length ? 0 : (a.length < b.length ? -1 : 1);
}
// replacement for python's dict.get function -- need easy default values
function __dictget(dict, key, defaultValue) {
    return dict.hasOwnProperty(key) ? dict[key] : defaultValue;
}
var SequenceMatcher = (function () {
    function SequenceMatcher(a, b) {
        this.a = a;
        this.b = b;
        this.b2j = {};
        for (var i = 0; i < b.length; i++) {
            var elt = b[i];
            if (this.b2j.hasOwnProperty(elt)) {
                this.b2j[elt].push(i);
            }
            else {
                this.b2j[elt] = [i];
            }
        }
    }
    SequenceMatcher.prototype.find_longest_match = function (alo, ahi, blo, bhi) {
        var a = this.a;
        var b = this.b;
        var b2j = this.b2j;
        var besti = alo;
        var bestj = blo;
        var bestsize = 0;
        var j2len = {};
        for (var i = alo; i < ahi; i++) {
            var newj2len = {};
            var jdict = __dictget(b2j, a[i], []);
            for (var jkey in jdict) {
                if (jdict.hasOwnProperty(jkey)) {
                    var j = jdict[jkey];
                    if (j < blo)
                        continue;
                    if (j >= bhi)
                        break;
                    var k = __dictget(j2len, j - 1, 0) + 1;
                    newj2len[j] = k;
                    if (k > bestsize) {
                        besti = i - k + 1;
                        bestj = j - k + 1;
                        bestsize = k;
                    }
                }
            }
            j2len = newj2len;
        }
        while (besti > alo && bestj > blo && a[besti - 1] == b[bestj - 1]) {
            besti--;
            bestj--;
            bestsize++;
        }
        while (besti + bestsize < ahi && bestj + bestsize < bhi &&
            a[besti + bestsize] == b[bestj + bestsize]) {
            bestsize++;
        }
        return [besti, bestj, bestsize];
    };
    SequenceMatcher.prototype.get_matching_blocks = function () {
        if (this.matching_blocks != null)
            return this.matching_blocks;
        var la = this.a.length;
        var lb = this.b.length;
        var queue = [[0, la, 0, lb]];
        var matching_blocks = [];
        while (queue.length) {
            var qi = queue.pop();
            var alo = qi[0];
            var ahi = qi[1];
            var blo = qi[2];
            var bhi = qi[3];
            var x = this.find_longest_match(alo, ahi, blo, bhi);
            var i = x[0];
            var j = x[1];
            var k = x[2];
            if (k) {
                matching_blocks.push(x);
                if (alo < i && blo < j)
                    queue.push([alo, i, blo, j]);
                if (i + k < ahi && j + k < bhi)
                    queue.push([i + k, ahi, j + k, bhi]);
            }
        }
        matching_blocks.sort(__ntuplecomp);
        var i1 = 0, j1 = 0, k1 = 0;
        var non_adjacent = [];
        for (var idx = 0; idx < matching_blocks.length; idx++) {
            var block = matching_blocks[idx];
            var i2 = block[0];
            var j2 = block[1];
            var k2 = block[2];
            if (i1 + k1 == i2 && j1 + k1 == j2) {
                k1 += k2;
            }
            else {
                if (k1)
                    non_adjacent.push([i1, j1, k1]);
                i1 = i2;
                j1 = j2;
                k1 = k2;
            }
        }
        if (k1)
            non_adjacent.push([i1, j1, k1]);
        non_adjacent.push([la, lb, 0]);
        this.matching_blocks = non_adjacent;
        return this.matching_blocks;
    };
    SequenceMatcher.prototype.get_opcodes = function () {
        if (this.opcodes != null)
            return this.opcodes;
        var i = 0;
        var j = 0;
        var answer = [];
        this.opcodes = answer;
        var blocks = this.get_matching_blocks();
        for (var idx = 0; idx < blocks.length; idx++) {
            var block = blocks[idx];
            var ai = block[0];
            var bj = block[1];
            var size = block[2];
            var tag = '';
            if (i < ai && j < bj) {
                tag = 'replace';
            }
            else if (i < ai) {
                tag = 'delete';
            }
            else if (j < bj) {
                tag = 'insert';
            }
            if (tag)
                answer.push([tag, i, ai, j, bj]);
            i = ai + size;
            j = bj + size;
            if (size)
                answer.push(['equal', ai, i, bj, j]);
        }
        return answer;
    };
    SequenceMatcher.prototype.text_diff = function (context) {
        var opcodes = this.get_opcodes();
        var diff = [];
        var a_side = [];
        var b_side = [];
        var a_max_len = 0;
        var last_seen = -1;
        for (var op_idx = 0; op_idx < opcodes.length; op_idx++) {
            var op = opcodes[op_idx];
            if (op[0] === 'equal')
                continue;
            var ai = op[1];
            var bi = op[3];
            var aj = op[2] - 1;
            var bj = op[4] - 1;
            var start = Math.min(ai, bi);
            var end = Math.max(aj, bj);
            var c = '';
            switch (op[0]) {
                case 'delete':
                    c = ' < ';
                    break;
                case 'insert':
                    c = ' > ';
                    break;
                case 'replace':
                    c = ' | ';
                    break;
            }
            for (var i = Math.max(last_seen + 1, start - context); i < start; i++) {
                var prefix = i + ': ';
                if (i < this.a.length) {
                    a_side.push(prefix + this.a[i]);
                    a_max_len = Math.max(a_max_len, this.a[i].length + prefix.length);
                }
                else {
                    a_side.push(prefix);
                }
                if (i < this.b.length) {
                    b_side.push(this.b[i]);
                }
                else {
                    b_side.push('');
                }
                diff.push('   ');
            }
            for (var i = start; i <= end; i++) {
                var prefix = i + ': ';
                if (i >= ai && i <= aj) {
                    a_side.push(prefix + this.a[i]);
                    a_max_len = Math.max(a_max_len, this.a[i].length + prefix.length);
                }
                else {
                    a_side.push(prefix);
                }
                if (i >= bi && i <= bj) {
                    b_side.push(this.b[i]);
                }
                else {
                    b_side.push('');
                }
                diff.push(c);
            }
            last_seen = end;
        }
        for (var i = 0; i < diff.length; i++) {
            var a = a_side[i];
            var b = b_side[i];
            if (a.length < a_max_len)
                a += (new Array(a_max_len - a.length + 1)).join(' ');
            diff[i] = a + diff[i] + b;
        }
        return diff;
    };
    return SequenceMatcher;
}());
exports.SequenceMatcher = SequenceMatcher;
//# sourceMappingURL=data:application/json;base64,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