"use strict";
/**
 * A class can be in one of these states at any given point in time.
 */
(function (ClassState) {
    // The class has yet to be loaded.
    ClassState[ClassState["NOT_LOADED"] = 0] = "NOT_LOADED";
    // The class's definition has been downloaded and parsed.
    ClassState[ClassState["LOADED"] = 1] = "LOADED";
    // This class and its super classes' definitions have been downloaded and
    // parsed.
    ClassState[ClassState["RESOLVED"] = 2] = "RESOLVED";
    // This class, its super classes', and its interfaces have been downloaded,
    // parsed, and statically initialized.
    ClassState[ClassState["INITIALIZED"] = 3] = "INITIALIZED";
})(exports.ClassState || (exports.ClassState = {}));
var ClassState = exports.ClassState;
/**
 * A thread can be in one of these states at any given point in time.
 *
 * NOTE: When altering ThreadStatus, remember to update the following things.
 *
 * - Thread.validTransitions: Describes each valid thread transition.
 * - sun.misc.VM.getThreadStateValues: Maps ThreadStatus values to Thread.State
 *   values.
 * - Assertion statements in Thread regarding its status.
 */
(function (ThreadStatus) {
    // A thread that has not yet started is in this state.
    ThreadStatus[ThreadStatus["NEW"] = 0] = "NEW";
    // A thread that is able to be run. The thread may actually be running.
    // Query the ThreadPool to determine if this is the case.
    ThreadStatus[ThreadStatus["RUNNABLE"] = 1] = "RUNNABLE";
    // A thread that is blocked waiting for a monitor lock is in this state.
    ThreadStatus[ThreadStatus["BLOCKED"] = 2] = "BLOCKED";
    // A thread that is blocked waiting for a monitor lock that was previously
    // interrupted from waiting on a monitor is in this state.
    // Why? Well, the thread has *already* been interrupted once, but cannot
    // process the interruption until it regains the lock.
    ThreadStatus[ThreadStatus["UNINTERRUPTABLY_BLOCKED"] = 3] = "UNINTERRUPTABLY_BLOCKED";
    // A thread that is waiting indefinitely for another thread to perform a
    // particular action is in this state.
    ThreadStatus[ThreadStatus["WAITING"] = 4] = "WAITING";
    // A thread that is waiting for another thread to perform an action for up to
    // a specified waiting time is in this state.
    ThreadStatus[ThreadStatus["TIMED_WAITING"] = 5] = "TIMED_WAITING";
    // A thread that is waiting for an asynchronous browser operation to complete.
    ThreadStatus[ThreadStatus["ASYNC_WAITING"] = 6] = "ASYNC_WAITING";
    // A thread that is parked.
    ThreadStatus[ThreadStatus["PARKED"] = 7] = "PARKED";
    // A thread that has exited is in this state.
    ThreadStatus[ThreadStatus["TERMINATED"] = 8] = "TERMINATED";
})(exports.ThreadStatus || (exports.ThreadStatus = {}));
var ThreadStatus = exports.ThreadStatus;
/**
 * Java-visible thread state values.
 */
(function (JVMTIThreadState) {
    JVMTIThreadState[JVMTIThreadState["ALIVE"] = 1] = "ALIVE";
    JVMTIThreadState[JVMTIThreadState["TERMINATED"] = 2] = "TERMINATED";
    JVMTIThreadState[JVMTIThreadState["RUNNABLE"] = 4] = "RUNNABLE";
    JVMTIThreadState[JVMTIThreadState["BLOCKED_ON_MONITOR_ENTER"] = 1024] = "BLOCKED_ON_MONITOR_ENTER";
    JVMTIThreadState[JVMTIThreadState["WAITING_INDEFINITELY"] = 16] = "WAITING_INDEFINITELY";
    JVMTIThreadState[JVMTIThreadState["WAITING_WITH_TIMEOUT"] = 32] = "WAITING_WITH_TIMEOUT";
})(exports.JVMTIThreadState || (exports.JVMTIThreadState = {}));
var JVMTIThreadState = exports.JVMTIThreadState;
/**
 * Three-state boolean.
 */
(function (TriState) {
    TriState[TriState["TRUE"] = 0] = "TRUE";
    TriState[TriState["FALSE"] = 1] = "FALSE";
    TriState[TriState["INDETERMINATE"] = 2] = "INDETERMINATE";
})(exports.TriState || (exports.TriState = {}));
var TriState = exports.TriState;
/**
 * The current status of the JVM.
 */
(function (JVMStatus) {
    // The JVM is booting up.
    JVMStatus[JVMStatus["BOOTING"] = 0] = "BOOTING";
    // The JVM is booted, and waiting for a class to run.
    JVMStatus[JVMStatus["BOOTED"] = 1] = "BOOTED";
    // The JVM is running.
    JVMStatus[JVMStatus["RUNNING"] = 2] = "RUNNING";
    // The JVM has completed running, and is performing termination steps.
    JVMStatus[JVMStatus["TERMINATING"] = 3] = "TERMINATING";
    // The JVM is completely finished executing.
    JVMStatus[JVMStatus["TERMINATED"] = 4] = "TERMINATED";
})(exports.JVMStatus || (exports.JVMStatus = {}));
var JVMStatus = exports.JVMStatus;
/**
 * Indicates the type of a stack frame.
 */
(function (StackFrameType) {
    /**
     * A JVM internal stack frame. These should be completely invisible to the
     * JVM program.
     */
    StackFrameType[StackFrameType["INTERNAL"] = 0] = "INTERNAL";
    /**
     * A bytecode method's stack frame. These have an actual stack.
     */
    StackFrameType[StackFrameType["BYTECODE"] = 1] = "BYTECODE";
    /**
     * A native method's stack frame. These typically consist of just a JavaScript
     * function and a method association.
     */
    StackFrameType[StackFrameType["NATIVE"] = 2] = "NATIVE";
})(exports.StackFrameType || (exports.StackFrameType = {}));
var StackFrameType = exports.StackFrameType;
/**
 * Various constant values. Enum'd so they are inlined by the TypeScript
 * compiler.
 */
(function (Constants) {
    Constants[Constants["INT_MAX"] = Math.pow(2, 31) - 1] = "INT_MAX";
    Constants[Constants["INT_MIN"] = -Constants.INT_MAX - 1] = "INT_MIN";
    Constants[Constants["FLOAT_POS_INFINITY"] = Math.pow(2, 128)] = "FLOAT_POS_INFINITY";
    Constants[Constants["FLOAT_NEG_INFINITY"] = -1 * Constants.FLOAT_POS_INFINITY] = "FLOAT_NEG_INFINITY";
    Constants[Constants["FLOAT_POS_INFINITY_AS_INT"] = 2139095040] = "FLOAT_POS_INFINITY_AS_INT";
    Constants[Constants["FLOAT_NEG_INFINITY_AS_INT"] = -8388608] = "FLOAT_NEG_INFINITY_AS_INT";
    // We use the JavaScript NaN as our NaN value, and convert it to
    // a NaN value in the SNaN range when an int equivalent is requested.
    Constants[Constants["FLOAT_NaN_AS_INT"] = 2143289344] = "FLOAT_NaN_AS_INT";
})(exports.Constants || (exports.Constants = {}));
var Constants = exports.Constants;
/**
 * Integer indicating the type of a constant pool item.
 * @url https://docs.oracle.com/javase/specs/jvms/se7/html/jvms-4.html#jvms-4.4-140
 */
(function (ConstantPoolItemType) {
    ConstantPoolItemType[ConstantPoolItemType["CLASS"] = 7] = "CLASS";
    ConstantPoolItemType[ConstantPoolItemType["FIELDREF"] = 9] = "FIELDREF";
    ConstantPoolItemType[ConstantPoolItemType["METHODREF"] = 10] = "METHODREF";
    ConstantPoolItemType[ConstantPoolItemType["INTERFACE_METHODREF"] = 11] = "INTERFACE_METHODREF";
    ConstantPoolItemType[ConstantPoolItemType["STRING"] = 8] = "STRING";
    ConstantPoolItemType[ConstantPoolItemType["INTEGER"] = 3] = "INTEGER";
    ConstantPoolItemType[ConstantPoolItemType["FLOAT"] = 4] = "FLOAT";
    ConstantPoolItemType[ConstantPoolItemType["LONG"] = 5] = "LONG";
    ConstantPoolItemType[ConstantPoolItemType["DOUBLE"] = 6] = "DOUBLE";
    ConstantPoolItemType[ConstantPoolItemType["NAME_AND_TYPE"] = 12] = "NAME_AND_TYPE";
    ConstantPoolItemType[ConstantPoolItemType["UTF8"] = 1] = "UTF8";
    ConstantPoolItemType[ConstantPoolItemType["METHOD_HANDLE"] = 15] = "METHOD_HANDLE";
    ConstantPoolItemType[ConstantPoolItemType["METHOD_TYPE"] = 16] = "METHOD_TYPE";
    ConstantPoolItemType[ConstantPoolItemType["INVOKE_DYNAMIC"] = 18] = "INVOKE_DYNAMIC";
})(exports.ConstantPoolItemType || (exports.ConstantPoolItemType = {}));
var ConstantPoolItemType = exports.ConstantPoolItemType;
/**
 * Integer indicating the type of a StackMapTable entry.
 * @see https://docs.oracle.com/javase/specs/jvms/se7/html/jvms-4.html#jvms-4.7.4
 */
(function (StackMapTableEntryType) {
    StackMapTableEntryType[StackMapTableEntryType["SAME_FRAME"] = 0] = "SAME_FRAME";
    StackMapTableEntryType[StackMapTableEntryType["SAME_LOCALS_1_STACK_ITEM_FRAME"] = 1] = "SAME_LOCALS_1_STACK_ITEM_FRAME";
    StackMapTableEntryType[StackMapTableEntryType["SAME_LOCALS_1_STACK_ITEM_FRAME_EXTENDED"] = 2] = "SAME_LOCALS_1_STACK_ITEM_FRAME_EXTENDED";
    StackMapTableEntryType[StackMapTableEntryType["CHOP_FRAME"] = 3] = "CHOP_FRAME";
    StackMapTableEntryType[StackMapTableEntryType["SAME_FRAME_EXTENDED"] = 4] = "SAME_FRAME_EXTENDED";
    StackMapTableEntryType[StackMapTableEntryType["APPEND_FRAME"] = 5] = "APPEND_FRAME";
    StackMapTableEntryType[StackMapTableEntryType["FULL_FRAME"] = 6] = "FULL_FRAME";
})(exports.StackMapTableEntryType || (exports.StackMapTableEntryType = {}));
var StackMapTableEntryType = exports.StackMapTableEntryType;
/**
 * Integer indicating the reference type of a MethodHandle item in the constant
 * pool.
 * @see https://docs.oracle.com/javase/specs/jvms/se7/html/jvms-4.html#jvms-4.4.8
 */
(function (MethodHandleReferenceKind) {
    MethodHandleReferenceKind[MethodHandleReferenceKind["GETFIELD"] = 1] = "GETFIELD";
    MethodHandleReferenceKind[MethodHandleReferenceKind["GETSTATIC"] = 2] = "GETSTATIC";
    MethodHandleReferenceKind[MethodHandleReferenceKind["PUTFIELD"] = 3] = "PUTFIELD";
    MethodHandleReferenceKind[MethodHandleReferenceKind["PUTSTATIC"] = 4] = "PUTSTATIC";
    MethodHandleReferenceKind[MethodHandleReferenceKind["INVOKEVIRTUAL"] = 5] = "INVOKEVIRTUAL";
    MethodHandleReferenceKind[MethodHandleReferenceKind["INVOKESTATIC"] = 6] = "INVOKESTATIC";
    MethodHandleReferenceKind[MethodHandleReferenceKind["INVOKESPECIAL"] = 7] = "INVOKESPECIAL";
    MethodHandleReferenceKind[MethodHandleReferenceKind["NEWINVOKESPECIAL"] = 8] = "NEWINVOKESPECIAL";
    MethodHandleReferenceKind[MethodHandleReferenceKind["INVOKEINTERFACE"] = 9] = "INVOKEINTERFACE";
})(exports.MethodHandleReferenceKind || (exports.MethodHandleReferenceKind = {}));
var MethodHandleReferenceKind = exports.MethodHandleReferenceKind;
/**
 * JVM op codes. The enum value corresponds to that opcode's value.
 */
(function (OpCode) {
    OpCode[OpCode["AALOAD"] = 50] = "AALOAD";
    OpCode[OpCode["AASTORE"] = 83] = "AASTORE";
    OpCode[OpCode["ACONST_NULL"] = 1] = "ACONST_NULL";
    OpCode[OpCode["ALOAD"] = 25] = "ALOAD";
    OpCode[OpCode["ALOAD_0"] = 42] = "ALOAD_0";
    OpCode[OpCode["ALOAD_1"] = 43] = "ALOAD_1";
    OpCode[OpCode["ALOAD_2"] = 44] = "ALOAD_2";
    OpCode[OpCode["ALOAD_3"] = 45] = "ALOAD_3";
    OpCode[OpCode["ANEWARRAY"] = 189] = "ANEWARRAY";
    OpCode[OpCode["ARETURN"] = 176] = "ARETURN";
    OpCode[OpCode["ARRAYLENGTH"] = 190] = "ARRAYLENGTH";
    OpCode[OpCode["ASTORE"] = 58] = "ASTORE";
    OpCode[OpCode["ASTORE_0"] = 75] = "ASTORE_0";
    OpCode[OpCode["ASTORE_1"] = 76] = "ASTORE_1";
    OpCode[OpCode["ASTORE_2"] = 77] = "ASTORE_2";
    OpCode[OpCode["ASTORE_3"] = 78] = "ASTORE_3";
    OpCode[OpCode["ATHROW"] = 191] = "ATHROW";
    OpCode[OpCode["BALOAD"] = 51] = "BALOAD";
    OpCode[OpCode["BASTORE"] = 84] = "BASTORE";
    OpCode[OpCode["BIPUSH"] = 16] = "BIPUSH";
    OpCode[OpCode["BREAKPOINT"] = 202] = "BREAKPOINT";
    OpCode[OpCode["CALOAD"] = 52] = "CALOAD";
    OpCode[OpCode["CASTORE"] = 85] = "CASTORE";
    OpCode[OpCode["CHECKCAST"] = 192] = "CHECKCAST";
    OpCode[OpCode["D2F"] = 144] = "D2F";
    OpCode[OpCode["D2I"] = 142] = "D2I";
    OpCode[OpCode["D2L"] = 143] = "D2L";
    OpCode[OpCode["DADD"] = 99] = "DADD";
    OpCode[OpCode["DALOAD"] = 49] = "DALOAD";
    OpCode[OpCode["DASTORE"] = 82] = "DASTORE";
    OpCode[OpCode["DCMPG"] = 152] = "DCMPG";
    OpCode[OpCode["DCMPL"] = 151] = "DCMPL";
    OpCode[OpCode["DCONST_0"] = 14] = "DCONST_0";
    OpCode[OpCode["DCONST_1"] = 15] = "DCONST_1";
    OpCode[OpCode["DDIV"] = 111] = "DDIV";
    OpCode[OpCode["DLOAD"] = 24] = "DLOAD";
    OpCode[OpCode["DLOAD_0"] = 38] = "DLOAD_0";
    OpCode[OpCode["DLOAD_1"] = 39] = "DLOAD_1";
    OpCode[OpCode["DLOAD_2"] = 40] = "DLOAD_2";
    OpCode[OpCode["DLOAD_3"] = 41] = "DLOAD_3";
    OpCode[OpCode["DMUL"] = 107] = "DMUL";
    OpCode[OpCode["DNEG"] = 119] = "DNEG";
    OpCode[OpCode["DREM"] = 115] = "DREM";
    OpCode[OpCode["DRETURN"] = 175] = "DRETURN";
    OpCode[OpCode["DSTORE"] = 57] = "DSTORE";
    OpCode[OpCode["DSTORE_0"] = 71] = "DSTORE_0";
    OpCode[OpCode["DSTORE_1"] = 72] = "DSTORE_1";
    OpCode[OpCode["DSTORE_2"] = 73] = "DSTORE_2";
    OpCode[OpCode["DSTORE_3"] = 74] = "DSTORE_3";
    OpCode[OpCode["DSUB"] = 103] = "DSUB";
    OpCode[OpCode["DUP"] = 89] = "DUP";
    OpCode[OpCode["DUP_X1"] = 90] = "DUP_X1";
    OpCode[OpCode["DUP_X2"] = 91] = "DUP_X2";
    OpCode[OpCode["DUP2"] = 92] = "DUP2";
    OpCode[OpCode["DUP2_X1"] = 93] = "DUP2_X1";
    OpCode[OpCode["DUP2_X2"] = 94] = "DUP2_X2";
    OpCode[OpCode["F2D"] = 141] = "F2D";
    OpCode[OpCode["F2I"] = 139] = "F2I";
    OpCode[OpCode["F2L"] = 140] = "F2L";
    OpCode[OpCode["FADD"] = 98] = "FADD";
    OpCode[OpCode["FALOAD"] = 48] = "FALOAD";
    OpCode[OpCode["FASTORE"] = 81] = "FASTORE";
    OpCode[OpCode["FCMPG"] = 150] = "FCMPG";
    OpCode[OpCode["FCMPL"] = 149] = "FCMPL";
    OpCode[OpCode["FCONST_0"] = 11] = "FCONST_0";
    OpCode[OpCode["FCONST_1"] = 12] = "FCONST_1";
    OpCode[OpCode["FCONST_2"] = 13] = "FCONST_2";
    OpCode[OpCode["FDIV"] = 110] = "FDIV";
    OpCode[OpCode["FLOAD"] = 23] = "FLOAD";
    OpCode[OpCode["FLOAD_0"] = 34] = "FLOAD_0";
    OpCode[OpCode["FLOAD_1"] = 35] = "FLOAD_1";
    OpCode[OpCode["FLOAD_2"] = 36] = "FLOAD_2";
    OpCode[OpCode["FLOAD_3"] = 37] = "FLOAD_3";
    OpCode[OpCode["FMUL"] = 106] = "FMUL";
    OpCode[OpCode["FNEG"] = 118] = "FNEG";
    OpCode[OpCode["FREM"] = 114] = "FREM";
    OpCode[OpCode["FRETURN"] = 174] = "FRETURN";
    OpCode[OpCode["FSTORE"] = 56] = "FSTORE";
    OpCode[OpCode["FSTORE_0"] = 67] = "FSTORE_0";
    OpCode[OpCode["FSTORE_1"] = 68] = "FSTORE_1";
    OpCode[OpCode["FSTORE_2"] = 69] = "FSTORE_2";
    OpCode[OpCode["FSTORE_3"] = 70] = "FSTORE_3";
    OpCode[OpCode["FSUB"] = 102] = "FSUB";
    OpCode[OpCode["GETFIELD"] = 180] = "GETFIELD";
    OpCode[OpCode["GETSTATIC"] = 178] = "GETSTATIC";
    OpCode[OpCode["GOTO"] = 167] = "GOTO";
    OpCode[OpCode["GOTO_W"] = 200] = "GOTO_W";
    OpCode[OpCode["I2B"] = 145] = "I2B";
    OpCode[OpCode["I2C"] = 146] = "I2C";
    OpCode[OpCode["I2D"] = 135] = "I2D";
    OpCode[OpCode["I2F"] = 134] = "I2F";
    OpCode[OpCode["I2L"] = 133] = "I2L";
    OpCode[OpCode["I2S"] = 147] = "I2S";
    OpCode[OpCode["IADD"] = 96] = "IADD";
    OpCode[OpCode["IALOAD"] = 46] = "IALOAD";
    OpCode[OpCode["IAND"] = 126] = "IAND";
    OpCode[OpCode["IASTORE"] = 79] = "IASTORE";
    OpCode[OpCode["ICONST_M1"] = 2] = "ICONST_M1";
    OpCode[OpCode["ICONST_0"] = 3] = "ICONST_0";
    OpCode[OpCode["ICONST_1"] = 4] = "ICONST_1";
    OpCode[OpCode["ICONST_2"] = 5] = "ICONST_2";
    OpCode[OpCode["ICONST_3"] = 6] = "ICONST_3";
    OpCode[OpCode["ICONST_4"] = 7] = "ICONST_4";
    OpCode[OpCode["ICONST_5"] = 8] = "ICONST_5";
    OpCode[OpCode["IDIV"] = 108] = "IDIV";
    OpCode[OpCode["IF_ACMPEQ"] = 165] = "IF_ACMPEQ";
    OpCode[OpCode["IF_ACMPNE"] = 166] = "IF_ACMPNE";
    OpCode[OpCode["IF_ICMPEQ"] = 159] = "IF_ICMPEQ";
    OpCode[OpCode["IF_ICMPGE"] = 162] = "IF_ICMPGE";
    OpCode[OpCode["IF_ICMPGT"] = 163] = "IF_ICMPGT";
    OpCode[OpCode["IF_ICMPLE"] = 164] = "IF_ICMPLE";
    OpCode[OpCode["IF_ICMPLT"] = 161] = "IF_ICMPLT";
    OpCode[OpCode["IF_ICMPNE"] = 160] = "IF_ICMPNE";
    OpCode[OpCode["IFEQ"] = 153] = "IFEQ";
    OpCode[OpCode["IFGE"] = 156] = "IFGE";
    OpCode[OpCode["IFGT"] = 157] = "IFGT";
    OpCode[OpCode["IFLE"] = 158] = "IFLE";
    OpCode[OpCode["IFLT"] = 155] = "IFLT";
    OpCode[OpCode["IFNE"] = 154] = "IFNE";
    OpCode[OpCode["IFNONNULL"] = 199] = "IFNONNULL";
    OpCode[OpCode["IFNULL"] = 198] = "IFNULL";
    OpCode[OpCode["IINC"] = 132] = "IINC";
    OpCode[OpCode["ILOAD"] = 21] = "ILOAD";
    OpCode[OpCode["ILOAD_0"] = 26] = "ILOAD_0";
    OpCode[OpCode["ILOAD_1"] = 27] = "ILOAD_1";
    OpCode[OpCode["ILOAD_2"] = 28] = "ILOAD_2";
    OpCode[OpCode["ILOAD_3"] = 29] = "ILOAD_3";
    // IMPDEP1 = 0xfe,
    // IMPDEP2 = 0xff,
    OpCode[OpCode["IMUL"] = 104] = "IMUL";
    OpCode[OpCode["INEG"] = 116] = "INEG";
    OpCode[OpCode["INSTANCEOF"] = 193] = "INSTANCEOF";
    OpCode[OpCode["INVOKEDYNAMIC"] = 186] = "INVOKEDYNAMIC";
    OpCode[OpCode["INVOKEINTERFACE"] = 185] = "INVOKEINTERFACE";
    OpCode[OpCode["INVOKESPECIAL"] = 183] = "INVOKESPECIAL";
    OpCode[OpCode["INVOKESTATIC"] = 184] = "INVOKESTATIC";
    OpCode[OpCode["INVOKEVIRTUAL"] = 182] = "INVOKEVIRTUAL";
    OpCode[OpCode["IOR"] = 128] = "IOR";
    OpCode[OpCode["IREM"] = 112] = "IREM";
    OpCode[OpCode["IRETURN"] = 172] = "IRETURN";
    OpCode[OpCode["ISHL"] = 120] = "ISHL";
    OpCode[OpCode["ISHR"] = 122] = "ISHR";
    OpCode[OpCode["ISTORE"] = 54] = "ISTORE";
    OpCode[OpCode["ISTORE_0"] = 59] = "ISTORE_0";
    OpCode[OpCode["ISTORE_1"] = 60] = "ISTORE_1";
    OpCode[OpCode["ISTORE_2"] = 61] = "ISTORE_2";
    OpCode[OpCode["ISTORE_3"] = 62] = "ISTORE_3";
    OpCode[OpCode["ISUB"] = 100] = "ISUB";
    OpCode[OpCode["IUSHR"] = 124] = "IUSHR";
    OpCode[OpCode["IXOR"] = 130] = "IXOR";
    OpCode[OpCode["JSR"] = 168] = "JSR";
    OpCode[OpCode["JSR_W"] = 201] = "JSR_W";
    OpCode[OpCode["L2D"] = 138] = "L2D";
    OpCode[OpCode["L2F"] = 137] = "L2F";
    OpCode[OpCode["L2I"] = 136] = "L2I";
    OpCode[OpCode["LADD"] = 97] = "LADD";
    OpCode[OpCode["LALOAD"] = 47] = "LALOAD";
    OpCode[OpCode["LAND"] = 127] = "LAND";
    OpCode[OpCode["LASTORE"] = 80] = "LASTORE";
    OpCode[OpCode["LCMP"] = 148] = "LCMP";
    OpCode[OpCode["LCONST_0"] = 9] = "LCONST_0";
    OpCode[OpCode["LCONST_1"] = 10] = "LCONST_1";
    OpCode[OpCode["LDC"] = 18] = "LDC";
    OpCode[OpCode["LDC_W"] = 19] = "LDC_W";
    OpCode[OpCode["LDC2_W"] = 20] = "LDC2_W";
    OpCode[OpCode["LDIV"] = 109] = "LDIV";
    OpCode[OpCode["LLOAD"] = 22] = "LLOAD";
    OpCode[OpCode["LLOAD_0"] = 30] = "LLOAD_0";
    OpCode[OpCode["LLOAD_1"] = 31] = "LLOAD_1";
    OpCode[OpCode["LLOAD_2"] = 32] = "LLOAD_2";
    OpCode[OpCode["LLOAD_3"] = 33] = "LLOAD_3";
    OpCode[OpCode["LMUL"] = 105] = "LMUL";
    OpCode[OpCode["LNEG"] = 117] = "LNEG";
    OpCode[OpCode["LOOKUPSWITCH"] = 171] = "LOOKUPSWITCH";
    OpCode[OpCode["LOR"] = 129] = "LOR";
    OpCode[OpCode["LREM"] = 113] = "LREM";
    OpCode[OpCode["LRETURN"] = 173] = "LRETURN";
    OpCode[OpCode["LSHL"] = 121] = "LSHL";
    OpCode[OpCode["LSHR"] = 123] = "LSHR";
    OpCode[OpCode["LSTORE"] = 55] = "LSTORE";
    OpCode[OpCode["LSTORE_0"] = 63] = "LSTORE_0";
    OpCode[OpCode["LSTORE_1"] = 64] = "LSTORE_1";
    OpCode[OpCode["LSTORE_2"] = 65] = "LSTORE_2";
    OpCode[OpCode["LSTORE_3"] = 66] = "LSTORE_3";
    OpCode[OpCode["LSUB"] = 101] = "LSUB";
    OpCode[OpCode["LUSHR"] = 125] = "LUSHR";
    OpCode[OpCode["LXOR"] = 131] = "LXOR";
    OpCode[OpCode["MONITORENTER"] = 194] = "MONITORENTER";
    OpCode[OpCode["MONITOREXIT"] = 195] = "MONITOREXIT";
    OpCode[OpCode["MULTIANEWARRAY"] = 197] = "MULTIANEWARRAY";
    OpCode[OpCode["NEW"] = 187] = "NEW";
    OpCode[OpCode["NEWARRAY"] = 188] = "NEWARRAY";
    OpCode[OpCode["NOP"] = 0] = "NOP";
    OpCode[OpCode["POP"] = 87] = "POP";
    OpCode[OpCode["POP2"] = 88] = "POP2";
    OpCode[OpCode["PUTFIELD"] = 181] = "PUTFIELD";
    OpCode[OpCode["PUTSTATIC"] = 179] = "PUTSTATIC";
    OpCode[OpCode["RET"] = 169] = "RET";
    OpCode[OpCode["RETURN"] = 177] = "RETURN";
    OpCode[OpCode["SALOAD"] = 53] = "SALOAD";
    OpCode[OpCode["SASTORE"] = 86] = "SASTORE";
    OpCode[OpCode["SIPUSH"] = 17] = "SIPUSH";
    OpCode[OpCode["SWAP"] = 95] = "SWAP";
    OpCode[OpCode["TABLESWITCH"] = 170] = "TABLESWITCH";
    OpCode[OpCode["WIDE"] = 196] = "WIDE";
    // Special Doppio 'fast' opcodes
    OpCode[OpCode["GETSTATIC_FAST32"] = 208] = "GETSTATIC_FAST32";
    OpCode[OpCode["GETSTATIC_FAST64"] = 209] = "GETSTATIC_FAST64";
    OpCode[OpCode["NEW_FAST"] = 210] = "NEW_FAST";
    OpCode[OpCode["ANEWARRAY_FAST"] = 213] = "ANEWARRAY_FAST";
    OpCode[OpCode["CHECKCAST_FAST"] = 214] = "CHECKCAST_FAST";
    OpCode[OpCode["INSTANCEOF_FAST"] = 215] = "INSTANCEOF_FAST";
    OpCode[OpCode["MULTIANEWARRAY_FAST"] = 216] = "MULTIANEWARRAY_FAST";
    OpCode[OpCode["PUTSTATIC_FAST32"] = 217] = "PUTSTATIC_FAST32";
    OpCode[OpCode["PUTSTATIC_FAST64"] = 218] = "PUTSTATIC_FAST64";
    OpCode[OpCode["GETFIELD_FAST32"] = 219] = "GETFIELD_FAST32";
    OpCode[OpCode["GETFIELD_FAST64"] = 220] = "GETFIELD_FAST64";
    OpCode[OpCode["PUTFIELD_FAST32"] = 221] = "PUTFIELD_FAST32";
    OpCode[OpCode["PUTFIELD_FAST64"] = 222] = "PUTFIELD_FAST64";
    OpCode[OpCode["INVOKENONVIRTUAL_FAST"] = 223] = "INVOKENONVIRTUAL_FAST";
    OpCode[OpCode["INVOKESTATIC_FAST"] = 240] = "INVOKESTATIC_FAST";
    OpCode[OpCode["INVOKEVIRTUAL_FAST"] = 241] = "INVOKEVIRTUAL_FAST";
    OpCode[OpCode["INVOKEINTERFACE_FAST"] = 242] = "INVOKEINTERFACE_FAST";
    OpCode[OpCode["INVOKEHANDLE"] = 243] = "INVOKEHANDLE";
    OpCode[OpCode["INVOKEBASIC"] = 244] = "INVOKEBASIC";
    OpCode[OpCode["LINKTOSPECIAL"] = 245] = "LINKTOSPECIAL";
    OpCode[OpCode["LINKTOVIRTUAL"] = 247] = "LINKTOVIRTUAL";
    OpCode[OpCode["INVOKEDYNAMIC_FAST"] = 248] = "INVOKEDYNAMIC_FAST";
})(exports.OpCode || (exports.OpCode = {}));
var OpCode = exports.OpCode;
(function (OpcodeLayoutType) {
    OpcodeLayoutType[OpcodeLayoutType["OPCODE_ONLY"] = 0] = "OPCODE_ONLY";
    OpcodeLayoutType[OpcodeLayoutType["CONSTANT_POOL_UINT8"] = 1] = "CONSTANT_POOL_UINT8";
    OpcodeLayoutType[OpcodeLayoutType["CONSTANT_POOL"] = 2] = "CONSTANT_POOL";
    OpcodeLayoutType[OpcodeLayoutType["CONSTANT_POOL_AND_UINT8_VALUE"] = 3] = "CONSTANT_POOL_AND_UINT8_VALUE";
    OpcodeLayoutType[OpcodeLayoutType["UINT8_VALUE"] = 4] = "UINT8_VALUE";
    OpcodeLayoutType[OpcodeLayoutType["UINT8_AND_INT8_VALUE"] = 5] = "UINT8_AND_INT8_VALUE";
    OpcodeLayoutType[OpcodeLayoutType["INT8_VALUE"] = 6] = "INT8_VALUE";
    OpcodeLayoutType[OpcodeLayoutType["INT16_VALUE"] = 7] = "INT16_VALUE";
    OpcodeLayoutType[OpcodeLayoutType["INT32_VALUE"] = 8] = "INT32_VALUE";
    // LOOKUPSWITCH,
    // TABLESWITCH,
    OpcodeLayoutType[OpcodeLayoutType["ARRAY_TYPE"] = 9] = "ARRAY_TYPE";
    OpcodeLayoutType[OpcodeLayoutType["WIDE"] = 10] = "WIDE";
})(exports.OpcodeLayoutType || (exports.OpcodeLayoutType = {}));
var OpcodeLayoutType = exports.OpcodeLayoutType;
// Contains the opcode layout types for each valid opcode.
// To conserve code space, it's assumed all opcodes not in the table
// are OPCODE_ONLY.
var olt = new Array(0xff);
(function () {
    for (var i = 0; i < 0xff; i++) {
        olt[i] = OpcodeLayoutType.OPCODE_ONLY;
    }
})();
function assignOpcodeLayout(layoutType, opcodes) {
    opcodes.forEach(function (opcode) {
        olt[opcode] = layoutType;
    });
}
assignOpcodeLayout(OpcodeLayoutType.UINT8_VALUE, [OpCode.ALOAD, OpCode.ASTORE, OpCode.DLOAD, OpCode.DSTORE,
    OpCode.FLOAD, OpCode.FSTORE, OpCode.ILOAD, OpCode.ISTORE,
    OpCode.LLOAD, OpCode.LSTORE, OpCode.RET]);
assignOpcodeLayout(OpcodeLayoutType.CONSTANT_POOL_UINT8, [OpCode.LDC]);
assignOpcodeLayout(OpcodeLayoutType.CONSTANT_POOL, [OpCode.LDC_W, OpCode.LDC2_W,
    OpCode.ANEWARRAY, OpCode.CHECKCAST, OpCode.GETFIELD,
    OpCode.GETSTATIC, OpCode.INSTANCEOF, OpCode.INVOKEDYNAMIC,
    OpCode.INVOKESPECIAL, OpCode.INVOKESTATIC, OpCode.INVOKEVIRTUAL,
    OpCode.NEW, OpCode.PUTFIELD, OpCode.PUTSTATIC, OpCode.MULTIANEWARRAY_FAST,
    OpCode.INVOKENONVIRTUAL_FAST, OpCode.INVOKESTATIC_FAST, OpCode.CHECKCAST_FAST,
    OpCode.NEW_FAST,
    OpCode.ANEWARRAY_FAST, OpCode.INSTANCEOF_FAST, OpCode.GETSTATIC_FAST32,
    OpCode.GETSTATIC_FAST64, OpCode.PUTSTATIC_FAST32, OpCode.PUTSTATIC_FAST64,
    OpCode.PUTFIELD_FAST32, OpCode.PUTFIELD_FAST64,
    OpCode.GETFIELD_FAST32, OpCode.GETFIELD_FAST64, OpCode.INVOKEVIRTUAL_FAST
]);
assignOpcodeLayout(OpcodeLayoutType.CONSTANT_POOL_AND_UINT8_VALUE, [OpCode.INVOKEINTERFACE, OpCode.INVOKEINTERFACE_FAST, OpCode.MULTIANEWARRAY]);
assignOpcodeLayout(OpcodeLayoutType.INT8_VALUE, [OpCode.BIPUSH]);
assignOpcodeLayout(OpcodeLayoutType.INT16_VALUE, [OpCode.SIPUSH, OpCode.GOTO, OpCode.IFGT, OpCode.IFEQ, OpCode.IFGE, OpCode.IFLE,
    OpCode.IFLT, OpCode.IFNE, OpCode.IFNULL, OpCode.IFNONNULL, OpCode.IF_ICMPLE,
    OpCode.IF_ACMPEQ, OpCode.IF_ACMPNE, OpCode.IF_ICMPEQ, OpCode.IF_ICMPGE,
    OpCode.IF_ICMPGT, OpCode.IF_ICMPLT, OpCode.IF_ICMPNE, OpCode.JSR]);
assignOpcodeLayout(OpcodeLayoutType.INT32_VALUE, [OpCode.GOTO_W, OpCode.JSR_W]);
assignOpcodeLayout(OpcodeLayoutType.UINT8_AND_INT8_VALUE, [OpCode.IINC]);
assignOpcodeLayout(OpcodeLayoutType.ARRAY_TYPE, [OpCode.NEWARRAY]);
exports.OpcodeLayouts = olt;
//# sourceMappingURL=data:application/json;base64,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