// Copyright 2009 The Closure Library Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS-IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
"use strict";
/**
 * @fileoverview Defines a Long class for representing a 64-bit two's-complement
 * integer value, which faithfully simulates the behavior of a Java "long". This
 * implementation is derived from LongLib in GWT.
 *
 */
var gLong = (function () {
    /**
     * Constructs a 64-bit two's-complement integer, given its low and high 32-bit
     * values as *signed* integers.  See the from* functions below for more
     * convenient ways of constructing Longs.
     *
     * The internal representation of a long is the two given signed, 32-bit values.
     * We use 32-bit pieces because these are the size of integers on which
     * Javascript performs bit-operations.  For operations like addition and
     * multiplication, we split each number into 16-bit pieces, which can easily be
     * multiplied within Javascript's floating-point representation without overflow
     * or change in sign.
     *
     * In the algorithms below, we frequently reduce the negative case to the
     * positive case by negating the input(s) and then post-processing the result.
     * Note that we must ALWAYS check specially whether those values are MIN_VALUE
     * (-2^63) because -MIN_VALUE == MIN_VALUE (since 2^63 cannot be represented as
     * a positive number, it overflows back into a negative).  Not handling this
     * case would often result in infinite recursion.
     *
     * @param {number} low  The low (signed) 32 bits of the long.
     * @param {number} high  The high (signed) 32 bits of the long.
     * @constructor
     */
    function gLong(low, high) {
        this.low_ = low | 0; // force into 32 signed bits.
        this.high_ = high | 0; // force into 32 signed bits.
    }
    /**
     * Returns a Long representing the given (32-bit) integer value.
     * @param {number} value The 32-bit integer in question.
     * @return {!gLong} The corresponding Long value.
     */
    gLong.fromInt = function (value) {
        if (-128 <= value && value < 128) {
            var cachedObj = gLong.IntCache_[value];
            if (cachedObj) {
                return cachedObj;
            }
        }
        var obj = new gLong(value, value < 0 ? -1 : 0);
        if (-128 <= value && value < 128) {
            gLong.IntCache_[value] = obj;
        }
        return obj;
    };
    /**
     * Returns a Long representing the given value, provided that it is a finite
     * number.  Otherwise, zero is returned.
     * @param {number} value The number in question.
     * @return {!gLong} The corresponding Long value.
     */
    gLong.fromNumber = function (value) {
        if (isNaN(value) || !isFinite(value)) {
            return gLong.ZERO;
        }
        else if (value <= -gLong.TWO_PWR_63_DBL_) {
            return gLong.MIN_VALUE;
        }
        else if (value + 1 >= gLong.TWO_PWR_63_DBL_) {
            return gLong.MAX_VALUE;
        }
        else if (value < 0) {
            return gLong.fromNumber(-value).negate();
        }
        else {
            return new gLong((value % gLong.TWO_PWR_32_DBL_) | 0, (value / gLong.TWO_PWR_32_DBL_) | 0);
        }
    };
    /**
     * Returns a Long representing the 64-bit integer that comes by concatenating
     * the given high and low bits.  Each is assumed to use 32 bits.
     * @param {number} lowBits The low 32-bits.
     * @param {number} highBits The high 32-bits.
     * @return {!gLong} The corresponding Long value.
     */
    gLong.fromBits = function (lowBits, highBits) {
        return new gLong(lowBits, highBits);
    };
    /**
     * Returns a Long representation of the given string, written using the given
     * radix.
     * @param {string} str The textual representation of the Long.
     * @param {number=} opt_radix The radix in which the text is written.
     * @return {!gLong} The corresponding Long value.
     */
    gLong.fromString = function (str, opt_radix) {
        if (str.length == 0) {
            throw Error('number format error: empty string');
        }
        var radix = opt_radix || 10;
        if (radix < 2 || 36 < radix) {
            throw Error('radix out of range: ' + radix);
        }
        if (str.charAt(0) == '-') {
            return gLong.fromString(str.substring(1), radix).negate();
        }
        else if (str.indexOf('-') >= 0) {
            throw Error('number format error: interior "-" character: ' + str);
        }
        // Do several (8) digits each time through the loop, so as to
        // minimize the calls to the very expensive emulated div.
        var radixToPower = gLong.fromNumber(Math.pow(radix, 8));
        var result = gLong.ZERO;
        for (var i = 0; i < str.length; i += 8) {
            var size = Math.min(8, str.length - i);
            var value = parseInt(str.substring(i, i + size), radix);
            if (size < 8) {
                var power = gLong.fromNumber(Math.pow(radix, size));
                result = result.multiply(power).add(gLong.fromNumber(value));
            }
            else {
                result = result.multiply(radixToPower);
                result = result.add(gLong.fromNumber(value));
            }
        }
        return result;
    };
    /** @return {number} The value, assuming it is a 32-bit integer. */
    gLong.prototype.toInt = function () {
        return this.low_;
    };
    /** @return {number} The closest floating-point representation to this value. */
    gLong.prototype.toNumber = function () {
        return this.high_ * gLong.TWO_PWR_32_DBL_ + this.getLowBitsUnsigned();
    };
    /**
     * @param {number=} opt_radix The radix in which the text should be written.
     * @return {string} The textual representation of this value.
     */
    gLong.prototype.toString = function (opt_radix) {
        var radix = opt_radix || 10;
        if (radix < 2 || 36 < radix) {
            throw Error('radix out of range: ' + radix);
        }
        if (this.isZero()) {
            return '0';
        }
        if (this.isNegative()) {
            if (this.equals(gLong.MIN_VALUE)) {
                // We need to change the Long value before it can be negated, so we remove
                // the bottom-most digit in this base and then recurse to do the rest.
                var radixLong = gLong.fromNumber(radix);
                var div = this.div(radixLong);
                var rem = div.multiply(radixLong).subtract(this);
                return div.toString(radix) + rem.toInt().toString(radix);
            }
            else {
                return '-' + this.negate().toString(radix);
            }
        }
        // Do several (6) digits each time through the loop, so as to
        // minimize the calls to the very expensive emulated div.
        var radixToPower = gLong.fromNumber(Math.pow(radix, 6));
        var rem = this;
        var result = '';
        while (true) {
            var remDiv = rem.div(radixToPower);
            var intval = rem.subtract(remDiv.multiply(radixToPower)).toInt();
            var digits = intval.toString(radix);
            rem = remDiv;
            if (rem.isZero()) {
                return digits + result;
            }
            else {
                while (digits.length < 6) {
                    digits = '0' + digits;
                }
                result = '' + digits + result;
            }
        }
    };
    /** @return {number} The high 32-bits as a signed value. */
    gLong.prototype.getHighBits = function () {
        return this.high_;
    };
    /** @return {number} The low 32-bits as a signed value. */
    gLong.prototype.getLowBits = function () {
        return this.low_;
    };
    /** @return {number} The low 32-bits as an unsigned value. */
    gLong.prototype.getLowBitsUnsigned = function () {
        return (this.low_ >= 0) ? this.low_ : gLong.TWO_PWR_32_DBL_ + this.low_;
    };
    /**
     * @return {number} Returns the number of bits needed to represent the absolute
     *     value of this Long.
     */
    gLong.prototype.getNumBitsAbs = function () {
        if (this.isNegative()) {
            if (this.equals(gLong.MIN_VALUE)) {
                return 64;
            }
            else {
                return this.negate().getNumBitsAbs();
            }
        }
        else {
            var val = this.high_ != 0 ? this.high_ : this.low_;
            for (var bit = 31; bit > 0; bit--) {
                if ((val & (1 << bit)) != 0) {
                    break;
                }
            }
            return this.high_ != 0 ? bit + 33 : bit + 1;
        }
    };
    /** @return {boolean} Whether this value is zero. */
    gLong.prototype.isZero = function () {
        return this.high_ == 0 && this.low_ == 0;
    };
    /** @return {boolean} Whether this value is negative. */
    gLong.prototype.isNegative = function () {
        return this.high_ < 0;
    };
    /** @return {boolean} Whether this value is odd. */
    gLong.prototype.isOdd = function () {
        return (this.low_ & 1) == 1;
    };
    /**
     * @param {gLong} other Long to compare against.
     * @return {boolean} Whether this Long equals the other.
     */
    gLong.prototype.equals = function (other) {
        return (this.high_ == other.high_) && (this.low_ == other.low_);
    };
    /**
     * @param {gLong} other Long to compare against.
     * @return {boolean} Whether this Long does not equal the other.
     */
    gLong.prototype.notEquals = function (other) {
        return (this.high_ != other.high_) || (this.low_ != other.low_);
    };
    /**
     * @param {gLong} other Long to compare against.
     * @return {boolean} Whether this Long is less than the other.
     */
    gLong.prototype.lessThan = function (other) {
        return this.compare(other) < 0;
    };
    /**
     * @param {gLong} other Long to compare against.
     * @return {boolean} Whether this Long is less than or equal to the other.
     */
    gLong.prototype.lessThanOrEqual = function (other) {
        return this.compare(other) <= 0;
    };
    /**
     * @param {gLong} other Long to compare against.
     * @return {boolean} Whether this Long is greater than the other.
     */
    gLong.prototype.greaterThan = function (other) {
        return this.compare(other) > 0;
    };
    /**
     * @param {gLong} other Long to compare against.
     * @return {boolean} Whether this Long is greater than or equal to the other.
     */
    gLong.prototype.greaterThanOrEqual = function (other) {
        return this.compare(other) >= 0;
    };
    /**
     * Compares this Long with the given one.
     * @param {gLong} other Long to compare against.
     * @return {number} 0 if they are the same, 1 if the this is greater, and -1
     *     if the given one is greater.
     */
    gLong.prototype.compare = function (other) {
        if (this.equals(other)) {
            return 0;
        }
        var thisNeg = this.isNegative();
        var otherNeg = other.isNegative();
        if (thisNeg && !otherNeg) {
            return -1;
        }
        if (!thisNeg && otherNeg) {
            return 1;
        }
        // at this point, the signs are the same, so subtraction will not overflow
        if (this.subtract(other).isNegative()) {
            return -1;
        }
        else {
            return 1;
        }
    };
    /** @return {!gLong} The negation of this value. */
    gLong.prototype.negate = function () {
        if (this.equals(gLong.MIN_VALUE)) {
            return gLong.MIN_VALUE;
        }
        else {
            return this.not().add(gLong.ONE);
        }
    };
    /**
     * Returns the sum of this and the given Long.
     * @param {gLong} other Long to add to this one.
     * @return {!gLong} The sum of this and the given Long.
     */
    gLong.prototype.add = function (other) {
        // Divide each number into 4 chunks of 16 bits, and then sum the chunks.
        var a48 = this.high_ >>> 16;
        var a32 = this.high_ & 0xFFFF;
        var a16 = this.low_ >>> 16;
        var a00 = this.low_ & 0xFFFF;
        var b48 = other.high_ >>> 16;
        var b32 = other.high_ & 0xFFFF;
        var b16 = other.low_ >>> 16;
        var b00 = other.low_ & 0xFFFF;
        var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
        c00 += a00 + b00;
        c16 += c00 >>> 16;
        c00 &= 0xFFFF;
        c16 += a16 + b16;
        c32 += c16 >>> 16;
        c16 &= 0xFFFF;
        c32 += a32 + b32;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c48 += a48 + b48;
        c48 &= 0xFFFF;
        return gLong.fromBits((c16 << 16) | c00, (c48 << 16) | c32);
    };
    /**
     * Returns the difference of this and the given Long.
     * @param {gLong} other Long to subtract from this.
     * @return {!gLong} The difference of this and the given Long.
     */
    gLong.prototype.subtract = function (other) {
        return this.add(other.negate());
    };
    /**
     * Returns the product of this and the given long.
     * @param {gLong} other Long to multiply with this.
     * @return {!gLong} The product of this and the other.
     */
    gLong.prototype.multiply = function (other) {
        if (this.isZero()) {
            return gLong.ZERO;
        }
        else if (other.isZero()) {
            return gLong.ZERO;
        }
        if (this.equals(gLong.MIN_VALUE)) {
            return other.isOdd() ? gLong.MIN_VALUE : gLong.ZERO;
        }
        else if (other.equals(gLong.MIN_VALUE)) {
            return this.isOdd() ? gLong.MIN_VALUE : gLong.ZERO;
        }
        if (this.isNegative()) {
            if (other.isNegative()) {
                return this.negate().multiply(other.negate());
            }
            else {
                return this.negate().multiply(other).negate();
            }
        }
        else if (other.isNegative()) {
            return this.multiply(other.negate()).negate();
        }
        // If both longs are small, use float multiplication
        if (this.lessThan(gLong.TWO_PWR_24_) &&
            other.lessThan(gLong.TWO_PWR_24_)) {
            return gLong.fromNumber(this.toNumber() * other.toNumber());
        }
        // Divide each long into 4 chunks of 16 bits, and then add up 4x4 products.
        // We can skip products that would overflow.
        var a48 = this.high_ >>> 16;
        var a32 = this.high_ & 0xFFFF;
        var a16 = this.low_ >>> 16;
        var a00 = this.low_ & 0xFFFF;
        var b48 = other.high_ >>> 16;
        var b32 = other.high_ & 0xFFFF;
        var b16 = other.low_ >>> 16;
        var b00 = other.low_ & 0xFFFF;
        var c48 = 0, c32 = 0, c16 = 0, c00 = 0;
        c00 += a00 * b00;
        c16 += c00 >>> 16;
        c00 &= 0xFFFF;
        c16 += a16 * b00;
        c32 += c16 >>> 16;
        c16 &= 0xFFFF;
        c16 += a00 * b16;
        c32 += c16 >>> 16;
        c16 &= 0xFFFF;
        c32 += a32 * b00;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c32 += a16 * b16;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c32 += a00 * b32;
        c48 += c32 >>> 16;
        c32 &= 0xFFFF;
        c48 += a48 * b00 + a32 * b16 + a16 * b32 + a00 * b48;
        c48 &= 0xFFFF;
        return gLong.fromBits((c16 << 16) | c00, (c48 << 16) | c32);
    };
    /**
     * Returns this Long divided by the given one.
     * @param {gLong} other Long by which to divide.
     * @return {!gLong} This Long divided by the given one.
     */
    gLong.prototype.div = function (other) {
        if (other.isZero()) {
            throw Error('division by zero');
        }
        else if (this.isZero()) {
            return gLong.ZERO;
        }
        if (this.equals(gLong.MIN_VALUE)) {
            if (other.equals(gLong.ONE) ||
                other.equals(gLong.NEG_ONE)) {
                return gLong.MIN_VALUE; // recall that -MIN_VALUE == MIN_VALUE
            }
            else if (other.equals(gLong.MIN_VALUE)) {
                return gLong.ONE;
            }
            else {
                // At this point, we have |other| >= 2, so |this/other| < |MIN_VALUE|.
                var halfThis = this.shiftRight(1);
                var l_approx = halfThis.div(other).shiftLeft(1);
                if (l_approx.equals(gLong.ZERO)) {
                    return other.isNegative() ? gLong.ONE : gLong.NEG_ONE;
                }
                else {
                    var rem = this.subtract(other.multiply(l_approx));
                    var result = l_approx.add(rem.div(other));
                    return result;
                }
            }
        }
        else if (other.equals(gLong.MIN_VALUE)) {
            return gLong.ZERO;
        }
        if (this.isNegative()) {
            if (other.isNegative()) {
                return this.negate().div(other.negate());
            }
            else {
                return this.negate().div(other).negate();
            }
        }
        else if (other.isNegative()) {
            return this.div(other.negate()).negate();
        }
        // Repeat the following until the remainder is less than other:  find a
        // floating-point that approximates remainder / other *from below*, add this
        // into the result, and subtract it from the remainder.  It is critical that
        // the approximate value is less than or equal to the real value so that the
        // remainder never becomes negative.
        var res = gLong.ZERO;
        var rem = this;
        while (rem.greaterThanOrEqual(other)) {
            // Approximate the result of division. This may be a little greater or
            // smaller than the actual value.
            var approx = Math.max(1, Math.floor(rem.toNumber() / other.toNumber()));
            // We will tweak the approximate result by changing it in the 48-th digit or
            // the smallest non-fractional digit, whichever is larger.
            var log2 = Math.ceil(Math.log(approx) / Math.LN2);
            var delta = 1;
            if (log2 > 48)
                delta = Math.pow(2, log2 - 48);
            // Decrease the approximation until it is smaller than the remainder.  Note
            // that if it is too large, the product overflows and is negative.
            var approxRes = gLong.fromNumber(approx);
            var approxRem = approxRes.multiply(other);
            while (approxRem.isNegative() || approxRem.greaterThan(rem)) {
                approx -= delta;
                approxRes = gLong.fromNumber(approx);
                approxRem = approxRes.multiply(other);
            }
            // We know the answer can't be zero... and actually, zero would cause
            // infinite recursion since we would make no progress.
            if (approxRes.isZero()) {
                approxRes = gLong.ONE;
            }
            res = res.add(approxRes);
            rem = rem.subtract(approxRem);
        }
        return res;
    };
    /**
     * Returns this Long modulo the given one.
     * @param {gLong} other Long by which to mod.
     * @return {!gLong} This Long modulo the given one.
     */
    gLong.prototype.modulo = function (other) {
        return this.subtract(this.div(other).multiply(other));
    };
    /** @return {!gLong} The bitwise-NOT of this value. */
    gLong.prototype.not = function () {
        return gLong.fromBits(~this.low_, ~this.high_);
    };
    /**
     * Returns the bitwise-AND of this Long and the given one.
     * @param {gLong} other The Long with which to AND.
     * @return {!gLong} The bitwise-AND of this and the other.
     */
    gLong.prototype.and = function (other) {
        return gLong.fromBits(this.low_ & other.low_, this.high_ & other.high_);
    };
    /**
     * Returns the bitwise-OR of this Long and the given one.
     * @param {gLong} other The Long with which to OR.
     * @return {!gLong} The bitwise-OR of this and the other.
     */
    gLong.prototype.or = function (other) {
        return gLong.fromBits(this.low_ | other.low_, this.high_ | other.high_);
    };
    /**
     * Returns the bitwise-XOR of this Long and the given one.
     * @param {gLong} other The Long with which to XOR.
     * @return {!gLong} The bitwise-XOR of this and the other.
     */
    gLong.prototype.xor = function (other) {
        return gLong.fromBits(this.low_ ^ other.low_, this.high_ ^ other.high_);
    };
    /**
     * Returns this Long with bits shifted to the left by the given amount.
     * @param {number} numBits The number of bits by which to shift.
     * @return {!gLong} This shifted to the left by the given amount.
     */
    gLong.prototype.shiftLeft = function (numBits) {
        numBits &= 63;
        if (numBits == 0) {
            return this;
        }
        else {
            var low = this.low_;
            if (numBits < 32) {
                var high = this.high_;
                return gLong.fromBits(low << numBits, (high << numBits) | (low >>> (32 - numBits)));
            }
            else {
                return gLong.fromBits(0, low << (numBits - 32));
            }
        }
    };
    /**
     * Returns this Long with bits shifted to the right by the given amount.
     * @param {number} numBits The number of bits by which to shift.
     * @return {!gLong} This shifted to the right by the given amount.
     */
    gLong.prototype.shiftRight = function (numBits) {
        numBits &= 63;
        if (numBits == 0) {
            return this;
        }
        else {
            var high = this.high_;
            if (numBits < 32) {
                var low = this.low_;
                return gLong.fromBits((low >>> numBits) | (high << (32 - numBits)), high >> numBits);
            }
            else {
                return gLong.fromBits(high >> (numBits - 32), high >= 0 ? 0 : -1);
            }
        }
    };
    /**
     * Returns this Long with bits shifted to the right by the given amount, with
     * the new top bits matching the current sign bit.
     * @param {number} numBits The number of bits by which to shift.
     * @return {!gLong} This shifted to the right by the given amount, with
     *     zeros placed into the new leading bits.
     */
    gLong.prototype.shiftRightUnsigned = function (numBits) {
        numBits &= 63;
        if (numBits == 0) {
            return this;
        }
        else {
            var high = this.high_;
            if (numBits < 32) {
                var low = this.low_;
                return gLong.fromBits((low >>> numBits) | (high << (32 - numBits)), high >>> numBits);
            }
            else if (numBits == 32) {
                return gLong.fromBits(high, 0);
            }
            else {
                return gLong.fromBits(high >>> (numBits - 32), 0);
            }
        }
    };
    // A cache of the Long representations of small integer values.
    gLong.IntCache_ = {};
    // Commonly used constant values.
    gLong.TWO_PWR_16_DBL_ = 1 << 16;
    gLong.TWO_PWR_24_DBL_ = 1 << 24;
    gLong.TWO_PWR_32_DBL_ = gLong.TWO_PWR_16_DBL_ * gLong.TWO_PWR_16_DBL_;
    gLong.TWO_PWR_31_DBL_ = gLong.TWO_PWR_32_DBL_ / 2;
    gLong.TWO_PWR_48_DBL_ = gLong.TWO_PWR_32_DBL_ * gLong.TWO_PWR_16_DBL_;
    gLong.TWO_PWR_64_DBL_ = gLong.TWO_PWR_32_DBL_ * gLong.TWO_PWR_32_DBL_;
    gLong.TWO_PWR_63_DBL_ = gLong.TWO_PWR_64_DBL_ / 2;
    gLong.ZERO = gLong.fromInt(0);
    gLong.ONE = gLong.fromInt(1);
    gLong.NEG_ONE = gLong.fromInt(-1);
    gLong.MAX_VALUE = gLong.fromBits(0xFFFFFFFF, 0x7FFFFFFF);
    gLong.MIN_VALUE = gLong.fromBits(0, 0x80000000);
    gLong.TWO_PWR_24_ = gLong.fromInt(gLong.TWO_PWR_24_DBL_);
    return gLong;
}());
exports.__esModule = true;
exports["default"] = gLong;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ0xvbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvZ0xvbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsbUVBQW1FO0FBQ25FLEVBQUU7QUFDRixrRUFBa0U7QUFDbEUsbUVBQW1FO0FBQ25FLDBDQUEwQztBQUMxQyxFQUFFO0FBQ0Ysa0RBQWtEO0FBQ2xELEVBQUU7QUFDRixzRUFBc0U7QUFDdEUsb0VBQW9FO0FBQ3BFLDJFQUEyRTtBQUMzRSxzRUFBc0U7QUFDdEUsaUNBQWlDOztBQUVqQzs7Ozs7R0FLRztBQUNIO0lBcUJFOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0JHO0lBQ0gsZUFBWSxHQUFXLEVBQUUsSUFBWTtRQUNuQyxJQUFJLENBQUMsSUFBSSxHQUFHLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBRSw2QkFBNkI7UUFDbkQsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxDQUFDLENBQUUsNkJBQTZCO0lBQ3ZELENBQUM7SUFFRDs7OztPQUlHO0lBQ1csYUFBTyxHQUFyQixVQUFzQixLQUFhO1FBQ2pDLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxJQUFJLEtBQUssSUFBSSxLQUFLLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUNqQyxJQUFJLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3ZDLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2QsTUFBTSxDQUFDLFNBQVMsQ0FBQztZQUNuQixDQUFDO1FBQ0gsQ0FBQztRQUVELElBQUksR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQy9DLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxJQUFJLEtBQUssSUFBSSxLQUFLLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUNqQyxLQUFLLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxHQUFHLEdBQUcsQ0FBQztRQUMvQixDQUFDO1FBQ0QsTUFBTSxDQUFDLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNXLGdCQUFVLEdBQXhCLFVBQXlCLEtBQWE7UUFDcEMsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQztRQUNwQixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO1lBQzNDLE1BQU0sQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ3pCLENBQUM7UUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxHQUFHLENBQUMsSUFBSSxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQztZQUM5QyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUN6QixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JCLE1BQU0sQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDM0MsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sTUFBTSxDQUFDLElBQUksS0FBSyxDQUNaLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLEVBQ25DLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUMzQyxDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNXLGNBQVEsR0FBdEIsVUFBdUIsT0FBZSxFQUFFLFFBQWdCO1FBQ3RELE1BQU0sQ0FBQyxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNXLGdCQUFVLEdBQXhCLFVBQXlCLEdBQVcsRUFBRSxTQUFrQjtRQUN0RCxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDcEIsTUFBTSxLQUFLLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUNuRCxDQUFDO1FBRUQsSUFBSSxLQUFLLEdBQUcsU0FBUyxJQUFJLEVBQUUsQ0FBQztRQUM1QixFQUFFLENBQUMsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQzVCLE1BQU0sS0FBSyxDQUFDLHNCQUFzQixHQUFHLEtBQUssQ0FBQyxDQUFDO1FBQzlDLENBQUM7UUFFRCxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDekIsTUFBTSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUM1RCxDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNqQyxNQUFNLEtBQUssQ0FBQywrQ0FBK0MsR0FBRyxHQUFHLENBQUMsQ0FBQztRQUNyRSxDQUFDO1FBRUQsNkRBQTZEO1FBQzdELHlEQUF5RDtRQUN6RCxJQUFJLFlBQVksR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFeEQsSUFBSSxNQUFNLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztRQUN4QixHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ3ZDLElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDdkMsSUFBSSxLQUFLLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxJQUFJLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUN4RCxFQUFFLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDYixJQUFJLEtBQUssR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7Z0JBQ3BELE1BQU0sR0FBRyxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDL0QsQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLE1BQU0sR0FBRyxNQUFNLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDO2dCQUN2QyxNQUFNLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDL0MsQ0FBQztRQUNILENBQUM7UUFDRCxNQUFNLENBQUMsTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFRCxtRUFBbUU7SUFDNUQscUJBQUssR0FBWjtRQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQ25CLENBQUM7SUFFRCxnRkFBZ0Y7SUFDekUsd0JBQVEsR0FBZjtRQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUM7SUFDeEUsQ0FBQztJQUVEOzs7T0FHRztJQUNJLHdCQUFRLEdBQWYsVUFBZ0IsU0FBa0I7UUFDaEMsSUFBSSxLQUFLLEdBQUcsU0FBUyxJQUFJLEVBQUUsQ0FBQztRQUM1QixFQUFFLENBQUMsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQzVCLE1BQU0sS0FBSyxDQUFDLHNCQUFzQixHQUFHLEtBQUssQ0FBQyxDQUFDO1FBQzlDLENBQUM7UUFFRCxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2xCLE1BQU0sQ0FBQyxHQUFHLENBQUM7UUFDYixDQUFDO1FBRUQsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUN0QixFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2pDLDBFQUEwRTtnQkFDMUUsc0VBQXNFO2dCQUN0RSxJQUFJLFNBQVMsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUN4QyxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUM5QixJQUFJLEdBQUcsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDakQsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMzRCxDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sTUFBTSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzdDLENBQUM7UUFDSCxDQUFDO1FBRUQsNkRBQTZEO1FBQzdELHlEQUF5RDtRQUN6RCxJQUFJLFlBQVksR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDeEQsSUFBSSxHQUFHLEdBQVUsSUFBSSxDQUFDO1FBQ3RCLElBQUksTUFBTSxHQUFHLEVBQUUsQ0FBQztRQUNoQixPQUFPLElBQUksRUFBRSxDQUFDO1lBQ1osSUFBSSxNQUFNLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsQ0FBQztZQUNuQyxJQUFJLE1BQU0sR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNqRSxJQUFJLE1BQU0sR0FBRyxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRXBDLEdBQUcsR0FBRyxNQUFNLENBQUM7WUFDYixFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUNqQixNQUFNLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztZQUN6QixDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sT0FBTyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO29CQUN6QixNQUFNLEdBQUcsR0FBRyxHQUFHLE1BQU0sQ0FBQztnQkFDeEIsQ0FBQztnQkFDRCxNQUFNLEdBQUcsRUFBRSxHQUFHLE1BQU0sR0FBRyxNQUFNLENBQUM7WUFDaEMsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRUQsMkRBQTJEO0lBQ3BELDJCQUFXLEdBQWxCO1FBQ0UsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUM7SUFDcEIsQ0FBQztJQUVELDBEQUEwRDtJQUNuRCwwQkFBVSxHQUFqQjtRQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQ25CLENBQUM7SUFFRCw2REFBNkQ7SUFDdEQsa0NBQWtCLEdBQXpCO1FBQ0UsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksSUFBSSxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztJQUMxRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksNkJBQWEsR0FBcEI7UUFDRSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3RCLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDakMsTUFBTSxDQUFDLEVBQUUsQ0FBQztZQUNaLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQ3ZDLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7WUFDbkQsR0FBRyxDQUFDLENBQUMsSUFBSSxHQUFHLEdBQUcsRUFBRSxFQUFFLEdBQUcsR0FBRyxDQUFDLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQztnQkFDbEMsRUFBRSxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDLElBQUksR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO29CQUM1QixLQUFLLENBQUM7Z0JBQ1IsQ0FBQztZQUNILENBQUM7WUFDRCxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssSUFBSSxDQUFDLEdBQUcsR0FBRyxHQUFHLEVBQUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLENBQUM7SUFDSCxDQUFDO0lBRUQsb0RBQW9EO0lBQzdDLHNCQUFNLEdBQWI7UUFDRSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssSUFBSSxDQUFDLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVELHdEQUF3RDtJQUNqRCwwQkFBVSxHQUFqQjtRQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxHQUFHLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRUQsbURBQW1EO0lBQzVDLHFCQUFLLEdBQVo7UUFDRSxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksc0JBQU0sR0FBYixVQUFjLEtBQVk7UUFDeEIsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0kseUJBQVMsR0FBaEIsVUFBaUIsS0FBWTtRQUMzQixNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7O09BR0c7SUFDSSx3QkFBUSxHQUFmLFVBQWdCLEtBQVk7UUFDMUIsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7O09BR0c7SUFDSSwrQkFBZSxHQUF0QixVQUF1QixLQUFZO1FBQ2pDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksMkJBQVcsR0FBbEIsVUFBbUIsS0FBWTtRQUM3QixNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLGtDQUFrQixHQUF6QixVQUEwQixLQUFZO1FBQ3BDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSx1QkFBTyxHQUFkLFVBQWUsS0FBWTtRQUN6QixFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN2QixNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ1gsQ0FBQztRQUVELElBQUksT0FBTyxHQUFHLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUNoQyxJQUFJLFFBQVEsR0FBRyxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDbEMsRUFBRSxDQUFDLENBQUMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUN6QixNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDWixDQUFDO1FBQ0QsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLElBQUksUUFBUSxDQUFDLENBQUMsQ0FBQztZQUN6QixNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ1gsQ0FBQztRQUVELDBFQUEwRTtRQUMxRSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUN0QyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDWixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ1gsQ0FBQztJQUNILENBQUM7SUFFRCxtREFBbUQ7SUFDNUMsc0JBQU0sR0FBYjtRQUNFLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNqQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUN6QixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbkMsQ0FBQztJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQUcsR0FBVixVQUFXLEtBQVk7UUFDckIsd0VBQXdFO1FBRXhFLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLEtBQUssRUFBRSxDQUFDO1FBQzVCLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLEdBQUcsTUFBTSxDQUFDO1FBQzlCLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDO1FBQzNCLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxJQUFJLEdBQUcsTUFBTSxDQUFDO1FBRTdCLElBQUksR0FBRyxHQUFHLEtBQUssQ0FBQyxLQUFLLEtBQUssRUFBRSxDQUFDO1FBQzdCLElBQUksR0FBRyxHQUFHLEtBQUssQ0FBQyxLQUFLLEdBQUcsTUFBTSxDQUFDO1FBQy9CLElBQUksR0FBRyxHQUFHLEtBQUssQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDO1FBQzVCLElBQUksR0FBRyxHQUFHLEtBQUssQ0FBQyxJQUFJLEdBQUcsTUFBTSxDQUFDO1FBRTlCLElBQUksR0FBRyxHQUFHLENBQUMsRUFBRSxHQUFHLEdBQUcsQ0FBQyxFQUFFLEdBQUcsR0FBRyxDQUFDLEVBQUUsR0FBRyxHQUFHLENBQUMsQ0FBQztRQUN2QyxHQUFHLElBQUksR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNqQixHQUFHLElBQUksR0FBRyxLQUFLLEVBQUUsQ0FBQztRQUNsQixHQUFHLElBQUksTUFBTSxDQUFDO1FBQ2QsR0FBRyxJQUFJLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDakIsR0FBRyxJQUFJLEdBQUcsS0FBSyxFQUFFLENBQUM7UUFDbEIsR0FBRyxJQUFJLE1BQU0sQ0FBQztRQUNkLEdBQUcsSUFBSSxHQUFHLEdBQUcsR0FBRyxDQUFDO1FBQ2pCLEdBQUcsSUFBSSxHQUFHLEtBQUssRUFBRSxDQUFDO1FBQ2xCLEdBQUcsSUFBSSxNQUFNLENBQUM7UUFDZCxHQUFHLElBQUksR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNqQixHQUFHLElBQUksTUFBTSxDQUFDO1FBQ2QsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxHQUFHLElBQUksRUFBRSxDQUFDLEdBQUcsR0FBRyxFQUFFLENBQUMsR0FBRyxJQUFJLEVBQUUsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksd0JBQVEsR0FBZixVQUFnQixLQUFZO1FBQzFCLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksd0JBQVEsR0FBZixVQUFnQixLQUFZO1FBQzFCLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDbEIsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDcEIsQ0FBQztRQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzFCLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQ3BCLENBQUM7UUFFRCxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakMsTUFBTSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsR0FBRyxLQUFLLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDdEQsQ0FBQztRQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDekMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsR0FBRyxLQUFLLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDckQsQ0FBQztRQUVELEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDdEIsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDdkIsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFDaEQsQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2hELENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDOUIsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDaEQsQ0FBQztRQUVELG9EQUFvRDtRQUNwRCxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7WUFDaEMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3RDLE1BQU0sQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsR0FBRyxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUM5RCxDQUFDO1FBRUQsMkVBQTJFO1FBQzNFLDRDQUE0QztRQUU1QyxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQztRQUM1QixJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxHQUFHLE1BQU0sQ0FBQztRQUM5QixJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQztRQUMzQixJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsSUFBSSxHQUFHLE1BQU0sQ0FBQztRQUU3QixJQUFJLEdBQUcsR0FBRyxLQUFLLENBQUMsS0FBSyxLQUFLLEVBQUUsQ0FBQztRQUM3QixJQUFJLEdBQUcsR0FBRyxLQUFLLENBQUMsS0FBSyxHQUFHLE1BQU0sQ0FBQztRQUMvQixJQUFJLEdBQUcsR0FBRyxLQUFLLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQztRQUM1QixJQUFJLEdBQUcsR0FBRyxLQUFLLENBQUMsSUFBSSxHQUFHLE1BQU0sQ0FBQztRQUU5QixJQUFJLEdBQUcsR0FBRyxDQUFDLEVBQUUsR0FBRyxHQUFHLENBQUMsRUFBRSxHQUFHLEdBQUcsQ0FBQyxFQUFFLEdBQUcsR0FBRyxDQUFDLENBQUM7UUFDdkMsR0FBRyxJQUFJLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDakIsR0FBRyxJQUFJLEdBQUcsS0FBSyxFQUFFLENBQUM7UUFDbEIsR0FBRyxJQUFJLE1BQU0sQ0FBQztRQUNkLEdBQUcsSUFBSSxHQUFHLEdBQUcsR0FBRyxDQUFDO1FBQ2pCLEdBQUcsSUFBSSxHQUFHLEtBQUssRUFBRSxDQUFDO1FBQ2xCLEdBQUcsSUFBSSxNQUFNLENBQUM7UUFDZCxHQUFHLElBQUksR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNqQixHQUFHLElBQUksR0FBRyxLQUFLLEVBQUUsQ0FBQztRQUNsQixHQUFHLElBQUksTUFBTSxDQUFDO1FBQ2QsR0FBRyxJQUFJLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDakIsR0FBRyxJQUFJLEdBQUcsS0FBSyxFQUFFLENBQUM7UUFDbEIsR0FBRyxJQUFJLE1BQU0sQ0FBQztRQUNkLEdBQUcsSUFBSSxHQUFHLEdBQUcsR0FBRyxDQUFDO1FBQ2pCLEdBQUcsSUFBSSxHQUFHLEtBQUssRUFBRSxDQUFDO1FBQ2xCLEdBQUcsSUFBSSxNQUFNLENBQUM7UUFDZCxHQUFHLElBQUksR0FBRyxHQUFHLEdBQUcsQ0FBQztRQUNqQixHQUFHLElBQUksR0FBRyxLQUFLLEVBQUUsQ0FBQztRQUNsQixHQUFHLElBQUksTUFBTSxDQUFDO1FBQ2QsR0FBRyxJQUFJLEdBQUcsR0FBRyxHQUFHLEdBQUcsR0FBRyxHQUFHLEdBQUcsR0FBRyxHQUFHLEdBQUcsR0FBRyxHQUFHLEdBQUcsR0FBRyxHQUFHLENBQUM7UUFDckQsR0FBRyxJQUFJLE1BQU0sQ0FBQztRQUNkLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxJQUFJLEVBQUUsQ0FBQyxHQUFHLEdBQUcsRUFBRSxDQUFDLEdBQUcsSUFBSSxFQUFFLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG1CQUFHLEdBQVYsVUFBVyxLQUFZO1FBQ3JCLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDbkIsTUFBTSxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUNsQyxDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDekIsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDcEIsQ0FBQztRQUVELEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNqQyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUM7Z0JBQ3ZCLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDaEMsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBRSxzQ0FBc0M7WUFDakUsQ0FBQztZQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3pDLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDO1lBQ25CLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixzRUFBc0U7Z0JBQ3RFLElBQUksUUFBUSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2xDLElBQUksUUFBUSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNoRCxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ2hDLE1BQU0sQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLEdBQUcsS0FBSyxDQUFDLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO2dCQUN4RCxDQUFDO2dCQUFDLElBQUksQ0FBQyxDQUFDO29CQUNOLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO29CQUNsRCxJQUFJLE1BQU0sR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztvQkFDMUMsTUFBTSxDQUFDLE1BQU0sQ0FBQztnQkFDaEIsQ0FBQztZQUNILENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN6QyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQztRQUNwQixDQUFDO1FBRUQsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUN0QixFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUN2QixNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztZQUMzQyxDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDM0MsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUM5QixNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUMzQyxDQUFDO1FBRUQsdUVBQXVFO1FBQ3ZFLDRFQUE0RTtRQUM1RSw0RUFBNEU7UUFDNUUsNEVBQTRFO1FBQzVFLG9DQUFvQztRQUNwQyxJQUFJLEdBQUcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQ3JCLElBQUksR0FBRyxHQUFVLElBQUksQ0FBQztRQUN0QixPQUFPLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQ3JDLHNFQUFzRTtZQUN0RSxpQ0FBaUM7WUFDakMsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUV4RSw0RUFBNEU7WUFDNUUsMERBQTBEO1lBQzFELElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDbEQsSUFBSSxLQUFLLEdBQUcsQ0FBQyxDQUFDO1lBQ2QsRUFBRSxDQUFDLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQztnQkFDWixLQUFLLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsSUFBSSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1lBRWpDLDJFQUEyRTtZQUMzRSxrRUFBa0U7WUFDbEUsSUFBSSxTQUFTLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN6QyxJQUFJLFNBQVMsR0FBRyxTQUFTLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQzFDLE9BQU8sU0FBUyxDQUFDLFVBQVUsRUFBRSxJQUFJLFNBQVMsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztnQkFDNUQsTUFBTSxJQUFJLEtBQUssQ0FBQztnQkFDaEIsU0FBUyxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ3JDLFNBQVMsR0FBRyxTQUFTLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3hDLENBQUM7WUFFRCxxRUFBcUU7WUFDckUsc0RBQXNEO1lBQ3RELEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQ3ZCLFNBQVMsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDO1lBQ3hCLENBQUM7WUFFRCxHQUFHLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUN6QixHQUFHLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoQyxDQUFDO1FBQ0QsTUFBTSxDQUFDLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQU0sR0FBYixVQUFjLEtBQVk7UUFDeEIsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQsc0RBQXNEO0lBQy9DLG1CQUFHLEdBQVY7UUFDRSxNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxtQkFBRyxHQUFWLFVBQVcsS0FBWTtRQUNyQixNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLEVBQ3RCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0JBQUUsR0FBVCxVQUFVLEtBQVk7UUFDcEIsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxFQUN0QixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLG1CQUFHLEdBQVYsVUFBVyxLQUFZO1FBQ3JCLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMxRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHlCQUFTLEdBQWhCLFVBQWlCLE9BQWU7UUFDOUIsT0FBTyxJQUFJLEVBQUUsQ0FBQztRQUNkLEVBQUUsQ0FBQyxDQUFDLE9BQU8sSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pCLE1BQU0sQ0FBQyxJQUFJLENBQUM7UUFDZCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ3BCLEVBQUUsQ0FBQyxDQUFDLE9BQU8sR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO2dCQUNqQixJQUFJLElBQUksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO2dCQUN0QixNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxHQUFHLElBQUksT0FBTyxFQUNkLENBQUMsSUFBSSxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsRUFBRSxHQUFHLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN0RSxDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxFQUFFLEdBQUcsSUFBSSxDQUFDLE9BQU8sR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2xELENBQUM7UUFDSCxDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSwwQkFBVSxHQUFqQixVQUFrQixPQUFlO1FBQy9CLE9BQU8sSUFBSSxFQUFFLENBQUM7UUFDZCxFQUFFLENBQUMsQ0FBQyxPQUFPLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNqQixNQUFNLENBQUMsSUFBSSxDQUFDO1FBQ2QsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sSUFBSSxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQztZQUN0QixFQUFFLENBQUMsQ0FBQyxPQUFPLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDakIsSUFBSSxHQUFHLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQztnQkFDcEIsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQ2pCLENBQUMsR0FBRyxLQUFLLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsRUFBRSxHQUFHLE9BQU8sQ0FBQyxDQUFDLEVBQzVDLElBQUksSUFBSSxPQUFPLENBQUMsQ0FBQztZQUN2QixDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQ2pCLElBQUksSUFBSSxDQUFDLE9BQU8sR0FBRyxFQUFFLENBQUMsRUFDdEIsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMxQixDQUFDO1FBQ0gsQ0FBQztJQUNILENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0IsR0FBekIsVUFBMEIsT0FBZTtRQUN2QyxPQUFPLElBQUksRUFBRSxDQUFDO1FBQ2QsRUFBRSxDQUFDLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakIsTUFBTSxDQUFDLElBQUksQ0FBQztRQUNkLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLElBQUksSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUM7WUFDdEIsRUFBRSxDQUFDLENBQUMsT0FBTyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQ2pCLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7Z0JBQ3BCLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUNqQixDQUFDLEdBQUcsS0FBSyxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQyxFQUM1QyxJQUFJLEtBQUssT0FBTyxDQUFDLENBQUM7WUFDeEIsQ0FBQztZQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDekIsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixNQUFNLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEtBQUssQ0FBQyxPQUFPLEdBQUcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDcEQsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBdm9CRCwrREFBK0Q7SUFDaEQsZUFBUyxHQUE2QixFQUFFLENBQUE7SUFDdkQsaUNBQWlDO0lBQ2xCLHFCQUFlLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUMxQixxQkFBZSxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDMUIscUJBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUM7SUFDaEUscUJBQWUsR0FBRyxLQUFLLENBQUMsZUFBZSxHQUFHLENBQUMsQ0FBQztJQUM1QyxxQkFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztJQUNoRSxxQkFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQztJQUNoRSxxQkFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLEdBQUcsQ0FBQyxDQUFDO0lBRTdDLFVBQUksR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3hCLFNBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3ZCLGFBQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUIsZUFBUyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQ25ELGVBQVMsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsRUFBRSxVQUFVLENBQUMsQ0FBQztJQUN6QyxpQkFBVyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBd25CcEUsWUFBQztBQUFELENBQUMsQUEzb0JELElBMm9CQztBQTNvQkQ7MEJBMm9CQyxDQUFBIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IDIwMDkgVGhlIENsb3N1cmUgTGlicmFyeSBBdXRob3JzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuLy9cbi8vIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIik7XG4vLyB5b3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4vLyBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcbi8vXG4vLyAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuLy9cbi8vIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbi8vIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMtSVNcIiBCQVNJUyxcbi8vIFdJVEhPVVQgV0FSUkFOVElFUyBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBlaXRoZXIgZXhwcmVzcyBvciBpbXBsaWVkLlxuLy8gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuLy8gbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG5cbi8qKlxuICogQGZpbGVvdmVydmlldyBEZWZpbmVzIGEgTG9uZyBjbGFzcyBmb3IgcmVwcmVzZW50aW5nIGEgNjQtYml0IHR3bydzLWNvbXBsZW1lbnRcbiAqIGludGVnZXIgdmFsdWUsIHdoaWNoIGZhaXRoZnVsbHkgc2ltdWxhdGVzIHRoZSBiZWhhdmlvciBvZiBhIEphdmEgXCJsb25nXCIuIFRoaXNcbiAqIGltcGxlbWVudGF0aW9uIGlzIGRlcml2ZWQgZnJvbSBMb25nTGliIGluIEdXVC5cbiAqXG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIGdMb25nIHtcbiAgcHJpdmF0ZSBsb3dfIDogbnVtYmVyXG4gIHByaXZhdGUgaGlnaF8gOiBudW1iZXJcbiAgLy8gQSBjYWNoZSBvZiB0aGUgTG9uZyByZXByZXNlbnRhdGlvbnMgb2Ygc21hbGwgaW50ZWdlciB2YWx1ZXMuXG4gIHByaXZhdGUgc3RhdGljIEludENhY2hlXzogeyBbdmFsOiBudW1iZXJdOiBnTG9uZyB9ID0ge31cbiAgLy8gQ29tbW9ubHkgdXNlZCBjb25zdGFudCB2YWx1ZXMuXG4gIHByaXZhdGUgc3RhdGljIFRXT19QV1JfMTZfREJMXyA9IDEgPDwgMTY7XG4gIHByaXZhdGUgc3RhdGljIFRXT19QV1JfMjRfREJMXyA9IDEgPDwgMjQ7XG4gIHByaXZhdGUgc3RhdGljIFRXT19QV1JfMzJfREJMXyA9IGdMb25nLlRXT19QV1JfMTZfREJMXyAqIGdMb25nLlRXT19QV1JfMTZfREJMXztcbiAgcHJpdmF0ZSBzdGF0aWMgVFdPX1BXUl8zMV9EQkxfID0gZ0xvbmcuVFdPX1BXUl8zMl9EQkxfIC8gMjtcbiAgcHJpdmF0ZSBzdGF0aWMgVFdPX1BXUl80OF9EQkxfID0gZ0xvbmcuVFdPX1BXUl8zMl9EQkxfICogZ0xvbmcuVFdPX1BXUl8xNl9EQkxfO1xuICBwcml2YXRlIHN0YXRpYyBUV09fUFdSXzY0X0RCTF8gPSBnTG9uZy5UV09fUFdSXzMyX0RCTF8gKiBnTG9uZy5UV09fUFdSXzMyX0RCTF87XG4gIHByaXZhdGUgc3RhdGljIFRXT19QV1JfNjNfREJMXyA9IGdMb25nLlRXT19QV1JfNjRfREJMXyAvIDI7XG5cbiAgcHVibGljIHN0YXRpYyBaRVJPID0gZ0xvbmcuZnJvbUludCgwKTtcbiAgcHVibGljIHN0YXRpYyBPTkUgPSBnTG9uZy5mcm9tSW50KDEpO1xuICBwdWJsaWMgc3RhdGljIE5FR19PTkUgPSBnTG9uZy5mcm9tSW50KC0xKTtcbiAgcHVibGljIHN0YXRpYyBNQVhfVkFMVUUgPSBnTG9uZy5mcm9tQml0cygweEZGRkZGRkZGLCAweDdGRkZGRkZGKTtcbiAgcHVibGljIHN0YXRpYyBNSU5fVkFMVUUgPSBnTG9uZy5mcm9tQml0cygwLCAweDgwMDAwMDAwKTtcbiAgcHJpdmF0ZSBzdGF0aWMgVFdPX1BXUl8yNF8gPSBnTG9uZy5mcm9tSW50KGdMb25nLlRXT19QV1JfMjRfREJMXyk7XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYSA2NC1iaXQgdHdvJ3MtY29tcGxlbWVudCBpbnRlZ2VyLCBnaXZlbiBpdHMgbG93IGFuZCBoaWdoIDMyLWJpdFxuICAgKiB2YWx1ZXMgYXMgKnNpZ25lZCogaW50ZWdlcnMuICBTZWUgdGhlIGZyb20qIGZ1bmN0aW9ucyBiZWxvdyBmb3IgbW9yZVxuICAgKiBjb252ZW5pZW50IHdheXMgb2YgY29uc3RydWN0aW5nIExvbmdzLlxuICAgKlxuICAgKiBUaGUgaW50ZXJuYWwgcmVwcmVzZW50YXRpb24gb2YgYSBsb25nIGlzIHRoZSB0d28gZ2l2ZW4gc2lnbmVkLCAzMi1iaXQgdmFsdWVzLlxuICAgKiBXZSB1c2UgMzItYml0IHBpZWNlcyBiZWNhdXNlIHRoZXNlIGFyZSB0aGUgc2l6ZSBvZiBpbnRlZ2VycyBvbiB3aGljaFxuICAgKiBKYXZhc2NyaXB0IHBlcmZvcm1zIGJpdC1vcGVyYXRpb25zLiAgRm9yIG9wZXJhdGlvbnMgbGlrZSBhZGRpdGlvbiBhbmRcbiAgICogbXVsdGlwbGljYXRpb24sIHdlIHNwbGl0IGVhY2ggbnVtYmVyIGludG8gMTYtYml0IHBpZWNlcywgd2hpY2ggY2FuIGVhc2lseSBiZVxuICAgKiBtdWx0aXBsaWVkIHdpdGhpbiBKYXZhc2NyaXB0J3MgZmxvYXRpbmctcG9pbnQgcmVwcmVzZW50YXRpb24gd2l0aG91dCBvdmVyZmxvd1xuICAgKiBvciBjaGFuZ2UgaW4gc2lnbi5cbiAgICpcbiAgICogSW4gdGhlIGFsZ29yaXRobXMgYmVsb3csIHdlIGZyZXF1ZW50bHkgcmVkdWNlIHRoZSBuZWdhdGl2ZSBjYXNlIHRvIHRoZVxuICAgKiBwb3NpdGl2ZSBjYXNlIGJ5IG5lZ2F0aW5nIHRoZSBpbnB1dChzKSBhbmQgdGhlbiBwb3N0LXByb2Nlc3NpbmcgdGhlIHJlc3VsdC5cbiAgICogTm90ZSB0aGF0IHdlIG11c3QgQUxXQVlTIGNoZWNrIHNwZWNpYWxseSB3aGV0aGVyIHRob3NlIHZhbHVlcyBhcmUgTUlOX1ZBTFVFXG4gICAqICgtMl42MykgYmVjYXVzZSAtTUlOX1ZBTFVFID09IE1JTl9WQUxVRSAoc2luY2UgMl42MyBjYW5ub3QgYmUgcmVwcmVzZW50ZWQgYXNcbiAgICogYSBwb3NpdGl2ZSBudW1iZXIsIGl0IG92ZXJmbG93cyBiYWNrIGludG8gYSBuZWdhdGl2ZSkuICBOb3QgaGFuZGxpbmcgdGhpc1xuICAgKiBjYXNlIHdvdWxkIG9mdGVuIHJlc3VsdCBpbiBpbmZpbml0ZSByZWN1cnNpb24uXG4gICAqXG4gICAqIEBwYXJhbSB7bnVtYmVyfSBsb3cgIFRoZSBsb3cgKHNpZ25lZCkgMzIgYml0cyBvZiB0aGUgbG9uZy5cbiAgICogQHBhcmFtIHtudW1iZXJ9IGhpZ2ggIFRoZSBoaWdoIChzaWduZWQpIDMyIGJpdHMgb2YgdGhlIGxvbmcuXG4gICAqIEBjb25zdHJ1Y3RvclxuICAgKi9cbiAgY29uc3RydWN0b3IobG93OiBudW1iZXIsIGhpZ2g6IG51bWJlcikge1xuICAgIHRoaXMubG93XyA9IGxvdyB8IDA7ICAvLyBmb3JjZSBpbnRvIDMyIHNpZ25lZCBiaXRzLlxuICAgIHRoaXMuaGlnaF8gPSBoaWdoIHwgMDsgIC8vIGZvcmNlIGludG8gMzIgc2lnbmVkIGJpdHMuXG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIExvbmcgcmVwcmVzZW50aW5nIHRoZSBnaXZlbiAoMzItYml0KSBpbnRlZ2VyIHZhbHVlLlxuICAgKiBAcGFyYW0ge251bWJlcn0gdmFsdWUgVGhlIDMyLWJpdCBpbnRlZ2VyIGluIHF1ZXN0aW9uLlxuICAgKiBAcmV0dXJuIHshZ0xvbmd9IFRoZSBjb3JyZXNwb25kaW5nIExvbmcgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21JbnQodmFsdWU6IG51bWJlcik6IGdMb25nIHtcbiAgICBpZiAoLTEyOCA8PSB2YWx1ZSAmJiB2YWx1ZSA8IDEyOCkge1xuICAgICAgdmFyIGNhY2hlZE9iaiA9IGdMb25nLkludENhY2hlX1t2YWx1ZV07XG4gICAgICBpZiAoY2FjaGVkT2JqKSB7XG4gICAgICAgIHJldHVybiBjYWNoZWRPYmo7XG4gICAgICB9XG4gICAgfVxuXG4gICAgdmFyIG9iaiA9IG5ldyBnTG9uZyh2YWx1ZSwgdmFsdWUgPCAwID8gLTEgOiAwKTtcbiAgICBpZiAoLTEyOCA8PSB2YWx1ZSAmJiB2YWx1ZSA8IDEyOCkge1xuICAgICAgZ0xvbmcuSW50Q2FjaGVfW3ZhbHVlXSA9IG9iajtcbiAgICB9XG4gICAgcmV0dXJuIG9iajtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgTG9uZyByZXByZXNlbnRpbmcgdGhlIGdpdmVuIHZhbHVlLCBwcm92aWRlZCB0aGF0IGl0IGlzIGEgZmluaXRlXG4gICAqIG51bWJlci4gIE90aGVyd2lzZSwgemVybyBpcyByZXR1cm5lZC5cbiAgICogQHBhcmFtIHtudW1iZXJ9IHZhbHVlIFRoZSBudW1iZXIgaW4gcXVlc3Rpb24uXG4gICAqIEByZXR1cm4geyFnTG9uZ30gVGhlIGNvcnJlc3BvbmRpbmcgTG9uZyB2YWx1ZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbU51bWJlcih2YWx1ZTogbnVtYmVyKTogZ0xvbmcge1xuICAgIGlmIChpc05hTih2YWx1ZSkgfHwgIWlzRmluaXRlKHZhbHVlKSkge1xuICAgICAgcmV0dXJuIGdMb25nLlpFUk87XG4gICAgfSBlbHNlIGlmICh2YWx1ZSA8PSAtZ0xvbmcuVFdPX1BXUl82M19EQkxfKSB7XG4gICAgICByZXR1cm4gZ0xvbmcuTUlOX1ZBTFVFO1xuICAgIH0gZWxzZSBpZiAodmFsdWUgKyAxID49IGdMb25nLlRXT19QV1JfNjNfREJMXykge1xuICAgICAgcmV0dXJuIGdMb25nLk1BWF9WQUxVRTtcbiAgICB9IGVsc2UgaWYgKHZhbHVlIDwgMCkge1xuICAgICAgcmV0dXJuIGdMb25nLmZyb21OdW1iZXIoLXZhbHVlKS5uZWdhdGUoKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIG5ldyBnTG9uZyhcbiAgICAgICAgICAodmFsdWUgJSBnTG9uZy5UV09fUFdSXzMyX0RCTF8pIHwgMCxcbiAgICAgICAgICAodmFsdWUgLyBnTG9uZy5UV09fUFdSXzMyX0RCTF8pIHwgMCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBMb25nIHJlcHJlc2VudGluZyB0aGUgNjQtYml0IGludGVnZXIgdGhhdCBjb21lcyBieSBjb25jYXRlbmF0aW5nXG4gICAqIHRoZSBnaXZlbiBoaWdoIGFuZCBsb3cgYml0cy4gIEVhY2ggaXMgYXNzdW1lZCB0byB1c2UgMzIgYml0cy5cbiAgICogQHBhcmFtIHtudW1iZXJ9IGxvd0JpdHMgVGhlIGxvdyAzMi1iaXRzLlxuICAgKiBAcGFyYW0ge251bWJlcn0gaGlnaEJpdHMgVGhlIGhpZ2ggMzItYml0cy5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGUgY29ycmVzcG9uZGluZyBMb25nIHZhbHVlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQml0cyhsb3dCaXRzOiBudW1iZXIsIGhpZ2hCaXRzOiBudW1iZXIpOiBnTG9uZyB7XG4gICAgcmV0dXJuIG5ldyBnTG9uZyhsb3dCaXRzLCBoaWdoQml0cyk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIExvbmcgcmVwcmVzZW50YXRpb24gb2YgdGhlIGdpdmVuIHN0cmluZywgd3JpdHRlbiB1c2luZyB0aGUgZ2l2ZW5cbiAgICogcmFkaXguXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBzdHIgVGhlIHRleHR1YWwgcmVwcmVzZW50YXRpb24gb2YgdGhlIExvbmcuXG4gICAqIEBwYXJhbSB7bnVtYmVyPX0gb3B0X3JhZGl4IFRoZSByYWRpeCBpbiB3aGljaCB0aGUgdGV4dCBpcyB3cml0dGVuLlxuICAgKiBAcmV0dXJuIHshZ0xvbmd9IFRoZSBjb3JyZXNwb25kaW5nIExvbmcgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoc3RyOiBzdHJpbmcsIG9wdF9yYWRpeD86IG51bWJlcik6IGdMb25nIHtcbiAgICBpZiAoc3RyLmxlbmd0aCA9PSAwKSB7XG4gICAgICB0aHJvdyBFcnJvcignbnVtYmVyIGZvcm1hdCBlcnJvcjogZW1wdHkgc3RyaW5nJyk7XG4gICAgfVxuXG4gICAgdmFyIHJhZGl4ID0gb3B0X3JhZGl4IHx8IDEwO1xuICAgIGlmIChyYWRpeCA8IDIgfHwgMzYgPCByYWRpeCkge1xuICAgICAgdGhyb3cgRXJyb3IoJ3JhZGl4IG91dCBvZiByYW5nZTogJyArIHJhZGl4KTtcbiAgICB9XG5cbiAgICBpZiAoc3RyLmNoYXJBdCgwKSA9PSAnLScpIHtcbiAgICAgIHJldHVybiBnTG9uZy5mcm9tU3RyaW5nKHN0ci5zdWJzdHJpbmcoMSksIHJhZGl4KS5uZWdhdGUoKTtcbiAgICB9IGVsc2UgaWYgKHN0ci5pbmRleE9mKCctJykgPj0gMCkge1xuICAgICAgdGhyb3cgRXJyb3IoJ251bWJlciBmb3JtYXQgZXJyb3I6IGludGVyaW9yIFwiLVwiIGNoYXJhY3RlcjogJyArIHN0cik7XG4gICAgfVxuXG4gICAgLy8gRG8gc2V2ZXJhbCAoOCkgZGlnaXRzIGVhY2ggdGltZSB0aHJvdWdoIHRoZSBsb29wLCBzbyBhcyB0b1xuICAgIC8vIG1pbmltaXplIHRoZSBjYWxscyB0byB0aGUgdmVyeSBleHBlbnNpdmUgZW11bGF0ZWQgZGl2LlxuICAgIHZhciByYWRpeFRvUG93ZXIgPSBnTG9uZy5mcm9tTnVtYmVyKE1hdGgucG93KHJhZGl4LCA4KSk7XG5cbiAgICB2YXIgcmVzdWx0ID0gZ0xvbmcuWkVSTztcbiAgICBmb3IgKHZhciBpID0gMDsgaSA8IHN0ci5sZW5ndGg7IGkgKz0gOCkge1xuICAgICAgdmFyIHNpemUgPSBNYXRoLm1pbig4LCBzdHIubGVuZ3RoIC0gaSk7XG4gICAgICB2YXIgdmFsdWUgPSBwYXJzZUludChzdHIuc3Vic3RyaW5nKGksIGkgKyBzaXplKSwgcmFkaXgpO1xuICAgICAgaWYgKHNpemUgPCA4KSB7XG4gICAgICAgIHZhciBwb3dlciA9IGdMb25nLmZyb21OdW1iZXIoTWF0aC5wb3cocmFkaXgsIHNpemUpKTtcbiAgICAgICAgcmVzdWx0ID0gcmVzdWx0Lm11bHRpcGx5KHBvd2VyKS5hZGQoZ0xvbmcuZnJvbU51bWJlcih2YWx1ZSkpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmVzdWx0ID0gcmVzdWx0Lm11bHRpcGx5KHJhZGl4VG9Qb3dlcik7XG4gICAgICAgIHJlc3VsdCA9IHJlc3VsdC5hZGQoZ0xvbmcuZnJvbU51bWJlcih2YWx1ZSkpO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG5cbiAgLyoqIEByZXR1cm4ge251bWJlcn0gVGhlIHZhbHVlLCBhc3N1bWluZyBpdCBpcyBhIDMyLWJpdCBpbnRlZ2VyLiAqL1xuICBwdWJsaWMgdG9JbnQoKTogbnVtYmVyIHtcbiAgICByZXR1cm4gdGhpcy5sb3dfO1xuICB9XG5cbiAgLyoqIEByZXR1cm4ge251bWJlcn0gVGhlIGNsb3Nlc3QgZmxvYXRpbmctcG9pbnQgcmVwcmVzZW50YXRpb24gdG8gdGhpcyB2YWx1ZS4gKi9cbiAgcHVibGljIHRvTnVtYmVyKCk6IG51bWJlciB7XG4gICAgcmV0dXJuIHRoaXMuaGlnaF8gKiBnTG9uZy5UV09fUFdSXzMyX0RCTF8gKyB0aGlzLmdldExvd0JpdHNVbnNpZ25lZCgpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwYXJhbSB7bnVtYmVyPX0gb3B0X3JhZGl4IFRoZSByYWRpeCBpbiB3aGljaCB0aGUgdGV4dCBzaG91bGQgYmUgd3JpdHRlbi5cbiAgICogQHJldHVybiB7c3RyaW5nfSBUaGUgdGV4dHVhbCByZXByZXNlbnRhdGlvbiBvZiB0aGlzIHZhbHVlLlxuICAgKi9cbiAgcHVibGljIHRvU3RyaW5nKG9wdF9yYWRpeD86IG51bWJlcik6IHN0cmluZyB7XG4gICAgdmFyIHJhZGl4ID0gb3B0X3JhZGl4IHx8IDEwO1xuICAgIGlmIChyYWRpeCA8IDIgfHwgMzYgPCByYWRpeCkge1xuICAgICAgdGhyb3cgRXJyb3IoJ3JhZGl4IG91dCBvZiByYW5nZTogJyArIHJhZGl4KTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5pc1plcm8oKSkge1xuICAgICAgcmV0dXJuICcwJztcbiAgICB9XG5cbiAgICBpZiAodGhpcy5pc05lZ2F0aXZlKCkpIHtcbiAgICAgIGlmICh0aGlzLmVxdWFscyhnTG9uZy5NSU5fVkFMVUUpKSB7XG4gICAgICAgIC8vIFdlIG5lZWQgdG8gY2hhbmdlIHRoZSBMb25nIHZhbHVlIGJlZm9yZSBpdCBjYW4gYmUgbmVnYXRlZCwgc28gd2UgcmVtb3ZlXG4gICAgICAgIC8vIHRoZSBib3R0b20tbW9zdCBkaWdpdCBpbiB0aGlzIGJhc2UgYW5kIHRoZW4gcmVjdXJzZSB0byBkbyB0aGUgcmVzdC5cbiAgICAgICAgdmFyIHJhZGl4TG9uZyA9IGdMb25nLmZyb21OdW1iZXIocmFkaXgpO1xuICAgICAgICB2YXIgZGl2ID0gdGhpcy5kaXYocmFkaXhMb25nKTtcbiAgICAgICAgdmFyIHJlbSA9IGRpdi5tdWx0aXBseShyYWRpeExvbmcpLnN1YnRyYWN0KHRoaXMpO1xuICAgICAgICByZXR1cm4gZGl2LnRvU3RyaW5nKHJhZGl4KSArIHJlbS50b0ludCgpLnRvU3RyaW5nKHJhZGl4KTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiAnLScgKyB0aGlzLm5lZ2F0ZSgpLnRvU3RyaW5nKHJhZGl4KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBEbyBzZXZlcmFsICg2KSBkaWdpdHMgZWFjaCB0aW1lIHRocm91Z2ggdGhlIGxvb3AsIHNvIGFzIHRvXG4gICAgLy8gbWluaW1pemUgdGhlIGNhbGxzIHRvIHRoZSB2ZXJ5IGV4cGVuc2l2ZSBlbXVsYXRlZCBkaXYuXG4gICAgdmFyIHJhZGl4VG9Qb3dlciA9IGdMb25nLmZyb21OdW1iZXIoTWF0aC5wb3cocmFkaXgsIDYpKTtcbiAgICB2YXIgcmVtOiBnTG9uZyA9IHRoaXM7XG4gICAgdmFyIHJlc3VsdCA9ICcnO1xuICAgIHdoaWxlICh0cnVlKSB7XG4gICAgICB2YXIgcmVtRGl2ID0gcmVtLmRpdihyYWRpeFRvUG93ZXIpO1xuICAgICAgdmFyIGludHZhbCA9IHJlbS5zdWJ0cmFjdChyZW1EaXYubXVsdGlwbHkocmFkaXhUb1Bvd2VyKSkudG9JbnQoKTtcbiAgICAgIHZhciBkaWdpdHMgPSBpbnR2YWwudG9TdHJpbmcocmFkaXgpO1xuXG4gICAgICByZW0gPSByZW1EaXY7XG4gICAgICBpZiAocmVtLmlzWmVybygpKSB7XG4gICAgICAgIHJldHVybiBkaWdpdHMgKyByZXN1bHQ7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB3aGlsZSAoZGlnaXRzLmxlbmd0aCA8IDYpIHtcbiAgICAgICAgICBkaWdpdHMgPSAnMCcgKyBkaWdpdHM7XG4gICAgICAgIH1cbiAgICAgICAgcmVzdWx0ID0gJycgKyBkaWdpdHMgKyByZXN1bHQ7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqIEByZXR1cm4ge251bWJlcn0gVGhlIGhpZ2ggMzItYml0cyBhcyBhIHNpZ25lZCB2YWx1ZS4gKi9cbiAgcHVibGljIGdldEhpZ2hCaXRzKCk6IG51bWJlciB7XG4gICAgcmV0dXJuIHRoaXMuaGlnaF87XG4gIH1cblxuICAvKiogQHJldHVybiB7bnVtYmVyfSBUaGUgbG93IDMyLWJpdHMgYXMgYSBzaWduZWQgdmFsdWUuICovXG4gIHB1YmxpYyBnZXRMb3dCaXRzKCk6IG51bWJlciB7XG4gICAgcmV0dXJuIHRoaXMubG93XztcbiAgfVxuXG4gIC8qKiBAcmV0dXJuIHtudW1iZXJ9IFRoZSBsb3cgMzItYml0cyBhcyBhbiB1bnNpZ25lZCB2YWx1ZS4gKi9cbiAgcHVibGljIGdldExvd0JpdHNVbnNpZ25lZCgpOiBudW1iZXIge1xuICAgIHJldHVybiAodGhpcy5sb3dfID49IDApID8gdGhpcy5sb3dfIDogZ0xvbmcuVFdPX1BXUl8zMl9EQkxfICsgdGhpcy5sb3dfO1xuICB9XG5cbiAgLyoqXG4gICAqIEByZXR1cm4ge251bWJlcn0gUmV0dXJucyB0aGUgbnVtYmVyIG9mIGJpdHMgbmVlZGVkIHRvIHJlcHJlc2VudCB0aGUgYWJzb2x1dGVcbiAgICogICAgIHZhbHVlIG9mIHRoaXMgTG9uZy5cbiAgICovXG4gIHB1YmxpYyBnZXROdW1CaXRzQWJzKCk6IG51bWJlciB7XG4gICAgaWYgKHRoaXMuaXNOZWdhdGl2ZSgpKSB7XG4gICAgICBpZiAodGhpcy5lcXVhbHMoZ0xvbmcuTUlOX1ZBTFVFKSkge1xuICAgICAgICByZXR1cm4gNjQ7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gdGhpcy5uZWdhdGUoKS5nZXROdW1CaXRzQWJzKCk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIHZhciB2YWwgPSB0aGlzLmhpZ2hfICE9IDAgPyB0aGlzLmhpZ2hfIDogdGhpcy5sb3dfO1xuICAgICAgZm9yICh2YXIgYml0ID0gMzE7IGJpdCA+IDA7IGJpdC0tKSB7XG4gICAgICAgIGlmICgodmFsICYgKDEgPDwgYml0KSkgIT0gMCkge1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICByZXR1cm4gdGhpcy5oaWdoXyAhPSAwID8gYml0ICsgMzMgOiBiaXQgKyAxO1xuICAgIH1cbiAgfVxuXG4gIC8qKiBAcmV0dXJuIHtib29sZWFufSBXaGV0aGVyIHRoaXMgdmFsdWUgaXMgemVyby4gKi9cbiAgcHVibGljIGlzWmVybygpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5oaWdoXyA9PSAwICYmIHRoaXMubG93XyA9PSAwO1xuICB9XG5cbiAgLyoqIEByZXR1cm4ge2Jvb2xlYW59IFdoZXRoZXIgdGhpcyB2YWx1ZSBpcyBuZWdhdGl2ZS4gKi9cbiAgcHVibGljIGlzTmVnYXRpdmUoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMuaGlnaF8gPCAwO1xuICB9XG5cbiAgLyoqIEByZXR1cm4ge2Jvb2xlYW59IFdoZXRoZXIgdGhpcyB2YWx1ZSBpcyBvZGQuICovXG4gIHB1YmxpYyBpc09kZCgpOiBib29sZWFuIHtcbiAgICByZXR1cm4gKHRoaXMubG93XyAmIDEpID09IDE7XG4gIH1cblxuICAvKipcbiAgICogQHBhcmFtIHtnTG9uZ30gb3RoZXIgTG9uZyB0byBjb21wYXJlIGFnYWluc3QuXG4gICAqIEByZXR1cm4ge2Jvb2xlYW59IFdoZXRoZXIgdGhpcyBMb25nIGVxdWFscyB0aGUgb3RoZXIuXG4gICAqL1xuICBwdWJsaWMgZXF1YWxzKG90aGVyOiBnTG9uZyk6IGJvb2xlYW4ge1xuICAgIHJldHVybiAodGhpcy5oaWdoXyA9PSBvdGhlci5oaWdoXykgJiYgKHRoaXMubG93XyA9PSBvdGhlci5sb3dfKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIHRvIGNvbXBhcmUgYWdhaW5zdC5cbiAgICogQHJldHVybiB7Ym9vbGVhbn0gV2hldGhlciB0aGlzIExvbmcgZG9lcyBub3QgZXF1YWwgdGhlIG90aGVyLlxuICAgKi9cbiAgcHVibGljIG5vdEVxdWFscyhvdGhlcjogZ0xvbmcpOiBib29sZWFuIHtcbiAgICByZXR1cm4gKHRoaXMuaGlnaF8gIT0gb3RoZXIuaGlnaF8pIHx8ICh0aGlzLmxvd18gIT0gb3RoZXIubG93Xyk7XG4gIH1cblxuICAvKipcbiAgICogQHBhcmFtIHtnTG9uZ30gb3RoZXIgTG9uZyB0byBjb21wYXJlIGFnYWluc3QuXG4gICAqIEByZXR1cm4ge2Jvb2xlYW59IFdoZXRoZXIgdGhpcyBMb25nIGlzIGxlc3MgdGhhbiB0aGUgb3RoZXIuXG4gICAqL1xuICBwdWJsaWMgbGVzc1RoYW4ob3RoZXI6IGdMb25nKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMuY29tcGFyZShvdGhlcikgPCAwO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwYXJhbSB7Z0xvbmd9IG90aGVyIExvbmcgdG8gY29tcGFyZSBhZ2FpbnN0LlxuICAgKiBAcmV0dXJuIHtib29sZWFufSBXaGV0aGVyIHRoaXMgTG9uZyBpcyBsZXNzIHRoYW4gb3IgZXF1YWwgdG8gdGhlIG90aGVyLlxuICAgKi9cbiAgcHVibGljIGxlc3NUaGFuT3JFcXVhbChvdGhlcjogZ0xvbmcpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5jb21wYXJlKG90aGVyKSA8PSAwO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwYXJhbSB7Z0xvbmd9IG90aGVyIExvbmcgdG8gY29tcGFyZSBhZ2FpbnN0LlxuICAgKiBAcmV0dXJuIHtib29sZWFufSBXaGV0aGVyIHRoaXMgTG9uZyBpcyBncmVhdGVyIHRoYW4gdGhlIG90aGVyLlxuICAgKi9cbiAgcHVibGljIGdyZWF0ZXJUaGFuKG90aGVyOiBnTG9uZyk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLmNvbXBhcmUob3RoZXIpID4gMDtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIHRvIGNvbXBhcmUgYWdhaW5zdC5cbiAgICogQHJldHVybiB7Ym9vbGVhbn0gV2hldGhlciB0aGlzIExvbmcgaXMgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIHRoZSBvdGhlci5cbiAgICovXG4gIHB1YmxpYyBncmVhdGVyVGhhbk9yRXF1YWwob3RoZXI6IGdMb25nKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMuY29tcGFyZShvdGhlcikgPj0gMDtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb21wYXJlcyB0aGlzIExvbmcgd2l0aCB0aGUgZ2l2ZW4gb25lLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIHRvIGNvbXBhcmUgYWdhaW5zdC5cbiAgICogQHJldHVybiB7bnVtYmVyfSAwIGlmIHRoZXkgYXJlIHRoZSBzYW1lLCAxIGlmIHRoZSB0aGlzIGlzIGdyZWF0ZXIsIGFuZCAtMVxuICAgKiAgICAgaWYgdGhlIGdpdmVuIG9uZSBpcyBncmVhdGVyLlxuICAgKi9cbiAgcHVibGljIGNvbXBhcmUob3RoZXI6IGdMb25nKTogbnVtYmVyIHtcbiAgICBpZiAodGhpcy5lcXVhbHMob3RoZXIpKSB7XG4gICAgICByZXR1cm4gMDtcbiAgICB9XG5cbiAgICB2YXIgdGhpc05lZyA9IHRoaXMuaXNOZWdhdGl2ZSgpO1xuICAgIHZhciBvdGhlck5lZyA9IG90aGVyLmlzTmVnYXRpdmUoKTtcbiAgICBpZiAodGhpc05lZyAmJiAhb3RoZXJOZWcpIHtcbiAgICAgIHJldHVybiAtMTtcbiAgICB9XG4gICAgaWYgKCF0aGlzTmVnICYmIG90aGVyTmVnKSB7XG4gICAgICByZXR1cm4gMTtcbiAgICB9XG5cbiAgICAvLyBhdCB0aGlzIHBvaW50LCB0aGUgc2lnbnMgYXJlIHRoZSBzYW1lLCBzbyBzdWJ0cmFjdGlvbiB3aWxsIG5vdCBvdmVyZmxvd1xuICAgIGlmICh0aGlzLnN1YnRyYWN0KG90aGVyKS5pc05lZ2F0aXZlKCkpIHtcbiAgICAgIHJldHVybiAtMTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIDE7XG4gICAgfVxuICB9XG5cbiAgLyoqIEByZXR1cm4geyFnTG9uZ30gVGhlIG5lZ2F0aW9uIG9mIHRoaXMgdmFsdWUuICovXG4gIHB1YmxpYyBuZWdhdGUoKTogZ0xvbmcge1xuICAgIGlmICh0aGlzLmVxdWFscyhnTG9uZy5NSU5fVkFMVUUpKSB7XG4gICAgICByZXR1cm4gZ0xvbmcuTUlOX1ZBTFVFO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gdGhpcy5ub3QoKS5hZGQoZ0xvbmcuT05FKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgc3VtIG9mIHRoaXMgYW5kIHRoZSBnaXZlbiBMb25nLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIHRvIGFkZCB0byB0aGlzIG9uZS5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGUgc3VtIG9mIHRoaXMgYW5kIHRoZSBnaXZlbiBMb25nLlxuICAgKi9cbiAgcHVibGljIGFkZChvdGhlcjogZ0xvbmcpOiBnTG9uZyB7XG4gICAgLy8gRGl2aWRlIGVhY2ggbnVtYmVyIGludG8gNCBjaHVua3Mgb2YgMTYgYml0cywgYW5kIHRoZW4gc3VtIHRoZSBjaHVua3MuXG5cbiAgICB2YXIgYTQ4ID0gdGhpcy5oaWdoXyA+Pj4gMTY7XG4gICAgdmFyIGEzMiA9IHRoaXMuaGlnaF8gJiAweEZGRkY7XG4gICAgdmFyIGExNiA9IHRoaXMubG93XyA+Pj4gMTY7XG4gICAgdmFyIGEwMCA9IHRoaXMubG93XyAmIDB4RkZGRjtcblxuICAgIHZhciBiNDggPSBvdGhlci5oaWdoXyA+Pj4gMTY7XG4gICAgdmFyIGIzMiA9IG90aGVyLmhpZ2hfICYgMHhGRkZGO1xuICAgIHZhciBiMTYgPSBvdGhlci5sb3dfID4+PiAxNjtcbiAgICB2YXIgYjAwID0gb3RoZXIubG93XyAmIDB4RkZGRjtcblxuICAgIHZhciBjNDggPSAwLCBjMzIgPSAwLCBjMTYgPSAwLCBjMDAgPSAwO1xuICAgIGMwMCArPSBhMDAgKyBiMDA7XG4gICAgYzE2ICs9IGMwMCA+Pj4gMTY7XG4gICAgYzAwICY9IDB4RkZGRjtcbiAgICBjMTYgKz0gYTE2ICsgYjE2O1xuICAgIGMzMiArPSBjMTYgPj4+IDE2O1xuICAgIGMxNiAmPSAweEZGRkY7XG4gICAgYzMyICs9IGEzMiArIGIzMjtcbiAgICBjNDggKz0gYzMyID4+PiAxNjtcbiAgICBjMzIgJj0gMHhGRkZGO1xuICAgIGM0OCArPSBhNDggKyBiNDg7XG4gICAgYzQ4ICY9IDB4RkZGRjtcbiAgICByZXR1cm4gZ0xvbmcuZnJvbUJpdHMoKGMxNiA8PCAxNikgfCBjMDAsIChjNDggPDwgMTYpIHwgYzMyKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkaWZmZXJlbmNlIG9mIHRoaXMgYW5kIHRoZSBnaXZlbiBMb25nLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIHRvIHN1YnRyYWN0IGZyb20gdGhpcy5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGUgZGlmZmVyZW5jZSBvZiB0aGlzIGFuZCB0aGUgZ2l2ZW4gTG9uZy5cbiAgICovXG4gIHB1YmxpYyBzdWJ0cmFjdChvdGhlcjogZ0xvbmcpOiBnTG9uZyB7XG4gICAgcmV0dXJuIHRoaXMuYWRkKG90aGVyLm5lZ2F0ZSgpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBwcm9kdWN0IG9mIHRoaXMgYW5kIHRoZSBnaXZlbiBsb25nLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIHRvIG11bHRpcGx5IHdpdGggdGhpcy5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGUgcHJvZHVjdCBvZiB0aGlzIGFuZCB0aGUgb3RoZXIuXG4gICAqL1xuICBwdWJsaWMgbXVsdGlwbHkob3RoZXI6IGdMb25nKTogZ0xvbmcge1xuICAgIGlmICh0aGlzLmlzWmVybygpKSB7XG4gICAgICByZXR1cm4gZ0xvbmcuWkVSTztcbiAgICB9IGVsc2UgaWYgKG90aGVyLmlzWmVybygpKSB7XG4gICAgICByZXR1cm4gZ0xvbmcuWkVSTztcbiAgICB9XG5cbiAgICBpZiAodGhpcy5lcXVhbHMoZ0xvbmcuTUlOX1ZBTFVFKSkge1xuICAgICAgcmV0dXJuIG90aGVyLmlzT2RkKCkgPyBnTG9uZy5NSU5fVkFMVUUgOiBnTG9uZy5aRVJPO1xuICAgIH0gZWxzZSBpZiAob3RoZXIuZXF1YWxzKGdMb25nLk1JTl9WQUxVRSkpIHtcbiAgICAgIHJldHVybiB0aGlzLmlzT2RkKCkgPyBnTG9uZy5NSU5fVkFMVUUgOiBnTG9uZy5aRVJPO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmlzTmVnYXRpdmUoKSkge1xuICAgICAgaWYgKG90aGVyLmlzTmVnYXRpdmUoKSkge1xuICAgICAgICByZXR1cm4gdGhpcy5uZWdhdGUoKS5tdWx0aXBseShvdGhlci5uZWdhdGUoKSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gdGhpcy5uZWdhdGUoKS5tdWx0aXBseShvdGhlcikubmVnYXRlKCk7XG4gICAgICB9XG4gICAgfSBlbHNlIGlmIChvdGhlci5pc05lZ2F0aXZlKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLm11bHRpcGx5KG90aGVyLm5lZ2F0ZSgpKS5uZWdhdGUoKTtcbiAgICB9XG5cbiAgICAvLyBJZiBib3RoIGxvbmdzIGFyZSBzbWFsbCwgdXNlIGZsb2F0IG11bHRpcGxpY2F0aW9uXG4gICAgaWYgKHRoaXMubGVzc1RoYW4oZ0xvbmcuVFdPX1BXUl8yNF8pICYmXG4gICAgICAgIG90aGVyLmxlc3NUaGFuKGdMb25nLlRXT19QV1JfMjRfKSkge1xuICAgICAgcmV0dXJuIGdMb25nLmZyb21OdW1iZXIodGhpcy50b051bWJlcigpICogb3RoZXIudG9OdW1iZXIoKSk7XG4gICAgfVxuXG4gICAgLy8gRGl2aWRlIGVhY2ggbG9uZyBpbnRvIDQgY2h1bmtzIG9mIDE2IGJpdHMsIGFuZCB0aGVuIGFkZCB1cCA0eDQgcHJvZHVjdHMuXG4gICAgLy8gV2UgY2FuIHNraXAgcHJvZHVjdHMgdGhhdCB3b3VsZCBvdmVyZmxvdy5cblxuICAgIHZhciBhNDggPSB0aGlzLmhpZ2hfID4+PiAxNjtcbiAgICB2YXIgYTMyID0gdGhpcy5oaWdoXyAmIDB4RkZGRjtcbiAgICB2YXIgYTE2ID0gdGhpcy5sb3dfID4+PiAxNjtcbiAgICB2YXIgYTAwID0gdGhpcy5sb3dfICYgMHhGRkZGO1xuXG4gICAgdmFyIGI0OCA9IG90aGVyLmhpZ2hfID4+PiAxNjtcbiAgICB2YXIgYjMyID0gb3RoZXIuaGlnaF8gJiAweEZGRkY7XG4gICAgdmFyIGIxNiA9IG90aGVyLmxvd18gPj4+IDE2O1xuICAgIHZhciBiMDAgPSBvdGhlci5sb3dfICYgMHhGRkZGO1xuXG4gICAgdmFyIGM0OCA9IDAsIGMzMiA9IDAsIGMxNiA9IDAsIGMwMCA9IDA7XG4gICAgYzAwICs9IGEwMCAqIGIwMDtcbiAgICBjMTYgKz0gYzAwID4+PiAxNjtcbiAgICBjMDAgJj0gMHhGRkZGO1xuICAgIGMxNiArPSBhMTYgKiBiMDA7XG4gICAgYzMyICs9IGMxNiA+Pj4gMTY7XG4gICAgYzE2ICY9IDB4RkZGRjtcbiAgICBjMTYgKz0gYTAwICogYjE2O1xuICAgIGMzMiArPSBjMTYgPj4+IDE2O1xuICAgIGMxNiAmPSAweEZGRkY7XG4gICAgYzMyICs9IGEzMiAqIGIwMDtcbiAgICBjNDggKz0gYzMyID4+PiAxNjtcbiAgICBjMzIgJj0gMHhGRkZGO1xuICAgIGMzMiArPSBhMTYgKiBiMTY7XG4gICAgYzQ4ICs9IGMzMiA+Pj4gMTY7XG4gICAgYzMyICY9IDB4RkZGRjtcbiAgICBjMzIgKz0gYTAwICogYjMyO1xuICAgIGM0OCArPSBjMzIgPj4+IDE2O1xuICAgIGMzMiAmPSAweEZGRkY7XG4gICAgYzQ4ICs9IGE0OCAqIGIwMCArIGEzMiAqIGIxNiArIGExNiAqIGIzMiArIGEwMCAqIGI0ODtcbiAgICBjNDggJj0gMHhGRkZGO1xuICAgIHJldHVybiBnTG9uZy5mcm9tQml0cygoYzE2IDw8IDE2KSB8IGMwMCwgKGM0OCA8PCAxNikgfCBjMzIpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhpcyBMb25nIGRpdmlkZWQgYnkgdGhlIGdpdmVuIG9uZS5cbiAgICogQHBhcmFtIHtnTG9uZ30gb3RoZXIgTG9uZyBieSB3aGljaCB0byBkaXZpZGUuXG4gICAqIEByZXR1cm4geyFnTG9uZ30gVGhpcyBMb25nIGRpdmlkZWQgYnkgdGhlIGdpdmVuIG9uZS5cbiAgICovXG4gIHB1YmxpYyBkaXYob3RoZXI6IGdMb25nKTogZ0xvbmcge1xuICAgIGlmIChvdGhlci5pc1plcm8oKSkge1xuICAgICAgdGhyb3cgRXJyb3IoJ2RpdmlzaW9uIGJ5IHplcm8nKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuaXNaZXJvKCkpIHtcbiAgICAgIHJldHVybiBnTG9uZy5aRVJPO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmVxdWFscyhnTG9uZy5NSU5fVkFMVUUpKSB7XG4gICAgICBpZiAob3RoZXIuZXF1YWxzKGdMb25nLk9ORSkgfHxcbiAgICAgICAgICBvdGhlci5lcXVhbHMoZ0xvbmcuTkVHX09ORSkpIHtcbiAgICAgICAgcmV0dXJuIGdMb25nLk1JTl9WQUxVRTsgIC8vIHJlY2FsbCB0aGF0IC1NSU5fVkFMVUUgPT0gTUlOX1ZBTFVFXG4gICAgICB9IGVsc2UgaWYgKG90aGVyLmVxdWFscyhnTG9uZy5NSU5fVkFMVUUpKSB7XG4gICAgICAgIHJldHVybiBnTG9uZy5PTkU7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICAvLyBBdCB0aGlzIHBvaW50LCB3ZSBoYXZlIHxvdGhlcnwgPj0gMiwgc28gfHRoaXMvb3RoZXJ8IDwgfE1JTl9WQUxVRXwuXG4gICAgICAgIHZhciBoYWxmVGhpcyA9IHRoaXMuc2hpZnRSaWdodCgxKTtcbiAgICAgICAgdmFyIGxfYXBwcm94ID0gaGFsZlRoaXMuZGl2KG90aGVyKS5zaGlmdExlZnQoMSk7XG4gICAgICAgIGlmIChsX2FwcHJveC5lcXVhbHMoZ0xvbmcuWkVSTykpIHtcbiAgICAgICAgICByZXR1cm4gb3RoZXIuaXNOZWdhdGl2ZSgpID8gZ0xvbmcuT05FIDogZ0xvbmcuTkVHX09ORTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICB2YXIgcmVtID0gdGhpcy5zdWJ0cmFjdChvdGhlci5tdWx0aXBseShsX2FwcHJveCkpO1xuICAgICAgICAgIHZhciByZXN1bHQgPSBsX2FwcHJveC5hZGQocmVtLmRpdihvdGhlcikpO1xuICAgICAgICAgIHJldHVybiByZXN1bHQ7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKG90aGVyLmVxdWFscyhnTG9uZy5NSU5fVkFMVUUpKSB7XG4gICAgICByZXR1cm4gZ0xvbmcuWkVSTztcbiAgICB9XG5cbiAgICBpZiAodGhpcy5pc05lZ2F0aXZlKCkpIHtcbiAgICAgIGlmIChvdGhlci5pc05lZ2F0aXZlKCkpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubmVnYXRlKCkuZGl2KG90aGVyLm5lZ2F0ZSgpKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiB0aGlzLm5lZ2F0ZSgpLmRpdihvdGhlcikubmVnYXRlKCk7XG4gICAgICB9XG4gICAgfSBlbHNlIGlmIChvdGhlci5pc05lZ2F0aXZlKCkpIHtcbiAgICAgIHJldHVybiB0aGlzLmRpdihvdGhlci5uZWdhdGUoKSkubmVnYXRlKCk7XG4gICAgfVxuXG4gICAgLy8gUmVwZWF0IHRoZSBmb2xsb3dpbmcgdW50aWwgdGhlIHJlbWFpbmRlciBpcyBsZXNzIHRoYW4gb3RoZXI6ICBmaW5kIGFcbiAgICAvLyBmbG9hdGluZy1wb2ludCB0aGF0IGFwcHJveGltYXRlcyByZW1haW5kZXIgLyBvdGhlciAqZnJvbSBiZWxvdyosIGFkZCB0aGlzXG4gICAgLy8gaW50byB0aGUgcmVzdWx0LCBhbmQgc3VidHJhY3QgaXQgZnJvbSB0aGUgcmVtYWluZGVyLiAgSXQgaXMgY3JpdGljYWwgdGhhdFxuICAgIC8vIHRoZSBhcHByb3hpbWF0ZSB2YWx1ZSBpcyBsZXNzIHRoYW4gb3IgZXF1YWwgdG8gdGhlIHJlYWwgdmFsdWUgc28gdGhhdCB0aGVcbiAgICAvLyByZW1haW5kZXIgbmV2ZXIgYmVjb21lcyBuZWdhdGl2ZS5cbiAgICB2YXIgcmVzID0gZ0xvbmcuWkVSTztcbiAgICB2YXIgcmVtOiBnTG9uZyA9IHRoaXM7XG4gICAgd2hpbGUgKHJlbS5ncmVhdGVyVGhhbk9yRXF1YWwob3RoZXIpKSB7XG4gICAgICAvLyBBcHByb3hpbWF0ZSB0aGUgcmVzdWx0IG9mIGRpdmlzaW9uLiBUaGlzIG1heSBiZSBhIGxpdHRsZSBncmVhdGVyIG9yXG4gICAgICAvLyBzbWFsbGVyIHRoYW4gdGhlIGFjdHVhbCB2YWx1ZS5cbiAgICAgIHZhciBhcHByb3ggPSBNYXRoLm1heCgxLCBNYXRoLmZsb29yKHJlbS50b051bWJlcigpIC8gb3RoZXIudG9OdW1iZXIoKSkpO1xuXG4gICAgICAvLyBXZSB3aWxsIHR3ZWFrIHRoZSBhcHByb3hpbWF0ZSByZXN1bHQgYnkgY2hhbmdpbmcgaXQgaW4gdGhlIDQ4LXRoIGRpZ2l0IG9yXG4gICAgICAvLyB0aGUgc21hbGxlc3Qgbm9uLWZyYWN0aW9uYWwgZGlnaXQsIHdoaWNoZXZlciBpcyBsYXJnZXIuXG4gICAgICB2YXIgbG9nMiA9IE1hdGguY2VpbChNYXRoLmxvZyhhcHByb3gpIC8gTWF0aC5MTjIpO1xuICAgICAgdmFyIGRlbHRhID0gMTtcbiAgICAgIGlmIChsb2cyID4gNDgpXG4gICAgICAgIGRlbHRhID0gTWF0aC5wb3coMiwgbG9nMiAtIDQ4KTtcblxuICAgICAgLy8gRGVjcmVhc2UgdGhlIGFwcHJveGltYXRpb24gdW50aWwgaXQgaXMgc21hbGxlciB0aGFuIHRoZSByZW1haW5kZXIuICBOb3RlXG4gICAgICAvLyB0aGF0IGlmIGl0IGlzIHRvbyBsYXJnZSwgdGhlIHByb2R1Y3Qgb3ZlcmZsb3dzIGFuZCBpcyBuZWdhdGl2ZS5cbiAgICAgIHZhciBhcHByb3hSZXMgPSBnTG9uZy5mcm9tTnVtYmVyKGFwcHJveCk7XG4gICAgICB2YXIgYXBwcm94UmVtID0gYXBwcm94UmVzLm11bHRpcGx5KG90aGVyKTtcbiAgICAgIHdoaWxlIChhcHByb3hSZW0uaXNOZWdhdGl2ZSgpIHx8IGFwcHJveFJlbS5ncmVhdGVyVGhhbihyZW0pKSB7XG4gICAgICAgIGFwcHJveCAtPSBkZWx0YTtcbiAgICAgICAgYXBwcm94UmVzID0gZ0xvbmcuZnJvbU51bWJlcihhcHByb3gpO1xuICAgICAgICBhcHByb3hSZW0gPSBhcHByb3hSZXMubXVsdGlwbHkob3RoZXIpO1xuICAgICAgfVxuXG4gICAgICAvLyBXZSBrbm93IHRoZSBhbnN3ZXIgY2FuJ3QgYmUgemVyby4uLiBhbmQgYWN0dWFsbHksIHplcm8gd291bGQgY2F1c2VcbiAgICAgIC8vIGluZmluaXRlIHJlY3Vyc2lvbiBzaW5jZSB3ZSB3b3VsZCBtYWtlIG5vIHByb2dyZXNzLlxuICAgICAgaWYgKGFwcHJveFJlcy5pc1plcm8oKSkge1xuICAgICAgICBhcHByb3hSZXMgPSBnTG9uZy5PTkU7XG4gICAgICB9XG5cbiAgICAgIHJlcyA9IHJlcy5hZGQoYXBwcm94UmVzKTtcbiAgICAgIHJlbSA9IHJlbS5zdWJ0cmFjdChhcHByb3hSZW0pO1xuICAgIH1cbiAgICByZXR1cm4gcmVzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhpcyBMb25nIG1vZHVsbyB0aGUgZ2l2ZW4gb25lLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBMb25nIGJ5IHdoaWNoIHRvIG1vZC5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGlzIExvbmcgbW9kdWxvIHRoZSBnaXZlbiBvbmUuXG4gICAqL1xuICBwdWJsaWMgbW9kdWxvKG90aGVyOiBnTG9uZyk6IGdMb25nIHtcbiAgICByZXR1cm4gdGhpcy5zdWJ0cmFjdCh0aGlzLmRpdihvdGhlcikubXVsdGlwbHkob3RoZXIpKTtcbiAgfVxuXG4gIC8qKiBAcmV0dXJuIHshZ0xvbmd9IFRoZSBiaXR3aXNlLU5PVCBvZiB0aGlzIHZhbHVlLiAqL1xuICBwdWJsaWMgbm90KCk6IGdMb25nIHtcbiAgICByZXR1cm4gZ0xvbmcuZnJvbUJpdHMofnRoaXMubG93XywgfnRoaXMuaGlnaF8pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGJpdHdpc2UtQU5EIG9mIHRoaXMgTG9uZyBhbmQgdGhlIGdpdmVuIG9uZS5cbiAgICogQHBhcmFtIHtnTG9uZ30gb3RoZXIgVGhlIExvbmcgd2l0aCB3aGljaCB0byBBTkQuXG4gICAqIEByZXR1cm4geyFnTG9uZ30gVGhlIGJpdHdpc2UtQU5EIG9mIHRoaXMgYW5kIHRoZSBvdGhlci5cbiAgICovXG4gIHB1YmxpYyBhbmQob3RoZXI6IGdMb25nKTogZ0xvbmcge1xuICAgIHJldHVybiBnTG9uZy5mcm9tQml0cyh0aGlzLmxvd18gJiBvdGhlci5sb3dfLFxuICAgICAgICAgICAgICAgICAgICAgICAgICB0aGlzLmhpZ2hfICYgb3RoZXIuaGlnaF8pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGJpdHdpc2UtT1Igb2YgdGhpcyBMb25nIGFuZCB0aGUgZ2l2ZW4gb25lLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBUaGUgTG9uZyB3aXRoIHdoaWNoIHRvIE9SLlxuICAgKiBAcmV0dXJuIHshZ0xvbmd9IFRoZSBiaXR3aXNlLU9SIG9mIHRoaXMgYW5kIHRoZSBvdGhlci5cbiAgICovXG4gIHB1YmxpYyBvcihvdGhlcjogZ0xvbmcpOiBnTG9uZyB7XG4gICAgcmV0dXJuIGdMb25nLmZyb21CaXRzKHRoaXMubG93XyB8IG90aGVyLmxvd18sXG4gICAgICAgICAgICAgICAgICAgICAgICAgIHRoaXMuaGlnaF8gfCBvdGhlci5oaWdoXyk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgYml0d2lzZS1YT1Igb2YgdGhpcyBMb25nIGFuZCB0aGUgZ2l2ZW4gb25lLlxuICAgKiBAcGFyYW0ge2dMb25nfSBvdGhlciBUaGUgTG9uZyB3aXRoIHdoaWNoIHRvIFhPUi5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGUgYml0d2lzZS1YT1Igb2YgdGhpcyBhbmQgdGhlIG90aGVyLlxuICAgKi9cbiAgcHVibGljIHhvcihvdGhlcjogZ0xvbmcpOiBnTG9uZyB7XG4gICAgcmV0dXJuIGdMb25nLmZyb21CaXRzKHRoaXMubG93XyBeIG90aGVyLmxvd18sIHRoaXMuaGlnaF8gXiBvdGhlci5oaWdoXyk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGlzIExvbmcgd2l0aCBiaXRzIHNoaWZ0ZWQgdG8gdGhlIGxlZnQgYnkgdGhlIGdpdmVuIGFtb3VudC5cbiAgICogQHBhcmFtIHtudW1iZXJ9IG51bUJpdHMgVGhlIG51bWJlciBvZiBiaXRzIGJ5IHdoaWNoIHRvIHNoaWZ0LlxuICAgKiBAcmV0dXJuIHshZ0xvbmd9IFRoaXMgc2hpZnRlZCB0byB0aGUgbGVmdCBieSB0aGUgZ2l2ZW4gYW1vdW50LlxuICAgKi9cbiAgcHVibGljIHNoaWZ0TGVmdChudW1CaXRzOiBudW1iZXIpOiBnTG9uZyB7XG4gICAgbnVtQml0cyAmPSA2MztcbiAgICBpZiAobnVtQml0cyA9PSAwKSB7XG4gICAgICByZXR1cm4gdGhpcztcbiAgICB9IGVsc2Uge1xuICAgICAgdmFyIGxvdyA9IHRoaXMubG93XztcbiAgICAgIGlmIChudW1CaXRzIDwgMzIpIHtcbiAgICAgICAgdmFyIGhpZ2ggPSB0aGlzLmhpZ2hfO1xuICAgICAgICByZXR1cm4gZ0xvbmcuZnJvbUJpdHMobG93IDw8IG51bUJpdHMsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAoaGlnaCA8PCBudW1CaXRzKSB8IChsb3cgPj4+ICgzMiAtIG51bUJpdHMpKSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gZ0xvbmcuZnJvbUJpdHMoMCwgbG93IDw8IChudW1CaXRzIC0gMzIpKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGlzIExvbmcgd2l0aCBiaXRzIHNoaWZ0ZWQgdG8gdGhlIHJpZ2h0IGJ5IHRoZSBnaXZlbiBhbW91bnQuXG4gICAqIEBwYXJhbSB7bnVtYmVyfSBudW1CaXRzIFRoZSBudW1iZXIgb2YgYml0cyBieSB3aGljaCB0byBzaGlmdC5cbiAgICogQHJldHVybiB7IWdMb25nfSBUaGlzIHNoaWZ0ZWQgdG8gdGhlIHJpZ2h0IGJ5IHRoZSBnaXZlbiBhbW91bnQuXG4gICAqL1xuICBwdWJsaWMgc2hpZnRSaWdodChudW1CaXRzOiBudW1iZXIpOiBnTG9uZyB7XG4gICAgbnVtQml0cyAmPSA2MztcbiAgICBpZiAobnVtQml0cyA9PSAwKSB7XG4gICAgICByZXR1cm4gdGhpcztcbiAgICB9IGVsc2Uge1xuICAgICAgdmFyIGhpZ2ggPSB0aGlzLmhpZ2hfO1xuICAgICAgaWYgKG51bUJpdHMgPCAzMikge1xuICAgICAgICB2YXIgbG93ID0gdGhpcy5sb3dfO1xuICAgICAgICByZXR1cm4gZ0xvbmcuZnJvbUJpdHMoXG4gICAgICAgICAgICAobG93ID4+PiBudW1CaXRzKSB8IChoaWdoIDw8ICgzMiAtIG51bUJpdHMpKSxcbiAgICAgICAgICAgIGhpZ2ggPj4gbnVtQml0cyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXR1cm4gZ0xvbmcuZnJvbUJpdHMoXG4gICAgICAgICAgICBoaWdoID4+IChudW1CaXRzIC0gMzIpLFxuICAgICAgICAgICAgaGlnaCA+PSAwID8gMCA6IC0xKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGlzIExvbmcgd2l0aCBiaXRzIHNoaWZ0ZWQgdG8gdGhlIHJpZ2h0IGJ5IHRoZSBnaXZlbiBhbW91bnQsIHdpdGhcbiAgICogdGhlIG5ldyB0b3AgYml0cyBtYXRjaGluZyB0aGUgY3VycmVudCBzaWduIGJpdC5cbiAgICogQHBhcmFtIHtudW1iZXJ9IG51bUJpdHMgVGhlIG51bWJlciBvZiBiaXRzIGJ5IHdoaWNoIHRvIHNoaWZ0LlxuICAgKiBAcmV0dXJuIHshZ0xvbmd9IFRoaXMgc2hpZnRlZCB0byB0aGUgcmlnaHQgYnkgdGhlIGdpdmVuIGFtb3VudCwgd2l0aFxuICAgKiAgICAgemVyb3MgcGxhY2VkIGludG8gdGhlIG5ldyBsZWFkaW5nIGJpdHMuXG4gICAqL1xuICBwdWJsaWMgc2hpZnRSaWdodFVuc2lnbmVkKG51bUJpdHM6IG51bWJlcik6IGdMb25nIHtcbiAgICBudW1CaXRzICY9IDYzO1xuICAgIGlmIChudW1CaXRzID09IDApIHtcbiAgICAgIHJldHVybiB0aGlzO1xuICAgIH0gZWxzZSB7XG4gICAgICB2YXIgaGlnaCA9IHRoaXMuaGlnaF87XG4gICAgICBpZiAobnVtQml0cyA8IDMyKSB7XG4gICAgICAgIHZhciBsb3cgPSB0aGlzLmxvd187XG4gICAgICAgIHJldHVybiBnTG9uZy5mcm9tQml0cyhcbiAgICAgICAgICAgIChsb3cgPj4+IG51bUJpdHMpIHwgKGhpZ2ggPDwgKDMyIC0gbnVtQml0cykpLFxuICAgICAgICAgICAgaGlnaCA+Pj4gbnVtQml0cyk7XG4gICAgICB9IGVsc2UgaWYgKG51bUJpdHMgPT0gMzIpIHtcbiAgICAgICAgcmV0dXJuIGdMb25nLmZyb21CaXRzKGhpZ2gsIDApO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIGdMb25nLmZyb21CaXRzKGhpZ2ggPj4+IChudW1CaXRzIC0gMzIpLCAwKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cbiJdfQ==