// A power-of-two segregated freelist "heap",
// for explicit memory management into a buffer.
// by Emery Berger, www.cs.umass.edu/~emery
"use strict";
var Heap = (function () {
    // size = total amount of memory for the heap.
    function Heap(size) {
        this.size = size;
        // A map of size classes per chunk (see above).
        this._sizeMap = {};
        this._buffer = new Buffer(size);
        this._remaining = size; // the whole thing is available.
        this._offset = 0; // start of the buffer.
        // Initialize the freelists.
        this._freeLists = new Array(Heap._numSizeClasses);
        for (var i = 0; i < Heap._numSizeClasses; i++) {
            this._freeLists[i] = [];
        }
    }
    // Allocate size bytes, returning the "address".
    Heap.prototype.malloc = function (size) {
        // if size is less than a word, round it up to a word (4 bytes).
        if (size <= 4) {
            size = 4;
        }
        // if we are out of memory, throw an exception.
        if (this._remaining < size) {
            // TODO: could actually scan larger freelists to see if there's
            // free memory there.
            throw "out of memory";
        }
        // compute the size class.
        var addr;
        var cl;
        cl = Heap.size_to_class(size);
        addr = this._freeLists[cl].pop();
        // if there's no more memory, get some.
        if (addr === undefined) {
            addr = this.refill(cl);
        }
        return addr;
    };
    // Deallocate memory.
    Heap.prototype.free = function (addr) {
        // push this address onto the appropriate freelist.
        // first, mask the address.
        var masked = addr & ~(Heap._chunkSize - 1);
        // next, look up the class using the masked address.
        var cl = this._sizeMap[masked];
        // finally, push onto the appropriate free list.
        // TODO: for sanity, we could check to see if this was *really* freed
        // and drop it if not.
        this._freeLists[cl].push(addr);
    };
    // Store a word (32-bits) at this address.
    Heap.prototype.store_word = function (addr, value) {
        // TODO: add sanity checks?
        this._buffer.writeInt32LE(value, addr);
    };
    // Access a byte at this location.
    Heap.prototype.get_byte = function (addr) {
        // TODO: add sanity checks?
        return this._buffer[addr];
    };
    Heap.prototype.get_word = function (addr) {
        return this._buffer.readInt32LE(addr);
    };
    Heap.prototype.get_buffer = function (addr, len) {
        return this._buffer.slice(addr, addr + len);
    };
    Heap.prototype.get_signed_byte = function (addr) {
        return this._buffer.readInt8(addr);
    };
    Heap.prototype.set_byte = function (addr, value) {
        this._buffer[addr] = value;
        ;
    };
    Heap.prototype.set_signed_byte = function (addr, value) {
        this._buffer.writeInt8(value, addr);
    };
    /**
     * Copy len bytes from srcAddr to dstAddr.
     */
    Heap.prototype.memcpy = function (srcAddr, dstAddr, len) {
        this._buffer.copy(this._buffer, dstAddr, srcAddr, srcAddr + len);
    };
    // Get more memory for a particular size class.
    Heap.prototype.refill = function (cl) {
        // Get the largest size for this class.
        var sz = this.cl_to_size(cl);
        // Figure out how many objects we are going to "allocate".
        var count = Math.floor(Heap._chunkSize / sz);
        if (count < 1) {
            count = 1;
        }
        // Now store the size class *for the first object* only.
        // We will later look up this object via "pointer arithmetic".
        var addr = this._offset;
        this._sizeMap[addr] = cl;
        // Add each one to the freelist.
        for (var i = 0; i < count; i++) {
            this._remaining -= sz;
            addr = this._offset;
            this._freeLists[cl].push(addr);
            this._offset += sz;
        }
        return addr;
    };
    // Computes ceil(log2(num)).
    Heap.ilog2 = function (num) {
        var log2 = 0;
        var value = 1;
        while (value < num) {
            value <<= (1);
            log2++;
        }
        return (log2);
    };
    // power-of-two size classes (just a ref to ilog2).
    Heap.size_to_class = function (size) {
        return Heap.ilog2(size);
    };
    // see above: classes are just powers of two.
    Heap.prototype.cl_to_size = function (cl) {
        return (1 << cl);
    };
    // The total number of size classes.
    Heap._numSizeClasses = 64; // way more than we'll ever need.
    // How much to grab at one time.
    Heap._chunkSize = 4096; // should be a power of two.
    return Heap;
}());
exports.__esModule = true;
exports["default"] = Heap;
//# sourceMappingURL=data:application/json;base64,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