"use strict";
var path = require('path');
var fs = require('fs');
/**
 * Given a folder that contains the contents of a JAR file, parses the MANIFEST
 * file and other metadata and makes it available to Doppio in a convenient
 * format.
 *
 * @see http://docs.oracle.com/javase/7/docs/technotes/guides/jar/jar.html#Notes_on_Manifest_and_Signature_Files
 * @todo Add a function for producing a Java object from this.
 */
var JAR = (function () {
    /**
     * Calls doneCb when it is finished opening and parsing the MANIFEST file.
     */
    function JAR(dir, doneCb) {
        var _this = this;
        this.attributes = {};
        this.classpath = [];
        // Try to read the MANIFEST file.
        fs.readFile(path.resolve(dir, 'META-INF/MANIFEST.MF'), function (err, data) {
            if (err)
                return doneCb(err);
            // Parse the file!
            // If a line *begins* with a space, then it's a continuation of the
            // previous value.
            var currentAttribute = null;
            var currentValue = "";
            // Figure out which line ending the MANIFEST uses.
            var ending = '\n';
            var dataStr = data.toString("utf8");
            var npos = dataStr.indexOf('\n');
            var rpos = dataStr.indexOf('\r');
            if (npos === rpos + 1)
                ending = '\r\n';
            else if (rpos > -1)
                ending = '\r';
            var lines = dataStr.split(ending);
            for (var i = 0; i < lines.length; i++) {
                var line = lines[i];
                if (line === '\r') {
                    // Windows-style line endings...
                    continue;
                }
                if (line.length === 0) {
                    // EOF!
                    if (currentAttribute !== null) {
                        // Set the previously-parsed value.
                        _this._setAttribute(currentAttribute, currentValue);
                        currentAttribute = null;
                    }
                }
                else if (line.charAt(0) === " ") {
                    // Continuation of previous value. Shave off the initial space.
                    currentValue += line.substr(1);
                }
                else {
                    // New value.
                    if (currentAttribute !== null) {
                        // Set the previously-parsed value.
                        _this._setAttribute(currentAttribute, currentValue);
                    }
                    // Split on :.
                    var split = line.split(':');
                    // TODO: Print a warning if parsing fails.
                    currentAttribute = split[0];
                    // Shave off the space separator.
                    currentValue = split[1].substr(1);
                }
            }
            // Completed without error.
            doneCb();
        });
    }
    /**
     * Sets the given attribute from parsed information. This function decides if
     * we digest the information further, or toss it.
     */
    JAR.prototype._setAttribute = function (attribute, value) {
        // Attribute names are case insensitive.
        attribute = attribute.toLowerCase();
        this.attributes[attribute] = value;
        if (attribute === 'class-path') {
            // Parse this! All items are JAR files and are separated by spaces, and
            // may contain \ as separator.
            // We'll just process all words that end in "JAR".
            var words = value.split(' ');
            for (var i = 0; i < words.length; i++) {
                var word = words[i];
                if (word.substr(-4) === '.jar') {
                    this.classpath.push(word);
                }
            }
        }
    };
    /**
     * Return the items in this JAR file's classpath.
     */
    JAR.prototype.getClassPath = function () {
        return this.classpath.slice(0);
    };
    /**
     * Returns NULL if the attribute does not exist.
     * Attribute names are case insensitive
     */
    JAR.prototype.getAttribute = function (attribute) {
        return this.attributes[attribute.toLowerCase()];
    };
    /**
     * Primarily used for debugging.
     */
    JAR.prototype.toString = function () {
        return JSON.stringify(this.attributes);
    };
    return JAR;
}());
exports.__esModule = true;
exports["default"] = JAR;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiamFyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2phci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsSUFBWSxJQUFJLFdBQU0sTUFBTSxDQUFDLENBQUE7QUFDN0IsSUFBWSxFQUFFLFdBQU0sSUFBSSxDQUFDLENBQUE7QUFFekI7Ozs7Ozs7R0FPRztBQUNIO0lBR0U7O09BRUc7SUFDSCxhQUFZLEdBQVcsRUFBRSxNQUE2QjtRQU54RCxpQkF1R0M7UUF0R1MsZUFBVSxHQUE2QixFQUFFLENBQUM7UUFDMUMsY0FBUyxHQUFhLEVBQUUsQ0FBQztRQUsvQixpQ0FBaUM7UUFDakMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxzQkFBc0IsQ0FBQyxFQUFFLFVBQUMsR0FBRyxFQUFFLElBQUk7WUFDL0QsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDO2dCQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDNUIsa0JBQWtCO1lBQ2xCLG1FQUFtRTtZQUNuRSxrQkFBa0I7WUFDbEIsSUFBSSxnQkFBZ0IsR0FBVyxJQUFJLENBQUM7WUFDcEMsSUFBSSxZQUFZLEdBQVcsRUFBRSxDQUFDO1lBQzlCLGtEQUFrRDtZQUNsRCxJQUFJLE1BQU0sR0FBRyxJQUFJLENBQUM7WUFDbEIsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNwQyxJQUFJLElBQUksR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ2pDLElBQUksSUFBSSxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDakMsRUFBRSxDQUFDLENBQUMsSUFBSSxLQUFLLElBQUksR0FBRyxDQUFDLENBQUM7Z0JBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztZQUN2QyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDO2dCQUFDLE1BQU0sR0FBRyxJQUFJLENBQUM7WUFFbEMsSUFBSSxLQUFLLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNsQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztnQkFDdEMsSUFBSSxJQUFJLEdBQUcsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNwQixFQUFFLENBQUMsQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQztvQkFDbEIsZ0NBQWdDO29CQUNoQyxRQUFRLENBQUM7Z0JBQ1gsQ0FBQztnQkFDRCxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQ3RCLE9BQU87b0JBQ1AsRUFBRSxDQUFDLENBQUMsZ0JBQWdCLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQzt3QkFDOUIsbUNBQW1DO3dCQUNuQyxLQUFJLENBQUMsYUFBYSxDQUFDLGdCQUFnQixFQUFFLFlBQVksQ0FBQyxDQUFDO3dCQUNuRCxnQkFBZ0IsR0FBRyxJQUFJLENBQUM7b0JBQzFCLENBQUM7Z0JBQ0gsQ0FBQztnQkFDRCxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsS0FBSyxHQUFHLENBQUMsQ0FBQyxDQUFDO29CQUNoQywrREFBK0Q7b0JBQy9ELFlBQVksSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNqQyxDQUFDO2dCQUFDLElBQUksQ0FBQyxDQUFDO29CQUNOLGFBQWE7b0JBQ2IsRUFBRSxDQUFDLENBQUMsZ0JBQWdCLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQzt3QkFDOUIsbUNBQW1DO3dCQUNuQyxLQUFJLENBQUMsYUFBYSxDQUFDLGdCQUFnQixFQUFFLFlBQVksQ0FBQyxDQUFDO29CQUNyRCxDQUFDO29CQUNELGNBQWM7b0JBQ2QsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztvQkFDNUIsMENBQTBDO29CQUMxQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7b0JBQzVCLGlDQUFpQztvQkFDakMsWUFBWSxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3BDLENBQUM7WUFDSCxDQUFDO1lBQ0QsMkJBQTJCO1lBQzNCLE1BQU0sRUFBRSxDQUFDO1FBQ1gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssMkJBQWEsR0FBckIsVUFBc0IsU0FBaUIsRUFBRSxLQUFhO1FBQ3BELHdDQUF3QztRQUN4QyxTQUFTLEdBQUcsU0FBUyxDQUFDLFdBQVcsRUFBRSxDQUFDO1FBQ3BDLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLEdBQUcsS0FBSyxDQUFDO1FBQ25DLEVBQUUsQ0FBQyxDQUFDLFNBQVMsS0FBSyxZQUFZLENBQUMsQ0FBQyxDQUFDO1lBQy9CLHVFQUF1RTtZQUN2RSw4QkFBOEI7WUFDOUIsa0RBQWtEO1lBQ2xELElBQUksS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDN0IsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7Z0JBQ3RDLElBQUksSUFBSSxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDcEIsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLE1BQU0sQ0FBQyxDQUFDLENBQUM7b0JBQy9CLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUM1QixDQUFDO1lBQ0gsQ0FBQztRQUNILENBQUM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSSwwQkFBWSxHQUFuQjtRQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksMEJBQVksR0FBbkIsVUFBb0IsU0FBaUI7UUFDbkMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksc0JBQVEsR0FBZjtRQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBQ0gsVUFBQztBQUFELENBQUMsQUF2R0QsSUF1R0M7QUF2R0Q7d0JBdUdDLENBQUEiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuXG4vKipcbiAqIEdpdmVuIGEgZm9sZGVyIHRoYXQgY29udGFpbnMgdGhlIGNvbnRlbnRzIG9mIGEgSkFSIGZpbGUsIHBhcnNlcyB0aGUgTUFOSUZFU1RcbiAqIGZpbGUgYW5kIG90aGVyIG1ldGFkYXRhIGFuZCBtYWtlcyBpdCBhdmFpbGFibGUgdG8gRG9wcGlvIGluIGEgY29udmVuaWVudFxuICogZm9ybWF0LlxuICpcbiAqIEBzZWUgaHR0cDovL2RvY3Mub3JhY2xlLmNvbS9qYXZhc2UvNy9kb2NzL3RlY2hub3Rlcy9ndWlkZXMvamFyL2phci5odG1sI05vdGVzX29uX01hbmlmZXN0X2FuZF9TaWduYXR1cmVfRmlsZXNcbiAqIEB0b2RvIEFkZCBhIGZ1bmN0aW9uIGZvciBwcm9kdWNpbmcgYSBKYXZhIG9iamVjdCBmcm9tIHRoaXMuXG4gKi9cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIEpBUiB7XG4gIHByaXZhdGUgYXR0cmlidXRlczoge1tuYW1lOiBzdHJpbmddOiBzdHJpbmd9ID0ge307XG4gIHByaXZhdGUgY2xhc3NwYXRoOiBzdHJpbmdbXSA9IFtdO1xuICAvKipcbiAgICogQ2FsbHMgZG9uZUNiIHdoZW4gaXQgaXMgZmluaXNoZWQgb3BlbmluZyBhbmQgcGFyc2luZyB0aGUgTUFOSUZFU1QgZmlsZS5cbiAgICovXG4gIGNvbnN0cnVjdG9yKGRpcjogc3RyaW5nLCBkb25lQ2I6IChlcnI/OiBFcnJvcikgPT4gdm9pZCkge1xuICAgIC8vIFRyeSB0byByZWFkIHRoZSBNQU5JRkVTVCBmaWxlLlxuICAgIGZzLnJlYWRGaWxlKHBhdGgucmVzb2x2ZShkaXIsICdNRVRBLUlORi9NQU5JRkVTVC5NRicpLCAoZXJyLCBkYXRhKSA9PiB7XG4gICAgICBpZiAoZXJyKSByZXR1cm4gZG9uZUNiKGVycik7XG4gICAgICAvLyBQYXJzZSB0aGUgZmlsZSFcbiAgICAgIC8vIElmIGEgbGluZSAqYmVnaW5zKiB3aXRoIGEgc3BhY2UsIHRoZW4gaXQncyBhIGNvbnRpbnVhdGlvbiBvZiB0aGVcbiAgICAgIC8vIHByZXZpb3VzIHZhbHVlLlxuICAgICAgdmFyIGN1cnJlbnRBdHRyaWJ1dGU6IHN0cmluZyA9IG51bGw7XG4gICAgICB2YXIgY3VycmVudFZhbHVlOiBzdHJpbmcgPSBcIlwiO1xuICAgICAgLy8gRmlndXJlIG91dCB3aGljaCBsaW5lIGVuZGluZyB0aGUgTUFOSUZFU1QgdXNlcy5cbiAgICAgIHZhciBlbmRpbmcgPSAnXFxuJztcbiAgICAgIHZhciBkYXRhU3RyID0gZGF0YS50b1N0cmluZyhcInV0ZjhcIik7XG4gICAgICB2YXIgbnBvcyA9IGRhdGFTdHIuaW5kZXhPZignXFxuJyk7XG4gICAgICB2YXIgcnBvcyA9IGRhdGFTdHIuaW5kZXhPZignXFxyJyk7XG4gICAgICBpZiAobnBvcyA9PT0gcnBvcyArIDEpIGVuZGluZyA9ICdcXHJcXG4nO1xuICAgICAgZWxzZSBpZiAocnBvcyA+IC0xKSBlbmRpbmcgPSAnXFxyJztcblxuICAgICAgdmFyIGxpbmVzID0gZGF0YVN0ci5zcGxpdChlbmRpbmcpO1xuICAgICAgZm9yICh2YXIgaSA9IDA7IGkgPCBsaW5lcy5sZW5ndGg7IGkrKykge1xuICAgICAgICB2YXIgbGluZSA9IGxpbmVzW2ldO1xuICAgICAgICBpZiAobGluZSA9PT0gJ1xccicpIHtcbiAgICAgICAgICAvLyBXaW5kb3dzLXN0eWxlIGxpbmUgZW5kaW5ncy4uLlxuICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICB9XG4gICAgICAgIGlmIChsaW5lLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgIC8vIEVPRiFcbiAgICAgICAgICBpZiAoY3VycmVudEF0dHJpYnV0ZSAhPT0gbnVsbCkge1xuICAgICAgICAgICAgLy8gU2V0IHRoZSBwcmV2aW91c2x5LXBhcnNlZCB2YWx1ZS5cbiAgICAgICAgICAgIHRoaXMuX3NldEF0dHJpYnV0ZShjdXJyZW50QXR0cmlidXRlLCBjdXJyZW50VmFsdWUpO1xuICAgICAgICAgICAgY3VycmVudEF0dHJpYnV0ZSA9IG51bGw7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKGxpbmUuY2hhckF0KDApID09PSBcIiBcIikge1xuICAgICAgICAgIC8vIENvbnRpbnVhdGlvbiBvZiBwcmV2aW91cyB2YWx1ZS4gU2hhdmUgb2ZmIHRoZSBpbml0aWFsIHNwYWNlLlxuICAgICAgICAgIGN1cnJlbnRWYWx1ZSArPSBsaW5lLnN1YnN0cigxKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAvLyBOZXcgdmFsdWUuXG4gICAgICAgICAgaWYgKGN1cnJlbnRBdHRyaWJ1dGUgIT09IG51bGwpIHtcbiAgICAgICAgICAgIC8vIFNldCB0aGUgcHJldmlvdXNseS1wYXJzZWQgdmFsdWUuXG4gICAgICAgICAgICB0aGlzLl9zZXRBdHRyaWJ1dGUoY3VycmVudEF0dHJpYnV0ZSwgY3VycmVudFZhbHVlKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgLy8gU3BsaXQgb24gOi5cbiAgICAgICAgICB2YXIgc3BsaXQgPSBsaW5lLnNwbGl0KCc6Jyk7XG4gICAgICAgICAgLy8gVE9ETzogUHJpbnQgYSB3YXJuaW5nIGlmIHBhcnNpbmcgZmFpbHMuXG4gICAgICAgICAgY3VycmVudEF0dHJpYnV0ZSA9IHNwbGl0WzBdO1xuICAgICAgICAgIC8vIFNoYXZlIG9mZiB0aGUgc3BhY2Ugc2VwYXJhdG9yLlxuICAgICAgICAgIGN1cnJlbnRWYWx1ZSA9IHNwbGl0WzFdLnN1YnN0cigxKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgLy8gQ29tcGxldGVkIHdpdGhvdXQgZXJyb3IuXG4gICAgICBkb25lQ2IoKTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXRzIHRoZSBnaXZlbiBhdHRyaWJ1dGUgZnJvbSBwYXJzZWQgaW5mb3JtYXRpb24uIFRoaXMgZnVuY3Rpb24gZGVjaWRlcyBpZlxuICAgKiB3ZSBkaWdlc3QgdGhlIGluZm9ybWF0aW9uIGZ1cnRoZXIsIG9yIHRvc3MgaXQuXG4gICAqL1xuICBwcml2YXRlIF9zZXRBdHRyaWJ1dGUoYXR0cmlidXRlOiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcpIHtcbiAgICAvLyBBdHRyaWJ1dGUgbmFtZXMgYXJlIGNhc2UgaW5zZW5zaXRpdmUuXG4gICAgYXR0cmlidXRlID0gYXR0cmlidXRlLnRvTG93ZXJDYXNlKCk7XG4gICAgdGhpcy5hdHRyaWJ1dGVzW2F0dHJpYnV0ZV0gPSB2YWx1ZTtcbiAgICBpZiAoYXR0cmlidXRlID09PSAnY2xhc3MtcGF0aCcpIHtcbiAgICAgIC8vIFBhcnNlIHRoaXMhIEFsbCBpdGVtcyBhcmUgSkFSIGZpbGVzIGFuZCBhcmUgc2VwYXJhdGVkIGJ5IHNwYWNlcywgYW5kXG4gICAgICAvLyBtYXkgY29udGFpbiBcXCBhcyBzZXBhcmF0b3IuXG4gICAgICAvLyBXZSdsbCBqdXN0IHByb2Nlc3MgYWxsIHdvcmRzIHRoYXQgZW5kIGluIFwiSkFSXCIuXG4gICAgICB2YXIgd29yZHMgPSB2YWx1ZS5zcGxpdCgnICcpO1xuICAgICAgZm9yICh2YXIgaSA9IDA7IGkgPCB3b3Jkcy5sZW5ndGg7IGkrKykge1xuICAgICAgICB2YXIgd29yZCA9IHdvcmRzW2ldO1xuICAgICAgICBpZiAod29yZC5zdWJzdHIoLTQpID09PSAnLmphcicpIHtcbiAgICAgICAgICB0aGlzLmNsYXNzcGF0aC5wdXNoKHdvcmQpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgaXRlbXMgaW4gdGhpcyBKQVIgZmlsZSdzIGNsYXNzcGF0aC5cbiAgICovXG4gIHB1YmxpYyBnZXRDbGFzc1BhdGgoKTogc3RyaW5nW10ge1xuICAgIHJldHVybiB0aGlzLmNsYXNzcGF0aC5zbGljZSgwKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIE5VTEwgaWYgdGhlIGF0dHJpYnV0ZSBkb2VzIG5vdCBleGlzdC5cbiAgICogQXR0cmlidXRlIG5hbWVzIGFyZSBjYXNlIGluc2Vuc2l0aXZlXG4gICAqL1xuICBwdWJsaWMgZ2V0QXR0cmlidXRlKGF0dHJpYnV0ZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5hdHRyaWJ1dGVzW2F0dHJpYnV0ZS50b0xvd2VyQ2FzZSgpXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBQcmltYXJpbHkgdXNlZCBmb3IgZGVidWdnaW5nLlxuICAgKi9cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBKU09OLnN0cmluZ2lmeSh0aGlzLmF0dHJpYnV0ZXMpO1xuICB9XG59XG4iXX0=