"use strict";
var option_parser_1 = require('./option_parser');
var jvm_1 = require('./jvm');
var util_1 = require('./util');
var logging_1 = require('./logging');
var parser = new option_parser_1.OptionParser({
    default: {
        classpath: {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            alias: 'cp',
            optDesc: ' <class search path of directories and zip/jar files>',
            desc: 'A : separated list of directories, JAR archives, and ZIP archives to search for class files.'
        },
        D: {
            type: 4 /* MAP_SYNTAX */,
            optDesc: '<name>=<value>',
            desc: 'set a system property'
        },
        jar: {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            stopParsing: true
        },
        help: { alias: '?', desc: 'print this help message' },
        X: { desc: 'print help on non-standard options' },
        enableassertions: {
            type: 2 /* COLON_VALUE_OR_FLAG_SYNTAX */,
            optDesc: '[:<packagename>...|:<classname>]',
            alias: 'ea',
            desc: 'enable assertions with specified granularity'
        },
        disableassertions: {
            type: 2 /* COLON_VALUE_OR_FLAG_SYNTAX */,
            optDesc: '[:<packagename>...|:<classname>]',
            alias: 'da',
            desc: 'disable assertions with specified granularity'
        },
        enablesystemassertions: { alias: 'esa', desc: 'enable system assertions' },
        disablesystemassertions: { alias: 'dsa', desc: 'disable system assertions ' }
    },
    X: {
        'int': {
            desc: 'interpreted mode execution only'
        },
        'dump-JIT-stats': {
            desc: 'dump JIT statistics',
            enabled: !RELEASE
        },
        log: {
            desc: 'log level, [0-10]|vtrace|trace|debug|error',
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            enabled: !RELEASE
        },
        'vtrace-methods': {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            optDesc: ' <java/lang/Object/getHashCode()I:...>',
            desc: 'specify particular methods to vtrace separated by colons',
            enabled: !RELEASE
        },
        'list-class-cache': {
            desc: 'list all of the bootstrap loaded classes after execution'
        },
        'dump-compiled-code': {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            optDesc: ' <directory>',
            desc: 'location to dump compiled object definitions',
            enabled: !RELEASE
        },
        // TODO: Use -Djava.library.path
        'native-classpath': {
            type: 3 /* NORMAL_VALUE_SYNTAX */,
            optDesc: ' <class search path of directories>',
            desc: 'A : separated list of directories to search for native mathods in JS files.'
        },
        'bootclasspath/a': {
            type: 1 /* COLON_VALUE_SYNTAX */,
            optDesc: ':<directories and zip/jar files separated by :>',
            desc: 'append to end of bootstrap class path'
        },
        'bootclasspath/p': {
            type: 1 /* COLON_VALUE_SYNTAX */,
            optDesc: ':<directories and zip/jar files separated by :>',
            desc: 'prepend in front of bootstrap class path'
        },
        'bootclasspath': {
            type: 1 /* COLON_VALUE_SYNTAX */,
            optDesc: ':<directories and zip/jar files separated by :>',
            desc: 'set search path for bootstrap classes and resources'
        },
        'X:+PrintCompilation': {
            desc: 'Print JIT compilation details',
            enabled: !RELEASE
        }
    }
});
/**
 * Consumes a `java` command line string. Constructs a JVM, launches the command, and
 * returns the JVM object. Throws an exception if parsing fails.
 *
 * Returns `null` if no JVM needed to be constructed (e.g. -h flag).
 *
 * @param args Arguments to the 'java' command.
 * @param opts Default options.
 * @param doneCb Called when JVM execution finishes. Passes a
 *   number to the callback indicating the exit value.
 * @param [jvmStarted] Called with the JVM object once we have invoked it.
 */
function java(args, opts, doneCb, jvmStarted) {
    if (jvmStarted === void 0) { jvmStarted = function (jvm) { }; }
    var parsedArgs = parser.parse(args), standard = parsedArgs['default'], nonStandard = parsedArgs['X'], jvmState;
    // System properties.
    opts.properties = standard.mapOption('D');
    if (standard.flag('help', false)) {
        return printHelp(opts.launcherName, parser.help('default'), doneCb, 0);
    }
    else if (standard.flag('X', false)) {
        return printNonStandardHelp(opts.launcherName, parser.help('X'), doneCb, 0);
    }
    // GLOBAL CONFIGURATION
    var logOption = nonStandard.stringOption('log', 'ERROR');
    opts.intMode = nonStandard.flag('int', false);
    opts.dumpJITStats = nonStandard.flag('dump-JIT-stats', false);
    if (/^[0-9]+$/.test(logOption)) {
        logging_1.setLogLevel(parseInt(logOption, 10));
    }
    else {
        var level = logging_1.LogLevel[logOption.toUpperCase()];
        if (level == null) {
            process.stderr.write("Unrecognized log level: " + logOption + ".");
            return printHelp(opts.launcherName, parser.help('default'), doneCb, 1);
        }
        logging_1.setLogLevel(level);
    }
    if (nonStandard.flag('list-class-cache', false)) {
        // Redefine done_cb so we print the loaded class files on JVM exit.
        doneCb = (function (old_done_cb) {
            return function (result) {
                var fpaths = jvmState.getBootstrapClassLoader().getLoadedClassFiles();
                process.stdout.write(fpaths.join('\n') + '\n');
                old_done_cb(result);
            };
        })(doneCb);
    }
    if (standard.flag('enablesystemassertions', false)) {
        opts.enableSystemAssertions = true;
    }
    if (standard.flag('disablesystemassertions', false)) {
        opts.enableSystemAssertions = false;
    }
    if (standard.flag('enableassertions', false)) {
        opts.enableAssertions = true;
    }
    else if (standard.stringOption('enableassertions', null)) {
        opts.enableAssertions = standard.stringOption('enableassertions', null).split(':');
    }
    if (standard.stringOption('disableassertions', null)) {
        opts.disableAssertions = standard.stringOption('disableassertions', null).split(':');
    }
    // NOTE: Boolean form of -disableassertions is a NOP.
    // Bootstrap classpath items.
    var bscl = nonStandard.stringOption('bootclasspath', null);
    if (bscl !== null) {
        opts.bootstrapClasspath = bscl.split(':');
    }
    var bsClAppend = nonStandard.stringOption('bootclasspath/a', null);
    if (bsClAppend) {
        opts.bootstrapClasspath = opts.bootstrapClasspath.concat(bsClAppend.split(':'));
    }
    var bsClPrepend = nonStandard.stringOption('bootclasspath/p', null);
    if (bsClPrepend) {
        opts.bootstrapClasspath = bsClPrepend.split(':').concat(opts.bootstrapClasspath);
    }
    // User-supplied classpath items.
    if (!opts.classpath) {
        opts.classpath = [];
    }
    if (standard.stringOption('jar', null)) {
        opts.classpath.push(standard.stringOption('jar', null));
    }
    else if (standard.stringOption('classpath', null)) {
        opts.classpath = opts.classpath.concat(standard.stringOption('classpath', null).split(':'));
    }
    else {
        // DEFAULT: If no user-supplied classpath, add the current directory to
        // the class path.
        opts.classpath.push(process.cwd());
    }
    // User-supplied native classpath.
    var nativeClasspath = standard.stringOption('native-classpath', null);
    if (nativeClasspath) {
        opts.nativeClasspath = opts.nativeClasspath.concat(nativeClasspath.split(':'));
    }
    // Construct the JVM.
    jvmState = new jvm_1["default"](opts, function (err) {
        if (err) {
            process.stderr.write("Error constructing JVM:\n");
            process.stderr.write(err.toString() + "\n");
            doneCb(1);
        }
        else {
            launchJvm(standard, opts, jvmState, doneCb, jvmStarted);
        }
    });
    jvmState.setPrintJITCompilation(nonStandard.flag('X:+PrintCompilation', false));
    var vtraceMethods = nonStandard.stringOption('vtrace-methods', null);
    if (vtraceMethods) {
        vtraceMethods.split(':').forEach(function (m) { return jvmState.vtraceMethod(m); });
    }
    var dumpCompiledCode = nonStandard.stringOption('dump-compiled-code', null);
    if (dumpCompiledCode) {
        jvmState.dumpCompiledCode(dumpCompiledCode);
    }
}
exports.__esModule = true;
exports["default"] = java;
/**
 * Consumes a fully-configured JVM, parsed arguments, and a callback.
 * Figures out from this how to launch the JVM (e.g. using a JAR file or a
 * particular class).
 */
function launchJvm(standardOptions, opts, jvmState, doneCb, jvmStarted) {
    var mainArgs = standardOptions.unparsedArgs();
    if (standardOptions.stringOption('jar', null)) {
        jvmState.runJar(mainArgs, doneCb);
        jvmStarted(jvmState);
    }
    else if (mainArgs.length > 0) {
        var cname = mainArgs[0];
        if (cname.slice(-6) === '.class') {
            cname = cname.slice(0, -6);
        }
        if (cname.indexOf('.') !== -1) {
            // hack: convert java.foo.Bar to java/foo/Bar
            cname = util_1.descriptor2typestr(util_1.int_classname(cname));
        }
        jvmState.runClass(cname, mainArgs.slice(1), doneCb);
        jvmStarted(jvmState);
    }
    else {
        // No class specified, no jar specified!
        printHelp(opts.launcherName, parser.help('default'), doneCb, 0);
    }
}
function printHelp(launcherName, str, doneCb, rv) {
    process.stdout.write("Usage: " + launcherName + " [-options] class [args...]\n        (to execute a class)\nor  " + launcherName + " [-options] -jar jarfile [args...]\n        (to execute a jar file)\nwhere options include:\n" + str);
    doneCb(rv);
}
function printNonStandardHelp(launcherName, str, doneCb, rv) {
    process.stdout.write(str + "\n\nThe -X options are non-standard and subject to change without notice.\n");
    doneCb(rv);
}
//# sourceMappingURL=data:application/json;base64,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