"use strict";
var util_1 = require('./util');
var SafeMap_1 = require('./SafeMap');
var methods_1 = require('./methods');
var ClassLoader_1 = require('./ClassLoader');
var fs = require('fs');
var path = require('path');
var buffer = require('buffer');
var threading_1 = require('./threading');
var enums_1 = require('./enums');
var heap_1 = require('./heap');
var assert_1 = require('./assert');
var parker_1 = require('./parker');
var threadpool_1 = require('./threadpool');
var JDKInfo = require('../vendor/java_home/jdk.json');
var global_1 = require('./global');
var global_require_1 = require('./global_require');
var BrowserFS = require('browserfs');
var DoppioJVM = require('./doppiojvm');
if (typeof RELEASE === 'undefined')
    global_1["default"].RELEASE = false;
// For version information.
var pkg;
if (util_1.are_in_browser()) {
    pkg = require('../package.json');
}
else {
    pkg = require('../../../package.json');
}
// XXX: We currently initialize these classes at JVM bootup. This is expensive.
// We should attempt to prune this list as much as possible.
var coreClasses = [
    'Ljava/lang/String;',
    'Ljava/lang/Class;', 'Ljava/lang/ClassLoader;',
    'Ljava/lang/reflect/Constructor;', 'Ljava/lang/reflect/Field;',
    'Ljava/lang/reflect/Method;',
    'Ljava/lang/Error;', 'Ljava/lang/StackTraceElement;',
    'Ljava/lang/System;',
    'Ljava/lang/Thread;',
    'Ljava/lang/ThreadGroup;',
    'Ljava/lang/Throwable;',
    'Ljava/nio/ByteOrder;',
    'Lsun/misc/VM;', 'Lsun/reflect/ConstantPool;', 'Ljava/lang/Byte;',
    'Ljava/lang/Character;', 'Ljava/lang/Double;', 'Ljava/lang/Float;',
    'Ljava/lang/Integer;', 'Ljava/lang/Long;', 'Ljava/lang/Short;',
    'Ljava/lang/Void;', 'Ljava/io/FileDescriptor;',
    'Ljava/lang/Boolean;', '[Lsun/management/MemoryManagerImpl;',
    '[Lsun/management/MemoryPoolImpl;',
    // Contains important FS constants used by natives. These constants are
    // inlined into JCL class files, so it typically never gets initialized
    // implicitly by the JVM.
    'Lsun/nio/fs/UnixConstants;'
];
/**
 * Encapsulates a single JVM instance.
 */
var JVM = (function () {
    /**
     * (Async) Construct a new instance of the Java Virtual Machine.
     */
    function JVM(opts, cb) {
        var _this = this;
        this.systemProperties = null;
        this.internedStrings = new SafeMap_1["default"]();
        this.bsCl = null;
        this.threadPool = null;
        this.natives = {};
        // 20MB heap
        // @todo Make heap resizeable.
        this.heap = new heap_1["default"](20 * 1024 * 1024);
        this.nativeClasspath = null;
        this.startupTime = new Date();
        this.terminationCb = null;
        // The initial JVM thread used to kick off execution.
        this.firstThread = null;
        this.responsiveness = null;
        this.enableSystemAssertions = false;
        this.enabledAssertions = false;
        this.disabledAssertions = [];
        this.printJITCompilation = false;
        this.systemClassLoader = null;
        this.nextRef = 0;
        // Set of all of the methods we want vtrace to be enabled on.
        // DEBUG builds only.
        this.vtraceMethods = {};
        // [DEBUG] directory to dump compiled code to.
        this.dumpCompiledCodeDir = null;
        // Handles parking/unparking threads.
        this.parker = new parker_1["default"]();
        // The current status of the JVM.
        this.status = enums_1.JVMStatus.BOOTING;
        // The JVM's planned exit code.
        this.exitCode = 0;
        // is JIT disabled?
        this.jitDisabled = false;
        this.dumpJITStats = false;
        // Get the environment's require variable, indirectly.
        // Hidden from webpack and other builders, as it confuses them.
        this.globalRequire = null;
        if (typeof (opts.doppioHomePath) !== 'string') {
            throw new TypeError("opts.doppioHomePath *must* be specified.");
        }
        opts = util_1.merge(JVM.getDefaultOptions(opts.doppioHomePath), opts);
        this.jitDisabled = opts.intMode;
        this.dumpJITStats = opts.dumpJITStats;
        var bootstrapClasspath = opts.bootstrapClasspath.map(function (p) { return path.resolve(p); }), 
        // JVM bootup tasks, from first to last task.
        bootupTasks = [], firstThread, firstThreadObj;
        // Sanity checks.
        if (!Array.isArray(opts.bootstrapClasspath) || opts.bootstrapClasspath.length === 0) {
            throw new TypeError("opts.bootstrapClasspath must be specified as an array of file paths.");
        }
        if (!Array.isArray(opts.classpath)) {
            throw new TypeError("opts.classpath must be specified as an array of file paths.");
        }
        if (typeof (opts.javaHomePath) !== 'string') {
            throw new TypeError("opts.javaHomePath must be specified.");
        }
        if (!opts.nativeClasspath) {
            opts.nativeClasspath = [];
        }
        if (!Array.isArray(opts.nativeClasspath)) {
            throw new TypeError("opts.nativeClasspath must be specified as an array of file paths.");
        }
        this.nativeClasspath = opts.nativeClasspath;
        if (opts.enableSystemAssertions) {
            this.enableSystemAssertions = opts.enableSystemAssertions;
        }
        if (opts.enableAssertions) {
            this.enabledAssertions = opts.enableAssertions;
        }
        if (opts.disableAssertions) {
            this.disabledAssertions = opts.disableAssertions;
        }
        this.responsiveness = opts.responsiveness;
        this._initSystemProperties(bootstrapClasspath, opts.classpath.map(function (p) { return path.resolve(p); }), path.resolve(opts.javaHomePath), path.resolve(opts.tmpDir), opts.properties);
        /**
         * Task #1: Initialize native methods.
         */
        bootupTasks.push(function (next) {
            _this.initializeNatives(next);
        });
        /**
         * Task #2: Construct the bootstrap class loader.
         */
        bootupTasks.push(function (next) {
            _this.bsCl =
                new ClassLoader_1.BootstrapClassLoader(_this.systemProperties['java.home'], bootstrapClasspath, next);
        });
        /**
         * Task #3: Construct the thread pool, resolve thread class, and construct
         * the first thread.
         */
        bootupTasks.push(function (next) {
            _this.threadPool = new threadpool_1["default"](function () { return _this.threadPoolIsEmpty(); });
            // Resolve Ljava/lang/Thread so we can fake a thread.
            // NOTE: This should never actually use the Thread object unless
            // there's an error loading java/lang/Thread and associated classes.
            _this.bsCl.resolveClass(null, 'Ljava/lang/Thread;', function (threadCdata) {
                if (threadCdata == null) {
                    // Failed.
                    next("Failed to resolve java/lang/Thread.");
                }
                else {
                    // Construct a thread.
                    firstThreadObj = new (threadCdata.getConstructor(null))(null);
                    firstThreadObj.$thread = firstThread = _this.firstThread = new threading_1.JVMThread(_this, _this.threadPool, firstThreadObj);
                    firstThreadObj.ref = 1;
                    firstThreadObj['java/lang/Thread/priority'] = 5;
                    firstThreadObj['java/lang/Thread/name'] = util_1.initCarr(_this.bsCl, 'main');
                    firstThreadObj['java/lang/Thread/blockerLock'] = new (_this.bsCl.getResolvedClass('Ljava/lang/Object;').getConstructor(firstThread))(firstThread);
                    next();
                }
            });
        });
        /**
         * Task #4: Preinitialize some essential JVM classes, and initializes the
         * JVM's ThreadGroup once that class is initialized.
         */
        bootupTasks.push(function (next) {
            util_1.asyncForEach(coreClasses, function (coreClass, nextItem) {
                _this.bsCl.initializeClass(firstThread, coreClass, function (cdata) {
                    if (cdata == null) {
                        nextItem("Failed to initialize " + coreClass);
                    }
                    else {
                        // One of the later preinitialized classes references Thread.group.
                        // Initialize the system's ThreadGroup now.
                        if (coreClass === 'Ljava/lang/ThreadGroup;') {
                            // Construct a ThreadGroup object for the first thread.
                            var threadGroupCons = cdata.getConstructor(firstThread), groupObj = new threadGroupCons(firstThread);
                            groupObj['<init>()V'](firstThread, null, function (e) {
                                // Tell the initial thread to use this group.
                                firstThreadObj['java/lang/Thread/group'] = groupObj;
                                nextItem(e);
                            });
                        }
                        else {
                            nextItem();
                        }
                    }
                });
            }, next);
        });
        /**
         * Task #5: Initialize the system class.
         */
        bootupTasks.push(function (next) {
            // Initialize the system class (initializes things like println/etc).
            var sysInit = _this.bsCl.getInitializedClass(firstThread, 'Ljava/lang/System;').getConstructor(firstThread);
            sysInit['java/lang/System/initializeSystemClass()V'](firstThread, null, next);
            ;
        });
        /**
         * Task #6: Initialize the application's
         */
        bootupTasks.push(function (next) {
            var clCons = _this.bsCl.getInitializedClass(firstThread, 'Ljava/lang/ClassLoader;').getConstructor(firstThread);
            clCons['java/lang/ClassLoader/getSystemClassLoader()Ljava/lang/ClassLoader;'](firstThread, null, function (e, rv) {
                if (e) {
                    next(e);
                }
                else {
                    _this.systemClassLoader = rv.$loader;
                    firstThreadObj['java/lang/Thread/contextClassLoader'] = rv;
                    // Initialize assertion data.
                    // TODO: Is there a better way to force this? :|
                    var defaultAssertionStatus = _this.enabledAssertions === true ? 1 : 0;
                    rv['java/lang/ClassLoader/setDefaultAssertionStatus(Z)V'](firstThread, [defaultAssertionStatus], next);
                }
            });
        });
        /**
         * Task #7: Initialize DoppioJVM's security provider for things like cryptographically strong RNG.
         */
        bootupTasks.push(function (next) {
            _this.bsCl.initializeClass(firstThread, 'Ldoppio/security/DoppioProvider;', function (cdata) {
                next(cdata ? null : new Error("Failed to initialize DoppioProvider."));
            });
        });
        // Perform bootup tasks, and then trigger the callback function.
        util_1.asyncSeries(bootupTasks, function (err) {
            // XXX: Without setImmediate, the firstThread won't clear out the stack
            // frame that triggered us, and the firstThread won't transition to a
            // 'terminated' status.
            setImmediate(function () {
                if (err) {
                    _this.status = enums_1.JVMStatus.TERMINATED;
                    cb(err);
                }
                else {
                    _this.status = enums_1.JVMStatus.BOOTED;
                    cb(null, _this);
                }
            });
        });
    }
    JVM.isReleaseBuild = function () {
        return typeof (RELEASE) !== 'undefined' && RELEASE;
    };
    JVM.getNativeMethodModules = function () {
        if (!this._haveAddedBuiltinNativeModules) {
            // NOTE: Replace with an ES6 import when we switch to a supporting bundler like Rollup.
            // Currently cannot import these above to avoid circular imports, which Webpack does not
            // support.
            JVM.registerNativeModule(require('./natives/doppio').default);
            JVM.registerNativeModule(require('./natives/java_io').default);
            JVM.registerNativeModule(require('./natives/java_lang').default);
            JVM.registerNativeModule(require('./natives/java_net').default);
            JVM.registerNativeModule(require('./natives/java_nio').default);
            JVM.registerNativeModule(require('./natives/java_security').default);
            JVM.registerNativeModule(require('./natives/java_util').default);
            JVM.registerNativeModule(require('./natives/sun_font').default);
            JVM.registerNativeModule(require('./natives/sun_management').default);
            JVM.registerNativeModule(require('./natives/sun_misc').default);
            JVM.registerNativeModule(require('./natives/sun_net').default);
            JVM.registerNativeModule(require('./natives/sun_nio').default);
            JVM.registerNativeModule(require('./natives/sun_reflect').default);
            this._haveAddedBuiltinNativeModules = true;
        }
        return this._nativeMethodModules;
    };
    /**
     * Registers a JavaScript module that provides particular native methods with Doppio.
     * All new JVMs constructed will auto-run this module to add its natives.
     */
    JVM.registerNativeModule = function (mod) {
        this._nativeMethodModules.push(mod);
    };
    JVM.prototype.getResponsiveness = function () {
        var resp = this.responsiveness;
        if (typeof resp === 'number') {
            return resp;
        }
        else if (typeof resp === 'function') {
            return resp();
        }
    };
    JVM.getDefaultOptions = function (doppioHome) {
        var javaHome = path.join(doppioHome, 'vendor', 'java_home');
        return {
            doppioHomePath: doppioHome,
            classpath: ['.'],
            bootstrapClasspath: JDKInfo.classpath.map(function (item) { return path.join(javaHome, item); }),
            javaHomePath: javaHome,
            nativeClasspath: [],
            enableSystemAssertions: false,
            enableAssertions: false,
            disableAssertions: null,
            properties: {},
            tmpDir: '/tmp',
            responsiveness: 1000,
            intMode: false,
            dumpJITStats: false
        };
    };
    /**
     * Get the URL to the version of the JDK that DoppioJVM was compiled with.
     */
    JVM.getCompiledJDKURL = function () {
        return JDKInfo.url;
    };
    /**
     * Get the JDK information that DoppioJVM was compiled against.
     */
    JVM.getJDKInfo = function () {
        return JDKInfo;
    };
    JVM.prototype.getSystemClassLoader = function () {
        return this.systemClassLoader;
    };
    /**
     * Get the next "ref" number for JVM objects.
     */
    JVM.prototype.getNextRef = function () {
        return this.nextRef++;
    };
    /**
     * Retrieve the JVM's parker. Handles parking/unparking threads.
     */
    JVM.prototype.getParker = function () {
        return this.parker;
    };
    /**
     * Run the specified class on this JVM instance.
     * @param className The name of the class to run. Can be specified in either
     *   foo.bar.Baz or foo/bar/Baz format.
     * @param args Command line arguments passed to the class.
     * @param cb Called when the JVM finishes executing. Called with 'true' if
     *   the JVM exited normally, 'false' if there was an error.
     */
    JVM.prototype.runClass = function (className, args, cb) {
        var _this = this;
        if (this.status !== enums_1.JVMStatus.BOOTED) {
            switch (this.status) {
                case enums_1.JVMStatus.BOOTING:
                    throw new Error("JVM is currently booting up. Please wait for it to call the bootup callback, which you passed to the constructor.");
                case enums_1.JVMStatus.RUNNING:
                    throw new Error("JVM is already running.");
                case enums_1.JVMStatus.TERMINATED:
                    throw new Error("This JVM has already terminated. Please create a new JVM.");
                case enums_1.JVMStatus.TERMINATING:
                    throw new Error("This JVM is currently terminating. You should create a new JVM for each class you wish to run.");
            }
        }
        this.terminationCb = cb;
        var thread = this.firstThread;
        assert_1["default"](thread != null, "Thread isn't created yet?");
        // Convert foo.bar.Baz => Lfoo/bar/Baz;
        className = util_1.int_classname(className);
        // Initialize the class.
        this.systemClassLoader.initializeClass(thread, className, function (cdata) {
            // If cdata is null, there was an error that ended execution.
            if (cdata != null) {
                // Convert the arguments.
                var strArrCons = _this.bsCl.getInitializedClass(thread, '[Ljava/lang/String;').getConstructor(thread), jvmifiedArgs = new strArrCons(thread, args.length), i;
                for (i = 0; i < args.length; i++) {
                    jvmifiedArgs.array[i] = util_1.initString(_this.bsCl, args[i]);
                }
                // Find the main method, and run it.
                _this.status = enums_1.JVMStatus.RUNNING;
                var cdataStatics = cdata.getConstructor(thread);
                if (cdataStatics['main([Ljava/lang/String;)V']) {
                    cdataStatics['main([Ljava/lang/String;)V'](thread, [jvmifiedArgs]);
                }
                else {
                    thread.throwNewException("Ljava/lang/NoSuchMethodError;", "Could not find main method in class " + cdata.getExternalName() + ".");
                }
            }
            else {
                process.stdout.write("Error: Could not find or load main class " + util_1.ext_classname(className) + "\n");
                // Erroneous exit.
                _this.terminationCb(1);
            }
        });
    };
    /**
     * Returns 'true' if confined to interpreter mode
     */
    JVM.prototype.isJITDisabled = function () {
        return this.jitDisabled;
    };
    /**
     * [DEBUG] Returns 'true' if the specified method should be vtraced.
     */
    JVM.prototype.shouldVtrace = function (sig) {
        return this.vtraceMethods[sig] === true;
    };
    /**
     * [DEBUG] Specify a method to vtrace.
     */
    JVM.prototype.vtraceMethod = function (sig) {
        this.vtraceMethods[sig] = true;
    };
    /**
     * Run the specified JAR file on this JVM instance.
     * @param args Command line arguments passed to the class.
     * @param cb Called when the JVM finishes executing. Called with 'true' if
     *   the JVM exited normally, 'false' if there was an error.
     */
    JVM.prototype.runJar = function (args, cb) {
        this.runClass('doppio.JarLauncher', args, cb);
    };
    /**
     * Called when the ThreadPool is empty.
     */
    JVM.prototype.threadPoolIsEmpty = function () {
        var systemClass, systemCons;
        switch (this.status) {
            case enums_1.JVMStatus.BOOTING:
                // Ignore empty thread pools during boot process.
                return false;
            case enums_1.JVMStatus.BOOTED:
                assert_1["default"](false, "Thread pool should not become empty after JVM is booted, but before it begins to run.");
                return false;
            case enums_1.JVMStatus.RUNNING:
                this.status = enums_1.JVMStatus.TERMINATING;
                systemClass = this.bsCl.getInitializedClass(this.firstThread, 'Ljava/lang/System;');
                assert_1["default"](systemClass !== null, "Invariant failure: System class must be initialized when JVM is in RUNNING state.");
                systemCons = systemClass.getConstructor(this.firstThread);
                // This is a normal, non-erroneous exit. When this function completes, threadPoolIsEmpty() will be invoked again.
                systemCons['java/lang/System/exit(I)V'](this.firstThread, [0]);
                return false;
            case enums_1.JVMStatus.TERMINATED:
                assert_1["default"](false, "Invariant failure: Thread pool cannot be emptied post-JVM termination.");
                return false;
            case enums_1.JVMStatus.TERMINATING:
                if (!RELEASE && this.dumpJITStats) {
                    methods_1.dumpStats();
                }
                this.status = enums_1.JVMStatus.TERMINATED;
                if (this.terminationCb) {
                    this.terminationCb(this.exitCode);
                }
                this.firstThread.close();
                return true;
        }
    };
    /**
     * Check if the JVM has started running the main class.
     */
    JVM.prototype.hasVMBooted = function () {
        return !(this.status === enums_1.JVMStatus.BOOTING || this.status === enums_1.JVMStatus.BOOTED);
    };
    /**
     * Completely halt the JVM.
     */
    JVM.prototype.halt = function (status) {
        this.exitCode = status;
        this.status = enums_1.JVMStatus.TERMINATING;
        this.threadPool.getThreads().forEach(function (t) {
            t.setStatus(enums_1.ThreadStatus.TERMINATED);
        });
    };
    /**
     * Retrieve the given system property.
     */
    JVM.prototype.getSystemProperty = function (prop) {
        return this.systemProperties[prop];
    };
    /**
     * Retrieve an array of all of the system property names.
     */
    JVM.prototype.getSystemPropertyNames = function () {
        return Object.keys(this.systemProperties);
    };
    /**
     * Retrieve the unmanaged heap.
     */
    JVM.prototype.getHeap = function () {
        return this.heap;
    };
    /**
     * Interns the given JavaScript string. Returns the interned string.
     */
    JVM.prototype.internString = function (str, javaObj) {
        if (this.internedStrings.has(str)) {
            return this.internedStrings.get(str);
        }
        else {
            if (!javaObj) {
                javaObj = util_1.initString(this.bsCl, str);
            }
            this.internedStrings.set(str, javaObj);
            return javaObj;
        }
    };
    /**
     * Evaluate native modules. Emulates CommonJS functionality.
     */
    JVM.prototype.evalNativeModule = function (mod) {
        if (!this.globalRequire) {
            this.globalRequire = global_require_1["default"]();
        }
        var rv;
        /**
         * Called by the native method file. Registers the package's native
         * methods with the JVM.
         */
        function registerNatives(defs) {
            rv = defs;
        }
        // Provide the natives with the Doppio API, if needed.
        var globalRequire = this.globalRequire;
        /**
         * An emulation of CommonJS require() for the modules.
         */
        function moduleRequire(name) {
            switch (name) {
                case 'doppiojvm':
                case '../doppiojvm':
                    return DoppioJVM;
                case 'fs':
                    return fs;
                case 'path':
                    return path;
                case 'buffer':
                    return buffer;
                case 'browserfs':
                    return BrowserFS;
                default:
                    return globalRequire(name);
            }
        }
        /**
         * Emulate AMD module 'define' function for natives compiled as AMD modules.
         */
        function moduleDefine(resources, module) {
            var args = [];
            resources.forEach(function (resource) {
                switch (resource) {
                    case 'require':
                        args.push(moduleRequire);
                        break;
                    case 'exports':
                        args.push({});
                        break;
                    default:
                        args.push(moduleRequire(resource));
                        break;
                }
            });
            module.apply(null, args);
        }
        var modFcn = new Function("require", "define", "registerNatives", "process", "DoppioJVM", "Buffer", mod);
        modFcn(moduleRequire, moduleDefine, registerNatives, process, DoppioJVM, Buffer);
        return rv;
    };
    /**
     * Register native methods with the virtual machine.
     */
    JVM.prototype.registerNatives = function (newNatives) {
        var clsName, methSig;
        for (clsName in newNatives) {
            if (newNatives.hasOwnProperty(clsName)) {
                if (!this.natives.hasOwnProperty(clsName)) {
                    this.natives[clsName] = {};
                }
                var clsMethods = newNatives[clsName];
                for (methSig in clsMethods) {
                    if (clsMethods.hasOwnProperty(methSig)) {
                        // Don't check if it exists already. This allows us to overwrite
                        // native methods dynamically at runtime.
                        this.natives[clsName][methSig] = clsMethods[methSig];
                    }
                }
            }
        }
    };
    /**
     * Convenience function. Register a single native method with the virtual
     * machine. Can be used to update existing native methods based on runtime
     * information.
     */
    JVM.prototype.registerNative = function (clsName, methSig, native) {
        this.registerNatives({ clsName: { methSig: native } });
    };
    /**
     * Retrieve the native method for the given method of the given class.
     * Returns null if none found.
     */
    JVM.prototype.getNative = function (clsName, methSig) {
        clsName = util_1.descriptor2typestr(clsName);
        if (this.natives.hasOwnProperty(clsName)) {
            var clsMethods = this.natives[clsName];
            if (clsMethods.hasOwnProperty(methSig)) {
                return clsMethods[methSig];
            }
        }
        return null;
    };
    /**
     * !!DO NOT MUTATE THE RETURNED VALUE!!
     * Used by the find_invalid_natives tool.
     */
    JVM.prototype.getNatives = function () {
        return this.natives;
    };
    /**
     * Loads in all of the native method modules prior to execution.
     * Currently a hack around our
     * @todo Make neater with async stuff.
     */
    JVM.prototype.initializeNatives = function (doneCb) {
        var _this = this;
        var registeredModules = JVM.getNativeMethodModules();
        for (var i_1 = 0; i_1 < registeredModules.length; i_1++) {
            this.registerNatives(registeredModules[i_1]());
        }
        var nextDir = function () {
            if (i === _this.nativeClasspath.length) {
                // Next phase: Load up the files.
                var count_1 = processFiles.length;
                if (count_1 === 0) {
                    return doneCb();
                }
                processFiles.forEach(function (file) {
                    fs.readFile(file, function (err, data) {
                        if (!err) {
                            _this.registerNatives(_this.evalNativeModule(data.toString()));
                        }
                        if (--count_1 === 0) {
                            doneCb();
                        }
                    });
                });
            }
            else {
                var dir = _this.nativeClasspath[i++];
                fs.readdir(dir, function (err, files) {
                    if (err) {
                        return doneCb();
                    }
                    var j, file;
                    for (j = 0; j < files.length; j++) {
                        file = files[j];
                        if (file.substring(file.length - 3, file.length) === '.js') {
                            processFiles.push(path.join(dir, file));
                        }
                    }
                    nextDir();
                });
            }
        }, i = 0, processFiles = [];
        nextDir();
    };
    /**
     * [Private] Same as reset_system_properties, but called by the constructor.
     */
    JVM.prototype._initSystemProperties = function (bootstrapClasspath, javaClassPath, javaHomePath, tmpDir, opts) {
        this.systemProperties = util_1.merge({
            'java.class.path': javaClassPath.join(':'),
            'java.home': javaHomePath,
            'java.ext.dirs': path.join(javaHomePath, 'lib', 'ext'),
            'java.io.tmpdir': tmpDir,
            'sun.boot.class.path': bootstrapClasspath.join(':'),
            'file.encoding': 'UTF-8',
            'java.vendor': 'Doppio',
            'java.version': '1.8',
            'java.vendor.url': 'https://github.com/plasma-umass/doppio',
            'java.class.version': '52.0',
            'java.specification.version': '1.8',
            'line.separator': '\n',
            'file.separator': path.sep,
            'path.separator': ':',
            'user.dir': path.resolve('.'),
            'user.home': '.',
            'user.name': 'DoppioUser',
            'os.name': 'doppio',
            'os.arch': 'js',
            'os.version': '0',
            'java.vm.name': 'DoppioJVM 32-bit VM',
            'java.vm.version': pkg.version,
            'java.vm.vendor': 'PLASMA@UMass',
            'java.awt.headless': (util_1.are_in_browser()).toString(),
            'java.awt.graphicsenv': 'classes.awt.CanvasGraphicsEnvironment',
            'jline.terminal': 'jline.UnsupportedTerminal',
            'sun.arch.data.model': '32',
            'sun.jnu.encoding': "UTF-8" // Determines how Java parses command line options.
        }, opts);
    };
    /**
     * Retrieves the bootstrap class loader.
     */
    JVM.prototype.getBootstrapClassLoader = function () {
        return this.bsCl;
    };
    JVM.prototype.getStartupTime = function () {
        return this.startupTime;
    };
    /**
     * Returns `true` if system assertions are enabled, false otherwise.
     */
    JVM.prototype.areSystemAssertionsEnabled = function () {
        return this.enableSystemAssertions;
    };
    /**
     * Get a listing of classes with assertions enabled. Can also return 'true' or 'false.
     */
    JVM.prototype.getEnabledAssertions = function () {
        return this.enabledAssertions;
    };
    /**
     * Get a listing of classes with assertions disabled.
     */
    JVM.prototype.getDisabledAssertions = function () {
        return this.disabledAssertions;
    };
    JVM.prototype.setPrintJITCompilation = function (enabledOrNot) {
        this.printJITCompilation = enabledOrNot;
    };
    JVM.prototype.shouldPrintJITCompilation = function () {
        return this.printJITCompilation;
    };
    /**
     * Specifies a directory to dump compiled code to.
     */
    JVM.prototype.dumpCompiledCode = function (dir) {
        this.dumpCompiledCodeDir = dir;
    };
    JVM.prototype.shouldDumpCompiledCode = function () {
        return this.dumpCompiledCodeDir !== null;
    };
    JVM.prototype.dumpObjectDefinition = function (cls, evalText) {
        if (this.shouldDumpCompiledCode()) {
            fs.writeFile(path.resolve(this.dumpCompiledCodeDir, cls.getExternalName() + ".js"), evalText, function () { });
        }
    };
    JVM.prototype.dumpBridgeMethod = function (methodSig, evalText) {
        if (this.shouldDumpCompiledCode()) {
            fs.appendFile(path.resolve(this.dumpCompiledCodeDir, "vmtarget_bridge_methods.dump"), methodSig + ":\n" + evalText + "\n\n", function () { });
        }
    };
    JVM.prototype.dumpCompiledMethod = function (methodSig, pc, code) {
        if (this.shouldDumpCompiledCode()) {
            fs.appendFile(path.resolve(this.dumpCompiledCodeDir, 'JIT_compiled_methods.dump'), methodSig + ":" + pc + ":\n" + code + "\n\n", function () { });
        }
    };
    /**
     * Asynchronously dumps JVM state to a file. Currently limited to thread
     * state.
     */
    JVM.prototype.dumpState = function (filename, cb) {
        fs.appendFile(filename, this.threadPool.getThreads().map(function (t) { return ("Thread " + t.getRef() + ":\n") + t.getPrintableStackTrace(); }).join("\n\n"), cb);
    };
    JVM._nativeMethodModules = [];
    JVM._haveAddedBuiltinNativeModules = false;
    return JVM;
}());
exports.__esModule = true;
exports["default"] = JVM;
//# sourceMappingURL=data:application/json;base64,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