"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var util_1 = require('./util');
var util = require('./util');
var attributes_1 = require('./attributes');
var threading_1 = require('./threading');
var assert_1 = require('./assert');
var enums_1 = require('./enums');
var StringOutputStream_1 = require('./StringOutputStream');
var global_1 = require('./global');
var jit_1 = require('./jit');
if (typeof RELEASE === 'undefined')
    global_1["default"].RELEASE = false;
var trapped_methods = {
    'java/lang/ref/Reference': {
        // NOP, because we don't do our own GC and also this starts a thread?!?!?!
        '<clinit>()V': function (thread) { }
    },
    'java/lang/System': {
        'loadLibrary(Ljava/lang/String;)V': function (thread, libName) {
            // Some libraries test if native libraries are available,
            // and expect an exception if they are not.
            // List all of the native libraries we support.
            var lib = libName.toString();
            switch (lib) {
                case 'zip':
                case 'net':
                case 'nio':
                case 'awt':
                case 'fontmanager':
                case 'management':
                    return;
                default:
                    thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', "no " + lib + " in java.library.path");
                    break;
            }
        }
    },
    'java/lang/Terminator': {
        'setup()V': function (thread) {
            // XXX: We should probably fix this; we support threads now.
            // Historically: NOP'd because we didn't support threads.
        }
    },
    'java/nio/charset/Charset$3': {
        // this is trapped and NOP'ed for speed
        'run()Ljava/lang/Object;': function (thread, javaThis) {
            return null;
        }
    },
    'sun/nio/fs/DefaultFileSystemProvider': {
        // OpenJDK doesn't know what the "Doppio" platform is. Tell it to use the Linux file system.
        'create()Ljava/nio/file/spi/FileSystemProvider;': function (thread) {
            thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
            var dfsp = thread.getBsCl().getInitializedClass(thread, 'Lsun/nio/fs/DefaultFileSystemProvider;'), dfspCls = dfsp.getConstructor(thread);
            dfspCls['createProvider(Ljava/lang/String;)Ljava/nio/file/spi/FileSystemProvider;'](thread, [thread.getJVM().internString('sun.nio.fs.LinuxFileSystemProvider')], util_1.forwardResult(thread));
        }
    }
};
function getTrappedMethod(clsName, methSig) {
    clsName = util_1.descriptor2typestr(clsName);
    if (trapped_methods.hasOwnProperty(clsName) && trapped_methods[clsName].hasOwnProperty(methSig)) {
        return trapped_methods[clsName][methSig];
    }
    return null;
}
/**
 * Shared functionality between Method and Field objects, as they are
 * represented similarly in class files.
 */
var AbstractMethodField = (function () {
    /**
     * Constructs a field or method object from raw class data.
     */
    function AbstractMethodField(cls, constantPool, slot, byteStream) {
        this.cls = cls;
        this.slot = slot;
        this.accessFlags = new util_1.Flags(byteStream.getUint16());
        this.name = constantPool.get(byteStream.getUint16()).value;
        this.rawDescriptor = constantPool.get(byteStream.getUint16()).value;
        this.attrs = attributes_1.makeAttributes(byteStream, constantPool);
    }
    AbstractMethodField.prototype.getAttribute = function (name) {
        for (var i = 0; i < this.attrs.length; i++) {
            var attr = this.attrs[i];
            if (attr.getName() === name) {
                return attr;
            }
        }
        return null;
    };
    AbstractMethodField.prototype.getAttributes = function (name) {
        return this.attrs.filter(function (attr) { return attr.getName() === name; });
    };
    /**
     * Get the particular type of annotation as a JVM byte array. Returns null
     * if the annotation does not exist.
     */
    AbstractMethodField.prototype.getAnnotationType = function (thread, name) {
        var annotation = this.getAttribute(name);
        if (annotation === null) {
            return null;
        }
        var byteArrCons = thread.getBsCl().getInitializedClass(thread, '[B').getConstructor(thread), rv = new byteArrCons(thread, 0);
        // TODO: Convert to typed array.
        var i, len = annotation.rawBytes.length, arr = new Array(len);
        for (i = 0; i < len; i++) {
            arr[i] = annotation.rawBytes.readInt8(i);
        }
        rv.array = arr;
        return rv;
    };
    // To satiate TypeScript. Consider it an 'abstract' method.
    AbstractMethodField.prototype.parseDescriptor = function (raw_descriptor) {
        throw new Error("Unimplemented error.");
    };
    return AbstractMethodField;
}());
exports.AbstractMethodField = AbstractMethodField;
var Field = (function (_super) {
    __extends(Field, _super);
    function Field(cls, constantPool, slot, byteStream) {
        _super.call(this, cls, constantPool, slot, byteStream);
        this.fullName = util_1.descriptor2typestr(cls.getInternalName()) + "/" + this.name;
    }
    /**
     * Calls cb with the reflectedField if it succeeds. Calls cb with null if it
     * fails.
     */
    Field.prototype.reflector = function (thread, cb) {
        var _this = this;
        var signatureAttr = this.getAttribute("Signature"), jvm = thread.getJVM(), bsCl = thread.getBsCl();
        var createObj = function (typeObj) {
            var fieldCls = bsCl.getInitializedClass(thread, 'Ljava/lang/reflect/Field;'), fieldObj = new (fieldCls.getConstructor(thread))(thread);
            fieldObj['java/lang/reflect/Field/clazz'] = _this.cls.getClassObject(thread);
            fieldObj['java/lang/reflect/Field/name'] = jvm.internString(_this.name);
            fieldObj['java/lang/reflect/Field/type'] = typeObj;
            fieldObj['java/lang/reflect/Field/modifiers'] = _this.accessFlags.getRawByte();
            fieldObj['java/lang/reflect/Field/slot'] = _this.slot;
            fieldObj['java/lang/reflect/Field/signature'] = signatureAttr !== null ? util_1.initString(bsCl, signatureAttr.sig) : null;
            fieldObj['java/lang/reflect/Field/annotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleAnnotations');
            return fieldObj;
        };
        // Our field's type may not be loaded, so we asynchronously load it here.
        // In the future, we can speed up reflection by having a synchronous_reflector
        // method that we can try first, and which may fail.
        this.cls.getLoader().resolveClass(thread, this.rawDescriptor, function (cdata) {
            if (cdata != null) {
                cb(createObj(cdata.getClassObject(thread)));
            }
            else {
                cb(null);
            }
        });
    };
    Field.prototype.getDefaultFieldValue = function () {
        var desc = this.rawDescriptor;
        if (desc === 'J')
            return 'gLongZero';
        var c = desc[0];
        if (c === '[' || c === 'L')
            return 'null';
        return '0';
    };
    /**
     * Outputs a JavaScript field assignment for this field.
     */
    Field.prototype.outputJavaScriptField = function (jsConsName, outputStream) {
        if (this.accessFlags.isStatic()) {
            outputStream.write(jsConsName + "[\"" + util_1.reescapeJVMName(this.fullName) + "\"] = cls._getInitialStaticFieldValue(thread, \"" + util_1.reescapeJVMName(this.name) + "\");\n");
        }
        else {
            outputStream.write("this[\"" + util_1.reescapeJVMName(this.fullName) + "\"] = " + this.getDefaultFieldValue() + ";\n");
        }
    };
    return Field;
}(AbstractMethodField));
exports.Field = Field;
var opcodeSize = function () {
    var table = [];
    table[enums_1.OpcodeLayoutType.OPCODE_ONLY] = 1;
    table[enums_1.OpcodeLayoutType.CONSTANT_POOL_UINT8] = 2;
    table[enums_1.OpcodeLayoutType.CONSTANT_POOL] = 3;
    table[enums_1.OpcodeLayoutType.CONSTANT_POOL_AND_UINT8_VALUE] = 4;
    table[enums_1.OpcodeLayoutType.UINT8_VALUE] = 2;
    table[enums_1.OpcodeLayoutType.UINT8_AND_INT8_VALUE] = 3;
    table[enums_1.OpcodeLayoutType.INT8_VALUE] = 2;
    table[enums_1.OpcodeLayoutType.INT16_VALUE] = 3;
    table[enums_1.OpcodeLayoutType.INT32_VALUE] = 5;
    table[enums_1.OpcodeLayoutType.ARRAY_TYPE] = 2;
    table[enums_1.OpcodeLayoutType.WIDE] = 1;
    return table;
}();
var TraceInfo = (function () {
    function TraceInfo(pc, jitInfo) {
        this.pc = pc;
        this.jitInfo = jitInfo;
        this.pops = [];
        this.pushes = [];
        this.prefixEmit = "";
    }
    return TraceInfo;
}());
var Trace = (function () {
    function Trace(startPC, code, method) {
        this.startPC = startPC;
        this.code = code;
        this.method = method;
        this.infos = [];
        this.endPc = -1;
    }
    /**
     * Emits a PC update statement at the end of the trace.
     */
    Trace.prototype.emitEndPC = function (pc) {
        this.endPc = pc;
    };
    Trace.prototype.addOp = function (pc, jitInfo) {
        this.infos.push(new TraceInfo(pc, jitInfo));
    };
    Trace.prototype.close = function (thread) {
        if (this.infos.length > 1) {
            var symbolicStack = [];
            var symbolCount = 0;
            // Ensure that the last statement sets the PC if the
            // last opcode doesn't.
            var emitted = this.endPc > -1 ? "f.pc=" + this.endPc + ";" : "";
            for (var i = 0; i < this.infos.length; i++) {
                var info = this.infos[i];
                var jitInfo = info.jitInfo;
                var pops = info.pops;
                var normalizedPops = jitInfo.pops < 0 ? Math.min(-jitInfo.pops, symbolicStack.length) : jitInfo.pops;
                for (var j = 0; j < normalizedPops; j++) {
                    if (symbolicStack.length > 0) {
                        pops.push(symbolicStack.pop());
                    }
                    else {
                        var symbol = "s" + symbolCount++;
                        info.prefixEmit += "var " + symbol + " = f.opStack.pop();";
                        pops.push(symbol);
                    }
                }
                info.onErrorPushes = symbolicStack.slice();
                var pushes = info.pushes;
                for (var j = 0; j < jitInfo.pushes; j++) {
                    var symbol = "s" + symbolCount++;
                    symbolicStack.push(symbol);
                    pushes.push(symbol);
                }
            }
            if (symbolicStack.length === 1) {
                emitted += "f.opStack.push(" + symbolicStack[0] + ");";
            }
            else if (symbolicStack.length > 1) {
                emitted += "f.opStack.pushAll(" + symbolicStack.join(',') + ");";
            }
            for (var i = this.infos.length - 1; i >= 0; i--) {
                var info = this.infos[i];
                var jitInfo = info.jitInfo;
                emitted = info.prefixEmit + jitInfo.emit(info.pops, info.pushes, "" + i, emitted, this.code, info.pc, info.onErrorPushes, this.method);
            }
            if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
                console.log(("Emitted trace of " + this.infos.length + " ops: ") + emitted);
            }
            // f = frame, t = thread, u = util
            return new Function("f", "t", "u", emitted);
        }
        else {
            if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
                console.log("Trace was cancelled");
            }
            return null;
        }
    };
    return Trace;
}());
var Method = (function (_super) {
    __extends(Method, _super);
    function Method(cls, constantPool, slot, byteStream) {
        _super.call(this, cls, constantPool, slot, byteStream);
        /**
         * number of basic block entries
         */
        this.numBBEntries = 0;
        this.compiledFunctions = [];
        this.failedCompile = [];
        var parsedDescriptor = util_1.getTypes(this.rawDescriptor), i, p;
        this.signature = this.name + this.rawDescriptor;
        this.fullSignature = util_1.descriptor2typestr(this.cls.getInternalName()) + "/" + this.signature;
        this.returnType = parsedDescriptor.pop();
        this.parameterTypes = parsedDescriptor;
        this.parameterWords = parsedDescriptor.length;
        // Double count doubles / longs.
        for (i = 0; i < this.parameterTypes.length; i++) {
            p = this.parameterTypes[i];
            if (p === 'D' || p === 'J') {
                this.parameterWords++;
            }
        }
        // Initialize 'code' property.
        var clsName = this.cls.getInternalName();
        if (getTrappedMethod(clsName, this.signature) !== null) {
            this.code = getTrappedMethod(clsName, this.signature);
            this.accessFlags.setNative(true);
        }
        else if (this.accessFlags.isNative()) {
            if (this.signature.indexOf('registerNatives()V', 0) < 0 && this.signature.indexOf('initIDs()V', 0) < 0) {
                // The first version of the native method attempts to fetch itself and
                // rewrite itself.
                var self = this;
                this.code = function (thread) {
                    // Try to fetch the native method.
                    var jvm = thread.getJVM(), c = jvm.getNative(clsName, self.signature);
                    if (c == null) {
                        thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', "Native method '" + self.getFullSignature() + "' not implemented.\nPlease fix or file a bug at https://github.com/plasma-umass/doppio/issues");
                    }
                    else {
                        self.code = c;
                        return c.apply(self, arguments);
                    }
                };
            }
            else {
                // Stub out initIDs and registerNatives.
                this.code = function () { };
            }
        }
        else if (!this.accessFlags.isAbstract()) {
            this.code = this.getAttribute('Code');
            var codeLength = this.code.code.length;
            // jit threshold. we countdown to zero from here.
            this.numBBEntries = codeLength > 3 ? 200 : 1000 * codeLength;
        }
    }
    Method.prototype.incrBBEntries = function () {
        // Optimisiation: we countdown to zero, instead of storing a positive limit in a separate variable
        this.numBBEntries--;
    };
    /**
     * Checks if the method is a default method.
     * A default method is a public non-abstract instance method, that
     * is, a non-static method with a body, declared in an interface
     * type.
     */
    Method.prototype.isDefault = function () {
        return (this.accessFlags.isPublic() && !this.accessFlags.isAbstract() && !this.accessFlags.isStatic() && this.cls.accessFlags.isInterface());
    };
    Method.prototype.getFullSignature = function () {
        return this.cls.getExternalName() + "." + this.name + this.rawDescriptor;
    };
    /**
     * Checks if this particular method should be hidden in stack frames.
     * Used by OpenJDK's lambda implementation to hide lambda boilerplate.
     */
    Method.prototype.isHidden = function () {
        var rva = this.getAttribute('RuntimeVisibleAnnotations');
        return rva !== null && rva.isHidden;
    };
    /**
     * Checks if this particular method has the CallerSensitive annotation.
     */
    Method.prototype.isCallerSensitive = function () {
        var rva = this.getAttribute('RuntimeVisibleAnnotations');
        return rva !== null && rva.isCallerSensitive;
    };
    /**
     * Get the number of machine words (32-bit words) required to store the
     * parameters to this function. Includes adding in a machine word for 'this'
     * for non-static functions.
     */
    Method.prototype.getParamWordSize = function () {
        return this.parameterWords;
    };
    Method.prototype.getCodeAttribute = function () {
        assert_1["default"](!this.accessFlags.isNative() && !this.accessFlags.isAbstract());
        return this.code;
    };
    Method.prototype.getOp = function (pc, codeBuffer, thread) {
        if (this.numBBEntries <= 0) {
            if (!this.failedCompile[pc]) {
                var cachedCompiledFunction = this.compiledFunctions[pc];
                if (!cachedCompiledFunction) {
                    var compiledFunction = this.jitCompileFrom(pc, thread);
                    if (compiledFunction) {
                        return compiledFunction;
                    }
                    else {
                        this.failedCompile[pc] = true;
                    }
                }
                else {
                    return cachedCompiledFunction;
                }
            }
        }
        return codeBuffer[pc];
    };
    Method.prototype.makeInvokeStaticJitInfo = function (code, pc) {
        var index = code.readUInt16BE(pc + 1);
        var methodReference = this.cls.constantPool.get(index);
        var paramSize = methodReference.paramWordSize;
        var method = methodReference.jsConstructor[methodReference.fullSignature];
        return { hasBranch: true, pops: -paramSize, pushes: 0, emit: function (pops, pushes, suffix, onSuccess) {
                var argInitialiser = paramSize > pops.length ? "f.opStack.sliceAndDropFromTop(" + (paramSize - pops.length) + ");" : "[" + pops.reduce(function (a, b) { return b + ',' + a; }, '') + "];";
                var argMaker = ("var args" + suffix + "=") + argInitialiser;
                if ((paramSize > pops.length) && (pops.length > 0)) {
                    argMaker += "args" + suffix + ".push(" + pops.slice().reverse().join(',') + ");";
                }
                return argMaker + ("\nvar methodReference" + suffix + "=f.method.cls.constantPool.get(" + index + ");\nf.pc=" + pc + ";\nmethodReference" + suffix + ".jsConstructor[methodReference" + suffix + ".fullSignature](t,args" + suffix + ");\nf.returnToThreadLoop=true;\n" + onSuccess);
            } };
    };
    Method.prototype.makeInvokeVirtualJitInfo = function (code, pc) {
        var index = code.readUInt16BE(pc + 1);
        var methodReference = this.cls.constantPool.get(index);
        var paramSize = methodReference.paramWordSize;
        return { hasBranch: true, pops: -(paramSize + 1), pushes: 0, emit: function (pops, pushes, suffix, onSuccess, code, pc, onErrorPushes) {
                var onError = makeOnError(onErrorPushes);
                var argInitialiser = paramSize > pops.length ? "f.opStack.sliceAndDropFromTop(" + (paramSize - pops.length) + ");" : "[" + pops.slice(0, paramSize).reduce(function (a, b) { return b + ',' + a; }, '') + "];";
                var argMaker = ("var args" + suffix + "=") + argInitialiser;
                if ((paramSize > pops.length) && (pops.length > 0)) {
                    argMaker += "args" + suffix + ".push(" + pops.slice().reverse().join(',') + ");";
                }
                return argMaker + ("var obj" + suffix + "=" + ((paramSize + 1) === pops.length ? pops[paramSize] : "f.opStack.pop()") + ";f.pc=" + pc + ";\nif(!u.isNull(t,f,obj" + suffix + ")){obj" + suffix + "['" + methodReference.signature + "'](t,args" + suffix + ");f.returnToThreadLoop=true;" + onSuccess + "}else{" + onError + "}");
            } };
    };
    Method.prototype.makeInvokeNonVirtualJitInfo = function (code, pc) {
        var index = code.readUInt16BE(pc + 1);
        var methodReference = this.cls.constantPool.get(index);
        var paramSize = methodReference.paramWordSize;
        return { hasBranch: true, pops: -(paramSize + 1), pushes: 0, emit: function (pops, pushes, suffix, onSuccess, code, pc, onErrorPushes) {
                var onError = makeOnError(onErrorPushes);
                var argInitialiser = paramSize > pops.length ? "f.opStack.sliceAndDropFromTop(" + (paramSize - pops.length) + ");" : "[" + pops.slice(0, paramSize).reduce(function (a, b) { return b + ',' + a; }, '') + "];";
                var argMaker = ("var args" + suffix + "=") + argInitialiser;
                if ((paramSize > pops.length) && (pops.length > 0)) {
                    argMaker += "args" + suffix + ".push(" + pops.slice().reverse().join(',') + ");";
                }
                return argMaker + ("var obj" + suffix + "=" + ((paramSize + 1) === pops.length ? pops[paramSize] : "f.opStack.pop()") + ";f.pc=" + pc + ";\nif(!u.isNull(t,f,obj" + suffix + ")){obj" + suffix + "['" + methodReference.fullSignature + "'](t, args" + suffix + ");f.returnToThreadLoop=true;" + onSuccess + "}else{" + onError + "}");
            } };
    };
    Method.prototype.jitCompileFrom = function (startPC, thread) {
        if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
            console.log("Planning to JIT: " + this.fullSignature + " from " + startPC);
        }
        var code = this.getCodeAttribute().getCode();
        var trace = null;
        var _this = this;
        var done = false;
        function closeCurrentTrace() {
            if (trace !== null) {
                // console.log("Tracing method: " + _this.fullSignature);
                var compiledFunction = trace.close(thread);
                if (compiledFunction) {
                    _this.compiledFunctions[trace.startPC] = compiledFunction;
                    if (!RELEASE && thread.getJVM().shouldDumpCompiledCode()) {
                        thread.getJVM().dumpCompiledMethod(_this.fullSignature, trace.startPC, compiledFunction.toString());
                    }
                }
                trace = null;
            }
            done = true;
        }
        for (var i = startPC; i < code.length && !done;) {
            var op = code[i];
            // TODO: handle wide()
            if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
                console.log(i + ": " + threading_1.annotateOpcode(op, this, code, i));
            }
            var jitInfo = jit_1.opJitInfo[op];
            if (jitInfo) {
                if (trace === null) {
                    trace = new Trace(i, code, _this);
                }
                trace.addOp(i, jitInfo);
                if (jitInfo.hasBranch) {
                    this.failedCompile[i] = true;
                    closeCurrentTrace();
                }
            }
            else if (op === enums_1.OpCode.INVOKESTATIC_FAST && trace !== null) {
                var invokeJitInfo = this.makeInvokeStaticJitInfo(code, i);
                trace.addOp(i, invokeJitInfo);
                this.failedCompile[i] = true;
                closeCurrentTrace();
            }
            else if (((op === enums_1.OpCode.INVOKEVIRTUAL_FAST) || (op === enums_1.OpCode.INVOKEINTERFACE_FAST)) && trace !== null) {
                var invokeJitInfo = this.makeInvokeVirtualJitInfo(code, i);
                trace.addOp(i, invokeJitInfo);
                this.failedCompile[i] = true;
                closeCurrentTrace();
            }
            else if ((op === enums_1.OpCode.INVOKENONVIRTUAL_FAST) && trace !== null) {
                var invokeJitInfo = this.makeInvokeNonVirtualJitInfo(code, i);
                trace.addOp(i, invokeJitInfo);
                this.failedCompile[i] = true;
                closeCurrentTrace();
            }
            else {
                if (!RELEASE) {
                    if (trace !== null) {
                        statTraceCloser[op]++;
                    }
                }
                this.failedCompile[i] = true;
                if (trace) {
                    trace.emitEndPC(i);
                }
                closeCurrentTrace();
            }
            i += opcodeSize[enums_1.OpcodeLayouts[op]];
        }
        return _this.compiledFunctions[startPC];
    };
    Method.prototype.getNativeFunction = function () {
        assert_1["default"](this.accessFlags.isNative() && typeof (this.code) === 'function');
        return this.code;
    };
    /**
     * Resolves all of the classes referenced through this method. Required in
     * order to create its reflection object.
     */
    Method.prototype._resolveReferencedClasses = function (thread, cb) {
        // Start with the return type + parameter types + reflection object types.
        var toResolve = this.parameterTypes.concat(this.returnType), code = this.code, exceptionAttribute = this.getAttribute("Exceptions");
        // Exception handler types.
        if (!this.accessFlags.isNative() && !this.accessFlags.isAbstract() && code.exceptionHandlers.length > 0) {
            toResolve.push('Ljava/lang/Throwable;'); // Mimic native Java (in case <any> is the only handler).
            // Filter out the <any> handlers.
            toResolve = toResolve.concat(code.exceptionHandlers.filter(function (handler) { return handler.catchType !== '<any>'; }).map(function (handler) { return handler.catchType; }));
        }
        // Resolve checked exception types.
        if (exceptionAttribute !== null) {
            toResolve = toResolve.concat(exceptionAttribute.exceptions);
        }
        this.cls.getLoader().resolveClasses(thread, toResolve, function (classes) {
            // Use bootstrap classloader for reflection classes.
            thread.getBsCl().resolveClasses(thread, ['Ljava/lang/reflect/Method;', 'Ljava/lang/reflect/Constructor;'], function (classes2) {
                if (classes === null || classes2 === null) {
                    cb(null);
                }
                else {
                    classes['Ljava/lang/reflect/Method;'] = classes2['Ljava/lang/reflect/Method;'];
                    classes['Ljava/lang/reflect/Constructor;'] = classes2['Ljava/lang/reflect/Constructor;'];
                    cb(classes);
                }
            });
        });
    };
    /**
     * Get a reflection object representing this method.
     */
    Method.prototype.reflector = function (thread, cb) {
        var _this = this;
        var bsCl = thread.getBsCl(), 
        // Grab the classes required to construct the needed arrays.
        clazzArray = bsCl.getInitializedClass(thread, '[Ljava/lang/Class;').getConstructor(thread), jvm = thread.getJVM(), 
        // Grab the needed
        signatureAttr = this.getAttribute("Signature"), exceptionAttr = this.getAttribute("Exceptions");
        // Retrieve all of the required class references.
        this._resolveReferencedClasses(thread, function (classes) {
            if (classes === null) {
                return cb(null);
            }
            // Construct the needed objects for the reflection object.
            var clazz = _this.cls.getClassObject(thread), name = jvm.internString(_this.name), parameterTypes = new clazzArray(thread, 0), returnType = classes[_this.returnType].getClassObject(thread), exceptionTypes = new clazzArray(thread, 0), modifiers = _this.accessFlags.getRawByte(), signature = signatureAttr !== null ? jvm.internString(signatureAttr.sig) : null;
            // Prepare the class arrays.
            parameterTypes.array = _this.parameterTypes.map(function (ptype) { return classes[ptype].getClassObject(thread); });
            if (exceptionAttr !== null) {
                exceptionTypes.array = exceptionAttr.exceptions.map(function (eType) { return classes[eType].getClassObject(thread); });
            }
            if (_this.name === '<init>') {
                // Constructor object.
                var consCons = classes['Ljava/lang/reflect/Constructor;'].getConstructor(thread), consObj = new consCons(thread);
                consObj['java/lang/reflect/Constructor/clazz'] = clazz;
                consObj['java/lang/reflect/Constructor/parameterTypes'] = parameterTypes;
                consObj['java/lang/reflect/Constructor/exceptionTypes'] = exceptionTypes;
                consObj['java/lang/reflect/Constructor/modifiers'] = modifiers;
                consObj['java/lang/reflect/Constructor/slot'] = _this.slot;
                consObj['java/lang/reflect/Constructor/signature'] = signature;
                consObj['java/lang/reflect/Constructor/annotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleAnnotations');
                consObj['java/lang/reflect/Constructor/parameterAnnotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleParameterAnnotations');
                cb(consObj);
            }
            else {
                // Method object.
                var methodCons = classes['Ljava/lang/reflect/Method;'].getConstructor(thread), methodObj = new methodCons(thread);
                methodObj['java/lang/reflect/Method/clazz'] = clazz;
                methodObj['java/lang/reflect/Method/name'] = name;
                methodObj['java/lang/reflect/Method/parameterTypes'] = parameterTypes;
                methodObj['java/lang/reflect/Method/returnType'] = returnType;
                methodObj['java/lang/reflect/Method/exceptionTypes'] = exceptionTypes;
                methodObj['java/lang/reflect/Method/modifiers'] = modifiers;
                methodObj['java/lang/reflect/Method/slot'] = _this.slot;
                methodObj['java/lang/reflect/Method/signature'] = signature;
                methodObj['java/lang/reflect/Method/annotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleAnnotations');
                methodObj['java/lang/reflect/Method/annotationDefault'] = _this.getAnnotationType(thread, 'AnnotationDefault');
                methodObj['java/lang/reflect/Method/parameterAnnotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleParameterAnnotations');
                cb(methodObj);
            }
        });
    };
    /**
     * Convert the arguments to this method into a form suitable for a native
     * implementation.
     *
     * The JVM uses two parameter slots for double and long values, since they
     * consist of two JVM machine words (32-bits). Doppio stores the entire value
     * in one slot, and stores a NULL in the second.
     *
     * This function strips out these NULLs so the arguments are in a more
     * consistent form. The return value is the arguments to this function without
     * these NULL values. It also adds the 'thread' object to the start of the
     * arguments array.
     */
    Method.prototype.convertArgs = function (thread, params) {
        if (this.isSignaturePolymorphic()) {
            // These don't need any conversion, and have arbitrary arguments.
            // Just append the thread object.
            params.unshift(thread);
            return params;
        }
        var convertedArgs = [thread], argIdx = 0, i;
        if (!this.accessFlags.isStatic()) {
            convertedArgs.push(params[0]);
            argIdx = 1;
        }
        for (i = 0; i < this.parameterTypes.length; i++) {
            var p = this.parameterTypes[i];
            convertedArgs.push(params[argIdx]);
            argIdx += (p === 'J' || p === 'D') ? 2 : 1;
        }
        return convertedArgs;
    };
    /**
     * Lock this particular method.
     */
    Method.prototype.methodLock = function (thread, frame) {
        if (this.accessFlags.isStatic()) {
            // Static methods lock the class.
            return this.cls.getClassObject(thread).getMonitor();
        }
        else {
            // Non-static methods lock the instance.
            return frame.locals[0].getMonitor();
        }
    };
    /**
     * Check if this is a signature polymorphic method.
     * From S2.9:
     * A method is signature polymorphic if and only if all of the following conditions hold :
     * * It is declared in the java.lang.invoke.MethodHandle class.
     * * It has a single formal parameter of type Object[].
     * * It has a return type of Object.
     * * It has the ACC_VARARGS and ACC_NATIVE flags set.
     */
    Method.prototype.isSignaturePolymorphic = function () {
        return this.cls.getInternalName() === 'Ljava/lang/invoke/MethodHandle;' &&
            this.accessFlags.isNative() && this.accessFlags.isVarArgs() &&
            this.rawDescriptor === '([Ljava/lang/Object;)Ljava/lang/Object;';
    };
    /**
     * Retrieve the MemberName/invokedynamic JavaScript "bridge method" that
     * encapsulates the logic required to call this particular method.
     */
    Method.prototype.getVMTargetBridgeMethod = function (thread, refKind) {
        // TODO: Could cache these in the Method object if desired.
        var outStream = new StringOutputStream_1["default"](), virtualDispatch = !(refKind === enums_1.MethodHandleReferenceKind.INVOKESTATIC || refKind === enums_1.MethodHandleReferenceKind.INVOKESPECIAL);
        // Args: thread, cls, util
        if (this.accessFlags.isStatic()) {
            assert_1["default"](!virtualDispatch, "Can't have static virtual dispatch.");
            outStream.write("var jsCons = cls.getConstructor(thread);\n");
        }
        outStream.write("function bridgeMethod(thread, descriptor, args, cb) {\n");
        if (!this.accessFlags.isStatic()) {
            outStream.write("  var obj = args.shift();\n");
            outStream.write("  if (obj === null) { return thread.throwNewException('Ljava/lang/NullPointerException;', ''); }\n");
            outStream.write("  obj[\"" + util_1.reescapeJVMName(virtualDispatch ? this.signature : this.fullSignature) + "\"](thread, ");
        }
        else {
            outStream.write("  jsCons[\"" + util_1.reescapeJVMName(this.fullSignature) + "\"](thread, ");
        }
        // TODO: Is it ever appropriate to box arguments for varargs functions? It appears not.
        outStream.write("args");
        outStream.write(", cb);\n  }\n  return bridgeMethod;");
        var evalText = outStream.flush();
        if (!RELEASE && thread !== null && thread.getJVM().shouldDumpCompiledCode()) {
            thread.getJVM().dumpBridgeMethod(this.fullSignature, evalText);
        }
        return new Function("thread", "cls", "util", evalText)(thread, this.cls, util);
    };
    /**
     * Generates JavaScript code for this particular method.
     * TODO: Move lock logic and such into this function! And other specialization.
     * TODO: Signature polymorphic functions...?
     */
    Method.prototype.outputJavaScriptFunction = function (jsConsName, outStream, nonVirtualOnly) {
        if (nonVirtualOnly === void 0) { nonVirtualOnly = false; }
        var i;
        if (this.accessFlags.isStatic()) {
            outStream.write(jsConsName + "[\"" + util_1.reescapeJVMName(this.fullSignature) + "\"] = " + jsConsName + "[\"" + util_1.reescapeJVMName(this.signature) + "\"] = ");
        }
        else {
            if (!nonVirtualOnly) {
                outStream.write(jsConsName + ".prototype[\"" + util_1.reescapeJVMName(this.signature) + "\"] = ");
            }
            outStream.write(jsConsName + ".prototype[\"" + util_1.reescapeJVMName(this.fullSignature) + "\"] = ");
        }
        // cb check is boilerplate, required for natives calling into JVM land.
        outStream.write("(function(method) {\n  return function(thread, args, cb) {\n    if (typeof cb === 'function') {\n      thread.stack.push(new InternalStackFrame(cb));\n    }\n    thread.stack.push(new " + (this.accessFlags.isNative() ? "NativeStackFrame" : "BytecodeStackFrame") + "(method, ");
        if (!this.accessFlags.isStatic()) {
            // Non-static functions need to add the implicit 'this' variable to the
            // local variables.
            outStream.write("[this");
            // Give the JS engine hints about the size, type, and contents of the array
            // by making it a literal.
            for (i = 0; i < this.parameterWords; i++) {
                outStream.write(", args[" + i + "]");
            }
            outStream.write("]");
        }
        else {
            // Static function doesn't need to mutate the arguments.
            if (this.parameterWords > 0) {
                outStream.write("args");
            }
            else {
                outStream.write("[]");
            }
        }
        outStream.write("));\n    thread.setStatus(" + enums_1.ThreadStatus.RUNNABLE + ");\n  };\n})(cls.getSpecificMethod(\"" + util_1.reescapeJVMName(this.cls.getInternalName()) + "\", \"" + util_1.reescapeJVMName(this.signature) + "\"));\n");
    };
    return Method;
}(AbstractMethodField));
exports.Method = Method;
function makeOnError(onErrorPushes) {
    return onErrorPushes.length > 0 ? "f.opStack.pushAll(" + onErrorPushes.join(',') + ");" : '';
}
var statTraceCloser = new Array(256);
if (!RELEASE) {
    for (var i = 0; i < 256; i++) {
        statTraceCloser[i] = 0;
    }
}
function dumpStats() {
    var range = new Array(256);
    for (var i = 0; i < 256; i++) {
        range[i] = i;
    }
    range.sort(function (x, y) { return statTraceCloser[y] - statTraceCloser[x]; });
    var top = range.slice(0, 24);
    console.log("Opcodes that closed a trace (number of times encountered):");
    for (var i = 0; i < top.length; i++) {
        var op = top[i];
        if (statTraceCloser[op] > 0) {
            console.log(enums_1.OpCode[op], statTraceCloser[op]);
        }
    }
}
exports.dumpStats = dumpStats;
//# sourceMappingURL=data:application/json;base64,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