"use strict";
var fs = require('fs');
var path = require('path');
var Doppio = require('../doppiojvm');
var util = Doppio.VM.Util;
var ThreadStatus = Doppio.VM.Enums.ThreadStatus;
var Long = Doppio.VM.Long;
var FDState = Doppio.VM.FDState;
function throwNodeError(thread, err) {
    var type = "Ljava/io/IOException;";
    if (err.code === "ENOENT") {
        type = 'Ljava/io/FileNotFoundException;';
    }
    thread.throwNewException(type, err.message);
}
function default_1() {
    /**
     * Provide buffering for the underlying input function, returning at most
     * n_bytes of data.
     */
    function async_input(n_bytes, resume) {
        // Try to read n_bytes from stdin's buffer.
        var read = function (nBytes) {
            // XXX: Returns a Buffer, but DefinitelyTyped says string|Buffer.
            var bytes = process.stdin.read(nBytes);
            if (bytes === null) {
                // We might have asked for too many bytes. Retrieve the entire stream
                // buffer.
                bytes = process.stdin.read();
            }
            // \0 => EOF.
            if (bytes !== null && bytes.length === 1 && bytes[0] === 0) {
                bytes = new Buffer(0);
            }
            return bytes;
        }, bytes = read(n_bytes);
        if (bytes === null) {
            // No input available. Wait for further input.
            process.stdin.once('readable', function (data) {
                var bytes = read(n_bytes);
                if (bytes === null) {
                    bytes = new Buffer(0);
                }
                resume(bytes);
            });
        }
        else {
            // Reset stack depth and resume with the given data.
            setImmediate(function () { resume(bytes); });
        }
    }
    function statFile(fname, cb) {
        fs.stat(fname, function (err, stat) {
            if (err != null) {
                cb(null);
            }
            else {
                cb(stat);
            }
        });
    }
    var java_io_Console = (function () {
        function java_io_Console() {
        }
        java_io_Console['encoding()Ljava/lang/String;'] = function (thread) {
            return null;
        };
        java_io_Console['echo(Z)Z'] = function (thread, echoOn) {
            var echoOff = !echoOn;
            process.stdin.setRawMode(echoOff);
            return echoOff;
        };
        java_io_Console['istty()Z'] = function (thread) {
            return process.stdout.isTTY;
        };
        return java_io_Console;
    }());
    var java_io_FileDescriptor = (function () {
        function java_io_FileDescriptor() {
        }
        java_io_FileDescriptor['sync()V'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_io_FileDescriptor['initIDs()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_io_FileDescriptor;
    }());
    var java_io_FileInputStream = (function () {
        function java_io_FileInputStream() {
        }
        java_io_FileInputStream['open0(Ljava/lang/String;)V'] = function (thread, javaThis, filename) {
            var filepath = filename.toString();
            // TODO: actually look at the mode
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.open(filepath, 'r', function (e, fd) {
                if (e != null) {
                    if (e.code === 'ENOENT') {
                        thread.throwNewException('Ljava/io/FileNotFoundException;', "" + filepath + " (No such file or directory)");
                    }
                    else {
                        thread.throwNewException('Ljava/lang/InternalError', 'Internal JVM error: ' + e);
                    }
                }
                else {
                    var fdObj = javaThis['java/io/FileInputStream/fd'];
                    fdObj['java/io/FileDescriptor/fd'] = fd;
                    FDState.open(fd, 0);
                    thread.asyncReturn();
                }
            });
        };
        java_io_FileInputStream['read0()I'] = function (thread, javaThis) {
            var fdObj = javaThis["java/io/FileInputStream/fd"], fd = fdObj["java/io/FileDescriptor/fd"];
            if (-1 === fd) {
                thread.throwNewException("Ljava/io/IOException;", "Bad file descriptor");
            }
            else if (0 !== fd) {
                // this is a real file that we've already opened
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                var buf = new Buffer(1);
                fs.read(fd, buf, 0, 1, FDState.getPos(fd), function (err, bytes_read) {
                    if (err) {
                        return throwNodeError(thread, err);
                    }
                    FDState.incrementPos(fd, 1);
                    thread.asyncReturn(0 === bytes_read ? -1 : buf[0]);
                });
            }
            else {
                // reading from System.in, do it async
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                async_input(1, function (byte) {
                    thread.asyncReturn(0 === byte.length ? -1 : byte[0]);
                });
            }
        };
        java_io_FileInputStream['readBytes([BII)I'] = function (thread, javaThis, byteArr, offset, nBytes) {
            var buf, pos, fdObj = javaThis["java/io/FileInputStream/fd"], fd = fdObj["java/io/FileDescriptor/fd"];
            if (offset + nBytes > byteArr.array.length) {
                thread.throwNewException('Ljava/lang/IndexOutOfBoundsException;', "");
                return;
            }
            if (nBytes === 0) {
                return 0;
            }
            else if (-1 === fd) {
                thread.throwNewException("Ljava/io/IOException;", "Bad file descriptor");
            }
            else if (0 !== fd) {
                // this is a real file that we've already opened
                pos = FDState.getPos(fd);
                buf = new Buffer(nBytes);
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                fs.read(fd, buf, 0, nBytes, pos, function (err, bytesRead) {
                    if (null != err) {
                        throwNodeError(thread, err);
                    }
                    else {
                        // not clear why, but sometimes node doesn't move the
                        // file pointer, so we do it here ourselves.
                        FDState.incrementPos(fd, bytesRead);
                        for (var i = 0; i < bytesRead; i++) {
                            byteArr.array[offset + i] = buf.readInt8(i);
                        }
                        thread.asyncReturn(0 === bytesRead ? -1 : bytesRead);
                    }
                });
            }
            else {
                // reading from System.in, do it async
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                async_input(nBytes, function (bytes) {
                    var b, idx;
                    for (idx = 0; idx < bytes.length; idx++) {
                        b = bytes.readInt8(idx);
                        byteArr.array[offset + idx] = b;
                    }
                    thread.asyncReturn(bytes.length === 0 ? -1 : bytes.length);
                });
            }
        };
        java_io_FileInputStream['skip(J)J'] = function (thread, javaThis, nBytes) {
            var fdObj = javaThis["java/io/FileInputStream/fd"];
            var fd = fdObj["java/io/FileDescriptor/fd"];
            if (-1 === fd) {
                thread.throwNewException("Ljava/io/IOException;", "Bad file descriptor");
            }
            else if (0 !== fd) {
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                fs.fstat(fd, function (err, stats) {
                    if (err) {
                        return throwNodeError(thread, err);
                    }
                    var bytesLeft = stats.size - FDState.getPos(fd), toSkip = Math.min(nBytes.toNumber(), bytesLeft);
                    FDState.incrementPos(fd, toSkip);
                    thread.asyncReturn(Long.fromNumber(toSkip), null);
                });
            }
            else {
                // reading from System.in, do it async
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                async_input(nBytes.toNumber(), function (bytes) {
                    // we don't care about what the input actually was
                    thread.asyncReturn(Long.fromNumber(bytes.length), null);
                });
            }
        };
        java_io_FileInputStream['available()I'] = function (thread, javaThis) {
            var fdObj = javaThis["java/io/FileInputStream/fd"], fd = fdObj["java/io/FileDescriptor/fd"];
            if (fd === -1) {
                thread.throwNewException("Ljava/io/IOException;", "Bad file descriptor");
            }
            else if (fd === 0) {
                // no buffering for stdin (if fd is 0)
                return 0;
            }
            else {
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                fs.fstat(fd, function (err, stats) {
                    if (err) {
                        return throwNodeError(thread, err);
                    }
                    thread.asyncReturn(stats.size - FDState.getPos(fd));
                });
            }
        };
        java_io_FileInputStream['initIDs()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_io_FileInputStream['close0()V'] = function (thread, javaThis) {
            var fdObj = javaThis['java/io/FileInputStream/fd'], fd = fdObj['java/io/FileDescriptor/fd'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.close(fd, function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    fdObj['java/io/FileDescriptor/fd'] = -1;
                    FDState.close(fd);
                    thread.asyncReturn();
                }
            });
        };
        return java_io_FileInputStream;
    }());
    var java_io_FileOutputStream = (function () {
        function java_io_FileOutputStream() {
        }
        /**
         * Opens a file, with the specified name, for overwriting or appending.
         * @param name name of file to be opened
         * @param append whether the file is to be opened in append mode
         */
        java_io_FileOutputStream['open0(Ljava/lang/String;Z)V'] = function (thread, javaThis, name, append) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.open(name.toString(), append ? 'a' : 'w', function (err, fd) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                var fdObj = javaThis['java/io/FileOutputStream/fd'];
                fdObj['java/io/FileDescriptor/fd'] = fd;
                fs.fstat(fd, function (err, stats) {
                    FDState.setPos(fd, stats.size);
                    thread.asyncReturn();
                });
            });
        };
        /**
         * Writes the specified byte to this file output stream.
         *
         * @param   b   the byte to be written.
         * @param   append   {@code true} if the write operation first
         *     advances the position to the end of file
         */
        java_io_FileOutputStream['write(IZ)V'] = function (thread, javaThis, b, append) {
            // HACK: Avoid reimplementing below for single byte case.
            java_io_FileOutputStream['writeBytes([BIIZ)V'](thread, javaThis, { array: [b] }, 0, 1, append);
        };
        /**
         * Writes a sub array as a sequence of bytes.
         * @param b the data to be written
         * @param off the start offset in the data
         * @param len the number of bytes that are written
         * @param append {@code true} to first advance the position to the
         *     end of file
         * @exception IOException If an I/O error has occurred.
         */
        java_io_FileOutputStream['writeBytes([BIIZ)V'] = function (thread, javaThis, bytes, offset, len, append) {
            var buf = new Buffer(bytes.array), fdObj = javaThis['java/io/FileOutputStream/fd'], fd = fdObj['java/io/FileDescriptor/fd'];
            if (fd === -1) {
                thread.throwNewException('Ljava/io/IOException;', "Bad file descriptor");
            }
            else if (fd !== 1 && fd !== 2) {
                // normal file
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                fs.write(fd, buf, offset, len, FDState.getPos(fd), function (err, numBytes) {
                    if (err) {
                        return throwNodeError(thread, err);
                    }
                    FDState.incrementPos(fd, numBytes);
                    thread.asyncReturn();
                });
            }
            else {
                // The string is in UTF-8 format. But now we need to convert them to UTF-16 to print 'em out. :(
                var output = buf.toString("utf8", offset, offset + len);
                if (fd === 1) {
                    process.stdout.write(output);
                }
                else if (fd === 2) {
                    process.stderr.write(output);
                }
                // For the browser implementation -- the DOM doesn't get repainted
                // unless we give the event loop a chance to spin.
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                setImmediate(function () {
                    thread.asyncReturn();
                });
            }
        };
        java_io_FileOutputStream['close0()V'] = function (thread, javaThis) {
            var fdObj = javaThis['java/io/FileOutputStream/fd'], fd = fdObj['java/io/FileDescriptor/fd'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.close(fd, function (err) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                else {
                    fdObj['java/io/FileDescriptor/fd'] = -1;
                    FDState.close(fd);
                    thread.asyncReturn();
                }
            });
        };
        java_io_FileOutputStream['initIDs()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_io_FileOutputStream;
    }());
    var java_io_ObjectInputStream = (function () {
        function java_io_ObjectInputStream() {
        }
        java_io_ObjectInputStream['bytesToFloats([BI[FII)V'] = function (thread, arg0, arg1, arg2, arg3, arg4) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_io_ObjectInputStream['bytesToDoubles([BI[DII)V'] = function (thread, arg0, arg1, arg2, arg3, arg4) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_io_ObjectInputStream;
    }());
    var java_io_ObjectOutputStream = (function () {
        function java_io_ObjectOutputStream() {
        }
        java_io_ObjectOutputStream['floatsToBytes([FI[BII)V'] = function (thread, arg0, arg1, arg2, arg3, arg4) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_io_ObjectOutputStream['doublesToBytes([DI[BII)V'] = function (thread, arg0, arg1, arg2, arg3, arg4) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_io_ObjectOutputStream;
    }());
    var java_io_ObjectStreamClass = (function () {
        function java_io_ObjectStreamClass() {
        }
        java_io_ObjectStreamClass['initNative()V'] = function (thread) {
            // NOP
        };
        java_io_ObjectStreamClass['hasStaticInitializer(Ljava/lang/Class;)Z'] = function (thread, jco) {
            // check if cls has a <clinit> method
            return jco.$cls.getMethod('<clinit>()V') !== null;
        };
        return java_io_ObjectStreamClass;
    }());
    var java_io_RandomAccessFile = (function () {
        function java_io_RandomAccessFile() {
        }
        java_io_RandomAccessFile['open0(Ljava/lang/String;I)V'] = function (thread, javaThis, filename, mode) {
            var filepath = filename.toString(), rafStatics = javaThis.getClass().getConstructor(thread), modeStr;
            switch (mode) {
                case rafStatics["java/io/RandomAccessFile/O_RDONLY"]:
                    modeStr = 'r';
                    break;
                case rafStatics["java/io/RandomAccessFile/O_RDWR"]:
                    modeStr = 'r+';
                    break;
                case rafStatics["java/io/RandomAccessFile/O_SYNC"]:
                case rafStatics["java/io/RandomAccessFile/O_DSYNC"]:
                    modeStr = 'rs+';
                    break;
            }
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.open(filepath, modeStr, function (e, fd) {
                if (e) {
                    return throwNodeError(thread, e);
                }
                else {
                    var fdObj = javaThis['java/io/RandomAccessFile/fd'];
                    fdObj['java/io/FileDescriptor/fd'] = fd;
                    FDState.open(fd, 0);
                    thread.asyncReturn();
                }
            });
        };
        /**
         * Reads a byte of data from this file. The byte is returned as an
         * integer in the range 0 to 255 ({@code 0x00-0x0ff}). This
         * method blocks if no input is yet available.
         * <p>
         * Although {@code RandomAccessFile} is not a subclass of
         * {@code InputStream}, this method behaves in exactly the same
         * way as the {@link InputStream#read()} method of
         * {@code InputStream}.
         *
         * @return     the next byte of data, or {@code -1} if the end of the
         *             file has been reached.
         * @exception  IOException  if an I/O error occurs. Not thrown if
         *                          end-of-file has been reached.
         */
        java_io_RandomAccessFile['read0()I'] = function (thread, javaThis) {
            var fdObj = javaThis["java/io/RandomAccessFile/fd"], fd = fdObj["java/io/FileDescriptor/fd"], buf = new Buffer(1);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.read(fd, buf, 0, 1, FDState.getPos(fd), function (err, bytesRead) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                else {
                    FDState.incrementPos(fd, bytesRead);
                    // Read as uint, since return value is unsigned.
                    thread.asyncReturn(bytesRead === 0 ? -1 : buf[0]);
                }
            });
        };
        java_io_RandomAccessFile['readBytes([BII)I'] = function (thread, javaThis, byte_arr, offset, len) {
            var fdObj = javaThis["java/io/RandomAccessFile/fd"], fd = fdObj["java/io/FileDescriptor/fd"], buf = new Buffer(len);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.read(fd, buf, 0, len, FDState.getPos(fd), function (err, bytesRead) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                else {
                    for (var i = 0; i < bytesRead; i++) {
                        byte_arr.array[offset + i] = buf.readInt8(i);
                    }
                    FDState.incrementPos(fd, bytesRead);
                    thread.asyncReturn(0 === bytesRead && 0 !== len ? -1 : bytesRead);
                }
            });
        };
        java_io_RandomAccessFile['write0(I)V'] = function (thread, javaThis, value) {
            var fdObj = javaThis["java/io/RandomAccessFile/fd"];
            var fd = fdObj["java/io/FileDescriptor/fd"];
            var data = new Buffer(1);
            data.writeInt8(value, 0);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.write(fd, data, 0, 1, FDState.getPos(fd), function (err, numBytes) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                FDState.incrementPos(fd, numBytes);
                thread.asyncReturn();
            });
        };
        java_io_RandomAccessFile['writeBytes([BII)V'] = function (thread, javaThis, byteArr, offset, len) {
            var fdObj = javaThis["java/io/RandomAccessFile/fd"], fd = fdObj["java/io/FileDescriptor/fd"], buf = new Buffer(byteArr.array);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.write(fd, buf, offset, len, FDState.getPos(fd), function (err, numBytes) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                FDState.incrementPos(fd, numBytes);
                thread.asyncReturn();
            });
        };
        java_io_RandomAccessFile['getFilePointer()J'] = function (thread, javaThis) {
            return Long.fromNumber(FDState.getPos(javaThis['java/io/RandomAccessFile/fd']['java/io/FileDescriptor/fd']));
        };
        java_io_RandomAccessFile['seek0(J)V'] = function (thread, javaThis, pos) {
            FDState.setPos(javaThis['java/io/RandomAccessFile/fd']['java/io/FileDescriptor/fd'], pos.toNumber());
        };
        java_io_RandomAccessFile['length()J'] = function (thread, javaThis) {
            var fdObj = javaThis['java/io/RandomAccessFile/fd'], fd = fdObj['java/io/FileDescriptor/fd'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.fstat(fd, function (err, stats) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                thread.asyncReturn(Long.fromNumber(stats.size), null);
            });
        };
        java_io_RandomAccessFile['setLength(J)V'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_io_RandomAccessFile['initIDs()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_io_RandomAccessFile['close0()V'] = function (thread, javaThis) {
            var fdObj = javaThis['java/io/RandomAccessFile/fd'], fd = fdObj['java/io/FileDescriptor/fd'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.close(fd, function (err) {
                if (err) {
                    return throwNodeError(thread, err);
                }
                else {
                    fdObj['java/io/FileDescriptor/fd'] = -1;
                    FDState.close(fd);
                    thread.asyncReturn();
                }
            });
        };
        return java_io_RandomAccessFile;
    }());
    var java_io_UnixFileSystem = (function () {
        function java_io_UnixFileSystem() {
        }
        java_io_UnixFileSystem['canonicalize0(Ljava/lang/String;)Ljava/lang/String;'] = function (thread, javaThis, jvmPathStr) {
            var jsStr = jvmPathStr.toString();
            return util.initString(thread.getBsCl(), path.resolve(path.normalize(jsStr)));
        };
        java_io_UnixFileSystem['getBooleanAttributes0(Ljava/io/File;)I'] = function (thread, javaThis, file) {
            var filepath = file['java/io/File/path'], fileSystem = thread.getBsCl().getInitializedClass(thread, 'Ljava/io/FileSystem;').getConstructor(thread);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath.toString(), function (stats) {
                // Returns 0 if file does not exist, or any other error occurs.
                var rv = 0;
                if (stats !== null) {
                    rv |= fileSystem['java/io/FileSystem/BA_EXISTS'];
                    if (stats.isFile()) {
                        rv |= fileSystem['java/io/FileSystem/BA_REGULAR'];
                    }
                    else if (stats.isDirectory()) {
                        rv |= fileSystem['java/io/FileSystem/BA_DIRECTORY'];
                    }
                }
                thread.asyncReturn(rv);
            });
        };
        java_io_UnixFileSystem['checkAccess(Ljava/io/File;I)Z'] = function (thread, javaThis, file, access) {
            var filepath = file['java/io/File/path'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath.toString(), function (stats) {
                if (stats == null) {
                    thread.asyncReturn(0);
                }
                else {
                    // XXX: Assuming we're owner/group/other. :)
                    // Shift access so it's present in owner/group/other.
                    // Then, AND with the actual mode, and check if the result is above 0.
                    // That indicates that the access bit we're looking for was set on
                    // one of owner/group/other.
                    var mask = access | (access << 3) | (access << 6);
                    thread.asyncReturn((stats.mode & mask) > 0 ? 1 : 0);
                }
            });
        };
        java_io_UnixFileSystem['getLastModifiedTime(Ljava/io/File;)J'] = function (thread, javaThis, file) {
            var filepath = file['java/io/File/path'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath.toString(), function (stats) {
                if (stats == null) {
                    thread.asyncReturn(Long.ZERO, null);
                }
                else {
                    thread.asyncReturn(Long.fromNumber(stats.mtime.getTime()), null);
                }
            });
        };
        java_io_UnixFileSystem['getLength(Ljava/io/File;)J'] = function (thread, javaThis, file) {
            var filepath = file['java/io/File/path'];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.stat(filepath.toString(), function (err, stat) {
                thread.asyncReturn(err != null ? Long.ZERO : Long.fromNumber(stat.size), null);
            });
        };
        java_io_UnixFileSystem['setPermission(Ljava/io/File;IZZ)Z'] = function (thread, javaThis, file, access, enable, owneronly) {
            // Access is equal to one of the following static fields:
            // * FileSystem.ACCESS_READ (0x04)
            // * FileSystem.ACCESS_WRITE (0x02)
            // * FileSystem.ACCESS_EXECUTE (0x01)
            // These are conveniently identical to their Unix equivalents, which
            // we have to convert to for Node.
            // XXX: Currently assuming that the above assumption holds across JCLs.
            var filepath = file['java/io/File/path'].toString();
            if (owneronly) {
                // Shift it 6 bits over into the 'owner' region of the access mode.
                access <<= 6;
            }
            else {
                // Clone it into the 'owner' and 'group' regions.
                access |= (access << 6) | (access << 3);
            }
            if (!enable) {
                // Do an invert and we'll AND rather than OR.
                access = ~access;
            }
            // Returns true on success, false on failure.
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            // Fetch existing permissions on file.
            statFile(filepath, function (stats) {
                if (stats == null) {
                    thread.asyncReturn(0);
                }
                else {
                    var existing_access = stats.mode;
                    // Apply mask.
                    access = enable ? existing_access | access : existing_access & access;
                    // Set new permissions.
                    fs.chmod(filepath, access, function (err) {
                        thread.asyncReturn(err != null ? 0 : 1);
                    });
                }
            });
        };
        java_io_UnixFileSystem['createFileExclusively(Ljava/lang/String;)Z'] = function (thread, javaThis, path) {
            var filepath = path.toString();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath, function (stat) {
                if (stat != null) {
                    thread.asyncReturn(0);
                }
                else {
                    fs.open(filepath, 'w', function (err, fd) {
                        if (err != null) {
                            thread.throwNewException('Ljava/io/IOException;', err.message);
                        }
                        else {
                            fs.close(fd, function (err) {
                                if (err != null) {
                                    thread.throwNewException('Ljava/io/IOException;', err.message);
                                }
                                else {
                                    thread.asyncReturn(1);
                                }
                            });
                        }
                    });
                }
            });
        };
        java_io_UnixFileSystem['delete0(Ljava/io/File;)Z'] = function (thread, javaThis, file) {
            // Delete the file or directory denoted by the given abstract
            // pathname, returning true if and only if the operation succeeds.
            // If file is a directory, it must be empty.
            var filepath = file['java/io/File/path'].toString();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath, function (stats) {
                if (stats == null) {
                    thread.asyncReturn(0);
                }
                else if (stats.isDirectory()) {
                    fs.readdir(filepath, function (err, files) {
                        if (files.length > 0) {
                            thread.asyncReturn(0);
                        }
                        else {
                            fs.rmdir(filepath, function (err) {
                                thread.asyncReturn(1);
                            });
                        }
                    });
                }
                else {
                    fs.unlink(filepath, function (err) {
                        thread.asyncReturn(1);
                    });
                }
            });
        };
        java_io_UnixFileSystem['list(Ljava/io/File;)[Ljava/lang/String;'] = function (thread, javaThis, file) {
            var filepath = file['java/io/File/path'], bsCl = thread.getBsCl();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.readdir(filepath.toString(), function (err, files) {
                if (err != null) {
                    thread.asyncReturn(null);
                }
                else {
                    thread.asyncReturn(util.newArrayFromData(thread, thread.getBsCl(), '[Ljava/lang/String;', files.map(function (file) { return util.initString(thread.getBsCl(), file); })));
                }
            });
        };
        java_io_UnixFileSystem['createDirectory(Ljava/io/File;)Z'] = function (thread, javaThis, file) {
            var filepath = file['java/io/File/path'].toString();
            // Already exists.
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath, function (stat) {
                if (stat != null) {
                    thread.asyncReturn(0);
                }
                else {
                    fs.mkdir(filepath, function (err) {
                        thread.asyncReturn(err != null ? 0 : 1);
                    });
                }
            });
        };
        java_io_UnixFileSystem['rename0(Ljava/io/File;Ljava/io/File;)Z'] = function (thread, javaThis, file1, file2) {
            var file1path = file1['java/io/File/path'].toString(), file2path = file2['java/io/File/path'].toString();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.rename(file1path, file2path, function (err) {
                thread.asyncReturn(err != null ? 0 : 1);
            });
        };
        java_io_UnixFileSystem['setLastModifiedTime(Ljava/io/File;J)Z'] = function (thread, javaThis, file, time) {
            var mtime = time.toNumber(), atime = (new Date).getTime(), filepath = file['java/io/File/path'].toString();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.utimes(filepath, atime, mtime, function (err) {
                thread.asyncReturn(1);
            });
        };
        java_io_UnixFileSystem['setReadOnly(Ljava/io/File;)Z'] = function (thread, javaThis, file) {
            // We'll be unsetting write permissions.
            // Leading 0o indicates octal.
            var filepath = file['java/io/File/path'].toString(), mask = ~0x92;
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            statFile(filepath, function (stats) {
                if (stats == null) {
                    thread.asyncReturn(0);
                }
                else {
                    fs.chmod(filepath, stats.mode & mask, function (err) {
                        thread.asyncReturn(err != null ? 0 : 1);
                    });
                }
            });
        };
        java_io_UnixFileSystem['getSpace(Ljava/io/File;I)J'] = function (thread, javaThis, file, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        java_io_UnixFileSystem['initIDs()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_io_UnixFileSystem;
    }());
    return {
        'java/io/Console': java_io_Console,
        'java/io/FileDescriptor': java_io_FileDescriptor,
        'java/io/FileInputStream': java_io_FileInputStream,
        'java/io/FileOutputStream': java_io_FileOutputStream,
        'java/io/ObjectInputStream': java_io_ObjectInputStream,
        'java/io/ObjectOutputStream': java_io_ObjectOutputStream,
        'java/io/ObjectStreamClass': java_io_ObjectStreamClass,
        'java/io/RandomAccessFile': java_io_RandomAccessFile,
        'java/io/UnixFileSystem': java_io_UnixFileSystem
    };
}
exports.__esModule = true;
exports["default"] = default_1;
;
//# sourceMappingURL=data:application/json;base64,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