"use strict";
var Doppio = require('../doppiojvm');
var ReferenceClassData = Doppio.VM.ClassFile.ReferenceClassData;
var logging = Doppio.Debug.Logging;
var util = Doppio.VM.Util;
var ArrayClassData = Doppio.VM.ClassFile.ArrayClassData;
var ThreadStatus = Doppio.VM.Enums.ThreadStatus;
var Method = Doppio.VM.ClassFile.Method;
var Long = Doppio.VM.Long;
var assert = Doppio.Debug.Assert;
var PrimitiveClassData = Doppio.VM.ClassFile.PrimitiveClassData;
var MethodHandleReferenceKind = Doppio.VM.Enums.MethodHandleReferenceKind;
function default_1() {
    var debug = logging.debug;
    function arrayGet(thread, arr, idx) {
        if (arr == null) {
            thread.throwNewException('Ljava/lang/NullPointerException;', '');
        }
        else {
            var array = arr.array;
            if (idx < 0 || idx >= array.length) {
                thread.throwNewException('Ljava/lang/ArrayIndexOutOfBoundsException;', 'Tried to access an illegal index in an array.');
            }
            else {
                return array[idx];
            }
        }
    }
    function isNotNull(thread, obj) {
        if (obj == null) {
            thread.throwNewException('Ljava/lang/NullPointerException;', '');
            return false;
        }
        else {
            return true;
        }
    }
    function verifyArray(thread, obj) {
        if (!(obj.getClass() instanceof ArrayClassData)) {
            thread.throwNewException('Ljava/lang/IllegalArgumentException;', 'Object is not an array.');
            return false;
        }
        else {
            return true;
        }
    }
    var java_lang_Class = (function () {
        function java_lang_Class() {
        }
        java_lang_Class['forName0(Ljava/lang/String;ZLjava/lang/ClassLoader;Ljava/lang/Class;)Ljava/lang/Class;'] = function (thread, jvmStr, initialize, jclo, caller) {
            var classname = util.int_classname(jvmStr.toString());
            if (!util.verify_int_classname(classname)) {
                thread.throwNewException('Ljava/lang/ClassNotFoundException;', classname);
            }
            else {
                var loader = util.getLoader(thread, jclo);
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                if (initialize) {
                    loader.initializeClass(thread, classname, function (cls) {
                        if (cls != null) {
                            thread.asyncReturn(cls.getClassObject(thread));
                        }
                    });
                }
                else {
                    loader.resolveClass(thread, classname, function (cls) {
                        if (cls != null) {
                            thread.asyncReturn(cls.getClassObject(thread));
                        }
                    });
                }
            }
        };
        java_lang_Class['isInstance(Ljava/lang/Object;)Z'] = function (thread, javaThis, obj) {
            if (obj !== null) {
                return obj.getClass().isCastable(javaThis.$cls);
            }
            else {
                return false;
            }
        };
        java_lang_Class['isAssignableFrom(Ljava/lang/Class;)Z'] = function (thread, javaThis, cls) {
            return cls.$cls.isCastable(javaThis.$cls);
        };
        java_lang_Class['isInterface()Z'] = function (thread, javaThis) {
            if (!(javaThis.$cls instanceof ReferenceClassData)) {
                return false;
            }
            return javaThis.$cls.accessFlags.isInterface();
        };
        java_lang_Class['isArray()Z'] = function (thread, javaThis) {
            return javaThis.$cls instanceof ArrayClassData;
        };
        java_lang_Class['isPrimitive()Z'] = function (thread, javaThis) {
            return javaThis.$cls instanceof PrimitiveClassData;
        };
        java_lang_Class['getName0()Ljava/lang/String;'] = function (thread, javaThis) {
            return util.initString(thread.getBsCl(), javaThis.$cls.getExternalName());
        };
        java_lang_Class['getSuperclass()Ljava/lang/Class;'] = function (thread, javaThis) {
            if (javaThis.$cls instanceof PrimitiveClassData) {
                return null;
            }
            var cls = javaThis.$cls;
            if (cls.accessFlags.isInterface() || (cls.getSuperClass() == null)) {
                return null;
            }
            return cls.getSuperClass().getClassObject(thread);
        };
        java_lang_Class['getInterfaces0()[Ljava/lang/Class;'] = function (thread, javaThis) {
            return util.newArrayFromData(thread, thread.getBsCl(), '[Ljava/lang/Class;', javaThis.$cls.getInterfaces().map(function (iface) { return iface.getClassObject(thread); }));
        };
        java_lang_Class['getComponentType()Ljava/lang/Class;'] = function (thread, javaThis) {
            if (!(javaThis.$cls instanceof ArrayClassData)) {
                return null;
            }
            // As this array type is loaded, the component type is guaranteed
            // to be loaded as well. No need for asynchronicity.
            return javaThis.$cls.getComponentClass().getClassObject(thread);
        };
        java_lang_Class['getModifiers()I'] = function (thread, javaThis) {
            return javaThis.$cls.accessFlags.getRawByte();
        };
        java_lang_Class['getSigners()[Ljava/lang/Object;'] = function (thread, javaThis) {
            var internalName = javaThis.$cls.getInternalName();
            var internalType = internalName[0];
            if (util.is_reference_type(internalType)) {
                return javaThis.signers == null ? null : util.newArrayFromData(thread, thread.getBsCl(), "[Ljava/lang/Object;", javaThis.signers);
            }
            return null;
        };
        java_lang_Class['setSigners([Ljava/lang/Object;)V'] = function (thread, javaThis, arg0) {
            var internalName = javaThis.$cls.getInternalName();
            var internalType = internalName[0];
            if (util.is_reference_type(internalType)) {
                javaThis.signers = arg0 == null ? null : arg0.array.slice();
            }
        };
        java_lang_Class['getEnclosingMethod0()[Ljava/lang/Object;'] = function (thread, javaThis) {
            var encDesc = null, enc_name = null, bsCl = thread.getBsCl();
            if (javaThis.$cls instanceof ReferenceClassData) {
                var cls = javaThis.$cls, em = cls.getAttribute('EnclosingMethod');
                if (em == null) {
                    return null;
                }
                // array w/ 3 elements:
                // - the immediately enclosing class (java/lang/Class)
                // - the immediately enclosing method or constructor's name (can be null). (String)
                // - the immediately enclosing method or constructor's descriptor (null iff name is). (String)
                var rv = util.newArray(thread, bsCl, '[Ljava/lang/Object;', 3), encClassRef = em.encClass;
                if (em.encMethod != null) {
                    rv.array[1] = util.initString(bsCl, em.encMethod.name);
                    rv.array[2] = util.initString(bsCl, em.encMethod.descriptor);
                }
                if (encClassRef.isResolved()) {
                    rv.array[0] = encClassRef.cls.getClassObject(thread);
                    return rv;
                }
                else {
                    thread.setStatus(ThreadStatus.ASYNC_WAITING);
                    encClassRef.resolve(thread, cls.getLoader(), cls, function (status) {
                        if (status) {
                            rv.array[0] = encClassRef.cls.getClassObject(thread);
                            thread.asyncReturn(rv);
                        }
                    });
                }
            }
            return null;
        };
        java_lang_Class['getDeclaringClass0()Ljava/lang/Class;'] = function (thread, javaThis) {
            var declaringName, entry, name, i, len;
            if (javaThis.$cls instanceof ReferenceClassData) {
                var cls = javaThis.$cls, icls = cls.getAttribute('InnerClasses');
                if (icls == null) {
                    return null;
                }
                var myClass = cls.getInternalName(), innerClassInfo = icls.classes;
                for (i = 0, len = innerClassInfo.length; i < len; i++) {
                    entry = innerClassInfo[i];
                    if (entry.outerInfoIndex <= 0) {
                        continue;
                    }
                    name = cls.constantPool.get(entry.innerInfoIndex).name;
                    if (name !== myClass) {
                        continue;
                    }
                    // XXX(jez): this assumes that the first enclosing entry is also
                    // the immediate enclosing parent, and I'm not 100% sure this is
                    // guaranteed by the spec
                    declaringName = cls.constantPool.get(entry.outerInfoIndex);
                    if (declaringName.isResolved()) {
                        return declaringName.cls.getClassObject(thread);
                    }
                    else {
                        thread.setStatus(ThreadStatus.ASYNC_WAITING);
                        declaringName.resolve(thread, cls.getLoader(), cls, function (status) {
                            if (status) {
                                thread.asyncReturn(declaringName.cls.getClassObject(thread));
                            }
                        });
                    }
                }
            }
            return null;
        };
        java_lang_Class['getProtectionDomain0()Ljava/security/ProtectionDomain;'] = function (thread, javaThis) {
            return javaThis.$cls.getProtectionDomain();
        };
        java_lang_Class['getPrimitiveClass(Ljava/lang/String;)Ljava/lang/Class;'] = function (thread, jvmStr) {
            var type_desc = util.typestr2descriptor(jvmStr.toString()), prim_cls = thread.getBsCl().getInitializedClass(thread, type_desc);
            return prim_cls.getClassObject(thread);
        };
        java_lang_Class['getGenericSignature0()Ljava/lang/String;'] = function (thread, javaThis) {
            var cls = javaThis.$cls;
            // TODO: What if it is a primitive type? What do I return?
            if (!util.is_primitive_type(cls.getInternalName())) {
                var sigAttr = cls.getAttribute('Signature');
                if (sigAttr != null && sigAttr.sig != null) {
                    return util.initString(thread.getBsCl(), sigAttr.sig);
                }
            }
            return null;
        };
        /**
         * Returns RuntimeVisibleAnnotations defined on the class.
         */
        java_lang_Class['getRawAnnotations()[B'] = function (thread, javaThis) {
            var cls = javaThis.$cls, annotationsVisible = cls.getAttribute('RuntimeVisibleAnnotations'), methods, i, m;
            if (annotationsVisible !== null) {
                // TODO: Use a typed array?
                var bytes = annotationsVisible.rawBytes, data = new Array(bytes.length);
                for (var i = 0; i < bytes.length; i++) {
                    data[i] = bytes.readInt8(i);
                }
                return util.newArrayFromData(thread, thread.getBsCl(), '[B', data);
            }
            return null;
        };
        java_lang_Class['getConstantPool()Lsun/reflect/ConstantPool;'] = function (thread, javaThis) {
            var cls = javaThis.$cls, cpObj = util.newObject(thread, thread.getBsCl(), 'Lsun/reflect/ConstantPool;');
            // @todo Make this a proper JavaObject. I don't think the JCL uses it as such,
            // but right now this function fails any automated sanity checks on return values.
            cpObj['sun/reflect/ConstantPool/constantPoolOop'] = cls.constantPool;
            return cpObj;
        };
        java_lang_Class['getDeclaredFields0(Z)[Ljava/lang/reflect/Field;'] = function (thread, javaThis, publicOnly) {
            var fields = javaThis.$cls.getFields();
            if (publicOnly) {
                fields = fields.filter(function (f) { return f.accessFlags.isPublic(); });
            }
            var rv = util.newArray(thread, thread.getBsCl(), '[Ljava/lang/reflect/Field;', fields.length), i = 0;
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            util.asyncForEach(fields, function (f, nextItem) {
                f.reflector(thread, function (fieldObj) {
                    if (fieldObj !== null) {
                        rv.array[i++] = fieldObj;
                        nextItem();
                    }
                });
            }, function () {
                thread.asyncReturn(rv);
            });
        };
        java_lang_Class['getDeclaredMethods0(Z)[Ljava/lang/reflect/Method;'] = function (thread, javaThis, publicOnly) {
            var methods = javaThis.$cls.getMethods().filter(function (m) {
                return m.name[0] !== '<' && (m.accessFlags.isPublic() || !publicOnly);
            }), rv = util.newArray(thread, thread.getBsCl(), '[Ljava/lang/reflect/Method;', methods.length), i = 0;
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            util.asyncForEach(methods, function (m, nextItem) {
                m.reflector(thread, function (methodObj) {
                    if (methodObj !== null) {
                        rv.array[i++] = methodObj;
                        nextItem();
                    }
                });
            }, function () {
                thread.asyncReturn(rv);
            });
        };
        java_lang_Class['getDeclaredConstructors0(Z)[Ljava/lang/reflect/Constructor;'] = function (thread, javaThis, publicOnly) {
            var methods = javaThis.$cls.getMethods().filter(function (m) {
                return m.name === '<init>' && (!publicOnly || m.accessFlags.isPublic());
            }), rv = util.newArray(thread, thread.getBsCl(), '[Ljava/lang/reflect/Constructor;', methods.length), i = 0;
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            util.asyncForEach(methods, function (m, nextItem) {
                m.reflector(thread, function (methodObj) {
                    if (methodObj !== null) {
                        rv.array[i++] = methodObj;
                        nextItem();
                    }
                });
            }, function () {
                thread.asyncReturn(rv);
            });
        };
        java_lang_Class['getDeclaredClasses0()[Ljava/lang/Class;'] = function (thread, javaThis) {
            var ret = util.newArray(thread, thread.getBsCl(), '[Ljava/lang/Class;', 0), cls = javaThis.$cls;
            if (cls instanceof ReferenceClassData) {
                var myClass_1 = cls.getInternalName(), iclses = cls.getAttributes('InnerClasses'), flatNames = [];
                if (iclses.length === 0) {
                    return ret;
                }
                for (var i = 0; i < iclses.length; i++) {
                    flatNames = flatNames.concat(iclses[i].classes.filter(function (c) {
                        // select inner classes where the enclosing class is my_class
                        return c.outerInfoIndex > 0 && cls.constantPool.get(c.outerInfoIndex).name === myClass_1;
                    })
                        .map(function (c) { return (cls.constantPool.get(c.innerInfoIndex)); }));
                }
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                util.asyncForEach(flatNames, function (clsRef, nextItem) {
                    if (clsRef.isResolved()) {
                        ret.array.push(clsRef.cls.getClassObject(thread));
                        nextItem();
                    }
                    else {
                        clsRef.resolve(thread, cls.getLoader(), javaThis.getClass(), function (status) {
                            if (status) {
                                ret.array.push(clsRef.cls.getClassObject(thread));
                                nextItem();
                            }
                        });
                    }
                }, function () { return thread.asyncReturn(ret); });
            }
            else {
                return ret;
            }
        };
        java_lang_Class['desiredAssertionStatus0(Ljava/lang/Class;)Z'] = function (thread, arg0) {
            if (arg0.$cls.getLoader().getLoaderObject() === null) {
                return thread.getJVM().areSystemAssertionsEnabled();
            }
            return false;
        };
        return java_lang_Class;
    }());
    var java_lang_ClassLoader$NativeLibrary = (function () {
        function java_lang_ClassLoader$NativeLibrary() {
        }
        java_lang_ClassLoader$NativeLibrary['load(Ljava/lang/String;Z)V'] = function (thread, javaThis, name, isBuiltIn) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_ClassLoader$NativeLibrary['find(Ljava/lang/String;)J'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        java_lang_ClassLoader$NativeLibrary['unload(Ljava/lang/String;Z)V'] = function (thread, javaThis, name) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_lang_ClassLoader$NativeLibrary;
    }());
    // Fun Note: The bootstrap classloader object is represented by null.
    var java_lang_ClassLoader = (function () {
        function java_lang_ClassLoader() {
        }
        java_lang_ClassLoader['defineClass0(Ljava/lang/String;[BIILjava/security/ProtectionDomain;)Ljava/lang/Class;'] = function (thread, javaThis, arg0, arg1, arg2, arg3, arg4) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        java_lang_ClassLoader['defineClass1(Ljava/lang/String;[BIILjava/security/ProtectionDomain;Ljava/lang/String;)Ljava/lang/Class;'] = function (thread, javaThis, name, bytes, offset, len, pd, source) {
            var loader = util.getLoader(thread, javaThis), type = util.int_classname(name.toString()), cls = loader.defineClass(thread, type, util.byteArray2Buffer(bytes.array, offset, len), pd);
            if (cls == null) {
                return null;
            }
            // Ensure that this class is resolved.
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            cls.resolve(thread, function (status) {
                // NULL status means resolution failed.
                if (status !== null) {
                    thread.asyncReturn(cls.getClassObject(thread));
                }
            }, true);
        };
        java_lang_ClassLoader['defineClass2(Ljava/lang/String;Ljava/nio/ByteBuffer;IILjava/security/ProtectionDomain;Ljava/lang/String;)Ljava/lang/Class;'] = function (thread, javaThis, name, b, off, len, pd, source) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        java_lang_ClassLoader['resolveClass0(Ljava/lang/Class;)V'] = function (thread, javaThis, cls) {
            var loader = util.getLoader(thread, javaThis);
            if (cls.$cls.isResolved()) {
                return;
            }
            // Ensure that this class is resolved.
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            cls.$cls.resolve(thread, function (cdata) {
                if (cdata !== null) {
                    thread.asyncReturn();
                }
                // Else: An exception occurred.
            }, true);
        };
        java_lang_ClassLoader['findBootstrapClass(Ljava/lang/String;)Ljava/lang/Class;'] = function (thread, javaThis, name) {
            var type = util.int_classname(name.toString());
            // This returns null in OpenJDK7, but actually can throw an exception
            // in OpenJDK6.
            // TODO: Fix currently incorrect behavior for our JDK. Should return null, not throw an exception on failure.
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            thread.getBsCl().resolveClass(thread, type, function (cls) {
                if (cls != null) {
                    thread.asyncReturn(cls.getClassObject(thread));
                }
            }, true);
        };
        java_lang_ClassLoader['findLoadedClass0(Ljava/lang/String;)Ljava/lang/Class;'] = function (thread, javaThis, name) {
            var loader = util.getLoader(thread, javaThis), type = util.int_classname(name.toString()), 
            // Return JavaClassObject if loaded, or null otherwise.
            cls = loader.getResolvedClass(type);
            if (cls != null) {
                return cls.getClassObject(thread);
            }
            else {
                return null;
            }
        };
        java_lang_ClassLoader['retrieveDirectives()Ljava/lang/AssertionStatusDirectives;'] = function (thread) {
            var jvm = thread.getJVM(), bsCl = thread.getBsCl();
            thread.import('Ljava/lang/AssertionStatusDirectives;', function (asd) {
                var directives = new asd();
                var enabledAssertions = jvm.getEnabledAssertions();
                // The classes for which assertions are to be enabled or disabled.
                var classes = [], 
                // A parallel array to classes, indicating whether each class
                // is to have assertions enabled or disabled.
                classEnabled = [], 
                // The package-trees for which assertions are to be enabled or disabled.
                packages = [], 
                // A parallel array to packages, indicating whether each
                // package-tree is to have assertions enabled or disabled.
                packageEnabled = [], deflt = false, processAssertions = function (enabled) {
                    return function (name) {
                        var dotIndex = name.indexOf('...');
                        if (dotIndex === -1) {
                            classes.push(name);
                            classEnabled.push(enabled);
                        }
                        else {
                            packages.push(name.slice(0, dotIndex));
                            packageEnabled.push(enabled);
                        }
                    };
                };
                jvm.getDisabledAssertions().forEach(processAssertions(0));
                if (typeof (enabledAssertions) === 'boolean') {
                    deflt = enabledAssertions;
                }
                else if (Array.isArray(enabledAssertions)) {
                    enabledAssertions.forEach(processAssertions(1));
                }
                else {
                    return thread.throwNewException('Ljava/lang/InternalError;', "Expected enableAssertions option to be a boolean or an array of strings.");
                }
                directives['java/lang/AssertionStatusDirectives/classes'] = util.newArrayFromData(thread, bsCl, '[Ljava/lang/String;', classes.map(function (cls) { return util.initString(bsCl, cls); }));
                directives['java/lang/AssertionStatusDirectives/classEnabled'] = util.newArrayFromData(thread, bsCl, '[Z', classEnabled);
                directives['java/lang/AssertionStatusDirectives/packages'] = util.newArrayFromData(thread, bsCl, '[Ljava/lang/String;', packages.map(function (pkg) { return util.initString(bsCl, pkg); }));
                directives['java/lang/AssertionStatusDirectives/packageEnabled'] = util.newArrayFromData(thread, bsCl, '[Z', packageEnabled);
                directives['java/lang/AssertionStatusDirectives/deflt'] = enabledAssertions ? 1 : 0;
                thread.asyncReturn(directives);
            });
        };
        return java_lang_ClassLoader;
    }());
    var java_lang_Compiler = (function () {
        function java_lang_Compiler() {
        }
        java_lang_Compiler['initialize()V'] = function (thread) {
            // NOP.
        };
        java_lang_Compiler['registerNatives()V'] = function (thread) {
            // NOP.
        };
        java_lang_Compiler['compileClass(Ljava/lang/Class;)Z'] = function (thread, arg0) {
            // Return false: No compiler available.
            return 0;
        };
        java_lang_Compiler['compileClasses(Ljava/lang/String;)Z'] = function (thread, arg0) {
            // Return false: No compiler available.
            return 0;
        };
        java_lang_Compiler['command(Ljava/lang/Object;)Ljava/lang/Object;'] = function (thread, arg0) {
            // Return null; no compiler available.
            return null;
        };
        // NOP'd.
        java_lang_Compiler['enable()V'] = function (thread) { };
        java_lang_Compiler['disable()V'] = function (thread) { };
        return java_lang_Compiler;
    }());
    // Used for converting between numerical representations.
    var conversionBuffer = new Buffer(8);
    var java_lang_Double = (function () {
        function java_lang_Double() {
        }
        java_lang_Double['doubleToRawLongBits(D)J'] = function (thread, num) {
            conversionBuffer.writeDoubleLE(num, 0);
            return Long.fromBits(conversionBuffer.readUInt32LE(0), conversionBuffer.readUInt32LE(4));
        };
        java_lang_Double['longBitsToDouble(J)D'] = function (thread, num) {
            conversionBuffer.writeInt32LE(num.getLowBits(), 0);
            conversionBuffer.writeInt32LE(num.getHighBits(), 4);
            return conversionBuffer.readDoubleLE(0);
        };
        return java_lang_Double;
    }());
    var java_lang_Float = (function () {
        function java_lang_Float() {
        }
        java_lang_Float['floatToRawIntBits(F)I'] = function (thread, num) {
            conversionBuffer.writeFloatLE(num, 0);
            return conversionBuffer.readInt32LE(0);
        };
        java_lang_Float['intBitsToFloat(I)F'] = function (thread, num) {
            conversionBuffer.writeInt32LE(num, 0);
            return conversionBuffer.readFloatLE(0);
        };
        return java_lang_Float;
    }());
    var java_lang_Object = (function () {
        function java_lang_Object() {
        }
        java_lang_Object['getClass()Ljava/lang/Class;'] = function (thread, javaThis) {
            return javaThis.getClass().getClassObject(thread);
        };
        java_lang_Object['hashCode()I'] = function (thread, javaThis) {
            return javaThis.ref;
        };
        java_lang_Object['clone()Ljava/lang/Object;'] = function (thread, javaThis) {
            var cls = javaThis.getClass();
            if (cls.getInternalName()[0] === '[') {
                // Array clone. It's always a shallow clone.
                return javaThis.slice(0);
            }
            else {
                var clonedObj = util.newObjectFromClass(thread, javaThis.getClass());
                Object.keys(javaThis).forEach(function (fieldName) {
                    clonedObj[fieldName] = javaThis[fieldName];
                });
                return clonedObj;
            }
        };
        java_lang_Object['notify()V'] = function (thread, javaThis) {
            debug("TE(notify): on lock *" + javaThis.ref);
            javaThis.getMonitor().notify(thread);
        };
        java_lang_Object['notifyAll()V'] = function (thread, javaThis) {
            debug("TE(notifyAll): on lock *" + javaThis.ref);
            javaThis.getMonitor().notifyAll(thread);
        };
        java_lang_Object['wait(J)V'] = function (thread, javaThis, timeout) {
            debug("TE(wait): on lock *" + javaThis.ref);
            javaThis.getMonitor().wait(thread, function (fromTimer) {
                thread.asyncReturn();
            }, timeout.toNumber());
        };
        return java_lang_Object;
    }());
    var java_lang_Package = (function () {
        function java_lang_Package() {
        }
        java_lang_Package['getSystemPackage0(Ljava/lang/String;)Ljava/lang/String;'] = function (thread, pkgNameObj) {
            var pkgName = pkgNameObj.toString();
            // Slice off ending /
            pkgName = pkgName.slice(0, pkgName.length - 1);
            var pkgs = thread.getBsCl().getPackages();
            for (var i = 0; i < pkgs.length; i++) {
                if (pkgs[i][0] === pkgName) {
                    // XXX: Ignore secondary load locations.
                    return util.initString(thread.getBsCl(), pkgs[i][1][0]);
                }
            }
            // Could not find package.
            return null;
        };
        java_lang_Package['getSystemPackages0()[Ljava/lang/String;'] = function (thread) {
            var pkgNames = thread.getBsCl().getPackages();
            // Note: We add / to end of package name, since it appears that is what OpenJDK expects.
            return util.newArrayFromData(thread, thread.getBsCl(), '[Ljava/lang/String;', pkgNames.map(function (pkgName) { return util.initString(thread.getBsCl(), pkgName[0] + "/"); }));
        };
        return java_lang_Package;
    }());
    var java_lang_ProcessEnvironment = (function () {
        function java_lang_ProcessEnvironment() {
        }
        java_lang_ProcessEnvironment['environ()[[B'] = function (thread) {
            var envArr = util.newArray(thread, thread.getBsCl(), '[[B', 0), env = process.env, key, v, bArr;
            // convert to an array of strings of the form [key, value, key, value ...]
            for (key in env) {
                v = env[key];
                bArr = util.newArray(thread, thread.getBsCl(), '[B', 0);
                bArr.array = util.bytestr2Array(key);
                envArr.array.push(bArr);
                bArr = util.newArray(thread, thread.getBsCl(), '[B', 0);
                bArr.array = util.bytestr2Array(v);
                envArr.array.push(bArr);
            }
            return envArr;
        };
        return java_lang_ProcessEnvironment;
    }());
    var java_lang_reflect_Array = (function () {
        function java_lang_reflect_Array() {
        }
        java_lang_reflect_Array['getLength(Ljava/lang/Object;)I'] = function (thread, arr) {
            if (verifyArray(thread, arr)) {
                if (isNotNull(thread, arr)) {
                    return arr.array.length;
                }
            }
        };
        java_lang_reflect_Array['get(Ljava/lang/Object;I)Ljava/lang/Object;'] = function (thread, arr, idx) {
            var val = arrayGet(thread, arr, idx);
            if (val != null) {
                var component = arr.getClass().getComponentClass();
                if (util.is_primitive_type(component.getInternalName())) {
                    // Box primitive values.
                    return component.createWrapperObject(thread, val);
                }
            }
            return val;
        };
        java_lang_reflect_Array['set(Ljava/lang/Object;ILjava/lang/Object;)V'] = function (thread, arr, idx, val) {
            if (verifyArray(thread, arr) && isNotNull(thread, arr)) {
                if (idx < 0 || idx >= arr.array.length) {
                    thread.throwNewException('Ljava/lang/ArrayIndexOutOfBoundsException;', 'Tried to write to an illegal index in an array.');
                }
                else {
                    var ccls = arr.getClass().getComponentClass();
                    if (ccls instanceof PrimitiveClassData) {
                        if (val.getClass().isSubclass(thread.getBsCl().getInitializedClass(thread, ccls.boxClassName()))) {
                            var ccname = ccls.getInternalName();
                            val[(util.internal2external[ccname] + "Value()" + ccname)](thread, null, function (e, rv) {
                                if (e) {
                                    thread.throwException(e);
                                }
                                else {
                                    arr.array[idx] = rv;
                                    thread.asyncReturn();
                                }
                            });
                        }
                        else {
                            thread.throwNewException('Ljava/lang/IllegalArgumentException;', 'argument type mismatch');
                        }
                    }
                    else if (val.getClass().isSubclass(ccls)) {
                        arr.array[idx] = val;
                    }
                    else {
                        thread.throwNewException('Ljava/lang/IllegalArgumentException;', 'argument type mismatch');
                    }
                }
            }
        };
        java_lang_reflect_Array['setBoolean(Ljava/lang/Object;IZ)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setByte(Ljava/lang/Object;IB)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setChar(Ljava/lang/Object;IC)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setShort(Ljava/lang/Object;IS)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setInt(Ljava/lang/Object;II)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setLong(Ljava/lang/Object;IJ)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setFloat(Ljava/lang/Object;IF)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['setDouble(Ljava/lang/Object;ID)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_reflect_Array['newArray(Ljava/lang/Class;I)Ljava/lang/Object;'] = function (thread, cls, len) {
            return util.newArray(thread, cls.$cls.getLoader(), "[" + cls.$cls.getInternalName(), len);
        };
        java_lang_reflect_Array['multiNewArray(Ljava/lang/Class;[I)Ljava/lang/Object;'] = function (thread, jco, lens) {
            var typeStr = (new Array(lens.array.length + 1)).join('[') + jco.$cls.getInternalName();
            if (jco.$cls.isInitialized(thread)) {
                return util.multiNewArray(thread, jco.$cls.getLoader(), typeStr, lens.array);
            }
            else {
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                jco.$cls.initialize(thread, function (cls) {
                    thread.asyncReturn(util.multiNewArray(thread, jco.$cls.getLoader(), typeStr, lens.array));
                });
            }
        };
        java_lang_reflect_Array['getBoolean(Ljava/lang/Object;I)Z'] = arrayGet;
        java_lang_reflect_Array['getByte(Ljava/lang/Object;I)B'] = arrayGet;
        java_lang_reflect_Array['getChar(Ljava/lang/Object;I)C'] = arrayGet;
        java_lang_reflect_Array['getShort(Ljava/lang/Object;I)S'] = arrayGet;
        java_lang_reflect_Array['getInt(Ljava/lang/Object;I)I'] = arrayGet;
        java_lang_reflect_Array['getLong(Ljava/lang/Object;I)J'] = arrayGet;
        java_lang_reflect_Array['getFloat(Ljava/lang/Object;I)F'] = arrayGet;
        java_lang_reflect_Array['getDouble(Ljava/lang/Object;I)D'] = arrayGet;
        return java_lang_reflect_Array;
    }());
    var java_lang_reflect_Proxy = (function () {
        function java_lang_reflect_Proxy() {
        }
        java_lang_reflect_Proxy['defineClass0(Ljava/lang/ClassLoader;Ljava/lang/String;[BII)Ljava/lang/Class;'] = function (thread, cl, name, bytes, offset, len) {
            var loader = util.getLoader(thread, cl), cls = loader.defineClass(thread, util.int_classname(name.toString()), util.byteArray2Buffer(bytes.array, offset, len), null);
            if (cls != null) {
                return cls.getClassObject(thread);
            }
        };
        return java_lang_reflect_Proxy;
    }());
    var java_lang_Runtime = (function () {
        function java_lang_Runtime() {
        }
        java_lang_Runtime['availableProcessors()I'] = function (thread, javaThis) {
            return 1;
        };
        java_lang_Runtime['freeMemory()J'] = function (thread, javaThis) {
            return Long.MAX_VALUE;
        };
        java_lang_Runtime['totalMemory()J'] = function (thread, javaThis) {
            return Long.MAX_VALUE;
        };
        /**
         * Returns the maximum amount of memory that the Java Virtual Machine will
         * attempt to use, in bytes, as a Long. If there is no inherent limit then the
         * value Long.MAX_VALUE will be returned.
         *
         * Currently returns Long.MAX_VALUE because unlike other JVMs Doppio has no
         * hard limit on the heap size.
         */
        java_lang_Runtime['maxMemory()J'] = function (thread, javaThis) {
            return Long.MAX_VALUE;
        };
        /**
         * No universal way of forcing browser to GC, so we yield in hopes
         * that the browser will use it as an opportunity to GC.
         */
        java_lang_Runtime['gc()V'] = function (thread, javaThis) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            setImmediate(function () {
                thread.asyncReturn();
            });
        };
        java_lang_Runtime['runFinalization0()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_Runtime['traceInstructions(Z)V'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_Runtime['traceMethodCalls(Z)V'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_lang_Runtime;
    }());
    var java_lang_SecurityManager = (function () {
        function java_lang_SecurityManager() {
        }
        java_lang_SecurityManager['getClassContext()[Ljava/lang/Class;'] = function (thread, javaThis) {
            // return an array of classes for each method on the stack
            // starting with the current method and going up the call chain
            return util.newArrayFromData(thread, thread.getBsCl(), '[Ljava/lang/Class;', thread.getStackTrace().map(function (item) { return item.method.cls.getClassObject(thread); }));
            ;
        };
        java_lang_SecurityManager['currentClassLoader0()Ljava/lang/ClassLoader;'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        java_lang_SecurityManager['classDepth(Ljava/lang/String;)I'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return 0;
        };
        java_lang_SecurityManager['classLoaderDepth0()I'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return 0;
        };
        java_lang_SecurityManager['currentLoadedClass0()Ljava/lang/Class;'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        return java_lang_SecurityManager;
    }());
    var java_lang_Shutdown = (function () {
        function java_lang_Shutdown() {
        }
        java_lang_Shutdown['halt0(I)V'] = function (thread, status) {
            thread.getJVM().halt(status);
        };
        java_lang_Shutdown['runAllFinalizers()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_lang_Shutdown;
    }());
    var java_lang_StrictMath = (function () {
        function java_lang_StrictMath() {
        }
        java_lang_StrictMath['sin(D)D'] = function (thread, d_val) {
            return Math.sin(d_val);
        };
        java_lang_StrictMath['cos(D)D'] = function (thread, d_val) {
            return Math.cos(d_val);
        };
        java_lang_StrictMath['tan(D)D'] = function (thread, d_val) {
            return Math.tan(d_val);
        };
        java_lang_StrictMath['asin(D)D'] = function (thread, d_val) {
            return Math.asin(d_val);
        };
        java_lang_StrictMath['acos(D)D'] = function (thread, d_val) {
            return Math.acos(d_val);
        };
        java_lang_StrictMath['atan(D)D'] = function (thread, d_val) {
            return Math.atan(d_val);
        };
        java_lang_StrictMath['exp(D)D'] = function (thread, d_val) {
            return Math.exp(d_val);
        };
        java_lang_StrictMath['log(D)D'] = function (thread, d_val) {
            return Math.log(d_val);
        };
        java_lang_StrictMath['log10(D)D'] = function (thread, d_val) {
            return Math.log(d_val) / Math.LN10;
        };
        java_lang_StrictMath['sqrt(D)D'] = function (thread, d_val) {
            return Math.sqrt(d_val);
        };
        java_lang_StrictMath['cbrt(D)D'] = function (thread, d_val) {
            var is_neg = d_val < 0;
            if (is_neg) {
                return -Math.pow(-d_val, 1 / 3);
            }
            else {
                return Math.pow(d_val, 1 / 3);
            }
        };
        java_lang_StrictMath['IEEEremainder(DD)D'] = function (thread, x, y) {
            // Purge off exception values.
            if (x == Number.NEGATIVE_INFINITY || !(x < Number.POSITIVE_INFINITY)
                || y == 0 || y != y)
                return Number.NaN;
            var TWO_1023 = 8.98846567431158e307; // Long bits 0x7fe0000000000000L.
            var negative = x < 0;
            x = Math.abs(x);
            y = Math.abs(y);
            if (x == y || x == 0)
                return 0 * x; // Get correct sign.
            // Achieve x < 2y, then take first shot at remainder.
            if (y < TWO_1023)
                x %= y + y;
            // Now adjust x to get correct precision.
            if (y < 4 / TWO_1023) {
                if (x + x > y) {
                    x -= y;
                    if (x + x >= y)
                        x -= y;
                }
            }
            else {
                y *= 0.5;
                if (x > y) {
                    x -= y;
                    if (x >= y)
                        x -= y;
                }
            }
            return negative ? -x : x;
        };
        java_lang_StrictMath['atan2(DD)D'] = function (thread, y, x) {
            return Math.atan2(y, x);
        };
        java_lang_StrictMath['pow(DD)D'] = function (thread, base, exp) {
            return Math.pow(base, exp);
        };
        java_lang_StrictMath['sinh(D)D'] = function (thread, d_val) {
            return Math.sinh(d_val);
        };
        java_lang_StrictMath['cosh(D)D'] = function (thread, d_val) {
            var exp = Math.exp(d_val);
            return (exp + 1 / exp) / 2;
        };
        java_lang_StrictMath['tanh(D)D'] = function (thread, d_val) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return 0;
        };
        java_lang_StrictMath['hypot(DD)D'] = function (thread, arg0, arg1) {
            return Math.sqrt(Math.pow(arg0, 2) + Math.pow(arg1, 2));
        };
        java_lang_StrictMath['expm1(D)D'] = function (thread, d_val) {
            return Math.expm1(d_val);
        };
        java_lang_StrictMath['log1p(D)D'] = function (thread, d_val) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return 0;
        };
        return java_lang_StrictMath;
    }());
    var java_lang_String = (function () {
        function java_lang_String() {
        }
        java_lang_String['intern()Ljava/lang/String;'] = function (thread, javaThis) {
            return thread.getJVM().internString(javaThis.toString(), javaThis);
        };
        return java_lang_String;
    }());
    var java_lang_System = (function () {
        function java_lang_System() {
        }
        java_lang_System['setIn0(Ljava/io/InputStream;)V'] = function (thread, stream) {
            var sys = util.getStaticFields(thread, thread.getBsCl(), 'Ljava/lang/System;');
            sys['java/lang/System/in'] = stream;
        };
        java_lang_System['setOut0(Ljava/io/PrintStream;)V'] = function (thread, stream) {
            var sys = util.getStaticFields(thread, thread.getBsCl(), 'Ljava/lang/System;');
            sys['java/lang/System/out'] = stream;
        };
        java_lang_System['setErr0(Ljava/io/PrintStream;)V'] = function (thread, stream) {
            var sys = util.getStaticFields(thread, thread.getBsCl(), 'Ljava/lang/System;');
            sys['java/lang/System/err'] = stream;
        };
        java_lang_System['currentTimeMillis()J'] = function (thread) {
            return Long.fromNumber((new Date).getTime());
        };
        /**
         * @todo Use performance.now() if available.
         */
        java_lang_System['nanoTime()J'] = function (thread) {
            return Long.fromNumber((new Date).getTime()).multiply(Long.fromNumber(1000000));
        };
        java_lang_System['arraycopy(Ljava/lang/Object;ILjava/lang/Object;II)V'] = function (thread, src, srcPos, dest, destPos, length) {
            // Needs to be checked *even if length is 0*.
            if ((src == null) || (dest == null)) {
                thread.throwNewException('Ljava/lang/NullPointerException;', 'Cannot copy to/from a null array.');
            }
            else if (!(src.getClass() instanceof ArrayClassData) || !(dest.getClass() instanceof ArrayClassData)) {
                thread.throwNewException('Ljava/lang/ArrayStoreException;', 'src and dest arguments must be of array type.');
            }
            else if (srcPos < 0 || (srcPos + length) > src.array.length || destPos < 0 || (destPos + length) > dest.array.length || length < 0) {
                // System.arraycopy requires IndexOutOfBoundsException, but Java throws an array variant of the exception in practice.
                thread.throwNewException('Ljava/lang/ArrayIndexOutOfBoundsException;', 'Tried to write to an illegal index in an array.');
            }
            else {
                var srcClass = src.getClass(), destClass = dest.getClass();
                // Special case; need to copy the section of src that is being copied into a temporary array before actually doing the copy.
                if (src === dest) {
                    src = dest.slice(srcPos, srcPos + length);
                    srcPos = 0;
                }
                if (srcClass.isCastable(destClass)) {
                    // Fast path
                    util.arraycopyNoCheck(src, srcPos, dest, destPos, length);
                }
                else {
                    // Slow path
                    // Absolutely cannot do this when two different primitive types, or a primitive type and a reference type.
                    var srcCompCls = src.getClass().getComponentClass(), destCompCls = dest.getClass().getComponentClass();
                    if ((srcCompCls instanceof PrimitiveClassData) || (destCompCls instanceof PrimitiveClassData)) {
                        thread.throwNewException('Ljava/lang/ArrayStoreException;', 'If calling arraycopy with a primitive array, both src and dest must be of the same primitive type.');
                    }
                    else {
                        // Must be two reference types.
                        util.arraycopyCheck(thread, src, srcPos, dest, destPos, length);
                    }
                }
            }
        };
        java_lang_System['identityHashCode(Ljava/lang/Object;)I'] = function (thread, x) {
            if (x != null && x.ref != null) {
                return x.ref;
            }
            return 0;
        };
        java_lang_System['initProperties(Ljava/util/Properties;)Ljava/util/Properties;'] = function (thread, props) {
            var jvm = thread.getJVM(), properties = jvm.getSystemPropertyNames();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            util.asyncForEach(properties, function (propertyName, nextItem) {
                var propertyVal = jvm.getSystemProperty(propertyName);
                props["setProperty(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/Object;"](thread, [jvm.internString(propertyName), jvm.internString(propertyVal)], nextItem);
            }, function (err) {
                if (err) {
                    thread.throwException(err);
                }
                else {
                    thread.asyncReturn(props);
                }
            });
        };
        java_lang_System['mapLibraryName(Ljava/lang/String;)Ljava/lang/String;'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        return java_lang_System;
    }());
    var java_lang_Thread = (function () {
        function java_lang_Thread() {
        }
        java_lang_Thread['currentThread()Ljava/lang/Thread;'] = function (thread) {
            return thread.getJVMObject();
        };
        java_lang_Thread['yield()V'] = function (thread) {
            // Force the thread scheduler to pick another thread by waiting for a short
            // amount of time.
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            setImmediate(function () {
                thread.setStatus(ThreadStatus.RUNNABLE);
                thread.asyncReturn();
            });
        };
        java_lang_Thread['sleep(J)V'] = function (thread, millis) {
            var beforeMethod = thread.currentMethod();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            setTimeout(function () {
                // Check if the thread was interrupted during our sleep. Interrupting
                // sleep causes an exception, so we need to ignore the setTimeout
                // callback in this case.
                if (beforeMethod === thread.currentMethod()) {
                    thread.setStatus(ThreadStatus.RUNNABLE);
                    thread.asyncReturn();
                }
            }, millis.toNumber());
        };
        java_lang_Thread['start0()V'] = function (thread, javaThis) {
            javaThis['run()V'](javaThis.$thread, null);
        };
        java_lang_Thread['setNativeName(Ljava/lang/String;)V'] = function (thread, javaThis, name) {
            // NOP. No need to do anything.
        };
        java_lang_Thread['isInterrupted(Z)Z'] = function (thread, javaThis, clearFlag) {
            var isInterrupted = javaThis.$thread.isInterrupted();
            if (clearFlag) {
                javaThis.$thread.setInterrupted(false);
            }
            return isInterrupted;
        };
        java_lang_Thread['isAlive()Z'] = function (thread, javaThis) {
            var state = javaThis.$thread.getStatus();
            return state !== ThreadStatus.TERMINATED && state !== ThreadStatus.NEW;
        };
        java_lang_Thread['countStackFrames()I'] = function (thread, javaThis) {
            return javaThis.$thread.getStackTrace().length;
        };
        java_lang_Thread['holdsLock(Ljava/lang/Object;)Z'] = function (thread, obj) {
            var mon = obj.getMonitor();
            return mon.getOwner() === thread;
        };
        java_lang_Thread['dumpThreads([Ljava/lang/Thread;)[[Ljava/lang/StackTraceElement;'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        java_lang_Thread['getThreads()[Ljava/lang/Thread;'] = function (thread) {
            return util.newArrayFromData(thread, thread.getBsCl(), '[Ljava/lang/Thread;', thread.getThreadPool().getThreads().map(function (thread) { return thread.getJVMObject(); }));
        };
        java_lang_Thread['setPriority0(I)V'] = function (thread, javaThis, arg0) {
            thread.signalPriorityChange();
        };
        java_lang_Thread['stop0(Ljava/lang/Object;)V'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_Thread['suspend0()V'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_Thread['resume0()V'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        /**
         * Interrupts this thread.
         *
         * Unless the current thread is interrupting itself, which is always
         * permitted, the checkAccess method of this thread is invoked, which may
         * cause a SecurityException to be thrown.
         *
         * - If this thread is blocked in an invocation of the Object.wait(),
         *   wait(long), or Object.wait(long,int) methods of the Object class, or of
         *   the join(), join(long), join(long,int), sleep(long), or sleep(long,int),
         *   methods of this class, then its interrupt status will be cleared and it
         *   will receive an InterruptedException.
         *
         * - If this thread is blocked in an I/O operation upon an
         *   java.nio.channels.InterruptibleChannel then the channel will be closed,
         *   the thread's interrupt status will be set, and the thread will receive a
         *   java.nio.channels.ClosedByInterruptException.
         *
         * - If this thread is blocked in a java.nio.channels.Selector then the
         *   thread's interrupt status will be set and it will return immediately from
         *   the selection operation, possibly with a non-zero value, just as if the
         *   selector's java.nio.channels.Selector.wakeup() method were invoked.
         *
         * - If none of the previous conditions hold then this thread's interrupt
         *   status will be set.
         *
         * Interrupting a thread that is not alive need not have any effect.
         */
        java_lang_Thread['interrupt0()V'] = function (thread, javaThis) {
            function throwInterruptedException() {
                javaThis.$thread.throwNewException('Ljava/lang/InterruptedException;', 'interrupt0 called');
            }
            var nativeThreadObj = javaThis.$thread;
            // See if we have access to modify this thread.
            javaThis['checkAccess()V'](thread, null, function (e) {
                if (e) {
                    // SecurityException. Rethrow it.
                    thread.throwException(e);
                }
                else {
                    // Check if thread is alive.
                    var status = nativeThreadObj.getStatus();
                    switch (status) {
                        case ThreadStatus.NEW:
                        case ThreadStatus.TERMINATED:
                            // Thread is not alive. NOP.
                            return thread.asyncReturn();
                        case ThreadStatus.BLOCKED:
                        case ThreadStatus.WAITING:
                        case ThreadStatus.TIMED_WAITING:
                            // Thread is waiting or blocked on a monitor. Clear interrupted
                            // status, and throw an interrupted exception.
                            nativeThreadObj.setInterrupted(false);
                            // Exit the monitor.
                            var monitor = nativeThreadObj.getMonitorBlock();
                            if (status === ThreadStatus.BLOCKED) {
                                monitor.unblock(nativeThreadObj, true);
                                throwInterruptedException();
                            }
                            else {
                                monitor.unwait(nativeThreadObj, false, true, throwInterruptedException);
                            }
                            return thread.asyncReturn();
                        case ThreadStatus.PARKED:
                            // Parked threads become unparked when interrupted.
                            thread.getJVM().getParker().completelyUnpark(nativeThreadObj);
                        // FALL-THROUGH
                        default:
                            var threadCls = thread.getBsCl().getInitializedClass(thread, 'Ljava/lang/Thread;'), 
                            // If we are in the following methods, we throw an InterruptedException:
                            interruptMethods = [
                                threadCls.methodLookup('join()V'),
                                threadCls.methodLookup('join(J)V'),
                                threadCls.methodLookup('join(JI)V'),
                                threadCls.methodLookup('sleep(J)V'),
                                threadCls.methodLookup('sleep(JI)V') // * Thread.sleep(long, int)
                            ], stackTrace = nativeThreadObj.getStackTrace(), currentMethod = stackTrace[stackTrace.length - 1].method;
                            if (interruptMethods.indexOf(currentMethod) !== -1) {
                                // Clear interrupt state before throwing the exception.
                                nativeThreadObj.setInterrupted(false);
                                nativeThreadObj.throwNewException('Ljava/lang/InterruptedException;', 'interrupt0 called');
                            }
                            else {
                                // Set the interrupted status.
                                nativeThreadObj.setInterrupted(true);
                            }
                            return thread.asyncReturn();
                    }
                }
            });
        };
        return java_lang_Thread;
    }());
    var java_lang_Throwable = (function () {
        function java_lang_Throwable() {
        }
        /**
         * NOTE: Integer is only there to distinguish this function from non-native fillInStackTrace()V.
         */
        java_lang_Throwable['fillInStackTrace(I)Ljava/lang/Throwable;'] = function (thread, javaThis, dummy) {
            var stackTraceElementCls = thread.getBsCl().getInitializedClass(thread, 'Ljava/lang/StackTraceElement;'), stacktrace = util.newArray(thread, thread.getBsCl(), '[Ljava/lang/StackTraceElement;', 0), cstack = thread.getStackTrace(), i, j, bsCl = thread.getBsCl();
            /**
             * OK, so we need to toss the following stack frames:
             * - The stack frame for this method.
             * - If we're still constructing the throwable object, we need to toss any
             *   stack frames involved in constructing the throwable. But if we're not,
             *   then there's no other frames we should cut.
             */
            cstack.pop(); // The stack frame for this method.
            // Bytecode methods involved in constructing the throwable. We assume that
            // there are no native methods involved in the mix other than this one.
            while (cstack.length > 0 &&
                !cstack[cstack.length - 1].method.accessFlags.isNative() &&
                cstack[cstack.length - 1].locals[0] === javaThis) {
                cstack.pop();
            }
            // Construct the stack such that the method on top of the stack is at index
            // 0.
            for (i = cstack.length - 1; i >= 0; i--) {
                var sf = cstack[i], cls = sf.method.cls, ln = -1, sourceFile;
                // Java 8: Ignore 'Hidden' methods. These are involved in constructing
                // Lambdas, and shouldn't be use-visible.
                if (sf.method.isHidden()) {
                    continue;
                }
                if (sf.method.accessFlags.isNative()) {
                    sourceFile = 'Native Method';
                }
                else {
                    var srcAttr = cls.getAttribute('SourceFile'), code = sf.method.getCodeAttribute(), table = code.getAttribute('LineNumberTable');
                    sourceFile = (srcAttr != null) ? srcAttr.filename : 'unknown';
                    if (table != null) {
                        ln = table.getLineNumber(sf.pc);
                    }
                    else {
                        ln = -1;
                    }
                }
                var newElement = util.newObjectFromClass(thread, stackTraceElementCls);
                newElement['java/lang/StackTraceElement/declaringClass'] = util.initString(bsCl, util.ext_classname(cls.getInternalName()));
                newElement['java/lang/StackTraceElement/methodName'] = util.initString(bsCl, sf.method.name != null ? sf.method.name : 'unknown');
                newElement['java/lang/StackTraceElement/fileName'] = util.initString(bsCl, sourceFile);
                newElement['java/lang/StackTraceElement/lineNumber'] = ln;
                stacktrace.array.push(newElement);
            }
            javaThis['java/lang/Throwable/backtrace'] = stacktrace;
            return javaThis;
        };
        java_lang_Throwable['getStackTraceDepth()I'] = function (thread, javaThis) {
            // 'backtrace' is typed as an Object so JVMs have flexibility in what to store there.
            // We simply store the stack trace element array.
            return javaThis['java/lang/Throwable/backtrace'].array.length;
        };
        java_lang_Throwable['getStackTraceElement(I)Ljava/lang/StackTraceElement;'] = function (thread, javaThis, depth) {
            return javaThis['java/lang/Throwable/backtrace'].array[depth];
        };
        return java_lang_Throwable;
    }());
    var java_lang_UNIXProcess = (function () {
        function java_lang_UNIXProcess() {
        }
        java_lang_UNIXProcess['waitForProcessExit(I)I'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return 0;
        };
        java_lang_UNIXProcess['forkAndExec(I[B[B[BI[BI[B[IZ)I'] = function (thread, javaThis) {
            thread.throwNewException('Ljava/lang/Error;', "Doppio doesn't support forking processes.");
        };
        java_lang_UNIXProcess['destroyProcess(IZ)V'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        java_lang_UNIXProcess['init()V'] = function (thread) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        return java_lang_UNIXProcess;
    }());
    /**
     * Misc. MemberName-specific constants, enum'd so they get inlined.
     */
    var MemberNameConstants;
    (function (MemberNameConstants) {
        /* Bit masks for FLAGS for particular types */
        MemberNameConstants[MemberNameConstants["IS_METHOD"] = 65536] = "IS_METHOD";
        MemberNameConstants[MemberNameConstants["IS_CONSTRUCTOR"] = 131072] = "IS_CONSTRUCTOR";
        MemberNameConstants[MemberNameConstants["IS_FIELD"] = 262144] = "IS_FIELD";
        MemberNameConstants[MemberNameConstants["IS_TYPE"] = 524288] = "IS_TYPE";
        MemberNameConstants[MemberNameConstants["CALLER_SENSITIVE"] = 1048576] = "CALLER_SENSITIVE";
        /* Passed in in matchFlags argument of MHN.getMembers */
        MemberNameConstants[MemberNameConstants["SEARCH_SUPERCLASSES"] = 1048576] = "SEARCH_SUPERCLASSES";
        MemberNameConstants[MemberNameConstants["SEARCH_INTERFACES"] = 2097152] = "SEARCH_INTERFACES";
        /* Number of bits to shift over the reference kind into the MN's flags. */
        MemberNameConstants[MemberNameConstants["REFERENCE_KIND_SHIFT"] = 24] = "REFERENCE_KIND_SHIFT";
        /* Mask to extract member type. */
        MemberNameConstants[MemberNameConstants["ALL_KINDS"] = 983040] = "ALL_KINDS";
    })(MemberNameConstants || (MemberNameConstants = {}));
    /**
     * Given a MemberName object and a reflective field/method/constructor,
     * initializes the member name:
     * - name: Name of the field/method.
     * - clazz: Referenced class that contains the method.
     * - flags: Encodes the reference type of the member and the member's access flags.
     * - type: String encoding of the type (method descriptor, or class name of field type in descriptor form)
     * - vmtarget: Contains the VM-specific pointer to the member (in our case, a Method or Field object)
     * (set clazz, updates flags, sets vmtarget).
     */
    function initializeMemberName(thread, mn, ref) {
        var flags = mn['java/lang/invoke/MemberName/flags'], type = mn['java/lang/invoke/MemberName/type'], name = mn['java/lang/invoke/MemberName/name'], refKind, existingRefKind = flags >>> MemberNameConstants.REFERENCE_KIND_SHIFT;
        // Determine the reference type.
        if (ref instanceof Method) {
            flags = MemberNameConstants.IS_METHOD;
            if (ref.cls.accessFlags.isInterface()) {
                refKind = MethodHandleReferenceKind.INVOKEINTERFACE;
            }
            else if (ref.accessFlags.isStatic()) {
                refKind = MethodHandleReferenceKind.INVOKESTATIC;
            }
            else if (ref.name[0] === '<') {
                flags = MemberNameConstants.IS_CONSTRUCTOR;
                refKind = MethodHandleReferenceKind.INVOKESPECIAL;
            }
            else {
                refKind = MethodHandleReferenceKind.INVOKEVIRTUAL;
            }
            mn.vmtarget = ref.getVMTargetBridgeMethod(thread, existingRefKind ? existingRefKind : refKind);
            if (refKind === MethodHandleReferenceKind.INVOKEINTERFACE ||
                refKind === MethodHandleReferenceKind.INVOKEVIRTUAL) {
                mn.vmindex = ref.cls.getVMIndexForMethod(ref);
            }
            flags |= (refKind << MemberNameConstants.REFERENCE_KIND_SHIFT) | methodFlags(ref);
        }
        else {
            flags = MemberNameConstants.IS_FIELD;
            // Assume a GET.
            if (ref.accessFlags.isStatic()) {
                refKind = MethodHandleReferenceKind.GETSTATIC;
            }
            else {
                refKind = MethodHandleReferenceKind.GETFIELD;
            }
            mn.vmindex = ref.cls.getVMIndexForField(ref);
            flags |= (refKind << MemberNameConstants.REFERENCE_KIND_SHIFT) | ref.accessFlags.getRawByte();
        }
        // Initialize type if we need to.
        if (type === null) {
            type = thread.getJVM().internString(ref.rawDescriptor);
        }
        // Initialize name if we need to.
        if (name === null) {
            name = thread.getJVM().internString(ref.name);
        }
        mn['java/lang/invoke/MemberName/clazz'] = ref.cls.getClassObject(thread);
        mn['java/lang/invoke/MemberName/flags'] = flags;
        mn['java/lang/invoke/MemberName/type'] = type;
        mn['java/lang/invoke/MemberName/name'] = name;
    }
    /**
     * Returns the MemberName flags for the given method.
     */
    function methodFlags(method) {
        var flags = method.accessFlags.getRawByte();
        if (method.isCallerSensitive()) {
            flags |= MemberNameConstants.CALLER_SENSITIVE;
        }
        return flags;
    }
    var java_lang_invoke_MethodHandleNatives = (function () {
        function java_lang_invoke_MethodHandleNatives() {
        }
        /**
         * I'm going by JAMVM's implementation of this method, which is very easy
         * to understand:
         * http://sourceforge.net/p/jamvm/code/ci/master/tree/src/classlib/openjdk/mh.c#l388
         *
         * The second argument is a Reflection object for the specified member,
         * which is either a Field, Method, or Constructor.
         *
         * We need to:
         * * Set "clazz" field to item's declaring class in the reflection object.
         * * Set "flags" field to items's flags, OR'd with its type (method/field/
         *   constructor), and OR'd with its reference kind shifted up by 24.
         * * Set "vmtarget" if relevant.
         * * Set "vmindex" if relevant.
         *
         * This method "resolves" the MemberName unambiguously using the provided
         * reflection object.
         *
         */
        java_lang_invoke_MethodHandleNatives['init(Ljava/lang/invoke/MemberName;Ljava/lang/Object;)V'] = function (thread, self, ref) {
            var clazz, clazzData, flags, m, f;
            switch (ref.getClass().getInternalName()) {
                case "Ljava/lang/reflect/Method;":
                    var methodObj = ref, refKind;
                    clazz = methodObj['java/lang/reflect/Method/clazz'];
                    clazzData = clazz.$cls;
                    m = clazzData.getMethodFromSlot(methodObj['java/lang/reflect/Method/slot']);
                    flags = methodFlags(m) | MemberNameConstants.IS_METHOD;
                    if (m.accessFlags.isStatic()) {
                        refKind = MethodHandleReferenceKind.INVOKESTATIC;
                    }
                    else if (clazzData.accessFlags.isInterface()) {
                        refKind = MethodHandleReferenceKind.INVOKEINTERFACE;
                    }
                    else {
                        refKind = MethodHandleReferenceKind.INVOKEVIRTUAL;
                    }
                    flags |= refKind << MemberNameConstants.REFERENCE_KIND_SHIFT;
                    self['java/lang/invoke/MemberName/clazz'] = clazz;
                    self['java/lang/invoke/MemberName/flags'] = flags;
                    self.vmtarget = m.getVMTargetBridgeMethod(thread, refKind);
                    // Only set vmindex for virtual dispatch.
                    if (refKind === MethodHandleReferenceKind.INVOKEVIRTUAL || refKind === MethodHandleReferenceKind.INVOKEINTERFACE) {
                        self.vmindex = clazzData.getVMIndexForMethod(m);
                    }
                    break;
                case "Ljava/lang/reflect/Constructor;":
                    var consObj = ref;
                    clazz = consObj['java/lang/reflect/Constructor/clazz'];
                    clazzData = clazz.$cls;
                    m = clazzData.getMethodFromSlot(consObj['java/lang/reflect/Constructor/slot']);
                    flags = methodFlags(m) | MemberNameConstants.IS_CONSTRUCTOR | (MethodHandleReferenceKind.INVOKESPECIAL << MemberNameConstants.REFERENCE_KIND_SHIFT);
                    self['java/lang/invoke/MemberName/clazz'] = clazz;
                    self['java/lang/invoke/MemberName/flags'] = flags;
                    self.vmtarget = m.getVMTargetBridgeMethod(thread, refKind);
                    // vmindex not relevant; nonvirtual dispatch.
                    break;
                case "Ljava/lang/reflect/Field;":
                    var fieldObj = ref;
                    clazz = fieldObj['java/lang/reflect/Field/clazz'];
                    clazzData = clazz.$cls;
                    f = clazzData.getFieldFromSlot(fieldObj['java/lang/reflect/Field/slot']);
                    flags = f.accessFlags.getRawByte() | MemberNameConstants.IS_FIELD;
                    flags |= (f.accessFlags.isStatic() ? MethodHandleReferenceKind.GETSTATIC : MethodHandleReferenceKind.GETFIELD) << MemberNameConstants.REFERENCE_KIND_SHIFT;
                    self['java/lang/invoke/MemberName/clazz'] = clazz;
                    self['java/lang/invoke/MemberName/flags'] = flags;
                    self.vmindex = clazzData.getVMIndexForField(f);
                    // vmtarget not relevant.
                    break;
                default:
                    thread.throwNewException("Ljava/lang/InternalError;", "init: Invalid target.");
                    break;
            }
        };
        java_lang_invoke_MethodHandleNatives['getConstant(I)I'] = function (thread, arg0) {
            // I have no idea what the semantics are, but returning 0 disables some internal MH-related counting.
            return 0;
        };
        /**
         * I'm going by JAMVM's implementation of resolve:
         * http://sourceforge.net/p/jamvm/code/ci/master/tree/src/classlib/openjdk/mh.c#l1266
         * @todo It doesn't do anything with the lookupClass... is that for permission checks?
         *
         * Input: A MemberName object that already has a name, reference kind, and class set.
         * Uses that info to resolve a concrete method, and then updates the MemberName's flags,
         * sets "vmtarget", and sets "vmindex".
         */
        java_lang_invoke_MethodHandleNatives['resolve(Ljava/lang/invoke/MemberName;Ljava/lang/Class;)Ljava/lang/invoke/MemberName;'] = function (thread, memberName, lookupClass) {
            var type = memberName['java/lang/invoke/MemberName/type'], name = memberName['java/lang/invoke/MemberName/name'].toString(), clazz = memberName['java/lang/invoke/MemberName/clazz'].$cls, flags = memberName['java/lang/invoke/MemberName/flags'], refKind = flags >>> MemberNameConstants.REFERENCE_KIND_SHIFT;
            if (clazz == null || name == null || type == null) {
                thread.throwNewException("Ljava/lang/IllegalArgumentException;", "Invalid MemberName.");
                return;
            }
            assert((flags & MemberNameConstants.CALLER_SENSITIVE) === 0, "Not yet supported: Caller sensitive methods.");
            switch (flags & MemberNameConstants.ALL_KINDS) {
                case MemberNameConstants.IS_CONSTRUCTOR:
                case MemberNameConstants.IS_METHOD:
                    // Need to perform method lookup.
                    var methodTarget = clazz.signaturePolymorphicAwareMethodLookup(name + type.toString());
                    if (methodTarget !== null) {
                        flags |= methodFlags(methodTarget);
                        memberName['java/lang/invoke/MemberName/flags'] = flags;
                        memberName.vmtarget = methodTarget.getVMTargetBridgeMethod(thread, flags >>> MemberNameConstants.REFERENCE_KIND_SHIFT);
                        // vmindex is only relevant for virtual dispatch.
                        if (refKind === MethodHandleReferenceKind.INVOKEINTERFACE || refKind === MethodHandleReferenceKind.INVOKEVIRTUAL) {
                            memberName.vmindex = clazz.getVMIndexForMethod(methodTarget);
                        }
                        return memberName;
                    }
                    else {
                        thread.throwNewException('Ljava/lang/NoSuchMethodError;', "Invalid method " + (name + type.toString()) + " in class " + clazz.getExternalName() + ".");
                    }
                    break;
                case MemberNameConstants.IS_FIELD:
                    var fieldTarget = clazz.fieldLookup(name);
                    if (fieldTarget !== null) {
                        flags |= fieldTarget.accessFlags.getRawByte();
                        memberName['java/lang/invoke/MemberName/flags'] = flags;
                        memberName.vmindex = clazz.getVMIndexForField(fieldTarget);
                        return memberName;
                    }
                    else {
                        thread.throwNewException('Ljava/lang/NoSuchFieldError;', "Invalid method " + name + " in class " + clazz.getExternalName() + ".");
                    }
                    break;
                default:
                    thread.throwNewException('Ljava/lang/LinkageError;', 'resolve member name');
                    break;
            }
        };
        /**
         * Follows the same logic as sun.misc.Unsafe's objectFieldOffset.
         */
        java_lang_invoke_MethodHandleNatives['objectFieldOffset(Ljava/lang/invoke/MemberName;)J'] = function (thread, memberName) {
            if (memberName['vmindex'] === -1) {
                thread.throwNewException("Ljava/lang/IllegalStateException;", "Attempted to retrieve the object offset for an unresolved or non-object MemberName.");
            }
            else {
                return Long.fromNumber(memberName.vmindex);
            }
        };
        /**
         * Follows the same logic as sun.misc.Unsafe's staticFieldOffset.
         */
        java_lang_invoke_MethodHandleNatives['staticFieldOffset(Ljava/lang/invoke/MemberName;)J'] = function (thread, memberName) {
            if (memberName['vmindex'] === -1) {
                thread.throwNewException("Ljava/lang/IllegalStateException;", "Attempted to retrieve the object offset for an unresolved or non-object MemberName.");
            }
            else {
                return Long.fromNumber(memberName.vmindex);
            }
        };
        /**
         * Follows the same logic as sun.misc.Unsafe's staticFieldBase.
         */
        java_lang_invoke_MethodHandleNatives['staticFieldBase(Ljava/lang/invoke/MemberName;)Ljava/lang/Object;'] = function (thread, memberName) {
            // Return a special JVM object.
            // TODO: Actually create a special DoppioJVM class for this.
            var rv = new (thread.getBsCl().getInitializedClass(thread, 'Ljava/lang/Object;').getConstructor(thread))(thread);
            rv.$staticFieldBase = memberName['java/lang/invoke/MemberName/clazz'].$cls;
            return rv;
        };
        /**
         * Get the members of the given class that match the specified flags, skipping
         * the specified number of members. For each non-skipped matching member,
         * fill in the fields of a MemberName objects in the results array.
         * If there are more matches than can fit in the array, do *not* overrun
         * the array. Return the total number of matching non-skipped members.
         * TODO: Access checks?
         */
        java_lang_invoke_MethodHandleNatives['getMembers(Ljava/lang/Class;Ljava/lang/String;Ljava/lang/String;ILjava/lang/Class;I[Ljava/lang/invoke/MemberName;)I'] = function (thread, defc, matchName, matchSig, matchFlags, caller, skip, results) {
            // General search flags.
            var searchSuperclasses = 0 !== (matchFlags & MemberNameConstants.SEARCH_SUPERCLASSES), searchInterfaces = 0 !== (matchFlags & MemberNameConstants.SEARCH_INTERFACES), matched = 0, targetClass = defc.$cls, methods, fields, matchArray = results.array, name = matchName !== null ? matchName.toString() : null, sig = matchSig !== null ? matchSig.toString() : null;
            /**
             * Helper function: Adds matched items to the array once we've skipped
             * enough.
             */
            function addMatch(item) {
                if (skip >= 0) {
                    if (matched < matchArray.length) {
                        initializeMemberName(thread, matchArray[matched], item);
                    }
                    matched++;
                }
                else {
                    skip--;
                }
            }
            // TODO: Support these flags.
            assert(!searchSuperclasses && !searchInterfaces, "Unsupported: Non-local getMembers calls.");
            // Constructors
            if (0 !== (matchFlags & MemberNameConstants.IS_CONSTRUCTOR) && (name === null || name === "<init>")) {
                methods = targetClass.getMethods();
                methods.forEach(function (m) {
                    if (m.name === "<init>" && (sig === null || sig === m.rawDescriptor)) {
                        addMatch(m);
                    }
                });
            }
            // Methods
            if (0 !== (matchFlags & MemberNameConstants.IS_METHOD)) {
                methods = targetClass.getMethods();
                methods.forEach(function (m) {
                    if (m.name !== "<init>" && (name === null || name === m.name) && (sig === null || sig === m.rawDescriptor)) {
                        addMatch(m);
                    }
                });
            }
            // Fields
            if (0 !== (matchFlags & MemberNameConstants.IS_FIELD) && sig === null) {
                fields = targetClass.getFields();
                fields.forEach(function (f) {
                    if (name === null || name === f.name) {
                        addMatch(f);
                    }
                });
            }
            // TODO: Inner types (IS_TYPE).
            assert(0 == (matchFlags & MemberNameConstants.IS_TYPE), "Unsupported: Getting inner type MemberNames.");
            return matched;
        };
        /**
         * Debug native in the JDK: Gets a named constant from MethodHandleNatives.Constants.
         */
        java_lang_invoke_MethodHandleNatives['getNamedCon(I[Ljava/lang/Object;)I'] = function (thread, fieldNum, args) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            thread.getBsCl().initializeClass(thread, "Ljava/lang/invoke/MethodHandleNatives$Constants;", function (constantsCls) {
                if (constantsCls === null) {
                    return;
                }
                var constants = constantsCls.getFields().filter(function (field) { return field.accessFlags.isStatic() && field.accessFlags.isFinal(); });
                if (fieldNum < constants.length) {
                    var field = constants[fieldNum];
                    args.array[0] = util.initString(thread.getBsCl(), field.name);
                    thread.asyncReturn(constantsCls.getConstructor(thread)[field.fullName]);
                }
                else {
                    thread.asyncReturn(-1);
                }
            });
        };
        java_lang_invoke_MethodHandleNatives['getMemberVMInfo(Ljava/lang/invoke/MemberName;)Ljava/lang/Object;'] = function (thread, mname) {
            var rv = util.newArray(thread, thread.getBsCl(), '[Ljava/lang/Object;', 2), flags = mname['java/lang/invoke/MemberName/flags'], refKind = flags >>> MemberNameConstants.REFERENCE_KIND_SHIFT, longCls = thread.getBsCl().getInitializedClass(thread, 'J');
            // VMIndex of the target.
            rv.array[0] = longCls.createWrapperObject(thread, Long.fromNumber(mname.vmindex));
            // Class if field, membername if method
            rv.array[1] = (((flags & MemberNameConstants.ALL_KINDS) & MemberNameConstants.IS_FIELD) > 0) ? mname['java/lang/invoke/MemberName/clazz'] : mname;
            return rv;
        };
        java_lang_invoke_MethodHandleNatives['setCallSiteTargetNormal(Ljava/lang/invoke/CallSite;Ljava/lang/invoke/MethodHandle;)V'] = function (thread, callSite, methodHandle) {
            callSite['java/lang/invoke/CallSite/target'] = methodHandle;
        };
        return java_lang_invoke_MethodHandleNatives;
    }());
    var java_lang_invoke_MethodHandle = (function () {
        function java_lang_invoke_MethodHandle() {
        }
        /**
         * Invokes the method handle, allowing any caller type descriptor, but requiring an exact type match.
         *
         * If this native method is invoked directly via java.lang.reflect.Method.invoke,
         * via JNI, or indirectly via java.lang.invoke.MethodHandles.Lookup.unreflect,
         * it will throw an UnsupportedOperationException.
         *
         * @throws WrongMethodTypeException if the target's type is not identical with the caller's symbolic type descriptor
         * @throws Throwable anything thrown by the underlying method propagates unchanged through the method handle call
         */
        java_lang_invoke_MethodHandle['invokeExact([Ljava/lang/Object;)Ljava/lang/Object;'] = function (thread, mh, args) {
            // Like other JVMs, we bake the semantics of invoke/invokeExact directly
            // into the bytecode. Thus, this version of the method will *only* be
            // invoked via reflection, causing this exception.
            thread.throwNewException("Ljava/lang/UnsupportedOperationException;", "MethodHandle.invokeExact cannot be invoked reflectively");
        };
        java_lang_invoke_MethodHandle['invoke([Ljava/lang/Object;)Ljava/lang/Object;'] = function (thread, mh, args) {
            // Like other JVMs, we bake the semantics of invoke/invokeExact directly
            // into the bytecode. Thus, this version of the method will *only* be
            // invoked via reflection, causing this exception.
            thread.throwNewException("Ljava/lang/UnsupportedOperationException;", "MethodHandle.invoke cannot be invoked reflectively");
        };
        /**
         * Unlike invoke and invokeExact, invokeBasic *can* be invoked reflectively,
         * and thus it has an implementation here. Note that invokeBasic is private,
         * and thus can only be invoked by trusted OpenJDK code.
         *
         * When invoked reflectively, arguments to invokeBasic will be boxed.
         *
         * The return value is *never* boxed. Yes, this is weird. It's only called by
         * trusted code, though.
         */
        java_lang_invoke_MethodHandle['invokeBasic([Ljava/lang/Object;)Ljava/lang/Object;'] = function (thread, mh, argsBoxed) {
            var lmbdaForm = mh['java/lang/invoke/MethodHandle/form'], mn = lmbdaForm['java/lang/invoke/LambdaForm/vmentry'], descriptor, paramTypes;
            assert(mh.getClass().isCastable(thread.getBsCl().getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandle;')), "First argument to invokeBasic must be a method handle.");
            assert(mn.vmtarget !== null && mn.vmtarget !== undefined, "vmtarget must be defined");
            assert(mn['java/lang/invoke/MemberName/type'].getClass().getInternalName() === 'Ljava/lang/invoke/MethodType;', "Expected a MethodType object.");
            descriptor = mn['java/lang/invoke/MemberName/type'].toString();
            paramTypes = util.getTypes(descriptor);
            // Remove return value.
            paramTypes.pop();
            // Remove method handle; it's not boxed.
            paramTypes.shift();
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            // Need to include methodhandle in the arguments to vmtarget, which handles
            // invoking it appropriately.
            mn.vmtarget(thread, descriptor, [mh].concat(util.unboxArguments(thread, paramTypes, argsBoxed.array)), function (e, rv) {
                if (e) {
                    thread.throwException(e);
                }
                else {
                    thread.asyncReturn(rv);
                }
            });
        };
        return java_lang_invoke_MethodHandle;
    }());
    return {
        'java/lang/Class': java_lang_Class,
        'java/lang/ClassLoader$NativeLibrary': java_lang_ClassLoader$NativeLibrary,
        'java/lang/ClassLoader': java_lang_ClassLoader,
        'java/lang/Compiler': java_lang_Compiler,
        'java/lang/Double': java_lang_Double,
        'java/lang/Float': java_lang_Float,
        'java/lang/Object': java_lang_Object,
        'java/lang/Package': java_lang_Package,
        'java/lang/ProcessEnvironment': java_lang_ProcessEnvironment,
        'java/lang/reflect/Array': java_lang_reflect_Array,
        'java/lang/reflect/Proxy': java_lang_reflect_Proxy,
        'java/lang/Runtime': java_lang_Runtime,
        'java/lang/SecurityManager': java_lang_SecurityManager,
        'java/lang/Shutdown': java_lang_Shutdown,
        'java/lang/StrictMath': java_lang_StrictMath,
        'java/lang/String': java_lang_String,
        'java/lang/System': java_lang_System,
        'java/lang/Thread': java_lang_Thread,
        'java/lang/Throwable': java_lang_Throwable,
        'java/lang/UNIXProcess': java_lang_UNIXProcess,
        'java/lang/invoke/MethodHandleNatives': java_lang_invoke_MethodHandleNatives,
        'java/lang/invoke/MethodHandle': java_lang_invoke_MethodHandle
    };
}
exports.__esModule = true;
exports["default"] = default_1;
;
//# sourceMappingURL=data:application/json;base64,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