"use strict";
var Doppio = require('../doppiojvm');
var logging = Doppio.Debug.Logging;
var util = Doppio.VM.Util;
var ThreadStatus = Doppio.VM.Enums.ThreadStatus;
var debug = logging.debug;
function default_1() {
    /**
     * If an application wants to open a TCP/UDP connection to "foobar.com", the application must first perform
     * a DNS lookup to determine the IP of that domain, and then open a socket to that IP. Doppio needs to emulate
     * this same functionality in JavaScript.
     *
     * However, the browser does not expose any DNS interfaces, as DNS lookup is provided opaquely by the browser
     * platform. For example, an application can make a WebSocket connection directly to "https://foobar.com/", and
     * will never know that domain's IP.
     *
     * To get around this missing functionality, Doppio returns an unused private IP in the range of 240.0.0.0 to
     * 250.0.0.0 for each unique DNS lookup. Doppio uses this IP as a token for that particular DNS lookup. When
     * the application attempts to connect to an IP in this range, Doppio uses the IP as a key into a hash table,
     * which returns a domain name that Doppio uses in the resulting WebSocket connection. An application will never
     * try to connect to one of these invalid IP addresses directly, so Doppio can distinguish between connections to
     * specific IP addresses and connections to domains.
     */
    var host_lookup = {}, host_reverse_lookup = {}, 
    // 240.0.0.0 .. 250.0.0.0 is currently unused address space
    next_host_address = 0xF0000000;
    // See RFC 6455 section 7.4
    function websocket_status_to_message(status) {
        switch (status) {
            case 1000:
                return 'Normal closure';
            case 1001:
                return 'Endpoint is going away';
            case 1002:
                return 'WebSocket protocol error';
            case 1003:
                return 'Server received invalid data';
        }
        return 'Unknown status code or error';
    }
    function next_address() {
        next_host_address++;
        if (next_host_address > 0xFA000000) {
            logging.error('Out of addresses');
            next_host_address = 0xF0000000;
        }
        return next_host_address;
    }
    function pack_address(address) {
        var i, ret = 0;
        for (i = 3; i >= 0; i--) {
            ret |= address[i] & 0xFF;
            ret <<= 8;
        }
        return ret;
    }
    function host_allocate_address(address) {
        var ret = next_address();
        host_lookup[ret] = address;
        host_reverse_lookup[address] = ret;
        return ret;
    }
    /**
     * Asynchronously read data from a socket. Note that if this passes 0 to the
     * callback, Java will think it has received an EOF. Thus, we should wait until:
     * - We have at least one byte to return.
     * - The socket is closed.
     */
    function socket_read_async(impl, b, offset, len, resume_cb) {
        var i, available = impl.$ws.rQlen(), trimmed_len = available < len ? available : len, read = impl.$ws.rQshiftBytes(trimmed_len);
        for (i = 0; i < trimmed_len; i++) {
            b.array[offset++] = read[i];
        }
        resume_cb(trimmed_len);
    }
    var java_net_Inet4Address = (function () {
        function java_net_Inet4Address() {
        }
        java_net_Inet4Address['init()V'] = function (thread) {
            // NOP
        };
        return java_net_Inet4Address;
    }());
    var java_net_Inet4AddressImpl = (function () {
        function java_net_Inet4AddressImpl() {
        }
        java_net_Inet4AddressImpl['getLocalHostName()Ljava/lang/String;'] = function (thread, javaThis) {
            return thread.getJVM().internString('localhost');
        };
        java_net_Inet4AddressImpl['lookupAllHostAddr(Ljava/lang/String;)[Ljava/net/InetAddress;'] = function (thread, javaThis, hostname) {
            var rv = util.newObject(thread, thread.getBsCl(), 'Ljava/net/Inet4Address;');
            rv['<init>(Ljava/lang/String;I)V'](thread, [hostname, host_allocate_address(hostname.toString())], function (e) {
                if (e) {
                    thread.throwException(e);
                }
                else {
                    thread.asyncReturn(util.newArrayFromData(thread, thread.getBsCl(), '[Ljava/net/InetAddress;', [rv]));
                }
            });
        };
        java_net_Inet4AddressImpl['getHostByAddr([B)Ljava/lang/String;'] = function (thread, javaThis, addr) {
            var ret = host_reverse_lookup[pack_address(addr.array)];
            if (ret == null) {
                return null;
            }
            return util.initString(thread.getBsCl(), "" + ret);
        };
        java_net_Inet4AddressImpl['isReachable0([BI[BI)Z'] = function (thread, javaThis, arg0, arg1, arg2, arg3) {
            return false;
        };
        return java_net_Inet4AddressImpl;
    }());
    var java_net_Inet6Address = (function () {
        function java_net_Inet6Address() {
        }
        java_net_Inet6Address['init()V'] = function (thread) {
            // NOP
        };
        return java_net_Inet6Address;
    }());
    var java_net_InetAddress = (function () {
        function java_net_InetAddress() {
        }
        java_net_InetAddress['init()V'] = function (thread) {
            // NOP
        };
        return java_net_InetAddress;
    }());
    var java_net_InetAddressImplFactory = (function () {
        function java_net_InetAddressImplFactory() {
        }
        java_net_InetAddressImplFactory['isIPv6Supported()Z'] = function (thread) {
            return false;
        };
        return java_net_InetAddressImplFactory;
    }());
    var java_net_PlainSocketImpl = (function () {
        function java_net_PlainSocketImpl() {
        }
        java_net_PlainSocketImpl['socketCreate(Z)V'] = function (thread, javaThis, isServer) {
            // Check to make sure we're in a browser and the websocket libraries are present
            if (!util.are_in_browser()) {
                thread.throwNewException('Ljava/io/IOException;', 'WebSockets are disabled');
            }
            else {
                var fd = javaThis['java/net/SocketImpl/fd'];
                // Make the FileDescriptor valid with a dummy fd
                fd['java/io/FileDescriptor/fd'] = 8374;
                // Finally, create our websocket instance
                javaThis.$ws = new Websock();
                javaThis.$is_shutdown = false;
            }
        };
        java_net_PlainSocketImpl['socketConnect(Ljava/net/InetAddress;II)V'] = function (thread, javaThis, address, port, timeout) {
            var i, 
            // The IPv4 case
            holder = address['java/net/InetAddress/holder'], addy = holder['java/net/InetAddress$InetAddressHolder/address'], 
            // Assume scheme is ws for now
            host = 'ws://';
            if (host_lookup[addy] == null) {
                // Populate host string based off of IP address
                for (i = 3; i >= 0; i--) {
                    var shift = i * 8;
                    host += "" + ((addy & (0xFF << shift)) >>> shift) + ".";
                }
                // trim last '.'
                host = host.substring(0, host.length - 1);
            }
            else {
                host += host_lookup[addy];
            }
            // Add port
            host += ":" + port;
            debug("Connecting to " + host + " with timeout = " + timeout + " ms");
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var id = 0, clear_state = function () {
                window.clearTimeout(id);
                javaThis.$ws.on('open', function () { });
                javaThis.$ws.on('close', function () { });
                javaThis.$ws.on('error', function () { });
            }, error_cb = function (msg) {
                return function (e) {
                    clear_state();
                    thread.throwNewException('Ljava/io/IOException;', msg + ": " + e);
                };
            }, close_cb = function (msg) {
                return function (e) {
                    clear_state();
                    thread.throwNewException('Ljava/io/IOException;', msg + ": " + websocket_status_to_message(e.status));
                };
            };
            // Success case
            javaThis.$ws.on('open', function () {
                debug('Open!');
                clear_state();
                thread.asyncReturn();
            });
            // Error cases
            javaThis.$ws.on('close', close_cb('Connection failed! (Closed)'));
            // Timeout case. In the case of no timeout, we set a default one of 10s.
            if (timeout === 0) {
                timeout = 10000;
            }
            // XXX: Casting to a number because NodeJS typings specify a Timer object.
            id = setTimeout(error_cb('Connection timeout!'), timeout);
            debug("Host: " + host);
            // Launch!
            try {
                javaThis.$ws.open(host);
            }
            catch (err) {
                error_cb('Connection failed! (exception)')(err.message);
            }
        };
        java_net_PlainSocketImpl['socketBind(Ljava/net/InetAddress;I)V'] = function (thread, javaThis, arg0, arg1) {
            thread.throwNewException('Ljava/io/IOException;', 'WebSockets doesn\'t know how to bind');
        };
        java_net_PlainSocketImpl['socketListen(I)V'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/io/IOException;', 'WebSockets doesn\'t know how to listen');
        };
        java_net_PlainSocketImpl['socketAccept(Ljava/net/SocketImpl;)V'] = function (thread, javaThis, arg0) {
            thread.throwNewException('Ljava/io/IOException;', 'WebSockets doesn\'t know how to accept');
        };
        java_net_PlainSocketImpl['socketAvailable()I'] = function (thread, javaThis) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            setImmediate(function () {
                thread.asyncReturn(javaThis.$ws.rQlen());
            });
        };
        java_net_PlainSocketImpl['socketClose0(Z)V'] = function (thread, javaThis, arg0) {
            // TODO: Something isn't working here
            javaThis.$ws.close();
        };
        java_net_PlainSocketImpl['socketShutdown(I)V'] = function (thread, javaThis, arg0) {
            javaThis.$is_shutdown = true;
        };
        java_net_PlainSocketImpl['initProto()V'] = function (thread) {
            // NOP
        };
        java_net_PlainSocketImpl['socketSetOption0(IZLjava/lang/Object;)V'] = function (thread, javaThis, arg0, arg1, arg2) {
            // NOP
        };
        java_net_PlainSocketImpl['socketGetOption(ILjava/lang/Object;)I'] = function (thread, javaThis, arg0, arg1) {
            // NOP
            return 0;
        };
        java_net_PlainSocketImpl['socketSendUrgentData(I)V'] = function (thread, javaThis, data) {
            // Urgent data is meant to jump ahead of the
            // outbound stream. We keep no notion of this,
            // so queue up the byte like normal
            javaThis.$ws.send(data);
        };
        return java_net_PlainSocketImpl;
    }());
    var java_net_SocketInputStream = (function () {
        function java_net_SocketInputStream() {
        }
        java_net_SocketInputStream['socketRead0(Ljava/io/FileDescriptor;[BIII)I'] = function (thread, javaThis, fd, b, offset, len, timeout) {
            var impl = javaThis['java/net/SocketInputStream/impl'];
            if (impl.$is_shutdown === true) {
                thread.throwNewException('Ljava/io/IOException;', 'Socket is shutdown.');
            }
            else {
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                setTimeout(function () {
                    socket_read_async(impl, b, offset, len, function (arg) {
                        thread.asyncReturn(arg);
                    });
                }, timeout);
            }
        };
        java_net_SocketInputStream['init()V'] = function (thread) {
            // NOP
        };
        return java_net_SocketInputStream;
    }());
    var java_net_SocketOutputStream = (function () {
        function java_net_SocketOutputStream() {
        }
        java_net_SocketOutputStream['socketWrite0(Ljava/io/FileDescriptor;[BII)V'] = function (thread, javaThis, fd, b, offset, len) {
            var impl = javaThis['java/net/SocketOutputStream/impl'];
            if (impl.$is_shutdown === true) {
                thread.throwNewException('Ljava/io/IOException;', 'Socket is shutdown.');
            }
            else if (impl.$ws.get_raw_state() !== WebSocket.OPEN) {
                thread.throwNewException('Ljava/io/IOException;', 'Connection isn\'t open');
            }
            else {
                // TODO: This can be optimized by accessing the 'Q' directly
                impl.$ws.send(b.array.slice(offset, offset + len));
                // Let the browser write it out
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                setImmediate(function () {
                    thread.asyncReturn();
                });
            }
        };
        java_net_SocketOutputStream['init()V'] = function (thread) {
            // NOP
        };
        return java_net_SocketOutputStream;
    }());
    var java_net_NetworkInterface = (function () {
        function java_net_NetworkInterface() {
        }
        java_net_NetworkInterface['init()V'] = function (thread) {
            // NOP
        };
        java_net_NetworkInterface['getAll()[Ljava/net/NetworkInterface;'] = function (thread) {
            var bsCl = thread.getBsCl();
            // Create a fake network interface bound to 127.1.1.1.
            thread.import(['Ljava/net/NetworkInterface;', 'Ljava/net/InetAddress;'], function (rv) {
                var niCons = rv[0], inetStatics = rv[1], iName = thread.getJVM().internString('doppio1');
                inetStatics['getByAddress(Ljava/lang/String;[B)Ljava/net/InetAddress;'](thread, [iName, util.newArrayFromData(thread, thread.getBsCl(), '[B', [127, 1, 1, 1])], function (e, rv) {
                    if (e) {
                        thread.throwException(e);
                    }
                    else {
                        var niObj = new niCons(thread);
                        niObj['<init>(Ljava/lang/String;I[Ljava/net/InetAddress;)V'](thread, [iName, 0, util.newArrayFromData(thread, bsCl, '[Ljava/net/InetAddress;', [rv])], function (e) {
                            if (e) {
                                thread.throwException(e);
                            }
                            else {
                                thread.asyncReturn(util.newArrayFromData(thread, bsCl, '[Ljava/net/NetworkInterface;', [niObj]));
                            }
                        });
                    }
                });
            });
        };
        java_net_NetworkInterface['getMacAddr0([BLjava/lang/String;I)[B'] = function (thread, inAddr, name, ind) {
            return util.newArrayFromData(thread, thread.getBsCl(), '[B', [1, 1, 1, 1, 1, 1]);
        };
        return java_net_NetworkInterface;
    }());
    return {
        'java/net/Inet4Address': java_net_Inet4Address,
        'java/net/Inet4AddressImpl': java_net_Inet4AddressImpl,
        'java/net/Inet6Address': java_net_Inet6Address,
        'java/net/InetAddress': java_net_InetAddress,
        'java/net/InetAddressImplFactory': java_net_InetAddressImplFactory,
        'java/net/PlainSocketImpl': java_net_PlainSocketImpl,
        'java/net/SocketInputStream': java_net_SocketInputStream,
        'java/net/SocketOutputStream': java_net_SocketOutputStream,
        'java/net/NetworkInterface': java_net_NetworkInterface
    };
}
exports.__esModule = true;
exports["default"] = default_1;
;
//# sourceMappingURL=data:application/json;base64,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