"use strict";
var Doppio = require('../doppiojvm');
var util = Doppio.VM.Util;
var Long = Doppio.VM.Long;
var ThreadStatus = Doppio.VM.Enums.ThreadStatus;
var fs = require('fs');
var BrowserFS = require('browserfs');
var FDState = Doppio.VM.FDState;
var BFSUtils = BrowserFS.BFSRequire('bfs_utils');
function default_1() {
    var sun_nio_ch_FileChannelImpl = (function () {
        function sun_nio_ch_FileChannelImpl() {
        }
        sun_nio_ch_FileChannelImpl['map0(IJJ)J'] = function (thread, javaThis, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return null;
        };
        sun_nio_ch_FileChannelImpl['unmap0(JJ)I'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            // Satisfy TypeScript return type.
            return 0;
        };
        sun_nio_ch_FileChannelImpl['position0(Ljava/io/FileDescriptor;J)J'] = function (thread, javaThis, fdObj, offset) {
            var fd = fdObj['java/io/FileDescriptor/fd'];
            var rv;
            if (offset.equals(Long.NEG_ONE)) {
                // Get current FD offset.
                rv = FDState.getPos(fd);
            }
            else {
                // Set FD offset.
                rv = offset.toNumber();
                FDState.setPos(fd, rv);
            }
            return Long.fromNumber(rv);
        };
        /**
         * this poorly-named method actually specifies the page size for mmap
         * This is the Mac name for sun/misc/Unsafe::pageSize. Apparently they
         * wanted to ensure page sizes can be > 2GB...
         */
        sun_nio_ch_FileChannelImpl['initIDs()J'] = function (thread) {
            // Size of heap pages.
            return Long.fromNumber(4096);
        };
        return sun_nio_ch_FileChannelImpl;
    }());
    var sun_nio_ch_NativeThread = (function () {
        function sun_nio_ch_NativeThread() {
        }
        sun_nio_ch_NativeThread['current()J'] = function (thread) {
            // -1 means that we do not require signaling according to the
            // docs.
            return Long.fromNumber(-1);
        };
        sun_nio_ch_NativeThread['signal(J)V'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_ch_NativeThread['init()V'] = function (thread) {
            // NOP
        };
        return sun_nio_ch_NativeThread;
    }());
    var sun_nio_ch_IOUtil = (function () {
        function sun_nio_ch_IOUtil() {
        }
        sun_nio_ch_IOUtil['iovMax()I'] = function (thread) {
            // Maximum number of IOVectors supported. Let's punt and say zero.
            return 0;
        };
        return sun_nio_ch_IOUtil;
    }());
    var sun_nio_ch_FileDispatcherImpl = (function () {
        function sun_nio_ch_FileDispatcherImpl() {
        }
        sun_nio_ch_FileDispatcherImpl['init()V'] = function (thread) {
        };
        sun_nio_ch_FileDispatcherImpl['read0(Ljava/io/FileDescriptor;JI)I'] = function (thread, fdObj, address, len) {
            var fd = fdObj["java/io/FileDescriptor/fd"], 
            // read upto len bytes and store into mmap'd buffer at address
            addr = address.toNumber(), buf = thread.getJVM().getHeap().get_buffer(addr, len);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.read(fd, buf, 0, len, FDState.getPos(fd), function (err, bytesRead) {
                if (err) {
                    thread.throwNewException("Ljava/io/IOException;", 'Error reading file: ' + err);
                }
                else {
                    FDState.incrementPos(fd, bytesRead);
                    // Return -1 if we reached the end of the file.
                    thread.asyncReturn(bytesRead === 0 ? -1 : bytesRead);
                }
            });
        };
        sun_nio_ch_FileDispatcherImpl['preClose0(Ljava/io/FileDescriptor;)V'] = function (thread, arg0) {
            // NOP, I think the actual fs.close is called later. If not, NBD.
        };
        sun_nio_ch_FileDispatcherImpl['close0(Ljava/io/FileDescriptor;)V'] = function (thread, fdObj) {
            var fd = fdObj["java/io/FileDescriptor/fd"];
            sun_nio_ch_FileDispatcherImpl['closeIntFD(I)V'](thread, fd);
            FDState.close(fd);
            fdObj["java/io/FileDescriptor/fd"] = -1;
        };
        sun_nio_ch_FileDispatcherImpl['size0(Ljava/io/FileDescriptor;)J'] = function (thread, fdObj) {
            var fd = fdObj["java/io/FileDescriptor/fd"];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.fstat(fd, function (err, stats) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn(Long.fromNumber(stats.size), null);
                }
            });
        };
        sun_nio_ch_FileDispatcherImpl['truncate0(Ljava/io/FileDescriptor;J)I'] = function (thread, fdObj, size) {
            var fd = fdObj["java/io/FileDescriptor/fd"];
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.ftruncate(fd, size.toNumber(), function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    // For some reason, this expects a return value.
                    // Give it the success status code.
                    thread.asyncReturn(0);
                }
            });
        };
        sun_nio_ch_FileDispatcherImpl['closeIntFD(I)V'] = function (thread, fd) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.close(fd, function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    FDState.close(fd);
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_ch_FileDispatcherImpl['write0(Ljava/io/FileDescriptor;JI)I'] = function (thread, fdObj, addr, len) {
            var fd = fdObj["java/io/FileDescriptor/fd"];
            var heap = thread.getJVM().getHeap();
            var data = heap.get_buffer(addr.toNumber(), len);
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.write(fd, data, 0, len, FDState.getPos(fd), function (err, numBytes) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    FDState.incrementPos(fd, numBytes);
                    thread.asyncReturn(numBytes);
                }
            });
        };
        return sun_nio_ch_FileDispatcherImpl;
    }());
    var DirFd = (function () {
        function DirFd(listing) {
            this._pos = 0;
            this._listing = listing;
        }
        DirFd.prototype.next = function () {
            var next = this._listing[this._pos++];
            if (next === undefined) {
                next = null;
            }
            return next;
        };
        return DirFd;
    }());
    var FDMap = (function () {
        function FDMap() {
            this._map = {};
        }
        FDMap.prototype.newEntry = function (entry) {
            var fd = FDMap._nextFd++;
            this._map[fd] = entry;
            return fd;
        };
        FDMap.prototype.removeEntry = function (thread, fd, exceptionType) {
            if (this._map[fd]) {
                delete this._map[fd];
            }
            else {
                thread.throwNewException(exceptionType, "Invalid file descriptor: " + fd);
            }
        };
        FDMap.prototype.getEntry = function (thread, exceptionType, fd) {
            var entry = this._map[fd];
            if (!entry) {
                thread.throwNewException(exceptionType, "Invalid file descriptor: " + fd);
                return null;
            }
            else {
                return entry;
            }
        };
        FDMap._nextFd = 1;
        return FDMap;
    }());
    var dirMap = new FDMap();
    function getStringFromHeap(thread, ptrLong) {
        var heap = thread.getJVM().getHeap(), ptr = ptrLong.toNumber(), len = 0;
        while (heap.get_signed_byte(ptr + len) !== 0) {
            len++;
        }
        return heap.get_buffer(ptr, len).toString();
    }
    /**
     * Converts a string into a C string stored in a JVM array.
     * No NULL terminator needed, since arrays have length.
     */
    function stringToByteArray(thread, str) {
        if (!str) {
            return null;
        }
        // NULL terminate the string.
        var buff = new Buffer(str, 'utf8');
        var len = buff.length;
        var i8 = new Int8Array(len);
        for (var i = 0; i < len; i++) {
            i8[i] = buff.readInt8(i);
        }
        return util.newArrayFromData(thread, thread.getBsCl(), '[B', i8);
    }
    function convertError(thread, err, cb) {
        thread.setStatus(ThreadStatus.ASYNC_WAITING);
        if (err.code === 'ENOENT') {
            thread.getBsCl().initializeClass(thread, 'Ljava/nio/file/NoSuchFileException;', function (noSuchFileException) {
                var cons = noSuchFileException.getConstructor(thread), rv = new cons(thread);
                rv['<init>(Ljava/lang/String;)V'](thread, [util.initString(thread.getBsCl(), err.path)], function (e) {
                    thread.throwException(rv);
                });
            });
        }
        else if (err.code === 'EEXIST') {
            thread.getBsCl().initializeClass(thread, 'Ljava/nio/file/FileAlreadyExistsException;', function (fileAlreadyExistsException) {
                var cons = fileAlreadyExistsException.getConstructor(thread), rv = new cons(thread);
                rv['<init>(Ljava/lang/String;)V'](thread, [util.initString(thread.getBsCl(), err.path)], function (e) {
                    cb(rv);
                });
            });
        }
        else {
            thread.getBsCl().initializeClass(thread, 'Lsun/nio/fs/UnixException;', function (unixException) {
                thread.getBsCl().initializeClass(thread, 'Lsun/nio/fs/UnixConstants;', function (unixConstants) {
                    var cons = unixException.getConstructor(thread), rv = new cons(thread), unixCons = unixConstants.getConstructor(thread), errCode = unixCons[("sun/nio/fs/UnixConstants/" + err.code)];
                    if (typeof (errCode) !== 'number') {
                        errCode = -1;
                    }
                    rv['sun/nio/fs/UnixException/errno'] = errCode;
                    rv['sun/nio/fs/UnixException/msg'] = util.initString(thread.getBsCl(), err.message);
                    cb(rv);
                });
            });
        }
    }
    function convertStats(stats, jvmStats) {
        jvmStats['sun/nio/fs/UnixFileAttributes/st_mode'] = stats.mode;
        jvmStats['sun/nio/fs/UnixFileAttributes/st_ino'] = Long.fromNumber(stats.ino);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_dev'] = Long.fromNumber(stats.dev);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_rdev'] = Long.fromNumber(stats.rdev);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_nlink'] = stats.nlink;
        jvmStats['sun/nio/fs/UnixFileAttributes/st_uid'] = stats.uid;
        jvmStats['sun/nio/fs/UnixFileAttributes/st_gid'] = stats.gid;
        jvmStats['sun/nio/fs/UnixFileAttributes/st_size'] = Long.fromNumber(stats.size);
        var atime = date2components(stats.atime), mtime = date2components(stats.mtime), ctime = date2components(stats.ctime);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_atime_sec'] = Long.fromNumber(atime[0]);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_atime_nsec'] = Long.fromNumber(atime[1]);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_mtime_sec'] = Long.fromNumber(mtime[0]);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_mtime_nsec'] = Long.fromNumber(mtime[1]);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_ctime_sec'] = Long.fromNumber(ctime[0]);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_ctime_nsec'] = Long.fromNumber(ctime[1]);
        jvmStats['sun/nio/fs/UnixFileAttributes/st_birthtime_sec'] = Long.fromNumber(Math.floor(stats.birthtime.getTime() / 1000));
    }
    var UnixConstants = null;
    function flagTest(flag, mask, value) {
        if (value === void 0) { value = mask; }
        return (flag & mask) === value;
    }
    /**
     * Converts a numerical Unix open() flag to a NodeJS string open() flag.
     * Returns NULL upon failure; throws a UnixException on thread when that happens.
     */
    function flag2nodeflag(thread, flag) {
        if (UnixConstants === null) {
            var UCCls = thread.getBsCl().getInitializedClass(thread, 'Lsun/nio/fs/UnixConstants;');
            if (UCCls === null) {
                thread.throwNewException("Ljava/lang/InternalError;", "UnixConstants is not initialized?");
                return null;
            }
            UnixConstants = UCCls.getConstructor(thread);
        }
        var sync = flagTest(flag, UnixConstants['sun/nio/fs/UnixConstants/O_SYNC']) || flagTest(flag, UnixConstants['sun/nio/fs/UnixConstants/O_DSYNC']);
        var failIfExists = flagTest(flag, UnixConstants['sun/nio/fs/UnixConstants/O_EXCL'] | UnixConstants['sun/nio/fs/UnixConstants/O_CREAT']);
        var O_ACCMODE = 0x3;
        if (flagTest(flag, O_ACCMODE, UnixConstants['sun/nio/fs/UnixConstants/O_RDWR'])) {
            if (flagTest(flag, UnixConstants['sun/nio/fs/UnixConstants/O_APPEND'])) {
                // 'a+' - Open file for reading and appending. The file is created if it does not exist.
                // 'ax+' - Like 'a+' but fails if path exists.
                return failIfExists ? 'ax+' : 'a+';
            }
            else if (flagTest(flag, UnixConstants['sun/nio/fs/UnixConstants/O_CREAT'])) {
                // 'w+' - Open file for reading and writing. The file is created (if it does not exist) or truncated (if it exists).
                // 'wx+' - Like 'w+' but fails if path exists.
                return failIfExists ? 'wx+' : 'w+';
            }
            else {
                // 'r+' - Open file for reading and writing. An exception occurs if the file does not exist.
                // 'rs+' - Open file for reading and writing, telling the OS to open it synchronously.
                return sync ? 'rs+' : 'r+';
            }
        }
        else if (flagTest(flag, O_ACCMODE, UnixConstants['sun/nio/fs/UnixConstants/O_RDONLY'])) {
            // 'r' - Open file for reading. An exception occurs if the file does not exist.
            // 'rs' - Open file for reading in synchronous mode. Instructs the operating system to bypass the local file system cache.
            return sync ? 'rs' : 'r';
        }
        else if (flagTest(flag, O_ACCMODE, UnixConstants['sun/nio/fs/UnixConstants/O_WRONLY'])) {
            if (flagTest(flag, UnixConstants['sun/nio/fs/UnixConstants/O_APPEND'])) {
                // 'ax' - Like 'a' but fails if path exists.
                // 'a' - Open file for appending. The file is created if it does not exist.
                return failIfExists ? 'ax' : 'a';
            }
            else {
                // 'w' - Open file for writing. The file is created (if it does not exist) or truncated (if it exists).
                // 'wx' - Like 'w' but fails if path exists.
                return failIfExists ? 'wx' : 'w';
            }
        }
        else {
            thread.throwNewException('Lsun/nio/fs/UnixException;', "Invalid open flag: " + flag + ".");
            return null;
        }
    }
    function throwNodeError(thread, err) {
        convertError(thread, err, function (convertedErr) {
            thread.throwException(convertedErr);
        });
    }
    /**
     * Converts a Date object into [seconds, nanoseconds].
     */
    function date2components(date) {
        var dateInMs = date.getTime();
        return [Math.floor(dateInMs / 1000), (dateInMs % 1000) * 1000000];
    }
    var sun_nio_fs_UnixNativeDispatcher = (function () {
        function sun_nio_fs_UnixNativeDispatcher() {
        }
        sun_nio_fs_UnixNativeDispatcher['getcwd()[B'] = function (thread) {
            return stringToByteArray(thread, process.cwd());
        };
        sun_nio_fs_UnixNativeDispatcher['dup(I)I'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return 0;
        };
        sun_nio_fs_UnixNativeDispatcher['open0(JII)I'] = function (thread, pathAddress, flags, mode) {
            // Essentially, convert open() args to fopen() args.
            var flagStr = flag2nodeflag(thread, flags);
            if (flagStr !== null) {
                thread.setStatus(ThreadStatus.ASYNC_WAITING);
                var pathStr = getStringFromHeap(thread, pathAddress);
                fs.open(pathStr, flagStr, mode, function (err, fd) {
                    if (err) {
                        throwNodeError(thread, err);
                    }
                    else {
                        if (flagStr.indexOf('a') !== -1) {
                            // Need to figure out size of file to set position.
                            fs.fstat(fd, function (err, stats) {
                                if (err) {
                                    throwNodeError(thread, err);
                                }
                                else {
                                    FDState.open(fd, stats.size);
                                    thread.asyncReturn(fd);
                                }
                            });
                        }
                        else {
                            FDState.open(fd, 0);
                            thread.asyncReturn(fd);
                        }
                    }
                });
            }
            else {
                return -1;
            }
        };
        sun_nio_fs_UnixNativeDispatcher['openat0(IJII)I'] = function (thread, arg0, arg1, arg2, arg3) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return 0;
        };
        sun_nio_fs_UnixNativeDispatcher['close(I)V'] = function (thread, fd) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.close(fd, function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    FDState.close(fd);
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['fopen0(JJ)J'] = function (thread, pathAddress, flagsAddress) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var pathStr = getStringFromHeap(thread, pathAddress);
            var flagsStr = getStringFromHeap(thread, flagsAddress);
            fs.open(pathStr, flagsStr, function (err, fd) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    if (flagsStr.indexOf('a') !== -1) {
                        // Need to figure out file size to update file position.
                        fs.fstat(fd, function (err, stats) {
                            if (err) {
                                throwNodeError(thread, err);
                            }
                            else {
                                FDState.open(fd, stats.size);
                                thread.asyncReturn(Long.fromNumber(fd), null);
                            }
                        });
                    }
                    else {
                        FDState.open(fd, 0);
                        thread.asyncReturn(Long.fromNumber(fd), null);
                    }
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['fclose(J)V'] = function (thread, fdLong) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var fd = fdLong.toNumber();
            fs.close(fd, function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    FDState.close(fd);
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['link0(JJ)V'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['unlink0(J)V'] = function (thread, pathAddress) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.unlink(getStringFromHeap(thread, pathAddress), function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['unlinkat0(IJI)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['mknod0(JIJ)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['rename0(JJ)V'] = function (thread, oldAddr, newAddr) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.rename(getStringFromHeap(thread, oldAddr), getStringFromHeap(thread, newAddr), function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['renameat0(IJIJ)V'] = function (thread, arg0, arg1, arg2, arg3) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['mkdir0(JI)V'] = function (thread, pathAddr, mode) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.mkdir(getStringFromHeap(thread, pathAddr), mode, function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['rmdir0(J)V'] = function (thread, pathAddr) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.rmdir(getStringFromHeap(thread, pathAddr), function (err) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['readlink0(J)[B'] = function (thread, pathAddr) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.readlink(getStringFromHeap(thread, pathAddr), function (err, linkPath) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn(stringToByteArray(thread, linkPath));
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['realpath0(J)[B'] = function (thread, pathAddress) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.realpath(getStringFromHeap(thread, pathAddress), function (err, resolvedPath) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn(stringToByteArray(thread, resolvedPath));
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['symlink0(JJ)V'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['stat0(JLsun/nio/fs/UnixFileAttributes;)V'] = function (thread, pathAddress, jvmStats) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.stat(getStringFromHeap(thread, pathAddress), function (err, stats) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    convertStats(stats, jvmStats);
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['lstat0(JLsun/nio/fs/UnixFileAttributes;)V'] = function (thread, pathAddress, jvmStats) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.lstat(getStringFromHeap(thread, pathAddress), function (err, stats) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    convertStats(stats, jvmStats);
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['fstat(ILsun/nio/fs/UnixFileAttributes;)V'] = function (thread, fd, jvmStats) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.fstat(fd, function (err, stats) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    convertStats(stats, jvmStats);
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['fstatat0(IJILsun/nio/fs/UnixFileAttributes;)V'] = function (thread, arg0, arg1, arg2, arg3) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['chown0(JII)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['lchown0(JII)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['fchown(III)V'] = function (thread, arg0, arg1, arg2) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['chmod0(JI)V'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['fchmod(II)V'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['utimes0(JJJ)V'] = function (thread, pathAddress, times0, times1) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var p = getStringFromHeap(thread, pathAddress);
            var t0 = new Date(times0.toNumber());
            var t1 = new Date(times1.toNumber());
            fs.utimes(p, t0, t1, function (err) {
                // Ignore ENOTSUP errors; some BFS backends do not support this operation,
                // and ignoring the error isn't typically an issue.
                if (err && err.code !== 'ENOTSUP') {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['futimes(IJJ)V'] = function (thread, fd, times0, times1) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var t0 = new Date(times0.toNumber());
            var t1 = new Date(times1.toNumber());
            fs.futimes(fd, t0, t1, function (err) {
                // Ignore ENOTSUP errors; some BFS backends do not support this operation,
                // and ignoring the error isn't typically an issue.
                if (err && err.code !== 'ENOTSUP') {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['opendir0(J)J'] = function (thread, ptr) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            fs.readdir(getStringFromHeap(thread, ptr), function (err, files) {
                if (err) {
                    convertError(thread, err, function (errObj) {
                        thread.throwException(errObj);
                    });
                }
                else {
                    thread.asyncReturn(Long.fromNumber(dirMap.newEntry(new DirFd(files))), null);
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['fdopendir(I)J'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return null;
        };
        sun_nio_fs_UnixNativeDispatcher['closedir(J)V'] = function (thread, arg0) {
            dirMap.removeEntry(thread, arg0.toNumber(), 'Lsun/nio/fs/UnixException;');
        };
        sun_nio_fs_UnixNativeDispatcher['readdir(J)[B'] = function (thread, fd) {
            var dirFd = dirMap.getEntry(thread, 'Lsun/nio/fs/UnixException;', fd.toNumber());
            if (dirFd) {
                return stringToByteArray(thread, dirFd.next());
            }
        };
        sun_nio_fs_UnixNativeDispatcher['read(IJI)I'] = function (thread, fd, buf, nbyte) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var buff = thread.getJVM().getHeap().get_buffer(buf.toNumber(), nbyte);
            fs.read(fd, buff, 0, nbyte, FDState.getPos(fd), function (err, bytesRead) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    FDState.incrementPos(fd, bytesRead);
                    thread.asyncReturn(bytesRead);
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['write(IJI)I'] = function (thread, fd, buf, nbyte) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            var buff = thread.getJVM().getHeap().get_buffer(buf.toNumber(), nbyte);
            fs.write(fd, buff, 0, nbyte, FDState.getPos(fd), function (err, bytesWritten) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    FDState.incrementPos(fd, bytesWritten);
                    thread.asyncReturn(bytesWritten);
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['access0(JI)V'] = function (thread, pathAddress, arg1) {
            thread.setStatus(ThreadStatus.ASYNC_WAITING);
            // TODO: Need to check specific flags
            var pathString = getStringFromHeap(thread, pathAddress);
            // TODO: fs.access() is better but not currently supported in browserfs: https://github.com/jvilk/BrowserFS/issues/128
            var checker = util.are_in_browser() ? fs.stat : fs.access;
            checker(pathString, function (err, stat) {
                if (err) {
                    throwNodeError(thread, err);
                }
                else {
                    thread.asyncReturn();
                }
            });
        };
        sun_nio_fs_UnixNativeDispatcher['getpwuid(I)[B'] = function (thread, arg0) {
            // Make something up.
            return stringToByteArray(thread, 'doppio');
        };
        sun_nio_fs_UnixNativeDispatcher['getgrgid(I)[B'] = function (thread, arg0) {
            // Make something up.
            return stringToByteArray(thread, 'doppio');
        };
        sun_nio_fs_UnixNativeDispatcher['getpwnam0(J)I'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return 0;
        };
        sun_nio_fs_UnixNativeDispatcher['getgrnam0(J)I'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return 0;
        };
        sun_nio_fs_UnixNativeDispatcher['statvfs0(JLsun/nio/fs/UnixFileStoreAttributes;)V'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
        };
        sun_nio_fs_UnixNativeDispatcher['pathconf0(JI)J'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return null;
        };
        sun_nio_fs_UnixNativeDispatcher['fpathconf(II)J'] = function (thread, arg0, arg1) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return null;
        };
        sun_nio_fs_UnixNativeDispatcher['strerror(I)[B'] = function (thread, arg0) {
            thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method not implemented.');
            return null;
        };
        sun_nio_fs_UnixNativeDispatcher['init()I'] = function (thread) {
            return 0;
        };
        return sun_nio_fs_UnixNativeDispatcher;
    }());
    return {
        'sun/nio/ch/FileChannelImpl': sun_nio_ch_FileChannelImpl,
        'sun/nio/ch/NativeThread': sun_nio_ch_NativeThread,
        'sun/nio/ch/IOUtil': sun_nio_ch_IOUtil,
        'sun/nio/ch/FileDispatcherImpl': sun_nio_ch_FileDispatcherImpl,
        'sun/nio/fs/UnixNativeDispatcher': sun_nio_fs_UnixNativeDispatcher
    };
}
exports.__esModule = true;
exports["default"] = default_1;
;
//# sourceMappingURL=data:application/json;base64,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