"use strict";
/**
 * Contains JVM opcode implementations for the JVM interpreter.
 */
// We use snake case for the opcode names so they match the JVM spec.
// As for typedef:memberVariableDeclarator, we disable this so we can have
// member variable opcodes without explicitly typing them as IOpcodeImplementation.
/* tslint:disable:variable-name typedef:memberVariableDeclarator */
var gLong_1 = require('./gLong');
var util_1 = require('./util');
var enums_1 = require('./enums');
var assert_1 = require('./assert');
/**
 * Helper function: Checks if object is null. Throws a NullPointerException
 * if it is.
 * @return True if the object is null.
 */
function isNull(thread, frame, obj) {
    if (obj == null) {
        throwException(thread, frame, 'Ljava/lang/NullPointerException;', '');
        return true;
    }
    return false;
}
exports.isNull = isNull;
/**
 * Helper function: Pops off two items, returns the second.
 */
function pop2(opStack) {
    // Ignore NULL.
    opStack.pop();
    return opStack.pop();
}
exports.pop2 = pop2;
function resolveCPItem(thread, frame, cpItem) {
    thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
    cpItem.resolve(thread, frame.getLoader(), frame.method.cls, function (status) {
        if (status) {
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
        }
    }, false);
    frame.returnToThreadLoop = true;
}
exports.resolveCPItem = resolveCPItem;
function initializeClassFromClass(thread, frame, cls) {
    thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
    cls.initialize(thread, function (cdata) {
        if (cdata != null) {
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
        }
    }, false);
    frame.returnToThreadLoop = true;
}
exports.initializeClassFromClass = initializeClassFromClass;
/**
 * Helper function: Pauses the thread and initializes a class.
 */
function initializeClass(thread, frame, clsRef) {
    thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
    function initialize(cls) {
        cls.initialize(thread, function (cdata) {
            if (cdata != null) {
                thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            }
        });
    }
    if (!clsRef.isResolved()) {
        clsRef.resolve(thread, frame.getLoader(), frame.method.cls, function (status) {
            if (status) {
                initialize(clsRef.cls);
            }
        }, false);
    }
    else {
        initialize(clsRef.cls);
    }
    frame.returnToThreadLoop = true;
}
exports.initializeClass = initializeClass;
/**
 * Interrupts the current method's execution and throws an exception.
 *
 * NOTE: This does *not* interrupt JavaScript control flow, so any opcode
 * calling this function must *return* and not do anything else.
 */
function throwException(thread, frame, clsName, msg) {
    thread.throwNewException(clsName, msg);
    frame.returnToThreadLoop = true;
}
exports.throwException = throwException;
exports.ArrayTypes = {
    4: 'Z', 5: 'C', 6: 'F', 7: 'D', 8: 'B', 9: 'S', 10: 'I', 11: 'J'
};
/**
 * Contains definitions for all JVM opcodes.
 */
var Opcodes = (function () {
    function Opcodes() {
    }
    /* 32-bit array load opcodes */
    /**
     * 32-bit array load opcode
     */
    Opcodes._aload_32 = function (thread, frame) {
        var opStack = frame.opStack, idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + " not in length " + len + " array of type " + obj.getClass().getInternalName());
            }
            else {
                opStack.push(obj.array[idx]);
                frame.pc++;
            }
        }
        // 'obj' is NULL. isNull threw an exception for us.
    };
    /* 64-bit array load opcodes */
    /**
     * 64-bit array load opcode.
     */
    Opcodes._aload_64 = function (thread, frame) {
        var opStack = frame.opStack, idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + " not in length " + len + " array of type " + obj.getClass().getInternalName());
            }
            else {
                opStack.push(obj.array[idx]);
                // 64-bit value.
                opStack.push(null);
                frame.pc++;
            }
        }
        // 'obj' is NULL. isNull threw an exception for us.
    };
    /* 32-bit array store opcodes */
    /**
     * 32-bit array store.
     * @private
     */
    Opcodes._astore_32 = function (thread, frame) {
        var opStack = frame.opStack, value = opStack.pop(), idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + " not in length " + len + " array of type " + obj.getClass().getInternalName());
            }
            else {
                obj.array[idx] = value;
                frame.pc++;
            }
        }
        // 'obj' is NULL. isNull threw an exception for us.
    };
    /* 64-bit array store opcodes */
    /**
     * 64-bit array store.
     * @private
     */
    Opcodes._astore_64 = function (thread, frame) {
        var opStack = frame.opStack, value = opStack.pop2(), idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + " not in length " + len + " array of type " + obj.getClass().getInternalName());
            }
            else {
                obj.array[idx] = value;
                frame.pc++;
            }
        }
        // 'obj' is NULL. isNull threw an exception for us.
    };
    /* 32-bit constants */
    Opcodes.aconst_null = function (thread, frame) {
        frame.opStack.push(null);
        frame.pc++;
    };
    Opcodes._const_0_32 = function (thread, frame) {
        frame.opStack.push(0);
        frame.pc++;
    };
    Opcodes._const_1_32 = function (thread, frame) {
        frame.opStack.push(1);
        frame.pc++;
    };
    Opcodes._const_2_32 = function (thread, frame) {
        frame.opStack.push(2);
        frame.pc++;
    };
    Opcodes.iconst_m1 = function (thread, frame) {
        frame.opStack.push(-1);
        frame.pc++;
    };
    Opcodes.iconst_3 = function (thread, frame) {
        frame.opStack.push(3);
        frame.pc++;
    };
    Opcodes.iconst_4 = function (thread, frame) {
        frame.opStack.push(4);
        frame.pc++;
    };
    Opcodes.iconst_5 = function (thread, frame) {
        frame.opStack.push(5);
        frame.pc++;
    };
    /* 64-bit constants */
    Opcodes.lconst_0 = function (thread, frame) {
        frame.opStack.pushWithNull(gLong_1["default"].ZERO);
        frame.pc++;
    };
    Opcodes.lconst_1 = function (thread, frame) {
        frame.opStack.pushWithNull(gLong_1["default"].ONE);
        frame.pc++;
    };
    Opcodes.dconst_0 = function (thread, frame) {
        frame.opStack.pushWithNull(0);
        frame.pc++;
    };
    Opcodes.dconst_1 = function (thread, frame) {
        frame.opStack.pushWithNull(1);
        frame.pc++;
    };
    /* 32-bit load opcodes */
    Opcodes._load_32 = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(frame.locals[code[pc + 1]]);
        frame.pc += 2;
    };
    Opcodes._load_0_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[0]);
        frame.pc++;
    };
    Opcodes._load_1_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[1]);
        frame.pc++;
    };
    Opcodes._load_2_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[2]);
        frame.pc++;
    };
    Opcodes._load_3_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[3]);
        frame.pc++;
    };
    /* 64-bit load opcodes */
    Opcodes._load_64 = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.pushWithNull(frame.locals[code[pc + 1]]);
        frame.pc += 2;
    };
    Opcodes._load_0_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[0]);
        frame.pc++;
    };
    Opcodes._load_1_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[1]);
        frame.pc++;
    };
    Opcodes._load_2_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[2]);
        frame.pc++;
    };
    Opcodes._load_3_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[3]);
        frame.pc++;
    };
    /* 32-bit store opcodes */
    Opcodes._store_32 = function (thread, frame, code) {
        var pc = frame.pc;
        frame.locals[code[pc + 1]] = frame.opStack.pop();
        frame.pc += 2;
    };
    Opcodes._store_0_32 = function (thread, frame) {
        frame.locals[0] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_1_32 = function (thread, frame) {
        frame.locals[1] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_2_32 = function (thread, frame) {
        frame.locals[2] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_3_32 = function (thread, frame) {
        frame.locals[3] = frame.opStack.pop();
        frame.pc++;
    };
    /* 64-bit store opcodes */
    Opcodes._store_64 = function (thread, frame, code) {
        var pc = frame.pc;
        var offset = code[pc + 1];
        // NULL
        frame.locals[offset + 1] = frame.opStack.pop();
        // The actual value.
        frame.locals[offset] = frame.opStack.pop();
        frame.pc += 2;
    };
    Opcodes._store_0_64 = function (thread, frame) {
        frame.locals[1] = frame.opStack.pop();
        frame.locals[0] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_1_64 = function (thread, frame) {
        frame.locals[2] = frame.opStack.pop();
        frame.locals[1] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_2_64 = function (thread, frame) {
        frame.locals[3] = frame.opStack.pop();
        frame.locals[2] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_3_64 = function (thread, frame) {
        frame.locals[4] = frame.opStack.pop();
        frame.locals[3] = frame.opStack.pop();
        frame.pc++;
    };
    /* Misc. */
    Opcodes.sipush = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(code.readInt16BE(pc + 1));
        frame.pc += 3;
    };
    Opcodes.bipush = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(code.readInt8(pc + 1));
        frame.pc += 2;
    };
    Opcodes.pop = function (thread, frame) {
        frame.opStack.dropFromTop(1);
        frame.pc++;
    };
    Opcodes.pop2 = function (thread, frame) {
        // http://i.imgur.com/MieF0KG.jpg
        frame.opStack.dropFromTop(2);
        frame.pc++;
    };
    Opcodes.dup = function (thread, frame) {
        frame.opStack.dup();
        frame.pc++;
    };
    Opcodes.dup_x1 = function (thread, frame) {
        frame.opStack.dup_x1();
        frame.pc++;
    };
    Opcodes.dup_x2 = function (thread, frame) {
        frame.opStack.dup_x2();
        frame.pc++;
    };
    Opcodes.dup2 = function (thread, frame) {
        frame.opStack.dup2();
        frame.pc++;
    };
    Opcodes.dup2_x1 = function (thread, frame) {
        frame.opStack.dup2_x1();
        frame.pc++;
    };
    Opcodes.dup2_x2 = function (thread, frame) {
        var opStack = frame.opStack, v1 = opStack.pop(), v2 = opStack.pop(), v3 = opStack.pop(), v4 = opStack.pop();
        opStack.push6(v2, v1, v4, v3, v2, v1);
        frame.pc++;
    };
    Opcodes.swap = function (thread, frame) {
        frame.opStack.swap();
        frame.pc++;
    };
    /* Math Opcodes */
    Opcodes.iadd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push((opStack.pop() + opStack.pop()) | 0);
        frame.pc++;
    };
    Opcodes.ladd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().add(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.fadd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.wrapFloat(opStack.pop() + opStack.pop()));
        frame.pc++;
    };
    Opcodes.dadd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2() + opStack.pop2());
        frame.pc++;
    };
    Opcodes.isub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push((-opStack.pop() + opStack.pop()) | 0);
        frame.pc++;
    };
    Opcodes.fsub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.wrapFloat(-opStack.pop() + opStack.pop()));
        frame.pc++;
    };
    Opcodes.dsub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(-opStack.pop2() + opStack.pop2());
        frame.pc++;
    };
    Opcodes.lsub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().negate().add(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.imul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(Math.imul(opStack.pop(), opStack.pop()));
        frame.pc++;
    };
    Opcodes.lmul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().multiply(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.fmul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.wrapFloat(opStack.pop() * opStack.pop()));
        frame.pc++;
    };
    Opcodes.dmul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2() * opStack.pop2());
        frame.pc++;
    };
    Opcodes.idiv = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop(), a = opStack.pop();
        if (b === 0) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        }
        else {
            // spec: "if the dividend is the negative integer of largest possible magnitude
            // for the int type, and the divisor is -1, then overflow occurs, and the
            // result is equal to the dividend."
            if (a === enums_1.Constants.INT_MIN && b === -1) {
                opStack.push(a);
            }
            else {
                opStack.push((a / b) | 0);
            }
            frame.pc++;
        }
    };
    Opcodes.ldiv = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop2(), a = opStack.pop2();
        if (b.isZero()) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        }
        else {
            opStack.pushWithNull(a.div(b));
            frame.pc++;
        }
    };
    Opcodes.fdiv = function (thread, frame) {
        var opStack = frame.opStack, a = opStack.pop();
        opStack.push(util_1.wrapFloat(opStack.pop() / a));
        frame.pc++;
    };
    Opcodes.ddiv = function (thread, frame) {
        var opStack = frame.opStack, v = opStack.pop2();
        opStack.pushWithNull(opStack.pop2() / v);
        frame.pc++;
    };
    Opcodes.irem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop(), a = opStack.pop();
        if (b === 0) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        }
        else {
            opStack.push(a % b);
            frame.pc++;
        }
    };
    Opcodes.lrem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop2(), a = opStack.pop2();
        if (b.isZero()) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        }
        else {
            opStack.pushWithNull(a.modulo(b));
            frame.pc++;
        }
    };
    Opcodes.frem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop();
        opStack.push(opStack.pop() % b);
        frame.pc++;
    };
    Opcodes.drem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop2();
        opStack.pushWithNull(opStack.pop2() % b);
        frame.pc++;
    };
    Opcodes.ineg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(-opStack.pop() | 0);
        frame.pc++;
    };
    Opcodes.lneg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().negate());
        frame.pc++;
    };
    Opcodes.fneg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(-opStack.pop());
        frame.pc++;
    };
    Opcodes.dneg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(-opStack.pop2());
        frame.pc++;
    };
    /* Bitwise Operations */
    Opcodes.ishl = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.push(opStack.pop() << s);
        frame.pc++;
    };
    Opcodes.lshl = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.pushWithNull(opStack.pop2().shiftLeft(gLong_1["default"].fromInt(s)));
        frame.pc++;
    };
    Opcodes.ishr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.push(opStack.pop() >> s);
        frame.pc++;
    };
    Opcodes.lshr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.pushWithNull(opStack.pop2().shiftRight(gLong_1["default"].fromInt(s)));
        frame.pc++;
    };
    Opcodes.iushr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.push((opStack.pop() >>> s) | 0);
        frame.pc++;
    };
    Opcodes.lushr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.pushWithNull(opStack.pop2().shiftRightUnsigned(gLong_1["default"].fromInt(s)));
        frame.pc++;
    };
    Opcodes.iand = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() & opStack.pop());
        frame.pc++;
    };
    Opcodes.land = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().and(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.ior = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() | opStack.pop());
        frame.pc++;
    };
    Opcodes.lor = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().or(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.ixor = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() ^ opStack.pop());
        frame.pc++;
    };
    Opcodes.lxor = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().xor(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.iinc = function (thread, frame, code) {
        var pc = frame.pc;
        var idx = code[pc + 1], val = code.readInt8(pc + 2);
        frame.locals[idx] = (frame.locals[idx] + val) | 0;
        frame.pc += 3;
    };
    Opcodes.i2l = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(gLong_1["default"].fromInt(opStack.pop()));
        frame.pc++;
    };
    Opcodes.i2f = function (thread, frame) {
        // NOP; we represent ints as floats anyway.
        // @todo What about quantities unexpressable as floats?
        frame.pc++;
    };
    Opcodes.i2d = function (thread, frame) {
        frame.opStack.push(null);
        frame.pc++;
    };
    Opcodes.l2i = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop2().toInt());
        frame.pc++;
    };
    Opcodes.l2f = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop2().toNumber());
        frame.pc++;
    };
    Opcodes.l2d = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().toNumber());
        frame.pc++;
    };
    Opcodes.f2i = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.float2int(opStack.pop()));
        frame.pc++;
    };
    Opcodes.f2l = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(gLong_1["default"].fromNumber(opStack.pop()));
        frame.pc++;
    };
    Opcodes.f2d = function (thread, frame) {
        frame.opStack.push(null);
        frame.pc++;
    };
    Opcodes.d2i = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.float2int(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.d2l = function (thread, frame) {
        var opStack = frame.opStack, d_val = opStack.pop2();
        if (d_val === Number.POSITIVE_INFINITY) {
            opStack.pushWithNull(gLong_1["default"].MAX_VALUE);
        }
        else if (d_val === Number.NEGATIVE_INFINITY) {
            opStack.pushWithNull(gLong_1["default"].MIN_VALUE);
        }
        else {
            opStack.pushWithNull(gLong_1["default"].fromNumber(d_val));
        }
        frame.pc++;
    };
    Opcodes.d2f = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pop();
        opStack.push(util_1.wrapFloat(opStack.pop()));
        frame.pc++;
    };
    Opcodes.i2b = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push((opStack.pop() << 24) >> 24);
        frame.pc++;
    };
    Opcodes.i2c = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() & 0xFFFF);
        frame.pc++;
    };
    Opcodes.i2s = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push((opStack.pop() << 16) >> 16);
        frame.pc++;
    };
    Opcodes.lcmp = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop2();
        opStack.push(opStack.pop2().compare(v2));
        frame.pc++;
    };
    Opcodes.fcmpl = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop(), v1 = opStack.pop();
        if (v1 === v2) {
            opStack.push(0);
        }
        else if (v1 > v2) {
            opStack.push(1);
        }
        else {
            // v1 < v2, and if v1 or v2 is NaN.
            opStack.push(-1);
        }
        frame.pc++;
    };
    Opcodes.fcmpg = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop(), v1 = opStack.pop();
        if (v1 === v2) {
            opStack.push(0);
        }
        else if (v1 < v2) {
            opStack.push(-1);
        }
        else {
            // v1 > v2, and if v1 or v2 is NaN.
            opStack.push(1);
        }
        frame.pc++;
    };
    Opcodes.dcmpl = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop2(), v1 = opStack.pop2();
        if (v1 === v2) {
            opStack.push(0);
        }
        else if (v1 > v2) {
            opStack.push(1);
        }
        else {
            // v1 < v2, and if v1 or v2 is NaN.
            opStack.push(-1);
        }
        frame.pc++;
    };
    Opcodes.dcmpg = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop2(), v1 = opStack.pop2();
        if (v1 === v2) {
            opStack.push(0);
        }
        else if (v1 < v2) {
            opStack.push(-1);
        }
        else {
            // v1 > v2, and if v1 or v2 is NaN.
            opStack.push(1);
        }
        frame.pc++;
    };
    /* Unary branch opcodes */
    Opcodes.ifeq = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() === 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.ifne = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() !== 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.iflt = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() < 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.ifge = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() >= 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.ifgt = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() > 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.ifle = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() <= 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    /* Binary branch opcodes */
    Opcodes.if_icmpeq = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 === v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpne = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 !== v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmplt = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 < v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpge = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 >= v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpgt = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 > v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmple = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 <= v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_acmpeq = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 === v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.if_acmpne = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 !== v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    /* Jump opcodes */
    Opcodes.goto = function (thread, frame, code) {
        var pc = frame.pc;
        var offset = code.readInt16BE(pc + 1);
        frame.pc += offset;
        if (offset < 0) {
            frame.method.incrBBEntries();
        }
    };
    Opcodes.jsr = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(pc + 3);
        var offset = code.readInt16BE(pc + 1);
        frame.pc += offset;
        if (offset < 0) {
            frame.method.incrBBEntries();
        }
    };
    Opcodes.ret = function (thread, frame, code) {
        var pc = frame.pc;
        frame.pc = frame.locals[code[pc + 1]];
    };
    Opcodes.tableswitch = function (thread, frame, code) {
        var pc = frame.pc;
        // Ignore padding bytes. The +1 is to skip the opcode byte.
        pc += ((4 - (pc + 1) % 4) % 4) + 1;
        var defaultOffset = code.readInt32BE(pc), low = code.readInt32BE(pc + 4), high = code.readInt32BE(pc + 8), offset = frame.opStack.pop();
        if (offset >= low && offset <= high) {
            frame.pc += code.readInt32BE(pc + 12 + ((offset - low) * 4));
        }
        else {
            frame.pc += defaultOffset;
        }
    };
    Opcodes.lookupswitch = function (thread, frame, code) {
        var pc = frame.pc;
        // Skip padding bytes. The +1 is to skip the opcode byte.
        pc += ((4 - (pc + 1) % 4) % 4) + 1;
        var defaultOffset = code.readInt32BE(pc), nPairs = code.readInt32BE(pc + 4), i, v = frame.opStack.pop();
        pc += 8;
        for (i = 0; i < nPairs; i++) {
            if (code.readInt32BE(pc) === v) {
                var offset = code.readInt32BE(pc + 4);
                frame.pc += offset;
                if (offset < 0) {
                    frame.method.incrBBEntries();
                }
                return;
            }
            pc += 8;
        }
        // No match found.
        frame.pc += defaultOffset;
    };
    Opcodes.return = function (thread, frame) {
        frame.returnToThreadLoop = true;
        if (frame.method.accessFlags.isSynchronized()) {
            // monitorexit
            if (!frame.method.methodLock(thread, frame).exit(thread)) {
                // monitorexit threw an exception.
                return;
            }
        }
        thread.asyncReturn();
    };
    /* 32-bit return bytecodes */
    Opcodes._return_32 = function (thread, frame) {
        frame.returnToThreadLoop = true;
        if (frame.method.accessFlags.isSynchronized()) {
            // monitorexit
            if (!frame.method.methodLock(thread, frame).exit(thread)) {
                // monitorexit threw an exception.
                return;
            }
        }
        thread.asyncReturn(frame.opStack.bottom());
    };
    /* 64-bit return opcodes */
    Opcodes._return_64 = function (thread, frame) {
        frame.returnToThreadLoop = true;
        if (frame.method.accessFlags.isSynchronized()) {
            // monitorexit
            if (!frame.method.methodLock(thread, frame).exit(thread)) {
                // monitorexit threw an exception.
                return;
            }
        }
        thread.asyncReturn(frame.opStack.bottom(), null);
    };
    Opcodes.getstatic = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        assert_1["default"](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        if (fieldInfo.isResolved()) {
            // Get the *actual* class that owns this field.
            // This may not be initialized if it's an interface, so we need to check.
            var fieldOwnerCls = fieldInfo.field.cls;
            if (fieldOwnerCls.isInitialized(thread)) {
                // Opcode is ready to execute! Rewrite to a 'fast' version,
                // and run the fast version.
                if (fieldInfo.nameAndTypeInfo.descriptor === 'J' || fieldInfo.nameAndTypeInfo.descriptor === 'D') {
                    code[pc] = enums_1.OpCode.GETSTATIC_FAST64;
                }
                else {
                    code[pc] = enums_1.OpCode.GETSTATIC_FAST32;
                }
                // Stash the result of field lookup.
                fieldInfo.fieldOwnerConstructor = fieldOwnerCls.getConstructor(thread);
            }
            else {
                // Initialize class and rerun opcode
                initializeClassFromClass(thread, frame, fieldOwnerCls);
            }
        }
        else {
            // Resolve the field.
            resolveCPItem(thread, frame, fieldInfo);
        }
    };
    /**
     * A fast version of getstatic that assumes that relevant classes are
     * initialized.
     *
     * Retrieves a 32-bit value.
     */
    Opcodes.getstatic_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        frame.opStack.push(fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName]);
        frame.pc += 3;
    };
    /**
     * A fast version of getstatic that assumes that relevant classes are
     * initialized.
     *
     * Retrieves a 64-bit value.
     */
    Opcodes.getstatic_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        frame.opStack.pushWithNull(fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName]);
        frame.pc += 3;
    };
    Opcodes.putstatic = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        assert_1["default"](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        if (fieldInfo.isResolved()) {
            // Get the *actual* class that owns this field.
            // This may not be initialized if it's an interface, so we need to check.
            var fieldOwnerCls = fieldInfo.field.cls;
            if (fieldOwnerCls.isInitialized(thread)) {
                // Opcode is ready to execute! Rewrite to a 'fast' version,
                // and run the fast version.
                if (fieldInfo.nameAndTypeInfo.descriptor === 'J' || fieldInfo.nameAndTypeInfo.descriptor === 'D') {
                    code[pc] = enums_1.OpCode.PUTSTATIC_FAST64;
                }
                else {
                    code[pc] = enums_1.OpCode.PUTSTATIC_FAST32;
                }
                // Stash the result of field lookup.
                fieldInfo.fieldOwnerConstructor = fieldOwnerCls.getConstructor(thread);
            }
            else {
                // Initialize class and rerun opcode
                initializeClassFromClass(thread, frame, fieldOwnerCls);
            }
        }
        else {
            // Resolve the field.
            resolveCPItem(thread, frame, fieldInfo);
        }
    };
    /**
     * A fast version of putstatic that assumes that relevant classes are
     * initialized.
     *
     * Puts a 32-bit value.
     */
    Opcodes.putstatic_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName] = frame.opStack.pop();
        frame.pc += 3;
    };
    /**
     * A fast version of putstatic that assumes that relevant classes are
     * initialized.
     *
     * Puts a 64-bit value.
     */
    Opcodes.putstatic_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName] = frame.opStack.pop2();
        frame.pc += 3;
    };
    Opcodes.getfield = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), loader = frame.getLoader(), obj = frame.opStack.top();
        assert_1["default"](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        // Check if the object is null; if we do not do this before get_class, then
        // we might try to get a class that we have not initialized!
        if (!isNull(thread, frame, obj)) {
            // cls is guaranteed to be in the inheritance hierarchy of obj, so it must be
            // initialized. However, it may not be loaded in the current class's
            // ClassLoader...
            if (fieldInfo.isResolved()) {
                var field = fieldInfo.field;
                if (field.rawDescriptor == 'J' || field.rawDescriptor == 'D') {
                    code[pc] = enums_1.OpCode.GETFIELD_FAST64;
                }
                else {
                    code[pc] = enums_1.OpCode.GETFIELD_FAST32;
                }
            }
            else {
                resolveCPItem(thread, frame, fieldInfo);
            }
        }
    };
    Opcodes.getfield_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            opStack.push(obj[fieldInfo.fullFieldName]);
            frame.pc += 3;
        }
    };
    Opcodes.getfield_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            opStack.pushWithNull(obj[fieldInfo.fullFieldName]);
            frame.pc += 3;
        }
    };
    Opcodes.putfield = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), loader = frame.getLoader(), isLong = fieldInfo.nameAndTypeInfo.descriptor == 'J' || fieldInfo.nameAndTypeInfo.descriptor == 'D', obj = frame.opStack.fromTop(isLong ? 2 : 1);
        assert_1["default"](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        // Check if the object is null; if we do not do this before get_class, then
        // we might try to get a class that we have not initialized!
        if (!isNull(thread, frame, obj)) {
            // cls is guaranteed to be in the inheritance hierarchy of obj, so it must be
            // initialized. However, it may not be loaded in the current class's
            // ClassLoader...
            if (fieldInfo.isResolved()) {
                var field = fieldInfo.field;
                if (isLong) {
                    code[pc] = enums_1.OpCode.PUTFIELD_FAST64;
                }
                else {
                    code[pc] = enums_1.OpCode.PUTFIELD_FAST32;
                }
                // Stash the resolved full field name.
                fieldInfo.fullFieldName = util_1.descriptor2typestr(field.cls.getInternalName()) + "/" + fieldInfo.nameAndTypeInfo.name;
            }
            else {
                resolveCPItem(thread, frame, fieldInfo);
            }
        }
    };
    Opcodes.putfield_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, val = opStack.pop(), obj = opStack.pop(), fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (!isNull(thread, frame, obj)) {
            obj[fieldInfo.fullFieldName] = val;
            frame.pc += 3;
        }
        // NPE has been thrown.
    };
    Opcodes.putfield_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, val = opStack.pop2(), obj = opStack.pop(), fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (!isNull(thread, frame, obj)) {
            obj[fieldInfo.fullFieldName] = val;
            frame.pc += 3;
        }
        // NPE has been thrown.
    };
    Opcodes.invokevirtual = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        // Ensure referenced class is loaded in the current classloader.
        // Even though we don't use this class for anything, and we know that it
        // must be loaded because it is in the object's inheritance hierarchy,
        // it needs to be present in the current classloader.
        if (methodReference.isResolved()) {
            var m = methodReference.method;
            if (m.isSignaturePolymorphic()) {
                switch (m.name) {
                    case 'invokeBasic':
                        code[pc] = enums_1.OpCode.INVOKEBASIC;
                        break;
                    case 'invoke':
                    case 'invokeExact':
                        code[pc] = enums_1.OpCode.INVOKEHANDLE;
                        break;
                    default:
                        throwException(thread, frame, 'Ljava/lang/AbstractMethodError;', "Invalid signature polymorphic method: " + m.cls.getExternalName() + "." + m.name);
                        break;
                }
            }
            else {
                code[pc] = enums_1.OpCode.INVOKEVIRTUAL_FAST;
            }
        }
        else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokeinterface = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            if (methodReference.method.cls.isInitialized(thread)) {
                // Rewrite to fast and rerun.
                code[pc] = enums_1.OpCode.INVOKEINTERFACE_FAST;
            }
            else {
                // Initialize our class and rerun opcode.
                // Note that the existance of an object of an interface type does *not*
                // mean that the interface is initialized!
                initializeClass(thread, frame, methodReference.classInfo);
            }
        }
        else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokedynamic = function (thread, frame, code) {
        var pc = frame.pc;
        var callSiteSpecifier = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
        callSiteSpecifier.constructCallSiteObject(thread, frame.getLoader(), frame.method.cls, pc, function (status) {
            if (status) {
                assert_1["default"](typeof (callSiteSpecifier.getCallSiteObject(pc)[0].vmtarget) === 'function', "MethodName should be resolved...");
                code[pc] = enums_1.OpCode.INVOKEDYNAMIC_FAST;
                // Resume and rerun fast opcode.
                thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            }
        });
        frame.returnToThreadLoop = true;
    };
    /**
     * XXX: Actually perform superclass method lookup.
     */
    Opcodes.invokespecial = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            // Rewrite and rerun.
            code[pc] = enums_1.OpCode.INVOKENONVIRTUAL_FAST;
        }
        else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokestatic = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            var m = methodReference.method;
            if (m.cls.isInitialized(thread)) {
                var newOpcode = enums_1.OpCode.INVOKESTATIC_FAST;
                if (methodReference.method.isSignaturePolymorphic()) {
                    switch (methodReference.method.name) {
                        case 'linkToInterface':
                        case 'linkToVirtual':
                            newOpcode = enums_1.OpCode.LINKTOVIRTUAL;
                            break;
                        case 'linkToStatic':
                        case 'linkToSpecial':
                            newOpcode = enums_1.OpCode.LINKTOSPECIAL;
                            break;
                        default:
                            assert_1["default"](false, "Should be impossible.");
                            break;
                    }
                }
                // Rewrite and rerun.
                code[pc] = newOpcode;
            }
            else {
                initializeClassFromClass(thread, frame, m.cls);
            }
        }
        else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    /// Fast invoke opcodes.
    Opcodes.invokenonvirtual_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, paramSize = methodReference.paramWordSize, obj = opStack.fromTop(paramSize);
        if (!isNull(thread, frame, obj)) {
            var args = opStack.sliceFromTop(paramSize);
            opStack.dropFromTop(paramSize + 1);
            assert_1["default"](typeof obj[methodReference.fullSignature] === 'function', "Resolved method " + methodReference.fullSignature + " isn't defined?!", thread);
            obj[methodReference.fullSignature](thread, args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.invokestatic_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, paramSize = methodReference.paramWordSize, args = opStack.sliceAndDropFromTop(paramSize);
        assert_1["default"](methodReference.jsConstructor != null, "jsConstructor is missing?!");
        assert_1["default"](typeof (methodReference.jsConstructor[methodReference.fullSignature]) === 'function', "Resolved method isn't defined?!");
        methodReference.jsConstructor[methodReference.fullSignature](thread, args);
        frame.returnToThreadLoop = true;
    };
    Opcodes.invokevirtual_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), count = methodReference.paramWordSize, opStack = frame.opStack, obj = opStack.fromTop(count);
        if (!isNull(thread, frame, obj)) {
            // Use the class of the *object*.
            assert_1["default"](typeof obj[methodReference.signature] === 'function', "Resolved method " + methodReference.signature + " isn't defined?!");
            obj[methodReference.signature](thread, opStack.sliceFromTop(count));
            opStack.dropFromTop(count + 1);
            frame.returnToThreadLoop = true;
        }
        // Object is NULL; NPE has been thrown.
    };
    Opcodes.invokedynamic_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var callSiteSpecifier = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), cso = callSiteSpecifier.getCallSiteObject(pc), appendix = cso[1], fcn = cso[0].vmtarget, opStack = frame.opStack, paramSize = callSiteSpecifier.paramWordSize, args = opStack.sliceAndDropFromTop(paramSize);
        if (appendix !== null) {
            args.push(appendix);
        }
        fcn(thread, null, args);
        frame.returnToThreadLoop = true;
    };
    /**
     * Opcode for MethodHandle.invoke and MethodHandle.invokeExact.
     */
    Opcodes.invokehandle = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, fcn = methodReference.memberName.vmtarget, 
        // Add in 1 for the method handle itself.
        paramSize = methodReference.paramWordSize + 1, appendix = methodReference.appendix, args = opStack.sliceFromTop(paramSize);
        if (appendix !== null) {
            args.push(appendix);
        }
        if (!isNull(thread, frame, args[0])) {
            opStack.dropFromTop(paramSize);
            // fcn will handle invoking 'this' and such.
            // TODO: If this can be varargs, pass in parameter types to the function.
            fcn(thread, null, args);
            frame.returnToThreadLoop = true;
        }
    };
    /**
     * Opcode for MethodHandle.invokeBasic.
     * Unlike invoke/invokeExact, invokeBasic does not call a generated bytecode
     * method. It calls the vmtarget embedded in the MethodHandler directly.
     * This can cause crashes with malformed calls, thus it is only accesssible
     * to trusted JDK code.
     */
    Opcodes.invokebasic = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), paramSize = methodReference.getParamWordSize(), opStack = frame.opStack, obj = opStack.fromTop(paramSize), 
        // Need to include the MethodHandle in the arguments to vmtarget. vmtarget
        // will appropriately invoke it.
        args = opStack.sliceFromTop(paramSize + 1), lmbdaForm, mn, m;
        // obj is a MethodHandle.
        if (!isNull(thread, frame, obj)) {
            opStack.dropFromTop(paramSize + 1);
            lmbdaForm = obj['java/lang/invoke/MethodHandle/form'];
            mn = lmbdaForm['java/lang/invoke/LambdaForm/vmentry'];
            assert_1["default"](mn.vmtarget !== null && mn.vmtarget !== undefined, "vmtarget must be defined");
            mn.vmtarget(thread, methodReference.nameAndTypeInfo.descriptor, args);
            frame.returnToThreadLoop = true;
        }
    };
    /**
     * Also used for linkToStatic.
     * TODO: De-conflate the two.
     * TODO: Varargs functions.
     */
    Opcodes.linktospecial = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, paramSize = methodReference.paramWordSize, 
        // Final argument is the relevant MemberName. Function args are right
        // before it.
        args = opStack.sliceFromTop(paramSize), memberName = args.pop(), 
        // TODO: Use parsed descriptor.
        desc = methodReference.nameAndTypeInfo.descriptor;
        if (!isNull(thread, frame, memberName)) {
            opStack.dropFromTop(paramSize);
            assert_1["default"](memberName.getClass().getInternalName() === "Ljava/lang/invoke/MemberName;");
            // parameterTypes for function are the same as the method reference, but without the trailing MemberName.
            // TODO: Use parsed descriptor, avoid re-doing work here.
            memberName.vmtarget(thread, desc.replace("Ljava/lang/invoke/MemberName;)", ")"), args);
            frame.returnToThreadLoop = true;
        }
    };
    // XXX: Varargs functions. We're supposed to box args if target is varargs.
    Opcodes.linktovirtual = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), paramSize = methodReference.paramWordSize, opStack = frame.opStack, args = opStack.sliceFromTop(paramSize), 
        // Final argument is the relevant MemberName. Function args are right
        // before it.
        memberName = args.pop(), desc = methodReference.nameAndTypeInfo.descriptor;
        if (!isNull(thread, frame, memberName)) {
            opStack.dropFromTop(paramSize);
            assert_1["default"](memberName.getClass().getInternalName() === "Ljava/lang/invoke/MemberName;");
            // parameterTypes for function are the same as the method reference, but without the trailing MemberName.
            memberName.vmtarget(thread, desc.replace("Ljava/lang/invoke/MemberName;)", ")"), args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.breakpoint = function (thread, frame) {
        throwException(thread, frame, "Ljava/lang/Error;", "breakpoint not implemented.");
    };
    Opcodes.new = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            var cls = classRef.cls;
            if (cls.isInitialized(thread)) {
                code[pc] = enums_1.OpCode.NEW_FAST;
            }
            else {
                initializeClassFromClass(thread, frame, cls);
            }
        }
        else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.new_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        frame.opStack.push(new classRef.clsConstructor(thread));
        frame.pc += 3;
    };
    Opcodes.newarray = function (thread, frame, code) {
        var pc = frame.pc;
        // TODO: Stash all of these array types during JVM startup.
        var opStack = frame.opStack, type = "[" + exports.ArrayTypes[code[pc + 1]], cls = frame.getLoader().getInitializedClass(thread, type), length = opStack.pop();
        if (length >= 0) {
            opStack.push(new (cls.getConstructor(thread))(thread, length));
            frame.pc += 2;
        }
        else {
            throwException(thread, frame, 'Ljava/lang/NegativeArraySizeException;', "Tried to init " + type + " array with length " + length);
        }
    };
    Opcodes.anewarray = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            // Rewrite and rerun.
            code[pc] = enums_1.OpCode.ANEWARRAY_FAST;
            classRef.arrayClass = frame.getLoader().getInitializedClass(thread, "[" + classRef.cls.getInternalName());
            classRef.arrayClassConstructor = classRef.arrayClass.getConstructor(thread);
        }
        else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.anewarray_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), length = opStack.pop();
        if (length >= 0) {
            opStack.push(new classRef.arrayClassConstructor(thread, length));
            frame.pc += 3;
        }
        else {
            throwException(thread, frame, 'Ljava/lang/NegativeArraySizeException;', "Tried to init " + classRef.arrayClass.getInternalName() + " array with length " + length);
        }
    };
    Opcodes.arraylength = function (thread, frame) {
        var opStack = frame.opStack, obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            opStack.push(obj.array.length);
            frame.pc++;
        }
        // obj is NULL. isNull threw an exception for us.
    };
    Opcodes.athrow = function (thread, frame) {
        thread.throwException(frame.opStack.pop());
        frame.returnToThreadLoop = true;
    };
    Opcodes.checkcast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            // Rewrite to fast version, and re-execute.
            code[pc] = enums_1.OpCode.CHECKCAST_FAST;
        }
        else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.checkcast_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), cls = classRef.cls, opStack = frame.opStack, o = opStack.top();
        if ((o != null) && !o.getClass().isCastable(cls)) {
            var targetClass = cls.getExternalName();
            var candidateClass = o.getClass().getExternalName();
            throwException(thread, frame, 'Ljava/lang/ClassCastException;', candidateClass + " cannot be cast to " + targetClass);
        }
        else {
            // Success!
            frame.pc += 3;
        }
    };
    Opcodes.instanceof = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            // Rewrite and rerun.
            code[pc] = enums_1.OpCode.INSTANCEOF_FAST;
        }
        else {
            // Fetch class and rerun opcode.
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.instanceof_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), cls = classRef.cls, opStack = frame.opStack, o = opStack.pop();
        opStack.push(o !== null ? (o.getClass().isCastable(cls) ? 1 : 0) : 0);
        frame.pc += 3;
    };
    Opcodes.monitorenter = function (thread, frame) {
        var opStack = frame.opStack, monitorObj = opStack.pop(), monitorEntered = function () {
            // [Note: Thread is now in the RUNNABLE state.]
            // Increment the PC.
            frame.pc++;
        };
        if (!monitorObj.getMonitor().enter(thread, monitorEntered)) {
            // Opcode failed. monitorEntered will be run once we own the monitor.
            // The thread is now in the BLOCKED state. Tell the frame to return to
            // the thread loop.
            frame.returnToThreadLoop = true;
        }
        else {
            monitorEntered();
        }
    };
    Opcodes.monitorexit = function (thread, frame) {
        var monitorObj = frame.opStack.pop();
        if (monitorObj.getMonitor().exit(thread)) {
            frame.pc++;
        }
        else {
            // monitorexit failed, and threw an exception.
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.multianewarray = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            // Rewrite and rerun.
            code[pc] = enums_1.OpCode.MULTIANEWARRAY_FAST;
        }
        else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.multianewarray_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, dim = code[pc + 3], i, 
        // Arguments to the constructor.
        args = new Array(dim), dimSize;
        for (i = 0; i < dim; i++) {
            dimSize = opStack.pop();
            args[dim - i - 1] = dimSize;
            if (dimSize < 0) {
                throwException(thread, frame, 'Ljava/lang/NegativeArraySizeException;', "Tried to init " + classRef.cls.getInternalName() + " array with a dimension of length " + dimSize);
                return;
            }
        }
        opStack.push(new (classRef.cls.getConstructor(thread))(thread, args));
        frame.pc += 4;
    };
    Opcodes.ifnull = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() == null) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.ifnonnull = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() != null) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        }
        else {
            frame.pc += 3;
        }
    };
    Opcodes.goto_w = function (thread, frame, code) {
        var pc = frame.pc;
        var offset = code.readInt32BE(pc + 1);
        frame.pc += offset;
        if (offset < 0) {
            frame.method.incrBBEntries();
        }
    };
    Opcodes.jsr_w = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(frame.pc + 5);
        frame.pc += code.readInt32BE(pc + 1);
    };
    Opcodes.nop = function (thread, frame) {
        frame.pc += 1;
    };
    Opcodes.ldc = function (thread, frame, code) {
        var pc = frame.pc;
        var constant = frame.method.cls.constantPool.get(code[pc + 1]);
        if (constant.isResolved()) {
            assert_1["default"]((function () {
                switch (constant.getType()) {
                    case enums_1.ConstantPoolItemType.STRING:
                    case enums_1.ConstantPoolItemType.CLASS:
                    case enums_1.ConstantPoolItemType.METHOD_HANDLE:
                    case enums_1.ConstantPoolItemType.METHOD_TYPE:
                    case enums_1.ConstantPoolItemType.INTEGER:
                    case enums_1.ConstantPoolItemType.FLOAT:
                        return true;
                    default:
                        return false;
                }
            })(), "Constant pool item " + enums_1.ConstantPoolItemType[constant.getType()] + " is not appropriate for LDC.");
            frame.opStack.push(constant.getConstant(thread));
            frame.pc += 2;
        }
        else {
            resolveCPItem(thread, frame, constant);
        }
    };
    Opcodes.ldc_w = function (thread, frame, code) {
        var pc = frame.pc;
        var constant = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (constant.isResolved()) {
            assert_1["default"]((function () {
                switch (constant.getType()) {
                    case enums_1.ConstantPoolItemType.STRING:
                    case enums_1.ConstantPoolItemType.CLASS:
                    case enums_1.ConstantPoolItemType.METHOD_HANDLE:
                    case enums_1.ConstantPoolItemType.METHOD_TYPE:
                    case enums_1.ConstantPoolItemType.INTEGER:
                    case enums_1.ConstantPoolItemType.FLOAT:
                        return true;
                    default:
                        return false;
                }
            })(), "Constant pool item " + enums_1.ConstantPoolItemType[constant.getType()] + " is not appropriate for LDC_W.");
            frame.opStack.push(constant.getConstant(thread));
            frame.pc += 3;
        }
        else {
            resolveCPItem(thread, frame, constant);
        }
    };
    Opcodes.ldc2_w = function (thread, frame, code) {
        var pc = frame.pc;
        var constant = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        assert_1["default"](constant.getType() === enums_1.ConstantPoolItemType.LONG
            || constant.getType() === enums_1.ConstantPoolItemType.DOUBLE, "Invalid ldc_w constant pool type: " + enums_1.ConstantPoolItemType[constant.getType()]);
        frame.opStack.pushWithNull(constant.value);
        frame.pc += 3;
    };
    Opcodes.wide = function (thread, frame, code) {
        var pc = frame.pc;
        var index = code.readUInt16BE(pc + 2);
        // Increment PC before switch to avoid issue where ret chances PC and we
        // erroneously increment the PC further.
        frame.pc += 4;
        switch (code[pc + 1]) {
            case enums_1.OpCode.ILOAD:
            case enums_1.OpCode.FLOAD:
            case enums_1.OpCode.ALOAD:
                frame.opStack.push(frame.locals[index]);
                break;
            case enums_1.OpCode.LLOAD:
            case enums_1.OpCode.DLOAD:
                frame.opStack.pushWithNull(frame.locals[index]);
                break;
            case enums_1.OpCode.ISTORE:
            case enums_1.OpCode.FSTORE:
            case enums_1.OpCode.ASTORE:
                frame.locals[index] = frame.opStack.pop();
                break;
            case enums_1.OpCode.LSTORE:
            case enums_1.OpCode.DSTORE:
                // NULL
                frame.locals[index + 1] = frame.opStack.pop();
                // The actual value.
                frame.locals[index] = frame.opStack.pop();
                break;
            case enums_1.OpCode.RET:
                frame.pc = frame.locals[index];
                break;
            case enums_1.OpCode.IINC:
                var value = code.readInt16BE(pc + 4);
                frame.locals[index] = (frame.locals[index] + value) | 0;
                // wide iinc has 2 extra bytes.
                frame.pc += 2;
                break;
            default:
                assert_1["default"](false, "Unknown wide opcode: " + code[pc + 1]);
                break;
        }
    };
    Opcodes.iaload = Opcodes._aload_32;
    Opcodes.faload = Opcodes._aload_32;
    Opcodes.aaload = Opcodes._aload_32;
    Opcodes.baload = Opcodes._aload_32;
    Opcodes.caload = Opcodes._aload_32;
    Opcodes.saload = Opcodes._aload_32;
    Opcodes.daload = Opcodes._aload_64;
    Opcodes.laload = Opcodes._aload_64;
    Opcodes.iastore = Opcodes._astore_32;
    Opcodes.fastore = Opcodes._astore_32;
    Opcodes.aastore = Opcodes._astore_32;
    Opcodes.bastore = Opcodes._astore_32;
    Opcodes.castore = Opcodes._astore_32;
    Opcodes.sastore = Opcodes._astore_32;
    Opcodes.lastore = Opcodes._astore_64;
    Opcodes.dastore = Opcodes._astore_64;
    Opcodes.iconst_0 = Opcodes._const_0_32;
    Opcodes.iconst_1 = Opcodes._const_1_32;
    Opcodes.iconst_2 = Opcodes._const_2_32;
    Opcodes.fconst_0 = Opcodes._const_0_32;
    Opcodes.fconst_1 = Opcodes._const_1_32;
    Opcodes.fconst_2 = Opcodes._const_2_32;
    Opcodes.iload = Opcodes._load_32;
    Opcodes.iload_0 = Opcodes._load_0_32;
    Opcodes.iload_1 = Opcodes._load_1_32;
    Opcodes.iload_2 = Opcodes._load_2_32;
    Opcodes.iload_3 = Opcodes._load_3_32;
    Opcodes.fload = Opcodes._load_32;
    Opcodes.fload_0 = Opcodes._load_0_32;
    Opcodes.fload_1 = Opcodes._load_1_32;
    Opcodes.fload_2 = Opcodes._load_2_32;
    Opcodes.fload_3 = Opcodes._load_3_32;
    Opcodes.aload = Opcodes._load_32;
    Opcodes.aload_0 = Opcodes._load_0_32;
    Opcodes.aload_1 = Opcodes._load_1_32;
    Opcodes.aload_2 = Opcodes._load_2_32;
    Opcodes.aload_3 = Opcodes._load_3_32;
    Opcodes.lload = Opcodes._load_64;
    Opcodes.lload_0 = Opcodes._load_0_64;
    Opcodes.lload_1 = Opcodes._load_1_64;
    Opcodes.lload_2 = Opcodes._load_2_64;
    Opcodes.lload_3 = Opcodes._load_3_64;
    Opcodes.dload = Opcodes._load_64;
    Opcodes.dload_0 = Opcodes._load_0_64;
    Opcodes.dload_1 = Opcodes._load_1_64;
    Opcodes.dload_2 = Opcodes._load_2_64;
    Opcodes.dload_3 = Opcodes._load_3_64;
    Opcodes.istore = Opcodes._store_32;
    Opcodes.istore_0 = Opcodes._store_0_32;
    Opcodes.istore_1 = Opcodes._store_1_32;
    Opcodes.istore_2 = Opcodes._store_2_32;
    Opcodes.istore_3 = Opcodes._store_3_32;
    Opcodes.fstore = Opcodes._store_32;
    Opcodes.fstore_0 = Opcodes._store_0_32;
    Opcodes.fstore_1 = Opcodes._store_1_32;
    Opcodes.fstore_2 = Opcodes._store_2_32;
    Opcodes.fstore_3 = Opcodes._store_3_32;
    Opcodes.astore = Opcodes._store_32;
    Opcodes.astore_0 = Opcodes._store_0_32;
    Opcodes.astore_1 = Opcodes._store_1_32;
    Opcodes.astore_2 = Opcodes._store_2_32;
    Opcodes.astore_3 = Opcodes._store_3_32;
    Opcodes.lstore = Opcodes._store_64;
    Opcodes.lstore_0 = Opcodes._store_0_64;
    Opcodes.lstore_1 = Opcodes._store_1_64;
    Opcodes.lstore_2 = Opcodes._store_2_64;
    Opcodes.lstore_3 = Opcodes._store_3_64;
    Opcodes.dstore = Opcodes._store_64;
    Opcodes.dstore_0 = Opcodes._store_0_64;
    Opcodes.dstore_1 = Opcodes._store_1_64;
    Opcodes.dstore_2 = Opcodes._store_2_64;
    Opcodes.dstore_3 = Opcodes._store_3_64;
    Opcodes.ireturn = Opcodes._return_32;
    Opcodes.freturn = Opcodes._return_32;
    Opcodes.areturn = Opcodes._return_32;
    Opcodes.lreturn = Opcodes._return_64;
    Opcodes.dreturn = Opcodes._return_64;
    Opcodes.invokeinterface_fast = Opcodes.invokevirtual_fast;
    return Opcodes;
}());
exports.Opcodes = Opcodes;
exports.LookupTable = new Array(0xff);
// Put in function closure to prevent scope pollution.
(function () {
    for (var i = 0; i < 0xff; i++) {
        if (enums_1.OpCode.hasOwnProperty("" + i)) {
            exports.LookupTable[i] = Opcodes[enums_1.OpCode[i].toLowerCase()];
            assert_1["default"](exports.LookupTable[i] != null, "Missing implementation of opcode " + enums_1.OpCode[i]);
        }
    }
})();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3Bjb2Rlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9vcGNvZGVzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7R0FFRztBQUNILHFFQUFxRTtBQUNyRSwwRUFBMEU7QUFDMUUsbUZBQW1GO0FBQ25GLG1FQUFtRTtBQUNuRSxzQkFBa0IsU0FBUyxDQUFDLENBQUE7QUFDNUIscUJBQXVELFFBQVEsQ0FBQyxDQUFBO0FBSWhFLHNCQUFvRSxTQUFTLENBQUMsQ0FBQTtBQUM5RSx1QkFBbUIsVUFBVSxDQUFDLENBQUE7QUFXOUI7Ozs7R0FJRztBQUNILGdCQUF1QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsR0FBUTtJQUMzRSxFQUFFLENBQUMsQ0FBQyxHQUFHLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNoQixjQUFjLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxrQ0FBa0MsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUN0RSxNQUFNLENBQUMsSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUNELE1BQU0sQ0FBQyxLQUFLLENBQUM7QUFDZixDQUFDO0FBTmUsY0FBTSxTQU1yQixDQUFBO0FBRUQ7O0dBRUc7QUFDSCxjQUFxQixPQUFjO0lBQ2pDLGVBQWU7SUFDZixPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7SUFDZCxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO0FBQ3ZCLENBQUM7QUFKZSxZQUFJLE9BSW5CLENBQUE7QUFFRCx1QkFBOEIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLE1BQXlCO0lBQ25HLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQVksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM3QyxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsU0FBUyxFQUFFLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsVUFBQyxNQUFlO1FBQzFFLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7WUFDWCxNQUFNLENBQUMsU0FBUyxDQUFDLG9CQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDMUMsQ0FBQztJQUNILENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNWLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7QUFDbEMsQ0FBQztBQVJlLHFCQUFhLGdCQVE1QixDQUFBO0FBRUQsa0NBQXlDLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxHQUFjO0lBQ25HLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQVksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM3QyxHQUFHLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxVQUFDLEtBQWdCO1FBQ3RDLEVBQUUsQ0FBQyxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ2xCLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUMxQyxDQUFDO0lBQ0gsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ1YsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztBQUNsQyxDQUFDO0FBUmUsZ0NBQXdCLDJCQVF2QyxDQUFBO0FBRUQ7O0dBRUc7QUFDSCx5QkFBZ0MsTUFBaUIsRUFBRSxLQUF5QixFQUFFLE1BQXNCO0lBQ2xHLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQVksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUU3QyxvQkFBb0IsR0FBYztRQUNoQyxHQUFHLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxVQUFDLEtBQWdCO1lBQ3RDLEVBQUUsQ0FBQyxDQUFDLEtBQUssSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDO2dCQUNsQixNQUFNLENBQUMsU0FBUyxDQUFDLG9CQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDMUMsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN6QixNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsU0FBUyxFQUFFLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsVUFBQyxNQUFlO1lBQzFFLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7Z0JBQ1gsVUFBVSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUN6QixDQUFDO1FBQ0gsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ1osQ0FBQztJQUFDLElBQUksQ0FBQyxDQUFDO1FBQ04sVUFBVSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN6QixDQUFDO0lBQ0QsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztBQUNsQyxDQUFDO0FBckJlLHVCQUFlLGtCQXFCOUIsQ0FBQTtBQUVEOzs7OztHQUtHO0FBQ0gsd0JBQXVFLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxPQUFlLEVBQUUsR0FBVztJQUMvSSxNQUFNLENBQUMsaUJBQWlCLENBQUksT0FBTyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzFDLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7QUFDbEMsQ0FBQztBQUhlLHNCQUFjLGlCQUc3QixDQUFBO0FBRVUsa0JBQVUsR0FBNkI7SUFDaEQsQ0FBQyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsR0FBRztDQUNqRSxDQUFDO0FBRUY7O0dBRUc7QUFDSDtJQUFBO0lBZzhEQSxDQUFDO0lBLzdEQywrQkFBK0I7SUFFL0I7O09BRUc7SUFDWSxpQkFBUyxHQUF4QixVQUF5QixNQUFpQixFQUFFLEtBQXlCO1FBQ25FLElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLEdBQUcsR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQ25CLEdBQUcsR0FBNEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQy9DLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLElBQUksR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1lBQzNCLEVBQUUsQ0FBQyxDQUFDLEdBQUcsR0FBRyxDQUFDLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxDQUFDLENBQUM7Z0JBQzFCLGNBQWMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLDRDQUE0QyxFQUFLLEdBQUcsdUJBQWtCLEdBQUcsdUJBQWtCLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxlQUFlLEVBQUksQ0FBQyxDQUFDO1lBQy9KLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDN0IsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQ2IsQ0FBQztRQUNILENBQUM7UUFDRCxtREFBbUQ7SUFDckQsQ0FBQztJQVNELCtCQUErQjtJQUUvQjs7T0FFRztJQUNZLGlCQUFTLEdBQXhCLFVBQXlCLE1BQWlCLEVBQUUsS0FBeUI7UUFDbkUsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDekIsR0FBRyxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFDbkIsR0FBRyxHQUE0QixPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDL0MsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsSUFBSSxHQUFHLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUM7WUFDM0IsRUFBRSxDQUFDLENBQUMsR0FBRyxHQUFHLENBQUMsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDMUIsY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsNENBQTRDLEVBQUssR0FBRyx1QkFBa0IsR0FBRyx1QkFBa0IsR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDLGVBQWUsRUFBSSxDQUFDLENBQUM7WUFDL0osQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLE9BQU8sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO2dCQUM3QixnQkFBZ0I7Z0JBQ2hCLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ25CLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNiLENBQUM7UUFDSCxDQUFDO1FBQ0QsbURBQW1EO0lBQ3JELENBQUM7SUFLRCxnQ0FBZ0M7SUFFaEM7OztPQUdHO0lBQ1ksa0JBQVUsR0FBekIsVUFBMEIsTUFBaUIsRUFBRSxLQUF5QjtRQUNwRSxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixLQUFLLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUNyQixHQUFHLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUNuQixHQUFHLEdBQTRCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMvQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNoQyxJQUFJLEdBQUcsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQztZQUMzQixFQUFFLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDO2dCQUMxQixjQUFjLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSw0Q0FBNEMsRUFBSyxHQUFHLHVCQUFrQixHQUFHLHVCQUFrQixHQUFHLENBQUMsUUFBUSxFQUFFLENBQUMsZUFBZSxFQUFJLENBQUMsQ0FBQztZQUMvSixDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7Z0JBQ3ZCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNiLENBQUM7UUFDSCxDQUFDO1FBQ0QsbURBQW1EO0lBQ3JELENBQUM7SUFTRCxnQ0FBZ0M7SUFFaEM7OztPQUdHO0lBQ1ksa0JBQVUsR0FBekIsVUFBMEIsTUFBaUIsRUFBRSxLQUF5QjtRQUNwRSxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixLQUFLLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxFQUN0QixHQUFHLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUNuQixHQUFHLEdBQTRCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMvQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNoQyxJQUFJLEdBQUcsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQztZQUMzQixFQUFFLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDO2dCQUMxQixjQUFjLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSw0Q0FBNEMsRUFBSyxHQUFHLHVCQUFrQixHQUFHLHVCQUFrQixHQUFHLENBQUMsUUFBUSxFQUFFLENBQUMsZUFBZSxFQUFJLENBQUMsQ0FBQztZQUMvSixDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7Z0JBQ3ZCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUNiLENBQUM7UUFDSCxDQUFDO1FBQ0QsbURBQW1EO0lBQ3JELENBQUM7SUFLRCxzQkFBc0I7SUFDUixtQkFBVyxHQUF6QixVQUEwQixNQUFpQixFQUFFLEtBQXlCO1FBQ3BFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3pCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3RCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3RCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3RCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxpQkFBUyxHQUF2QixVQUF3QixNQUFpQixFQUFFLEtBQXlCO1FBQ2xFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdkIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQU1hLGdCQUFRLEdBQXRCLFVBQXVCLE1BQWlCLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGdCQUFRLEdBQXRCLFVBQXVCLE1BQWlCLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGdCQUFRLEdBQXRCLFVBQXVCLE1BQWlCLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQU1ELHNCQUFzQjtJQUNSLGdCQUFRLEdBQXRCLFVBQXVCLE1BQWlCLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsa0JBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsZ0JBQVEsR0FBdEIsVUFBdUIsTUFBaUIsRUFBRSxLQUF5QjtRQUNqRSxLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxrQkFBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3RDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxnQkFBUSxHQUF0QixVQUF1QixNQUFpQixFQUFFLEtBQXlCO1FBQ2pFLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxnQkFBUSxHQUF0QixVQUF1QixNQUFpQixFQUFFLEtBQXlCO1FBQ2pFLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFRCx5QkFBeUI7SUFDVixnQkFBUSxHQUF2QixVQUF3QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNoRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDL0MsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDaEIsQ0FBQztJQUVjLGtCQUFVLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUI7UUFDcEUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxrQkFBVSxHQUF6QixVQUEwQixNQUFpQixFQUFFLEtBQXlCO1FBQ3BFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNwQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWMsa0JBQVUsR0FBekIsVUFBMEIsTUFBaUIsRUFBRSxLQUF5QjtRQUNwRSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVjLGtCQUFVLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUI7UUFDcEUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFrQkQseUJBQXlCO0lBQ1YsZ0JBQVEsR0FBdkIsVUFBd0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDaEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYyxrQkFBVSxHQUF6QixVQUEwQixNQUFpQixFQUFFLEtBQXlCO1FBQ3BFLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWMsa0JBQVUsR0FBekIsVUFBMEIsTUFBaUIsRUFBRSxLQUF5QjtRQUNwRSxLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVjLGtCQUFVLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUI7UUFDcEUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzVDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxrQkFBVSxHQUF6QixVQUEwQixNQUFpQixFQUFFLEtBQXlCO1FBQ3BFLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBYUQsMEJBQTBCO0lBQ1gsaUJBQVMsR0FBeEIsVUFBeUIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDakYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2pELEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN0QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWMsbUJBQVcsR0FBMUIsVUFBMkIsTUFBaUIsRUFBRSxLQUF5QjtRQUNyRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDdEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVjLG1CQUFXLEdBQTFCLFVBQTJCLE1BQWlCLEVBQUUsS0FBeUI7UUFDckUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3RDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN0QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBa0JELDBCQUEwQjtJQUNYLGlCQUFTLEdBQXhCLFVBQXlCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2pGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUMxQixPQUFPO1FBQ1AsS0FBSyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMvQyxvQkFBb0I7UUFDcEIsS0FBSyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzNDLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN0QyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDdEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVjLG1CQUFXLEdBQTFCLFVBQTJCLE1BQWlCLEVBQUUsS0FBeUI7UUFDckUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3RDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN0QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWMsbUJBQVcsR0FBMUIsVUFBMkIsTUFBaUIsRUFBRSxLQUF5QjtRQUNyRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDdEMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3RDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYyxtQkFBVyxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCO1FBQ3JFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN0QyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDdEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQWFELFdBQVc7SUFFRyxjQUFNLEdBQXBCLFVBQXFCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzdFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3QyxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNoQixDQUFDO0lBRWEsY0FBTSxHQUFwQixVQUFxQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUM3RSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDMUMsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDaEIsQ0FBQztJQUVhLFdBQUcsR0FBakIsVUFBa0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM1RCxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3QixLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELGlDQUFpQztRQUNqQyxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3QixLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDcEIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGNBQU0sR0FBcEIsVUFBcUIsTUFBaUIsRUFBRSxLQUF5QjtRQUMvRCxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ3ZCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxjQUFNLEdBQXBCLFVBQXFCLE1BQWlCLEVBQUUsS0FBeUI7UUFDL0QsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUN2QixLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDckIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGVBQU8sR0FBckIsVUFBc0IsTUFBaUIsRUFBRSxLQUF5QjtRQUNoRSxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ3hCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxlQUFPLEdBQXJCLFVBQXNCLE1BQWlCLEVBQUUsS0FBeUI7UUFDaEUsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDekIsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFDbEIsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFDbEIsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFDbEIsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNyQixPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDdEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ3JCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFRCxrQkFBa0I7SUFDSixZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ2xELEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN6RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxnQkFBUyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUN0RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ25ELEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLGdCQUFTLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN4RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUN2RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsTUFBTSxFQUFFLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDbEUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQVEsSUFBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM5RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDOUQsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0JBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN2RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7UUFDdEQsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsR0FBVyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxHQUFXLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNsRixFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNaLGNBQWMsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLGlDQUFpQyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ2hGLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLCtFQUErRTtZQUMvRSx5RUFBeUU7WUFDekUsb0NBQW9DO1lBQ3BDLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxpQkFBUyxDQUFDLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN4QyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2xCLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQzVCLENBQUM7WUFDRCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDYixDQUFDO0lBQ0gsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixDQUFDLEdBQVUsT0FBTyxDQUFDLElBQUksRUFBRSxFQUN6QixDQUFDLEdBQVUsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQzVCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDZixjQUFjLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxpQ0FBaUMsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUNoRixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUMvQixLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDYixDQUFDO0lBQ0gsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixDQUFDLEdBQVcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0JBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMzQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLENBQUMsR0FBVyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDN0IsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDekMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixDQUFDLEdBQVcsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUN6QixDQUFDLEdBQVcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVCLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ1osY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsaUNBQWlDLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEYsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDcEIsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQ2IsQ0FBQztJQUNILENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDekIsQ0FBQyxHQUFVLE9BQU8sQ0FBQyxJQUFJLEVBQUUsRUFDekIsQ0FBQyxHQUFVLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUM1QixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2YsY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsaUNBQWlDLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDaEYsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQ2IsQ0FBQztJQUNILENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDekIsQ0FBQyxHQUFXLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUNoQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLENBQUMsR0FBVyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDN0IsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDekMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDakMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFDOUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUM3QixLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3RDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFRCx3QkFBd0I7SUFFVixZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDekIsQ0FBQyxHQUFXLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNqQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLENBQUMsR0FBVyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsU0FBUyxDQUFDLGtCQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNqRSxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLENBQUMsR0FBVyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDakMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixDQUFDLEdBQVcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxrQkFBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbEUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM5RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixDQUFDLEdBQVcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDeEMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM5RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixDQUFDLEdBQVcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLGtCQUFrQixDQUFDLGtCQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxRSxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDNUMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3pELEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxXQUFHLEdBQWpCLFVBQWtCLE1BQWlCLEVBQUUsS0FBeUI7UUFDNUQsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUM1QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDeEQsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QjtRQUM3RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQzVDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUI7UUFDN0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN6RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUMzRSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksR0FBRyxHQUFHLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLEVBQ3BCLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUM5QixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbEQsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDaEIsQ0FBQztJQUVhLFdBQUcsR0FBakIsVUFBa0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM1RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxZQUFZLENBQUMsa0JBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNuRCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELDJDQUEyQztRQUMzQyx1REFBdUQ7UUFDdkQsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFdBQUcsR0FBakIsVUFBa0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM1RCxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN6QixLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUNyQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUN4QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUNoRCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxnQkFBUyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDdkMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFdBQUcsR0FBakIsVUFBa0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM1RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxZQUFZLENBQUMsa0JBQUssQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN0RCxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3pCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxXQUFHLEdBQWpCLFVBQWtCLE1BQWlCLEVBQUUsS0FBeUI7UUFDNUQsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLGdCQUFTLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN4QyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLEtBQUssR0FBVyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakMsRUFBRSxDQUFDLENBQUMsS0FBSyxLQUFLLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUM7WUFDdkMsT0FBTyxDQUFDLFlBQVksQ0FBQyxrQkFBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLENBQUM7UUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsS0FBSyxLQUFLLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUM7WUFDOUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxrQkFBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLE9BQU8sQ0FBQyxZQUFZLENBQUMsa0JBQUssQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUNoRCxDQUFDO1FBQ0QsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLFdBQUcsR0FBakIsVUFBa0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM1RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzVCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNkLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0JBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3ZDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxXQUFHLEdBQWpCLFVBQWtCLE1BQWlCLEVBQUUsS0FBeUI7UUFDNUQsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzFDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztJQUNiLENBQUM7SUFFYSxXQUFHLEdBQWpCLFVBQWtCLE1BQWlCLEVBQUUsS0FBeUI7UUFDNUQsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM1QixPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxNQUFNLENBQUMsQ0FBQztRQUNyQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDNUIsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsSUFBSSxFQUFFLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUMxQyxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUM7SUFDYixDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCO1FBQzdELElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLEVBQUUsR0FBVSxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDN0IsT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDekMsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM5RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUNsQixFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3JCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2QsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ25CLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbEIsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sbUNBQW1DO1lBQ25DLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNuQixDQUFDO1FBQ0QsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM5RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUNsQixFQUFFLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3JCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2QsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ25CLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNuQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixtQ0FBbUM7WUFDbkMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixDQUFDO1FBQ0QsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM5RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxFQUNuQixFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ3RCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2QsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ25CLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbEIsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sbUNBQW1DO1lBQ25DLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNuQixDQUFDO1FBQ0QsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QjtRQUM5RCxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN6QixFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxFQUNuQixFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ3RCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2QsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixDQUFDO1FBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ25CLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNuQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixtQ0FBbUM7WUFDbkMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixDQUFDO1FBQ0QsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO0lBQ2IsQ0FBQztJQUVELDBCQUEwQjtJQUNaLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDM0UsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDOUIsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7WUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzNFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlCLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1lBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNmLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUMzRSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM1QixJQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUN4QyxLQUFLLENBQUMsRUFBRSxJQUFJLE1BQU0sQ0FBQztZQUNuQixFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQy9CLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO0lBQ0gsQ0FBQztJQUVhLFlBQUksR0FBbEIsVUFBbUIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDM0UsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDN0IsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7WUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzNFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzVCLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1lBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNmLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUMzRSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUM3QixJQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUN4QyxLQUFLLENBQUMsRUFBRSxJQUFJLE1BQU0sQ0FBQztZQUNuQixFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQy9CLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO0lBQ0gsQ0FBQztJQUVELDJCQUEyQjtJQUNiLGlCQUFTLEdBQXZCLFVBQXdCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2hGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM3QixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzdCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2QsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7WUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFFYSxpQkFBUyxHQUF2QixVQUF3QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNoRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDN0IsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM3QixFQUFFLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNkLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1lBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNmLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsaUJBQVMsR0FBdkIsVUFBd0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDaEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzdCLElBQUksRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDN0IsRUFBRSxDQUFDLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDWixJQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUN4QyxLQUFLLENBQUMsRUFBRSxJQUFJLE1BQU0sQ0FBQztZQUNuQixFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQy9CLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO0lBQ0gsQ0FBQztJQUVhLGlCQUFTLEdBQXZCLFVBQXdCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2hGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM3QixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzdCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2IsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7WUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFFYSxpQkFBUyxHQUF2QixVQUF3QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNoRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDN0IsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM3QixFQUFFLENBQUMsQ0FBQyxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNaLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1lBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNmLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsaUJBQVMsR0FBdkIsVUFBd0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDaEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzdCLElBQUksRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDN0IsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDYixJQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUN4QyxLQUFLLENBQUMsRUFBRSxJQUFJLE1BQU0sQ0FBQztZQUNuQixFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQy9CLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO0lBQ0gsQ0FBQztJQUVhLGlCQUFTLEdBQXZCLFVBQXdCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2hGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM3QixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzdCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2QsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7WUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFFYSxpQkFBUyxHQUF2QixVQUF3QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNoRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDN0IsSUFBSSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUM3QixFQUFFLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNkLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1lBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNmLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRUQsa0JBQWtCO0lBQ0osWUFBSSxHQUFsQixVQUFtQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUMzRSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1FBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUMvQixDQUFDO0lBQ0gsQ0FBQztJQUVhLFdBQUcsR0FBakIsVUFBa0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDMUUsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDM0IsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7UUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQy9CLENBQUM7SUFDSCxDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUMxRSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLEtBQUssQ0FBQyxFQUFFLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVhLG1CQUFXLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2xGLElBQUksRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDbEIsMkRBQTJEO1FBQzNELEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNuQyxJQUFJLGFBQWEsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxFQUN0QyxHQUFHLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLEVBQzlCLElBQUksR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsRUFDL0IsTUFBTSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFFL0IsRUFBRSxDQUFDLENBQUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQztZQUNwQyxLQUFLLENBQUMsRUFBRSxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDL0QsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxhQUFhLENBQUM7UUFDNUIsQ0FBQztJQUNILENBQUM7SUFFYSxvQkFBWSxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNuRixJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ2xCLHlEQUF5RDtRQUN6RCxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbkMsSUFBSSxhQUFhLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsRUFDdEMsTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxFQUNqQyxDQUFTLEVBQ1QsQ0FBQyxHQUFXLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7UUFFbEMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNSLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1lBQzVCLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDL0IsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7Z0JBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO2dCQUNuQixFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztvQkFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO2dCQUMvQixDQUFDO2dCQUNELE1BQU0sQ0FBQztZQUNULENBQUM7WUFDRCxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ1YsQ0FBQztRQUNELGtCQUFrQjtRQUNsQixLQUFLLENBQUMsRUFBRSxJQUFJLGFBQWEsQ0FBQztJQUM1QixDQUFDO0lBRWEsY0FBTSxHQUFwQixVQUFxQixNQUFpQixFQUFFLEtBQXlCO1FBQy9ELEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7UUFDaEMsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzlDLGNBQWM7WUFDZCxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN6RCxrQ0FBa0M7Z0JBQ2xDLE1BQU0sQ0FBQztZQUNULENBQUM7UUFDSCxDQUFDO1FBQ0QsTUFBTSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ3ZCLENBQUM7SUFFRCw2QkFBNkI7SUFFZCxrQkFBVSxHQUF6QixVQUEwQixNQUFpQixFQUFFLEtBQXlCO1FBQ3BFLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7UUFDaEMsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzlDLGNBQWM7WUFDZCxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN6RCxrQ0FBa0M7Z0JBQ2xDLE1BQU0sQ0FBQztZQUNULENBQUM7UUFDSCxDQUFDO1FBQ0QsTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQU1ELDJCQUEyQjtJQUVaLGtCQUFVLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUI7UUFDcEUsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztRQUNoQyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDOUMsY0FBYztZQUNkLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3pELGtDQUFrQztnQkFDbEMsTUFBTSxDQUFDO1lBQ1QsQ0FBQztRQUNILENBQUM7UUFDRCxNQUFNLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUthLGlCQUFTLEdBQXZCLFVBQXdCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2hGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxTQUFTLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM5RixtQkFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsS0FBSyw0QkFBb0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM5RCxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzNCLCtDQUErQztZQUMvQyx5RUFBeUU7WUFDekUsSUFBSSxhQUFhLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUM7WUFDeEMsRUFBRSxDQUFDLENBQUMsYUFBYSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3hDLDJEQUEyRDtnQkFDM0QsNEJBQTRCO2dCQUM1QixFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLFVBQVUsS0FBSyxHQUFHLElBQUksU0FBUyxDQUFDLGVBQWUsQ0FBQyxVQUFVLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQztvQkFDakcsSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQU0sQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDckMsQ0FBQztnQkFBQyxJQUFJLENBQUMsQ0FBQztvQkFDTixJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGdCQUFnQixDQUFDO2dCQUNyQyxDQUFDO2dCQUNELG9DQUFvQztnQkFDcEMsU0FBUyxDQUFDLHFCQUFxQixHQUFHLGFBQWEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDekUsQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLG9DQUFvQztnQkFDcEMsd0JBQXdCLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxhQUFhLENBQUMsQ0FBQztZQUN6RCxDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04scUJBQXFCO1lBQ3JCLGFBQWEsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzFDLENBQUM7SUFDSCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDVyx3QkFBZ0IsR0FBOUIsVUFBK0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDdkYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFNBQVMsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlGLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUM3RSxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNoQixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDVyx3QkFBZ0IsR0FBOUIsVUFBK0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDdkYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFNBQVMsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlGLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUNyRixLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNoQixDQUFDO0lBRWEsaUJBQVMsR0FBdkIsVUFBd0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDaEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFNBQVMsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlGLG1CQUFNLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxLQUFLLDRCQUFvQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTlELEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDM0IsK0NBQStDO1lBQy9DLHlFQUF5RTtZQUN6RSxJQUFJLGFBQWEsR0FBRyxTQUFTLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQztZQUN4QyxFQUFFLENBQUMsQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDeEMsMkRBQTJEO2dCQUMzRCw0QkFBNEI7Z0JBQzVCLEVBQUUsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsVUFBVSxLQUFLLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLFVBQVUsS0FBSyxHQUFHLENBQUMsQ0FBQyxDQUFDO29CQUNqRyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGdCQUFnQixDQUFDO2dCQUNyQyxDQUFDO2dCQUFDLElBQUksQ0FBQyxDQUFDO29CQUNOLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxjQUFNLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3JDLENBQUM7Z0JBQ0Qsb0NBQW9DO2dCQUNwQyxTQUFTLENBQUMscUJBQXFCLEdBQUcsYUFBYSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN6RSxDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04sb0NBQW9DO2dCQUNwQyx3QkFBd0IsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1lBQ3pELENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixxQkFBcUI7WUFDckIsYUFBYSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDMUMsQ0FBQztJQUNILENBQUM7SUFFRDs7Ozs7T0FLRztJQUNXLHdCQUFnQixHQUE5QixVQUErQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUN2RixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksU0FBUyxHQUFvQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDOUYsU0FBUyxDQUFDLHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQy9FLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNXLHdCQUFnQixHQUE5QixVQUErQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUN2RixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksU0FBUyxHQUFvQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDOUYsU0FBUyxDQUFDLHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2hGLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYSxnQkFBUSxHQUF0QixVQUF1QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUMvRSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksU0FBUyxHQUFvQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQzNGLE1BQU0sR0FBRyxLQUFLLENBQUMsU0FBUyxFQUFFLEVBQzFCLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzVCLG1CQUFNLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxLQUFLLDRCQUFvQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzlELDJFQUEyRTtRQUMzRSw0REFBNEQ7UUFDNUQsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsNkVBQTZFO1lBQzdFLG9FQUFvRTtZQUNwRSxpQkFBaUI7WUFDakIsRUFBRSxDQUFDLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDM0IsSUFBSSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQztnQkFDNUIsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSxHQUFHLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDO29CQUM3RCxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGVBQWUsQ0FBQztnQkFDcEMsQ0FBQztnQkFBQyxJQUFJLENBQUMsQ0FBQztvQkFDTixJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGVBQWUsQ0FBQztnQkFDcEMsQ0FBQztZQUVILENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztZQUMxQyxDQUFDO1FBQ0gsQ0FBQztJQUNILENBQUM7SUFFYSx1QkFBZSxHQUE3QixVQUE4QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUN0RixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksU0FBUyxHQUFvQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQzNGLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLEdBQUcsR0FBOEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLE9BQU8sQ0FBQyxJQUFJLENBQVEsR0FBSSxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO1lBQ25ELEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsdUJBQWUsR0FBN0IsVUFBOEIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDdEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFNBQVMsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUMzRixPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxHQUFHLEdBQThCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMxRSxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNoQyxPQUFPLENBQUMsWUFBWSxDQUFRLEdBQUksQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztZQUMzRCxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO0lBQ0gsQ0FBQztJQUVhLGdCQUFRLEdBQXRCLFVBQXVCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQy9FLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxTQUFTLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDM0YsTUFBTSxHQUFHLEtBQUssQ0FBQyxTQUFTLEVBQUUsRUFDMUIsTUFBTSxHQUFHLFNBQVMsQ0FBQyxlQUFlLENBQUMsVUFBVSxJQUFJLEdBQUcsSUFBSSxTQUFTLENBQUMsZUFBZSxDQUFDLFVBQVUsSUFBSSxHQUFHLEVBQ25HLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQzlDLG1CQUFNLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxLQUFLLDRCQUFvQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRTlELDJFQUEyRTtRQUMzRSw0REFBNEQ7UUFDNUQsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsNkVBQTZFO1lBQzdFLG9FQUFvRTtZQUNwRSxpQkFBaUI7WUFDakIsRUFBRSxDQUFDLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztnQkFDM0IsSUFBSSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQztnQkFDNUIsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztvQkFDWCxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGVBQWUsQ0FBQztnQkFDcEMsQ0FBQztnQkFBQyxJQUFJLENBQUMsQ0FBQztvQkFDTixJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGVBQWUsQ0FBQztnQkFDcEMsQ0FBQztnQkFDRCxzQ0FBc0M7Z0JBQ3RDLFNBQVMsQ0FBQyxhQUFhLEdBQU0seUJBQWtCLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxTQUFJLFNBQVMsQ0FBQyxlQUFlLENBQUMsSUFBTSxDQUFDO1lBRW5ILENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztZQUMxQyxDQUFDO1FBQ0gsQ0FBQztJQUNILENBQUM7SUFFYSx1QkFBZSxHQUE3QixVQUE4QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUN0RixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLEdBQUcsR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQ25CLEdBQUcsR0FBOEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUM5QyxTQUFTLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUU1RixFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN6QixHQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxHQUFHLEdBQUcsQ0FBQztZQUMzQyxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO1FBQ0QsdUJBQXVCO0lBQ3pCLENBQUM7SUFFYSx1QkFBZSxHQUE3QixVQUE4QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUN0RixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLEdBQUcsR0FBRyxPQUFPLENBQUMsSUFBSSxFQUFFLEVBQ3BCLEdBQUcsR0FBOEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUM5QyxTQUFTLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUU1RixFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN6QixHQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxHQUFHLEdBQUcsQ0FBQztZQUMzQyxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO1FBQ0QsdUJBQXVCO0lBQ3pCLENBQUM7SUFFYSxxQkFBYSxHQUEzQixVQUE0QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNwRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksZUFBZSxHQUFxQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFckcsZ0VBQWdFO1FBQ2hFLHdFQUF3RTtRQUN4RSxzRUFBc0U7UUFDdEUscURBQXFEO1FBQ3JELEVBQUUsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDakMsSUFBSSxDQUFDLEdBQUcsZUFBZSxDQUFDLE1BQU0sQ0FBQztZQUMvQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsc0JBQXNCLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQy9CLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO29CQUNmLEtBQUssYUFBYTt3QkFDaEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQU0sQ0FBQyxXQUFXLENBQUM7d0JBQzlCLEtBQUssQ0FBQztvQkFDUixLQUFLLFFBQVEsQ0FBQztvQkFDZCxLQUFLLGFBQWE7d0JBQ2hCLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxjQUFNLENBQUMsWUFBWSxDQUFDO3dCQUMvQixLQUFLLENBQUM7b0JBQ1I7d0JBQ0UsY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsaUNBQWlDLEVBQUUsMkNBQXlDLENBQUMsQ0FBQyxHQUFHLENBQUMsZUFBZSxFQUFFLFNBQUksQ0FBQyxDQUFDLElBQU0sQ0FBQyxDQUFDO3dCQUMvSSxLQUFLLENBQUM7Z0JBQ1YsQ0FBQztZQUNILENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTixJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGtCQUFrQixDQUFDO1lBQ3ZDLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNoRCxDQUFDO0lBQ0gsQ0FBQztJQUVhLHVCQUFlLEdBQTdCLFVBQThCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ3RGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxlQUFlLEdBQThCLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM5RyxFQUFFLENBQUMsQ0FBQyxlQUFlLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLEVBQUUsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3JELDZCQUE2QjtnQkFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQU0sQ0FBQyxvQkFBb0IsQ0FBQztZQUN6QyxDQUFDO1lBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ04seUNBQXlDO2dCQUN6Qyx1RUFBdUU7Z0JBQ3ZFLDBDQUEwQztnQkFDMUMsZUFBZSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsZUFBZSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQzVELENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNoRCxDQUFDO0lBQ0gsQ0FBQztJQUVhLHFCQUFhLEdBQTNCLFVBQTRCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ3BGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxpQkFBaUIsR0FBbUIsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JHLE1BQU0sQ0FBQyxTQUFTLENBQUMsb0JBQVksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUM3QyxpQkFBaUIsQ0FBQyx1QkFBdUIsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLFNBQVMsRUFBRSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLEVBQUUsRUFBRSxVQUFDLE1BQWU7WUFDekcsRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztnQkFDWCxtQkFBTSxDQUFDLE9BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxVQUFVLEVBQUUsa0NBQWtDLENBQUMsQ0FBQztnQkFDdkgsSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQU0sQ0FBQyxrQkFBa0IsQ0FBQztnQkFDckMsZ0NBQWdDO2dCQUNoQyxNQUFNLENBQUMsU0FBUyxDQUFDLG9CQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDMUMsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztJQUNsQyxDQUFDO0lBRUQ7O09BRUc7SUFDVyxxQkFBYSxHQUEzQixVQUE0QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNwRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksZUFBZSxHQUFnRCxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDaEksRUFBRSxDQUFDLENBQUMsZUFBZSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUNqQyxxQkFBcUI7WUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQU0sQ0FBQyxxQkFBcUIsQ0FBQztRQUMxQyxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNoRCxDQUFDO0lBQ0gsQ0FBQztJQUVhLG9CQUFZLEdBQTFCLFVBQTJCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ25GLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxlQUFlLEdBQWdELEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNoSSxFQUFFLENBQUMsQ0FBQyxlQUFlLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLElBQUksQ0FBQyxHQUFHLGVBQWUsQ0FBQyxNQUFNLENBQUM7WUFDL0IsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNoQyxJQUFJLFNBQVMsR0FBVyxjQUFNLENBQUMsaUJBQWlCLENBQUM7Z0JBQ2pELEVBQUUsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsc0JBQXNCLEVBQUUsQ0FBQyxDQUFDLENBQUM7b0JBQ3BELE1BQU0sQ0FBQyxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQzt3QkFDcEMsS0FBSyxpQkFBaUIsQ0FBQzt3QkFDdkIsS0FBSyxlQUFlOzRCQUNsQixTQUFTLEdBQUcsY0FBTSxDQUFDLGFBQWEsQ0FBQzs0QkFDakMsS0FBSyxDQUFDO3dCQUNSLEtBQUssY0FBYyxDQUFDO3dCQUNwQixLQUFLLGVBQWU7NEJBQ2xCLFNBQVMsR0FBRyxjQUFNLENBQUMsYUFBYSxDQUFDOzRCQUNqQyxLQUFLLENBQUM7d0JBQ1I7NEJBQ0UsbUJBQU0sQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLENBQUMsQ0FBQzs0QkFDdkMsS0FBSyxDQUFDO29CQUNWLENBQUM7Z0JBQ0gsQ0FBQztnQkFDRCxxQkFBcUI7Z0JBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxTQUFTLENBQUM7WUFDdkIsQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLHdCQUF3QixDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ2pELENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNoRCxDQUFDO0lBQ0gsQ0FBQztJQUVELHdCQUF3QjtJQUVWLDZCQUFxQixHQUFuQyxVQUFvQyxNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUM1RixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksZUFBZSxHQUFnRCxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQzdILE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLFNBQVMsR0FBRyxlQUFlLENBQUMsYUFBYSxFQUNsRSxHQUFHLEdBQThCLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFOUQsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsSUFBSSxJQUFJLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUMzQyxPQUFPLENBQUMsV0FBVyxDQUFDLFNBQVMsR0FBRyxDQUFDLENBQUMsQ0FBQztZQUNuQyxtQkFBTSxDQUFDLE9BQWMsR0FBSSxDQUFDLGVBQWUsQ0FBQyxhQUFhLENBQUMsS0FBSyxVQUFVLEVBQUUscUJBQW1CLGVBQWUsQ0FBQyxhQUFhLHFCQUFrQixFQUFFLE1BQU0sQ0FBQyxDQUFDO1lBQzlJLEdBQUksQ0FBQyxlQUFlLENBQUMsYUFBYSxDQUFDLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQ3pELEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7UUFDbEMsQ0FBQztJQUNILENBQUM7SUFFYSx5QkFBaUIsR0FBL0IsVUFBZ0MsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDeEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLGVBQWUsR0FBZ0QsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUM3SCxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxTQUFTLEdBQUcsZUFBZSxDQUFDLGFBQWEsRUFDbEUsSUFBSSxHQUFHLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoRCxtQkFBTSxDQUFDLGVBQWUsQ0FBQyxhQUFhLElBQUksSUFBSSxFQUFFLDRCQUE0QixDQUFDLENBQUM7UUFDNUUsbUJBQU0sQ0FBQyxPQUFNLENBQUMsZUFBZSxDQUFDLGFBQWEsQ0FBQyxlQUFlLENBQUMsYUFBYSxDQUFDLENBQUMsS0FBSyxVQUFVLEVBQUUsaUNBQWlDLENBQUMsQ0FBQztRQUMvSCxlQUFlLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDM0UsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztJQUNsQyxDQUFDO0lBRWEsMEJBQWtCLEdBQWhDLFVBQWlDLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ3pGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxlQUFlLEdBQWdELEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDN0gsS0FBSyxHQUFHLGVBQWUsQ0FBQyxhQUFhLEVBQ3JDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN2QixHQUFHLEdBQThCLE9BQU8sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDMUQsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsaUNBQWlDO1lBQ2pDLG1CQUFNLENBQUMsT0FBYyxHQUFJLENBQUMsZUFBZSxDQUFDLFNBQVMsQ0FBQyxLQUFLLFVBQVUsRUFBRSxxQkFBbUIsZUFBZSxDQUFDLFNBQVMscUJBQWtCLENBQUMsQ0FBQztZQUM5SCxHQUFJLENBQUMsZUFBZSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDNUUsT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDL0IsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztRQUNsQyxDQUFDO1FBQ0QsdUNBQXVDO0lBQ3pDLENBQUM7SUFJYSwwQkFBa0IsR0FBaEMsVUFBaUMsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDekYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLGlCQUFpQixHQUFtQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQ2xHLEdBQUcsR0FBRyxpQkFBaUIsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsRUFDN0MsUUFBUSxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDakIsR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQ3JCLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLFNBQVMsR0FBRyxpQkFBaUIsQ0FBQyxhQUFhLEVBQ3BFLElBQUksR0FBRyxPQUFPLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFaEQsRUFBRSxDQUFDLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDdEIsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN0QixDQUFDO1FBQ0QsR0FBRyxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDeEIsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztJQUNsQyxDQUFDO0lBRUQ7O09BRUc7SUFDVyxvQkFBWSxHQUExQixVQUEyQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNuRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksZUFBZSxHQUFxQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQ2xHLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN2QixHQUFHLEdBQUcsZUFBZSxDQUFDLFVBQVUsQ0FBQyxRQUFRO1FBQ3pDLHlDQUF5QztRQUN6QyxTQUFTLEdBQUcsZUFBZSxDQUFDLGFBQWEsR0FBRyxDQUFDLEVBQzdDLFFBQVEsR0FBRyxlQUFlLENBQUMsUUFBUSxFQUNuQyxJQUFJLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUV6QyxFQUFFLENBQUMsQ0FBQyxRQUFRLEtBQUssSUFBSSxDQUFDLENBQUMsQ0FBQztZQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3RCLENBQUM7UUFFRCxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNwQyxPQUFPLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQy9CLDRDQUE0QztZQUM1Qyx5RUFBeUU7WUFDekUsR0FBRyxDQUFDLE1BQU0sRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDeEIsS0FBSyxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQztRQUNsQyxDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNXLG1CQUFXLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2xGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxlQUFlLEdBQXFCLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDbEcsU0FBUyxHQUFHLGVBQWUsQ0FBQyxnQkFBZ0IsRUFBRSxFQUM5QyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDdkIsR0FBRyxHQUEyQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztRQUN4RSwwRUFBMEU7UUFDMUUsZ0NBQWdDO1FBQ2hDLElBQUksR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLFNBQVMsR0FBRyxDQUFDLENBQUMsRUFDMUMsU0FBK0MsRUFDL0MsRUFBd0MsRUFDeEMsQ0FBUyxDQUFDO1FBRVoseUJBQXlCO1FBQ3pCLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLE9BQU8sQ0FBQyxXQUFXLENBQUMsU0FBUyxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ25DLFNBQVMsR0FBRyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztZQUN0RCxFQUFFLEdBQUcsU0FBUyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7WUFDdEQsbUJBQU0sQ0FBQyxFQUFFLENBQUMsUUFBUSxLQUFLLElBQUksSUFBSSxFQUFFLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO1lBQ3RGLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLGVBQWUsQ0FBQyxlQUFlLENBQUMsVUFBVSxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQ3RFLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7UUFDbEMsQ0FBQztJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ1cscUJBQWEsR0FBM0IsVUFBNEIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDcEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLGVBQWUsR0FBcUIsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUNsRyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxTQUFTLEdBQUcsZUFBZSxDQUFDLGFBQWE7UUFDbEUscUVBQXFFO1FBQ3JFLGFBQWE7UUFDYixJQUFJLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsRUFDdEMsVUFBVSxHQUF5QyxJQUFJLENBQUMsR0FBRyxFQUFFO1FBQzdELCtCQUErQjtRQUMvQixJQUFJLEdBQUcsZUFBZSxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUM7UUFFcEQsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdkMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUMvQixtQkFBTSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxlQUFlLEVBQUUsS0FBSywrQkFBK0IsQ0FBQyxDQUFDO1lBQ3BGLHlHQUF5RztZQUN6Ryx5REFBeUQ7WUFDekQsVUFBVSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUN2RixLQUFLLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDO1FBQ2xDLENBQUM7SUFDSCxDQUFDO0lBRUQsMkVBQTJFO0lBQzdELHFCQUFhLEdBQTNCLFVBQTRCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ3BGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxlQUFlLEdBQWdELEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDN0gsU0FBUyxHQUFHLGVBQWUsQ0FBQyxhQUFhLEVBQ3pDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN2QixJQUFJLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUM7UUFDdEMscUVBQXFFO1FBQ3JFLGFBQWE7UUFDYixVQUFVLEdBQXlDLElBQUksQ0FBQyxHQUFHLEVBQUUsRUFDN0QsSUFBSSxHQUFHLGVBQWUsQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDO1FBRXBELEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3ZDLE9BQU8sQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDL0IsbUJBQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFLENBQUMsZUFBZSxFQUFFLEtBQUssK0JBQStCLENBQUMsQ0FBQztZQUNwRix5R0FBeUc7WUFDekcsVUFBVSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUN2RixLQUFLLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDO1FBQ2xDLENBQUM7SUFDSCxDQUFDO0lBRWEsa0JBQVUsR0FBeEIsVUFBeUIsTUFBaUIsRUFBRSxLQUF5QjtRQUNuRSxjQUFjLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxtQkFBbUIsRUFBRSw2QkFBNkIsQ0FBQyxDQUFDO0lBQ3BGLENBQUM7SUFFYSxXQUFHLEdBQWpCLFVBQWtCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzFFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxRQUFRLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3RixFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzFCLElBQUksR0FBRyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7WUFDdkIsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxjQUFNLENBQUMsUUFBUSxDQUFDO1lBRTdCLENBQUM7WUFBQyxJQUFJLENBQUMsQ0FBQztnQkFDTix3QkFBd0IsQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQy9DLENBQUM7UUFDSCxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN6QyxDQUFDO0lBQ0gsQ0FBQztJQUVhLGdCQUFRLEdBQXRCLFVBQXVCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQy9FLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxRQUFRLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3RixLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUN4RCxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNoQixDQUFDO0lBRWEsZ0JBQVEsR0FBdEIsVUFBdUIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDL0UsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQiwyREFBMkQ7UUFDM0QsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDekIsSUFBSSxHQUFHLEdBQUcsR0FBRyxrQkFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDckMsR0FBRyxHQUF5QixLQUFLLENBQUMsU0FBUyxFQUFFLENBQUMsbUJBQW1CLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxFQUMvRSxNQUFNLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3pCLEVBQUUsQ0FBQyxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hCLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUMvRCxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixjQUFjLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSx3Q0FBd0MsRUFBRSxtQkFBaUIsSUFBSSwyQkFBc0IsTUFBUSxDQUFDLENBQUM7UUFDL0gsQ0FBQztJQUNILENBQUM7SUFFYSxpQkFBUyxHQUF2QixVQUF3QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNoRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksUUFBUSxHQUFvQixLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0YsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUMxQixxQkFBcUI7WUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLGNBQU0sQ0FBQyxjQUFjLENBQUM7WUFDakMsUUFBUSxDQUFDLFVBQVUsR0FBeUIsS0FBSyxDQUFDLFNBQVMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxNQUFJLFFBQVEsQ0FBQyxHQUFHLENBQUMsZUFBZSxFQUFJLENBQUMsQ0FBQztZQUNoSSxRQUFRLENBQUMscUJBQXFCLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDOUUsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sYUFBYSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDekMsQ0FBQztJQUNILENBQUM7SUFFYSxzQkFBYyxHQUE1QixVQUE2QixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUNyRixJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLElBQUksT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQ3pCLFFBQVEsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUN4RixNQUFNLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBRXpCLEVBQUUsQ0FBQyxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hCLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxRQUFRLENBQUMscUJBQXFCLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7WUFDakUsS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsd0NBQXdDLEVBQUUsbUJBQWlCLFFBQVEsQ0FBQyxVQUFVLENBQUMsZUFBZSxFQUFFLDJCQUFzQixNQUFRLENBQUMsQ0FBQztRQUNoSyxDQUFDO0lBQ0gsQ0FBQztJQUVhLG1CQUFXLEdBQXpCLFVBQTBCLE1BQWlCLEVBQUUsS0FBeUI7UUFDcEUsSUFBSSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxHQUFHLEdBQTJCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN6RSxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNoQyxPQUFPLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDL0IsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQ2IsQ0FBQztRQUNELGlEQUFpRDtJQUNuRCxDQUFDO0lBRWEsY0FBTSxHQUFwQixVQUFxQixNQUFpQixFQUFFLEtBQXlCO1FBQy9ELE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQzNDLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7SUFDbEMsQ0FBQztJQUVhLGlCQUFTLEdBQXZCLFVBQXdCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ2hGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxRQUFRLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3RixFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzFCLDJDQUEyQztZQUMzQyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsY0FBTSxDQUFDLGNBQWMsQ0FBQztRQUNuQyxDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN6QyxDQUFDO0lBQ0gsQ0FBQztJQUVhLHNCQUFjLEdBQTVCLFVBQTZCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ3JGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxRQUFRLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDMUYsR0FBRyxHQUFHLFFBQVEsQ0FBQyxHQUFHLEVBQ2xCLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN2QixDQUFDLEdBQThCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMvQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2pELElBQUksV0FBVyxHQUFHLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUN4QyxJQUFJLGNBQWMsR0FBRyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsZUFBZSxFQUFFLENBQUM7WUFDcEQsY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsZ0NBQWdDLEVBQUssY0FBYywyQkFBc0IsV0FBYSxDQUFDLENBQUM7UUFDeEgsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sV0FBVztZQUNYLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsa0JBQVUsR0FBeEIsVUFBeUIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDakYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFFBQVEsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzdGLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDMUIscUJBQXFCO1lBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxjQUFNLENBQUMsZUFBZSxDQUFDO1FBQ3BDLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLGdDQUFnQztZQUNoQyxhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN6QyxDQUFDO0lBQ0gsQ0FBQztJQUVhLHVCQUFlLEdBQTdCLFVBQThCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQ3RGLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxRQUFRLEdBQW9CLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFDMUYsR0FBRyxHQUFHLFFBQVEsQ0FBQyxHQUFHLEVBQ2xCLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUN2QixDQUFDLEdBQStCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNoRCxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxJQUFJLEdBQUcsQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUN0RSxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUNoQixDQUFDO0lBRWEsb0JBQVksR0FBMUIsVUFBMkIsTUFBaUIsRUFBRSxLQUF5QjtRQUNyRSxJQUFJLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLFVBQVUsR0FBOEIsT0FBTyxDQUFDLEdBQUcsRUFBRSxFQUNoRixjQUFjLEdBQUc7WUFDZiwrQ0FBK0M7WUFDL0Msb0JBQW9CO1lBQ3BCLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUNiLENBQUMsQ0FBQztRQUVKLEVBQUUsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLFVBQVUsRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzNELHFFQUFxRTtZQUNyRSxzRUFBc0U7WUFDdEUsbUJBQW1CO1lBQ25CLEtBQUssQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUM7UUFDbEMsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sY0FBYyxFQUFFLENBQUM7UUFDbkIsQ0FBQztJQUNILENBQUM7SUFFYSxtQkFBVyxHQUF6QixVQUEwQixNQUFpQixFQUFFLEtBQXlCO1FBQ3BFLElBQUksVUFBVSxHQUE4QixLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2hFLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3pDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUNiLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLDhDQUE4QztZQUM5QyxLQUFLLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDO1FBQ2xDLENBQUM7SUFDSCxDQUFDO0lBRWEsc0JBQWMsR0FBNUIsVUFBNkIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDckYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFFBQVEsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzdGLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDMUIscUJBQXFCO1lBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxjQUFNLENBQUMsbUJBQW1CLENBQUM7UUFDeEMsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sYUFBYSxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDekMsQ0FBQztJQUNILENBQUM7SUFFYSwyQkFBbUIsR0FBakMsVUFBa0MsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDMUYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFFBQVEsR0FBb0IsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUMxRixPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFDdkIsR0FBRyxHQUFHLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLEVBQ2xCLENBQVM7UUFDVCxnQ0FBZ0M7UUFDaEMsSUFBSSxHQUFHLElBQUksS0FBSyxDQUFTLEdBQUcsQ0FBQyxFQUFFLE9BQWUsQ0FBQztRQUVqRCxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxHQUFHLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztZQUN6QixPQUFPLEdBQUcsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ3hCLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLE9BQU8sQ0FBQztZQUM1QixFQUFFLENBQUMsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDaEIsY0FBYyxDQUFDLE1BQU0sRUFBRSxLQUFLLEVBQUUsd0NBQXdDLEVBQUUsbUJBQWlCLFFBQVEsQ0FBQyxHQUFHLENBQUMsZUFBZSxFQUFFLDBDQUFxQyxPQUFTLENBQUMsQ0FBQztnQkFDdkssTUFBTSxDQUFDO1lBQ1QsQ0FBQztRQUNILENBQUM7UUFDRCxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ3RFLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYSxjQUFNLEdBQXBCLFVBQXFCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzdFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLElBQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ3hDLEtBQUssQ0FBQyxFQUFFLElBQUksTUFBTSxDQUFDO1lBQ25CLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNmLEtBQUssQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDL0IsQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ2hCLENBQUM7SUFDSCxDQUFDO0lBRWEsaUJBQVMsR0FBdkIsVUFBd0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDaEYsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDaEMsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7WUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2YsS0FBSyxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMvQixDQUFDO1FBQ0gsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sS0FBSyxDQUFDLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFFYSxjQUFNLEdBQXBCLFVBQXFCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzdFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDeEMsS0FBSyxDQUFDLEVBQUUsSUFBSSxNQUFNLENBQUM7UUFDbkIsRUFBRSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDZixLQUFLLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDO1FBQy9CLENBQUM7SUFDSCxDQUFDO0lBRWEsYUFBSyxHQUFuQixVQUFvQixNQUFpQixFQUFFLEtBQXlCLEVBQUUsSUFBWTtRQUM1RSxJQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3BCLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDakMsS0FBSyxDQUFDLEVBQUUsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRWEsV0FBRyxHQUFqQixVQUFrQixNQUFpQixFQUFFLEtBQXlCO1FBQzVELEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYSxXQUFHLEdBQWpCLFVBQWtCLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzFFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxRQUFRLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDL0QsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUMxQixtQkFBTSxDQUFDLENBQUM7Z0JBQ04sTUFBTSxDQUFDLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQztvQkFDM0IsS0FBSyw0QkFBb0IsQ0FBQyxNQUFNLENBQUM7b0JBQ2pDLEtBQUssNEJBQW9CLENBQUMsS0FBSyxDQUFDO29CQUNoQyxLQUFLLDRCQUFvQixDQUFDLGFBQWEsQ0FBQztvQkFDeEMsS0FBSyw0QkFBb0IsQ0FBQyxXQUFXLENBQUM7b0JBQ3RDLEtBQUssNEJBQW9CLENBQUMsT0FBTyxDQUFDO29CQUNsQyxLQUFLLDRCQUFvQixDQUFDLEtBQUs7d0JBQzdCLE1BQU0sQ0FBQyxJQUFJLENBQUM7b0JBQ2Q7d0JBQ0UsTUFBTSxDQUFDLEtBQUssQ0FBQztnQkFDakIsQ0FBQztZQUNILENBQUMsQ0FBQyxFQUFFLEVBQUUsd0JBQXNCLDRCQUFvQixDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxpQ0FBOEIsQ0FBQyxDQUFDO1lBQ3BHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUNqRCxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN6QyxDQUFDO0lBQ0gsQ0FBQztJQUVhLGFBQUssR0FBbkIsVUFBb0IsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDNUUsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFFBQVEsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUUsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUMxQixtQkFBTSxDQUFDLENBQUM7Z0JBQ04sTUFBTSxDQUFDLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQztvQkFDM0IsS0FBSyw0QkFBb0IsQ0FBQyxNQUFNLENBQUM7b0JBQ2pDLEtBQUssNEJBQW9CLENBQUMsS0FBSyxDQUFDO29CQUNoQyxLQUFLLDRCQUFvQixDQUFDLGFBQWEsQ0FBQztvQkFDeEMsS0FBSyw0QkFBb0IsQ0FBQyxXQUFXLENBQUM7b0JBQ3RDLEtBQUssNEJBQW9CLENBQUMsT0FBTyxDQUFDO29CQUNsQyxLQUFLLDRCQUFvQixDQUFDLEtBQUs7d0JBQzdCLE1BQU0sQ0FBQyxJQUFJLENBQUM7b0JBQ2Q7d0JBQ0UsTUFBTSxDQUFDLEtBQUssQ0FBQztnQkFDakIsQ0FBQztZQUNILENBQUMsQ0FBQyxFQUFFLEVBQUUsd0JBQXNCLDRCQUFvQixDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxtQ0FBZ0MsQ0FBQyxDQUFDO1lBQ3RHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztZQUNqRCxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNoQixDQUFDO1FBQUMsSUFBSSxDQUFDLENBQUM7WUFDTixhQUFhLENBQUMsTUFBTSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN6QyxDQUFDO0lBQ0gsQ0FBQztJQUVhLGNBQU0sR0FBcEIsVUFBcUIsTUFBaUIsRUFBRSxLQUF5QixFQUFFLElBQVk7UUFDN0UsSUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUNwQixJQUFJLFFBQVEsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUUsbUJBQU0sQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFLEtBQUssNEJBQW9CLENBQUMsSUFBSTtlQUNsRCxRQUFRLENBQUMsT0FBTyxFQUFFLEtBQUssNEJBQW9CLENBQUMsTUFBTSxFQUNyRCx1Q0FBcUMsNEJBQW9CLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFHLENBQUMsQ0FBQztRQUNuRixLQUFLLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBNEIsUUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3ZFLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ2hCLENBQUM7SUFFYSxZQUFJLEdBQWxCLFVBQW1CLE1BQWlCLEVBQUUsS0FBeUIsRUFBRSxJQUFZO1FBQzNFLElBQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDcEIsSUFBSSxLQUFLLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDdEMsd0VBQXdFO1FBQ3hFLHdDQUF3QztRQUN4QyxLQUFLLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNkLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JCLEtBQUssY0FBTSxDQUFDLEtBQUssQ0FBQztZQUNsQixLQUFLLGNBQU0sQ0FBQyxLQUFLLENBQUM7WUFDbEIsS0FBSyxjQUFNLENBQUMsS0FBSztnQkFDZixLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7Z0JBQ3hDLEtBQUssQ0FBQztZQUNSLEtBQUssY0FBTSxDQUFDLEtBQUssQ0FBQztZQUNsQixLQUFLLGNBQU0sQ0FBQyxLQUFLO2dCQUNmLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztnQkFDaEQsS0FBSyxDQUFDO1lBQ1IsS0FBSyxjQUFNLENBQUMsTUFBTSxDQUFDO1lBQ25CLEtBQUssY0FBTSxDQUFDLE1BQU0sQ0FBQztZQUNuQixLQUFLLGNBQU0sQ0FBQyxNQUFNO2dCQUNoQixLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7Z0JBQzFDLEtBQUssQ0FBQztZQUNSLEtBQUssY0FBTSxDQUFDLE1BQU0sQ0FBQztZQUNuQixLQUFLLGNBQU0sQ0FBQyxNQUFNO2dCQUNoQixPQUFPO2dCQUNQLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxHQUFHLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7Z0JBQzlDLG9CQUFvQjtnQkFDcEIsS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO2dCQUMxQyxLQUFLLENBQUM7WUFDUixLQUFLLGNBQU0sQ0FBQyxHQUFHO2dCQUNiLEtBQUssQ0FBQyxFQUFFLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDL0IsS0FBSyxDQUFDO1lBQ1IsS0FBSyxjQUFNLENBQUMsSUFBSTtnQkFDZCxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDckMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUN4RCwrQkFBK0I7Z0JBQy9CLEtBQUssQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO2dCQUNkLEtBQUssQ0FBQztZQUNSO2dCQUNFLG1CQUFNLENBQUMsS0FBSyxFQUFFLDBCQUF3QixJQUFJLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBRyxDQUFDLENBQUM7Z0JBQ3RELEtBQUssQ0FBQztRQUNWLENBQUM7SUFDSCxDQUFDO0lBejZEYSxjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQXlCM0IsY0FBTSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUFDM0IsY0FBTSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUF5QjNCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQzdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQzdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQzdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQzdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQzdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBeUI3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQTRCN0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQy9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFpQi9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBa0QvQixhQUFLLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUN6QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixhQUFLLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUN6QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixhQUFLLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztJQUN6QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQUM3QixlQUFPLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztJQTZCN0IsYUFBSyxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUM7SUFDekIsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsYUFBSyxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUM7SUFDekIsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUE2QjdCLGNBQU0sR0FBRyxPQUFPLENBQUMsU0FBUyxDQUFDO0lBQzNCLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQy9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQy9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsY0FBTSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUFDM0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQy9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBcUMvQixjQUFNLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQztJQUMzQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQy9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsY0FBTSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUM7SUFDM0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBQy9CLGdCQUFRLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztJQUMvQixnQkFBUSxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUM7SUFDL0IsZ0JBQVEsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO0lBb3hCL0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFDN0IsZUFBTyxHQUFHLE9BQU8sQ0FBQyxVQUFVLENBQUM7SUFnQjdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBQzdCLGVBQU8sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDO0lBaVg3Qiw0QkFBb0IsR0FBRyxPQUFPLENBQUMsa0JBQWtCLENBQUM7SUEyY2xFLGNBQUM7QUFBRCxDQUFDLEFBaDhERCxJQWc4REM7QUFoOERZLGVBQU8sVUFnOERuQixDQUFBO0FBRVUsbUJBQVcsR0FBNEIsSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDbEUsc0RBQXNEO0FBQ3RELENBQUM7SUFDQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO1FBQzlCLEVBQUUsQ0FBQyxDQUFDLGNBQU0sQ0FBQyxjQUFjLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNsQyxtQkFBVyxDQUFDLENBQUMsQ0FBQyxHQUFVLE9BQVEsQ0FBQyxjQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztZQUMxRCxtQkFBTSxDQUFDLG1CQUFXLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxFQUFFLHNDQUFvQyxjQUFNLENBQUMsQ0FBQyxDQUFHLENBQUMsQ0FBQztRQUNsRixDQUFDO0lBQ0gsQ0FBQztBQUNILENBQUMsQ0FBQyxFQUFFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvbnRhaW5zIEpWTSBvcGNvZGUgaW1wbGVtZW50YXRpb25zIGZvciB0aGUgSlZNIGludGVycHJldGVyLlxuICovXG4vLyBXZSB1c2Ugc25ha2UgY2FzZSBmb3IgdGhlIG9wY29kZSBuYW1lcyBzbyB0aGV5IG1hdGNoIHRoZSBKVk0gc3BlYy5cbi8vIEFzIGZvciB0eXBlZGVmOm1lbWJlclZhcmlhYmxlRGVjbGFyYXRvciwgd2UgZGlzYWJsZSB0aGlzIHNvIHdlIGNhbiBoYXZlXG4vLyBtZW1iZXIgdmFyaWFibGUgb3Bjb2RlcyB3aXRob3V0IGV4cGxpY2l0bHkgdHlwaW5nIHRoZW0gYXMgSU9wY29kZUltcGxlbWVudGF0aW9uLlxuLyogdHNsaW50OmRpc2FibGU6dmFyaWFibGUtbmFtZSB0eXBlZGVmOm1lbWJlclZhcmlhYmxlRGVjbGFyYXRvciAqL1xuaW1wb3J0IGdMb25nIGZyb20gJy4vZ0xvbmcnO1xuaW1wb3J0IHt3cmFwRmxvYXQsIGZsb2F0MmludCwgZGVzY3JpcHRvcjJ0eXBlc3RyfSBmcm9tICcuL3V0aWwnO1xuaW1wb3J0IHtDbGFzc1JlZmVyZW5jZSwgRmllbGRSZWZlcmVuY2UsIE1ldGhvZFJlZmVyZW5jZSwgSW50ZXJmYWNlTWV0aG9kUmVmZXJlbmNlLCBJQ29uc3RhbnRQb29sSXRlbSwgSW52b2tlRHluYW1pYywgQ29uc3REb3VibGUsIENvbnN0TG9uZ30gZnJvbSAnLi9Db25zdGFudFBvb2wnO1xuaW1wb3J0IHtDbGFzc0RhdGEsIEFycmF5Q2xhc3NEYXRhfSBmcm9tICcuL0NsYXNzRGF0YSc7XG5pbXBvcnQge0pWTVRocmVhZCwgQnl0ZWNvZGVTdGFja0ZyYW1lfSBmcm9tICcuL3RocmVhZGluZyc7XG5pbXBvcnQge1RocmVhZFN0YXR1cywgT3BDb2RlLCBDb25zdGFudFBvb2xJdGVtVHlwZSwgQ29uc3RhbnRzfSBmcm9tICcuL2VudW1zJztcbmltcG9ydCBhc3NlcnQgZnJvbSAnLi9hc3NlcnQnO1xuaW1wb3J0IHtNZXRob2R9IGZyb20gJy4vbWV0aG9kcyc7XG5pbXBvcnQgKiBhcyBKVk1UeXBlcyBmcm9tICcuLi9pbmNsdWRlcy9KVk1UeXBlcyc7XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBpbmRpdmlkdWFsIG9wY29kZSBpbXBsZW1lbnRhdGlvbnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU9wY29kZUltcGxlbWVudGF0aW9uIHtcbiAgKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlPzogQnVmZmVyKTogdm9pZDtcbn1cblxuLyoqXG4gKiBIZWxwZXIgZnVuY3Rpb246IENoZWNrcyBpZiBvYmplY3QgaXMgbnVsbC4gVGhyb3dzIGEgTnVsbFBvaW50ZXJFeGNlcHRpb25cbiAqIGlmIGl0IGlzLlxuICogQHJldHVybiBUcnVlIGlmIHRoZSBvYmplY3QgaXMgbnVsbC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzTnVsbCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgb2JqOiBhbnkpOiBib29sZWFuIHtcbiAgaWYgKG9iaiA9PSBudWxsKSB7XG4gICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvTnVsbFBvaW50ZXJFeGNlcHRpb247JywgJycpO1xuICAgIHJldHVybiB0cnVlO1xuICB9XG4gIHJldHVybiBmYWxzZTtcbn1cblxuLyoqXG4gKiBIZWxwZXIgZnVuY3Rpb246IFBvcHMgb2ZmIHR3byBpdGVtcywgcmV0dXJucyB0aGUgc2Vjb25kLlxuICovXG5leHBvcnQgZnVuY3Rpb24gcG9wMihvcFN0YWNrOiBhbnlbXSk6IGFueSB7XG4gIC8vIElnbm9yZSBOVUxMLlxuICBvcFN0YWNrLnBvcCgpO1xuICByZXR1cm4gb3BTdGFjay5wb3AoKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmVDUEl0ZW0odGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNwSXRlbTogSUNvbnN0YW50UG9vbEl0ZW0pOiB2b2lkIHtcbiAgdGhyZWFkLnNldFN0YXR1cyhUaHJlYWRTdGF0dXMuQVNZTkNfV0FJVElORyk7XG4gIGNwSXRlbS5yZXNvbHZlKHRocmVhZCwgZnJhbWUuZ2V0TG9hZGVyKCksIGZyYW1lLm1ldGhvZC5jbHMsIChzdGF0dXM6IGJvb2xlYW4pID0+IHtcbiAgICBpZiAoc3RhdHVzKSB7XG4gICAgICB0aHJlYWQuc2V0U3RhdHVzKFRocmVhZFN0YXR1cy5SVU5OQUJMRSk7XG4gICAgfVxuICB9LCBmYWxzZSk7XG4gIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpbml0aWFsaXplQ2xhc3NGcm9tQ2xhc3ModGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNsczogQ2xhc3NEYXRhKTogdm9pZCB7XG4gIHRocmVhZC5zZXRTdGF0dXMoVGhyZWFkU3RhdHVzLkFTWU5DX1dBSVRJTkcpO1xuICBjbHMuaW5pdGlhbGl6ZSh0aHJlYWQsIChjZGF0YTogQ2xhc3NEYXRhKSA9PiB7XG4gICAgaWYgKGNkYXRhICE9IG51bGwpIHtcbiAgICAgIHRocmVhZC5zZXRTdGF0dXMoVGhyZWFkU3RhdHVzLlJVTk5BQkxFKTtcbiAgICB9XG4gIH0sIGZhbHNlKTtcbiAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbn1cblxuLyoqXG4gKiBIZWxwZXIgZnVuY3Rpb246IFBhdXNlcyB0aGUgdGhyZWFkIGFuZCBpbml0aWFsaXplcyBhIGNsYXNzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gaW5pdGlhbGl6ZUNsYXNzKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjbHNSZWY6IENsYXNzUmVmZXJlbmNlKTogdm9pZCB7XG4gIHRocmVhZC5zZXRTdGF0dXMoVGhyZWFkU3RhdHVzLkFTWU5DX1dBSVRJTkcpO1xuXG4gIGZ1bmN0aW9uIGluaXRpYWxpemUoY2xzOiBDbGFzc0RhdGEpIHtcbiAgICBjbHMuaW5pdGlhbGl6ZSh0aHJlYWQsIChjZGF0YTogQ2xhc3NEYXRhKSA9PiB7XG4gICAgICBpZiAoY2RhdGEgIT0gbnVsbCkge1xuICAgICAgICB0aHJlYWQuc2V0U3RhdHVzKFRocmVhZFN0YXR1cy5SVU5OQUJMRSk7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICBpZiAoIWNsc1JlZi5pc1Jlc29sdmVkKCkpIHtcbiAgICBjbHNSZWYucmVzb2x2ZSh0aHJlYWQsIGZyYW1lLmdldExvYWRlcigpLCBmcmFtZS5tZXRob2QuY2xzLCAoc3RhdHVzOiBib29sZWFuKSA9PiB7XG4gICAgICBpZiAoc3RhdHVzKSB7XG4gICAgICAgIGluaXRpYWxpemUoY2xzUmVmLmNscyk7XG4gICAgICB9XG4gICAgfSwgZmFsc2UpO1xuICB9IGVsc2Uge1xuICAgIGluaXRpYWxpemUoY2xzUmVmLmNscyk7XG4gIH1cbiAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbn1cblxuLyoqXG4gKiBJbnRlcnJ1cHRzIHRoZSBjdXJyZW50IG1ldGhvZCdzIGV4ZWN1dGlvbiBhbmQgdGhyb3dzIGFuIGV4Y2VwdGlvbi5cbiAqXG4gKiBOT1RFOiBUaGlzIGRvZXMgKm5vdCogaW50ZXJydXB0IEphdmFTY3JpcHQgY29udHJvbCBmbG93LCBzbyBhbnkgb3Bjb2RlXG4gKiBjYWxsaW5nIHRoaXMgZnVuY3Rpb24gbXVzdCAqcmV0dXJuKiBhbmQgbm90IGRvIGFueXRoaW5nIGVsc2UuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB0aHJvd0V4Y2VwdGlvbjxUIGV4dGVuZHMgSlZNVHlwZXMuamF2YV9sYW5nX1Rocm93YWJsZT4odGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNsc05hbWU6IHN0cmluZywgbXNnOiBzdHJpbmcpOiB2b2lkIHtcbiAgdGhyZWFkLnRocm93TmV3RXhjZXB0aW9uPFQ+KGNsc05hbWUsIG1zZyk7XG4gIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG59XG5cbmV4cG9ydCB2YXIgQXJyYXlUeXBlcyA6IHtbdDogbnVtYmVyXTogc3RyaW5nOyB9ID0ge1xuICA0OiAnWicsIDU6ICdDJywgNjogJ0YnLCA3OiAnRCcsIDg6ICdCJywgOTogJ1MnLCAxMDogJ0knLCAxMTogJ0onXG59O1xuXG4vKipcbiAqIENvbnRhaW5zIGRlZmluaXRpb25zIGZvciBhbGwgSlZNIG9wY29kZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBPcGNvZGVzIHtcbiAgLyogMzItYml0IGFycmF5IGxvYWQgb3Bjb2RlcyAqL1xuXG4gIC8qKlxuICAgKiAzMi1iaXQgYXJyYXkgbG9hZCBvcGNvZGVcbiAgICovXG4gIHByaXZhdGUgc3RhdGljIF9hbG9hZF8zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGlkeCA9IG9wU3RhY2sucG9wKCksXG4gICAgICBvYmogPSA8SlZNVHlwZXMuSlZNQXJyYXk8YW55Pj4gb3BTdGFjay5wb3AoKTtcbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICB2YXIgbGVuID0gb2JqLmFycmF5Lmxlbmd0aDtcbiAgICAgIGlmIChpZHggPCAwIHx8IGlkeCA+PSBsZW4pIHtcbiAgICAgICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvQXJyYXlJbmRleE91dE9mQm91bmRzRXhjZXB0aW9uOycsIGAke2lkeH0gbm90IGluIGxlbmd0aCAke2xlbn0gYXJyYXkgb2YgdHlwZSAke29iai5nZXRDbGFzcygpLmdldEludGVybmFsTmFtZSgpfWApO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgb3BTdGFjay5wdXNoKG9iai5hcnJheVtpZHhdKTtcbiAgICAgICAgZnJhbWUucGMrKztcbiAgICAgIH1cbiAgICB9XG4gICAgLy8gJ29iaicgaXMgTlVMTC4gaXNOdWxsIHRocmV3IGFuIGV4Y2VwdGlvbiBmb3IgdXMuXG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzMyO1xuICBwdWJsaWMgc3RhdGljIGZhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzMyO1xuICBwdWJsaWMgc3RhdGljIGFhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzMyO1xuICBwdWJsaWMgc3RhdGljIGJhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzMyO1xuICBwdWJsaWMgc3RhdGljIGNhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzMyO1xuICBwdWJsaWMgc3RhdGljIHNhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzMyO1xuXG4gIC8qIDY0LWJpdCBhcnJheSBsb2FkIG9wY29kZXMgKi9cblxuICAvKipcbiAgICogNjQtYml0IGFycmF5IGxvYWQgb3Bjb2RlLlxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgX2Fsb2FkXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgaWR4ID0gb3BTdGFjay5wb3AoKSxcbiAgICAgIG9iaiA9IDxKVk1UeXBlcy5KVk1BcnJheTxhbnk+PiBvcFN0YWNrLnBvcCgpO1xuICAgIGlmICghaXNOdWxsKHRocmVhZCwgZnJhbWUsIG9iaikpIHtcbiAgICAgIHZhciBsZW4gPSBvYmouYXJyYXkubGVuZ3RoO1xuICAgICAgaWYgKGlkeCA8IDAgfHwgaWR4ID49IGxlbikge1xuICAgICAgICB0aHJvd0V4Y2VwdGlvbih0aHJlYWQsIGZyYW1lLCAnTGphdmEvbGFuZy9BcnJheUluZGV4T3V0T2ZCb3VuZHNFeGNlcHRpb247JywgYCR7aWR4fSBub3QgaW4gbGVuZ3RoICR7bGVufSBhcnJheSBvZiB0eXBlICR7b2JqLmdldENsYXNzKCkuZ2V0SW50ZXJuYWxOYW1lKCl9YCk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBvcFN0YWNrLnB1c2gob2JqLmFycmF5W2lkeF0pO1xuICAgICAgICAvLyA2NC1iaXQgdmFsdWUuXG4gICAgICAgIG9wU3RhY2sucHVzaChudWxsKTtcbiAgICAgICAgZnJhbWUucGMrKztcbiAgICAgIH1cbiAgICB9XG4gICAgLy8gJ29iaicgaXMgTlVMTC4gaXNOdWxsIHRocmV3IGFuIGV4Y2VwdGlvbiBmb3IgdXMuXG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGRhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzY0O1xuICBwdWJsaWMgc3RhdGljIGxhbG9hZCA9IE9wY29kZXMuX2Fsb2FkXzY0O1xuXG4gIC8qIDMyLWJpdCBhcnJheSBzdG9yZSBvcGNvZGVzICovXG5cbiAgLyoqXG4gICAqIDMyLWJpdCBhcnJheSBzdG9yZS5cbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByaXZhdGUgc3RhdGljIF9hc3RvcmVfMzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICB2YWx1ZSA9IG9wU3RhY2sucG9wKCksXG4gICAgICBpZHggPSBvcFN0YWNrLnBvcCgpLFxuICAgICAgb2JqID0gPEpWTVR5cGVzLkpWTUFycmF5PGFueT4+IG9wU3RhY2sucG9wKCk7XG4gICAgaWYgKCFpc051bGwodGhyZWFkLCBmcmFtZSwgb2JqKSkge1xuICAgICAgdmFyIGxlbiA9IG9iai5hcnJheS5sZW5ndGg7XG4gICAgICBpZiAoaWR4IDwgMCB8fCBpZHggPj0gbGVuKSB7XG4gICAgICAgIHRocm93RXhjZXB0aW9uKHRocmVhZCwgZnJhbWUsICdMamF2YS9sYW5nL0FycmF5SW5kZXhPdXRPZkJvdW5kc0V4Y2VwdGlvbjsnLCBgJHtpZHh9IG5vdCBpbiBsZW5ndGggJHtsZW59IGFycmF5IG9mIHR5cGUgJHtvYmouZ2V0Q2xhc3MoKS5nZXRJbnRlcm5hbE5hbWUoKX1gKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIG9iai5hcnJheVtpZHhdID0gdmFsdWU7XG4gICAgICAgIGZyYW1lLnBjKys7XG4gICAgICB9XG4gICAgfVxuICAgIC8vICdvYmonIGlzIE5VTEwuIGlzTnVsbCB0aHJldyBhbiBleGNlcHRpb24gZm9yIHVzLlxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpYXN0b3JlID0gT3Bjb2Rlcy5fYXN0b3JlXzMyO1xuICBwdWJsaWMgc3RhdGljIGZhc3RvcmUgPSBPcGNvZGVzLl9hc3RvcmVfMzI7XG4gIHB1YmxpYyBzdGF0aWMgYWFzdG9yZSA9IE9wY29kZXMuX2FzdG9yZV8zMjtcbiAgcHVibGljIHN0YXRpYyBiYXN0b3JlID0gT3Bjb2Rlcy5fYXN0b3JlXzMyO1xuICBwdWJsaWMgc3RhdGljIGNhc3RvcmUgPSBPcGNvZGVzLl9hc3RvcmVfMzI7XG4gIHB1YmxpYyBzdGF0aWMgc2FzdG9yZSA9IE9wY29kZXMuX2FzdG9yZV8zMjtcblxuICAvKiA2NC1iaXQgYXJyYXkgc3RvcmUgb3Bjb2RlcyAqL1xuXG4gIC8qKlxuICAgKiA2NC1iaXQgYXJyYXkgc3RvcmUuXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyBfYXN0b3JlXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgdmFsdWUgPSBvcFN0YWNrLnBvcDIoKSxcbiAgICAgIGlkeCA9IG9wU3RhY2sucG9wKCksXG4gICAgICBvYmogPSA8SlZNVHlwZXMuSlZNQXJyYXk8YW55Pj4gb3BTdGFjay5wb3AoKTtcbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICB2YXIgbGVuID0gb2JqLmFycmF5Lmxlbmd0aDtcbiAgICAgIGlmIChpZHggPCAwIHx8IGlkeCA+PSBsZW4pIHtcbiAgICAgICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvQXJyYXlJbmRleE91dE9mQm91bmRzRXhjZXB0aW9uOycsIGAke2lkeH0gbm90IGluIGxlbmd0aCAke2xlbn0gYXJyYXkgb2YgdHlwZSAke29iai5nZXRDbGFzcygpLmdldEludGVybmFsTmFtZSgpfWApO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgb2JqLmFycmF5W2lkeF0gPSB2YWx1ZTtcbiAgICAgICAgZnJhbWUucGMrKztcbiAgICAgIH1cbiAgICB9XG4gICAgLy8gJ29iaicgaXMgTlVMTC4gaXNOdWxsIHRocmV3IGFuIGV4Y2VwdGlvbiBmb3IgdXMuXG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxhc3RvcmUgPSBPcGNvZGVzLl9hc3RvcmVfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgZGFzdG9yZSA9IE9wY29kZXMuX2FzdG9yZV82NDtcblxuICAvKiAzMi1iaXQgY29uc3RhbnRzICovXG4gIHB1YmxpYyBzdGF0aWMgYWNvbnN0X251bGwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2gobnVsbCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIF9jb25zdF8wXzMyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKDApO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfY29uc3RfMV8zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaCgxKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX2NvbnN0XzJfMzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2goMik7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWNvbnN0X20xKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKC0xKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpY29uc3RfMCA9IE9wY29kZXMuX2NvbnN0XzBfMzI7XG4gIHB1YmxpYyBzdGF0aWMgaWNvbnN0XzEgPSBPcGNvZGVzLl9jb25zdF8xXzMyO1xuICBwdWJsaWMgc3RhdGljIGljb25zdF8yID0gT3Bjb2Rlcy5fY29uc3RfMl8zMjtcblxuICBwdWJsaWMgc3RhdGljIGljb25zdF8zKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKDMpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGljb25zdF80KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKDQpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGljb25zdF81KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKDUpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZjb25zdF8wID0gT3Bjb2Rlcy5fY29uc3RfMF8zMjtcbiAgcHVibGljIHN0YXRpYyBmY29uc3RfMSA9IE9wY29kZXMuX2NvbnN0XzFfMzI7XG4gIHB1YmxpYyBzdGF0aWMgZmNvbnN0XzIgPSBPcGNvZGVzLl9jb25zdF8yXzMyO1xuXG4gIC8qIDY0LWJpdCBjb25zdGFudHMgKi9cbiAgcHVibGljIHN0YXRpYyBsY29uc3RfMCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaFdpdGhOdWxsKGdMb25nLlpFUk8pO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxjb25zdF8xKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoV2l0aE51bGwoZ0xvbmcuT05FKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBkY29uc3RfMCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaFdpdGhOdWxsKDApO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGRjb25zdF8xKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoV2l0aE51bGwoMSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIC8qIDMyLWJpdCBsb2FkIG9wY29kZXMgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgX2xvYWRfMzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKGZyYW1lLmxvY2Fsc1tjb2RlW3BjICsgMV1dKTtcbiAgICBmcmFtZS5wYyArPSAyO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX2xvYWRfMF8zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaChmcmFtZS5sb2NhbHNbMF0pO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfbG9hZF8xXzMyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKGZyYW1lLmxvY2Fsc1sxXSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIF9sb2FkXzJfMzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2goZnJhbWUubG9jYWxzWzJdKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX2xvYWRfM18zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaChmcmFtZS5sb2NhbHNbM10pO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlsb2FkID0gT3Bjb2Rlcy5fbG9hZF8zMjtcbiAgcHVibGljIHN0YXRpYyBpbG9hZF8wID0gT3Bjb2Rlcy5fbG9hZF8wXzMyO1xuICBwdWJsaWMgc3RhdGljIGlsb2FkXzEgPSBPcGNvZGVzLl9sb2FkXzFfMzI7XG4gIHB1YmxpYyBzdGF0aWMgaWxvYWRfMiA9IE9wY29kZXMuX2xvYWRfMl8zMjtcbiAgcHVibGljIHN0YXRpYyBpbG9hZF8zID0gT3Bjb2Rlcy5fbG9hZF8zXzMyO1xuICBwdWJsaWMgc3RhdGljIGZsb2FkID0gT3Bjb2Rlcy5fbG9hZF8zMjtcbiAgcHVibGljIHN0YXRpYyBmbG9hZF8wID0gT3Bjb2Rlcy5fbG9hZF8wXzMyO1xuICBwdWJsaWMgc3RhdGljIGZsb2FkXzEgPSBPcGNvZGVzLl9sb2FkXzFfMzI7XG4gIHB1YmxpYyBzdGF0aWMgZmxvYWRfMiA9IE9wY29kZXMuX2xvYWRfMl8zMjtcbiAgcHVibGljIHN0YXRpYyBmbG9hZF8zID0gT3Bjb2Rlcy5fbG9hZF8zXzMyO1xuICBwdWJsaWMgc3RhdGljIGFsb2FkID0gT3Bjb2Rlcy5fbG9hZF8zMjtcbiAgcHVibGljIHN0YXRpYyBhbG9hZF8wID0gT3Bjb2Rlcy5fbG9hZF8wXzMyO1xuICBwdWJsaWMgc3RhdGljIGFsb2FkXzEgPSBPcGNvZGVzLl9sb2FkXzFfMzI7XG4gIHB1YmxpYyBzdGF0aWMgYWxvYWRfMiA9IE9wY29kZXMuX2xvYWRfMl8zMjtcbiAgcHVibGljIHN0YXRpYyBhbG9hZF8zID0gT3Bjb2Rlcy5fbG9hZF8zXzMyO1xuXG4gIC8qIDY0LWJpdCBsb2FkIG9wY29kZXMgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgX2xvYWRfNjQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoV2l0aE51bGwoZnJhbWUubG9jYWxzW2NvZGVbcGMgKyAxXV0pO1xuICAgIGZyYW1lLnBjICs9IDI7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfbG9hZF8wXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoV2l0aE51bGwoZnJhbWUubG9jYWxzWzBdKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX2xvYWRfMV82NCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaFdpdGhOdWxsKGZyYW1lLmxvY2Fsc1sxXSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIF9sb2FkXzJfNjQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2hXaXRoTnVsbChmcmFtZS5sb2NhbHNbMl0pO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfbG9hZF8zXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoV2l0aE51bGwoZnJhbWUubG9jYWxzWzNdKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsbG9hZCA9IE9wY29kZXMuX2xvYWRfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgbGxvYWRfMCA9IE9wY29kZXMuX2xvYWRfMF82NDtcbiAgcHVibGljIHN0YXRpYyBsbG9hZF8xID0gT3Bjb2Rlcy5fbG9hZF8xXzY0O1xuICBwdWJsaWMgc3RhdGljIGxsb2FkXzIgPSBPcGNvZGVzLl9sb2FkXzJfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgbGxvYWRfMyA9IE9wY29kZXMuX2xvYWRfM182NDtcbiAgcHVibGljIHN0YXRpYyBkbG9hZCA9IE9wY29kZXMuX2xvYWRfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgZGxvYWRfMCA9IE9wY29kZXMuX2xvYWRfMF82NDtcbiAgcHVibGljIHN0YXRpYyBkbG9hZF8xID0gT3Bjb2Rlcy5fbG9hZF8xXzY0O1xuICBwdWJsaWMgc3RhdGljIGRsb2FkXzIgPSBPcGNvZGVzLl9sb2FkXzJfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgZGxvYWRfMyA9IE9wY29kZXMuX2xvYWRfM182NDtcblxuICAvKiAzMi1iaXQgc3RvcmUgb3Bjb2RlcyAqL1xuICBwcml2YXRlIHN0YXRpYyBfc3RvcmVfMzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgZnJhbWUubG9jYWxzW2NvZGVbcGMgKyAxXV0gPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIGZyYW1lLnBjICs9IDI7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfc3RvcmVfMF8zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLmxvY2Fsc1swXSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIF9zdG9yZV8xXzMyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUubG9jYWxzWzFdID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX3N0b3JlXzJfMzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5sb2NhbHNbMl0gPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfc3RvcmVfM18zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLmxvY2Fsc1szXSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaXN0b3JlID0gT3Bjb2Rlcy5fc3RvcmVfMzI7XG4gIHB1YmxpYyBzdGF0aWMgaXN0b3JlXzAgPSBPcGNvZGVzLl9zdG9yZV8wXzMyO1xuICBwdWJsaWMgc3RhdGljIGlzdG9yZV8xID0gT3Bjb2Rlcy5fc3RvcmVfMV8zMjtcbiAgcHVibGljIHN0YXRpYyBpc3RvcmVfMiA9IE9wY29kZXMuX3N0b3JlXzJfMzI7XG4gIHB1YmxpYyBzdGF0aWMgaXN0b3JlXzMgPSBPcGNvZGVzLl9zdG9yZV8zXzMyO1xuICBwdWJsaWMgc3RhdGljIGZzdG9yZSA9IE9wY29kZXMuX3N0b3JlXzMyO1xuICBwdWJsaWMgc3RhdGljIGZzdG9yZV8wID0gT3Bjb2Rlcy5fc3RvcmVfMF8zMjtcbiAgcHVibGljIHN0YXRpYyBmc3RvcmVfMSA9IE9wY29kZXMuX3N0b3JlXzFfMzI7XG4gIHB1YmxpYyBzdGF0aWMgZnN0b3JlXzIgPSBPcGNvZGVzLl9zdG9yZV8yXzMyO1xuICBwdWJsaWMgc3RhdGljIGZzdG9yZV8zID0gT3Bjb2Rlcy5fc3RvcmVfM18zMjtcbiAgcHVibGljIHN0YXRpYyBhc3RvcmUgPSBPcGNvZGVzLl9zdG9yZV8zMjtcbiAgcHVibGljIHN0YXRpYyBhc3RvcmVfMCA9IE9wY29kZXMuX3N0b3JlXzBfMzI7XG4gIHB1YmxpYyBzdGF0aWMgYXN0b3JlXzEgPSBPcGNvZGVzLl9zdG9yZV8xXzMyO1xuICBwdWJsaWMgc3RhdGljIGFzdG9yZV8yID0gT3Bjb2Rlcy5fc3RvcmVfMl8zMjtcbiAgcHVibGljIHN0YXRpYyBhc3RvcmVfMyA9IE9wY29kZXMuX3N0b3JlXzNfMzI7XG5cbiAgLyogNjQtYml0IHN0b3JlIG9wY29kZXMgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgX3N0b3JlXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBvZmZzZXQgPSBjb2RlW3BjICsgMV07XG4gICAgLy8gTlVMTFxuICAgIGZyYW1lLmxvY2Fsc1tvZmZzZXQgKyAxXSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgLy8gVGhlIGFjdHVhbCB2YWx1ZS5cbiAgICBmcmFtZS5sb2NhbHNbb2Zmc2V0XSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgZnJhbWUucGMgKz0gMjtcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIF9zdG9yZV8wXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUubG9jYWxzWzFdID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBmcmFtZS5sb2NhbHNbMF0gPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBfc3RvcmVfMV82NCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLmxvY2Fsc1syXSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgZnJhbWUubG9jYWxzWzFdID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgX3N0b3JlXzJfNjQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5sb2NhbHNbM10gPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIGZyYW1lLmxvY2Fsc1syXSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHByaXZhdGUgc3RhdGljIF9zdG9yZV8zXzY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUubG9jYWxzWzRdID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBmcmFtZS5sb2NhbHNbM10gPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxzdG9yZSA9IE9wY29kZXMuX3N0b3JlXzY0O1xuICBwdWJsaWMgc3RhdGljIGxzdG9yZV8wID0gT3Bjb2Rlcy5fc3RvcmVfMF82NDtcbiAgcHVibGljIHN0YXRpYyBsc3RvcmVfMSA9IE9wY29kZXMuX3N0b3JlXzFfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgbHN0b3JlXzIgPSBPcGNvZGVzLl9zdG9yZV8yXzY0O1xuICBwdWJsaWMgc3RhdGljIGxzdG9yZV8zID0gT3Bjb2Rlcy5fc3RvcmVfM182NDtcbiAgcHVibGljIHN0YXRpYyBkc3RvcmUgPSBPcGNvZGVzLl9zdG9yZV82NDtcbiAgcHVibGljIHN0YXRpYyBkc3RvcmVfMCA9IE9wY29kZXMuX3N0b3JlXzBfNjQ7XG4gIHB1YmxpYyBzdGF0aWMgZHN0b3JlXzEgPSBPcGNvZGVzLl9zdG9yZV8xXzY0O1xuICBwdWJsaWMgc3RhdGljIGRzdG9yZV8yID0gT3Bjb2Rlcy5fc3RvcmVfMl82NDtcbiAgcHVibGljIHN0YXRpYyBkc3RvcmVfMyA9IE9wY29kZXMuX3N0b3JlXzNfNjQ7XG5cbiAgLyogTWlzYy4gKi9cblxuICBwdWJsaWMgc3RhdGljIHNpcHVzaCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2goY29kZS5yZWFkSW50MTZCRShwYyArIDEpKTtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBiaXB1c2godGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKGNvZGUucmVhZEludDgocGMgKyAxKSk7XG4gICAgZnJhbWUucGMgKz0gMjtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgcG9wKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5kcm9wRnJvbVRvcCgxKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBwb3AyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgLy8gaHR0cDovL2kuaW1ndXIuY29tL01pZUYwS0cuanBnXG4gICAgZnJhbWUub3BTdGFjay5kcm9wRnJvbVRvcCgyKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBkdXAodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5vcFN0YWNrLmR1cCgpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGR1cF94MSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2suZHVwX3gxKCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZHVwX3gyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5kdXBfeDIoKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBkdXAyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5kdXAyKCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZHVwMl94MSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2suZHVwMl94MSgpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGR1cDJfeDIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICB2MSA9IG9wU3RhY2sucG9wKCksXG4gICAgICB2MiA9IG9wU3RhY2sucG9wKCksXG4gICAgICB2MyA9IG9wU3RhY2sucG9wKCksXG4gICAgICB2NCA9IG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoNih2MiwgdjEsIHY0LCB2MywgdjIsIHYxKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBzd2FwKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgZnJhbWUub3BTdGFjay5zd2FwKCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIC8qIE1hdGggT3Bjb2RlcyAqL1xuICBwdWJsaWMgc3RhdGljIGlhZGQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKChvcFN0YWNrLnBvcCgpICsgb3BTdGFjay5wb3AoKSkgfCAwKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsYWRkKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKG9wU3RhY2sucG9wMigpLmFkZChvcFN0YWNrLnBvcDIoKSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZhZGQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKHdyYXBGbG9hdChvcFN0YWNrLnBvcCgpICsgb3BTdGFjay5wb3AoKSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGRhZGQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoV2l0aE51bGwob3BTdGFjay5wb3AyKCkgKyBvcFN0YWNrLnBvcDIoKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaXN1Yih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2goKC1vcFN0YWNrLnBvcCgpICsgb3BTdGFjay5wb3AoKSkgfCAwKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBmc3ViKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaCh3cmFwRmxvYXQoLW9wU3RhY2sucG9wKCkgKyBvcFN0YWNrLnBvcCgpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZHN1Yih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbCgtb3BTdGFjay5wb3AyKCkgKyBvcFN0YWNrLnBvcDIoKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbHN1Yih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChvcFN0YWNrLnBvcDIoKS5uZWdhdGUoKS5hZGQob3BTdGFjay5wb3AyKCkpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpbXVsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaCgoPGFueT4gTWF0aCkuaW11bChvcFN0YWNrLnBvcCgpLCBvcFN0YWNrLnBvcCgpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbG11bCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChvcFN0YWNrLnBvcDIoKS5tdWx0aXBseShvcFN0YWNrLnBvcDIoKSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZtdWwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKHdyYXBGbG9hdChvcFN0YWNrLnBvcCgpICogb3BTdGFjay5wb3AoKSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGRtdWwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoV2l0aE51bGwob3BTdGFjay5wb3AyKCkgKiBvcFN0YWNrLnBvcDIoKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWRpdih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaywgYjogbnVtYmVyID0gb3BTdGFjay5wb3AoKSwgYTogbnVtYmVyID0gb3BTdGFjay5wb3AoKTtcbiAgICBpZiAoYiA9PT0gMCkge1xuICAgICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvQXJpdGhtZXRpY0V4Y2VwdGlvbjsnLCAnLyBieSB6ZXJvJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIHNwZWM6IFwiaWYgdGhlIGRpdmlkZW5kIGlzIHRoZSBuZWdhdGl2ZSBpbnRlZ2VyIG9mIGxhcmdlc3QgcG9zc2libGUgbWFnbml0dWRlXG4gICAgICAvLyBmb3IgdGhlIGludCB0eXBlLCBhbmQgdGhlIGRpdmlzb3IgaXMgLTEsIHRoZW4gb3ZlcmZsb3cgb2NjdXJzLCBhbmQgdGhlXG4gICAgICAvLyByZXN1bHQgaXMgZXF1YWwgdG8gdGhlIGRpdmlkZW5kLlwiXG4gICAgICBpZiAoYSA9PT0gQ29uc3RhbnRzLklOVF9NSU4gJiYgYiA9PT0gLTEpIHtcbiAgICAgICAgb3BTdGFjay5wdXNoKGEpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgb3BTdGFjay5wdXNoKChhIC8gYikgfCAwKTtcbiAgICAgIH1cbiAgICAgIGZyYW1lLnBjKys7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsZGl2KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgYjogZ0xvbmcgPSBvcFN0YWNrLnBvcDIoKSxcbiAgICAgIGE6IGdMb25nID0gb3BTdGFjay5wb3AyKCk7XG4gICAgaWYgKGIuaXNaZXJvKCkpIHtcbiAgICAgIHRocm93RXhjZXB0aW9uKHRocmVhZCwgZnJhbWUsICdMamF2YS9sYW5nL0FyaXRobWV0aWNFeGNlcHRpb247JywgJy8gYnkgemVybycpO1xuICAgIH0gZWxzZSB7XG4gICAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChhLmRpdihiKSk7XG4gICAgICBmcmFtZS5wYysrO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZmRpdih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGE6IG51bWJlciA9IG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoKHdyYXBGbG9hdChvcFN0YWNrLnBvcCgpIC8gYSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGRkaXYodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICB2OiBudW1iZXIgPSBvcFN0YWNrLnBvcDIoKTtcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChvcFN0YWNrLnBvcDIoKSAvIHYpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlyZW0odGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBiOiBudW1iZXIgPSBvcFN0YWNrLnBvcCgpLFxuICAgICAgYTogbnVtYmVyID0gb3BTdGFjay5wb3AoKTtcbiAgICBpZiAoYiA9PT0gMCkge1xuICAgICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvQXJpdGhtZXRpY0V4Y2VwdGlvbjsnLCAnLyBieSB6ZXJvJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9wU3RhY2sucHVzaChhICUgYik7XG4gICAgICBmcmFtZS5wYysrO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbHJlbSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGI6IGdMb25nID0gb3BTdGFjay5wb3AyKCksXG4gICAgICBhOiBnTG9uZyA9IG9wU3RhY2sucG9wMigpO1xuICAgIGlmIChiLmlzWmVybygpKSB7XG4gICAgICB0aHJvd0V4Y2VwdGlvbih0aHJlYWQsIGZyYW1lLCAnTGphdmEvbGFuZy9Bcml0aG1ldGljRXhjZXB0aW9uOycsICcvIGJ5IHplcm8nKTtcbiAgICB9IGVsc2Uge1xuICAgICAgb3BTdGFjay5wdXNoV2l0aE51bGwoYS5tb2R1bG8oYikpO1xuICAgICAgZnJhbWUucGMrKztcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZyZW0odGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBiOiBudW1iZXIgPSBvcFN0YWNrLnBvcCgpO1xuICAgIG9wU3RhY2sucHVzaChvcFN0YWNrLnBvcCgpICUgYik7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZHJlbSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGI6IG51bWJlciA9IG9wU3RhY2sucG9wMigpO1xuICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKG9wU3RhY2sucG9wMigpICUgYik7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaW5lZyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2goLW9wU3RhY2sucG9wKCkgfCAwKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsbmVnKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKG9wU3RhY2sucG9wMigpLm5lZ2F0ZSgpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBmbmVnKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaCgtb3BTdGFjay5wb3AoKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZG5lZyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbCgtb3BTdGFjay5wb3AyKCkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICAvKiBCaXR3aXNlIE9wZXJhdGlvbnMgKi9cblxuICBwdWJsaWMgc3RhdGljIGlzaGwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBzOiBudW1iZXIgPSBvcFN0YWNrLnBvcCgpO1xuICAgIG9wU3RhY2sucHVzaChvcFN0YWNrLnBvcCgpIDw8IHMpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxzaGwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBzOiBudW1iZXIgPSBvcFN0YWNrLnBvcCgpO1xuICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKG9wU3RhY2sucG9wMigpLnNoaWZ0TGVmdChnTG9uZy5mcm9tSW50KHMpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaXNocih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIHM6IG51bWJlciA9IG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoKG9wU3RhY2sucG9wKCkgPj4gcyk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbHNocih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIHM6IG51bWJlciA9IG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoV2l0aE51bGwob3BTdGFjay5wb3AyKCkuc2hpZnRSaWdodChnTG9uZy5mcm9tSW50KHMpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaXVzaHIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBzOiBudW1iZXIgPSBvcFN0YWNrLnBvcCgpO1xuICAgIG9wU3RhY2sucHVzaCgob3BTdGFjay5wb3AoKSA+Pj4gcykgfCAwKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsdXNocih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIHM6IG51bWJlciA9IG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoV2l0aE51bGwob3BTdGFjay5wb3AyKCkuc2hpZnRSaWdodFVuc2lnbmVkKGdMb25nLmZyb21JbnQocykpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpYW5kKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaChvcFN0YWNrLnBvcCgpICYgb3BTdGFjay5wb3AoKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbGFuZCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChvcFN0YWNrLnBvcDIoKS5hbmQob3BTdGFjay5wb3AyKCkpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpb3IodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKG9wU3RhY2sucG9wKCkgfCBvcFN0YWNrLnBvcCgpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsb3IodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoV2l0aE51bGwob3BTdGFjay5wb3AyKCkub3Iob3BTdGFjay5wb3AyKCkpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpeG9yKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaChvcFN0YWNrLnBvcCgpIF4gb3BTdGFjay5wb3AoKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbHhvcih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChvcFN0YWNrLnBvcDIoKS54b3Iob3BTdGFjay5wb3AyKCkpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpaW5jKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBpZHggPSBjb2RlW3BjICsgMV0sXG4gICAgICB2YWwgPSBjb2RlLnJlYWRJbnQ4KHBjICsgMik7XG4gICAgZnJhbWUubG9jYWxzW2lkeF0gPSAoZnJhbWUubG9jYWxzW2lkeF0gKyB2YWwpIHwgMDtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpMmwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoV2l0aE51bGwoZ0xvbmcuZnJvbUludChvcFN0YWNrLnBvcCgpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaTJmKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgLy8gTk9QOyB3ZSByZXByZXNlbnQgaW50cyBhcyBmbG9hdHMgYW55d2F5LlxuICAgIC8vIEB0b2RvIFdoYXQgYWJvdXQgcXVhbnRpdGllcyB1bmV4cHJlc3NhYmxlIGFzIGZsb2F0cz9cbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpMmQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2gobnVsbCk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbDJpKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaChvcFN0YWNrLnBvcDIoKS50b0ludCgpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsMmYodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKG9wU3RhY2sucG9wMigpLnRvTnVtYmVyKCkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGwyZCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2hXaXRoTnVsbChvcFN0YWNrLnBvcDIoKS50b051bWJlcigpKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBmMmkodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKGZsb2F0MmludChvcFN0YWNrLnBvcCgpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZjJsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKGdMb25nLmZyb21OdW1iZXIob3BTdGFjay5wb3AoKSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGYyZCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaChudWxsKTtcbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBkMmkodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2s7XG4gICAgb3BTdGFjay5wdXNoKGZsb2F0MmludChvcFN0YWNrLnBvcDIoKSkpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGQybCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGRfdmFsOiBudW1iZXIgPSBvcFN0YWNrLnBvcDIoKTtcbiAgICBpZiAoZF92YWwgPT09IE51bWJlci5QT1NJVElWRV9JTkZJTklUWSkge1xuICAgICAgb3BTdGFjay5wdXNoV2l0aE51bGwoZ0xvbmcuTUFYX1ZBTFVFKTtcbiAgICB9IGVsc2UgaWYgKGRfdmFsID09PSBOdW1iZXIuTkVHQVRJVkVfSU5GSU5JVFkpIHtcbiAgICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKGdMb25nLk1JTl9WQUxVRSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKGdMb25nLmZyb21OdW1iZXIoZF92YWwpKTtcbiAgICB9XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZDJmKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoKHdyYXBGbG9hdChvcFN0YWNrLnBvcCgpKSk7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaTJiKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaCgob3BTdGFjay5wb3AoKSA8PCAyNCkgPj4gMjQpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGkyYyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaztcbiAgICBvcFN0YWNrLnB1c2gob3BTdGFjay5wb3AoKSAmIDB4RkZGRik7XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaTJzKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrO1xuICAgIG9wU3RhY2sucHVzaCgob3BTdGFjay5wb3AoKSA8PCAxNikgPj4gMTYpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxjbXAodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICB2MjogZ0xvbmcgPSBvcFN0YWNrLnBvcDIoKTtcbiAgICBvcFN0YWNrLnB1c2gob3BTdGFjay5wb3AyKCkuY29tcGFyZSh2MikpO1xuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZjbXBsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgdjIgPSBvcFN0YWNrLnBvcCgpLFxuICAgICAgdjEgPSBvcFN0YWNrLnBvcCgpO1xuICAgIGlmICh2MSA9PT0gdjIpIHtcbiAgICAgIG9wU3RhY2sucHVzaCgwKTtcbiAgICB9IGVsc2UgaWYgKHYxID4gdjIpIHtcbiAgICAgIG9wU3RhY2sucHVzaCgxKTtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gdjEgPCB2MiwgYW5kIGlmIHYxIG9yIHYyIGlzIE5hTi5cbiAgICAgIG9wU3RhY2sucHVzaCgtMSk7XG4gICAgfVxuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZjbXBnKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgdjIgPSBvcFN0YWNrLnBvcCgpLFxuICAgICAgdjEgPSBvcFN0YWNrLnBvcCgpO1xuICAgIGlmICh2MSA9PT0gdjIpIHtcbiAgICAgIG9wU3RhY2sucHVzaCgwKTtcbiAgICB9IGVsc2UgaWYgKHYxIDwgdjIpIHtcbiAgICAgIG9wU3RhY2sucHVzaCgtMSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIHYxID4gdjIsIGFuZCBpZiB2MSBvciB2MiBpcyBOYU4uXG4gICAgICBvcFN0YWNrLnB1c2goMSk7XG4gICAgfVxuICAgIGZyYW1lLnBjKys7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGRjbXBsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgdjIgPSBvcFN0YWNrLnBvcDIoKSxcbiAgICAgIHYxID0gb3BTdGFjay5wb3AyKCk7XG4gICAgaWYgKHYxID09PSB2Mikge1xuICAgICAgb3BTdGFjay5wdXNoKDApO1xuICAgIH0gZWxzZSBpZiAodjEgPiB2Mikge1xuICAgICAgb3BTdGFjay5wdXNoKDEpO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyB2MSA8IHYyLCBhbmQgaWYgdjEgb3IgdjIgaXMgTmFOLlxuICAgICAgb3BTdGFjay5wdXNoKC0xKTtcbiAgICB9XG4gICAgZnJhbWUucGMrKztcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZGNtcGcodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICB2YXIgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICB2MiA9IG9wU3RhY2sucG9wMigpLFxuICAgICAgdjEgPSBvcFN0YWNrLnBvcDIoKTtcbiAgICBpZiAodjEgPT09IHYyKSB7XG4gICAgICBvcFN0YWNrLnB1c2goMCk7XG4gICAgfSBlbHNlIGlmICh2MSA8IHYyKSB7XG4gICAgICBvcFN0YWNrLnB1c2goLTEpO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyB2MSA+IHYyLCBhbmQgaWYgdjEgb3IgdjIgaXMgTmFOLlxuICAgICAgb3BTdGFjay5wdXNoKDEpO1xuICAgIH1cbiAgICBmcmFtZS5wYysrO1xuICB9XG5cbiAgLyogVW5hcnkgYnJhbmNoIG9wY29kZXMgKi9cbiAgcHVibGljIHN0YXRpYyBpZmVxKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIGlmIChmcmFtZS5vcFN0YWNrLnBvcCgpID09PSAwKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZuZSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBpZiAoZnJhbWUub3BTdGFjay5wb3AoKSAhPT0gMCkge1xuICAgICAgY29uc3Qgb2Zmc2V0ID0gY29kZS5yZWFkSW50MTZCRShwYyArIDEpO1xuICAgICAgZnJhbWUucGMgKz0gb2Zmc2V0O1xuICAgICAgaWYgKG9mZnNldCA8IDApIHtcbiAgICAgICAgZnJhbWUubWV0aG9kLmluY3JCQkVudHJpZXMoKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgZnJhbWUucGMgKz0gMztcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlmbHQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgaWYgKGZyYW1lLm9wU3RhY2sucG9wKCkgPCAwKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZnZSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBpZiAoZnJhbWUub3BTdGFjay5wb3AoKSA+PSAwKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZndCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBpZiAoZnJhbWUub3BTdGFjay5wb3AoKSA+IDApIHtcbiAgICAgIGNvbnN0IG9mZnNldCA9IGNvZGUucmVhZEludDE2QkUocGMgKyAxKTtcbiAgICAgIGZyYW1lLnBjICs9IG9mZnNldDtcbiAgICAgIGlmIChvZmZzZXQgPCAwKSB7XG4gICAgICAgIGZyYW1lLm1ldGhvZC5pbmNyQkJFbnRyaWVzKCk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGZyYW1lLnBjICs9IDM7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpZmxlKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIGlmIChmcmFtZS5vcFN0YWNrLnBvcCgpIDw9IDApIHtcbiAgICAgIGNvbnN0IG9mZnNldCA9IGNvZGUucmVhZEludDE2QkUocGMgKyAxKTtcbiAgICAgIGZyYW1lLnBjICs9IG9mZnNldDtcbiAgICAgIGlmIChvZmZzZXQgPCAwKSB7XG4gICAgICAgIGZyYW1lLm1ldGhvZC5pbmNyQkJFbnRyaWVzKCk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGZyYW1lLnBjICs9IDM7XG4gICAgfVxuICB9XG5cbiAgLyogQmluYXJ5IGJyYW5jaCBvcGNvZGVzICovXG4gIHB1YmxpYyBzdGF0aWMgaWZfaWNtcGVxKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciB2MiA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgdmFyIHYxID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAodjEgPT09IHYyKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZfaWNtcG5lKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciB2MiA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgdmFyIHYxID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAodjEgIT09IHYyKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZfaWNtcGx0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciB2MiA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgdmFyIHYxID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAodjEgPCB2Mikge1xuICAgICAgY29uc3Qgb2Zmc2V0ID0gY29kZS5yZWFkSW50MTZCRShwYyArIDEpO1xuICAgICAgZnJhbWUucGMgKz0gb2Zmc2V0O1xuICAgICAgaWYgKG9mZnNldCA8IDApIHtcbiAgICAgICAgZnJhbWUubWV0aG9kLmluY3JCQkVudHJpZXMoKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgZnJhbWUucGMgKz0gMztcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlmX2ljbXBnZSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgdjIgPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIHZhciB2MSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgaWYgKHYxID49IHYyKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZfaWNtcGd0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciB2MiA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgdmFyIHYxID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAodjEgPiB2Mikge1xuICAgICAgY29uc3Qgb2Zmc2V0ID0gY29kZS5yZWFkSW50MTZCRShwYyArIDEpO1xuICAgICAgZnJhbWUucGMgKz0gb2Zmc2V0O1xuICAgICAgaWYgKG9mZnNldCA8IDApIHtcbiAgICAgICAgZnJhbWUubWV0aG9kLmluY3JCQkVudHJpZXMoKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgZnJhbWUucGMgKz0gMztcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlmX2ljbXBsZSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgdjIgPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuICAgIHZhciB2MSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgaWYgKHYxIDw9IHYyKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZfYWNtcGVxKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciB2MiA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgdmFyIHYxID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAodjEgPT09IHYyKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaWZfYWNtcG5lKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciB2MiA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgdmFyIHYxID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAodjEgIT09IHYyKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIC8qIEp1bXAgb3Bjb2RlcyAqL1xuICBwdWJsaWMgc3RhdGljIGdvdG8odGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgY29uc3Qgb2Zmc2V0ID0gY29kZS5yZWFkSW50MTZCRShwYyArIDEpO1xuICAgIGZyYW1lLnBjICs9IG9mZnNldDtcbiAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgZnJhbWUubWV0aG9kLmluY3JCQkVudHJpZXMoKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGpzcih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2gocGMgKyAzKTtcbiAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgZnJhbWUucGMgKz0gb2Zmc2V0O1xuICAgIGlmIChvZmZzZXQgPCAwKSB7XG4gICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgcmV0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIGZyYW1lLnBjID0gZnJhbWUubG9jYWxzW2NvZGVbcGMgKyAxXV07XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIHRhYmxlc3dpdGNoKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBsZXQgcGMgPSBmcmFtZS5wYztcbiAgICAvLyBJZ25vcmUgcGFkZGluZyBieXRlcy4gVGhlICsxIGlzIHRvIHNraXAgdGhlIG9wY29kZSBieXRlLlxuICAgIHBjICs9ICgoNCAtIChwYyArIDEpICUgNCkgJSA0KSArIDE7XG4gICAgdmFyIGRlZmF1bHRPZmZzZXQgPSBjb2RlLnJlYWRJbnQzMkJFKHBjKSxcbiAgICAgIGxvdyA9IGNvZGUucmVhZEludDMyQkUocGMgKyA0KSxcbiAgICAgIGhpZ2ggPSBjb2RlLnJlYWRJbnQzMkJFKHBjICsgOCksXG4gICAgICBvZmZzZXQgPSBmcmFtZS5vcFN0YWNrLnBvcCgpO1xuXG4gICAgaWYgKG9mZnNldCA+PSBsb3cgJiYgb2Zmc2V0IDw9IGhpZ2gpIHtcbiAgICAgIGZyYW1lLnBjICs9IGNvZGUucmVhZEludDMyQkUocGMgKyAxMiArICgob2Zmc2V0IC0gbG93KSAqIDQpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgZnJhbWUucGMgKz0gZGVmYXVsdE9mZnNldDtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxvb2t1cHN3aXRjaCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgbGV0IHBjID0gZnJhbWUucGM7XG4gICAgLy8gU2tpcCBwYWRkaW5nIGJ5dGVzLiBUaGUgKzEgaXMgdG8gc2tpcCB0aGUgb3Bjb2RlIGJ5dGUuXG4gICAgcGMgKz0gKCg0IC0gKHBjICsgMSkgJSA0KSAlIDQpICsgMTtcbiAgICB2YXIgZGVmYXVsdE9mZnNldCA9IGNvZGUucmVhZEludDMyQkUocGMpLFxuICAgICAgblBhaXJzID0gY29kZS5yZWFkSW50MzJCRShwYyArIDQpLFxuICAgICAgaTogbnVtYmVyLFxuICAgICAgdjogbnVtYmVyID0gZnJhbWUub3BTdGFjay5wb3AoKTtcblxuICAgIHBjICs9IDg7XG4gICAgZm9yIChpID0gMDsgaSA8IG5QYWlyczsgaSsrKSB7XG4gICAgICBpZiAoY29kZS5yZWFkSW50MzJCRShwYykgPT09IHYpIHtcbiAgICAgICAgY29uc3Qgb2Zmc2V0ID0gY29kZS5yZWFkSW50MzJCRShwYyArIDQpO1xuICAgICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICAgIGlmIChvZmZzZXQgPCAwKSB7XG4gICAgICAgICAgZnJhbWUubWV0aG9kLmluY3JCQkVudHJpZXMoKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgICBwYyArPSA4O1xuICAgIH1cbiAgICAvLyBObyBtYXRjaCBmb3VuZC5cbiAgICBmcmFtZS5wYyArPSBkZWZhdWx0T2Zmc2V0O1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyByZXR1cm4odGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUpIHtcbiAgICBmcmFtZS5yZXR1cm5Ub1RocmVhZExvb3AgPSB0cnVlO1xuICAgIGlmIChmcmFtZS5tZXRob2QuYWNjZXNzRmxhZ3MuaXNTeW5jaHJvbml6ZWQoKSkge1xuICAgICAgLy8gbW9uaXRvcmV4aXRcbiAgICAgIGlmICghZnJhbWUubWV0aG9kLm1ldGhvZExvY2sodGhyZWFkLCBmcmFtZSkuZXhpdCh0aHJlYWQpKSB7XG4gICAgICAgIC8vIG1vbml0b3JleGl0IHRocmV3IGFuIGV4Y2VwdGlvbi5cbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuICAgIH1cbiAgICB0aHJlYWQuYXN5bmNSZXR1cm4oKTtcbiAgfVxuXG4gIC8qIDMyLWJpdCByZXR1cm4gYnl0ZWNvZGVzICovXG5cbiAgcHJpdmF0ZSBzdGF0aWMgX3JldHVybl8zMih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gICAgaWYgKGZyYW1lLm1ldGhvZC5hY2Nlc3NGbGFncy5pc1N5bmNocm9uaXplZCgpKSB7XG4gICAgICAvLyBtb25pdG9yZXhpdFxuICAgICAgaWYgKCFmcmFtZS5tZXRob2QubWV0aG9kTG9jayh0aHJlYWQsIGZyYW1lKS5leGl0KHRocmVhZCkpIHtcbiAgICAgICAgLy8gbW9uaXRvcmV4aXQgdGhyZXcgYW4gZXhjZXB0aW9uLlxuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgfVxuICAgIHRocmVhZC5hc3luY1JldHVybihmcmFtZS5vcFN0YWNrLmJvdHRvbSgpKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaXJldHVybiA9IE9wY29kZXMuX3JldHVybl8zMjtcbiAgcHVibGljIHN0YXRpYyBmcmV0dXJuID0gT3Bjb2Rlcy5fcmV0dXJuXzMyO1xuICBwdWJsaWMgc3RhdGljIGFyZXR1cm4gPSBPcGNvZGVzLl9yZXR1cm5fMzI7XG5cbiAgLyogNjQtYml0IHJldHVybiBvcGNvZGVzICovXG5cbiAgcHJpdmF0ZSBzdGF0aWMgX3JldHVybl82NCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gICAgaWYgKGZyYW1lLm1ldGhvZC5hY2Nlc3NGbGFncy5pc1N5bmNocm9uaXplZCgpKSB7XG4gICAgICAvLyBtb25pdG9yZXhpdFxuICAgICAgaWYgKCFmcmFtZS5tZXRob2QubWV0aG9kTG9jayh0aHJlYWQsIGZyYW1lKS5leGl0KHRocmVhZCkpIHtcbiAgICAgICAgLy8gbW9uaXRvcmV4aXQgdGhyZXcgYW4gZXhjZXB0aW9uLlxuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgfVxuICAgIHRocmVhZC5hc3luY1JldHVybihmcmFtZS5vcFN0YWNrLmJvdHRvbSgpLCBudWxsKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbHJldHVybiA9IE9wY29kZXMuX3JldHVybl82NDtcbiAgcHVibGljIHN0YXRpYyBkcmV0dXJuID0gT3Bjb2Rlcy5fcmV0dXJuXzY0O1xuXG4gIHB1YmxpYyBzdGF0aWMgZ2V0c3RhdGljKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBmaWVsZEluZm8gPSA8RmllbGRSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKTtcbiAgICBhc3NlcnQoZmllbGRJbmZvLmdldFR5cGUoKSA9PT0gQ29uc3RhbnRQb29sSXRlbVR5cGUuRklFTERSRUYpO1xuICAgIGlmIChmaWVsZEluZm8uaXNSZXNvbHZlZCgpKSB7XG4gICAgICAvLyBHZXQgdGhlICphY3R1YWwqIGNsYXNzIHRoYXQgb3ducyB0aGlzIGZpZWxkLlxuICAgICAgLy8gVGhpcyBtYXkgbm90IGJlIGluaXRpYWxpemVkIGlmIGl0J3MgYW4gaW50ZXJmYWNlLCBzbyB3ZSBuZWVkIHRvIGNoZWNrLlxuICAgICAgdmFyIGZpZWxkT3duZXJDbHMgPSBmaWVsZEluZm8uZmllbGQuY2xzO1xuICAgICAgaWYgKGZpZWxkT3duZXJDbHMuaXNJbml0aWFsaXplZCh0aHJlYWQpKSB7XG4gICAgICAgIC8vIE9wY29kZSBpcyByZWFkeSB0byBleGVjdXRlISBSZXdyaXRlIHRvIGEgJ2Zhc3QnIHZlcnNpb24sXG4gICAgICAgIC8vIGFuZCBydW4gdGhlIGZhc3QgdmVyc2lvbi5cbiAgICAgICAgaWYgKGZpZWxkSW5mby5uYW1lQW5kVHlwZUluZm8uZGVzY3JpcHRvciA9PT0gJ0onIHx8IGZpZWxkSW5mby5uYW1lQW5kVHlwZUluZm8uZGVzY3JpcHRvciA9PT0gJ0QnKSB7XG4gICAgICAgICAgY29kZVtwY10gPSBPcENvZGUuR0VUU1RBVElDX0ZBU1Q2NDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb2RlW3BjXSA9IE9wQ29kZS5HRVRTVEFUSUNfRkFTVDMyO1xuICAgICAgICB9XG4gICAgICAgIC8vIFN0YXNoIHRoZSByZXN1bHQgb2YgZmllbGQgbG9va3VwLlxuICAgICAgICBmaWVsZEluZm8uZmllbGRPd25lckNvbnN0cnVjdG9yID0gZmllbGRPd25lckNscy5nZXRDb25zdHJ1Y3Rvcih0aHJlYWQpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gSW5pdGlhbGl6ZSBjbGFzcyBhbmQgcmVydW4gb3Bjb2RlXG4gICAgICAgIGluaXRpYWxpemVDbGFzc0Zyb21DbGFzcyh0aHJlYWQsIGZyYW1lLCBmaWVsZE93bmVyQ2xzKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgLy8gUmVzb2x2ZSB0aGUgZmllbGQuXG4gICAgICByZXNvbHZlQ1BJdGVtKHRocmVhZCwgZnJhbWUsIGZpZWxkSW5mbyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEEgZmFzdCB2ZXJzaW9uIG9mIGdldHN0YXRpYyB0aGF0IGFzc3VtZXMgdGhhdCByZWxldmFudCBjbGFzc2VzIGFyZVxuICAgKiBpbml0aWFsaXplZC5cbiAgICpcbiAgICogUmV0cmlldmVzIGEgMzItYml0IHZhbHVlLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBnZXRzdGF0aWNfZmFzdDMyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBmaWVsZEluZm8gPSA8RmllbGRSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKTtcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2goZmllbGRJbmZvLmZpZWxkT3duZXJDb25zdHJ1Y3RvcltmaWVsZEluZm8uZnVsbEZpZWxkTmFtZV0pO1xuICAgIGZyYW1lLnBjICs9IDM7XG4gIH1cblxuICAvKipcbiAgICogQSBmYXN0IHZlcnNpb24gb2YgZ2V0c3RhdGljIHRoYXQgYXNzdW1lcyB0aGF0IHJlbGV2YW50IGNsYXNzZXMgYXJlXG4gICAqIGluaXRpYWxpemVkLlxuICAgKlxuICAgKiBSZXRyaWV2ZXMgYSA2NC1iaXQgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGdldHN0YXRpY19mYXN0NjQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGZpZWxkSW5mbyA9IDxGaWVsZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGZyYW1lLm9wU3RhY2sucHVzaFdpdGhOdWxsKGZpZWxkSW5mby5maWVsZE93bmVyQ29uc3RydWN0b3JbZmllbGRJbmZvLmZ1bGxGaWVsZE5hbWVdKTtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBwdXRzdGF0aWModGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGZpZWxkSW5mbyA9IDxGaWVsZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGFzc2VydChmaWVsZEluZm8uZ2V0VHlwZSgpID09PSBDb25zdGFudFBvb2xJdGVtVHlwZS5GSUVMRFJFRik7XG5cbiAgICBpZiAoZmllbGRJbmZvLmlzUmVzb2x2ZWQoKSkge1xuICAgICAgLy8gR2V0IHRoZSAqYWN0dWFsKiBjbGFzcyB0aGF0IG93bnMgdGhpcyBmaWVsZC5cbiAgICAgIC8vIFRoaXMgbWF5IG5vdCBiZSBpbml0aWFsaXplZCBpZiBpdCdzIGFuIGludGVyZmFjZSwgc28gd2UgbmVlZCB0byBjaGVjay5cbiAgICAgIHZhciBmaWVsZE93bmVyQ2xzID0gZmllbGRJbmZvLmZpZWxkLmNscztcbiAgICAgIGlmIChmaWVsZE93bmVyQ2xzLmlzSW5pdGlhbGl6ZWQodGhyZWFkKSkge1xuICAgICAgICAvLyBPcGNvZGUgaXMgcmVhZHkgdG8gZXhlY3V0ZSEgUmV3cml0ZSB0byBhICdmYXN0JyB2ZXJzaW9uLFxuICAgICAgICAvLyBhbmQgcnVuIHRoZSBmYXN0IHZlcnNpb24uXG4gICAgICAgIGlmIChmaWVsZEluZm8ubmFtZUFuZFR5cGVJbmZvLmRlc2NyaXB0b3IgPT09ICdKJyB8fCBmaWVsZEluZm8ubmFtZUFuZFR5cGVJbmZvLmRlc2NyaXB0b3IgPT09ICdEJykge1xuICAgICAgICAgIGNvZGVbcGNdID0gT3BDb2RlLlBVVFNUQVRJQ19GQVNUNjQ7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29kZVtwY10gPSBPcENvZGUuUFVUU1RBVElDX0ZBU1QzMjtcbiAgICAgICAgfVxuICAgICAgICAvLyBTdGFzaCB0aGUgcmVzdWx0IG9mIGZpZWxkIGxvb2t1cC5cbiAgICAgICAgZmllbGRJbmZvLmZpZWxkT3duZXJDb25zdHJ1Y3RvciA9IGZpZWxkT3duZXJDbHMuZ2V0Q29uc3RydWN0b3IodGhyZWFkKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIEluaXRpYWxpemUgY2xhc3MgYW5kIHJlcnVuIG9wY29kZVxuICAgICAgICBpbml0aWFsaXplQ2xhc3NGcm9tQ2xhc3ModGhyZWFkLCBmcmFtZSwgZmllbGRPd25lckNscyk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIFJlc29sdmUgdGhlIGZpZWxkLlxuICAgICAgcmVzb2x2ZUNQSXRlbSh0aHJlYWQsIGZyYW1lLCBmaWVsZEluZm8pO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBBIGZhc3QgdmVyc2lvbiBvZiBwdXRzdGF0aWMgdGhhdCBhc3N1bWVzIHRoYXQgcmVsZXZhbnQgY2xhc3NlcyBhcmVcbiAgICogaW5pdGlhbGl6ZWQuXG4gICAqXG4gICAqIFB1dHMgYSAzMi1iaXQgdmFsdWUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHB1dHN0YXRpY19mYXN0MzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGZpZWxkSW5mbyA9IDxGaWVsZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGZpZWxkSW5mby5maWVsZE93bmVyQ29uc3RydWN0b3JbZmllbGRJbmZvLmZ1bGxGaWVsZE5hbWVdID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgLyoqXG4gICAqIEEgZmFzdCB2ZXJzaW9uIG9mIHB1dHN0YXRpYyB0aGF0IGFzc3VtZXMgdGhhdCByZWxldmFudCBjbGFzc2VzIGFyZVxuICAgKiBpbml0aWFsaXplZC5cbiAgICpcbiAgICogUHV0cyBhIDY0LWJpdCB2YWx1ZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcHV0c3RhdGljX2Zhc3Q2NCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgZmllbGRJbmZvID0gPEZpZWxkUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG4gICAgZmllbGRJbmZvLmZpZWxkT3duZXJDb25zdHJ1Y3RvcltmaWVsZEluZm8uZnVsbEZpZWxkTmFtZV0gPSBmcmFtZS5vcFN0YWNrLnBvcDIoKTtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBnZXRmaWVsZCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgZmllbGRJbmZvID0gPEZpZWxkUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBsb2FkZXIgPSBmcmFtZS5nZXRMb2FkZXIoKSxcbiAgICAgIG9iaiA9IGZyYW1lLm9wU3RhY2sudG9wKCk7XG4gICAgYXNzZXJ0KGZpZWxkSW5mby5nZXRUeXBlKCkgPT09IENvbnN0YW50UG9vbEl0ZW1UeXBlLkZJRUxEUkVGKTtcbiAgICAvLyBDaGVjayBpZiB0aGUgb2JqZWN0IGlzIG51bGw7IGlmIHdlIGRvIG5vdCBkbyB0aGlzIGJlZm9yZSBnZXRfY2xhc3MsIHRoZW5cbiAgICAvLyB3ZSBtaWdodCB0cnkgdG8gZ2V0IGEgY2xhc3MgdGhhdCB3ZSBoYXZlIG5vdCBpbml0aWFsaXplZCFcbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICAvLyBjbHMgaXMgZ3VhcmFudGVlZCB0byBiZSBpbiB0aGUgaW5oZXJpdGFuY2UgaGllcmFyY2h5IG9mIG9iaiwgc28gaXQgbXVzdCBiZVxuICAgICAgLy8gaW5pdGlhbGl6ZWQuIEhvd2V2ZXIsIGl0IG1heSBub3QgYmUgbG9hZGVkIGluIHRoZSBjdXJyZW50IGNsYXNzJ3NcbiAgICAgIC8vIENsYXNzTG9hZGVyLi4uXG4gICAgICBpZiAoZmllbGRJbmZvLmlzUmVzb2x2ZWQoKSkge1xuICAgICAgICB2YXIgZmllbGQgPSBmaWVsZEluZm8uZmllbGQ7XG4gICAgICAgIGlmIChmaWVsZC5yYXdEZXNjcmlwdG9yID09ICdKJyB8fCBmaWVsZC5yYXdEZXNjcmlwdG9yID09ICdEJykge1xuICAgICAgICAgIGNvZGVbcGNdID0gT3BDb2RlLkdFVEZJRUxEX0ZBU1Q2NDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb2RlW3BjXSA9IE9wQ29kZS5HRVRGSUVMRF9GQVNUMzI7XG4gICAgICAgIH1cbiAgICAgICAgLy8gUmVydW4gb3Bjb2RlXG4gICAgICB9IGVsc2Uge1xuICAgICAgICByZXNvbHZlQ1BJdGVtKHRocmVhZCwgZnJhbWUsIGZpZWxkSW5mbyk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBnZXRmaWVsZF9mYXN0MzIodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGZpZWxkSW5mbyA9IDxGaWVsZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssIG9iajogSlZNVHlwZXMuamF2YV9sYW5nX09iamVjdCA9IG9wU3RhY2sucG9wKCk7XG4gICAgaWYgKCFpc051bGwodGhyZWFkLCBmcmFtZSwgb2JqKSkge1xuICAgICAgb3BTdGFjay5wdXNoKCg8YW55PiBvYmopW2ZpZWxkSW5mby5mdWxsRmllbGROYW1lXSk7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZ2V0ZmllbGRfZmFzdDY0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBmaWVsZEluZm8gPSA8RmllbGRSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKSxcbiAgICAgIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLCBvYmo6IEpWTVR5cGVzLmphdmFfbGFuZ19PYmplY3QgPSBvcFN0YWNrLnBvcCgpO1xuICAgIGlmICghaXNOdWxsKHRocmVhZCwgZnJhbWUsIG9iaikpIHtcbiAgICAgIG9wU3RhY2sucHVzaFdpdGhOdWxsKCg8YW55PiBvYmopW2ZpZWxkSW5mby5mdWxsRmllbGROYW1lXSk7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgcHV0ZmllbGQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGZpZWxkSW5mbyA9IDxGaWVsZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpLFxuICAgICAgbG9hZGVyID0gZnJhbWUuZ2V0TG9hZGVyKCksXG4gICAgICBpc0xvbmcgPSBmaWVsZEluZm8ubmFtZUFuZFR5cGVJbmZvLmRlc2NyaXB0b3IgPT0gJ0onIHx8IGZpZWxkSW5mby5uYW1lQW5kVHlwZUluZm8uZGVzY3JpcHRvciA9PSAnRCcsXG4gICAgICBvYmogPSBmcmFtZS5vcFN0YWNrLmZyb21Ub3AoaXNMb25nID8gMiA6IDEpO1xuICAgIGFzc2VydChmaWVsZEluZm8uZ2V0VHlwZSgpID09PSBDb25zdGFudFBvb2xJdGVtVHlwZS5GSUVMRFJFRik7XG5cbiAgICAvLyBDaGVjayBpZiB0aGUgb2JqZWN0IGlzIG51bGw7IGlmIHdlIGRvIG5vdCBkbyB0aGlzIGJlZm9yZSBnZXRfY2xhc3MsIHRoZW5cbiAgICAvLyB3ZSBtaWdodCB0cnkgdG8gZ2V0IGEgY2xhc3MgdGhhdCB3ZSBoYXZlIG5vdCBpbml0aWFsaXplZCFcbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICAvLyBjbHMgaXMgZ3VhcmFudGVlZCB0byBiZSBpbiB0aGUgaW5oZXJpdGFuY2UgaGllcmFyY2h5IG9mIG9iaiwgc28gaXQgbXVzdCBiZVxuICAgICAgLy8gaW5pdGlhbGl6ZWQuIEhvd2V2ZXIsIGl0IG1heSBub3QgYmUgbG9hZGVkIGluIHRoZSBjdXJyZW50IGNsYXNzJ3NcbiAgICAgIC8vIENsYXNzTG9hZGVyLi4uXG4gICAgICBpZiAoZmllbGRJbmZvLmlzUmVzb2x2ZWQoKSkge1xuICAgICAgICB2YXIgZmllbGQgPSBmaWVsZEluZm8uZmllbGQ7XG4gICAgICAgIGlmIChpc0xvbmcpIHtcbiAgICAgICAgICBjb2RlW3BjXSA9IE9wQ29kZS5QVVRGSUVMRF9GQVNUNjQ7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgY29kZVtwY10gPSBPcENvZGUuUFVURklFTERfRkFTVDMyO1xuICAgICAgICB9XG4gICAgICAgIC8vIFN0YXNoIHRoZSByZXNvbHZlZCBmdWxsIGZpZWxkIG5hbWUuXG4gICAgICAgIGZpZWxkSW5mby5mdWxsRmllbGROYW1lID0gYCR7ZGVzY3JpcHRvcjJ0eXBlc3RyKGZpZWxkLmNscy5nZXRJbnRlcm5hbE5hbWUoKSl9LyR7ZmllbGRJbmZvLm5hbWVBbmRUeXBlSW5mby5uYW1lfWA7XG4gICAgICAgIC8vIFJlcnVuIG9wY29kZVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmVzb2x2ZUNQSXRlbSh0aHJlYWQsIGZyYW1lLCBmaWVsZEluZm8pO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgcHV0ZmllbGRfZmFzdDMyKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIHZhbCA9IG9wU3RhY2sucG9wKCksXG4gICAgICBvYmo6IEpWTVR5cGVzLmphdmFfbGFuZ19PYmplY3QgPSBvcFN0YWNrLnBvcCgpLFxuICAgICAgZmllbGRJbmZvID0gPEZpZWxkUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG5cbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICAoPGFueT4gb2JqKVtmaWVsZEluZm8uZnVsbEZpZWxkTmFtZV0gPSB2YWw7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgICAvLyBOUEUgaGFzIGJlZW4gdGhyb3duLlxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBwdXRmaWVsZF9mYXN0NjQodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgdmFsID0gb3BTdGFjay5wb3AyKCksXG4gICAgICBvYmo6IEpWTVR5cGVzLmphdmFfbGFuZ19PYmplY3QgPSBvcFN0YWNrLnBvcCgpLFxuICAgICAgZmllbGRJbmZvID0gPEZpZWxkUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG5cbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICAoPGFueT4gb2JqKVtmaWVsZEluZm8uZnVsbEZpZWxkTmFtZV0gPSB2YWw7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgICAvLyBOUEUgaGFzIGJlZW4gdGhyb3duLlxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpbnZva2V2aXJ0dWFsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBtZXRob2RSZWZlcmVuY2UgPSA8TWV0aG9kUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG5cbiAgICAvLyBFbnN1cmUgcmVmZXJlbmNlZCBjbGFzcyBpcyBsb2FkZWQgaW4gdGhlIGN1cnJlbnQgY2xhc3Nsb2FkZXIuXG4gICAgLy8gRXZlbiB0aG91Z2ggd2UgZG9uJ3QgdXNlIHRoaXMgY2xhc3MgZm9yIGFueXRoaW5nLCBhbmQgd2Uga25vdyB0aGF0IGl0XG4gICAgLy8gbXVzdCBiZSBsb2FkZWQgYmVjYXVzZSBpdCBpcyBpbiB0aGUgb2JqZWN0J3MgaW5oZXJpdGFuY2UgaGllcmFyY2h5LFxuICAgIC8vIGl0IG5lZWRzIHRvIGJlIHByZXNlbnQgaW4gdGhlIGN1cnJlbnQgY2xhc3Nsb2FkZXIuXG4gICAgaWYgKG1ldGhvZFJlZmVyZW5jZS5pc1Jlc29sdmVkKCkpIHtcbiAgICAgIHZhciBtID0gbWV0aG9kUmVmZXJlbmNlLm1ldGhvZDtcbiAgICAgIGlmIChtLmlzU2lnbmF0dXJlUG9seW1vcnBoaWMoKSkge1xuICAgICAgICBzd2l0Y2ggKG0ubmFtZSkge1xuICAgICAgICAgIGNhc2UgJ2ludm9rZUJhc2ljJzpcbiAgICAgICAgICAgIGNvZGVbcGNdID0gT3BDb2RlLklOVk9LRUJBU0lDO1xuICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgY2FzZSAnaW52b2tlJzpcbiAgICAgICAgICBjYXNlICdpbnZva2VFeGFjdCc6XG4gICAgICAgICAgICBjb2RlW3BjXSA9IE9wQ29kZS5JTlZPS0VIQU5ETEU7XG4gICAgICAgICAgICBicmVhaztcbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvQWJzdHJhY3RNZXRob2RFcnJvcjsnLCBgSW52YWxpZCBzaWduYXR1cmUgcG9seW1vcnBoaWMgbWV0aG9kOiAke20uY2xzLmdldEV4dGVybmFsTmFtZSgpfS4ke20ubmFtZX1gKTtcbiAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBjb2RlW3BjXSA9IE9wQ29kZS5JTlZPS0VWSVJUVUFMX0ZBU1Q7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIHJlc29sdmVDUEl0ZW0odGhyZWFkLCBmcmFtZSwgbWV0aG9kUmVmZXJlbmNlKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGludm9rZWludGVyZmFjZSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgbWV0aG9kUmVmZXJlbmNlID0gPEludGVyZmFjZU1ldGhvZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGlmIChtZXRob2RSZWZlcmVuY2UuaXNSZXNvbHZlZCgpKSB7XG4gICAgICBpZiAobWV0aG9kUmVmZXJlbmNlLm1ldGhvZC5jbHMuaXNJbml0aWFsaXplZCh0aHJlYWQpKSB7XG4gICAgICAgIC8vIFJld3JpdGUgdG8gZmFzdCBhbmQgcmVydW4uXG4gICAgICAgIGNvZGVbcGNdID0gT3BDb2RlLklOVk9LRUlOVEVSRkFDRV9GQVNUO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gSW5pdGlhbGl6ZSBvdXIgY2xhc3MgYW5kIHJlcnVuIG9wY29kZS5cbiAgICAgICAgLy8gTm90ZSB0aGF0IHRoZSBleGlzdGFuY2Ugb2YgYW4gb2JqZWN0IG9mIGFuIGludGVyZmFjZSB0eXBlIGRvZXMgKm5vdCpcbiAgICAgICAgLy8gbWVhbiB0aGF0IHRoZSBpbnRlcmZhY2UgaXMgaW5pdGlhbGl6ZWQhXG4gICAgICAgIGluaXRpYWxpemVDbGFzcyh0aHJlYWQsIGZyYW1lLCBtZXRob2RSZWZlcmVuY2UuY2xhc3NJbmZvKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgcmVzb2x2ZUNQSXRlbSh0aHJlYWQsIGZyYW1lLCBtZXRob2RSZWZlcmVuY2UpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaW52b2tlZHluYW1pYyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgY2FsbFNpdGVTcGVjaWZpZXIgPSA8SW52b2tlRHluYW1pYz4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIHRocmVhZC5zZXRTdGF0dXMoVGhyZWFkU3RhdHVzLkFTWU5DX1dBSVRJTkcpO1xuICAgIGNhbGxTaXRlU3BlY2lmaWVyLmNvbnN0cnVjdENhbGxTaXRlT2JqZWN0KHRocmVhZCwgZnJhbWUuZ2V0TG9hZGVyKCksIGZyYW1lLm1ldGhvZC5jbHMsIHBjLCAoc3RhdHVzOiBib29sZWFuKSA9PiB7XG4gICAgICBpZiAoc3RhdHVzKSB7XG4gICAgICAgIGFzc2VydCh0eXBlb2YoY2FsbFNpdGVTcGVjaWZpZXIuZ2V0Q2FsbFNpdGVPYmplY3QocGMpWzBdLnZtdGFyZ2V0KSA9PT0gJ2Z1bmN0aW9uJywgXCJNZXRob2ROYW1lIHNob3VsZCBiZSByZXNvbHZlZC4uLlwiKTtcbiAgICAgICAgY29kZVtwY10gPSBPcENvZGUuSU5WT0tFRFlOQU1JQ19GQVNUO1xuICAgICAgICAvLyBSZXN1bWUgYW5kIHJlcnVuIGZhc3Qgb3Bjb2RlLlxuICAgICAgICB0aHJlYWQuc2V0U3RhdHVzKFRocmVhZFN0YXR1cy5SVU5OQUJMRSk7XG4gICAgICB9XG4gICAgfSk7XG4gICAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBYWFg6IEFjdHVhbGx5IHBlcmZvcm0gc3VwZXJjbGFzcyBtZXRob2QgbG9va3VwLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpbnZva2VzcGVjaWFsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBtZXRob2RSZWZlcmVuY2UgPSA8TWV0aG9kUmVmZXJlbmNlIHwgSW50ZXJmYWNlTWV0aG9kUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG4gICAgaWYgKG1ldGhvZFJlZmVyZW5jZS5pc1Jlc29sdmVkKCkpIHtcbiAgICAgIC8vIFJld3JpdGUgYW5kIHJlcnVuLlxuICAgICAgY29kZVtwY10gPSBPcENvZGUuSU5WT0tFTk9OVklSVFVBTF9GQVNUO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXNvbHZlQ1BJdGVtKHRocmVhZCwgZnJhbWUsIG1ldGhvZFJlZmVyZW5jZSk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpbnZva2VzdGF0aWModGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIG1ldGhvZFJlZmVyZW5jZSA9IDxNZXRob2RSZWZlcmVuY2UgfCBJbnRlcmZhY2VNZXRob2RSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKTtcbiAgICBpZiAobWV0aG9kUmVmZXJlbmNlLmlzUmVzb2x2ZWQoKSkge1xuICAgICAgdmFyIG0gPSBtZXRob2RSZWZlcmVuY2UubWV0aG9kO1xuICAgICAgaWYgKG0uY2xzLmlzSW5pdGlhbGl6ZWQodGhyZWFkKSkge1xuICAgICAgICB2YXIgbmV3T3Bjb2RlOiBPcENvZGUgPSBPcENvZGUuSU5WT0tFU1RBVElDX0ZBU1Q7XG4gICAgICAgIGlmIChtZXRob2RSZWZlcmVuY2UubWV0aG9kLmlzU2lnbmF0dXJlUG9seW1vcnBoaWMoKSkge1xuICAgICAgICAgIHN3aXRjaCAobWV0aG9kUmVmZXJlbmNlLm1ldGhvZC5uYW1lKSB7XG4gICAgICAgICAgICBjYXNlICdsaW5rVG9JbnRlcmZhY2UnOlxuICAgICAgICAgICAgY2FzZSAnbGlua1RvVmlydHVhbCc6XG4gICAgICAgICAgICAgIG5ld09wY29kZSA9IE9wQ29kZS5MSU5LVE9WSVJUVUFMO1xuICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgIGNhc2UgJ2xpbmtUb1N0YXRpYyc6XG4gICAgICAgICAgICBjYXNlICdsaW5rVG9TcGVjaWFsJzpcbiAgICAgICAgICAgICAgbmV3T3Bjb2RlID0gT3BDb2RlLkxJTktUT1NQRUNJQUw7XG4gICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgICAgYXNzZXJ0KGZhbHNlLCBcIlNob3VsZCBiZSBpbXBvc3NpYmxlLlwiKTtcbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIC8vIFJld3JpdGUgYW5kIHJlcnVuLlxuICAgICAgICBjb2RlW3BjXSA9IG5ld09wY29kZTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGluaXRpYWxpemVDbGFzc0Zyb21DbGFzcyh0aHJlYWQsIGZyYW1lLCBtLmNscyk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIHJlc29sdmVDUEl0ZW0odGhyZWFkLCBmcmFtZSwgbWV0aG9kUmVmZXJlbmNlKTtcbiAgICB9XG4gIH1cblxuICAvLy8gRmFzdCBpbnZva2Ugb3Bjb2Rlcy5cblxuICBwdWJsaWMgc3RhdGljIGludm9rZW5vbnZpcnR1YWxfZmFzdCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgbWV0aG9kUmVmZXJlbmNlID0gPE1ldGhvZFJlZmVyZW5jZSB8IEludGVyZmFjZU1ldGhvZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssIHBhcmFtU2l6ZSA9IG1ldGhvZFJlZmVyZW5jZS5wYXJhbVdvcmRTaXplLFxuICAgICAgb2JqOiBKVk1UeXBlcy5qYXZhX2xhbmdfT2JqZWN0ID0gb3BTdGFjay5mcm9tVG9wKHBhcmFtU2l6ZSk7XG5cbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICB2YXIgYXJncyA9IG9wU3RhY2suc2xpY2VGcm9tVG9wKHBhcmFtU2l6ZSk7XG4gICAgICBvcFN0YWNrLmRyb3BGcm9tVG9wKHBhcmFtU2l6ZSArIDEpO1xuICAgICAgYXNzZXJ0KHR5cGVvZiAoPGFueT4gb2JqKVttZXRob2RSZWZlcmVuY2UuZnVsbFNpZ25hdHVyZV0gPT09ICdmdW5jdGlvbicsIGBSZXNvbHZlZCBtZXRob2QgJHttZXRob2RSZWZlcmVuY2UuZnVsbFNpZ25hdHVyZX0gaXNuJ3QgZGVmaW5lZD8hYCwgdGhyZWFkKTtcbiAgICAgICg8YW55PiBvYmopW21ldGhvZFJlZmVyZW5jZS5mdWxsU2lnbmF0dXJlXSh0aHJlYWQsIGFyZ3MpO1xuICAgICAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGludm9rZXN0YXRpY19mYXN0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBtZXRob2RSZWZlcmVuY2UgPSA8TWV0aG9kUmVmZXJlbmNlIHwgSW50ZXJmYWNlTWV0aG9kUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBvcFN0YWNrID0gZnJhbWUub3BTdGFjaywgcGFyYW1TaXplID0gbWV0aG9kUmVmZXJlbmNlLnBhcmFtV29yZFNpemUsXG4gICAgICBhcmdzID0gb3BTdGFjay5zbGljZUFuZERyb3BGcm9tVG9wKHBhcmFtU2l6ZSk7XG4gICAgYXNzZXJ0KG1ldGhvZFJlZmVyZW5jZS5qc0NvbnN0cnVjdG9yICE9IG51bGwsIFwianNDb25zdHJ1Y3RvciBpcyBtaXNzaW5nPyFcIik7XG4gICAgYXNzZXJ0KHR5cGVvZihtZXRob2RSZWZlcmVuY2UuanNDb25zdHJ1Y3RvclttZXRob2RSZWZlcmVuY2UuZnVsbFNpZ25hdHVyZV0pID09PSAnZnVuY3Rpb24nLCBcIlJlc29sdmVkIG1ldGhvZCBpc24ndCBkZWZpbmVkPyFcIik7XG4gICAgbWV0aG9kUmVmZXJlbmNlLmpzQ29uc3RydWN0b3JbbWV0aG9kUmVmZXJlbmNlLmZ1bGxTaWduYXR1cmVdKHRocmVhZCwgYXJncyk7XG4gICAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaW52b2tldmlydHVhbF9mYXN0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBtZXRob2RSZWZlcmVuY2UgPSA8TWV0aG9kUmVmZXJlbmNlIHwgSW50ZXJmYWNlTWV0aG9kUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBjb3VudCA9IG1ldGhvZFJlZmVyZW5jZS5wYXJhbVdvcmRTaXplLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBvYmo6IEpWTVR5cGVzLmphdmFfbGFuZ19PYmplY3QgPSBvcFN0YWNrLmZyb21Ub3AoY291bnQpO1xuICAgIGlmICghaXNOdWxsKHRocmVhZCwgZnJhbWUsIG9iaikpIHtcbiAgICAgIC8vIFVzZSB0aGUgY2xhc3Mgb2YgdGhlICpvYmplY3QqLlxuICAgICAgYXNzZXJ0KHR5cGVvZiAoPGFueT4gb2JqKVttZXRob2RSZWZlcmVuY2Uuc2lnbmF0dXJlXSA9PT0gJ2Z1bmN0aW9uJywgYFJlc29sdmVkIG1ldGhvZCAke21ldGhvZFJlZmVyZW5jZS5zaWduYXR1cmV9IGlzbid0IGRlZmluZWQ/IWApO1xuICAgICAgKDxhbnk+IG9iailbbWV0aG9kUmVmZXJlbmNlLnNpZ25hdHVyZV0odGhyZWFkLCBvcFN0YWNrLnNsaWNlRnJvbVRvcChjb3VudCkpO1xuICAgICAgb3BTdGFjay5kcm9wRnJvbVRvcChjb3VudCArIDEpO1xuICAgICAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbiAgICB9XG4gICAgLy8gT2JqZWN0IGlzIE5VTEw7IE5QRSBoYXMgYmVlbiB0aHJvd24uXG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGludm9rZWludGVyZmFjZV9mYXN0ID0gT3Bjb2Rlcy5pbnZva2V2aXJ0dWFsX2Zhc3Q7XG5cbiAgcHVibGljIHN0YXRpYyBpbnZva2VkeW5hbWljX2Zhc3QodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGNhbGxTaXRlU3BlY2lmaWVyID0gPEludm9rZUR5bmFtaWM+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKSxcbiAgICAgIGNzbyA9IGNhbGxTaXRlU3BlY2lmaWVyLmdldENhbGxTaXRlT2JqZWN0KHBjKSxcbiAgICAgIGFwcGVuZGl4ID0gY3NvWzFdLFxuICAgICAgZmNuID0gY3NvWzBdLnZtdGFyZ2V0LFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssIHBhcmFtU2l6ZSA9IGNhbGxTaXRlU3BlY2lmaWVyLnBhcmFtV29yZFNpemUsXG4gICAgICBhcmdzID0gb3BTdGFjay5zbGljZUFuZERyb3BGcm9tVG9wKHBhcmFtU2l6ZSk7XG5cbiAgICBpZiAoYXBwZW5kaXggIT09IG51bGwpIHtcbiAgICAgIGFyZ3MucHVzaChhcHBlbmRpeCk7XG4gICAgfVxuICAgIGZjbih0aHJlYWQsIG51bGwsIGFyZ3MpO1xuICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gIH1cblxuICAvKipcbiAgICogT3Bjb2RlIGZvciBNZXRob2RIYW5kbGUuaW52b2tlIGFuZCBNZXRob2RIYW5kbGUuaW52b2tlRXhhY3QuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGludm9rZWhhbmRsZSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgbWV0aG9kUmVmZXJlbmNlID0gPE1ldGhvZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBmY24gPSBtZXRob2RSZWZlcmVuY2UubWVtYmVyTmFtZS52bXRhcmdldCxcbiAgICAgIC8vIEFkZCBpbiAxIGZvciB0aGUgbWV0aG9kIGhhbmRsZSBpdHNlbGYuXG4gICAgICBwYXJhbVNpemUgPSBtZXRob2RSZWZlcmVuY2UucGFyYW1Xb3JkU2l6ZSArIDEsXG4gICAgICBhcHBlbmRpeCA9IG1ldGhvZFJlZmVyZW5jZS5hcHBlbmRpeCxcbiAgICAgIGFyZ3MgPSBvcFN0YWNrLnNsaWNlRnJvbVRvcChwYXJhbVNpemUpO1xuXG4gICAgaWYgKGFwcGVuZGl4ICE9PSBudWxsKSB7XG4gICAgICBhcmdzLnB1c2goYXBwZW5kaXgpO1xuICAgIH1cblxuICAgIGlmICghaXNOdWxsKHRocmVhZCwgZnJhbWUsIGFyZ3NbMF0pKSB7XG4gICAgICBvcFN0YWNrLmRyb3BGcm9tVG9wKHBhcmFtU2l6ZSk7XG4gICAgICAvLyBmY24gd2lsbCBoYW5kbGUgaW52b2tpbmcgJ3RoaXMnIGFuZCBzdWNoLlxuICAgICAgLy8gVE9ETzogSWYgdGhpcyBjYW4gYmUgdmFyYXJncywgcGFzcyBpbiBwYXJhbWV0ZXIgdHlwZXMgdG8gdGhlIGZ1bmN0aW9uLlxuICAgICAgZmNuKHRocmVhZCwgbnVsbCwgYXJncyk7XG4gICAgICBmcmFtZS5yZXR1cm5Ub1RocmVhZExvb3AgPSB0cnVlO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBPcGNvZGUgZm9yIE1ldGhvZEhhbmRsZS5pbnZva2VCYXNpYy5cbiAgICogVW5saWtlIGludm9rZS9pbnZva2VFeGFjdCwgaW52b2tlQmFzaWMgZG9lcyBub3QgY2FsbCBhIGdlbmVyYXRlZCBieXRlY29kZVxuICAgKiBtZXRob2QuIEl0IGNhbGxzIHRoZSB2bXRhcmdldCBlbWJlZGRlZCBpbiB0aGUgTWV0aG9kSGFuZGxlciBkaXJlY3RseS5cbiAgICogVGhpcyBjYW4gY2F1c2UgY3Jhc2hlcyB3aXRoIG1hbGZvcm1lZCBjYWxscywgdGh1cyBpdCBpcyBvbmx5IGFjY2Vzc3NpYmxlXG4gICAqIHRvIHRydXN0ZWQgSkRLIGNvZGUuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGludm9rZWJhc2ljKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBtZXRob2RSZWZlcmVuY2UgPSA8TWV0aG9kUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBwYXJhbVNpemUgPSBtZXRob2RSZWZlcmVuY2UuZ2V0UGFyYW1Xb3JkU2l6ZSgpLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBvYmo6IEpWTVR5cGVzLmphdmFfbGFuZ19pbnZva2VfTWV0aG9kSGFuZGxlID0gb3BTdGFjay5mcm9tVG9wKHBhcmFtU2l6ZSksXG4gICAgICAvLyBOZWVkIHRvIGluY2x1ZGUgdGhlIE1ldGhvZEhhbmRsZSBpbiB0aGUgYXJndW1lbnRzIHRvIHZtdGFyZ2V0LiB2bXRhcmdldFxuICAgICAgLy8gd2lsbCBhcHByb3ByaWF0ZWx5IGludm9rZSBpdC5cbiAgICAgIGFyZ3MgPSBvcFN0YWNrLnNsaWNlRnJvbVRvcChwYXJhbVNpemUgKyAxKSxcbiAgICAgIGxtYmRhRm9ybTogSlZNVHlwZXMuamF2YV9sYW5nX2ludm9rZV9MYW1iZGFGb3JtLFxuICAgICAgbW46IEpWTVR5cGVzLmphdmFfbGFuZ19pbnZva2VfTWVtYmVyTmFtZSxcbiAgICAgIG06IE1ldGhvZDtcblxuICAgIC8vIG9iaiBpcyBhIE1ldGhvZEhhbmRsZS5cbiAgICBpZiAoIWlzTnVsbCh0aHJlYWQsIGZyYW1lLCBvYmopKSB7XG4gICAgICBvcFN0YWNrLmRyb3BGcm9tVG9wKHBhcmFtU2l6ZSArIDEpO1xuICAgICAgbG1iZGFGb3JtID0gb2JqWydqYXZhL2xhbmcvaW52b2tlL01ldGhvZEhhbmRsZS9mb3JtJ107XG4gICAgICBtbiA9IGxtYmRhRm9ybVsnamF2YS9sYW5nL2ludm9rZS9MYW1iZGFGb3JtL3ZtZW50cnknXTtcbiAgICAgIGFzc2VydChtbi52bXRhcmdldCAhPT0gbnVsbCAmJiBtbi52bXRhcmdldCAhPT0gdW5kZWZpbmVkLCBcInZtdGFyZ2V0IG11c3QgYmUgZGVmaW5lZFwiKTtcbiAgICAgIG1uLnZtdGFyZ2V0KHRocmVhZCwgbWV0aG9kUmVmZXJlbmNlLm5hbWVBbmRUeXBlSW5mby5kZXNjcmlwdG9yLCBhcmdzKTtcbiAgICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFsc28gdXNlZCBmb3IgbGlua1RvU3RhdGljLlxuICAgKiBUT0RPOiBEZS1jb25mbGF0ZSB0aGUgdHdvLlxuICAgKiBUT0RPOiBWYXJhcmdzIGZ1bmN0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgbGlua3Rvc3BlY2lhbCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgbWV0aG9kUmVmZXJlbmNlID0gPE1ldGhvZFJlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssIHBhcmFtU2l6ZSA9IG1ldGhvZFJlZmVyZW5jZS5wYXJhbVdvcmRTaXplLFxuICAgICAgLy8gRmluYWwgYXJndW1lbnQgaXMgdGhlIHJlbGV2YW50IE1lbWJlck5hbWUuIEZ1bmN0aW9uIGFyZ3MgYXJlIHJpZ2h0XG4gICAgICAvLyBiZWZvcmUgaXQuXG4gICAgICBhcmdzID0gb3BTdGFjay5zbGljZUZyb21Ub3AocGFyYW1TaXplKSxcbiAgICAgIG1lbWJlck5hbWU6IEpWTVR5cGVzLmphdmFfbGFuZ19pbnZva2VfTWVtYmVyTmFtZSA9IGFyZ3MucG9wKCksXG4gICAgICAvLyBUT0RPOiBVc2UgcGFyc2VkIGRlc2NyaXB0b3IuXG4gICAgICBkZXNjID0gbWV0aG9kUmVmZXJlbmNlLm5hbWVBbmRUeXBlSW5mby5kZXNjcmlwdG9yO1xuXG4gICAgaWYgKCFpc051bGwodGhyZWFkLCBmcmFtZSwgbWVtYmVyTmFtZSkpIHtcbiAgICAgIG9wU3RhY2suZHJvcEZyb21Ub3AocGFyYW1TaXplKTtcbiAgICAgIGFzc2VydChtZW1iZXJOYW1lLmdldENsYXNzKCkuZ2V0SW50ZXJuYWxOYW1lKCkgPT09IFwiTGphdmEvbGFuZy9pbnZva2UvTWVtYmVyTmFtZTtcIik7XG4gICAgICAvLyBwYXJhbWV0ZXJUeXBlcyBmb3IgZnVuY3Rpb24gYXJlIHRoZSBzYW1lIGFzIHRoZSBtZXRob2QgcmVmZXJlbmNlLCBidXQgd2l0aG91dCB0aGUgdHJhaWxpbmcgTWVtYmVyTmFtZS5cbiAgICAgIC8vIFRPRE86IFVzZSBwYXJzZWQgZGVzY3JpcHRvciwgYXZvaWQgcmUtZG9pbmcgd29yayBoZXJlLlxuICAgICAgbWVtYmVyTmFtZS52bXRhcmdldCh0aHJlYWQsIGRlc2MucmVwbGFjZShcIkxqYXZhL2xhbmcvaW52b2tlL01lbWJlck5hbWU7KVwiLCBcIilcIiksIGFyZ3MpO1xuICAgICAgZnJhbWUucmV0dXJuVG9UaHJlYWRMb29wID0gdHJ1ZTtcbiAgICB9XG4gIH1cblxuICAvLyBYWFg6IFZhcmFyZ3MgZnVuY3Rpb25zLiBXZSdyZSBzdXBwb3NlZCB0byBib3ggYXJncyBpZiB0YXJnZXQgaXMgdmFyYXJncy5cbiAgcHVibGljIHN0YXRpYyBsaW5rdG92aXJ0dWFsKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBtZXRob2RSZWZlcmVuY2UgPSA8TWV0aG9kUmVmZXJlbmNlIHwgSW50ZXJmYWNlTWV0aG9kUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBwYXJhbVNpemUgPSBtZXRob2RSZWZlcmVuY2UucGFyYW1Xb3JkU2l6ZSxcbiAgICAgIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgYXJncyA9IG9wU3RhY2suc2xpY2VGcm9tVG9wKHBhcmFtU2l6ZSksXG4gICAgICAvLyBGaW5hbCBhcmd1bWVudCBpcyB0aGUgcmVsZXZhbnQgTWVtYmVyTmFtZS4gRnVuY3Rpb24gYXJncyBhcmUgcmlnaHRcbiAgICAgIC8vIGJlZm9yZSBpdC5cbiAgICAgIG1lbWJlck5hbWU6IEpWTVR5cGVzLmphdmFfbGFuZ19pbnZva2VfTWVtYmVyTmFtZSA9IGFyZ3MucG9wKCksXG4gICAgICBkZXNjID0gbWV0aG9kUmVmZXJlbmNlLm5hbWVBbmRUeXBlSW5mby5kZXNjcmlwdG9yO1xuXG4gICAgaWYgKCFpc051bGwodGhyZWFkLCBmcmFtZSwgbWVtYmVyTmFtZSkpIHtcbiAgICAgIG9wU3RhY2suZHJvcEZyb21Ub3AocGFyYW1TaXplKTtcbiAgICAgIGFzc2VydChtZW1iZXJOYW1lLmdldENsYXNzKCkuZ2V0SW50ZXJuYWxOYW1lKCkgPT09IFwiTGphdmEvbGFuZy9pbnZva2UvTWVtYmVyTmFtZTtcIik7XG4gICAgICAvLyBwYXJhbWV0ZXJUeXBlcyBmb3IgZnVuY3Rpb24gYXJlIHRoZSBzYW1lIGFzIHRoZSBtZXRob2QgcmVmZXJlbmNlLCBidXQgd2l0aG91dCB0aGUgdHJhaWxpbmcgTWVtYmVyTmFtZS5cbiAgICAgIG1lbWJlck5hbWUudm10YXJnZXQodGhyZWFkLCBkZXNjLnJlcGxhY2UoXCJMamF2YS9sYW5nL2ludm9rZS9NZW1iZXJOYW1lOylcIiwgXCIpXCIpLCBhcmdzKTtcbiAgICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBicmVha3BvaW50KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgXCJMamF2YS9sYW5nL0Vycm9yO1wiLCBcImJyZWFrcG9pbnQgbm90IGltcGxlbWVudGVkLlwiKTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbmV3KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBjbGFzc1JlZiA9IDxDbGFzc1JlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGlmIChjbGFzc1JlZi5pc1Jlc29sdmVkKCkpIHtcbiAgICAgIHZhciBjbHMgPSBjbGFzc1JlZi5jbHM7XG4gICAgICBpZiAoY2xzLmlzSW5pdGlhbGl6ZWQodGhyZWFkKSkge1xuICAgICAgICBjb2RlW3BjXSA9IE9wQ29kZS5ORVdfRkFTVDtcbiAgICAgICAgLy8gUmV0dXJuIHRvIHRocmVhZCwgcmVydW4gb3Bjb2RlLlxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgaW5pdGlhbGl6ZUNsYXNzRnJvbUNsYXNzKHRocmVhZCwgZnJhbWUsIGNscyk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIHJlc29sdmVDUEl0ZW0odGhyZWFkLCBmcmFtZSwgY2xhc3NSZWYpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbmV3X2Zhc3QodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGNsYXNzUmVmID0gPENsYXNzUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoKG5ldyBjbGFzc1JlZi5jbHNDb25zdHJ1Y3Rvcih0aHJlYWQpKTtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBuZXdhcnJheSh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICAvLyBUT0RPOiBTdGFzaCBhbGwgb2YgdGhlc2UgYXJyYXkgdHlwZXMgZHVyaW5nIEpWTSBzdGFydHVwLlxuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIHR5cGUgPSBcIltcIiArIEFycmF5VHlwZXNbY29kZVtwYyArIDFdXSxcbiAgICAgIGNscyA9IDxBcnJheUNsYXNzRGF0YTxhbnk+PiBmcmFtZS5nZXRMb2FkZXIoKS5nZXRJbml0aWFsaXplZENsYXNzKHRocmVhZCwgdHlwZSksXG4gICAgICBsZW5ndGggPSBvcFN0YWNrLnBvcCgpO1xuICAgIGlmIChsZW5ndGggPj0gMCkge1xuICAgICAgb3BTdGFjay5wdXNoKG5ldyAoY2xzLmdldENvbnN0cnVjdG9yKHRocmVhZCkpKHRocmVhZCwgbGVuZ3RoKSk7XG4gICAgICBmcmFtZS5wYyArPSAyO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvd0V4Y2VwdGlvbih0aHJlYWQsIGZyYW1lLCAnTGphdmEvbGFuZy9OZWdhdGl2ZUFycmF5U2l6ZUV4Y2VwdGlvbjsnLCBgVHJpZWQgdG8gaW5pdCAke3R5cGV9IGFycmF5IHdpdGggbGVuZ3RoICR7bGVuZ3RofWApO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgYW5ld2FycmF5KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBjbGFzc1JlZiA9IDxDbGFzc1JlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGlmIChjbGFzc1JlZi5pc1Jlc29sdmVkKCkpIHtcbiAgICAgIC8vIFJld3JpdGUgYW5kIHJlcnVuLlxuICAgICAgY29kZVtwY10gPSBPcENvZGUuQU5FV0FSUkFZX0ZBU1Q7XG4gICAgICBjbGFzc1JlZi5hcnJheUNsYXNzID0gPEFycmF5Q2xhc3NEYXRhPGFueT4+IGZyYW1lLmdldExvYWRlcigpLmdldEluaXRpYWxpemVkQ2xhc3ModGhyZWFkLCBgWyR7Y2xhc3NSZWYuY2xzLmdldEludGVybmFsTmFtZSgpfWApO1xuICAgICAgY2xhc3NSZWYuYXJyYXlDbGFzc0NvbnN0cnVjdG9yID0gY2xhc3NSZWYuYXJyYXlDbGFzcy5nZXRDb25zdHJ1Y3Rvcih0aHJlYWQpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXNvbHZlQ1BJdGVtKHRocmVhZCwgZnJhbWUsIGNsYXNzUmVmKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGFuZXdhcnJheV9mYXN0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGNsYXNzUmVmID0gPENsYXNzUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBsZW5ndGggPSBvcFN0YWNrLnBvcCgpO1xuXG4gICAgaWYgKGxlbmd0aCA+PSAwKSB7XG4gICAgICBvcFN0YWNrLnB1c2gobmV3IGNsYXNzUmVmLmFycmF5Q2xhc3NDb25zdHJ1Y3Rvcih0aHJlYWQsIGxlbmd0aCkpO1xuICAgICAgZnJhbWUucGMgKz0gMztcbiAgICB9IGVsc2Uge1xuICAgICAgdGhyb3dFeGNlcHRpb24odGhyZWFkLCBmcmFtZSwgJ0xqYXZhL2xhbmcvTmVnYXRpdmVBcnJheVNpemVFeGNlcHRpb247JywgYFRyaWVkIHRvIGluaXQgJHtjbGFzc1JlZi5hcnJheUNsYXNzLmdldEludGVybmFsTmFtZSgpfSBhcnJheSB3aXRoIGxlbmd0aCAke2xlbmd0aH1gKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGFycmF5bGVuZ3RoKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdmFyIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLCBvYmo6IEpWTVR5cGVzLkpWTUFycmF5PGFueT4gPSBvcFN0YWNrLnBvcCgpO1xuICAgIGlmICghaXNOdWxsKHRocmVhZCwgZnJhbWUsIG9iaikpIHtcbiAgICAgIG9wU3RhY2sucHVzaChvYmouYXJyYXkubGVuZ3RoKTtcbiAgICAgIGZyYW1lLnBjKys7XG4gICAgfVxuICAgIC8vIG9iaiBpcyBOVUxMLiBpc051bGwgdGhyZXcgYW4gZXhjZXB0aW9uIGZvciB1cy5cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgYXRocm93KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lKSB7XG4gICAgdGhyZWFkLnRocm93RXhjZXB0aW9uKGZyYW1lLm9wU3RhY2sucG9wKCkpO1xuICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGNoZWNrY2FzdCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgY2xhc3NSZWYgPSA8Q2xhc3NSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKTtcbiAgICBpZiAoY2xhc3NSZWYuaXNSZXNvbHZlZCgpKSB7XG4gICAgICAvLyBSZXdyaXRlIHRvIGZhc3QgdmVyc2lvbiwgYW5kIHJlLWV4ZWN1dGUuXG4gICAgICBjb2RlW3BjXSA9IE9wQ29kZS5DSEVDS0NBU1RfRkFTVDtcbiAgICB9IGVsc2Uge1xuICAgICAgcmVzb2x2ZUNQSXRlbSh0aHJlYWQsIGZyYW1lLCBjbGFzc1JlZik7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBjaGVja2Nhc3RfZmFzdCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgY2xhc3NSZWYgPSA8Q2xhc3NSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKSxcbiAgICAgIGNscyA9IGNsYXNzUmVmLmNscyxcbiAgICAgIG9wU3RhY2sgPSBmcmFtZS5vcFN0YWNrLFxuICAgICAgbzogSlZNVHlwZXMuamF2YV9sYW5nX09iamVjdCA9IG9wU3RhY2sudG9wKCk7XG4gICAgaWYgKChvICE9IG51bGwpICYmICFvLmdldENsYXNzKCkuaXNDYXN0YWJsZShjbHMpKSB7XG4gICAgICB2YXIgdGFyZ2V0Q2xhc3MgPSBjbHMuZ2V0RXh0ZXJuYWxOYW1lKCk7XG4gICAgICB2YXIgY2FuZGlkYXRlQ2xhc3MgPSBvLmdldENsYXNzKCkuZ2V0RXh0ZXJuYWxOYW1lKCk7XG4gICAgICB0aHJvd0V4Y2VwdGlvbih0aHJlYWQsIGZyYW1lLCAnTGphdmEvbGFuZy9DbGFzc0Nhc3RFeGNlcHRpb247JywgYCR7Y2FuZGlkYXRlQ2xhc3N9IGNhbm5vdCBiZSBjYXN0IHRvICR7dGFyZ2V0Q2xhc3N9YCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIFN1Y2Nlc3MhXG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaW5zdGFuY2VvZih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgY2xhc3NSZWYgPSA8Q2xhc3NSZWZlcmVuY2U+IGZyYW1lLm1ldGhvZC5jbHMuY29uc3RhbnRQb29sLmdldChjb2RlLnJlYWRVSW50MTZCRShwYyArIDEpKTtcbiAgICBpZiAoY2xhc3NSZWYuaXNSZXNvbHZlZCgpKSB7XG4gICAgICAvLyBSZXdyaXRlIGFuZCByZXJ1bi5cbiAgICAgIGNvZGVbcGNdID0gT3BDb2RlLklOU1RBTkNFT0ZfRkFTVDtcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gRmV0Y2ggY2xhc3MgYW5kIHJlcnVuIG9wY29kZS5cbiAgICAgIHJlc29sdmVDUEl0ZW0odGhyZWFkLCBmcmFtZSwgY2xhc3NSZWYpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgaW5zdGFuY2VvZl9mYXN0KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBjbGFzc1JlZiA9IDxDbGFzc1JlZmVyZW5jZT4gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpLFxuICAgICAgY2xzID0gY2xhc3NSZWYuY2xzLFxuICAgICAgb3BTdGFjayA9IGZyYW1lLm9wU3RhY2ssXG4gICAgICBvID0gPEpWTVR5cGVzLmphdmFfbGFuZ19PYmplY3Q+IG9wU3RhY2sucG9wKCk7XG4gICAgb3BTdGFjay5wdXNoKG8gIT09IG51bGwgPyAoby5nZXRDbGFzcygpLmlzQ2FzdGFibGUoY2xzKSA/IDEgOiAwKSA6IDApO1xuICAgIGZyYW1lLnBjICs9IDM7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIG1vbml0b3JlbnRlcih0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBvcFN0YWNrID0gZnJhbWUub3BTdGFjaywgbW9uaXRvck9iajogSlZNVHlwZXMuamF2YV9sYW5nX09iamVjdCA9IG9wU3RhY2sucG9wKCksXG4gICAgICBtb25pdG9yRW50ZXJlZCA9ICgpID0+IHtcbiAgICAgICAgLy8gW05vdGU6IFRocmVhZCBpcyBub3cgaW4gdGhlIFJVTk5BQkxFIHN0YXRlLl1cbiAgICAgICAgLy8gSW5jcmVtZW50IHRoZSBQQy5cbiAgICAgICAgZnJhbWUucGMrKztcbiAgICAgIH07XG5cbiAgICBpZiAoIW1vbml0b3JPYmouZ2V0TW9uaXRvcigpLmVudGVyKHRocmVhZCwgbW9uaXRvckVudGVyZWQpKSB7XG4gICAgICAvLyBPcGNvZGUgZmFpbGVkLiBtb25pdG9yRW50ZXJlZCB3aWxsIGJlIHJ1biBvbmNlIHdlIG93biB0aGUgbW9uaXRvci5cbiAgICAgIC8vIFRoZSB0aHJlYWQgaXMgbm93IGluIHRoZSBCTE9DS0VEIHN0YXRlLiBUZWxsIHRoZSBmcmFtZSB0byByZXR1cm4gdG9cbiAgICAgIC8vIHRoZSB0aHJlYWQgbG9vcC5cbiAgICAgIGZyYW1lLnJldHVyblRvVGhyZWFkTG9vcCA9IHRydWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIG1vbml0b3JFbnRlcmVkKCk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBtb25pdG9yZXhpdCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIHZhciBtb25pdG9yT2JqOiBKVk1UeXBlcy5qYXZhX2xhbmdfT2JqZWN0ID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICBpZiAobW9uaXRvck9iai5nZXRNb25pdG9yKCkuZXhpdCh0aHJlYWQpKSB7XG4gICAgICBmcmFtZS5wYysrO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBtb25pdG9yZXhpdCBmYWlsZWQsIGFuZCB0aHJldyBhbiBleGNlcHRpb24uXG4gICAgICBmcmFtZS5yZXR1cm5Ub1RocmVhZExvb3AgPSB0cnVlO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgbXVsdGlhbmV3YXJyYXkodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGNsYXNzUmVmID0gPENsYXNzUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG4gICAgaWYgKGNsYXNzUmVmLmlzUmVzb2x2ZWQoKSkge1xuICAgICAgLy8gUmV3cml0ZSBhbmQgcmVydW4uXG4gICAgICBjb2RlW3BjXSA9IE9wQ29kZS5NVUxUSUFORVdBUlJBWV9GQVNUO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXNvbHZlQ1BJdGVtKHRocmVhZCwgZnJhbWUsIGNsYXNzUmVmKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIG11bHRpYW5ld2FycmF5X2Zhc3QodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGNsYXNzUmVmID0gPENsYXNzUmVmZXJlbmNlPiBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSksXG4gICAgICBvcFN0YWNrID0gZnJhbWUub3BTdGFjayxcbiAgICAgIGRpbSA9IGNvZGVbcGMgKyAzXSxcbiAgICAgIGk6IG51bWJlcixcbiAgICAgIC8vIEFyZ3VtZW50cyB0byB0aGUgY29uc3RydWN0b3IuXG4gICAgICBhcmdzID0gbmV3IEFycmF5PG51bWJlcj4oZGltKSwgZGltU2l6ZTogbnVtYmVyO1xuXG4gICAgZm9yIChpID0gMDsgaSA8IGRpbTsgaSsrKSB7XG4gICAgICBkaW1TaXplID0gb3BTdGFjay5wb3AoKTtcbiAgICAgIGFyZ3NbZGltIC0gaSAtIDFdID0gZGltU2l6ZTtcbiAgICAgIGlmIChkaW1TaXplIDwgMCkge1xuICAgICAgICB0aHJvd0V4Y2VwdGlvbih0aHJlYWQsIGZyYW1lLCAnTGphdmEvbGFuZy9OZWdhdGl2ZUFycmF5U2l6ZUV4Y2VwdGlvbjsnLCBgVHJpZWQgdG8gaW5pdCAke2NsYXNzUmVmLmNscy5nZXRJbnRlcm5hbE5hbWUoKX0gYXJyYXkgd2l0aCBhIGRpbWVuc2lvbiBvZiBsZW5ndGggJHtkaW1TaXplfWApO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG4gICAgfVxuICAgIG9wU3RhY2sucHVzaChuZXcgKGNsYXNzUmVmLmNscy5nZXRDb25zdHJ1Y3Rvcih0aHJlYWQpKSh0aHJlYWQsIGFyZ3MpKTtcbiAgICBmcmFtZS5wYyArPSA0O1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBpZm51bGwodGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgaWYgKGZyYW1lLm9wU3RhY2sucG9wKCkgPT0gbnVsbCkge1xuICAgICAgY29uc3Qgb2Zmc2V0ID0gY29kZS5yZWFkSW50MTZCRShwYyArIDEpO1xuICAgICAgZnJhbWUucGMgKz0gb2Zmc2V0O1xuICAgICAgaWYgKG9mZnNldCA8IDApIHtcbiAgICAgICAgZnJhbWUubWV0aG9kLmluY3JCQkVudHJpZXMoKTtcbiAgICAgIH1cbiAgICB9IGVsc2Uge1xuICAgICAgZnJhbWUucGMgKz0gMztcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGlmbm9ubnVsbCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBpZiAoZnJhbWUub3BTdGFjay5wb3AoKSAhPSBudWxsKSB7XG4gICAgICBjb25zdCBvZmZzZXQgPSBjb2RlLnJlYWRJbnQxNkJFKHBjICsgMSk7XG4gICAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgICBpZiAob2Zmc2V0IDwgMCkge1xuICAgICAgICBmcmFtZS5tZXRob2QuaW5jckJCRW50cmllcygpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBmcmFtZS5wYyArPSAzO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgZ290b193KHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIGNvbnN0IG9mZnNldCA9IGNvZGUucmVhZEludDMyQkUocGMgKyAxKTtcbiAgICBmcmFtZS5wYyArPSBvZmZzZXQ7XG4gICAgaWYgKG9mZnNldCA8IDApIHtcbiAgICAgIGZyYW1lLm1ldGhvZC5pbmNyQkJFbnRyaWVzKCk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBqc3Jfdyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICBmcmFtZS5vcFN0YWNrLnB1c2goZnJhbWUucGMgKyA1KTtcbiAgICBmcmFtZS5wYyArPSBjb2RlLnJlYWRJbnQzMkJFKHBjICsgMSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIG5vcCh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSkge1xuICAgIGZyYW1lLnBjICs9IDE7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxkYyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgY29uc3RhbnQgPSBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZVtwYyArIDFdKTtcbiAgICBpZiAoY29uc3RhbnQuaXNSZXNvbHZlZCgpKSB7XG4gICAgICBhc3NlcnQoKCgpID0+IHtcbiAgICAgICAgc3dpdGNoIChjb25zdGFudC5nZXRUeXBlKCkpIHtcbiAgICAgICAgICBjYXNlIENvbnN0YW50UG9vbEl0ZW1UeXBlLlNUUklORzpcbiAgICAgICAgICBjYXNlIENvbnN0YW50UG9vbEl0ZW1UeXBlLkNMQVNTOlxuICAgICAgICAgIGNhc2UgQ29uc3RhbnRQb29sSXRlbVR5cGUuTUVUSE9EX0hBTkRMRTpcbiAgICAgICAgICBjYXNlIENvbnN0YW50UG9vbEl0ZW1UeXBlLk1FVEhPRF9UWVBFOlxuICAgICAgICAgIGNhc2UgQ29uc3RhbnRQb29sSXRlbVR5cGUuSU5URUdFUjpcbiAgICAgICAgICBjYXNlIENvbnN0YW50UG9vbEl0ZW1UeXBlLkZMT0FUOlxuICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgfVxuICAgICAgfSkoKSwgYENvbnN0YW50IHBvb2wgaXRlbSAke0NvbnN0YW50UG9vbEl0ZW1UeXBlW2NvbnN0YW50LmdldFR5cGUoKV19IGlzIG5vdCBhcHByb3ByaWF0ZSBmb3IgTERDLmApO1xuICAgICAgZnJhbWUub3BTdGFjay5wdXNoKGNvbnN0YW50LmdldENvbnN0YW50KHRocmVhZCkpO1xuICAgICAgZnJhbWUucGMgKz0gMjtcbiAgICB9IGVsc2Uge1xuICAgICAgcmVzb2x2ZUNQSXRlbSh0aHJlYWQsIGZyYW1lLCBjb25zdGFudCk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsZGNfdyh0aHJlYWQ6IEpWTVRocmVhZCwgZnJhbWU6IEJ5dGVjb2RlU3RhY2tGcmFtZSwgY29kZTogQnVmZmVyKSB7XG4gICAgY29uc3QgcGMgPSBmcmFtZS5wYztcbiAgICB2YXIgY29uc3RhbnQgPSBmcmFtZS5tZXRob2QuY2xzLmNvbnN0YW50UG9vbC5nZXQoY29kZS5yZWFkVUludDE2QkUocGMgKyAxKSk7XG4gICAgaWYgKGNvbnN0YW50LmlzUmVzb2x2ZWQoKSkge1xuICAgICAgYXNzZXJ0KCgoKSA9PiB7XG4gICAgICAgIHN3aXRjaCAoY29uc3RhbnQuZ2V0VHlwZSgpKSB7XG4gICAgICAgICAgY2FzZSBDb25zdGFudFBvb2xJdGVtVHlwZS5TVFJJTkc6XG4gICAgICAgICAgY2FzZSBDb25zdGFudFBvb2xJdGVtVHlwZS5DTEFTUzpcbiAgICAgICAgICBjYXNlIENvbnN0YW50UG9vbEl0ZW1UeXBlLk1FVEhPRF9IQU5ETEU6XG4gICAgICAgICAgY2FzZSBDb25zdGFudFBvb2xJdGVtVHlwZS5NRVRIT0RfVFlQRTpcbiAgICAgICAgICBjYXNlIENvbnN0YW50UG9vbEl0ZW1UeXBlLklOVEVHRVI6XG4gICAgICAgICAgY2FzZSBDb25zdGFudFBvb2xJdGVtVHlwZS5GTE9BVDpcbiAgICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICAgIH1cbiAgICAgIH0pKCksIGBDb25zdGFudCBwb29sIGl0ZW0gJHtDb25zdGFudFBvb2xJdGVtVHlwZVtjb25zdGFudC5nZXRUeXBlKCldfSBpcyBub3QgYXBwcm9wcmlhdGUgZm9yIExEQ19XLmApO1xuICAgICAgZnJhbWUub3BTdGFjay5wdXNoKGNvbnN0YW50LmdldENvbnN0YW50KHRocmVhZCkpO1xuICAgICAgZnJhbWUucGMgKz0gMztcbiAgICB9IGVsc2Uge1xuICAgICAgcmVzb2x2ZUNQSXRlbSh0aHJlYWQsIGZyYW1lLCBjb25zdGFudCk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIHN0YXRpYyBsZGMyX3codGhyZWFkOiBKVk1UaHJlYWQsIGZyYW1lOiBCeXRlY29kZVN0YWNrRnJhbWUsIGNvZGU6IEJ1ZmZlcikge1xuICAgIGNvbnN0IHBjID0gZnJhbWUucGM7XG4gICAgdmFyIGNvbnN0YW50ID0gZnJhbWUubWV0aG9kLmNscy5jb25zdGFudFBvb2wuZ2V0KGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMSkpO1xuICAgIGFzc2VydChjb25zdGFudC5nZXRUeXBlKCkgPT09IENvbnN0YW50UG9vbEl0ZW1UeXBlLkxPTkdcbiAgICAgIHx8IGNvbnN0YW50LmdldFR5cGUoKSA9PT0gQ29uc3RhbnRQb29sSXRlbVR5cGUuRE9VQkxFLFxuICAgICAgYEludmFsaWQgbGRjX3cgY29uc3RhbnQgcG9vbCB0eXBlOiAke0NvbnN0YW50UG9vbEl0ZW1UeXBlW2NvbnN0YW50LmdldFR5cGUoKV19YCk7XG4gICAgZnJhbWUub3BTdGFjay5wdXNoV2l0aE51bGwoKDxDb25zdExvbmcgfCBDb25zdERvdWJsZT4gY29uc3RhbnQpLnZhbHVlKTtcbiAgICBmcmFtZS5wYyArPSAzO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyB3aWRlKHRocmVhZDogSlZNVGhyZWFkLCBmcmFtZTogQnl0ZWNvZGVTdGFja0ZyYW1lLCBjb2RlOiBCdWZmZXIpIHtcbiAgICBjb25zdCBwYyA9IGZyYW1lLnBjO1xuICAgIHZhciBpbmRleCA9IGNvZGUucmVhZFVJbnQxNkJFKHBjICsgMik7XG4gICAgLy8gSW5jcmVtZW50IFBDIGJlZm9yZSBzd2l0Y2ggdG8gYXZvaWQgaXNzdWUgd2hlcmUgcmV0IGNoYW5jZXMgUEMgYW5kIHdlXG4gICAgLy8gZXJyb25lb3VzbHkgaW5jcmVtZW50IHRoZSBQQyBmdXJ0aGVyLlxuICAgIGZyYW1lLnBjICs9IDQ7XG4gICAgc3dpdGNoIChjb2RlW3BjICsgMV0pIHtcbiAgICAgIGNhc2UgT3BDb2RlLklMT0FEOlxuICAgICAgY2FzZSBPcENvZGUuRkxPQUQ6XG4gICAgICBjYXNlIE9wQ29kZS5BTE9BRDpcbiAgICAgICAgZnJhbWUub3BTdGFjay5wdXNoKGZyYW1lLmxvY2Fsc1tpbmRleF0pO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgT3BDb2RlLkxMT0FEOlxuICAgICAgY2FzZSBPcENvZGUuRExPQUQ6XG4gICAgICAgIGZyYW1lLm9wU3RhY2sucHVzaFdpdGhOdWxsKGZyYW1lLmxvY2Fsc1tpbmRleF0pO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgT3BDb2RlLklTVE9SRTpcbiAgICAgIGNhc2UgT3BDb2RlLkZTVE9SRTpcbiAgICAgIGNhc2UgT3BDb2RlLkFTVE9SRTpcbiAgICAgICAgZnJhbWUubG9jYWxzW2luZGV4XSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSBPcENvZGUuTFNUT1JFOlxuICAgICAgY2FzZSBPcENvZGUuRFNUT1JFOlxuICAgICAgICAvLyBOVUxMXG4gICAgICAgIGZyYW1lLmxvY2Fsc1tpbmRleCArIDFdID0gZnJhbWUub3BTdGFjay5wb3AoKTtcbiAgICAgICAgLy8gVGhlIGFjdHVhbCB2YWx1ZS5cbiAgICAgICAgZnJhbWUubG9jYWxzW2luZGV4XSA9IGZyYW1lLm9wU3RhY2sucG9wKCk7XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSBPcENvZGUuUkVUOlxuICAgICAgICBmcmFtZS5wYyA9IGZyYW1lLmxvY2Fsc1tpbmRleF07XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSBPcENvZGUuSUlOQzpcbiAgICAgICAgdmFyIHZhbHVlID0gY29kZS5yZWFkSW50MTZCRShwYyArIDQpO1xuICAgICAgICBmcmFtZS5sb2NhbHNbaW5kZXhdID0gKGZyYW1lLmxvY2Fsc1tpbmRleF0gKyB2YWx1ZSkgfCAwO1xuICAgICAgICAvLyB3aWRlIGlpbmMgaGFzIDIgZXh0cmEgYnl0ZXMuXG4gICAgICAgIGZyYW1lLnBjICs9IDI7XG4gICAgICAgIGJyZWFrO1xuICAgICAgZGVmYXVsdDpcbiAgICAgICAgYXNzZXJ0KGZhbHNlLCBgVW5rbm93biB3aWRlIG9wY29kZTogJHtjb2RlW3BjICsgMV19YCk7XG4gICAgICAgIGJyZWFrO1xuICAgIH1cbiAgfVxufVxuXG5leHBvcnQgdmFyIExvb2t1cFRhYmxlOiBJT3Bjb2RlSW1wbGVtZW50YXRpb25bXSA9IG5ldyBBcnJheSgweGZmKTtcbi8vIFB1dCBpbiBmdW5jdGlvbiBjbG9zdXJlIHRvIHByZXZlbnQgc2NvcGUgcG9sbHV0aW9uLlxuKCgpID0+IHtcbiAgZm9yICh2YXIgaSA9IDA7IGkgPCAweGZmOyBpKyspIHtcbiAgICBpZiAoT3BDb2RlLmhhc093blByb3BlcnR5KFwiXCIgKyBpKSkge1xuICAgICAgTG9va3VwVGFibGVbaV0gPSAoPGFueT4gT3Bjb2RlcylbT3BDb2RlW2ldLnRvTG93ZXJDYXNlKCldO1xuICAgICAgYXNzZXJ0KExvb2t1cFRhYmxlW2ldICE9IG51bGwsIGBNaXNzaW5nIGltcGxlbWVudGF0aW9uIG9mIG9wY29kZSAke09wQ29kZVtpXX1gKTtcbiAgICB9XG4gIH1cbn0pKCk7XG4iXX0=