"use strict";
// Usage.
var PrefixParseResult = (function () {
    function PrefixParseResult(result, unparsedArgs) {
        if (unparsedArgs === void 0) { unparsedArgs = []; }
        this._result = result;
        this._unparsedArgs = unparsedArgs;
    }
    PrefixParseResult.prototype.unparsedArgs = function () {
        return this._unparsedArgs;
    };
    PrefixParseResult.prototype.flag = function (name, defaultVal) {
        var val = this._result[name];
        if (typeof (val) === 'boolean') {
            return val;
        }
        return defaultVal;
    };
    PrefixParseResult.prototype.stringOption = function (name, defaultVal) {
        var val = this._result[name];
        if (typeof (val) === 'string') {
            return val;
        }
        return defaultVal;
    };
    PrefixParseResult.prototype.mapOption = function (name) {
        var val = this._result[name];
        if (typeof (val) === 'object') {
            return val;
        }
        return {};
    };
    return PrefixParseResult;
}());
exports.PrefixParseResult = PrefixParseResult;
function getOptName(prefix, name) {
    return prefix !== 'default' ? "" + prefix + name : name;
}
/**
 * Handles parsing for a specific options configuration.
 * Parses Java-style options.
 */
var OptionParser = (function () {
    function OptionParser(desc) {
        var _this = this;
        this._parseMap = {};
        this._prefixes = [];
        this._mapArgs = [];
        this._rawDesc = desc;
        this._prefixes = Object.keys(desc);
        this._prefixes.forEach(function (prefix) {
            var opts = desc[prefix];
            var optNames = Object.keys(opts);
            optNames.slice(0).forEach(function (optName) {
                var option = opts[optName];
                if (option.enabled === false) {
                    // Ignore disabled options.
                    delete desc[prefix][optName];
                    return;
                }
                if (!option.type) {
                    option.type = 0 /* FLAG */;
                }
                if (option.type === 4 /* MAP_SYNTAX */) {
                    // ASSUMPTION: These do not have aliases.
                    _this._mapArgs.push(optName);
                }
                option.prefix = prefix;
                option.name = optName;
                _this._parseMap[getOptName(prefix, optName)] = option;
                if (option.alias) {
                    optNames.push(option.alias);
                    _this._parseMap[getOptName(prefix, option.alias)] = option;
                }
            });
        });
    }
    /**
     * Parses the given arguments. Throws an exception on parsing failure.
     */
    OptionParser.prototype.parse = function (argv) {
        var _this = this;
        var result = {}, ptr = 0, len;
        this._prefixes.forEach(function (prefix) { return result[prefix] = {}; });
        argv = argv.map(function (arg) { return arg.trim(); }).filter(function (arg) { return arg !== ''; });
        len = argv.length;
        while (ptr < len) {
            var arg = argv[ptr];
            if (arg[0] === '-') {
                arg = arg.slice(1);
                var opt;
                if (opt = this._parseMap[arg]) {
                    switch (opt.type) {
                        case 0 /* FLAG */:
                        case 2 /* COLON_VALUE_OR_FLAG_SYNTAX */:
                            result[opt.prefix][opt.name] = true;
                            break;
                        case 3 /* NORMAL_VALUE_SYNTAX */:
                        case 1 /* COLON_VALUE_SYNTAX */:
                            ptr++;
                            if (ptr < len) {
                                result[opt.prefix][opt.name] = argv[ptr];
                            }
                            else {
                                throw new Error("-" + arg + " requires an argument.");
                            }
                            break;
                        case 4 /* MAP_SYNTAX */:
                            // NOP.
                            break;
                        default:
                            // Invalid.
                            throw new Error("INTERNAL ERROR: Invalid parse type for -" + arg + ".");
                    }
                }
                else if (this._mapArgs.filter(function (mapArg) {
                    if (arg.slice(0, mapArg.length) === mapArg) {
                        opt = _this._parseMap[mapArg];
                        return true;
                    }
                    return false;
                }).length > 0) {
                    // ASSUMPTION: Map args are mutually exclusive.
                    // Argument is -{mapArg}key=value
                    // If no value, set to ''.
                    var mapping = arg.slice(opt.name.length), map = result[opt.prefix][opt.name];
                    if (!map) {
                        map = result[opt.prefix][opt.name] = {};
                    }
                    var eqIdx = mapping.indexOf('=');
                    if (eqIdx !== -1) {
                        map[mapping.slice(0, eqIdx)] = mapping.slice(eqIdx + 1);
                    }
                    else {
                        map[mapping] = "";
                    }
                }
                else if (arg.indexOf(':') !== -1 && (opt = this._parseMap[arg.slice(0, arg.indexOf(':'))])) {
                    // Colon option.
                    if (opt.type === 1 /* COLON_VALUE_SYNTAX */ || opt.type === 2 /* COLON_VALUE_OR_FLAG_SYNTAX */) {
                        result[opt.prefix][opt.name] = arg.slice(arg.indexOf(':') + 1);
                    }
                    else {
                        // Unrecognized option.
                        throw new Error("Unrecognized option: -" + arg);
                    }
                }
                else {
                    throw new Error("Unrecognized option: -" + arg);
                }
                if (opt.stopParsing) {
                    ptr++;
                    break;
                }
            }
            else {
                break;
            }
            // Advance to next value.
            ptr++;
        }
        // ptr is at raw args to program / JVM.
        var unparsedArgs = argv.slice(ptr), rv = {};
        Object.keys(result).forEach(function (prefix) {
            rv[prefix] = new PrefixParseResult(result[prefix], unparsedArgs);
        });
        return rv;
    };
    /**
     * Generates help text for the given prefixed options.
     */
    OptionParser.prototype.help = function (prefix) {
        return _showHelp(this._rawDesc[prefix], prefix === 'default' ? '' : prefix);
    };
    return OptionParser;
}());
exports.OptionParser = OptionParser;
function printCol(value, width) {
    var rv = value;
    var padding = width - value.length;
    while (padding-- > 0) {
        rv += ' ';
    }
    return rv;
}
function _showHelp(category, prefix) {
    var combinedKeys = {};
    var keyColWidth = 13;
    Object.keys(category).forEach(function (key) {
        var opt = category[key];
        // Ignored in help text.
        if (opt.stopParsing) {
            return;
        }
        var keys = [key];
        if (opt.alias != null) {
            keys.push(opt.alias);
        }
        var ckey;
        if (opt.optDesc) {
            ckey = keys.map(function (key) { return ("-" + prefix + key + opt.optDesc); }).join("\n");
        }
        else {
            ckey = keys.map(function (key) { return ("-" + prefix + key); }).join(' | ');
        }
        combinedKeys[ckey] = opt;
    });
    return Object.keys(combinedKeys).map(function (key) {
        var option = combinedKeys[key];
        if (option.optDesc) {
            var cols = key.split('\n');
            var rv = cols.map(function (row) { return ("    " + row); });
            // Multiline.
            return rv.join('\n') + "\n                  " + option.desc;
        }
        else {
            var colText = printCol(key, keyColWidth);
            if (colText.length === keyColWidth) {
                return "    " + colText + " " + option.desc;
            }
            else {
                return "    " + colText + "\n                  " + option.desc;
            }
        }
    }).join('\n') + '\n';
}
//# sourceMappingURL=data:application/json;base64,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