"use strict";
var enums_1 = require('./enums');
var assert_1 = require('./assert');
/**
 * Manages parked threads and their callbacks.
 */
var Parker = (function () {
    function Parker() {
        this._parkCounts = {};
        this._parkCallbacks = {};
    }
    Parker.prototype.park = function (thread, cb) {
        var ref = thread.getRef();
        assert_1["default"](!this._parkCallbacks[ref] && thread.getStatus() !== enums_1.ThreadStatus.PARKED, "Thread " + ref + " is doubly parked? Should be impossible.");
        this._parkCallbacks[ref] = cb;
        this._mutateParkCount(thread, 1);
        // It's possible the thread was instantly unparked due to a previously
        // unbalancing park.
        if (this.isParked(thread)) {
            thread.setStatus(enums_1.ThreadStatus.PARKED);
        }
    };
    Parker.prototype.unpark = function (thread) {
        this._mutateParkCount(thread, -1);
    };
    Parker.prototype.completelyUnpark = function (thread) {
        var ref = thread.getRef(), count = this._parkCounts[ref];
        if (count) {
            this._mutateParkCount(thread, -count);
        }
    };
    Parker.prototype._mutateParkCount = function (thread, delta) {
        var ref = thread.getRef(), cb;
        // Initialize park count.
        if (!this._parkCounts[ref]) {
            this._parkCounts[ref] = 0;
        }
        if (0 === (this._parkCounts[ref] += delta)) {
            assert_1["default"](!!this._parkCallbacks[ref], "Balancing unpark for thread " + ref + " with no callback? Should be impossible.");
            cb = this._parkCallbacks[ref];
            // Cleanup.
            delete this._parkCounts[ref];
            delete this._parkCallbacks[ref];
            // Avoid situations where a terminated thread's timeout wakes up
            // and tries to revive its thread.
            if (thread.getStatus() === enums_1.ThreadStatus.PARKED) {
                thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
                cb();
            }
        }
    };
    Parker.prototype.isParked = function (thread) {
        return !!this._parkCounts[thread.getRef()];
    };
    return Parker;
}());
exports.__esModule = true;
exports["default"] = Parker;
//# sourceMappingURL=data:application/json;base64,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