"use strict";
var jvm_1 = require('./jvm');
var util_1 = require('./util');
var difflib_1 = require('./difflib');
var path = require('path');
var fs = require('fs');
function makeTestingError(msg, origErr, fatal) {
    var err = new Error(msg);
    err.originalError = origErr;
    err.fatal = fatal;
    return err;
}
/**
 * Captures stdout/stderr.
 * @todo Do this the proper Node way once BFS is more compliant.
 */
var OutputCapturer = (function () {
    function OutputCapturer() {
        this._stdoutWrite = process.stdout.write;
        this._stderrWrite = process.stderr.write;
        this._data = "";
        this._isCapturing = false;
    }
    OutputCapturer.prototype.debugWrite = function (str) {
        this._stdoutWrite.apply(process.stdout, [str, 'utf8']);
    };
    /**
     * Begin capturing output.
     */
    OutputCapturer.prototype.start = function (clear) {
        var _this = this;
        if (this._isCapturing) {
            throw new Error("Already capturing.");
        }
        this._isCapturing = true;
        if (clear) {
            this._data = "";
        }
        process.stderr.write = process.stdout.write = function (data, arg2, arg3) {
            if (typeof (data) !== 'string') {
                // Buffer.
                data = data.toString();
            }
            _this._data += data;
            return true;
        };
    };
    /**
     * Stop capturing output.
     */
    OutputCapturer.prototype.stop = function () {
        if (!this._isCapturing) {
            // May be called twice when there's a catastrophic error.
            return;
        }
        this._isCapturing = false;
        process.stderr.write = this._stderrWrite;
        process.stdout.write = this._stdoutWrite;
    };
    /**
     * Retrieve the captured output.
     * @param clear Clear the captured output.
     */
    OutputCapturer.prototype.getOutput = function (clear) {
        var data = this._data;
        if (clear) {
            this._data = "";
        }
        return data;
    };
    return OutputCapturer;
}());
/**
 * Represents a single unit test, where we compare Doppio's output to the native
 * JVM.
 */
var DoppioTest = (function () {
    function DoppioTest(opts, cls) {
        /**
         * The output capturer for this test.
         */
        this.outputCapturer = new OutputCapturer();
        this.opts = opts;
        if (cls.indexOf('.') !== -1) {
            // Convert foo.bar.Baz => foo/bar/Baz
            cls = util_1.descriptor2typestr(util_1.int_classname(cls));
        }
        this.cls = cls;
        this.outFile = path.resolve(opts.doppioHomePath, cls) + ".runout";
    }
    /**
     * Constructs a new JVM for the test.
     */
    DoppioTest.prototype.constructJVM = function (cb) {
        new jvm_1["default"](util_1.merge(jvm_1["default"].getDefaultOptions(this.opts.doppioHomePath), this.opts, {
            classpath: [this.opts.doppioHomePath],
            enableAssertions: true,
            enableSystemAssertions: true
        }), cb);
    };
    /**
     * Runs the unit test.
     */
    DoppioTest.prototype.run = function (registerGlobalErrorTrap, cb) {
        var _this = this;
        var outputCapturer = this.outputCapturer, _jvm = null, terminated = false, jvmConstructHasFinished = false, hasFinished = false;
        registerGlobalErrorTrap(function (err) {
            if (_jvm) {
                try {
                    _jvm.halt(1);
                }
                catch (e) {
                    err.message += "\n\nAdditionally, test runner received the following error while trying to halt the JVM: " + e + (e.stack ? "\n\n" + e.stack : '') + "\n\nOriginal error's stack trace:";
                }
            }
            outputCapturer.stop();
            cb(makeTestingError("Uncaught error. Aborting further tests.\n\t" + err + (err.stack ? "\n\n" + err.stack : ""), err, true));
        });
        this.constructJVM(function (err, jvm) {
            _jvm = jvm;
            if (terminated) {
                // Already handled.
                return;
            }
            if (jvmConstructHasFinished) {
                return cb(makeTestingError("constructJVM returned twice. Aborting further tests.", null, true));
            }
            jvmConstructHasFinished = true;
            if (err) {
                cb(makeTestingError("Could not construct JVM:\n" + err, err));
            }
            else {
                outputCapturer.start(true);
                jvm.runClass(_this.cls, [], function (status) {
                    if (terminated) {
                        // Already handled.
                        return;
                    }
                    outputCapturer.stop();
                    if (hasFinished) {
                        return cb(makeTestingError("JVM triggered completion callback twice. Aborting further tests.", null, true));
                    }
                    hasFinished = true;
                    var actual = outputCapturer.getOutput(true);
                    fs.readFile(_this.outFile, { encoding: 'utf8' }, function (err, expected) {
                        if (err) {
                            cb(makeTestingError("Could not read runout file:\n" + err, err));
                        }
                        else {
                            var diffText = diff(actual, expected), errMsg = null;
                            if (diffText !== null) {
                                errMsg = "Output does not match native JVM.";
                            }
                            cb(errMsg ? makeTestingError(errMsg) : null, actual, expected, diffText);
                        }
                    });
                });
            }
        });
    };
    return DoppioTest;
}());
exports.DoppioTest = DoppioTest;
/**
 * Locate all of Doppio's test classes, and pass them to the callback.
 */
function findTestClasses(doppioDir, cb) {
    var testDir = path.resolve(doppioDir, path.join('classes', 'test'));
    fs.readdir(testDir, function (err, files) {
        if (err) {
            cb([]);
        }
        else {
            cb(files.filter(function (file) { return path.extname(file) === '.java'; })
                .map(function (file) { return path.join('classes', 'test', path.basename(file, '.java')); }));
        }
    });
}
/**
 * Retrieve all of the unit tests.
 */
function getTests(opts, cb) {
    var testClasses = opts.testClasses, tests;
    if (testClasses == null || testClasses.length === 0) {
        // If no test classes are specified, get ALL the tests!
        findTestClasses(opts.doppioHomePath, function (testClasses) {
            opts.testClasses = testClasses;
            getTests(opts, cb);
        });
    }
    else {
        cb(testClasses.map(function (testClass) {
            return new DoppioTest(opts, testClass);
        }));
    }
}
exports.getTests = getTests;
/**
 * Returns a formatted diff between doppioOut and nativeOut.
 * Returns NULL if the strings are identical.
 */
function diff(doppioOut, nativeOut) {
    // @todo Robust to Windows line breaks!
    var doppioLines = doppioOut.split(/\n/), jvmLines = nativeOut.split(/\n/), diff = difflib_1.text_diff(doppioLines, jvmLines, 2);
    if (diff.length > 0) {
        return 'Doppio | Java\n' + diff.join('\n');
    }
    return null;
}
exports.diff = diff;
/**
 * Run the specified tests.
 */
function runTests(opts, quiet, continueAfterFailure, hideDiffs, registerGlobalErrorTrap, cb) {
    function print(str) {
        if (!quiet) {
            process.stdout.write(str);
        }
    }
    getTests(opts, function (tests) {
        util_1.asyncForEach(tests, function (test, nextTest) {
            var hasFinished = false;
            print("[" + test.cls + "]: Running... ");
            test.run(registerGlobalErrorTrap, function (err, actual, expected, diff) {
                if (err && !hideDiffs && diff) {
                    err.message += "\n" + diff;
                }
                if (err) {
                    print("fail.\n\t" + err.message + "\n");
                    if (err.originalError && err.originalError.stack) {
                        print(err.stack + "\n");
                    }
                    if (!continueAfterFailure || err['fatal']) {
                        err.message = "Failed " + test.cls + ": " + err.message;
                        nextTest(err);
                    }
                    else {
                        nextTest();
                    }
                }
                else {
                    print("pass.\n");
                    nextTest();
                }
            });
        }, cb);
    });
}
exports.runTests = runTests;
//# sourceMappingURL=data:application/json;base64,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