"use strict";
var enums_1 = require('./enums');
var assert_1 = require('./assert');
var gLong_1 = require('./gLong');
var opcodes_1 = require('./opcodes');
var logging_1 = require('./logging');
var logging = require('./logging');
var util_1 = require('./util');
var debug = logging.debug;
var vtrace = logging.vtrace;
var trace = logging.trace;
// The number of method resumes we should allow before yielding for
// responsiveness. Updated using a cumulative moving average to ensure
// Doppio is responsive.
var maxMethodResumes = 10000, 
// The number of method resumes until Doppio should yield again.
methodResumesLeft = maxMethodResumes, 
// Used for the CMA.
numSamples = 1;
var PreAllocatedStack = (function () {
    function PreAllocatedStack(initialSize) {
        this.curr = 0;
        this.store = new Array(initialSize);
    }
    PreAllocatedStack.prototype.push = function (x) {
        this.store[this.curr++] = x;
    };
    PreAllocatedStack.prototype.pushAll = function () {
        var n = arguments.length;
        for (var i = 0; i < n; i++) {
            this.store[this.curr++] = arguments[i];
        }
    };
    PreAllocatedStack.prototype.pushWithNull = function (x) {
        this.store[this.curr] = x;
        // XXX: Although setting `null` is required in theory, it has no functional impact in practice.
        // Performance is improved when commented.
        // this.store[this.curr + 1] = null;
        this.curr += 2;
    };
    PreAllocatedStack.prototype.push6 = function (x, y, z, z1, z2, z3) {
        this.store[this.curr++] = x;
        this.store[this.curr++] = y;
        this.store[this.curr++] = z;
        this.store[this.curr++] = z1;
        this.store[this.curr++] = z2;
        this.store[this.curr++] = z3;
    };
    PreAllocatedStack.prototype.swap = function () {
        var tmp = this.store[this.curr - 1];
        this.store[this.curr - 1] = this.store[this.curr - 2];
        this.store[this.curr - 2] = tmp;
    };
    PreAllocatedStack.prototype.dup = function () {
        this.store[this.curr] = this.store[this.curr - 1];
        this.curr++;
    };
    PreAllocatedStack.prototype.dup2 = function () {
        this.store[this.curr] = this.store[this.curr - 2];
        this.store[this.curr + 1] = this.store[this.curr - 1];
        this.curr += 2;
    };
    PreAllocatedStack.prototype.dup_x1 = function () {
        var v1 = this.store[this.curr - 1];
        this.store[this.curr - 1] = this.store[this.curr - 2];
        this.store[this.curr] = v1;
        this.store[this.curr - 2] = v1;
        this.curr++;
    };
    PreAllocatedStack.prototype.dup_x2 = function () {
        var v1 = this.store[this.curr - 1];
        this.store[this.curr - 1] = this.store[this.curr - 2];
        this.store[this.curr - 2] = this.store[this.curr - 3];
        this.store[this.curr] = v1;
        this.store[this.curr - 3] = v1;
        this.curr++;
    };
    PreAllocatedStack.prototype.dup2_x1 = function () {
        var v1 = this.store[this.curr - 1];
        var v2 = this.store[this.curr - 2];
        this.store[this.curr] = v2;
        this.store[this.curr + 1] = v1;
        this.store[this.curr - 1] = this.store[this.curr - 3];
        this.store[this.curr - 2] = v1;
        this.store[this.curr - 3] = v2;
        this.curr += 2;
    };
    PreAllocatedStack.prototype.pop = function () {
        return this.store[--this.curr];
    };
    PreAllocatedStack.prototype.pop2 = function () {
        this.curr -= 2;
        return this.store[this.curr];
    };
    PreAllocatedStack.prototype.bottom = function () {
        return this.store[0];
    };
    PreAllocatedStack.prototype.top = function () {
        return this.store[this.curr - 1];
    };
    PreAllocatedStack.prototype.fromTop = function (n) {
        return this.store[this.curr - (n + 1)];
    };
    PreAllocatedStack.prototype.sliceFromBottom = function (n) {
        return this.store.slice(n, this.curr);
    };
    PreAllocatedStack.prototype.sliceFromTop = function (n) {
        return this.store.slice(this.curr - n, this.curr);
    };
    PreAllocatedStack.prototype.dropFromTop = function (n) {
        this.curr -= n;
    };
    PreAllocatedStack.prototype.sliceAndDropFromTop = function (n) {
        var curr = this.curr;
        this.curr -= n;
        return this.store.slice(curr - n, curr);
    };
    PreAllocatedStack.prototype.getRaw = function () {
        return this.store.slice(0, this.curr);
    };
    PreAllocatedStack.prototype.clear = function () {
        this.curr = 0;
    };
    return PreAllocatedStack;
}());
exports.PreAllocatedStack = PreAllocatedStack;
var jitUtil = {
    isNull: opcodes_1.isNull,
    resolveCPItem: opcodes_1.resolveCPItem,
    throwException: opcodes_1.throwException,
    gLong: gLong_1["default"],
    float2int: util_1.float2int,
    wrapFloat: util_1.wrapFloat,
    Constants: enums_1.Constants
};
/**
 * Represents a stack frame for a bytecode method.
 */
var BytecodeStackFrame = (function () {
    /**
     * Constructs a bytecode method's stack frame.
     * @param method The bytecode method to run.
     * @param args The arguments to pass to the bytecode method.
     */
    function BytecodeStackFrame(method, args) {
        this.pc = 0;
        this.returnToThreadLoop = false;
        this.lockedMethodLock = false;
        /**
         * Indicates the type of this stack frame.
         */
        this.type = enums_1.StackFrameType.BYTECODE;
        this.method = method;
        method.incrBBEntries();
        assert_1["default"](!method.accessFlags.isNative(), 'Cannot run a native method using a BytecodeStackFrame.');
        // @todo This should be a runtime error, since reflection can cause you to
        // try to do this.
        assert_1["default"](!method.accessFlags.isAbstract(), 'Cannot run an abstract method!');
        this.locals = args;
        this.opStack = new PreAllocatedStack(method.getCodeAttribute().getMaxStack());
    }
    BytecodeStackFrame.prototype.run = function (thread) {
        var _this = this;
        var method = this.method, code = this.method.getCodeAttribute().getCode(), opcodeTable = opcodes_1.LookupTable;
        if (!RELEASE && logging_1.logLevel >= logging_1.LogLevel.TRACE) {
            if (this.pc === 0) {
                trace("\nT" + thread.getRef() + " D" + thread.getStackTrace().length + " Running " + this.method.getFullSignature() + " [Bytecode]:");
            }
            else {
                trace("\nT" + thread.getRef() + " D" + thread.getStackTrace().length + " Resuming " + this.method.getFullSignature() + ":" + this.pc + " [Bytecode]:");
            }
            vtrace("  S: [" + logging_1.debug_vars(this.opStack.getRaw()) + "], L: [" + logging_1.debug_vars(this.locals) + "]");
        }
        if (method.accessFlags.isSynchronized() && !this.lockedMethodLock) {
            // We are starting a synchronized method! These must implicitly enter
            // their respective locks.
            this.lockedMethodLock = method.methodLock(thread, this).enter(thread, function () {
                // Lock succeeded. Set the flag so we don't attempt to reacquire it
                // when this method reruns.
                _this.lockedMethodLock = true;
            });
            if (!this.lockedMethodLock) {
                // Failed. Thread is automatically blocked. Return.
                assert_1["default"](thread.getStatus() === enums_1.ThreadStatus.BLOCKED, "Failed to enter a monitor. Thread must be BLOCKED.");
                return;
            }
        }
        // Reset the returnToThreadLoop switch. The current value is leftover
        // from the previous time this method was run, and is meaningless.
        this.returnToThreadLoop = false;
        if (thread.getJVM().isJITDisabled()) {
            // Interpret until we get the signal to return to the thread loop.
            while (!this.returnToThreadLoop) {
                var opCode = code[this.pc];
                if (!RELEASE && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                    vtrace("  " + this.pc + " " + annotateOpcode(op, method, code, this.pc));
                }
                opcodeTable[opCode](thread, this, code);
                if (!RELEASE && !this.returnToThreadLoop && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                    vtrace("    S: [" + logging_1.debug_vars(this.opStack.getRaw()) + "], L: [" + logging_1.debug_vars(this.locals) + "]");
                }
            }
        }
        else {
            // Run until we get the signal to return to the thread loop.
            while (!this.returnToThreadLoop) {
                var op = method.getOp(this.pc, code, thread);
                if (typeof op === 'function') {
                    if (!RELEASE && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                        vtrace("  " + this.pc + " running JIT compiled function:\n" + op.toString());
                    }
                    op(this, thread, jitUtil);
                }
                else {
                    if (!RELEASE && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                        vtrace("  " + this.pc + " " + annotateOpcode(op, method, code, this.pc));
                    }
                    opcodeTable[op](thread, this, code);
                }
                if (!RELEASE && !this.returnToThreadLoop && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                    vtrace("    S: [" + logging_1.debug_vars(this.opStack.getRaw()) + "], L: [" + logging_1.debug_vars(this.locals) + "]");
                }
            }
        }
    };
    BytecodeStackFrame.prototype.scheduleResume = function (thread, rv, rv2) {
        // Advance to the next opcode.
        var prevOp = this.method.getCodeAttribute().getCode()[this.pc];
        switch (prevOp) {
            case enums_1.OpCode.INVOKEINTERFACE:
            case enums_1.OpCode.INVOKEINTERFACE_FAST:
                this.pc += 5;
                break;
            case enums_1.OpCode.INVOKESPECIAL:
            case enums_1.OpCode.INVOKESTATIC:
            case enums_1.OpCode.INVOKEVIRTUAL:
            case enums_1.OpCode.INVOKESTATIC_FAST:
            case enums_1.OpCode.INVOKENONVIRTUAL_FAST:
            case enums_1.OpCode.INVOKEVIRTUAL_FAST:
            case enums_1.OpCode.INVOKEHANDLE:
            case enums_1.OpCode.INVOKEBASIC:
            case enums_1.OpCode.LINKTOSPECIAL:
            case enums_1.OpCode.LINKTOVIRTUAL:
            case enums_1.OpCode.INVOKEDYNAMIC:
            case enums_1.OpCode.INVOKEDYNAMIC_FAST:
                this.pc += 3;
                break;
            default:
                // Should be impossible.
                assert_1["default"](false, "Resuming from a non-invoke opcode! Opcode: " + enums_1.OpCode[prevOp] + " [" + prevOp + "]");
                break;
        }
        if (rv !== undefined) {
            this.opStack.push(rv);
        }
        if (rv2 !== undefined) {
            this.opStack.push(rv2);
        }
    };
    /**
     * Checks if this method can handle the specified exception 'e'.
     * Returns true if it can, or if it needs to asynchronously resolve some
     * classes.
     *
     * In the latter case, scheduleException will handle rethrowing the exception
     * in the event that it can't actually handle it.
     */
    BytecodeStackFrame.prototype.scheduleException = function (thread, e) {
        var codeAttr = this.method.getCodeAttribute(), pc = this.pc, method = this.method, 
        // STEP 1: See if we can find an appropriate handler for this exception!
        exceptionHandlers = codeAttr.exceptionHandlers, ecls = e.getClass(), handler;
        for (var i = 0; i < exceptionHandlers.length; i++) {
            var eh = exceptionHandlers[i];
            if (eh.startPC <= pc && pc < eh.endPC) {
                if (eh.catchType === "<any>") {
                    handler = eh;
                    break;
                }
                else {
                    var resolvedCatchType = method.cls.getLoader().getResolvedClass(eh.catchType);
                    if (resolvedCatchType != null) {
                        if (ecls.isCastable(resolvedCatchType)) {
                            handler = eh;
                            break;
                        }
                    }
                    else {
                        // ASYNC PATH: We'll need to asynchronously resolve these handlers.
                        debug(method.getFullSignature() + " needs to resolve some exception types...");
                        var handlerClasses = [];
                        for (var i_1 = 0; i_1 < exceptionHandlers.length; i_1++) {
                            var handler_1 = exceptionHandlers[i_1];
                            if (handler_1.catchType !== "<any>") {
                                handlerClasses.push(handler_1.catchType);
                            }
                        }
                        debug(method.getFullSignature() + ": Has to resolve exception classes. Deferring scheduling...");
                        thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
                        method.cls.getLoader().resolveClasses(thread, handlerClasses, function (classes) {
                            if (classes !== null) {
                                debug(method.getFullSignature() + ": Rethrowing exception to handle!");
                                // Rethrow the exception to trigger scheduleException again.
                                // @todo If the ClassLoader throws an exception during resolution,
                                // this could result in an infinite loop. Fix would be to sync check
                                // if class failed to load previously.
                                thread.throwException(e);
                            }
                        });
                        // Tell the thread we'll handle it.
                        return true;
                    }
                }
            }
        }
        // STEP 2: Either continue on if we could not find an appropriate handler,
        // or set up the stack for appropriate resumption.
        if (handler != null) {
            // Found the handler.
            debug(method.getFullSignature() + ": Caught " + e.getClass().getInternalName() + " as subclass of " + handler.catchType);
            // clear out anything on the stack; it was made during the try block
            this.opStack.clear();
            this.opStack.push(e);
            this.pc = handler.handlerPC;
            return true;
        }
        else {
            // abrupt method invocation completion
            debug(method.getFullSignature() + ": Did not catch " + e.getClass().getInternalName() + ".");
            // STEP 3: Synchronized method? Exit from the method's monitor.
            if (method.accessFlags.isSynchronized()) {
                method.methodLock(thread, this).exit(thread);
            }
            return false;
        }
    };
    /**
     * Returns the classloader for the stack frame.
     */
    BytecodeStackFrame.prototype.getLoader = function () {
        return this.method.cls.getLoader();
    };
    BytecodeStackFrame.prototype.getStackTraceFrame = function () {
        return {
            method: this.method,
            pc: this.pc,
            stack: this.opStack.sliceFromBottom(0),
            locals: this.locals.slice(0)
        };
    };
    return BytecodeStackFrame;
}());
exports.BytecodeStackFrame = BytecodeStackFrame;
/**
 * Represents a native method's stack frame.
 */
var NativeStackFrame = (function () {
    /**
     * Constructs a native method's stack frame.
     * @param method The native method to run.
     * @param args The arguments to pass to the native method.
     */
    function NativeStackFrame(method, args) {
        this.type = enums_1.StackFrameType.NATIVE;
        this.method = method;
        this.args = args;
        assert_1["default"](method.accessFlags.isNative());
        this.nativeMethod = method.getNativeFunction();
    }
    /**
     * Calls the native method.
     * NOTE: Should only be called once.
     */
    NativeStackFrame.prototype.run = function (thread) {
        trace("\nT" + thread.getRef() + " D" + thread.getStackTrace().length + " Running " + this.method.getFullSignature() + " [Native]:");
        var rv = this.nativeMethod.apply(null, this.method.convertArgs(thread, this.args));
        // Ensure thread is running, and we are the running method.
        if (thread.getStatus() === enums_1.ThreadStatus.RUNNABLE && thread.currentMethod() === this.method) {
            // Normal native method exit.
            var returnType = this.method.returnType;
            switch (returnType) {
                case 'J':
                case 'D':
                    // Two stack return values for methods that return a long or a double.
                    thread.asyncReturn(rv, null);
                    break;
                case 'Z':
                    // Convert to a number.
                    thread.asyncReturn(rv ? 1 : 0);
                    break;
                default:
                    thread.asyncReturn(rv);
                    break;
            }
        }
    };
    /**
     * N/A
     */
    NativeStackFrame.prototype.scheduleResume = function (thread, rv, rv2) {
        // NOP
    };
    /**
     * Not relevant; the first execution block of a native method will never
     * receive an exception.
     */
    NativeStackFrame.prototype.scheduleException = function (thread, e) {
        return false;
    };
    NativeStackFrame.prototype.getStackTraceFrame = function () {
        return {
            method: this.method,
            pc: -1,
            stack: [],
            locals: []
        };
    };
    /**
     * Returns the classloader for the stack frame.
     */
    NativeStackFrame.prototype.getLoader = function () {
        return this.method.cls.getLoader();
    };
    return NativeStackFrame;
}());
exports.NativeStackFrame = NativeStackFrame;
/**
 * InternalStackFrames are used by the JVM to launch JVM functions that
 * eventually call back into JavaScript code when they complete or throw a
 * fatal exception.
 */
var InternalStackFrame = (function () {
    /**
     * @param cb Callback function. Called with an exception if one occurs, or
     *   the return value from the called method, if relevant.
     */
    function InternalStackFrame(cb) {
        this.isException = false;
        this.type = enums_1.StackFrameType.INTERNAL;
        this.cb = cb;
    }
    InternalStackFrame.prototype.run = function (thread) {
        // Pop myself off of the stack.
        thread.framePop();
        // Pause the thread before returning into native JavaScript code.
        thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
        if (this.isException) {
            this.cb(this.val);
        }
        else {
            this.cb(null, this.val);
        }
    };
    /**
     * Resumes the JavaScript code that created this stack frame.
     */
    InternalStackFrame.prototype.scheduleResume = function (thread, rv) {
        this.isException = false;
        this.val = rv;
    };
    /**
     * Resumes the JavaScript code that created this stack frame with the given
     * exception.
     */
    InternalStackFrame.prototype.scheduleException = function (thread, e) {
        this.isException = true;
        this.val = e;
        return true;
    };
    InternalStackFrame.prototype.getStackTraceFrame = function () {
        // These should not be language visible.
        return null;
    };
    InternalStackFrame.prototype.getLoader = function () {
        throw new Error("Internal stack frames have no loader.");
    };
    return InternalStackFrame;
}());
exports.InternalStackFrame = InternalStackFrame;
/**
 * Represents a single JVM thread.
 */
var JVMThread = (function () {
    /**
     * Initializes a new JVM thread. Starts the thread in the NEW state.
     */
    function JVMThread(jvm, tpool, threadObj) {
        /**
         * The current state of this thread, from the JVM level.
         */
        this.status = enums_1.ThreadStatus.NEW;
        /**
         * The call stack.
         */
        this.stack = [];
        /**
         * Whether or not this thread has been interrupted. It's a JVM thing.
         */
        this.interrupted = false;
        /**
         * If the thread is WAITING, BLOCKED, or TIMED_WAITING, this field holds the
         * monitor that is involved.
         */
        this.monitor = null;
        this.jvm = jvm;
        this.bsCl = jvm.getBootstrapClassLoader();
        this.tpool = tpool;
        this.jvmThreadObj = threadObj;
    }
    /**
     * Get the JVM thread object that represents this thread.
     */
    JVMThread.prototype.getJVMObject = function () {
        return this.jvmThreadObj;
    };
    /**
     * Is this thread a daemon?
     */
    JVMThread.prototype.isDaemon = function () {
        return this.jvmThreadObj['java/lang/Thread/daemon'] !== 0;
    };
    /**
     * Get the priority of this thread.
     */
    JVMThread.prototype.getPriority = function () {
        return this.jvmThreadObj['java/lang/Thread/priority'];
    };
    /**
     * XXX: Used during bootstrapping to set the first thread's Thread object.
     */
    JVMThread.prototype.setJVMObject = function (obj) {
        obj['java/lang/Thread/threadStatus'] = this.jvmThreadObj['java/lang/Thread/threadStatus'];
        this.jvmThreadObj = obj;
    };
    /**
     * Return the reference number for this thread.
     */
    JVMThread.prototype.getRef = function () {
        return this.jvmThreadObj.ref;
    };
    /**
     * Check if this thread's interrupted flag is set.
     */
    JVMThread.prototype.isInterrupted = function () {
        return this.interrupted;
    };
    /**
     * Returns the currently running method. Returns NULL if stack is empty.
     */
    JVMThread.prototype.currentMethod = function () {
        var stack = this.stack, idx = stack.length, method;
        while (--idx >= 0) {
            method = stack[idx].getStackTraceFrame().method;
            if (method !== null) {
                return method;
            }
        }
        return null;
    };
    /**
     * Set or unset this thread's interrupted flag.
     */
    JVMThread.prototype.setInterrupted = function (interrupted) {
        this.interrupted = interrupted;
    };
    /**
     * Retrieve the bootstrap classloader.
     */
    JVMThread.prototype.getBsCl = function () {
        return this.bsCl;
    };
    /**
     * Get the classloader for the current frame.
     */
    JVMThread.prototype.getLoader = function () {
        var loader = this.stack[this.stack.length - 1].getLoader();
        if (loader) {
            return loader;
        }
        else {
            // Crawl stack until we find one.
            var len = this.stack.length;
            for (var i = 2; i <= len; i++) {
                loader = this.stack[len - i].getLoader();
                if (loader) {
                    return loader;
                }
            }
            throw new Error("Unable to find loader.");
        }
    };
    JVMThread.prototype.import = function (names, cb, explicit) {
        var _this = this;
        if (explicit === void 0) { explicit = true; }
        var loader = this.getLoader();
        this.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
        if (Array.isArray(names)) {
            var rv_1 = [];
            util_1.asyncForEach(names, function (name, nextItem) {
                _this._import(name, loader, function (cons) {
                    rv_1.push(cons);
                    nextItem();
                }, explicit);
            }, function (e) {
                cb(rv_1);
            });
        }
        else {
            this._import(names, loader, cb, explicit);
        }
    };
    JVMThread.prototype._import = function (name, loader, cb, explicit) {
        var _this = this;
        var cls = loader.getInitializedClass(this, name);
        if (cls) {
            setImmediate(function () { return cb(cls.getConstructor(_this)); });
        }
        else {
            loader.initializeClass(this, name, function (cdata) {
                if (cdata) {
                    cb(cdata.getConstructor(_this));
                }
            }, explicit);
        }
    };
    /**
     * Retrieve the JVM instantiation that this thread belongs to.
     */
    JVMThread.prototype.getJVM = function () {
        return this.jvm;
    };
    /**
     * Retrieve the thread pool that this thread belongs to.
     */
    JVMThread.prototype.getThreadPool = function () {
        return this.tpool;
    };
    /**
     * Retrieves the current stack trace.
     */
    JVMThread.prototype.getStackTrace = function () {
        var trace = [], i, frame;
        for (i = 0; i < this.stack.length; i++) {
            frame = this.stack[i].getStackTraceFrame();
            if (frame != null) {
                trace.push(frame);
            }
        }
        return trace;
    };
    /**
     * [DEBUG] Return a printable string of the thread's current stack trace.
     */
    JVMThread.prototype.getPrintableStackTrace = function () {
        var rv = "";
        this.getStackTrace().reverse().forEach(function (trace) {
            rv += "\tat " + util_1.ext_classname(trace.method.cls.getInternalName()) + "::" + trace.method.name + "(";
            if (trace.pc >= 0) {
                // Bytecode method
                var code = trace.method.getCodeAttribute();
                var table = code.getAttribute('LineNumberTable');
                var srcAttr = trace.method.cls.getAttribute('SourceFile');
                if (srcAttr != null) {
                    rv += srcAttr.filename;
                }
                else {
                    rv += 'unknown';
                }
                if (table != null) {
                    var lineNumber = table.getLineNumber(trace.pc);
                    rv += ":" + lineNumber;
                    rv += " Bytecode offset: " + trace.pc;
                }
            }
            else {
                // Native method.
                rv += "native";
            }
            rv += ")\n";
        });
        return rv;
    };
    /**
     * The thread's main execution loop. Everything starts here!
     *
     * SHOULD ONLY BE INVOKED BY THE SCHEDULER.
     */
    JVMThread.prototype.run = function () {
        var stack = this.stack, startTime = (new Date()).getTime();
        // Reset counter. Threads always start from a fresh stack / yield.
        methodResumesLeft = maxMethodResumes;
        while (this.status === enums_1.ThreadStatus.RUNNABLE && stack.length > 0) {
            var sf = stack[stack.length - 1];
            if (!RELEASE) {
                if (sf.type === enums_1.StackFrameType.BYTECODE && this.jvm.shouldVtrace(sf.method.fullSignature)) {
                    var oldLevel = logging_1.logLevel;
                    logging_1.setLogLevel(logging_1.LogLevel.VTRACE);
                    sf.run(this);
                    logging_1.setLogLevel(oldLevel);
                }
                else {
                    sf.run(this);
                }
            }
            else {
                sf.run(this);
            }
            if (--methodResumesLeft === 0) {
                var endTime = (new Date()).getTime();
                var duration = endTime - startTime;
                // Estimated number of methods we can resume before needing to yield.
                var estMaxMethodResumes = ((maxMethodResumes / duration) * this.jvm.getResponsiveness()) | 0;
                // Update CMA.
                maxMethodResumes = ((estMaxMethodResumes + numSamples * maxMethodResumes) / (numSamples + 1)) | 0;
                if (maxMethodResumes <= 0) {
                    // Sanity check. Should never really occur.
                    maxMethodResumes = 10;
                }
                vtrace("T" + this.getRef() + " Quantum over. Method resumes: Max " + maxMethodResumes + " Est " + estMaxMethodResumes + " Samples " + numSamples);
                numSamples++;
                // Tell the scheduler that our quantum is over.
                this.tpool.quantumOver(this);
                // Break out of while loop.
                break;
            }
        }
        if (stack.length === 0) {
            // This thread has finished!
            this.setStatus(enums_1.ThreadStatus.TERMINATED);
        }
    };
    /**
     * [DEBUG] Performs a sanity check on the thread.
     */
    JVMThread.prototype.sanityCheck = function () {
        switch (this.status) {
            case enums_1.ThreadStatus.NEW:
                return true;
            case enums_1.ThreadStatus.RUNNABLE:
                assert_1["default"](this.stack.length > 0, 'A runnable thread must not have an empty stack.');
                return true;
            case enums_1.ThreadStatus.TIMED_WAITING:
                assert_1["default"](this.monitor != null && this.monitor.isTimedWaiting(this), 'A timed waiting thread must be waiting on a monitor.');
                return true;
            case enums_1.ThreadStatus.WAITING:
                assert_1["default"](this.monitor != null && this.monitor.isWaiting(this), "A waiting thread must be waiting on a monitor.");
                return true;
            case enums_1.ThreadStatus.BLOCKED:
            case enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED:
                assert_1["default"](this.monitor != null && this.monitor.isBlocked(this), "A blocked thread must be blocked on a monitor");
                return true;
            case enums_1.ThreadStatus.ASYNC_WAITING:
                return true;
            case enums_1.ThreadStatus.TERMINATED:
                assert_1["default"](this.stack.length === 0, "A terminated thread must have an empty stack.");
                return true;
            case enums_1.ThreadStatus.PARKED:
                assert_1["default"](this.jvm.getParker().isParked(this), "A parked thread must be parked.");
                return true;
            default:
                // Invalid ThreadStatus.
                return false;
        }
    };
    /**
     * Should only be called by setStatus.
     * Updates both the JVMThread object and this object.
     */
    JVMThread.prototype.rawSetStatus = function (newStatus) {
        var jvmNewStatus = 0, oldStatus = this.status;
        if (logging_1.logLevel === logging_1.LogLevel.VTRACE) {
            vtrace("\nT" + this.getRef() + " " + enums_1.ThreadStatus[oldStatus] + " => " + enums_1.ThreadStatus[newStatus]);
        }
        assert_1["default"](validateThreadTransition(oldStatus, newStatus), "Invalid thread transition: " + enums_1.ThreadStatus[oldStatus] + " => " + enums_1.ThreadStatus[newStatus]);
        this.status = newStatus;
        // Map our status value back to JVM's threadStatus value.
        // Ensures that JVM code can introspect on our threads.
        switch (newStatus) {
            case enums_1.ThreadStatus.NEW:
                jvmNewStatus |= enums_1.JVMTIThreadState.ALIVE;
                break;
            case enums_1.ThreadStatus.RUNNABLE:
                jvmNewStatus |= enums_1.JVMTIThreadState.RUNNABLE;
                break;
            case enums_1.ThreadStatus.BLOCKED:
            case enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED:
                jvmNewStatus |= enums_1.JVMTIThreadState.BLOCKED_ON_MONITOR_ENTER;
                break;
            case enums_1.ThreadStatus.WAITING:
            case enums_1.ThreadStatus.ASYNC_WAITING:
            case enums_1.ThreadStatus.PARKED:
                jvmNewStatus |= enums_1.JVMTIThreadState.WAITING_INDEFINITELY;
                break;
            case enums_1.ThreadStatus.TIMED_WAITING:
                jvmNewStatus |= enums_1.JVMTIThreadState.WAITING_WITH_TIMEOUT;
                break;
            case enums_1.ThreadStatus.TERMINATED:
                jvmNewStatus |= enums_1.JVMTIThreadState.TERMINATED;
                break;
            default:
                jvmNewStatus = enums_1.JVMTIThreadState.RUNNABLE;
                break;
        }
        this.jvmThreadObj['java/lang/Thread/threadStatus'] = jvmNewStatus;
        this.tpool.statusChange(this, oldStatus, this.status);
    };
    /**
     * Transitions the thread from one state to the next.
     * Contains JVM-specific thread logic.
     */
    JVMThread.prototype.setStatus = function (status, monitor) {
        if (monitor === void 0) { monitor = null; }
        if (this.status !== status) {
            var oldStatus = this.status;
            // Update the monitor.
            this.monitor = monitor;
            if (status !== enums_1.ThreadStatus.TERMINATED) {
                // Actually change state.
                this.rawSetStatus(status);
            }
            else {
                // Call exit() first.
                this.exit();
            }
            // Validate current state (debug builds only)
            assert_1["default"](this.sanityCheck(), "Invalid thread status.");
        }
    };
    /**
     * Called when a thread finishes executing.
     */
    JVMThread.prototype.exit = function () {
        var _this = this;
        var monitor = this.jvmThreadObj.getMonitor();
        if (monitor.isBlocked(this) || monitor.getOwner() === this || this.status === enums_1.ThreadStatus.TERMINATED) {
            // Thread is already shutting down.
            return;
        }
        if (this.stack.length === 0) {
            // De-schedule thread.
            this.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
            // Only applicable if it's not an early death, e.g. before VM bootup.
            if (this.jvm.hasVMBooted()) {
                trace("T" + this.getRef() + " Exiting.");
                var phase2 = function () {
                    trace("T" + _this.getRef() + " Entered exit monitor.");
                    // Exit.
                    _this.jvmThreadObj["exit()V"](_this, null, function (e) {
                        // Notify everyone.
                        monitor.notifyAll(_this);
                        // Exit monitor.
                        monitor.exit(_this);
                        trace("T" + _this.getRef() + " Terminated.");
                        // Actually become terminated.
                        _this.rawSetStatus(enums_1.ThreadStatus.TERMINATED);
                    });
                };
                // Acquire the monitor associated with our JavaObject.
                if (monitor.enter(this, phase2)) {
                    phase2();
                }
            }
            else {
                trace("T" + this.getRef() + " Not exiting; VM is still booting.");
            }
        }
        else {
            // There are things on the stack. This exit is occuring before the stack has emptied.
            // Clear the stack, set to terminated.
            while (this.stack.length > 0) {
                this.stack.pop();
            }
            trace("T" + this.getRef() + " Terminated.");
            this.rawSetStatus(enums_1.ThreadStatus.TERMINATED);
        }
    };
    /**
     * Called when the priority of the thread changes.
     * Should only be called by java.lang.setPriority0.
     */
    JVMThread.prototype.signalPriorityChange = function () {
        this.tpool.priorityChange(this);
    };
    /**
     * Get the monitor that this thread is waiting or blocked on.
     */
    JVMThread.prototype.getMonitorBlock = function () {
        return this.monitor;
    };
    /**
     * Get the thread's current state.
     */
    JVMThread.prototype.getStatus = function () {
        return this.status;
    };
    JVMThread.prototype.asyncReturn = function (rv, rv2) {
        var stack = this.stack;
        assert_1["default"](this.status === enums_1.ThreadStatus.RUNNABLE || this.status === enums_1.ThreadStatus.ASYNC_WAITING);
        assert_1["default"](typeof (rv) !== 'boolean' && rv2 == null);
        // Pop off the current method.
        var frame = stack.pop();
        if (frame.type != enums_1.StackFrameType.INTERNAL) {
            var frameCast = frame;
            if (frame.type === enums_1.StackFrameType.BYTECODE) {
                // This line will be preceded by a line that prints the method, so can be short n' sweet.
                trace("  Returning: " + logging_1.debug_var(rv));
            }
            trace("\nT" + this.getRef() + " D" + (this.getStackTrace().length + 1) + " Returning value from " + frameCast.method.getFullSignature() + " [" + (frameCast.method.accessFlags.isNative() ? 'Native' : 'Bytecode') + "]: " + logging_1.debug_var(rv));
            assert_1["default"](validateReturnValue(this, frameCast.method, frameCast.method.returnType, this.bsCl, frameCast.method.cls.getLoader(), rv, rv2), "Invalid return value for method " + frameCast.method.getFullSignature());
        }
        // Tell the top of the stack that this RV is waiting for it.
        var idx = stack.length - 1;
        // If idx is 0, then the thread will TERMINATE next time it enters its main
        // loop.
        if (idx >= 0) {
            stack[idx].scheduleResume(this, rv, rv2);
        }
        // Thread state transition.
        this.setStatus(enums_1.ThreadStatus.RUNNABLE);
    };
    /**
     * Pops the top stackframe off of the call stack.
     * WARNING: SHOULD ONLY BE CALLED BY InternalStackFrame.run()!
     */
    JVMThread.prototype.framePop = function () {
        this.stack.pop();
    };
    /**
     * Throws the given JVM exception. Causes the thread to unwind the stack until
     * it can find a stack frame that can handle the exception.
     *
     * Causes the following state transition:
     * * RUNNING => RUNNABLE
     * * RUNNABLE => RUNNABLE
     * * ASYNC_WAITING => RUNNABLE
     *
     * Or, if the exception is uncaught, one of the following transitions:
     * * RUNNING => TERMINATED
     * * RUNNABLE => TERMINATED
     * * ASYNC_WAITING => TERMINATED
     *
     * It is not valid to call this method if the thread is in any other state.
     */
    JVMThread.prototype.throwException = function (exception) {
        assert_1["default"](this.status === enums_1.ThreadStatus.RUNNABLE || this.status === enums_1.ThreadStatus.ASYNC_WAITING, "Tried to throw exception while thread was in state " + enums_1.ThreadStatus[this.status]);
        var stack = this.stack, idx = stack.length - 1;
        // Stack may actually be empty, so guard against this.
        if (idx >= 0) {
            // An internal stack frame cannot process its own thrown exception.
            if (stack[idx].type === enums_1.StackFrameType.INTERNAL) {
                stack.pop();
                idx--;
            }
            // Find a stack frame that can handle the exception.
            // Set our status *before* scheduling the exception. Some exception handlers
            // may want to do something asynchronous before resuming execution.
            this.setStatus(enums_1.ThreadStatus.RUNNABLE);
            while (stack.length > 0 && !stack[idx].scheduleException(this, exception)) {
                stack.pop();
                idx--;
            }
        }
        if (stack.length === 0) {
            // Uncaught exception!
            this.handleUncaughtException(exception);
        }
    };
    /**
     * Construct a new exception object of the given class with the given message.
     * Convenience function for native JavaScript code.
     * @param clsName Name of the class (e.g. "Ljava/lang/Throwable;")
     * @param msg The message to include with the exception.
     */
    JVMThread.prototype.throwNewException = function (clsName, msg) {
        var _this = this;
        var cls = this.getLoader().getInitializedClass(this, clsName), throwException = function () {
            var eCons = cls.getConstructor(_this), e = new eCons(_this);
            // Construct the exception, and throw it when done.
            e['<init>(Ljava/lang/String;)V'](_this, [util_1.initString(_this.bsCl, msg)], function (err) {
                if (err) {
                    _this.throwException(err);
                }
                else {
                    _this.throwException(e);
                }
            });
        };
        if (cls != null) {
            // No initialization required.
            throwException();
        }
        else {
            // Initialization required.
            this.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
            this.getLoader().initializeClass(this, clsName, function (cdata) {
                if (cdata != null) {
                    cls = cdata;
                    throwException();
                }
            }, false);
        }
    };
    /**
     * Handles an uncaught exception on a thread.
     */
    JVMThread.prototype.handleUncaughtException = function (exception) {
        this.jvmThreadObj['dispatchUncaughtException(Ljava/lang/Throwable;)V'](this, [exception]);
    };
    JVMThread.prototype.close = function () {
        this.jvm = null;
    };
    return JVMThread;
}());
exports.JVMThread = JVMThread;
/**
 * [DEBUG] Stores all of the valid thread transitions.
 * @todo Any way to make this smaller?
 * @todo Move into 'debug' module that we NOP out in release builds.
 */
exports.validTransitions = {};
exports.validTransitions[enums_1.ThreadStatus.NEW] = {};
exports.validTransitions[enums_1.ThreadStatus.NEW][enums_1.ThreadStatus.RUNNABLE] = "RunMethod invoked on new thread";
exports.validTransitions[enums_1.ThreadStatus.NEW][enums_1.ThreadStatus.ASYNC_WAITING] = "[JVM bootup only] Internal operation occurs on new thread";
exports.validTransitions[enums_1.ThreadStatus.NEW][enums_1.ThreadStatus.TERMINATED] = "[JVM halt0 only] When the JVM shuts down, it terminates all threads, including those that have never been run.";
exports.validTransitions[enums_1.ThreadStatus.ASYNC_WAITING] = {};
exports.validTransitions[enums_1.ThreadStatus.ASYNC_WAITING][enums_1.ThreadStatus.RUNNABLE] = "Async operation completes";
exports.validTransitions[enums_1.ThreadStatus.ASYNC_WAITING][enums_1.ThreadStatus.TERMINATED] = "RunMethod completes and callstack is empty";
exports.validTransitions[enums_1.ThreadStatus.BLOCKED] = {};
exports.validTransitions[enums_1.ThreadStatus.BLOCKED][enums_1.ThreadStatus.RUNNABLE] = "Acquires monitor, or is interrupted";
exports.validTransitions[enums_1.ThreadStatus.BLOCKED][enums_1.ThreadStatus.TERMINATED] = "Thread is terminated whilst blocked.";
exports.validTransitions[enums_1.ThreadStatus.PARKED] = {};
exports.validTransitions[enums_1.ThreadStatus.PARKED][enums_1.ThreadStatus.ASYNC_WAITING] = "Balancing unpark, or is interrupted";
exports.validTransitions[enums_1.ThreadStatus.PARKED][enums_1.ThreadStatus.TERMINATED] = "Thread is terminated whilst parked.";
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE] = {};
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.ASYNC_WAITING] = "Thread performs an asynchronous JavaScript operation";
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.TERMINATED] = "Callstack is empty";
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.BLOCKED] = "Thread waits to acquire monitor";
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.WAITING] = "Thread waits on monitor (Object.wait)";
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.TIMED_WAITING] = "Thread waits on monitor with timeout (Object.wait)";
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.PARKED] = "Thread parks itself";
exports.validTransitions[enums_1.ThreadStatus.TERMINATED] = {};
exports.validTransitions[enums_1.ThreadStatus.TERMINATED][enums_1.ThreadStatus.NEW] = "Thread is resurrected for re-use";
exports.validTransitions[enums_1.ThreadStatus.TERMINATED][enums_1.ThreadStatus.RUNNABLE] = "Thread is resurrected for re-use";
exports.validTransitions[enums_1.ThreadStatus.TERMINATED][enums_1.ThreadStatus.ASYNC_WAITING] = "[JVM Bootup] Thread is resurrected for internal operation";
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING] = {};
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING][enums_1.ThreadStatus.RUNNABLE] = "Timer expires, or thread is interrupted, and thread immediately acquires lock";
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING][enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED] = "Thread is interrupted or notified, or timer expires, and lock already owned";
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING][enums_1.ThreadStatus.TERMINATED] = "Thread is terminated whilst waiting.";
exports.validTransitions[enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED] = {};
exports.validTransitions[enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED][enums_1.ThreadStatus.RUNNABLE] = "Thread acquires monitor";
exports.validTransitions[enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED][enums_1.ThreadStatus.TERMINATED] = "Thread is terminated whilst blocked.";
exports.validTransitions[enums_1.ThreadStatus.WAITING] = {};
exports.validTransitions[enums_1.ThreadStatus.WAITING][enums_1.ThreadStatus.RUNNABLE] = "Thread is interrupted, and immediately acquires lock";
exports.validTransitions[enums_1.ThreadStatus.WAITING][enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED] = "Thread is notified or interrupted, and does not immediately acquire lock";
exports.validTransitions[enums_1.ThreadStatus.WAITING][enums_1.ThreadStatus.TERMINATED] = "Thread is terminated whilst waiting.";
/**
 * [DEBUG] Ensures that a thread transition is legal.
 */
function validateThreadTransition(oldStatus, newStatus) {
    var rv = exports.validTransitions.hasOwnProperty("" + oldStatus) &&
        exports.validTransitions[oldStatus].hasOwnProperty("" + newStatus);
    return rv;
}
/**
 * [DEBUG] Asserts that the return value of the function passes basic sanity
 * checks.
 */
function validateReturnValue(thread, method, returnType, bsCl, cl, rv1, rv2) {
    // invokeBasic is typed with an Object return value, but it can return any
    // damn type it wants, primitive or no.
    if (method.fullSignature === "java/lang/invoke/MethodHandle/invokeBasic([Ljava/lang/Object;)Ljava/lang/Object;") {
        return true;
    }
    var cls;
    if (util_1.is_primitive_type(returnType)) {
        switch (returnType) {
            case 'Z':
                assert_1["default"](rv2 === undefined, "Second return value must be undefined for Boolean type.");
                assert_1["default"](rv1 === 1 || rv1 === 0, "Booleans must be 0 or 1.");
                break;
            case 'B':
                assert_1["default"](rv2 === undefined, "Second return value must be undefined for Byte type.");
                assert_1["default"](rv1 <= 127 && rv1 >= -128, "Byte value for method " + method.name + " is out of bounds: " + rv1);
                break;
            case 'C':
                assert_1["default"](rv2 === undefined, "Second return value must be undefined for Character type.");
                assert_1["default"](rv1 <= 65535 && rv1 >= 0, "Character value is out of bounds: " + rv1);
                break;
            case 'S':
                assert_1["default"](rv2 === undefined, "Second return value must be undefined for Short type.");
                assert_1["default"](rv1 <= 32767 && rv1 >= -32768, "Short value is out of bounds: " + rv1);
                break;
            case 'I':
                assert_1["default"](rv2 === undefined, "Second return value must be undefined for Int type.");
                assert_1["default"](rv1 <= 2147483647 && rv1 >= -2147483648, "Int value is out of bounds: " + rv1);
                break;
            case 'J':
                assert_1["default"](rv2 === null, "Second return value must be NULL for Long type.");
                assert_1["default"](rv1.lessThanOrEqual(gLong_1["default"].MAX_VALUE) && rv1.greaterThanOrEqual(gLong_1["default"].MIN_VALUE), "Long value is out of bounds: " + rv1);
                break;
            case 'F':
                assert_1["default"](rv2 === undefined, "Second return value must be undefined for Float type.");
                // NaN !== NaN, so we have to have a special case here.
                assert_1["default"](util_1.wrapFloat(rv1) === rv1 || (isNaN(rv1) && isNaN(util_1.wrapFloat(rv1))), "Float value is out of bounds: " + rv1);
                break;
            case 'D':
                assert_1["default"](rv2 === null, "Second return value must be NULL for Double type.");
                assert_1["default"](typeof rv1 === 'number', "Invalid double value: " + rv1);
                break;
            case 'V':
                assert_1["default"](rv1 === undefined && rv2 === undefined, "Return values must be undefined for Void type");
                break;
        }
    }
    else if (util_1.is_array_type(returnType)) {
        assert_1["default"](rv2 === undefined, "Second return value must be undefined for array type.");
        assert_1["default"](rv1 === null || (typeof rv1 === 'object' && typeof rv1['getClass'] === 'function'), "Invalid array object: " + rv1);
        if (rv1 != null) {
            cls = assertClassInitializedOrResolved(thread, cl, returnType, true);
            assert_1["default"](rv1.getClass().isCastable(cls), "Return value of type " + rv1.getClass().getInternalName() + " unable to be cast to return type " + returnType + ".");
        }
    }
    else {
        assert_1["default"](util_1.is_reference_type(returnType), "Invalid reference type: " + returnType);
        assert_1["default"](rv2 === undefined, "Second return value must be undefined for reference type.");
        // All objects and arrays are instances of java/lang/Object.
        assert_1["default"](rv1 === null || rv1 instanceof bsCl.getInitializedClass(thread, 'Ljava/lang/Object;').getConstructor(thread), "Reference return type must be an instance of Object; value: " + rv1);
        if (rv1 != null) {
            cls = assertClassInitializedOrResolved(thread, cl, returnType, false);
            if (!cls.accessFlags.isInterface()) {
                // You can return an interface type without initializing it,
                // since they don't need to be initialized until you try to
                // invoke one of their
                // NOTE: We don't check if the class is in the INITIALIZED state,
                // since it is possible that it is currently in the process of being
                // initialized. getInitializedClass handles this subtlety.
                assertClassInitializedOrResolved(thread, cl, returnType, true);
            }
            assert_1["default"](rv1.getClass().isCastable(cls), "Unable to cast " + rv1.getClass().getInternalName() + " to " + returnType + ".");
        }
    }
    return true;
}
function assertClassInitializedOrResolved(thread, cl, type, initialized) {
    var cls = null;
    // Break out of loop once class is found.
    while (cls === null) {
        cls = initialized ? cl.getInitializedClass(thread, type) : cl.getResolvedClass(type);
        if (cl.getLoaderObject() !== null) {
            if (cl.getLoaderObject()['java/lang/ClassLoader/parent'] === null) {
                cl = thread.getBsCl();
            }
            else {
                cl = cl.getLoaderObject()['java/lang/ClassLoader/parent'].$loader;
            }
        }
        else {
            // We just checked the bootstrap classloader, so we reached the root.
            assert_1["default"](cls !== null, "Unable to get initialized class for type " + type + ".");
        }
    }
    return cls;
}
function printConstantPoolItem(cpi) {
    switch (cpi.getType()) {
        case enums_1.ConstantPoolItemType.METHODREF:
            var cpiMR = cpi;
            return util_1.ext_classname(cpiMR.classInfo.name) + "." + cpiMR.signature;
        case enums_1.ConstantPoolItemType.INTERFACE_METHODREF:
            var cpiIM = cpi;
            return util_1.ext_classname(cpiIM.classInfo.name) + "." + cpiIM.signature;
        case enums_1.ConstantPoolItemType.FIELDREF:
            var cpiFR = cpi;
            return util_1.ext_classname(cpiFR.classInfo.name) + "." + cpiFR.nameAndTypeInfo.name + ":" + util_1.ext_classname(cpiFR.nameAndTypeInfo.descriptor);
        case enums_1.ConstantPoolItemType.NAME_AND_TYPE:
            var cpiNAT = cpi;
            return cpiNAT.name + ":" + cpiNAT.descriptor;
        case enums_1.ConstantPoolItemType.CLASS:
            var cpiClass = cpi;
            return util_1.ext_classname(cpiClass.name);
        default:
            return logging_1.debug_var(cpi.value);
    }
}
// TODO: Prefix behind DEBUG, cache lowercase opcode names.
exports.OpcodeLayoutPrinters = {};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.OPCODE_ONLY] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase(); };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.CONSTANT_POOL] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + printConstantPoolItem(method.cls.constantPool.get(code.readUInt16BE(pc + 1))); };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.CONSTANT_POOL_UINT8] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + printConstantPoolItem(method.cls.constantPool.get(code[pc + 1])); };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.CONSTANT_POOL_AND_UINT8_VALUE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + printConstantPoolItem(method.cls.constantPool.get(code.readUInt16BE(pc + 1))) + " " + code[pc + 3]; };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.UINT8_VALUE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + code[pc + 1]; };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.UINT8_AND_INT8_VALUE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + code[pc + 1] + " " + code[pc + 2]; };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.INT8_VALUE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + code.readInt8(pc + 1); };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.INT16_VALUE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + code.readInt16BE(pc + 1); };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.INT32_VALUE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + code.readInt32BE(pc + 1); };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.ARRAY_TYPE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase() + " " + opcodes_1.ArrayTypes[code[pc + 1]]; };
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.WIDE] = function (method, code, pc) { return enums_1.OpCode[code[pc]].toLowerCase(); };
function annotateOpcode(op, method, code, pc) {
    return exports.OpcodeLayoutPrinters[enums_1.OpcodeLayouts[op]](method, code, pc);
}
exports.annotateOpcode = annotateOpcode;
//# sourceMappingURL=data:application/json;base64,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