"use strict";
var enums_1 = require('./enums');
var assert_1 = require('./assert');
/**
 * Checks if the given thread status indicates that the thread is scheduleable.
 */
function isRunnable(status) {
    return status === enums_1.ThreadStatus.RUNNABLE;
}
/**
 * A Weighted Round Robin thread scheduler.
 */
var WeightedRoundRobinScheduler = (function () {
    function WeightedRoundRobinScheduler() {
        // Number of quanta given to the current thread.
        this._count = 0;
        // The queue of threads.
        this._queue = [];
        // Read by runThread. Used as a lock.
        this._threadScheduled = false;
    }
    WeightedRoundRobinScheduler.prototype.scheduleThread = function (thread) {
        this._queue.push(thread);
        if (this._queue.length === 1) {
            // There aren't any threads running. Run this thread.
            this.runThread();
        }
    };
    /**
     * Run the thread at the head of the queue.
     */
    WeightedRoundRobinScheduler.prototype.runThread = function () {
        var _this = this;
        if (this._threadScheduled) {
            return;
        }
        this._threadScheduled = true;
        setImmediate(function () {
            var queue = _this._queue;
            _this._threadScheduled = false;
            if (queue.length > 0) {
                var thread = _this._queue[0];
                assert_1["default"](thread.getStatus() === enums_1.ThreadStatus.RUNNABLE, "Attempted to run non-runnable thread.");
                thread.run();
            }
        });
    };
    WeightedRoundRobinScheduler.prototype.unscheduleThread = function (thread) {
        var queue = this._queue;
        var isRunningThread = queue[0] === thread;
        assert_1["default"](queue.indexOf(thread) > -1, "Tried to unschedule thread that was not scheduled.");
        // Remove thread from queue.
        if (isRunningThread) {
            queue.shift();
            this._count = 0;
            // Schedule the next thread.
            this.runThread();
        }
        else {
            queue.splice(queue.indexOf(thread), 1);
        }
    };
    WeightedRoundRobinScheduler.prototype.getRunningThread = function () {
        var queue = this._queue;
        if (queue.length > 0) {
            return queue[0];
        }
        else {
            return null;
        }
    };
    WeightedRoundRobinScheduler.prototype.priorityChange = function (thread) {
        // Not important for the algorithm. We'll pick up the change
        // next time we schedule.
    };
    WeightedRoundRobinScheduler.prototype.quantumOver = function (thread) {
        assert_1["default"](this._queue[0] === thread, "A non-running thread has an expired quantum?");
        this._count++;
        if (this._count >= thread.getPriority() || thread.getStatus() !== enums_1.ThreadStatus.RUNNABLE) {
            // Move to back of queue, reset count.
            this._count = 0;
            this._queue.push(this._queue.shift());
        }
        // Schedule the next thread.
        this.runThread();
    };
    return WeightedRoundRobinScheduler;
}());
/**
 * Represents a thread pool. Handles scheduling duties.
 */
var ThreadPool = (function () {
    function ThreadPool(emptyCallback) {
        this.threads = [];
        this.scheduler = new WeightedRoundRobinScheduler();
        this.emptyCallback = emptyCallback;
    }
    /**
     * Retrieve all of the threads in the thread pool.
     */
    ThreadPool.prototype.getThreads = function () {
        // Return a copy of our internal array.
        return this.threads.slice(0);
    };
    /**
     * Checks if any remaining threads are non-daemonic and could be runnable.
     * If not, we can terminate execution.
     *
     * This check is invoked each time a thread terminates.
     */
    ThreadPool.prototype.anyNonDaemonicThreads = function () {
        for (var i = 0; i < this.threads.length; i++) {
            var t = this.threads[i];
            if (t.isDaemon()) {
                continue;
            }
            var status_1 = t.getStatus();
            if (status_1 !== enums_1.ThreadStatus.NEW &&
                status_1 !== enums_1.ThreadStatus.TERMINATED) {
                return true;
            }
        }
        return false;
    };
    ThreadPool.prototype.threadTerminated = function (thread) {
        var idx = this.threads.indexOf(thread);
        assert_1["default"](idx >= 0);
        // Remove the specified thread from the threadpool.
        this.threads.splice(idx, 1);
        if (!this.anyNonDaemonicThreads()) {
            var close_1 = this.emptyCallback();
            if (close_1) {
                this.emptyCallback = null;
            }
        }
    };
    /**
     * Called when a thread's status changes.
     */
    ThreadPool.prototype.statusChange = function (thread, oldStatus, newStatus) {
        var wasRunnable = isRunnable(oldStatus), nowRunnable = isRunnable(newStatus);
        if (oldStatus === enums_1.ThreadStatus.NEW || oldStatus === enums_1.ThreadStatus.TERMINATED) {
            if (this.threads.indexOf(thread) === -1) {
                this.threads.push(thread);
            }
        }
        // Inform scheduling algorithm if thread changes from runnable => unrunnable, or unrunnable => runnable.
        if (wasRunnable !== nowRunnable) {
            if (wasRunnable) {
                this.scheduler.unscheduleThread(thread);
            }
            else {
                this.scheduler.scheduleThread(thread);
            }
        }
        if (newStatus === enums_1.ThreadStatus.TERMINATED) {
            this.threadTerminated(thread);
        }
    };
    /**
     * Called when a thread's priority changes.
     */
    ThreadPool.prototype.priorityChange = function (thread) {
        this.scheduler.priorityChange(thread);
    };
    /**
     * Called when a thread's quantum is over.
     */
    ThreadPool.prototype.quantumOver = function (thread) {
        this.scheduler.quantumOver(thread);
    };
    return ThreadPool;
}());
exports.__esModule = true;
exports["default"] = ThreadPool;
//# sourceMappingURL=data:application/json;base64,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