"use strict";
var gLong_1 = require('./gLong');
var enums_1 = require('./enums');
var BrowserFS = require('browserfs');
var BFSUtils = BrowserFS.BFSRequire('bfs_utils');
/**
 * util contains stateless utility functions that are used around Doppio's
 * codebase.
 * TODO: Separate general JS utility methods from JVM utility methods.
 */
/**
 * Merges object literals together into a new object. Emulates underscore's merge function.
 */
function merge() {
    var literals = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        literals[_i - 0] = arguments[_i];
    }
    var newObject = {};
    literals.forEach(function (literal) {
        Object.keys(literal).forEach(function (key) {
            newObject[key] = literal[key];
        });
    });
    return newObject;
}
exports.merge = merge;
function are_in_browser() {
    return process.platform === 'browser';
}
exports.are_in_browser = are_in_browser;
exports.typedArraysSupported = typeof ArrayBuffer !== "undefined";
/**
 * Converts JVM internal names into JS-safe names. Only for use with reference
 * types.
 * Ljava/lang/Object; => java_lang_Object
 * Lfoo/Bar_baz; => foo_Bar__baz
 *
 * Is NOT meant to be unambiguous!
 *
 * Also handles the special characters described here:
 * https://blogs.oracle.com/jrose/entry/symbolic_freedom_in_the_vm
 */
function jvmName2JSName(jvmName) {
    switch (jvmName[0]) {
        case 'L':
            return jvmName.slice(1, jvmName.length - 1).replace(/_/g, '__')
                .replace(/[\/.;$<>\[\]:\\=^-]/g, '_');
        case '[':
            return "ARR_" + jvmName2JSName(jvmName.slice(1));
        default:
            return jvmName;
    }
}
exports.jvmName2JSName = jvmName2JSName;
/**
 * Re-escapes JVM names for eval'd code. Otherwise, JavaScript removes the escapes.
 */
function reescapeJVMName(jvmName) {
    return jvmName.replace(/\\/g, '\\\\');
}
exports.reescapeJVMName = reescapeJVMName;
/**
 * Applies an async function to each element of a list, in order.
 */
function asyncForEach(lst, fn, done_cb) {
    var i = -1;
    function processItem(err) {
        if (err) {
            done_cb(err);
        }
        else {
            i++;
            if (i < lst.length) {
                fn(lst[i], processItem);
            }
            else {
                done_cb();
            }
        }
    }
    processItem();
}
exports.asyncForEach = asyncForEach;
/**
 * Runs the specified tasks in series.
 */
function asyncSeries(tasks, doneCb) {
    var i = -1;
    function processItem(err) {
        if (err) {
            doneCb(err);
        }
        else {
            i++;
            if (i < tasks.length) {
                tasks[i](processItem);
            }
            else {
                doneCb();
            }
        }
    }
    processItem();
}
exports.asyncSeries = asyncSeries;
/**
 * Applies the function to each element of the list in order in series.
 * The first element that returns success halts the process, and triggers
 * done_cb. If no elements return success, done_cb is triggered with no
 * arguments.
 *
 * I wrote this specifically for classloading, but it may have uses elsewhere.
 */
function asyncFind(lst, fn, done_cb) {
    var i = -1;
    function processItem(success) {
        if (success) {
            done_cb(lst[i]);
        }
        else {
            i++;
            if (i < lst.length) {
                fn(lst[i], processItem);
            }
            else {
                done_cb();
            }
        }
    }
    processItem(false);
}
exports.asyncFind = asyncFind;
if (!Math['imul']) {
    Math['imul'] = function (a, b) {
        // polyfill from https://developer.mozilla.org/en-US/docs/JavaScript/Reference/Global_Objects/Math/imul
        var ah = (a >>> 16) & 0xffff;
        var al = a & 0xffff;
        var bh = (b >>> 16) & 0xffff;
        var bl = b & 0xffff;
        // the shift by 0 fixes the sign on the high part
        // the final |0 converts the unsigned value into a signed value
        return ((al * bl) + (((ah * bl + al * bh) << 16) >>> 0) | 0);
    };
}
if (!Math['expm1']) {
    Math['expm1'] = function (x) {
        if (Math.abs(x) < 1e-5) {
            return x + 0.5 * x * x;
        }
        else {
            return Math.exp(x) - 1.0;
        }
    };
}
if (!Math['sinh']) {
    Math['sinh'] = function (a) {
        var exp = Math.exp(a);
        return (exp - 1 / exp) / 2;
    };
}
if (!Array.prototype.indexOf) {
    Array.prototype.indexOf = function (searchElement, fromIndex) {
        if (this == null) {
            throw new TypeError();
        }
        var t = Object(this);
        var len = t.length >>> 0;
        if (len === 0) {
            return -1;
        }
        var n = 0;
        if (fromIndex !== undefined) {
            n = Number(fromIndex);
            if (n != n) {
                n = 0;
            }
            else if (n != 0 && n != Infinity && n != -Infinity) {
                n = ((n > 0 ? 1 : 0) || -1) * Math.floor(Math.abs(n));
            }
        }
        if (n >= len) {
            return -1;
        }
        var k = n >= 0 ? n : Math.max(len - Math.abs(n), 0);
        for (; k < len; k++) {
            if (k in t && t[k] === searchElement) {
                return k;
            }
        }
        return -1;
    };
}
/**
 * Checks if accessingCls has permission to a field or method with the given
 * flags on owningCls.
 *
 * Modifier    | Class | Package | Subclass | World
 * ————————————+———————+—————————+——————————+———————
 * public      |  y    |    y    |    y     |   y
 * ————————————+———————+—————————+——————————+———————
 * protected   |  y    |    y    |    y     |   n
 * ————————————+———————+—————————+——————————+———————
 * no modifier |  y    |    y    |    n     |   n
 * ————————————+———————+—————————+——————————+———————
 * private     |  y    |    n    |    n     |   n
 *
 * y: accessible
 * n: not accessible
 */
function checkAccess(accessingCls, owningCls, accessFlags) {
    if (accessFlags.isPublic()) {
        return true;
    }
    else if (accessFlags.isProtected()) {
        return accessingCls.getPackageName() === owningCls.getPackageName() || accessingCls.isSubclass(owningCls);
    }
    else if (accessFlags.isPrivate()) {
        return accessingCls === owningCls;
    }
    else {
        return accessingCls.getPackageName() === owningCls.getPackageName();
    }
}
exports.checkAccess = checkAccess;
/**
 * Truncates a floating point into an integer.
 */
function float2int(a) {
    if (a > enums_1.Constants.INT_MAX) {
        return enums_1.Constants.INT_MAX;
    }
    else if (a < enums_1.Constants.INT_MIN) {
        return enums_1.Constants.INT_MIN;
    }
    else {
        return a | 0;
    }
}
exports.float2int = float2int;
var supportsArrayBuffers = typeof (ArrayBuffer) !== 'undefined';
/**
 * Converts a byte array to a buffer. **Copies.**
 */
function byteArray2Buffer(bytes, offset, len) {
    if (offset === void 0) { offset = 0; }
    if (len === void 0) { len = bytes.length; }
    if (supportsArrayBuffers && ArrayBuffer.isView(bytes)) {
        var offset_1 = bytes.byteOffset;
        return new Buffer(bytes.buffer.slice(offset_1, offset_1 + bytes.length));
    }
    else {
        var buff = new Buffer(len), i;
        for (i = 0; i < len; i++) {
            buff.writeInt8(bytes[offset + i], i);
        }
        return buff;
    }
}
exports.byteArray2Buffer = byteArray2Buffer;
function isUint8Array(arr) {
    if (arr && typeof (Uint8Array) !== "undefined" && arr instanceof Uint8Array) {
        return true;
    }
    return false;
}
exports.isUint8Array = isUint8Array;
function isInt8Array(arr) {
    if (arr && typeof (Int8Array) !== "undefined" && arr instanceof Int8Array) {
        return true;
    }
    return false;
}
exports.isInt8Array = isInt8Array;
/**
 * Converts an Int8Array or an array of 8-bit signed ints into
 * a Uint8Array or an array of 8-bit unsigned ints.
 */
function i82u8(arr, start, len) {
    if (isInt8Array(arr)) {
        return new Uint8Array(arr.buffer, arr.byteOffset + start, len);
    }
    else if (Array.isArray(arr)) {
        if (typeof (Uint8Array) !== "undefined") {
            var i8arr = new Int8Array(len);
            if (start === 0 && len === arr.length) {
                i8arr.set(arr, 0);
            }
            else {
                i8arr.set(arr.slice(start, start + len), 0);
            }
            return new Uint8Array(i8arr.buffer);
        }
        else {
            // Slow way.
            var rv = new Array(len);
            for (var i = 0; i < len; i++) {
                rv[i] = arr[start + i] & 0xFF;
            }
            return rv;
        }
    }
    else {
        throw new TypeError("Invalid array.");
    }
}
exports.i82u8 = i82u8;
/**
 * Converts an Uint8Array or an array of 8-bit unsigned ints into
 * an Int8Array or an array of 8-bit signed ints.
 */
function u82i8(arr, start, len) {
    if (start === void 0) { start = 0; }
    if (len === void 0) { len = arr.length; }
    if (isUint8Array(arr)) {
        return new Int8Array(arr.buffer, arr.byteOffset + start, len);
    }
    else if (Array.isArray(arr)) {
        if (typeof (Int8Array) !== "undefined") {
            var u8arr = new Uint8Array(len);
            if (start === 0 && len === arr.length) {
                u8arr.set(arr, 0);
            }
            else {
                u8arr.set(arr.slice(start, start + len), 0);
            }
            return new Int8Array(u8arr.buffer);
        }
        else {
            // Slow way.
            var rv = new Array(len);
            for (var i = 0; i < len; i++) {
                rv[i] = arr[start + i];
                if (rv[i] > 127) {
                    // Sign extend.
                    rv[i] |= 0xFFFFFF80;
                }
            }
            return rv;
        }
    }
    else {
        throw new TypeError("Invalid array.");
    }
}
exports.u82i8 = u82i8;
// Call this ONLY on the result of two non-NaN numbers.
function wrapFloat(a) {
    if (a > 3.40282346638528860e+38) {
        return Number.POSITIVE_INFINITY;
    }
    if (0 < a && a < 1.40129846432481707e-45) {
        return 0;
    }
    if (a < -3.40282346638528860e+38) {
        return Number.NEGATIVE_INFINITY;
    }
    if (0 > a && a > -1.40129846432481707e-45) {
        return 0;
    }
    return a;
}
exports.wrapFloat = wrapFloat;
// Convert :count chars starting from :offset in a Java character array into a JS string
function chars2jsStr(jvmCarr, offset, count) {
    if (offset === void 0) { offset = 0; }
    if (count === void 0) { count = jvmCarr.array.length; }
    var i, carrArray = jvmCarr.array, rv = "", endOffset = offset + count;
    for (i = offset; i < endOffset; i++) {
        rv += String.fromCharCode(carrArray[i]);
    }
    return rv;
}
exports.chars2jsStr = chars2jsStr;
// TODO: Is this used anywhere where we are *not* inserting the bytestr into
// a JVMArray object?
// TODO: Could inject this as a static String method...
function bytestr2Array(byteStr) {
    var rv = [];
    for (var i = 0; i < byteStr.length; i++) {
        rv.push(byteStr.charCodeAt(i));
    }
    return rv;
}
exports.bytestr2Array = bytestr2Array;
function array2bytestr(byteArray) {
    // XXX: We'd like to use String.fromCharCode(bytecode_array...)
    //  but that fails on Webkit with arrays longer than 2^31. See issue #129 for details.
    var rv = '';
    for (var i = 0; i < byteArray.length; i++) {
        rv += String.fromCharCode(byteArray[i]);
    }
    return rv;
}
exports.array2bytestr = array2bytestr;
/**
 * Bit masks for the flag byte.
 */
(function (FlagMasks) {
    FlagMasks[FlagMasks["PUBLIC"] = 1] = "PUBLIC";
    FlagMasks[FlagMasks["PRIVATE"] = 2] = "PRIVATE";
    FlagMasks[FlagMasks["PROTECTED"] = 4] = "PROTECTED";
    FlagMasks[FlagMasks["STATIC"] = 8] = "STATIC";
    FlagMasks[FlagMasks["FINAL"] = 16] = "FINAL";
    FlagMasks[FlagMasks["SYNCHRONIZED"] = 32] = "SYNCHRONIZED";
    FlagMasks[FlagMasks["SUPER"] = 32] = "SUPER";
    FlagMasks[FlagMasks["VOLATILE"] = 64] = "VOLATILE";
    FlagMasks[FlagMasks["TRANSIENT"] = 128] = "TRANSIENT";
    FlagMasks[FlagMasks["VARARGS"] = 128] = "VARARGS";
    FlagMasks[FlagMasks["NATIVE"] = 256] = "NATIVE";
    FlagMasks[FlagMasks["INTERFACE"] = 512] = "INTERFACE";
    FlagMasks[FlagMasks["ABSTRACT"] = 1024] = "ABSTRACT";
    FlagMasks[FlagMasks["STRICT"] = 2048] = "STRICT";
})(exports.FlagMasks || (exports.FlagMasks = {}));
var FlagMasks = exports.FlagMasks;
/**
 * Represents a 'flag byte'. See �4 of the JVM spec.
 * @todo Separate METHOD flags and CLASS flags.
 */
var Flags = (function () {
    function Flags(byte) {
        this.byte = byte;
    }
    Flags.prototype.isPublic = function () {
        return (this.byte & FlagMasks.PUBLIC) > 0;
    };
    Flags.prototype.isPrivate = function () {
        return (this.byte & FlagMasks.PRIVATE) > 0;
    };
    Flags.prototype.isProtected = function () {
        return (this.byte & FlagMasks.PROTECTED) > 0;
    };
    Flags.prototype.isStatic = function () {
        return (this.byte & FlagMasks.STATIC) > 0;
    };
    Flags.prototype.isFinal = function () {
        return (this.byte & FlagMasks.FINAL) > 0;
    };
    Flags.prototype.isSynchronized = function () {
        return (this.byte & FlagMasks.SYNCHRONIZED) > 0;
    };
    Flags.prototype.isSuper = function () {
        return (this.byte & FlagMasks.SUPER) > 0;
    };
    Flags.prototype.isVolatile = function () {
        return (this.byte & FlagMasks.VOLATILE) > 0;
    };
    Flags.prototype.isTransient = function () {
        return (this.byte & FlagMasks.TRANSIENT) > 0;
    };
    Flags.prototype.isNative = function () {
        return (this.byte & FlagMasks.NATIVE) > 0;
    };
    Flags.prototype.isInterface = function () {
        return (this.byte & FlagMasks.INTERFACE) > 0;
    };
    Flags.prototype.isAbstract = function () {
        return (this.byte & FlagMasks.ABSTRACT) > 0;
    };
    Flags.prototype.isStrict = function () {
        return (this.byte & FlagMasks.STRICT) > 0;
    };
    /**
     * Changes a function to native. Used for trapped methods.
     */
    Flags.prototype.setNative = function (n) {
        if (n) {
            this.byte = this.byte | FlagMasks.NATIVE;
        }
        else {
            this.byte = this.byte & (~FlagMasks.NATIVE);
        }
    };
    Flags.prototype.isVarArgs = function () {
        return (this.byte & FlagMasks.VARARGS) > 0;
    };
    Flags.prototype.getRawByte = function () {
        return this.byte;
    };
    return Flags;
}());
exports.Flags = Flags;
function initialValue(type_str) {
    if (type_str === 'J')
        return gLong_1["default"].ZERO;
    var c = type_str[0];
    if (c === '[' || c === 'L')
        return null;
    return 0;
}
exports.initialValue = initialValue;
/**
 * Java classes are represented internally using slashes as delimiters.
 * These helper functions convert between the two representations.
 * Ljava/lang/Class; => java.lang.Class
 */
function ext_classname(str) {
    return descriptor2typestr(str).replace(/\//g, '.');
}
exports.ext_classname = ext_classname;
/**
 * java.lang.Class => Ljava/lang/Class;
 */
function int_classname(str) {
    return typestr2descriptor(str.replace(/\./g, '/'));
}
exports.int_classname = int_classname;
function verify_int_classname(str) {
    var array_nesting = str.match(/^\[*/)[0].length;
    if (array_nesting > 255) {
        return false;
    }
    if (array_nesting > 0) {
        str = str.slice(array_nesting);
    }
    if (str[0] === 'L') {
        if (str[str.length - 1] !== ';') {
            return false;
        }
        str = str.slice(1, -1);
    }
    if (str in exports.internal2external) {
        return true;
    }
    if (str.match(/\/{2,}/)) {
        return false;
    }
    var parts = str.split('/');
    for (var i = 0; i < parts.length; i++) {
        if (parts[i].match(/[^$_a-z0-9]/i)) {
            return false;
        }
    }
    return true;
}
exports.verify_int_classname = verify_int_classname;
exports.internal2external = {
    B: 'byte',
    C: 'char',
    D: 'double',
    F: 'float',
    I: 'int',
    J: 'long',
    S: 'short',
    V: 'void',
    Z: 'boolean'
};
exports.external2internal = {};
for (var k in exports.internal2external) {
    exports.external2internal[exports.internal2external[k]] = k;
}
/**
 * Given a method descriptor, returns the typestrings for the return type
 * and the parameters.
 *
 * e.g. (Ljava/lang/Class;Z)Ljava/lang/String; =>
 *        ["Ljava/lang/Class;", "Z", "Ljava/lang/String;"]
 */
function getTypes(methodDescriptor) {
    var i = 0, types = [], endIdx;
    for (i = 0; i < methodDescriptor.length; i++) {
        switch (methodDescriptor.charAt(i)) {
            case '(':
            case ')':
                //Skip.
                break;
            case 'L':
                // Reference type.
                endIdx = methodDescriptor.indexOf(';', i);
                types.push(methodDescriptor.slice(i, endIdx + 1));
                i = endIdx;
                break;
            case '[':
                endIdx = i + 1;
                // Find the start of the component.
                while (methodDescriptor.charAt(endIdx) === '[') {
                    endIdx++;
                }
                if (methodDescriptor.charAt(endIdx) === 'L') {
                    // Reference component. Read ahead to end.
                    endIdx = methodDescriptor.indexOf(';', endIdx);
                    types.push(methodDescriptor.slice(i, endIdx + 1));
                }
                else {
                    // Primitive component.
                    types.push(methodDescriptor.slice(i, endIdx + 1));
                }
                i = endIdx;
                break;
            default:
                // Primitive type.
                types.push(methodDescriptor.charAt(i));
                break;
        }
    }
    return types;
}
exports.getTypes = getTypes;
// Get the component type of an array type string.
// Cut off the [L and ; for arrays of classes.
function get_component_type(type_str) {
    return type_str.slice(1);
}
exports.get_component_type = get_component_type;
function is_array_type(type_str) {
    return type_str[0] === '[';
}
exports.is_array_type = is_array_type;
function is_primitive_type(type_str) {
    return type_str in exports.internal2external;
}
exports.is_primitive_type = is_primitive_type;
function is_reference_type(type_str) {
    return type_str[0] === 'L';
}
exports.is_reference_type = is_reference_type;
/**
 * Converts type descriptors into standardized internal type strings.
 * Ljava/lang/Class; => java/lang/Class   Reference types
 * [Ljava/lang/Class; is unchanged        Array types
 * C => char                              Primitive types
 */
function descriptor2typestr(type_str) {
    var c = type_str[0];
    if (c in exports.internal2external)
        return exports.internal2external[c];
    if (c === 'L')
        return type_str.slice(1, -1);
    if (c === '[')
        return type_str;
    // no match
    throw new Error("Unrecognized type string: " + type_str);
}
exports.descriptor2typestr = descriptor2typestr;
// Takes a character array of concatenated type descriptors and returns/removes the first one.
function carr2descriptor(carr) {
    var c = carr.shift();
    if (c == null)
        return null;
    if (exports.internal2external[c] !== void 0)
        return c;
    if (c === 'L') {
        var rv = 'L';
        while ((c = carr.shift()) !== ';') {
            rv += c;
        }
        return rv + ';';
    }
    if (c === '[')
        return "[" + carr2descriptor(carr);
    // no match
    carr.unshift(c);
    throw new Error("Unrecognized descriptor: " + carr.join(''));
}
exports.carr2descriptor = carr2descriptor;
// Converts internal type strings into type descriptors. Reverse of descriptor2typestr.
function typestr2descriptor(type_str) {
    if (exports.external2internal[type_str] !== void 0) {
        return exports.external2internal[type_str];
    }
    else if (type_str[0] === '[') {
        return type_str;
    }
    else {
        return "L" + type_str + ";";
    }
}
exports.typestr2descriptor = typestr2descriptor;
/**
 * Java's reflection APIs need to unbox primitive arguments to function calls,
 * as they are boxed in an Object array. This utility function converts
 * an array of arguments into the appropriate form prior to function invocation.
 * Note that this includes padding category 2 primitives, which consume two
 * slots in the array (doubles/longs).
 */
function unboxArguments(thread, paramTypes, args) {
    var rv = [], i, type, arg;
    for (i = 0; i < paramTypes.length; i++) {
        type = paramTypes[i];
        arg = args[i];
        if (is_primitive_type(type)) {
            // Unbox the primitive type.
            // TODO: Precisely type this better. Once TypeScript lets you import
            // union types, we can define a "JVMPrimitive" type...
            rv.push(arg.unbox());
            if (type === 'J' || type === 'D') {
                // 64-bit primitives take up two argument slots. Doppio uses a NULL for the second slot.
                rv.push(null);
            }
        }
        else {
            // Reference type; do not change.
            rv.push(arg);
        }
    }
    return rv;
}
exports.unboxArguments = unboxArguments;
/**
 * Given a method descriptor as a JS string, returns a corresponding MethodType
 * object.
 */
function createMethodType(thread, cl, descriptor, cb) {
    cl.initializeClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;', function (cdata) {
        if (cdata !== null) {
            var jsCons = cdata.getConstructor(thread), classes = getTypes(descriptor);
            classes.push('[Ljava/lang/Class;');
            // Need the return type and parameter types.
            cl.resolveClasses(thread, classes, function (classMap) {
                var types = classes.map(function (cls) { return classMap[cls].getClassObject(thread); });
                types.pop(); // Discard '[Ljava/lang/Class;'
                var rtype = types.pop(), // Return type.
                clsArrCons = classMap['[Ljava/lang/Class;'].getConstructor(thread), ptypes = new clsArrCons(thread, types.length);
                ptypes.array = types;
                jsCons['java/lang/invoke/MethodHandleNatives/findMethodHandleType(Ljava/lang/Class;[Ljava/lang/Class;)Ljava/lang/invoke/MethodType;'](thread, [rtype, ptypes], cb);
            });
        }
    });
}
exports.createMethodType = createMethodType;
/**
 * Given a method descriptor, returns the number of words required to store
 * its arguments.
 * Does not include considerations for e.g. the 'this' argument, since the
 * descriptor does not specify if the method is static or not.
 */
function getMethodDescriptorWordSize(descriptor) {
    var parsedDescriptor = getTypes(descriptor), words = parsedDescriptor.length - 1, i, p;
    // Remove return type.
    parsedDescriptor.pop();
    // Double count doubles / longs.
    for (i = 0; i < parsedDescriptor.length; i++) {
        p = parsedDescriptor[i];
        if (p === 'D' || p === 'J') {
            words++;
        }
    }
    return words;
}
exports.getMethodDescriptorWordSize = getMethodDescriptorWordSize;
/**
 * Given a return type as a Class object, and an array of class objects for
 * parameter types, returns the descriptor string for the method type.
 */
function getDescriptorString(rtype, ptypes) {
    var rv = "(";
    if (ptypes !== undefined && ptypes !== null) {
        ptypes.array.forEach(function (ptype) {
            rv += ptype.$cls.getInternalName();
        });
    }
    rv += ")" + rtype.$cls.getInternalName();
    return rv;
}
exports.getDescriptorString = getDescriptorString;
/**
 * Have a JavaClassLoaderObject and need its ClassLoader object? Use this method!
 * @todo Install on Java ClassLoader objects.
 */
function getLoader(thread, jclo) {
    if ((jclo != null) && (jclo.$loader != null)) {
        return jclo.$loader;
    }
    return thread.getBsCl();
}
exports.getLoader = getLoader;
/**
 * "Fast" array copy; does not have to check every element for illegal
 * assignments. You can do tricks here (if possible) to copy chunks of the array
 * at a time rather than element-by-element.
 * This function *cannot* access any attribute other than 'array' on src due to
 * the special case when src == dest (see code for System.arraycopy below).
 */
function arraycopyNoCheck(src, srcPos, dest, destPos, length) {
    var j = destPos;
    var end = srcPos + length;
    for (var i = srcPos; i < end; i++) {
        dest.array[j++] = src.array[i];
    }
}
exports.arraycopyNoCheck = arraycopyNoCheck;
/**
 * "Slow" array copy; has to check every element for illegal assignments.
 * You cannot do any tricks here; you must copy element by element until you
 * have either copied everything, or encountered an element that cannot be
 * assigned (which causes an exception).
 * Guarantees: src and dest are two different reference types. They cannot be
 *             primitive arrays.
 */
function arraycopyCheck(thread, src, srcPos, dest, destPos, length) {
    var j = destPos;
    var end = srcPos + length;
    var destCompCls = dest.getClass().getComponentClass();
    for (var i = srcPos; i < end; i++) {
        // Check if null or castable.
        if (src.array[i] === null || src.array[i].getClass().isCastable(destCompCls)) {
            dest.array[j] = src.array[i];
        }
        else {
            thread.throwNewException('Ljava/lang/ArrayStoreException;', 'Array element in src cannot be cast to dest array type.');
            return;
        }
        j++;
    }
}
exports.arraycopyCheck = arraycopyCheck;
function initString(cl, str) {
    var carr = initCarr(cl, str);
    var strCons = cl.getResolvedClass('Ljava/lang/String;').getConstructor(null);
    var strObj = new strCons(null);
    strObj['java/lang/String/value'] = carr;
    return strObj;
}
exports.initString = initString;
function initCarr(cl, str) {
    var arrClsCons = cl.getInitializedClass(null, '[C').getConstructor(null), carr = new arrClsCons(null, str.length), carrArray = carr.array;
    for (var i = 0; i < str.length; i++) {
        carrArray[i] = str.charCodeAt(i);
    }
    return carr;
}
exports.initCarr = initCarr;
function newArrayFromClass(thread, clazz, length) {
    return new (clazz.getConstructor(thread))(thread, length);
}
exports.newArrayFromClass = newArrayFromClass;
function newArray(thread, cl, desc, length) {
    var cls = cl.getInitializedClass(thread, desc);
    return newArrayFromClass(thread, cls, length);
}
exports.newArray = newArray;
/**
 * Separate from newArray to avoid programming mistakes where newArray and newArrayFromData are conflated.
 */
function multiNewArray(thread, cl, desc, lengths) {
    var cls = cl.getInitializedClass(thread, desc);
    return new (cls.getConstructor(thread))(thread, lengths);
}
exports.multiNewArray = multiNewArray;
function newObjectFromClass(thread, clazz) {
    return new (clazz.getConstructor(thread))(thread);
}
exports.newObjectFromClass = newObjectFromClass;
function newObject(thread, cl, desc) {
    var cls = cl.getInitializedClass(thread, desc);
    return newObjectFromClass(thread, cls);
}
exports.newObject = newObject;
function getStaticFields(thread, cl, desc) {
    return cl.getInitializedClass(thread, desc).getConstructor(thread);
}
exports.getStaticFields = getStaticFields;
function newArrayFromDataWithClass(thread, cls, data) {
    var arr = newArrayFromClass(thread, cls, 0);
    arr.array = data;
    return arr;
}
exports.newArrayFromDataWithClass = newArrayFromDataWithClass;
function newArrayFromData(thread, cl, desc, data) {
    var arr = newArray(thread, cl, desc, 0);
    arr.array = data;
    return arr;
}
exports.newArrayFromData = newArrayFromData;
/**
 * Returns the boxed class name of the given primitive type.
 */
function boxClassName(primType) {
    switch (primType) {
        case 'B':
            return 'Ljava/lang/Byte;';
        case 'C':
            return 'Ljava/lang/Character;';
        case 'D':
            return 'Ljava/lang/Double;';
        case 'F':
            return 'Ljava/lang/Float;';
        case 'I':
            return 'Ljava/lang/Integer;';
        case 'J':
            return 'Ljava/lang/Long;';
        case 'S':
            return 'Ljava/lang/Short;';
        case 'Z':
            return 'Ljava/lang/Boolean;';
        case 'V':
            return 'Ljava/lang/Void;';
        default:
            throw new Error("Tried to box a non-primitive class: " + this.className);
    }
}
exports.boxClassName = boxClassName;
/**
 * Boxes the given primitive value.
 */
function boxPrimitiveValue(thread, type, val) {
    // XXX: We assume Integer for typing purposes only; avoids a huge union type.
    var primCls = thread.getBsCl().getInitializedClass(thread, boxClassName(type)), primClsCons = primCls.getConstructor(thread);
    return primClsCons.box(val);
}
exports.boxPrimitiveValue = boxPrimitiveValue;
/**
 * Boxes the given arguments into an Object[].
 *
 * @param descriptor The descriptor at the *call site*.
 * @param data The actual arguments for this function call.
 * @param isStatic If false, disregard the first type in the descriptor, as it is the 'this' argument.
 */
function boxArguments(thread, objArrCls, descriptor, data, isStatic, skipArgs) {
    if (skipArgs === void 0) { skipArgs = 0; }
    var paramTypes = getTypes(descriptor), boxedArgs = newArrayFromClass(thread, objArrCls, paramTypes.length - (isStatic ? 1 : 2) - skipArgs), i, j = 0, boxedArgsArr = boxedArgs.array, type;
    // Ignore return value.
    paramTypes.pop();
    if (!isStatic) {
        // Ignore 'this' argument.
        paramTypes.shift();
    }
    if (skipArgs > 0) {
        // Ignore regular arguments
        paramTypes = paramTypes.slice(skipArgs);
        data = data.slice(skipArgs);
    }
    for (i = 0; i < paramTypes.length; i++) {
        type = paramTypes[i];
        switch (type[0]) {
            case '[':
            case 'L':
                // Single argument slot, no boxing required.
                boxedArgsArr[i] = data[j];
                break;
            case 'J':
            case 'D':
                boxedArgsArr[i] = boxPrimitiveValue(thread, type, data[j]);
                j++;
                break;
            default:
                boxedArgsArr[i] = boxPrimitiveValue(thread, type, data[j]);
                break;
        }
        j++;
    }
    return boxedArgs;
}
exports.boxArguments = boxArguments;
function forwardResult(thread) {
    return function (e, rv) {
        if (e) {
            thread.throwException(e);
        }
        else {
            thread.asyncReturn(rv);
        }
    };
}
exports.forwardResult = forwardResult;
//# sourceMappingURL=data:application/json;base64,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