'use strict';
var option_parser_1 = require('../src/option_parser');
var path = require('path');
var fs = require('fs');
var util_1 = require('../src/util');
var classpath_1 = require('../src/classpath');
var ClassData_1 = require('../src/ClassData');
var JDKInfo = require('../vendor/java_home/jdk.json');
var enums_1 = require('../src/enums');
var async = require('async');
require('source-map-support').install({ handleUncaughtExceptions: true });
var classpath = null, parser = new option_parser_1.OptionParser({
        default: {
            classpath: {
                type: 3,
                alias: 'cp',
                optDesc: ' <class search path of directories and zip/jar files>',
                desc: 'A : separated list of directories, JAR archives, and ZIP archives to search for class files.'
            },
            help: {
                alias: '?',
                desc: 'print this help message'
            },
            directory: {
                type: 3,
                alias: 'd',
                optDesc: ' <directory>',
                desc: 'Output directory'
            },
            javascript: {
                alias: 'js',
                desc: 'Generate JavaScript templates (Default is true)'
            },
            typescript: {
                alias: 'ts',
                desc: 'Generate TypeScript templates'
            },
            'doppiojvm-path': {
                type: 3,
                optDesc: ' <path to doppiojvm module>',
                alias: 'dpath',
                desc: 'Path to the doppiojvm module. Defaults to \'doppiojvm\', referring to the NPM module.'
            },
            'force_headers': {
                type: 3,
                optDesc: ':[<classname>:]',
                alias: 'f',
                desc: '[TypeScript only] Forces doppioh to generate TypeScript headers for specified JVM classes'
            },
            'headers_only': {
                alias: 'ho',
                desc: '[TypeScript only] Only generate header file.'
            }
        }
    });
function printEraseableLine(line) {
    if (process.stdout['clearLine']) {
        process.stdout.clearLine();
        process.stdout.cursorTo(0);
        process.stdout.write(line);
    }
}
function printHelp() {
    process.stdout.write('Usage: doppioh [flags] class_or_package_names\n' + parser.help('default') + '\n');
}
var parseResults = parser.parse(process.argv.slice(2)), args = parseResults['default'];
if (args.flag('help', false) || process.argv.length === 2) {
    printHelp();
    process.exit(1);
}
var outputDirectory = args.stringOption('directory', '.');
function file2desc(fname) {
    return 'L' + fname.slice(0, fname.length - 6).replace(/\\/g, '/') + ';';
}
var cache = {};
function getClasses(item) {
    var rv = [];
    var cpItems = [];
    for (var i = 0; i < classpath.length; i++) {
        var searchedItem = item;
        var stat = classpath[i].tryStatSync(searchedItem);
        if (!stat) {
            searchedItem = item + '.class';
            stat = classpath[i].tryStatSync(searchedItem);
        }
        if (!stat) {
            continue;
        } else {
            if (!stat.isDirectory()) {
                if (path.extname(searchedItem) === '.class') {
                    rv.push(file2desc(searchedItem));
                }
            } else {
                cpItems.push(classpath[i]);
            }
        }
    }
    if (rv.length === 0 && cpItems.length === 0) {
        throw new Error('Unable to find resource ' + item + '.');
    }
    if (cpItems.length > 0) {
        var dirStack = [item];
        while (dirStack.length > 0) {
            var dir = dirStack.pop();
            for (var i = 0; i < cpItems.length; i++) {
                var dirListing = cpItems[i].tryReaddirSync(dir);
                if (dirListing === null) {
                    continue;
                } else {
                    for (var i_1 = 0; i_1 < dirListing.length; i_1++) {
                        var item_1 = dirListing[i_1];
                        var itemPath = path.join(dir, item_1);
                        if (path.extname(itemPath) === '.class') {
                            rv.push(file2desc(itemPath));
                        } else {
                            dirStack.push(itemPath);
                        }
                    }
                }
            }
        }
    }
    return rv;
}
function loadClass(type) {
    for (var i = 0; i < classpath.length; i++) {
        var item = classpath[i];
        switch (item.hasClass(type)) {
        case enums_1.TriState.INDETERMINATE:
        case enums_1.TriState.TRUE:
            var buff = item.tryLoadClassSync(type);
            if (buff !== null) {
                return buff;
            }
            break;
        }
    }
    throw new Error('Unable to find class ' + type);
}
function findClass(descriptor) {
    if (cache[descriptor] !== undefined) {
        return cache[descriptor];
    }
    var rv;
    try {
        switch (descriptor[0]) {
        case 'L':
            rv = new ClassData_1.ReferenceClassData(loadClass(util_1.descriptor2typestr(descriptor)));
            var superClassRef = rv.getSuperClassReference(), interfaceClassRefs = rv.getInterfaceClassReferences(), superClass = null, interfaceClasses = [];
            if (superClassRef !== null) {
                superClass = findClass(superClassRef.name);
            }
            if (interfaceClassRefs.length > 0) {
                interfaceClasses = interfaceClassRefs.map(function (iface) {
                    return findClass(iface.name);
                });
            }
            rv.setResolved(superClass, interfaceClasses);
            break;
        case '[':
            rv = new ClassData_1.ArrayClassData(descriptor.slice(1), null);
            break;
        default:
            rv = new ClassData_1.PrimitiveClassData(descriptor, null);
            break;
        }
        cache[descriptor] = rv;
        return rv;
    } catch (e) {
        throw new Error('Unable to read class file for ' + descriptor + ': ' + e + '\n' + e.stack);
    }
}
function processClassData(stream, template, classData) {
    var fixedClassName = classData.getInternalName().replace(/\//g, '_'), nativeFound = false;
    fixedClassName = fixedClassName.substring(1, fixedClassName.length - 1);
    var methods = classData.getMethods();
    methods.forEach(function (method) {
        if (method.accessFlags.isNative()) {
            if (!nativeFound) {
                template.classStart(stream, fixedClassName);
                nativeFound = true;
            }
            template.method(stream, classData.getInternalName(), method.signature, method.accessFlags.isStatic(), method.parameterTypes, method.returnType);
        }
    });
    if (nativeFound) {
        template.classEnd(stream, fixedClassName);
    }
}
var TSDeclarationFile = function () {
    function TSDeclarationFile(doppiojvmPath, outputPath) {
        var _this = this;
        this.headerCount = 0;
        this.headerSet = {};
        this.generateQueue = [];
        this.headerPath = path.resolve(outputPath, 'JVMTypes.d.ts');
        this.doppiojvmPath = path.relative(outputPath, doppiojvmPath);
        try {
            var existingHeaders = fs.readFileSync(this.headerPath).toString(), searchIdx = 0, clsName;
            while ((searchIdx = existingHeaders.indexOf('export class ', searchIdx)) > -1) {
                clsName = existingHeaders.slice(searchIdx + 13, existingHeaders.indexOf(' ', searchIdx + 13));
                if (clsName.indexOf('JVMArray') !== 0) {
                    this.generateClassDefinition(this.tstype2jvmtype(clsName));
                }
                searchIdx++;
            }
            searchIdx = 0;
            while ((searchIdx = existingHeaders.indexOf('export interface ', searchIdx)) > -1) {
                clsName = existingHeaders.slice(searchIdx + 17, existingHeaders.indexOf(' ', searchIdx + 17));
                this.generateClassDefinition(this.tstype2jvmtype(clsName));
                searchIdx++;
            }
        } catch (e) {
        }
        this.headerStream = fs.createWriteStream(this.headerPath);
        this.headersStart();
        this.generateArrayDefinition();
        this.generateMiscDefinitions();
        this.generateClassDefinition('Ljava/lang/Throwable;');
        if (args.stringOption('force_headers', null)) {
            var clses = args.stringOption('force_headers', null).split(':');
            clses.forEach(function (clsName) {
                _this.generateClassDefinition(util_1.int_classname(clsName));
            });
        }
    }
    TSDeclarationFile.prototype.headersStart = function () {
        this.headerStream.write('// TypeScript declaration file for JVM types. Automatically generated by doppioh.\n// http://github.com/plasma-umass/doppio\nimport * as DoppioJVM from \'' + this.doppiojvmPath.replace(/\\/g, '/') + '\';\nimport JVMThread = DoppioJVM.VM.Threading.JVMThread;\nimport Long = DoppioJVM.VM.Long;\nimport ClassData = DoppioJVM.VM.ClassFile.ClassData;\nimport ArrayClassData = DoppioJVM.VM.ClassFile.ArrayClassData;\nimport ReferenceClassData = DoppioJVM.VM.ClassFile.ReferenceClassData;\nimport Monitor = DoppioJVM.VM.Monitor;\nimport ClassLoader = DoppioJVM.VM.ClassFile.ClassLoader;\nimport Interfaces = DoppioJVM.VM.Interfaces;\n\ndeclare module JVMTypes {\n');
    };
    TSDeclarationFile.prototype.headersEnd = function () {
        this._processGenerateQueue();
        printEraseableLine('Processed ' + this.headerCount + ' classes.\n');
        this.headerStream.end('}\nexport = JVMTypes;\n', function () {
        });
    };
    TSDeclarationFile.prototype._processHeader = function (cls) {
        var _this = this;
        var desc = cls.getInternalName(), interfaces = cls.getInterfaceClassReferences().map(function (iface) {
                return iface.name;
            }), superClass = cls.getSuperClassReference(), methods = cls.getMethods().concat(cls.getMirandaAndDefaultMethods()), fields = cls.getFields(), methodsSeen = {}, injectedFields = cls.getInjectedFields(), injectedMethods = cls.getInjectedMethods(), injectedStaticMethods = cls.getInjectedStaticMethods();
        printEraseableLine('[' + this.headerCount++ + '] Processing header for ' + util_1.descriptor2typestr(desc) + '...');
        if (cls.accessFlags.isInterface()) {
            this.headerStream.write('  export interface ' + this.jvmtype2tstype(desc, false));
        } else {
            this.headerStream.write('  export class ' + this.jvmtype2tstype(desc, false));
        }
        if (superClass !== null) {
            this.headerStream.write(' extends ' + this.jvmtype2tstype(superClass.name, false));
        }
        if (interfaces.length > 0) {
            if (cls.accessFlags.isInterface()) {
                this.headerStream.write(', ');
            } else {
                this.headerStream.write(' implements ');
            }
            this.headerStream.write('' + interfaces.map(function (ifaceName) {
                return _this.jvmtype2tstype(ifaceName, false);
            }).join(', '));
        }
        this.headerStream.write(' {\n');
        Object.keys(injectedFields).forEach(function (name) {
            return _this._outputInjectedField(name, injectedFields[name], _this.headerStream);
        });
        Object.keys(injectedMethods).forEach(function (name) {
            return _this._outputInjectedMethod(name, injectedMethods[name], _this.headerStream);
        });
        Object.keys(injectedStaticMethods).forEach(function (name) {
            return _this._outputInjectedStaticMethod(name, injectedStaticMethods[name], _this.headerStream);
        });
        fields.forEach(function (f) {
            return _this._outputField(f, _this.headerStream);
        });
        methods.forEach(function (m) {
            return _this._outputMethod(m, _this.headerStream);
        });
        cls.getUninheritedDefaultMethods().forEach(function (m) {
            return _this._outputMethod(m, _this.headerStream);
        });
        this.headerStream.write('  }\n');
    };
    TSDeclarationFile.prototype.jvmtype2tstype = function (desc, prefix) {
        if (prefix === void 0) {
            prefix = true;
        }
        switch (desc[0]) {
        case '[':
            return (prefix ? 'JVMTypes.' : '') + ('JVMArray<' + this.jvmtype2tstype(desc.slice(1), prefix) + '>');
        case 'L':
            this.generateClassDefinition(desc);
            return (prefix ? 'JVMTypes.' : '') + util_1.descriptor2typestr(desc).replace(/_/g, '__').replace(/\//g, '_');
        case 'J':
            return 'Long';
        case 'V':
            return 'void';
        default:
            return 'number';
        }
    };
    TSDeclarationFile.prototype.tstype2jvmtype = function (tsType) {
        if (tsType.indexOf('JVMArray') === 0) {
            return '[' + this.tstype2jvmtype(tsType.slice(9, tsType.length - 1));
        } else if (tsType === 'number') {
            throw new Error('Ambiguous.');
        } else if (tsType === 'void') {
            return 'V';
        } else {
            return 'L' + tsType.replace(/_/g, '/').replace(/\/\//g, '_') + ';';
        }
    };
    TSDeclarationFile.prototype.generateClassDefinition = function (desc) {
        if (this.headerSet[desc] !== undefined || util_1.is_primitive_type(desc)) {
            return;
        } else if (desc[0] === '[') {
            return this.generateClassDefinition(desc.slice(1));
        } else {
            this.headerSet[desc] = true;
            this.generateQueue.push(findClass(desc));
        }
    };
    TSDeclarationFile.prototype._outputMethod = function (m, stream, nonVirtualOnly) {
        var _this = this;
        if (nonVirtualOnly === void 0) {
            nonVirtualOnly = false;
        }
        var argTypes = m.parameterTypes, rType = m.returnType, args = '', cbSig = 'e?: java_lang_Throwable' + (rType === 'V' ? '' : ', rv?: ' + this.jvmtype2tstype(rType, false)), methodSig, methodFlags = 'public' + (m.accessFlags.isStatic() ? ' static' : '');
        if (argTypes.length > 0) {
            args = 'args: [' + argTypes.map(function (type, i) {
                return '' + _this.jvmtype2tstype(type, false) + (type === 'J' || type === 'D' ? ', any' : '');
            }).join(', ') + ']';
        } else {
            args = 'args: {}[]';
        }
        methodSig = '(thread: JVMThread, ' + args + ', cb?: (' + cbSig + ') => void): void';
        if (m.cls.accessFlags.isInterface()) {
            if (m.accessFlags.isStatic()) {
            } else {
                stream.write('    "' + m.signature + '"' + methodSig + ';\n');
            }
        } else {
            if (!nonVirtualOnly) {
                stream.write('    ' + methodFlags + ' "' + m.signature + '"' + methodSig + ';\n');
            }
            stream.write('    ' + methodFlags + ' "' + m.fullSignature + '"' + methodSig + ';\n');
        }
    };
    TSDeclarationFile.prototype._outputField = function (f, stream) {
        var fieldType = f.rawDescriptor, cls = f.cls;
        if (cls.accessFlags.isInterface()) {
            return;
        }
        if (f.accessFlags.isStatic()) {
            stream.write('    public static "' + util_1.descriptor2typestr(cls.getInternalName()) + '/' + f.name + '": ' + this.jvmtype2tstype(fieldType, false) + ';\n');
        } else {
            stream.write('    public "' + util_1.descriptor2typestr(cls.getInternalName()) + '/' + f.name + '": ' + this.jvmtype2tstype(fieldType, false) + ';\n');
        }
    };
    TSDeclarationFile.prototype._outputInjectedField = function (name, type, stream) {
        stream.write('    public ' + name + ': ' + type + ';\n');
    };
    TSDeclarationFile.prototype._outputInjectedMethod = function (name, type, stream) {
        stream.write('    public ' + name + type + ';\n');
    };
    TSDeclarationFile.prototype._outputInjectedStaticMethod = function (name, type, stream) {
        stream.write('    public static ' + name + type + ';\n');
    };
    TSDeclarationFile.prototype._processGenerateQueue = function () {
        while (this.generateQueue.length > 0) {
            this._processHeader(this.generateQueue.pop());
        }
    };
    TSDeclarationFile.prototype.generateArrayDefinition = function () {
        this.headerStream.write('  export class JVMArray<T> extends java_lang_Object {\n    /**\n     * NOTE: Our arrays are either JS arrays, or TypedArrays for primitive\n     * types.\n     */\n    public array: T[];\n    public getClass(): ArrayClassData<T>;\n    /**\n     * Create a new JVM array of this type that starts at start, and ends at\n     * end. End defaults to the end of the array.\n     */\n    public slice(start: number, end?: number): JVMArray<T>;\n  }\n');
    };
    TSDeclarationFile.prototype.generateMiscDefinitions = function () {
        this.headerStream.write('  // Basic, valid JVM types.\n  export type BasicType = number | java_lang_Object | Long;\n  export type JVMFunction = (thread: JVMThread, args: BasicType[], cb: (e?: JVMTypes.java_lang_Object, rv?: BasicType) => void) => void;\n');
    };
    return TSDeclarationFile;
}();
var TSTemplate = function () {
    function TSTemplate(doppiojvmPath, outputPath) {
        this.classesSeen = [];
        this.doppiojvmPath = path.relative(outputPath, doppiojvmPath);
        if (TSTemplate.declFile === null) {
            TSTemplate.declFile = new TSDeclarationFile(doppiojvmPath, outputPath);
        }
    }
    TSTemplate.prototype.getExtension = function () {
        return 'ts';
    };
    TSTemplate.prototype.fileStart = function (stream) {
        stream.write('import JVMTypes = require("./JVMTypes");\nimport DoppioJVM = require(\'' + this.doppiojvmPath.replace(/\\/g, '/') + '\');\nimport JVMThread = DoppioJVM.VM.Threading.JVMThread;\nimport Long = DoppioJVM.VM.Long;\ndeclare var registerNatives: (natives: any) => void;\n');
    };
    TSTemplate.prototype.fileEnd = function (stream) {
        var i;
        stream.write('\n// Export line. This is what DoppioJVM sees.\nregisterNatives({');
        for (i = 0; i < this.classesSeen.length; i++) {
            var kls = this.classesSeen[i];
            if (i > 0)
                stream.write(',');
            stream.write('\n  \'' + kls.replace(/_/g, '/') + '\': ' + kls);
        }
        stream.write('\n});\n');
    };
    TSTemplate.prototype.classStart = function (stream, className) {
        stream.write('\nclass ' + className + ' {\n');
        this.classesSeen.push(className);
        TSTemplate.declFile.generateClassDefinition('L' + className.replace(/_/g, '/') + ';');
    };
    TSTemplate.prototype.classEnd = function (stream, className) {
        stream.write('\n}\n');
    };
    TSTemplate.prototype.method = function (stream, classDesc, methodName, isStatic, argTypes, rType) {
        var trueRtype = TSTemplate.declFile.jvmtype2tstype(rType), rval = '';
        if (trueRtype === 'number') {
            rval = '0';
        } else if (trueRtype !== 'void') {
            rval = 'null';
        }
        argTypes.concat([rType]).forEach(function (type) {
            TSTemplate.declFile.generateClassDefinition(type);
        });
        stream.write('\n  public static \'' + methodName + '\'(thread: JVMThread' + (isStatic ? '' : ', javaThis: ' + TSTemplate.declFile.jvmtype2tstype(classDesc)) + (argTypes.length === 0 ? '' : ', ' + argTypes.map(function (type, i) {
            return 'arg' + i + ': ' + TSTemplate.declFile.jvmtype2tstype(type);
        }).join(', ')) + '): ' + TSTemplate.declFile.jvmtype2tstype(rType) + ' {\n    thread.throwNewException(\'Ljava/lang/UnsatisfiedLinkError;\', \'Native method not implemented.\');' + (rval !== '' ? '\n    return ' + rval + ';' : '') + '\n  }\n');
    };
    TSTemplate.declFile = null;
    return TSTemplate;
}();
var JSTemplate = function () {
    function JSTemplate() {
        this.firstMethod = true;
        this.firstClass = true;
    }
    JSTemplate.prototype.getExtension = function () {
        return 'js';
    };
    JSTemplate.prototype.fileStart = function (stream) {
        stream.write('// This entire object is exported. Feel free to define private helper functions above it.\nregisterNatives({');
    };
    JSTemplate.prototype.fileEnd = function (stream) {
        stream.write('\n});\n');
    };
    JSTemplate.prototype.classStart = function (stream, className) {
        this.firstMethod = true;
        if (this.firstClass) {
            this.firstClass = false;
        } else {
            stream.write(',\n');
        }
        stream.write('\n  \'' + className.replace(/_/g, '/') + '\': {\n');
    };
    JSTemplate.prototype.classEnd = function (stream, className) {
        stream.write('\n\n  }');
    };
    JSTemplate.prototype.method = function (stream, classDesc, methodName, isStatic, argTypes, rType) {
        var argSig = 'thread', i;
        if (!isStatic) {
            argSig += ', javaThis';
        }
        for (i = 0; i < argTypes.length; i++) {
            argSig += ', arg' + i;
        }
        if (this.firstMethod) {
            this.firstMethod = false;
        } else {
            stream.write(',\n');
        }
        stream.write('\n    \'' + methodName + '\': function(' + argSig + ') {');
        stream.write('\n      thread.throwNewException(\'Ljava/lang/UnsatisfiedLinkError;\', \'Native method not implemented.\');');
        stream.write('\n    }');
    };
    return JSTemplate;
}();
var JAVA_HOME = path.resolve(__dirname, '../vendor/java_home');
var classpathPaths = JDKInfo.classpath.map(function (item) {
    return path.resolve(JAVA_HOME, item);
}).concat(args.stringOption('classpath', '.').split(':'));
var classNames = args.unparsedArgs();
if (classNames.length === 0) {
    throw new Error('Must specify a class name.');
}
if (!fs.existsSync(outputDirectory)) {
    fs.mkdirSync(outputDirectory);
}
classpath_1.ClasspathFactory(JAVA_HOME, classpathPaths, function (items) {
    async.each(items, function (item, cb) {
        if (item instanceof classpath_1.UnindexedClasspathJar || item instanceof classpath_1.IndexedClasspathJar) {
            item.loadJar(cb);
        } else {
            cb();
        }
    }, function (e) {
        if (e) {
            throw e;
        }
        classpath = items;
        try {
            classNames.forEach(function (className) {
                var targetName = className.replace(/\//g, '_').replace(/\./g, '_'), targetPath = className.replace(/\./g, '/');
                var template = args.flag('typescript', false) ? new TSTemplate(args.stringOption('doppiojvm-path', 'doppiojvm'), outputDirectory) : new JSTemplate();
                var stream = fs.createWriteStream(path.join(outputDirectory, targetName + '.' + template.getExtension()));
                template.fileStart(stream);
                var classes = getClasses(targetPath);
                for (var i = 0; i < classes.length; i++) {
                    var desc = classes[i];
                    processClassData(stream, template, findClass(desc));
                }
                template.fileEnd(stream);
                stream.end(new Buffer(''), function () {
                });
                if (args.flag('typescript', false) && args.flag('headers_only', false)) {
                    fs.unlinkSync(path.join(outputDirectory, targetName + '.' + template.getExtension()));
                }
            });
            if (args.flag('typescript', false)) {
                TSTemplate.declFile.headersEnd();
            }
        } catch (e) {
            console.error('Encountered error: ' + e);
        }
    });
});
//# sourceMappingURL=doppioh.js.map