'use strict';
var fs = require('fs');
var path = require('path');
var https = require('https');
var rimraf = require('rimraf');
var glob = require('glob');
var gunzip = require('gunzip-maybe');
var tarFs = require('tar-fs');
var JDK_URL = 'https://github.com/plasma-umass/doppio_jcl/releases/download/v3.2/java_home.tar.gz';
var JDK_PATH = path.resolve(__dirname, '..', '..', '..', 'vendor');
var JDK_FOLDER = 'java_home';
function doesJDKExist() {
    var jdkInfoPath = path.resolve(JDK_PATH, JDK_FOLDER, 'jdk.json');
    if (fs.existsSync(jdkInfoPath)) {
        try {
            var jdkInfo = JSON.parse(fs.readFileSync(jdkInfoPath).toString());
            return jdkInfo['url'] === JDK_URL;
        } catch (e) {
            return false;
        }
    }
    return false;
}
function mkdirp(p) {
    p = path.resolve(p);
    if (!fs.existsSync(p)) {
        mkdirp(path.dirname(p));
        fs.mkdirSync(p);
    }
}
function downloadJDK(url, destPath, cb) {
    var ended = false;
    mkdirp(destPath);
    https.get(url, function (res) {
        if (res.statusCode === 302) {
            ended = true;
            downloadJDK(res.headers['location'], destPath, cb);
            return;
        }
        var contentLength = parseInt(res.headers['content-length']);
        var progressBytes = 0;
        var startTime = new Date().getTime();
        function printStatus() {
            var percent = '??';
            if (!isNaN(contentLength)) {
                percent = (progressBytes / contentLength * 100).toFixed(0);
            }
            var lastPrint = new Date().getTime();
            var dlSoFar = progressBytes >> 10;
            var rate = dlSoFar / ((lastPrint - startTime) / 1000);
            console.log('[' + percent + '%] Received ' + dlSoFar + ' KB [' + rate.toFixed(2) + ' KB/s]');
        }
        var interval = setInterval(function () {
            printStatus();
        }, 5000);
        function end(err) {
            if (!ended) {
                ended = true;
                clearInterval(interval);
                cb(err);
            }
        }
        res.pipe(gunzip()).pipe(tarFs.extract(destPath)).on('error', end).on('finish', end);
        res.on('data', function (d) {
            progressBytes += d.length;
        });
    }).on('error', function (err) {
        if (!ended) {
            ended = true;
            cb(err);
        }
    });
}
function writeJdkJson() {
    var globSearch = path.resolve(JDK_PATH, JDK_FOLDER).replace(/\\/g, '/') + '/lib/*.jar';
    glob(globSearch, function (e, classpath) {
        if (e) {
            console.error('Failed to locate JDK JAR items: ' + e);
            process.exit(1);
        }
        var rtIndex = -1;
        classpath = classpath.map(function (item, i) {
            switch (path.basename(item)) {
            case 'rt.jar':
                rtIndex = i;
                break;
            }
            return path.relative(path.resolve(JDK_PATH, JDK_FOLDER), item);
        });
        var rt = classpath[rtIndex];
        classpath.splice(rtIndex, 1);
        classpath.unshift(rt);
        var jdkJson = {
            url: JDK_URL,
            classpath: classpath.map(function (cpItem) {
                return cpItem.replace(/\\/g, '/');
            })
        };
        if (jdkJson.classpath.filter(function (cpItem) {
                return path.basename(cpItem) === 'doppio.jar';
            }).length === 0) {
            jdkJson.classpath.push('lib/doppio.jar');
        }
        fs.writeFileSync(path.resolve(JDK_PATH, JDK_FOLDER, 'jdk.json'), new Buffer(JSON.stringify(jdkJson), 'utf8'));
        fs.writeFileSync(path.resolve(JDK_PATH, JDK_FOLDER, 'jdk.json.d.ts'), new Buffer('declare let JDKInfo: {\n  url: string;\n  classpath: string[];\n};\nexport = JDKInfo;\n', 'utf8'));
    });
}
if (!doesJDKExist()) {
    console.log('JDK is out of date! Removing old JDK...');
    rimraf(path.resolve(JDK_PATH, JDK_FOLDER), function (err) {
        if (err) {
            console.error('Error removing old JDK: ' + err);
            process.exit(1);
        }
        console.log('Downloading new JDK...');
        downloadJDK(JDK_URL, JDK_PATH, function (err) {
            if (err) {
                console.error('Failed to download JDK: ' + err + '.');
                process.exit(1);
            } else {
                console.log('Successfully downloaded JDK.');
                writeJdkJson();
            }
        });
    });
} else {
    console.log('JDK is up-to-date.');
}
//# sourceMappingURL=download_jdk.js.map