'use strict';
var __extends = this && this.__extends || function (d, b) {
    for (var p in b)
        if (b.hasOwnProperty(p))
            d[p] = b[p];
    function __() {
        this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var ClassData_1 = require('./ClassData');
var ClassLock_1 = require('./ClassLock');
var classpath_1 = require('./classpath');
var enums_1 = require('./enums');
var util_1 = require('./util');
var logging = require('./logging');
var assert_1 = require('./assert');
var debug = logging.debug;
var error = logging.error;
var ClassLocks = function () {
    function ClassLocks() {
        this.locks = {};
    }
    ClassLocks.prototype.tryLock = function (typeStr, thread, cb) {
        if (typeof this.locks[typeStr] === 'undefined') {
            this.locks[typeStr] = new ClassLock_1['default']();
        }
        return this.locks[typeStr].tryLock(thread, cb);
    };
    ClassLocks.prototype.unlock = function (typeStr, cdata) {
        this.locks[typeStr].unlock(cdata);
        delete this.locks[typeStr];
    };
    ClassLocks.prototype.getOwner = function (typeStr) {
        if (this.locks[typeStr]) {
            return this.locks[typeStr].getOwner();
        }
        return null;
    };
    return ClassLocks;
}();
var ClassLoader = function () {
    function ClassLoader(bootstrap) {
        this.bootstrap = bootstrap;
        this.loadedClasses = {};
        this.loadClassLocks = new ClassLocks();
    }
    ClassLoader.prototype.getLoadedClassNames = function () {
        return Object.keys(this.loadedClasses);
    };
    ClassLoader.prototype.addClass = function (typeStr, classData) {
        assert_1['default'](this.loadedClasses[typeStr] != null ? this.loadedClasses[typeStr] === classData : true);
        this.loadedClasses[typeStr] = classData;
    };
    ClassLoader.prototype.getClass = function (typeStr) {
        return this.loadedClasses[typeStr];
    };
    ClassLoader.prototype.defineClass = function (thread, typeStr, data, protectionDomain) {
        try {
            var classData = new ClassData_1.ReferenceClassData(data, protectionDomain, this);
            this.addClass(typeStr, classData);
            if (this instanceof BootstrapClassLoader) {
                ;
            } else {
                ;
            }
            return classData;
        } catch (e) {
            if (thread === null) {
                error('JVM initialization failed: ' + e);
                error(e.stack);
            } else {
                thread.throwNewException('Ljava/lang/ClassFormatError;', e);
            }
            return null;
        }
    };
    ClassLoader.prototype.defineArrayClass = function (typeStr) {
        assert_1['default'](this.getLoadedClass(util_1.get_component_type(typeStr)) != null);
        var arrayClass = new ClassData_1.ArrayClassData(util_1.get_component_type(typeStr), this);
        this.addClass(typeStr, arrayClass);
        return arrayClass;
    };
    ClassLoader.prototype.getLoadedClass = function (typeStr) {
        var cls = this.loadedClasses[typeStr];
        if (cls != null) {
            return cls;
        } else {
            if (util_1.is_primitive_type(typeStr)) {
                return this.bootstrap.getPrimitiveClass(typeStr);
            } else if (util_1.is_array_type(typeStr)) {
                var component = this.getLoadedClass(util_1.get_component_type(typeStr));
                if (component != null) {
                    var componentCl = component.getLoader();
                    if (componentCl === this) {
                        return this.defineArrayClass(typeStr);
                    } else {
                        cls = componentCl.getLoadedClass(typeStr);
                        this.addClass(typeStr, cls);
                        return cls;
                    }
                }
            }
            return null;
        }
    };
    ClassLoader.prototype.getResolvedClass = function (typeStr) {
        var cls = this.getLoadedClass(typeStr);
        if (cls !== null) {
            if (cls.isResolved() || cls.tryToResolve()) {
                return cls;
            } else {
                return null;
            }
        } else {
            return null;
        }
    };
    ClassLoader.prototype.getInitializedClass = function (thread, typeStr) {
        var cls = this.getLoadedClass(typeStr);
        if (cls !== null) {
            if (cls.isInitialized(thread) || cls.tryToInitialize()) {
                return cls;
            } else {
                return null;
            }
        } else {
            return cls;
        }
    };
    ClassLoader.prototype.loadClass = function (thread, typeStr, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        var cdata = this.getLoadedClass(typeStr);
        if (cdata) {
            setImmediate(function () {
                cb(cdata);
            });
        } else {
            if (this.loadClassLocks.tryLock(typeStr, thread, cb)) {
                if (util_1.is_reference_type(typeStr)) {
                    this._loadClass(thread, typeStr, function (cdata) {
                        _this.loadClassLocks.unlock(typeStr, cdata);
                    }, explicit);
                } else {
                    this.loadClass(thread, util_1.get_component_type(typeStr), function (cdata) {
                        if (cdata != null) {
                            _this.loadClassLocks.unlock(typeStr, _this.getLoadedClass(typeStr));
                        }
                    }, explicit);
                }
            }
        }
    };
    ClassLoader.prototype.resolveClasses = function (thread, typeStrs, cb) {
        var _this = this;
        var classes = {};
        util_1.asyncForEach(typeStrs, function (typeStr, next_item) {
            _this.resolveClass(thread, typeStr, function (cdata) {
                if (cdata === null) {
                    next_item('Error resolving class: ' + typeStr);
                } else {
                    classes[typeStr] = cdata;
                    next_item();
                }
            });
        }, function (err) {
            if (err) {
                cb(null);
            } else {
                cb(classes);
            }
        });
    };
    ClassLoader.prototype.resolveClass = function (thread, typeStr, cb, explicit) {
        if (explicit === void 0) {
            explicit = true;
        }
        this.loadClass(thread, typeStr, function (cdata) {
            if (cdata === null || cdata.isResolved()) {
                setImmediate(function () {
                    cb(cdata);
                });
            } else {
                cdata.resolve(thread, cb, explicit);
            }
        }, explicit);
    };
    ClassLoader.prototype.initializeClass = function (thread, typeStr, cb, explicit) {
        if (explicit === void 0) {
            explicit = true;
        }
        this.resolveClass(thread, typeStr, function (cdata) {
            if (cdata === null || cdata.isInitialized(thread)) {
                setImmediate(function () {
                    cb(cdata);
                });
            } else {
                assert_1['default'](util_1.is_reference_type(typeStr));
                cdata.initialize(thread, cb, explicit);
            }
        }, explicit);
    };
    ClassLoader.prototype.throwClassNotFoundException = function (thread, typeStr, explicit) {
        thread.throwNewException(explicit ? 'Ljava/lang/ClassNotFoundException;' : 'Ljava/lang/NoClassDefFoundError;', 'Cannot load class: ' + util_1.ext_classname(typeStr));
    };
    return ClassLoader;
}();
exports.ClassLoader = ClassLoader;
var BootstrapClassLoader = function (_super) {
    __extends(BootstrapClassLoader, _super);
    function BootstrapClassLoader(javaHome, classpath, cb) {
        var _this = this;
        _super.call(this, null);
        this.bootstrap = this;
        this.classpath = null;
        this.loadedPackages = {};
        classpath_1.ClasspathFactory(javaHome, classpath, function (items) {
            _this.classpath = items.reverse();
            cb();
        });
    }
    BootstrapClassLoader.prototype._registerLoadedClass = function (clsType, cpItem) {
        var pkgName = clsType.slice(0, clsType.lastIndexOf('/')), itemLoader = this.loadedPackages[pkgName];
        if (!itemLoader) {
            this.loadedPackages[pkgName] = [cpItem];
        } else if (itemLoader[0] !== cpItem && itemLoader.indexOf(cpItem) === -1) {
            itemLoader.push(cpItem);
        }
    };
    BootstrapClassLoader.prototype.getPackages = function () {
        var _this = this;
        return Object.keys(this.loadedPackages).map(function (pkgName) {
            return [
                pkgName,
                _this.loadedPackages[pkgName].map(function (item) {
                    return item.getPath();
                })
            ];
        });
    };
    BootstrapClassLoader.prototype.getPrimitiveClass = function (typeStr) {
        var cdata = this.getClass(typeStr);
        if (cdata == null) {
            cdata = new ClassData_1.PrimitiveClassData(typeStr, this);
            this.addClass(typeStr, cdata);
        }
        return cdata;
    };
    BootstrapClassLoader.prototype._loadClass = function (thread, typeStr, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        ;
        assert_1['default'](util_1.is_reference_type(typeStr));
        var clsFilePath = util_1.descriptor2typestr(typeStr), cPathLen = this.classpath.length, toSearch = [], clsData;
        searchLoop:
            for (var i = 0; i < cPathLen; i++) {
                var item = this.classpath[i];
                switch (item.hasClass(clsFilePath)) {
                case enums_1.TriState.INDETERMINATE:
                    toSearch.push(item);
                    break;
                case enums_1.TriState.TRUE:
                    toSearch.push(item);
                    break searchLoop;
                }
            }
        util_1.asyncFind(toSearch, function (pItem, callback) {
            pItem.loadClass(clsFilePath, function (err, data) {
                if (err) {
                    callback(false);
                } else {
                    clsData = data;
                    callback(true);
                }
            });
        }, function (pItem) {
            if (pItem) {
                var cls = _this.defineClass(thread, typeStr, clsData, null);
                if (cls !== null) {
                    _this._registerLoadedClass(clsFilePath, pItem);
                }
                cb(cls);
            } else {
                ;
                _this.throwClassNotFoundException(thread, typeStr, explicit);
                cb(null);
            }
        });
    };
    BootstrapClassLoader.prototype.getLoadedClassFiles = function () {
        var loadedClasses = this.getLoadedClassNames();
        return loadedClasses.filter(function (clsName) {
            return util_1.is_reference_type(clsName);
        });
    };
    BootstrapClassLoader.prototype.getLoaderObject = function () {
        return null;
    };
    BootstrapClassLoader.prototype.getClassPath = function () {
        var cpLen = this.classpath.length, cpStrings = new Array(cpLen);
        for (var i = 0; i < cpLen; i++) {
            cpStrings[i] = this.classpath[cpLen - i - 1].getPath();
        }
        return cpStrings;
    };
    BootstrapClassLoader.prototype.getClassPathItems = function () {
        return this.classpath.slice(0);
    };
    return BootstrapClassLoader;
}(ClassLoader);
exports.BootstrapClassLoader = BootstrapClassLoader;
var CustomClassLoader = function (_super) {
    __extends(CustomClassLoader, _super);
    function CustomClassLoader(bootstrap, loaderObj) {
        _super.call(this, bootstrap);
        this.loaderObj = loaderObj;
    }
    CustomClassLoader.prototype._loadClass = function (thread, typeStr, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        ;
        assert_1['default'](util_1.is_reference_type(typeStr));
        this.loaderObj['loadClass(Ljava/lang/String;)Ljava/lang/Class;'](thread, [util_1.initString(this.bootstrap, util_1.ext_classname(typeStr))], function (e, jco) {
            if (e) {
                _this.throwClassNotFoundException(thread, typeStr, explicit);
                cb(null);
            } else {
                var cls = jco.$cls;
                _this.addClass(typeStr, cls);
                cb(cls);
            }
        });
    };
    CustomClassLoader.prototype.getLoaderObject = function () {
        return this.loaderObj;
    };
    return CustomClassLoader;
}(ClassLoader);
exports.CustomClassLoader = CustomClassLoader;
//# sourceMappingURL=ClassLoader.js.map