'use strict';
var util = require('./util');
var enums_1 = require('./enums');
var assert_1 = require('./assert');
var CP_CLASSES = {};
var ConstUTF8 = function () {
    function ConstUTF8(rawBytes) {
        this.value = this.bytes2str(rawBytes);
    }
    ConstUTF8.prototype.bytes2str = function (bytes) {
        var y, z, v, w, x, charCode, idx = 0, rv = '';
        while (idx < bytes.length) {
            x = bytes[idx++] & 255;
            if (x <= 127) {
                charCode = x;
            } else if (x <= 223) {
                y = bytes[idx++];
                charCode = ((x & 31) << 6) + (y & 63);
            } else {
                y = bytes[idx++];
                z = bytes[idx++];
                charCode = ((x & 15) << 12) + ((y & 63) << 6) + (z & 63);
            }
            rv += String.fromCharCode(charCode);
        }
        return rv;
    };
    ConstUTF8.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.UTF8;
    };
    ConstUTF8.prototype.getConstant = function (thread) {
        return this.value;
    };
    ConstUTF8.prototype.isResolved = function () {
        return true;
    };
    ConstUTF8.fromBytes = function (byteStream, constantPool) {
        var strlen = byteStream.getUint16();
        return new this(byteStream.read(strlen));
    };
    ConstUTF8.size = 1;
    ConstUTF8.infoByteSize = 0;
    return ConstUTF8;
}();
exports.ConstUTF8 = ConstUTF8;
CP_CLASSES[enums_1.ConstantPoolItemType.UTF8] = ConstUTF8;
var ConstInt32 = function () {
    function ConstInt32(value) {
        this.value = value;
    }
    ConstInt32.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.INTEGER;
    };
    ConstInt32.prototype.getConstant = function (thread) {
        return this.value;
    };
    ConstInt32.prototype.isResolved = function () {
        return true;
    };
    ConstInt32.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getInt32());
    };
    ConstInt32.size = 1;
    ConstInt32.infoByteSize = 4;
    return ConstInt32;
}();
exports.ConstInt32 = ConstInt32;
CP_CLASSES[enums_1.ConstantPoolItemType.INTEGER] = ConstInt32;
var ConstFloat = function () {
    function ConstFloat(value) {
        this.value = value;
    }
    ConstFloat.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.FLOAT;
    };
    ConstFloat.prototype.getConstant = function (thread) {
        return this.value;
    };
    ConstFloat.prototype.isResolved = function () {
        return true;
    };
    ConstFloat.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getFloat());
    };
    ConstFloat.size = 1;
    ConstFloat.infoByteSize = 4;
    return ConstFloat;
}();
exports.ConstFloat = ConstFloat;
CP_CLASSES[enums_1.ConstantPoolItemType.FLOAT] = ConstFloat;
var ConstLong = function () {
    function ConstLong(value) {
        this.value = value;
    }
    ConstLong.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.LONG;
    };
    ConstLong.prototype.getConstant = function (thread) {
        return this.value;
    };
    ConstLong.prototype.isResolved = function () {
        return true;
    };
    ConstLong.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getInt64());
    };
    ConstLong.size = 2;
    ConstLong.infoByteSize = 8;
    return ConstLong;
}();
exports.ConstLong = ConstLong;
CP_CLASSES[enums_1.ConstantPoolItemType.LONG] = ConstLong;
var ConstDouble = function () {
    function ConstDouble(value) {
        this.value = value;
    }
    ConstDouble.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.DOUBLE;
    };
    ConstDouble.prototype.getConstant = function (thread) {
        return this.value;
    };
    ConstDouble.prototype.isResolved = function () {
        return true;
    };
    ConstDouble.fromBytes = function (byteStream, constantPool) {
        return new this(byteStream.getDouble());
    };
    ConstDouble.size = 2;
    ConstDouble.infoByteSize = 8;
    return ConstDouble;
}();
exports.ConstDouble = ConstDouble;
CP_CLASSES[enums_1.ConstantPoolItemType.DOUBLE] = ConstDouble;
var ClassReference = function () {
    function ClassReference(name) {
        this.cls = null;
        this.clsConstructor = null;
        this.arrayClass = null;
        this.arrayClassConstructor = null;
        this.name = name;
    }
    ClassReference.prototype.tryResolve = function (loader) {
        if (this.cls === null) {
            this.cls = loader.getResolvedClass(this.name);
        }
        return this.cls !== null;
    };
    ClassReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        if (thread !== null) {
            var currentMethod = thread.currentMethod();
            if (currentMethod !== null && this.name === currentMethod.cls.getInternalName()) {
                this.setResolved(thread, thread.currentMethod().cls);
                return cb(true);
            }
        }
        loader.resolveClass(thread, this.name, function (cdata) {
            _this.setResolved(thread, cdata);
            cb(cdata !== null);
        }, explicit);
    };
    ClassReference.prototype.setResolved = function (thread, cls) {
        this.cls = cls;
        if (cls !== null) {
            this.clsConstructor = cls.getConstructor(thread);
        }
    };
    ClassReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.CLASS;
    };
    ClassReference.prototype.getConstant = function (thread) {
        return this.cls.getClassObject(thread);
    };
    ClassReference.prototype.isResolved = function () {
        return this.cls !== null;
    };
    ClassReference.fromBytes = function (byteStream, constantPool) {
        var nameIndex = byteStream.getUint16(), cpItem = constantPool.get(nameIndex);
        assert_1['default'](cpItem.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool ClassReference type != UTF8');
        return new this(util.typestr2descriptor(cpItem.value));
    };
    ClassReference.size = 1;
    ClassReference.infoByteSize = 2;
    return ClassReference;
}();
exports.ClassReference = ClassReference;
CP_CLASSES[enums_1.ConstantPoolItemType.CLASS] = ClassReference;
var NameAndTypeInfo = function () {
    function NameAndTypeInfo(name, descriptor) {
        this.name = name;
        this.descriptor = descriptor;
    }
    NameAndTypeInfo.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.NAME_AND_TYPE;
    };
    NameAndTypeInfo.prototype.isResolved = function () {
        return true;
    };
    NameAndTypeInfo.fromBytes = function (byteStream, constantPool) {
        var nameIndex = byteStream.getUint16(), descriptorIndex = byteStream.getUint16(), nameConst = constantPool.get(nameIndex), descriptorConst = constantPool.get(descriptorIndex);
        assert_1['default'](nameConst.getType() === enums_1.ConstantPoolItemType.UTF8 && descriptorConst.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool NameAndTypeInfo types != UTF8');
        return new this(nameConst.value, descriptorConst.value);
    };
    NameAndTypeInfo.size = 1;
    NameAndTypeInfo.infoByteSize = 4;
    return NameAndTypeInfo;
}();
exports.NameAndTypeInfo = NameAndTypeInfo;
CP_CLASSES[enums_1.ConstantPoolItemType.NAME_AND_TYPE] = NameAndTypeInfo;
var ConstString = function () {
    function ConstString(stringValue) {
        this.value = null;
        this.stringValue = stringValue;
    }
    ConstString.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.STRING;
    };
    ConstString.prototype.resolve = function (thread, loader, caller, cb) {
        this.value = thread.getJVM().internString(this.stringValue);
        setImmediate(function () {
            return cb(true);
        });
    };
    ConstString.prototype.getConstant = function (thread) {
        return this.value;
    };
    ConstString.prototype.isResolved = function () {
        return this.value !== null;
    };
    ConstString.fromBytes = function (byteStream, constantPool) {
        var stringIndex = byteStream.getUint16(), utf8Info = constantPool.get(stringIndex);
        assert_1['default'](utf8Info.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool ConstString type != UTF8');
        return new this(utf8Info.value);
    };
    ConstString.size = 1;
    ConstString.infoByteSize = 2;
    return ConstString;
}();
exports.ConstString = ConstString;
CP_CLASSES[enums_1.ConstantPoolItemType.STRING] = ConstString;
var MethodType = function () {
    function MethodType(descriptor) {
        this.methodType = null;
        this.descriptor = descriptor;
    }
    MethodType.prototype.resolve = function (thread, cl, caller, cb) {
        var _this = this;
        util.createMethodType(thread, cl, this.descriptor, function (e, type) {
            if (e) {
                thread.throwException(e);
                cb(false);
            } else {
                _this.methodType = type;
                cb(true);
            }
        });
    };
    MethodType.prototype.getConstant = function (thread) {
        return this.methodType;
    };
    MethodType.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.METHOD_TYPE;
    };
    MethodType.prototype.isResolved = function () {
        return this.methodType !== null;
    };
    MethodType.fromBytes = function (byteStream, constantPool) {
        var descriptorIndex = byteStream.getUint16(), utf8Info = constantPool.get(descriptorIndex);
        assert_1['default'](utf8Info.getType() === enums_1.ConstantPoolItemType.UTF8, 'ConstantPool MethodType type != UTF8');
        return new this(utf8Info.value);
    };
    MethodType.size = 1;
    MethodType.infoByteSize = 2;
    return MethodType;
}();
exports.MethodType = MethodType;
CP_CLASSES[enums_1.ConstantPoolItemType.METHOD_TYPE] = MethodType;
var MethodReference = function () {
    function MethodReference(classInfo, nameAndTypeInfo) {
        this.method = null;
        this.fullSignature = null;
        this.paramWordSize = -1;
        this.memberName = null;
        this.appendix = null;
        this.jsConstructor = null;
        this.classInfo = classInfo;
        this.nameAndTypeInfo = nameAndTypeInfo;
        this.signature = this.nameAndTypeInfo.name + this.nameAndTypeInfo.descriptor;
    }
    MethodReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.METHODREF;
    };
    MethodReference.prototype.hasAccess = function (thread, frame, isStatic) {
        var method = this.method, accessingCls = frame.method.cls;
        if (method.accessFlags.isStatic() !== isStatic) {
            thread.throwNewException('Ljava/lang/IncompatibleClassChangeError;', 'Method ' + method.name + ' from class ' + method.cls.getExternalName() + ' is ' + (isStatic ? 'not ' : '') + 'static.');
            frame.returnToThreadLoop = true;
            return false;
        } else if (!util.checkAccess(accessingCls, method.cls, method.accessFlags)) {
            thread.throwNewException('Ljava/lang/IllegalAccessError;', accessingCls.getExternalName() + ' cannot access ' + method.cls.getExternalName() + '.' + method.name);
            frame.returnToThreadLoop = true;
            return false;
        }
        return true;
    };
    MethodReference.prototype.resolveMemberName = function (method, thread, cl, caller, cb) {
        var _this = this;
        var memberHandleNatives = thread.getBsCl().getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;').getConstructor(thread), appendix = new (thread.getBsCl().getInitializedClass(thread, '[Ljava/lang/Object;').getConstructor(thread))(thread, 1);
        util.createMethodType(thread, cl, this.nameAndTypeInfo.descriptor, function (e, type) {
            if (e) {
                thread.throwException(e);
                cb(false);
            } else {
                memberHandleNatives['java/lang/invoke/MethodHandleNatives/linkMethod(Ljava/lang/Class;ILjava/lang/Class;Ljava/lang/String;Ljava/lang/Object;[Ljava/lang/Object;)Ljava/lang/invoke/MemberName;'](thread, [
                    caller.getClassObject(thread),
                    enums_1.MethodHandleReferenceKind.INVOKEVIRTUAL,
                    _this.classInfo.cls.getClassObject(thread),
                    thread.getJVM().internString(_this.nameAndTypeInfo.name),
                    type,
                    appendix
                ], function (e, rv) {
                    if (e !== null) {
                        thread.throwException(e);
                        cb(false);
                    } else {
                        _this.appendix = appendix.array[0];
                        _this.memberName = rv;
                        cb(true);
                    }
                });
            }
        });
    };
    MethodReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        if (!this.classInfo.isResolved()) {
            this.classInfo.resolve(thread, loader, caller, function (status) {
                if (!status) {
                    cb(false);
                } else {
                    _this.resolve(thread, loader, caller, cb, explicit);
                }
            }, explicit);
        } else {
            var cls = this.classInfo.cls, method = cls.methodLookup(this.signature);
            if (method === null) {
                if (util.is_reference_type(cls.getInternalName())) {
                    method = cls.signaturePolymorphicAwareMethodLookup(this.signature);
                    if (method !== null && (method.name === 'invoke' || method.name === 'invokeExact')) {
                        return this.resolveMemberName(method, thread, loader, caller, function (status) {
                            if (status === true) {
                                _this.setResolved(thread, method);
                            } else {
                                thread.throwNewException('Ljava/lang/NoSuchMethodError;', 'Method ' + _this.signature + ' does not exist in class ' + _this.classInfo.cls.getExternalName() + '.');
                            }
                            cb(status);
                        });
                    }
                }
            }
            if (method !== null) {
                this.setResolved(thread, method);
                cb(true);
            } else {
                thread.throwNewException('Ljava/lang/NoSuchMethodError;', 'Method ' + this.signature + ' does not exist in class ' + this.classInfo.cls.getExternalName() + '.');
                cb(false);
            }
        }
    };
    MethodReference.prototype.setResolved = function (thread, method) {
        this.method = method;
        this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        this.fullSignature = this.method.fullSignature;
        this.jsConstructor = this.method.cls.getConstructor(thread);
    };
    MethodReference.prototype.isResolved = function () {
        return this.method !== null;
    };
    MethodReference.prototype.getParamWordSize = function () {
        if (this.paramWordSize === -1) {
            this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        }
        return this.paramWordSize;
    };
    MethodReference.fromBytes = function (byteStream, constantPool) {
        var classIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), classInfo = constantPool.get(classIndex), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1['default'](classInfo.getType() === enums_1.ConstantPoolItemType.CLASS && nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool MethodReference types mismatch');
        return new this(classInfo, nameAndTypeInfo);
    };
    MethodReference.size = 1;
    MethodReference.infoByteSize = 4;
    return MethodReference;
}();
exports.MethodReference = MethodReference;
CP_CLASSES[enums_1.ConstantPoolItemType.METHODREF] = MethodReference;
var InterfaceMethodReference = function () {
    function InterfaceMethodReference(classInfo, nameAndTypeInfo) {
        this.fullSignature = null;
        this.method = null;
        this.paramWordSize = -1;
        this.jsConstructor = null;
        this.classInfo = classInfo;
        this.nameAndTypeInfo = nameAndTypeInfo;
        this.signature = this.nameAndTypeInfo.name + this.nameAndTypeInfo.descriptor;
    }
    InterfaceMethodReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.INTERFACE_METHODREF;
    };
    InterfaceMethodReference.prototype.hasAccess = function (thread, frame, isStatic) {
        var method = this.method, accessingCls = frame.method.cls;
        if (method.accessFlags.isStatic() !== isStatic) {
            thread.throwNewException('Ljava/lang/IncompatibleClassChangeError;', 'Method ' + method.name + ' from class ' + method.cls.getExternalName() + ' is ' + (isStatic ? 'not ' : '') + 'static.');
            frame.returnToThreadLoop = true;
            return false;
        } else if (!util.checkAccess(accessingCls, method.cls, method.accessFlags)) {
            thread.throwNewException('Ljava/lang/IllegalAccessError;', accessingCls.getExternalName() + ' cannot access ' + method.cls.getExternalName() + '.' + method.name);
            frame.returnToThreadLoop = true;
            return false;
        }
        return true;
    };
    InterfaceMethodReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        if (!this.classInfo.isResolved()) {
            this.classInfo.resolve(thread, loader, caller, function (status) {
                if (!status) {
                    cb(false);
                } else {
                    _this.resolve(thread, loader, caller, cb, explicit);
                }
            }, explicit);
        } else {
            var cls = this.classInfo.cls, method = cls.methodLookup(this.signature);
            this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
            if (method !== null) {
                this.setResolved(thread, method);
                cb(true);
            } else {
                thread.throwNewException('Ljava/lang/NoSuchMethodError;', 'Method ' + this.signature + ' does not exist in class ' + this.classInfo.cls.getExternalName() + '.');
                cb(false);
            }
        }
    };
    InterfaceMethodReference.prototype.setResolved = function (thread, method) {
        this.method = method;
        this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        this.fullSignature = this.method.fullSignature;
        this.jsConstructor = this.method.cls.getConstructor(thread);
    };
    InterfaceMethodReference.prototype.getParamWordSize = function () {
        if (this.paramWordSize === -1) {
            this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
        }
        return this.paramWordSize;
    };
    InterfaceMethodReference.prototype.isResolved = function () {
        return this.method !== null;
    };
    InterfaceMethodReference.fromBytes = function (byteStream, constantPool) {
        var classIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), classInfo = constantPool.get(classIndex), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1['default'](classInfo.getType() === enums_1.ConstantPoolItemType.CLASS && nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool InterfaceMethodReference types mismatch');
        return new this(classInfo, nameAndTypeInfo);
    };
    InterfaceMethodReference.size = 1;
    InterfaceMethodReference.infoByteSize = 4;
    return InterfaceMethodReference;
}();
exports.InterfaceMethodReference = InterfaceMethodReference;
CP_CLASSES[enums_1.ConstantPoolItemType.INTERFACE_METHODREF] = InterfaceMethodReference;
var FieldReference = function () {
    function FieldReference(classInfo, nameAndTypeInfo) {
        this.field = null;
        this.fullFieldName = null;
        this.fieldOwnerConstructor = null;
        this.classInfo = classInfo;
        this.nameAndTypeInfo = nameAndTypeInfo;
    }
    FieldReference.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.FIELDREF;
    };
    FieldReference.prototype.hasAccess = function (thread, frame, isStatic) {
        var field = this.field, accessingCls = frame.method.cls;
        if (field.accessFlags.isStatic() !== isStatic) {
            thread.throwNewException('Ljava/lang/IncompatibleClassChangeError;', 'Field ' + name + ' from class ' + field.cls.getExternalName() + ' is ' + (isStatic ? 'not ' : '') + 'static.');
            frame.returnToThreadLoop = true;
            return false;
        } else if (!util.checkAccess(accessingCls, field.cls, field.accessFlags)) {
            thread.throwNewException('Ljava/lang/IllegalAccessError;', accessingCls.getExternalName() + ' cannot access ' + field.cls.getExternalName() + '.' + name);
            frame.returnToThreadLoop = true;
            return false;
        }
        return true;
    };
    FieldReference.prototype.resolve = function (thread, loader, caller, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        if (!this.classInfo.isResolved()) {
            this.classInfo.resolve(thread, loader, caller, function (status) {
                if (!status) {
                    cb(false);
                } else {
                    _this.resolve(thread, loader, caller, cb, explicit);
                }
            }, explicit);
        } else {
            var cls = this.classInfo.cls, field = cls.fieldLookup(this.nameAndTypeInfo.name);
            if (field !== null) {
                this.fullFieldName = util.descriptor2typestr(field.cls.getInternalName()) + '/' + field.name;
                this.field = field;
                cb(true);
            } else {
                thread.throwNewException('Ljava/lang/NoSuchFieldError;', 'Field ' + this.nameAndTypeInfo.name + ' does not exist in class ' + this.classInfo.cls.getExternalName() + '.');
                cb(false);
            }
        }
    };
    FieldReference.prototype.isResolved = function () {
        return this.field !== null;
    };
    FieldReference.fromBytes = function (byteStream, constantPool) {
        var classIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), classInfo = constantPool.get(classIndex), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1['default'](classInfo.getType() === enums_1.ConstantPoolItemType.CLASS && nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool FieldReference types mismatch');
        return new this(classInfo, nameAndTypeInfo);
    };
    FieldReference.size = 1;
    FieldReference.infoByteSize = 4;
    return FieldReference;
}();
exports.FieldReference = FieldReference;
CP_CLASSES[enums_1.ConstantPoolItemType.FIELDREF] = FieldReference;
var InvokeDynamic = function () {
    function InvokeDynamic(bootstrapMethodAttrIndex, nameAndTypeInfo) {
        this.callSiteObjects = {};
        this.methodType = null;
        this.bootstrapMethodAttrIndex = bootstrapMethodAttrIndex;
        this.nameAndTypeInfo = nameAndTypeInfo;
        this.paramWordSize = util.getMethodDescriptorWordSize(this.nameAndTypeInfo.descriptor);
    }
    InvokeDynamic.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.INVOKE_DYNAMIC;
    };
    InvokeDynamic.prototype.isResolved = function () {
        return this.methodType !== null;
    };
    InvokeDynamic.prototype.resolve = function (thread, loader, caller, cb) {
        var _this = this;
        util.createMethodType(thread, loader, this.nameAndTypeInfo.descriptor, function (e, rv) {
            if (e) {
                thread.throwException(e);
                cb(false);
            } else {
                _this.methodType = rv;
                cb(true);
            }
        });
    };
    InvokeDynamic.prototype.getCallSiteObject = function (pc) {
        var cso = this.callSiteObjects[pc];
        if (cso) {
            return cso;
        } else {
            return null;
        }
    };
    InvokeDynamic.prototype.constructCallSiteObject = function (thread, cl, clazz, pc, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        var bootstrapMethod = clazz.getBootstrapMethod(this.bootstrapMethodAttrIndex), unresolvedItems = bootstrapMethod[1].concat(bootstrapMethod[0], this).filter(function (item) {
                return !item.isResolved();
            });
        if (unresolvedItems.length > 0) {
            return util.asyncForEach(unresolvedItems, function (cpItem, nextItem) {
                cpItem.resolve(thread, cl, clazz, function (status) {
                    if (!status) {
                        nextItem('Failed.');
                    } else {
                        nextItem();
                    }
                }, explicit);
            }, function (err) {
                if (err) {
                    cb(false);
                } else {
                    _this.constructCallSiteObject(thread, cl, clazz, pc, cb, explicit);
                }
            });
        }
        function getArguments() {
            var cpItems = bootstrapMethod[1], i, cpItem, rvObj = new (thread.getBsCl().getInitializedClass(thread, '[Ljava/lang/Object;').getConstructor(thread))(thread, cpItems.length), rv = rvObj.array;
            for (i = 0; i < cpItems.length; i++) {
                cpItem = cpItems[i];
                switch (cpItem.getType()) {
                case enums_1.ConstantPoolItemType.CLASS:
                    rv[i] = cpItem.cls.getClassObject(thread);
                    break;
                case enums_1.ConstantPoolItemType.METHOD_HANDLE:
                    rv[i] = cpItem.methodHandle;
                    break;
                case enums_1.ConstantPoolItemType.METHOD_TYPE:
                    rv[i] = cpItem.methodType;
                    break;
                case enums_1.ConstantPoolItemType.STRING:
                    rv[i] = cpItem.value;
                    break;
                case enums_1.ConstantPoolItemType.UTF8:
                    rv[i] = thread.getJVM().internString(cpItem.value);
                    break;
                case enums_1.ConstantPoolItemType.INTEGER:
                    rv[i] = cl.getInitializedClass(thread, 'I').createWrapperObject(thread, cpItem.value);
                    break;
                case enums_1.ConstantPoolItemType.LONG:
                    rv[i] = cl.getInitializedClass(thread, 'J').createWrapperObject(thread, cpItem.value);
                    break;
                case enums_1.ConstantPoolItemType.FLOAT:
                    rv[i] = cl.getInitializedClass(thread, 'F').createWrapperObject(thread, cpItem.value);
                    break;
                case enums_1.ConstantPoolItemType.DOUBLE:
                    rv[i] = cl.getInitializedClass(thread, 'D').createWrapperObject(thread, cpItem.value);
                    break;
                default:
                    assert_1['default'](false, 'Invalid CPItem for static args: ' + enums_1.ConstantPoolItemType[cpItem.getType()]);
                    break;
                }
            }
            assert_1['default'](function () {
                var status = true;
                cpItems.forEach(function (cpItem, i) {
                    if (rv[i] === undefined) {
                        console.log('Undefined item at arg ' + i + ': ' + enums_1.ConstantPoolItemType[cpItem.getType()]);
                        status = false;
                    } else if (rv[i] === null) {
                        console.log('Null item at arg ' + i + ': ' + enums_1.ConstantPoolItemType[cpItem.getType()]);
                        status = false;
                    }
                });
                return status;
            }(), 'Arguments cannot be undefined or null.');
            return rvObj;
        }
        var methodName = thread.getJVM().internString(this.nameAndTypeInfo.name), appendixArr = new (cl.getInitializedClass(thread, '[Ljava/lang/Object;').getConstructor(thread))(thread, 1), staticArgs = getArguments(), mhn = cl.getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;').getConstructor(thread);
        mhn['java/lang/invoke/MethodHandleNatives/linkCallSite(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;[Ljava/lang/Object;)Ljava/lang/invoke/MemberName;'](thread, [
            clazz.getClassObject(thread),
            bootstrapMethod[0].methodHandle,
            methodName,
            this.methodType,
            staticArgs,
            appendixArr
        ], function (e, rv) {
            if (e) {
                thread.throwException(e);
                cb(false);
            } else {
                _this.setResolved(pc, [
                    rv,
                    appendixArr.array[0]
                ]);
                cb(true);
            }
        });
    };
    InvokeDynamic.prototype.setResolved = function (pc, cso) {
        if (this.callSiteObjects[pc] === undefined) {
            this.callSiteObjects[pc] = cso;
        }
    };
    InvokeDynamic.fromBytes = function (byteStream, constantPool) {
        var bootstrapMethodAttrIndex = byteStream.getUint16(), nameAndTypeIndex = byteStream.getUint16(), nameAndTypeInfo = constantPool.get(nameAndTypeIndex);
        assert_1['default'](nameAndTypeInfo.getType() === enums_1.ConstantPoolItemType.NAME_AND_TYPE, 'ConstantPool InvokeDynamic types mismatch');
        return new this(bootstrapMethodAttrIndex, nameAndTypeInfo);
    };
    InvokeDynamic.size = 1;
    InvokeDynamic.infoByteSize = 4;
    return InvokeDynamic;
}();
exports.InvokeDynamic = InvokeDynamic;
CP_CLASSES[enums_1.ConstantPoolItemType.INVOKE_DYNAMIC] = InvokeDynamic;
var MethodHandle = function () {
    function MethodHandle(reference, referenceType) {
        this.methodHandle = null;
        this.reference = reference;
        this.referenceType = referenceType;
    }
    MethodHandle.prototype.getType = function () {
        return enums_1.ConstantPoolItemType.METHOD_HANDLE;
    };
    MethodHandle.prototype.isResolved = function () {
        return this.methodHandle !== null;
    };
    MethodHandle.prototype.getConstant = function (thread) {
        return this.methodHandle;
    };
    MethodHandle.prototype.resolve = function (thread, cl, caller, cb, explicit) {
        var _this = this;
        if (!this.reference.isResolved()) {
            return this.reference.resolve(thread, cl, caller, function (status) {
                if (!status) {
                    cb(false);
                } else {
                    _this.resolve(thread, cl, caller, cb, explicit);
                }
            }, explicit);
        }
        this.constructMethodHandleType(thread, cl, function (type) {
            if (type === null) {
                cb(false);
            } else {
                var methodHandleNatives = cl.getInitializedClass(thread, 'Ljava/lang/invoke/MethodHandleNatives;').getConstructor(thread);
                methodHandleNatives['linkMethodHandleConstant(Ljava/lang/Class;ILjava/lang/Class;Ljava/lang/String;Ljava/lang/Object;)Ljava/lang/invoke/MethodHandle;'](thread, [
                    caller.getClassObject(thread),
                    _this.referenceType,
                    _this.getDefiningClassObj(thread),
                    thread.getJVM().internString(_this.reference.nameAndTypeInfo.name),
                    type
                ], function (e, methodHandle) {
                    if (e) {
                        thread.throwException(e);
                        cb(false);
                    } else {
                        _this.methodHandle = methodHandle;
                        cb(true);
                    }
                });
            }
        });
    };
    MethodHandle.prototype.getDefiningClassObj = function (thread) {
        if (this.reference.getType() === enums_1.ConstantPoolItemType.FIELDREF) {
            return this.reference.field.cls.getClassObject(thread);
        } else {
            return this.reference.method.cls.getClassObject(thread);
        }
    };
    MethodHandle.prototype.constructMethodHandleType = function (thread, cl, cb) {
        if (this.reference.getType() === enums_1.ConstantPoolItemType.FIELDREF) {
            var resolveObj = this.reference.nameAndTypeInfo.descriptor;
            cl.resolveClass(thread, resolveObj, function (cdata) {
                if (cdata !== null) {
                    cb(cdata.getClassObject(thread));
                } else {
                    cb(null);
                }
            });
        } else {
            util.createMethodType(thread, cl, this.reference.nameAndTypeInfo.descriptor, function (e, rv) {
                if (e) {
                    thread.throwException(e);
                    cb(null);
                } else {
                    cb(rv);
                }
            });
        }
    };
    MethodHandle.fromBytes = function (byteStream, constantPool) {
        var referenceKind = byteStream.getUint8(), referenceIndex = byteStream.getUint16(), reference = constantPool.get(referenceIndex);
        assert_1['default'](0 < referenceKind && referenceKind < 10, 'ConstantPool MethodHandle invalid referenceKind: ' + referenceKind);
        assert_1['default'](function () {
            switch (referenceKind) {
            case enums_1.MethodHandleReferenceKind.GETFIELD:
            case enums_1.MethodHandleReferenceKind.GETSTATIC:
            case enums_1.MethodHandleReferenceKind.PUTFIELD:
            case enums_1.MethodHandleReferenceKind.PUTSTATIC:
                return reference.getType() === enums_1.ConstantPoolItemType.FIELDREF;
            case enums_1.MethodHandleReferenceKind.INVOKEINTERFACE:
                return reference.getType() === enums_1.ConstantPoolItemType.INTERFACE_METHODREF && reference.nameAndTypeInfo.name[0] !== '<';
            case enums_1.MethodHandleReferenceKind.INVOKEVIRTUAL:
            case enums_1.MethodHandleReferenceKind.INVOKESTATIC:
            case enums_1.MethodHandleReferenceKind.INVOKESPECIAL:
                return (reference.getType() === enums_1.ConstantPoolItemType.METHODREF || reference.getType() === enums_1.ConstantPoolItemType.INTERFACE_METHODREF) && reference.nameAndTypeInfo.name[0] !== '<';
            case enums_1.MethodHandleReferenceKind.NEWINVOKESPECIAL:
                return reference.getType() === enums_1.ConstantPoolItemType.METHODREF && reference.nameAndTypeInfo.name === '<init>';
            }
            return true;
        }(), 'Invalid constant pool reference for method handle reference type: ' + enums_1.MethodHandleReferenceKind[referenceKind]);
        return new this(reference, referenceKind);
    };
    MethodHandle.size = 1;
    MethodHandle.infoByteSize = 3;
    return MethodHandle;
}();
exports.MethodHandle = MethodHandle;
CP_CLASSES[enums_1.ConstantPoolItemType.METHOD_HANDLE] = MethodHandle;
var CONSTANT_POOL_TIER = [
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0
];
(function (tierInfos) {
    tierInfos.forEach(function (tierInfo, index) {
        tierInfo.forEach(function (type) {
            CONSTANT_POOL_TIER[type] = index;
        });
    });
}([
    [
        enums_1.ConstantPoolItemType.UTF8,
        enums_1.ConstantPoolItemType.INTEGER,
        enums_1.ConstantPoolItemType.FLOAT,
        enums_1.ConstantPoolItemType.LONG,
        enums_1.ConstantPoolItemType.DOUBLE
    ],
    [
        enums_1.ConstantPoolItemType.CLASS,
        enums_1.ConstantPoolItemType.STRING,
        enums_1.ConstantPoolItemType.NAME_AND_TYPE,
        enums_1.ConstantPoolItemType.METHOD_TYPE
    ],
    [
        enums_1.ConstantPoolItemType.FIELDREF,
        enums_1.ConstantPoolItemType.METHODREF,
        enums_1.ConstantPoolItemType.INTERFACE_METHODREF,
        enums_1.ConstantPoolItemType.INVOKE_DYNAMIC
    ],
    [enums_1.ConstantPoolItemType.METHOD_HANDLE]
]));
var ConstantPool = function () {
    function ConstantPool() {
    }
    ConstantPool.prototype.parse = function (byteStream, cpPatches) {
        var _this = this;
        if (cpPatches === void 0) {
            cpPatches = null;
        }
        var cpCount = byteStream.getUint16(), deferredQueue = [
                [],
                [],
                []
            ], endIdx = 0, idx = 1, tag = 0, itemOffset = 0, itemTier = 0;
        this.constantPool = new Array(cpCount);
        while (idx < cpCount) {
            itemOffset = byteStream.pos();
            tag = byteStream.getUint8();
            assert_1['default'](CP_CLASSES[tag] !== null && CP_CLASSES[tag] !== undefined, 'Unknown ConstantPool tag: ' + tag);
            itemTier = CONSTANT_POOL_TIER[tag];
            if (itemTier > 0) {
                deferredQueue[itemTier - 1].push({
                    offset: itemOffset,
                    index: idx
                });
                byteStream.skip(CP_CLASSES[tag].infoByteSize);
            } else {
                this.constantPool[idx] = CP_CLASSES[tag].fromBytes(byteStream, this);
            }
            idx += CP_CLASSES[tag].size;
        }
        endIdx = byteStream.pos();
        deferredQueue.forEach(function (deferredItems) {
            deferredItems.forEach(function (item) {
                byteStream.seek(item.offset);
                tag = byteStream.getUint8();
                _this.constantPool[item.index] = CP_CLASSES[tag].fromBytes(byteStream, _this);
                if (cpPatches !== null && cpPatches.array[item.index] !== null && cpPatches.array[item.index] !== undefined) {
                    var patchObj = cpPatches.array[item.index];
                    switch (patchObj.getClass().getInternalName()) {
                    case 'Ljava/lang/Integer;':
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.INTEGER);
                        _this.constantPool[item.index].value = patchObj['java/lang/Integer/value'];
                        break;
                    case 'Ljava/lang/Long;':
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.LONG);
                        _this.constantPool[item.index].value = patchObj['java/lang/Long/value'];
                        break;
                    case 'Ljava/lang/Float;':
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.FLOAT);
                        _this.constantPool[item.index].value = patchObj['java/lang/Float/value'];
                        break;
                    case 'Ljava/lang/Double;':
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.DOUBLE);
                        _this.constantPool[item.index].value = patchObj['java/lang/Double/value'];
                        break;
                    case 'Ljava/lang/String;':
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.UTF8);
                        _this.constantPool[item.index].value = patchObj.toString();
                        break;
                    case 'Ljava/lang/Class;':
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.CLASS);
                        _this.constantPool[item.index].name = patchObj.$cls.getInternalName();
                        _this.constantPool[item.index].cls = patchObj.$cls;
                        break;
                    default:
                        assert_1['default'](tag === enums_1.ConstantPoolItemType.STRING);
                        _this.constantPool[item.index].stringValue = '';
                        _this.constantPool[item.index].value = patchObj;
                        break;
                    }
                }
            });
        });
        byteStream.seek(endIdx);
        return byteStream;
    };
    ConstantPool.prototype.get = function (idx) {
        assert_1['default'](this.constantPool[idx] !== undefined, 'Invalid ConstantPool reference.');
        return this.constantPool[idx];
    };
    ConstantPool.prototype.each = function (fn) {
        this.constantPool.forEach(function (item, idx) {
            if (item !== undefined) {
                fn(idx, item);
            }
        });
    };
    return ConstantPool;
}();
exports.ConstantPool = ConstantPool;
//# sourceMappingURL=ConstantPool.js.map