'use strict';
var enums_1 = require('./enums');
var assert_1 = require('./assert');
var Monitor = function () {
    function Monitor() {
        this.owner = null;
        this.count = 0;
        this.blocked = {};
        this.waiting = {};
    }
    Monitor.prototype.enter = function (thread, cb) {
        if (this.owner === thread) {
            this.count++;
            return true;
        } else {
            return this.contendForLock(thread, 1, enums_1.ThreadStatus.BLOCKED, cb);
        }
    };
    Monitor.prototype.contendForLock = function (thread, count, blockStatus, cb) {
        var owner = this.owner;
        assert_1['default'](owner != thread, 'Thread attempting to contend for lock it already owns!');
        if (owner === null) {
            assert_1['default'](this.count === 0);
            this.owner = thread;
            this.count = count;
            return true;
        } else {
            this.blocked[thread.getRef()] = {
                thread: thread,
                cb: cb,
                count: count
            };
            thread.setStatus(blockStatus, this);
            return false;
        }
    };
    Monitor.prototype.exit = function (thread) {
        var owner = this.owner;
        if (owner === thread) {
            if (--this.count === 0) {
                this.owner = null;
                this.appointNewOwner();
            }
        } else {
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', 'Cannot exit a monitor that you do not own.');
        }
        return owner === thread;
    };
    Monitor.prototype.appointNewOwner = function () {
        var blockedThreadRefs = Object.keys(this.blocked);
        if (blockedThreadRefs.length > 0) {
            var unblockedRef = blockedThreadRefs[Math.floor(Math.random() * blockedThreadRefs.length)], unblocked = this.blocked[unblockedRef];
            this.unblock(unblocked.thread, false);
        }
    };
    Monitor.prototype.wait = function (thread, cb, timeoutMs, timeoutNs) {
        var _this = this;
        if (this.getOwner() === thread) {
            assert_1['default'](thread.getStatus() !== enums_1.ThreadStatus.BLOCKED);
            this.waiting[thread.getRef()] = {
                thread: thread,
                cb: cb,
                count: this.count,
                isTimed: timeoutMs != null && timeoutMs !== 0
            };
            this.owner = null;
            this.count = 0;
            if (timeoutMs != null && timeoutMs !== 0) {
                this.waiting[thread.getRef()].timer = setTimeout(function () {
                    _this.unwait(thread, true);
                }, timeoutMs);
                thread.setStatus(enums_1.ThreadStatus.TIMED_WAITING, this);
            } else {
                thread.setStatus(enums_1.ThreadStatus.WAITING, this);
            }
            this.appointNewOwner();
            return true;
        } else {
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', 'Cannot wait on an object that you do not own.');
            return false;
        }
    };
    Monitor.prototype.unwait = function (thread, fromTimer, interrupting, unwaitCb) {
        if (interrupting === void 0) {
            interrupting = false;
        }
        if (unwaitCb === void 0) {
            unwaitCb = null;
        }
        var waitEntry = this.waiting[thread.getRef()], blockStatus = enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED, blockCb = function () {
                thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
                if (interrupting) {
                    unwaitCb();
                } else {
                    waitEntry.cb(fromTimer);
                }
            };
        assert_1['default'](waitEntry != null);
        delete this.waiting[thread.getRef()];
        if (thread.getStatus() === enums_1.ThreadStatus.TIMED_WAITING && !fromTimer) {
            var timerId = waitEntry.timer;
            assert_1['default'](timerId != null);
            clearTimeout(timerId);
        }
        if (this.contendForLock(thread, waitEntry.count, blockStatus, blockCb)) {
            blockCb();
        }
    };
    Monitor.prototype.unblock = function (thread, interrupting) {
        if (interrupting === void 0) {
            interrupting = false;
        }
        var blockEntry = this.blocked[thread.getRef()];
        assert_1['default'](interrupting ? thread.getStatus() === enums_1.ThreadStatus.BLOCKED : true);
        if (blockEntry != null) {
            delete this.blocked[thread.getRef()];
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            if (!interrupting) {
                assert_1['default'](this.owner == null && this.count === 0, 'T' + thread.getRef() + ': We\'re not interrupting a block, but someone else owns the monitor?! Owned by ' + (this.owner == null ? '[no one]' : '' + this.owner.getRef()) + ' Count: ' + this.count);
                this.owner = thread;
                this.count = blockEntry.count;
                blockEntry.cb();
            }
        }
    };
    Monitor.prototype.notify = function (thread) {
        if (this.owner === thread) {
            var waitingRefs = Object.keys(this.waiting);
            if (waitingRefs.length > 0) {
                this.unwait(this.waiting[waitingRefs[Math.floor(Math.random() * waitingRefs.length)]].thread, false);
            }
        } else {
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', 'Cannot notify on a monitor that you do not own.');
        }
    };
    Monitor.prototype.notifyAll = function (thread) {
        if (this.owner === thread) {
            var waitingRefs = Object.keys(this.waiting), i;
            for (i = 0; i < waitingRefs.length; i++) {
                this.unwait(this.waiting[waitingRefs[i]].thread, false);
            }
        } else {
            thread.throwNewException('Ljava/lang/IllegalMonitorStateException;', 'Cannot notifyAll on a monitor that you do not own.');
        }
    };
    Monitor.prototype.getOwner = function () {
        return this.owner;
    };
    Monitor.prototype.isWaiting = function (thread) {
        return this.waiting[thread.getRef()] != null && !this.waiting[thread.getRef()].isTimed;
    };
    Monitor.prototype.isTimedWaiting = function (thread) {
        return this.waiting[thread.getRef()] != null && this.waiting[thread.getRef()].isTimed;
    };
    Monitor.prototype.isBlocked = function (thread) {
        return this.blocked[thread.getRef()] != null;
    };
    return Monitor;
}();
exports.__esModule = true;
exports['default'] = Monitor;
//# sourceMappingURL=Monitor.js.map