'use strict';
var Heap = function () {
    function Heap(size) {
        this.size = size;
        this._sizeMap = {};
        this._buffer = new Buffer(size);
        this._remaining = size;
        this._offset = 0;
        this._freeLists = new Array(Heap._numSizeClasses);
        for (var i = 0; i < Heap._numSizeClasses; i++) {
            this._freeLists[i] = [];
        }
    }
    Heap.prototype.malloc = function (size) {
        if (size <= 4) {
            size = 4;
        }
        if (this._remaining < size) {
            throw 'out of memory';
        }
        var addr;
        var cl;
        cl = Heap.size_to_class(size);
        addr = this._freeLists[cl].pop();
        if (addr === undefined) {
            addr = this.refill(cl);
        }
        return addr;
    };
    Heap.prototype.free = function (addr) {
        var masked = addr & ~(Heap._chunkSize - 1);
        var cl = this._sizeMap[masked];
        this._freeLists[cl].push(addr);
    };
    Heap.prototype.store_word = function (addr, value) {
        this._buffer.writeInt32LE(value, addr);
    };
    Heap.prototype.get_byte = function (addr) {
        return this._buffer[addr];
    };
    Heap.prototype.get_word = function (addr) {
        return this._buffer.readInt32LE(addr);
    };
    Heap.prototype.get_buffer = function (addr, len) {
        return this._buffer.slice(addr, addr + len);
    };
    Heap.prototype.get_signed_byte = function (addr) {
        return this._buffer.readInt8(addr);
    };
    Heap.prototype.set_byte = function (addr, value) {
        this._buffer[addr] = value;
        ;
    };
    Heap.prototype.set_signed_byte = function (addr, value) {
        this._buffer.writeInt8(value, addr);
    };
    Heap.prototype.memcpy = function (srcAddr, dstAddr, len) {
        this._buffer.copy(this._buffer, dstAddr, srcAddr, srcAddr + len);
    };
    Heap.prototype.refill = function (cl) {
        var sz = this.cl_to_size(cl);
        var count = Math.floor(Heap._chunkSize / sz);
        if (count < 1) {
            count = 1;
        }
        var addr = this._offset;
        this._sizeMap[addr] = cl;
        for (var i = 0; i < count; i++) {
            this._remaining -= sz;
            addr = this._offset;
            this._freeLists[cl].push(addr);
            this._offset += sz;
        }
        return addr;
    };
    Heap.ilog2 = function (num) {
        var log2 = 0;
        var value = 1;
        while (value < num) {
            value <<= 1;
            log2++;
        }
        return log2;
    };
    Heap.size_to_class = function (size) {
        return Heap.ilog2(size);
    };
    Heap.prototype.cl_to_size = function (cl) {
        return 1 << cl;
    };
    Heap._numSizeClasses = 64;
    Heap._chunkSize = 4096;
    return Heap;
}();
exports.__esModule = true;
exports['default'] = Heap;
//# sourceMappingURL=heap.js.map