'use strict';
var option_parser_1 = require('./option_parser');
var jvm_1 = require('./jvm');
var util_1 = require('./util');
var logging_1 = require('./logging');
var parser = new option_parser_1.OptionParser({
    default: {
        classpath: {
            type: 3,
            alias: 'cp',
            optDesc: ' <class search path of directories and zip/jar files>',
            desc: 'A : separated list of directories, JAR archives, and ZIP archives to search for class files.'
        },
        D: {
            type: 4,
            optDesc: '<name>=<value>',
            desc: 'set a system property'
        },
        jar: {
            type: 3,
            stopParsing: true
        },
        help: {
            alias: '?',
            desc: 'print this help message'
        },
        X: { desc: 'print help on non-standard options' },
        enableassertions: {
            type: 2,
            optDesc: '[:<packagename>...|:<classname>]',
            alias: 'ea',
            desc: 'enable assertions with specified granularity'
        },
        disableassertions: {
            type: 2,
            optDesc: '[:<packagename>...|:<classname>]',
            alias: 'da',
            desc: 'disable assertions with specified granularity'
        },
        enablesystemassertions: {
            alias: 'esa',
            desc: 'enable system assertions'
        },
        disablesystemassertions: {
            alias: 'dsa',
            desc: 'disable system assertions '
        }
    },
    X: {
        'int': { desc: 'interpreted mode execution only' },
        'dump-JIT-stats': {
            desc: 'dump JIT statistics',
            enabled: !RELEASE
        },
        log: {
            desc: 'log level, [0-10]|vtrace|trace|debug|error',
            type: 3,
            enabled: !RELEASE
        },
        'vtrace-methods': {
            type: 3,
            optDesc: ' <java/lang/Object/getHashCode()I:...>',
            desc: 'specify particular methods to vtrace separated by colons',
            enabled: !RELEASE
        },
        'list-class-cache': { desc: 'list all of the bootstrap loaded classes after execution' },
        'dump-compiled-code': {
            type: 3,
            optDesc: ' <directory>',
            desc: 'location to dump compiled object definitions',
            enabled: !RELEASE
        },
        'native-classpath': {
            type: 3,
            optDesc: ' <class search path of directories>',
            desc: 'A : separated list of directories to search for native mathods in JS files.'
        },
        'bootclasspath/a': {
            type: 1,
            optDesc: ':<directories and zip/jar files separated by :>',
            desc: 'append to end of bootstrap class path'
        },
        'bootclasspath/p': {
            type: 1,
            optDesc: ':<directories and zip/jar files separated by :>',
            desc: 'prepend in front of bootstrap class path'
        },
        'bootclasspath': {
            type: 1,
            optDesc: ':<directories and zip/jar files separated by :>',
            desc: 'set search path for bootstrap classes and resources'
        },
        'X:+PrintCompilation': {
            desc: 'Print JIT compilation details',
            enabled: !RELEASE
        }
    }
});
function java(args, opts, doneCb, jvmStarted) {
    if (jvmStarted === void 0) {
        jvmStarted = function (jvm) {
        };
    }
    var parsedArgs = parser.parse(args), standard = parsedArgs['default'], nonStandard = parsedArgs['X'], jvmState;
    opts.properties = standard.mapOption('D');
    if (standard.flag('help', false)) {
        return printHelp(opts.launcherName, parser.help('default'), doneCb, 0);
    } else if (standard.flag('X', false)) {
        return printNonStandardHelp(opts.launcherName, parser.help('X'), doneCb, 0);
    }
    var logOption = nonStandard.stringOption('log', 'ERROR');
    opts.intMode = nonStandard.flag('int', false);
    opts.dumpJITStats = nonStandard.flag('dump-JIT-stats', false);
    if (/^[0-9]+$/.test(logOption)) {
        logging_1.setLogLevel(parseInt(logOption, 10));
    } else {
        var level = logging_1.LogLevel[logOption.toUpperCase()];
        if (level == null) {
            process.stderr.write('Unrecognized log level: ' + logOption + '.');
            return printHelp(opts.launcherName, parser.help('default'), doneCb, 1);
        }
        logging_1.setLogLevel(level);
    }
    if (nonStandard.flag('list-class-cache', false)) {
        doneCb = function (old_done_cb) {
            return function (result) {
                var fpaths = jvmState.getBootstrapClassLoader().getLoadedClassFiles();
                process.stdout.write(fpaths.join('\n') + '\n');
                old_done_cb(result);
            };
        }(doneCb);
    }
    if (standard.flag('enablesystemassertions', false)) {
        opts.enableSystemAssertions = true;
    }
    if (standard.flag('disablesystemassertions', false)) {
        opts.enableSystemAssertions = false;
    }
    if (standard.flag('enableassertions', false)) {
        opts.enableAssertions = true;
    } else if (standard.stringOption('enableassertions', null)) {
        opts.enableAssertions = standard.stringOption('enableassertions', null).split(':');
    }
    if (standard.stringOption('disableassertions', null)) {
        opts.disableAssertions = standard.stringOption('disableassertions', null).split(':');
    }
    var bscl = nonStandard.stringOption('bootclasspath', null);
    if (bscl !== null) {
        opts.bootstrapClasspath = bscl.split(':');
    }
    var bsClAppend = nonStandard.stringOption('bootclasspath/a', null);
    if (bsClAppend) {
        opts.bootstrapClasspath = opts.bootstrapClasspath.concat(bsClAppend.split(':'));
    }
    var bsClPrepend = nonStandard.stringOption('bootclasspath/p', null);
    if (bsClPrepend) {
        opts.bootstrapClasspath = bsClPrepend.split(':').concat(opts.bootstrapClasspath);
    }
    if (!opts.classpath) {
        opts.classpath = [];
    }
    if (standard.stringOption('jar', null)) {
        opts.classpath.push(standard.stringOption('jar', null));
    } else if (standard.stringOption('classpath', null)) {
        opts.classpath = opts.classpath.concat(standard.stringOption('classpath', null).split(':'));
    } else {
        opts.classpath.push(process.cwd());
    }
    var nativeClasspath = standard.stringOption('native-classpath', null);
    if (nativeClasspath) {
        opts.nativeClasspath = opts.nativeClasspath.concat(nativeClasspath.split(':'));
    }
    jvmState = new jvm_1['default'](opts, function (err) {
        if (err) {
            process.stderr.write('Error constructing JVM:\n');
            process.stderr.write(err.toString() + '\n');
            doneCb(1);
        } else {
            launchJvm(standard, opts, jvmState, doneCb, jvmStarted);
        }
    });
    jvmState.setPrintJITCompilation(nonStandard.flag('X:+PrintCompilation', false));
    var vtraceMethods = nonStandard.stringOption('vtrace-methods', null);
    if (vtraceMethods) {
        vtraceMethods.split(':').forEach(function (m) {
            return jvmState.vtraceMethod(m);
        });
    }
    var dumpCompiledCode = nonStandard.stringOption('dump-compiled-code', null);
    if (dumpCompiledCode) {
        jvmState.dumpCompiledCode(dumpCompiledCode);
    }
}
exports.__esModule = true;
exports['default'] = java;
function launchJvm(standardOptions, opts, jvmState, doneCb, jvmStarted) {
    var mainArgs = standardOptions.unparsedArgs();
    if (standardOptions.stringOption('jar', null)) {
        jvmState.runJar(mainArgs, doneCb);
        jvmStarted(jvmState);
    } else if (mainArgs.length > 0) {
        var cname = mainArgs[0];
        if (cname.slice(-6) === '.class') {
            cname = cname.slice(0, -6);
        }
        if (cname.indexOf('.') !== -1) {
            cname = util_1.descriptor2typestr(util_1.int_classname(cname));
        }
        jvmState.runClass(cname, mainArgs.slice(1), doneCb);
        jvmStarted(jvmState);
    } else {
        printHelp(opts.launcherName, parser.help('default'), doneCb, 0);
    }
}
function printHelp(launcherName, str, doneCb, rv) {
    process.stdout.write('Usage: ' + launcherName + ' [-options] class [args...]\n        (to execute a class)\nor  ' + launcherName + ' [-options] -jar jarfile [args...]\n        (to execute a jar file)\nwhere options include:\n' + str);
    doneCb(rv);
}
function printNonStandardHelp(launcherName, str, doneCb, rv) {
    process.stdout.write(str + '\n\nThe -X options are non-standard and subject to change without notice.\n');
    doneCb(rv);
}
//# sourceMappingURL=java_cli.js.map