'use strict';
var util_1 = require('./util');
var SafeMap_1 = require('./SafeMap');
var methods_1 = require('./methods');
var ClassLoader_1 = require('./ClassLoader');
var fs = require('fs');
var path = require('path');
var buffer = require('buffer');
var threading_1 = require('./threading');
var enums_1 = require('./enums');
var heap_1 = require('./heap');
var assert_1 = require('./assert');
var parker_1 = require('./parker');
var threadpool_1 = require('./threadpool');
var JDKInfo = require('../vendor/java_home/jdk.json');
var global_1 = require('./global');
var global_require_1 = require('./global_require');
var BrowserFS = require('browserfs');
var DoppioJVM = require('./doppiojvm');
if (typeof RELEASE === 'undefined')
    global_1['default'].RELEASE = false;
var pkg;
if (util_1.are_in_browser()) {
    pkg = require('../package.json');
} else {
    pkg = require('../../../package.json');
}
var coreClasses = [
    'Ljava/lang/String;',
    'Ljava/lang/Class;',
    'Ljava/lang/ClassLoader;',
    'Ljava/lang/reflect/Constructor;',
    'Ljava/lang/reflect/Field;',
    'Ljava/lang/reflect/Method;',
    'Ljava/lang/Error;',
    'Ljava/lang/StackTraceElement;',
    'Ljava/lang/System;',
    'Ljava/lang/Thread;',
    'Ljava/lang/ThreadGroup;',
    'Ljava/lang/Throwable;',
    'Ljava/nio/ByteOrder;',
    'Lsun/misc/VM;',
    'Lsun/reflect/ConstantPool;',
    'Ljava/lang/Byte;',
    'Ljava/lang/Character;',
    'Ljava/lang/Double;',
    'Ljava/lang/Float;',
    'Ljava/lang/Integer;',
    'Ljava/lang/Long;',
    'Ljava/lang/Short;',
    'Ljava/lang/Void;',
    'Ljava/io/FileDescriptor;',
    'Ljava/lang/Boolean;',
    '[Lsun/management/MemoryManagerImpl;',
    '[Lsun/management/MemoryPoolImpl;',
    'Lsun/nio/fs/UnixConstants;'
];
var JVM = function () {
    function JVM(opts, cb) {
        var _this = this;
        this.systemProperties = null;
        this.internedStrings = new SafeMap_1['default']();
        this.bsCl = null;
        this.threadPool = null;
        this.natives = {};
        this.heap = new heap_1['default'](20 * 1024 * 1024);
        this.nativeClasspath = null;
        this.startupTime = new Date();
        this.terminationCb = null;
        this.firstThread = null;
        this.responsiveness = null;
        this.enableSystemAssertions = false;
        this.enabledAssertions = false;
        this.disabledAssertions = [];
        this.printJITCompilation = false;
        this.systemClassLoader = null;
        this.nextRef = 0;
        this.vtraceMethods = {};
        this.dumpCompiledCodeDir = null;
        this.parker = new parker_1['default']();
        this.status = enums_1.JVMStatus.BOOTING;
        this.exitCode = 0;
        this.jitDisabled = false;
        this.dumpJITStats = false;
        this.globalRequire = null;
        if (typeof opts.doppioHomePath !== 'string') {
            throw new TypeError('opts.doppioHomePath *must* be specified.');
        }
        opts = util_1.merge(JVM.getDefaultOptions(opts.doppioHomePath), opts);
        this.jitDisabled = opts.intMode;
        this.dumpJITStats = opts.dumpJITStats;
        var bootstrapClasspath = opts.bootstrapClasspath.map(function (p) {
                return path.resolve(p);
            }), bootupTasks = [], firstThread, firstThreadObj;
        if (!Array.isArray(opts.bootstrapClasspath) || opts.bootstrapClasspath.length === 0) {
            throw new TypeError('opts.bootstrapClasspath must be specified as an array of file paths.');
        }
        if (!Array.isArray(opts.classpath)) {
            throw new TypeError('opts.classpath must be specified as an array of file paths.');
        }
        if (typeof opts.javaHomePath !== 'string') {
            throw new TypeError('opts.javaHomePath must be specified.');
        }
        if (!opts.nativeClasspath) {
            opts.nativeClasspath = [];
        }
        if (!Array.isArray(opts.nativeClasspath)) {
            throw new TypeError('opts.nativeClasspath must be specified as an array of file paths.');
        }
        this.nativeClasspath = opts.nativeClasspath;
        if (opts.enableSystemAssertions) {
            this.enableSystemAssertions = opts.enableSystemAssertions;
        }
        if (opts.enableAssertions) {
            this.enabledAssertions = opts.enableAssertions;
        }
        if (opts.disableAssertions) {
            this.disabledAssertions = opts.disableAssertions;
        }
        this.responsiveness = opts.responsiveness;
        this._initSystemProperties(bootstrapClasspath, opts.classpath.map(function (p) {
            return path.resolve(p);
        }), path.resolve(opts.javaHomePath), path.resolve(opts.tmpDir), opts.properties);
        bootupTasks.push(function (next) {
            _this.initializeNatives(next);
        });
        bootupTasks.push(function (next) {
            _this.bsCl = new ClassLoader_1.BootstrapClassLoader(_this.systemProperties['java.home'], bootstrapClasspath, next);
        });
        bootupTasks.push(function (next) {
            _this.threadPool = new threadpool_1['default'](function () {
                return _this.threadPoolIsEmpty();
            });
            _this.bsCl.resolveClass(null, 'Ljava/lang/Thread;', function (threadCdata) {
                if (threadCdata == null) {
                    next('Failed to resolve java/lang/Thread.');
                } else {
                    firstThreadObj = new (threadCdata.getConstructor(null))(null);
                    firstThreadObj.$thread = firstThread = _this.firstThread = new threading_1.JVMThread(_this, _this.threadPool, firstThreadObj);
                    firstThreadObj.ref = 1;
                    firstThreadObj['java/lang/Thread/priority'] = 5;
                    firstThreadObj['java/lang/Thread/name'] = util_1.initCarr(_this.bsCl, 'main');
                    firstThreadObj['java/lang/Thread/blockerLock'] = new (_this.bsCl.getResolvedClass('Ljava/lang/Object;').getConstructor(firstThread))(firstThread);
                    next();
                }
            });
        });
        bootupTasks.push(function (next) {
            util_1.asyncForEach(coreClasses, function (coreClass, nextItem) {
                _this.bsCl.initializeClass(firstThread, coreClass, function (cdata) {
                    if (cdata == null) {
                        nextItem('Failed to initialize ' + coreClass);
                    } else {
                        if (coreClass === 'Ljava/lang/ThreadGroup;') {
                            var threadGroupCons = cdata.getConstructor(firstThread), groupObj = new threadGroupCons(firstThread);
                            groupObj['<init>()V'](firstThread, null, function (e) {
                                firstThreadObj['java/lang/Thread/group'] = groupObj;
                                nextItem(e);
                            });
                        } else {
                            nextItem();
                        }
                    }
                });
            }, next);
        });
        bootupTasks.push(function (next) {
            var sysInit = _this.bsCl.getInitializedClass(firstThread, 'Ljava/lang/System;').getConstructor(firstThread);
            sysInit['java/lang/System/initializeSystemClass()V'](firstThread, null, next);
            ;
        });
        bootupTasks.push(function (next) {
            var clCons = _this.bsCl.getInitializedClass(firstThread, 'Ljava/lang/ClassLoader;').getConstructor(firstThread);
            clCons['java/lang/ClassLoader/getSystemClassLoader()Ljava/lang/ClassLoader;'](firstThread, null, function (e, rv) {
                if (e) {
                    next(e);
                } else {
                    _this.systemClassLoader = rv.$loader;
                    firstThreadObj['java/lang/Thread/contextClassLoader'] = rv;
                    var defaultAssertionStatus = _this.enabledAssertions === true ? 1 : 0;
                    rv['java/lang/ClassLoader/setDefaultAssertionStatus(Z)V'](firstThread, [defaultAssertionStatus], next);
                }
            });
        });
        bootupTasks.push(function (next) {
            _this.bsCl.initializeClass(firstThread, 'Ldoppio/security/DoppioProvider;', function (cdata) {
                next(cdata ? null : new Error('Failed to initialize DoppioProvider.'));
            });
        });
        util_1.asyncSeries(bootupTasks, function (err) {
            setImmediate(function () {
                if (err) {
                    _this.status = enums_1.JVMStatus.TERMINATED;
                    cb(err);
                } else {
                    _this.status = enums_1.JVMStatus.BOOTED;
                    cb(null, _this);
                }
            });
        });
    }
    JVM.isReleaseBuild = function () {
        return typeof RELEASE !== 'undefined' && RELEASE;
    };
    JVM.getNativeMethodModules = function () {
        if (!this._haveAddedBuiltinNativeModules) {
            JVM.registerNativeModule(require('./natives/doppio').default);
            JVM.registerNativeModule(require('./natives/java_io').default);
            JVM.registerNativeModule(require('./natives/java_lang').default);
            JVM.registerNativeModule(require('./natives/java_net').default);
            JVM.registerNativeModule(require('./natives/java_nio').default);
            JVM.registerNativeModule(require('./natives/java_security').default);
            JVM.registerNativeModule(require('./natives/java_util').default);
            JVM.registerNativeModule(require('./natives/sun_font').default);
            JVM.registerNativeModule(require('./natives/sun_management').default);
            JVM.registerNativeModule(require('./natives/sun_misc').default);
            JVM.registerNativeModule(require('./natives/sun_net').default);
            JVM.registerNativeModule(require('./natives/sun_nio').default);
            JVM.registerNativeModule(require('./natives/sun_reflect').default);
            this._haveAddedBuiltinNativeModules = true;
        }
        return this._nativeMethodModules;
    };
    JVM.registerNativeModule = function (mod) {
        this._nativeMethodModules.push(mod);
    };
    JVM.prototype.getResponsiveness = function () {
        var resp = this.responsiveness;
        if (typeof resp === 'number') {
            return resp;
        } else if (typeof resp === 'function') {
            return resp();
        }
    };
    JVM.getDefaultOptions = function (doppioHome) {
        var javaHome = path.join(doppioHome, 'vendor', 'java_home');
        return {
            doppioHomePath: doppioHome,
            classpath: ['.'],
            bootstrapClasspath: JDKInfo.classpath.map(function (item) {
                return path.join(javaHome, item);
            }),
            javaHomePath: javaHome,
            nativeClasspath: [],
            enableSystemAssertions: false,
            enableAssertions: false,
            disableAssertions: null,
            properties: {},
            tmpDir: '/tmp',
            responsiveness: 1000,
            intMode: false,
            dumpJITStats: false
        };
    };
    JVM.getCompiledJDKURL = function () {
        return JDKInfo.url;
    };
    JVM.getJDKInfo = function () {
        return JDKInfo;
    };
    JVM.prototype.getSystemClassLoader = function () {
        return this.systemClassLoader;
    };
    JVM.prototype.getNextRef = function () {
        return this.nextRef++;
    };
    JVM.prototype.getParker = function () {
        return this.parker;
    };
    JVM.prototype.runClass = function (className, args, cb) {
        var _this = this;
        if (this.status !== enums_1.JVMStatus.BOOTED) {
            switch (this.status) {
            case enums_1.JVMStatus.BOOTING:
                throw new Error('JVM is currently booting up. Please wait for it to call the bootup callback, which you passed to the constructor.');
            case enums_1.JVMStatus.RUNNING:
                throw new Error('JVM is already running.');
            case enums_1.JVMStatus.TERMINATED:
                throw new Error('This JVM has already terminated. Please create a new JVM.');
            case enums_1.JVMStatus.TERMINATING:
                throw new Error('This JVM is currently terminating. You should create a new JVM for each class you wish to run.');
            }
        }
        this.terminationCb = cb;
        var thread = this.firstThread;
        assert_1['default'](thread != null, 'Thread isn\'t created yet?');
        className = util_1.int_classname(className);
        this.systemClassLoader.initializeClass(thread, className, function (cdata) {
            if (cdata != null) {
                var strArrCons = _this.bsCl.getInitializedClass(thread, '[Ljava/lang/String;').getConstructor(thread), jvmifiedArgs = new strArrCons(thread, args.length), i;
                for (i = 0; i < args.length; i++) {
                    jvmifiedArgs.array[i] = util_1.initString(_this.bsCl, args[i]);
                }
                _this.status = enums_1.JVMStatus.RUNNING;
                var cdataStatics = cdata.getConstructor(thread);
                if (cdataStatics['main([Ljava/lang/String;)V']) {
                    cdataStatics['main([Ljava/lang/String;)V'](thread, [jvmifiedArgs]);
                } else {
                    thread.throwNewException('Ljava/lang/NoSuchMethodError;', 'Could not find main method in class ' + cdata.getExternalName() + '.');
                }
            } else {
                process.stdout.write('Error: Could not find or load main class ' + util_1.ext_classname(className) + '\n');
                _this.terminationCb(1);
            }
        });
    };
    JVM.prototype.isJITDisabled = function () {
        return this.jitDisabled;
    };
    JVM.prototype.shouldVtrace = function (sig) {
        return this.vtraceMethods[sig] === true;
    };
    JVM.prototype.vtraceMethod = function (sig) {
        this.vtraceMethods[sig] = true;
    };
    JVM.prototype.runJar = function (args, cb) {
        this.runClass('doppio.JarLauncher', args, cb);
    };
    JVM.prototype.threadPoolIsEmpty = function () {
        var systemClass, systemCons;
        switch (this.status) {
        case enums_1.JVMStatus.BOOTING:
            return false;
        case enums_1.JVMStatus.BOOTED:
            assert_1['default'](false, 'Thread pool should not become empty after JVM is booted, but before it begins to run.');
            return false;
        case enums_1.JVMStatus.RUNNING:
            this.status = enums_1.JVMStatus.TERMINATING;
            systemClass = this.bsCl.getInitializedClass(this.firstThread, 'Ljava/lang/System;');
            assert_1['default'](systemClass !== null, 'Invariant failure: System class must be initialized when JVM is in RUNNING state.');
            systemCons = systemClass.getConstructor(this.firstThread);
            systemCons['java/lang/System/exit(I)V'](this.firstThread, [0]);
            return false;
        case enums_1.JVMStatus.TERMINATED:
            assert_1['default'](false, 'Invariant failure: Thread pool cannot be emptied post-JVM termination.');
            return false;
        case enums_1.JVMStatus.TERMINATING:
            if (!RELEASE && this.dumpJITStats) {
                methods_1.dumpStats();
            }
            this.status = enums_1.JVMStatus.TERMINATED;
            if (this.terminationCb) {
                this.terminationCb(this.exitCode);
            }
            this.firstThread.close();
            return true;
        }
    };
    JVM.prototype.hasVMBooted = function () {
        return !(this.status === enums_1.JVMStatus.BOOTING || this.status === enums_1.JVMStatus.BOOTED);
    };
    JVM.prototype.halt = function (status) {
        this.exitCode = status;
        this.status = enums_1.JVMStatus.TERMINATING;
        this.threadPool.getThreads().forEach(function (t) {
            t.setStatus(enums_1.ThreadStatus.TERMINATED);
        });
    };
    JVM.prototype.getSystemProperty = function (prop) {
        return this.systemProperties[prop];
    };
    JVM.prototype.getSystemPropertyNames = function () {
        return Object.keys(this.systemProperties);
    };
    JVM.prototype.getHeap = function () {
        return this.heap;
    };
    JVM.prototype.internString = function (str, javaObj) {
        if (this.internedStrings.has(str)) {
            return this.internedStrings.get(str);
        } else {
            if (!javaObj) {
                javaObj = util_1.initString(this.bsCl, str);
            }
            this.internedStrings.set(str, javaObj);
            return javaObj;
        }
    };
    JVM.prototype.evalNativeModule = function (mod) {
        if (!this.globalRequire) {
            this.globalRequire = global_require_1['default']();
        }
        var rv;
        function registerNatives(defs) {
            rv = defs;
        }
        var globalRequire = this.globalRequire;
        function moduleRequire(name) {
            switch (name) {
            case 'doppiojvm':
            case '../doppiojvm':
                return DoppioJVM;
            case 'fs':
                return fs;
            case 'path':
                return path;
            case 'buffer':
                return buffer;
            case 'browserfs':
                return BrowserFS;
            default:
                return globalRequire(name);
            }
        }
        function moduleDefine(resources, module) {
            var args = [];
            resources.forEach(function (resource) {
                switch (resource) {
                case 'require':
                    args.push(moduleRequire);
                    break;
                case 'exports':
                    args.push({});
                    break;
                default:
                    args.push(moduleRequire(resource));
                    break;
                }
            });
            module.apply(null, args);
        }
        var modFcn = new Function('require', 'define', 'registerNatives', 'process', 'DoppioJVM', 'Buffer', mod);
        modFcn(moduleRequire, moduleDefine, registerNatives, process, DoppioJVM, Buffer);
        return rv;
    };
    JVM.prototype.registerNatives = function (newNatives) {
        var clsName, methSig;
        for (clsName in newNatives) {
            if (newNatives.hasOwnProperty(clsName)) {
                if (!this.natives.hasOwnProperty(clsName)) {
                    this.natives[clsName] = {};
                }
                var clsMethods = newNatives[clsName];
                for (methSig in clsMethods) {
                    if (clsMethods.hasOwnProperty(methSig)) {
                        this.natives[clsName][methSig] = clsMethods[methSig];
                    }
                }
            }
        }
    };
    JVM.prototype.registerNative = function (clsName, methSig, native) {
        this.registerNatives({ clsName: { methSig: native } });
    };
    JVM.prototype.getNative = function (clsName, methSig) {
        clsName = util_1.descriptor2typestr(clsName);
        if (this.natives.hasOwnProperty(clsName)) {
            var clsMethods = this.natives[clsName];
            if (clsMethods.hasOwnProperty(methSig)) {
                return clsMethods[methSig];
            }
        }
        return null;
    };
    JVM.prototype.getNatives = function () {
        return this.natives;
    };
    JVM.prototype.initializeNatives = function (doneCb) {
        var _this = this;
        var registeredModules = JVM.getNativeMethodModules();
        for (var i_1 = 0; i_1 < registeredModules.length; i_1++) {
            this.registerNatives(registeredModules[i_1]());
        }
        var nextDir = function () {
                if (i === _this.nativeClasspath.length) {
                    var count_1 = processFiles.length;
                    if (count_1 === 0) {
                        return doneCb();
                    }
                    processFiles.forEach(function (file) {
                        fs.readFile(file, function (err, data) {
                            if (!err) {
                                _this.registerNatives(_this.evalNativeModule(data.toString()));
                            }
                            if (--count_1 === 0) {
                                doneCb();
                            }
                        });
                    });
                } else {
                    var dir = _this.nativeClasspath[i++];
                    fs.readdir(dir, function (err, files) {
                        if (err) {
                            return doneCb();
                        }
                        var j, file;
                        for (j = 0; j < files.length; j++) {
                            file = files[j];
                            if (file.substring(file.length - 3, file.length) === '.js') {
                                processFiles.push(path.join(dir, file));
                            }
                        }
                        nextDir();
                    });
                }
            }, i = 0, processFiles = [];
        nextDir();
    };
    JVM.prototype._initSystemProperties = function (bootstrapClasspath, javaClassPath, javaHomePath, tmpDir, opts) {
        this.systemProperties = util_1.merge({
            'java.class.path': javaClassPath.join(':'),
            'java.home': javaHomePath,
            'java.ext.dirs': path.join(javaHomePath, 'lib', 'ext'),
            'java.io.tmpdir': tmpDir,
            'sun.boot.class.path': bootstrapClasspath.join(':'),
            'file.encoding': 'UTF-8',
            'java.vendor': 'Doppio',
            'java.version': '1.8',
            'java.vendor.url': 'https://github.com/plasma-umass/doppio',
            'java.class.version': '52.0',
            'java.specification.version': '1.8',
            'line.separator': '\n',
            'file.separator': path.sep,
            'path.separator': ':',
            'user.dir': path.resolve('.'),
            'user.home': '.',
            'user.name': 'DoppioUser',
            'os.name': 'doppio',
            'os.arch': 'js',
            'os.version': '0',
            'java.vm.name': 'DoppioJVM 32-bit VM',
            'java.vm.version': pkg.version,
            'java.vm.vendor': 'PLASMA@UMass',
            'java.awt.headless': util_1.are_in_browser().toString(),
            'java.awt.graphicsenv': 'classes.awt.CanvasGraphicsEnvironment',
            'jline.terminal': 'jline.UnsupportedTerminal',
            'sun.arch.data.model': '32',
            'sun.jnu.encoding': 'UTF-8'
        }, opts);
    };
    JVM.prototype.getBootstrapClassLoader = function () {
        return this.bsCl;
    };
    JVM.prototype.getStartupTime = function () {
        return this.startupTime;
    };
    JVM.prototype.areSystemAssertionsEnabled = function () {
        return this.enableSystemAssertions;
    };
    JVM.prototype.getEnabledAssertions = function () {
        return this.enabledAssertions;
    };
    JVM.prototype.getDisabledAssertions = function () {
        return this.disabledAssertions;
    };
    JVM.prototype.setPrintJITCompilation = function (enabledOrNot) {
        this.printJITCompilation = enabledOrNot;
    };
    JVM.prototype.shouldPrintJITCompilation = function () {
        return this.printJITCompilation;
    };
    JVM.prototype.dumpCompiledCode = function (dir) {
        this.dumpCompiledCodeDir = dir;
    };
    JVM.prototype.shouldDumpCompiledCode = function () {
        return this.dumpCompiledCodeDir !== null;
    };
    JVM.prototype.dumpObjectDefinition = function (cls, evalText) {
        if (this.shouldDumpCompiledCode()) {
            fs.writeFile(path.resolve(this.dumpCompiledCodeDir, cls.getExternalName() + '.js'), evalText, function () {
            });
        }
    };
    JVM.prototype.dumpBridgeMethod = function (methodSig, evalText) {
        if (this.shouldDumpCompiledCode()) {
            fs.appendFile(path.resolve(this.dumpCompiledCodeDir, 'vmtarget_bridge_methods.dump'), methodSig + ':\n' + evalText + '\n\n', function () {
            });
        }
    };
    JVM.prototype.dumpCompiledMethod = function (methodSig, pc, code) {
        if (this.shouldDumpCompiledCode()) {
            fs.appendFile(path.resolve(this.dumpCompiledCodeDir, 'JIT_compiled_methods.dump'), methodSig + ':' + pc + ':\n' + code + '\n\n', function () {
            });
        }
    };
    JVM.prototype.dumpState = function (filename, cb) {
        fs.appendFile(filename, this.threadPool.getThreads().map(function (t) {
            return 'Thread ' + t.getRef() + ':\n' + t.getPrintableStackTrace();
        }).join('\n\n'), cb);
    };
    JVM._nativeMethodModules = [];
    JVM._haveAddedBuiltinNativeModules = false;
    return JVM;
}();
exports.__esModule = true;
exports['default'] = JVM;
//# sourceMappingURL=jvm.js.map