'use strict';
var __extends = this && this.__extends || function (d, b) {
    for (var p in b)
        if (b.hasOwnProperty(p))
            d[p] = b[p];
    function __() {
        this.constructor = d;
    }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var util_1 = require('./util');
var util = require('./util');
var attributes_1 = require('./attributes');
var threading_1 = require('./threading');
var assert_1 = require('./assert');
var enums_1 = require('./enums');
var StringOutputStream_1 = require('./StringOutputStream');
var global_1 = require('./global');
var jit_1 = require('./jit');
if (typeof RELEASE === 'undefined')
    global_1['default'].RELEASE = false;
var trapped_methods = {
    'java/lang/ref/Reference': {
        '<clinit>()V': function (thread) {
        }
    },
    'java/lang/System': {
        'loadLibrary(Ljava/lang/String;)V': function (thread, libName) {
            var lib = libName.toString();
            switch (lib) {
            case 'zip':
            case 'net':
            case 'nio':
            case 'awt':
            case 'fontmanager':
            case 'management':
                return;
            default:
                thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'no ' + lib + ' in java.library.path');
                break;
            }
        }
    },
    'java/lang/Terminator': {
        'setup()V': function (thread) {
        }
    },
    'java/nio/charset/Charset$3': {
        'run()Ljava/lang/Object;': function (thread, javaThis) {
            return null;
        }
    },
    'sun/nio/fs/DefaultFileSystemProvider': {
        'create()Ljava/nio/file/spi/FileSystemProvider;': function (thread) {
            thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
            var dfsp = thread.getBsCl().getInitializedClass(thread, 'Lsun/nio/fs/DefaultFileSystemProvider;'), dfspCls = dfsp.getConstructor(thread);
            dfspCls['createProvider(Ljava/lang/String;)Ljava/nio/file/spi/FileSystemProvider;'](thread, [thread.getJVM().internString('sun.nio.fs.LinuxFileSystemProvider')], util_1.forwardResult(thread));
        }
    }
};
function getTrappedMethod(clsName, methSig) {
    clsName = util_1.descriptor2typestr(clsName);
    if (trapped_methods.hasOwnProperty(clsName) && trapped_methods[clsName].hasOwnProperty(methSig)) {
        return trapped_methods[clsName][methSig];
    }
    return null;
}
var AbstractMethodField = function () {
    function AbstractMethodField(cls, constantPool, slot, byteStream) {
        this.cls = cls;
        this.slot = slot;
        this.accessFlags = new util_1.Flags(byteStream.getUint16());
        this.name = constantPool.get(byteStream.getUint16()).value;
        this.rawDescriptor = constantPool.get(byteStream.getUint16()).value;
        this.attrs = attributes_1.makeAttributes(byteStream, constantPool);
    }
    AbstractMethodField.prototype.getAttribute = function (name) {
        for (var i = 0; i < this.attrs.length; i++) {
            var attr = this.attrs[i];
            if (attr.getName() === name) {
                return attr;
            }
        }
        return null;
    };
    AbstractMethodField.prototype.getAttributes = function (name) {
        return this.attrs.filter(function (attr) {
            return attr.getName() === name;
        });
    };
    AbstractMethodField.prototype.getAnnotationType = function (thread, name) {
        var annotation = this.getAttribute(name);
        if (annotation === null) {
            return null;
        }
        var byteArrCons = thread.getBsCl().getInitializedClass(thread, '[B').getConstructor(thread), rv = new byteArrCons(thread, 0);
        var i, len = annotation.rawBytes.length, arr = new Array(len);
        for (i = 0; i < len; i++) {
            arr[i] = annotation.rawBytes.readInt8(i);
        }
        rv.array = arr;
        return rv;
    };
    AbstractMethodField.prototype.parseDescriptor = function (raw_descriptor) {
        throw new Error('Unimplemented error.');
    };
    return AbstractMethodField;
}();
exports.AbstractMethodField = AbstractMethodField;
var Field = function (_super) {
    __extends(Field, _super);
    function Field(cls, constantPool, slot, byteStream) {
        _super.call(this, cls, constantPool, slot, byteStream);
        this.fullName = util_1.descriptor2typestr(cls.getInternalName()) + '/' + this.name;
    }
    Field.prototype.reflector = function (thread, cb) {
        var _this = this;
        var signatureAttr = this.getAttribute('Signature'), jvm = thread.getJVM(), bsCl = thread.getBsCl();
        var createObj = function (typeObj) {
            var fieldCls = bsCl.getInitializedClass(thread, 'Ljava/lang/reflect/Field;'), fieldObj = new (fieldCls.getConstructor(thread))(thread);
            fieldObj['java/lang/reflect/Field/clazz'] = _this.cls.getClassObject(thread);
            fieldObj['java/lang/reflect/Field/name'] = jvm.internString(_this.name);
            fieldObj['java/lang/reflect/Field/type'] = typeObj;
            fieldObj['java/lang/reflect/Field/modifiers'] = _this.accessFlags.getRawByte();
            fieldObj['java/lang/reflect/Field/slot'] = _this.slot;
            fieldObj['java/lang/reflect/Field/signature'] = signatureAttr !== null ? util_1.initString(bsCl, signatureAttr.sig) : null;
            fieldObj['java/lang/reflect/Field/annotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleAnnotations');
            return fieldObj;
        };
        this.cls.getLoader().resolveClass(thread, this.rawDescriptor, function (cdata) {
            if (cdata != null) {
                cb(createObj(cdata.getClassObject(thread)));
            } else {
                cb(null);
            }
        });
    };
    Field.prototype.getDefaultFieldValue = function () {
        var desc = this.rawDescriptor;
        if (desc === 'J')
            return 'gLongZero';
        var c = desc[0];
        if (c === '[' || c === 'L')
            return 'null';
        return '0';
    };
    Field.prototype.outputJavaScriptField = function (jsConsName, outputStream) {
        if (this.accessFlags.isStatic()) {
            outputStream.write(jsConsName + '["' + util_1.reescapeJVMName(this.fullName) + '"] = cls._getInitialStaticFieldValue(thread, "' + util_1.reescapeJVMName(this.name) + '");\n');
        } else {
            outputStream.write('this["' + util_1.reescapeJVMName(this.fullName) + '"] = ' + this.getDefaultFieldValue() + ';\n');
        }
    };
    return Field;
}(AbstractMethodField);
exports.Field = Field;
var opcodeSize = function () {
    var table = [];
    table[enums_1.OpcodeLayoutType.OPCODE_ONLY] = 1;
    table[enums_1.OpcodeLayoutType.CONSTANT_POOL_UINT8] = 2;
    table[enums_1.OpcodeLayoutType.CONSTANT_POOL] = 3;
    table[enums_1.OpcodeLayoutType.CONSTANT_POOL_AND_UINT8_VALUE] = 4;
    table[enums_1.OpcodeLayoutType.UINT8_VALUE] = 2;
    table[enums_1.OpcodeLayoutType.UINT8_AND_INT8_VALUE] = 3;
    table[enums_1.OpcodeLayoutType.INT8_VALUE] = 2;
    table[enums_1.OpcodeLayoutType.INT16_VALUE] = 3;
    table[enums_1.OpcodeLayoutType.INT32_VALUE] = 5;
    table[enums_1.OpcodeLayoutType.ARRAY_TYPE] = 2;
    table[enums_1.OpcodeLayoutType.WIDE] = 1;
    return table;
}();
var TraceInfo = function () {
    function TraceInfo(pc, jitInfo) {
        this.pc = pc;
        this.jitInfo = jitInfo;
        this.pops = [];
        this.pushes = [];
        this.prefixEmit = '';
    }
    return TraceInfo;
}();
var Trace = function () {
    function Trace(startPC, code, method) {
        this.startPC = startPC;
        this.code = code;
        this.method = method;
        this.infos = [];
        this.endPc = -1;
    }
    Trace.prototype.emitEndPC = function (pc) {
        this.endPc = pc;
    };
    Trace.prototype.addOp = function (pc, jitInfo) {
        this.infos.push(new TraceInfo(pc, jitInfo));
    };
    Trace.prototype.close = function (thread) {
        if (this.infos.length > 1) {
            var symbolicStack = [];
            var symbolCount = 0;
            var emitted = this.endPc > -1 ? 'f.pc=' + this.endPc + ';' : '';
            for (var i = 0; i < this.infos.length; i++) {
                var info = this.infos[i];
                var jitInfo = info.jitInfo;
                var pops = info.pops;
                var normalizedPops = jitInfo.pops < 0 ? Math.min(-jitInfo.pops, symbolicStack.length) : jitInfo.pops;
                for (var j = 0; j < normalizedPops; j++) {
                    if (symbolicStack.length > 0) {
                        pops.push(symbolicStack.pop());
                    } else {
                        var symbol = 's' + symbolCount++;
                        info.prefixEmit += 'var ' + symbol + ' = f.opStack.pop();';
                        pops.push(symbol);
                    }
                }
                info.onErrorPushes = symbolicStack.slice();
                var pushes = info.pushes;
                for (var j = 0; j < jitInfo.pushes; j++) {
                    var symbol = 's' + symbolCount++;
                    symbolicStack.push(symbol);
                    pushes.push(symbol);
                }
            }
            if (symbolicStack.length === 1) {
                emitted += 'f.opStack.push(' + symbolicStack[0] + ');';
            } else if (symbolicStack.length > 1) {
                emitted += 'f.opStack.pushAll(' + symbolicStack.join(',') + ');';
            }
            for (var i = this.infos.length - 1; i >= 0; i--) {
                var info = this.infos[i];
                var jitInfo = info.jitInfo;
                emitted = info.prefixEmit + jitInfo.emit(info.pops, info.pushes, '' + i, emitted, this.code, info.pc, info.onErrorPushes, this.method);
            }
            if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
                console.log('Emitted trace of ' + this.infos.length + ' ops: ' + emitted);
            }
            return new Function('f', 't', 'u', emitted);
        } else {
            if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
                console.log('Trace was cancelled');
            }
            return null;
        }
    };
    return Trace;
}();
var Method = function (_super) {
    __extends(Method, _super);
    function Method(cls, constantPool, slot, byteStream) {
        _super.call(this, cls, constantPool, slot, byteStream);
        this.numBBEntries = 0;
        this.compiledFunctions = [];
        this.failedCompile = [];
        var parsedDescriptor = util_1.getTypes(this.rawDescriptor), i, p;
        this.signature = this.name + this.rawDescriptor;
        this.fullSignature = util_1.descriptor2typestr(this.cls.getInternalName()) + '/' + this.signature;
        this.returnType = parsedDescriptor.pop();
        this.parameterTypes = parsedDescriptor;
        this.parameterWords = parsedDescriptor.length;
        for (i = 0; i < this.parameterTypes.length; i++) {
            p = this.parameterTypes[i];
            if (p === 'D' || p === 'J') {
                this.parameterWords++;
            }
        }
        var clsName = this.cls.getInternalName();
        if (getTrappedMethod(clsName, this.signature) !== null) {
            this.code = getTrappedMethod(clsName, this.signature);
            this.accessFlags.setNative(true);
        } else if (this.accessFlags.isNative()) {
            if (this.signature.indexOf('registerNatives()V', 0) < 0 && this.signature.indexOf('initIDs()V', 0) < 0) {
                var self = this;
                this.code = function (thread) {
                    var jvm = thread.getJVM(), c = jvm.getNative(clsName, self.signature);
                    if (c == null) {
                        thread.throwNewException('Ljava/lang/UnsatisfiedLinkError;', 'Native method \'' + self.getFullSignature() + '\' not implemented.\nPlease fix or file a bug at https://github.com/plasma-umass/doppio/issues');
                    } else {
                        self.code = c;
                        return c.apply(self, arguments);
                    }
                };
            } else {
                this.code = function () {
                };
            }
        } else if (!this.accessFlags.isAbstract()) {
            this.code = this.getAttribute('Code');
            var codeLength = this.code.code.length;
            this.numBBEntries = codeLength > 3 ? 200 : 1000 * codeLength;
        }
    }
    Method.prototype.incrBBEntries = function () {
        this.numBBEntries--;
    };
    Method.prototype.isDefault = function () {
        return this.accessFlags.isPublic() && !this.accessFlags.isAbstract() && !this.accessFlags.isStatic() && this.cls.accessFlags.isInterface();
    };
    Method.prototype.getFullSignature = function () {
        return this.cls.getExternalName() + '.' + this.name + this.rawDescriptor;
    };
    Method.prototype.isHidden = function () {
        var rva = this.getAttribute('RuntimeVisibleAnnotations');
        return rva !== null && rva.isHidden;
    };
    Method.prototype.isCallerSensitive = function () {
        var rva = this.getAttribute('RuntimeVisibleAnnotations');
        return rva !== null && rva.isCallerSensitive;
    };
    Method.prototype.getParamWordSize = function () {
        return this.parameterWords;
    };
    Method.prototype.getCodeAttribute = function () {
        assert_1['default'](!this.accessFlags.isNative() && !this.accessFlags.isAbstract());
        return this.code;
    };
    Method.prototype.getOp = function (pc, codeBuffer, thread) {
        if (this.numBBEntries <= 0) {
            if (!this.failedCompile[pc]) {
                var cachedCompiledFunction = this.compiledFunctions[pc];
                if (!cachedCompiledFunction) {
                    var compiledFunction = this.jitCompileFrom(pc, thread);
                    if (compiledFunction) {
                        return compiledFunction;
                    } else {
                        this.failedCompile[pc] = true;
                    }
                } else {
                    return cachedCompiledFunction;
                }
            }
        }
        return codeBuffer[pc];
    };
    Method.prototype.makeInvokeStaticJitInfo = function (code, pc) {
        var index = code.readUInt16BE(pc + 1);
        var methodReference = this.cls.constantPool.get(index);
        var paramSize = methodReference.paramWordSize;
        var method = methodReference.jsConstructor[methodReference.fullSignature];
        return {
            hasBranch: true,
            pops: -paramSize,
            pushes: 0,
            emit: function (pops, pushes, suffix, onSuccess) {
                var argInitialiser = paramSize > pops.length ? 'f.opStack.sliceAndDropFromTop(' + (paramSize - pops.length) + ');' : '[' + pops.reduce(function (a, b) {
                    return b + ',' + a;
                }, '') + '];';
                var argMaker = 'var args' + suffix + '=' + argInitialiser;
                if (paramSize > pops.length && pops.length > 0) {
                    argMaker += 'args' + suffix + '.push(' + pops.slice().reverse().join(',') + ');';
                }
                return argMaker + ('\nvar methodReference' + suffix + '=f.method.cls.constantPool.get(' + index + ');\nf.pc=' + pc + ';\nmethodReference' + suffix + '.jsConstructor[methodReference' + suffix + '.fullSignature](t,args' + suffix + ');\nf.returnToThreadLoop=true;\n' + onSuccess);
            }
        };
    };
    Method.prototype.makeInvokeVirtualJitInfo = function (code, pc) {
        var index = code.readUInt16BE(pc + 1);
        var methodReference = this.cls.constantPool.get(index);
        var paramSize = methodReference.paramWordSize;
        return {
            hasBranch: true,
            pops: -(paramSize + 1),
            pushes: 0,
            emit: function (pops, pushes, suffix, onSuccess, code, pc, onErrorPushes) {
                var onError = makeOnError(onErrorPushes);
                var argInitialiser = paramSize > pops.length ? 'f.opStack.sliceAndDropFromTop(' + (paramSize - pops.length) + ');' : '[' + pops.slice(0, paramSize).reduce(function (a, b) {
                    return b + ',' + a;
                }, '') + '];';
                var argMaker = 'var args' + suffix + '=' + argInitialiser;
                if (paramSize > pops.length && pops.length > 0) {
                    argMaker += 'args' + suffix + '.push(' + pops.slice().reverse().join(',') + ');';
                }
                return argMaker + ('var obj' + suffix + '=' + (paramSize + 1 === pops.length ? pops[paramSize] : 'f.opStack.pop()') + ';f.pc=' + pc + ';\nif(!u.isNull(t,f,obj' + suffix + ')){obj' + suffix + '[\'' + methodReference.signature + '\'](t,args' + suffix + ');f.returnToThreadLoop=true;' + onSuccess + '}else{' + onError + '}');
            }
        };
    };
    Method.prototype.makeInvokeNonVirtualJitInfo = function (code, pc) {
        var index = code.readUInt16BE(pc + 1);
        var methodReference = this.cls.constantPool.get(index);
        var paramSize = methodReference.paramWordSize;
        return {
            hasBranch: true,
            pops: -(paramSize + 1),
            pushes: 0,
            emit: function (pops, pushes, suffix, onSuccess, code, pc, onErrorPushes) {
                var onError = makeOnError(onErrorPushes);
                var argInitialiser = paramSize > pops.length ? 'f.opStack.sliceAndDropFromTop(' + (paramSize - pops.length) + ');' : '[' + pops.slice(0, paramSize).reduce(function (a, b) {
                    return b + ',' + a;
                }, '') + '];';
                var argMaker = 'var args' + suffix + '=' + argInitialiser;
                if (paramSize > pops.length && pops.length > 0) {
                    argMaker += 'args' + suffix + '.push(' + pops.slice().reverse().join(',') + ');';
                }
                return argMaker + ('var obj' + suffix + '=' + (paramSize + 1 === pops.length ? pops[paramSize] : 'f.opStack.pop()') + ';f.pc=' + pc + ';\nif(!u.isNull(t,f,obj' + suffix + ')){obj' + suffix + '[\'' + methodReference.fullSignature + '\'](t, args' + suffix + ');f.returnToThreadLoop=true;' + onSuccess + '}else{' + onError + '}');
            }
        };
    };
    Method.prototype.jitCompileFrom = function (startPC, thread) {
        if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
            console.log('Planning to JIT: ' + this.fullSignature + ' from ' + startPC);
        }
        var code = this.getCodeAttribute().getCode();
        var trace = null;
        var _this = this;
        var done = false;
        function closeCurrentTrace() {
            if (trace !== null) {
                var compiledFunction = trace.close(thread);
                if (compiledFunction) {
                    _this.compiledFunctions[trace.startPC] = compiledFunction;
                    if (!RELEASE && thread.getJVM().shouldDumpCompiledCode()) {
                        thread.getJVM().dumpCompiledMethod(_this.fullSignature, trace.startPC, compiledFunction.toString());
                    }
                }
                trace = null;
            }
            done = true;
        }
        for (var i = startPC; i < code.length && !done;) {
            var op = code[i];
            if (!RELEASE && thread.getJVM().shouldPrintJITCompilation()) {
                console.log(i + ': ' + threading_1.annotateOpcode(op, this, code, i));
            }
            var jitInfo = jit_1.opJitInfo[op];
            if (jitInfo) {
                if (trace === null) {
                    trace = new Trace(i, code, _this);
                }
                trace.addOp(i, jitInfo);
                if (jitInfo.hasBranch) {
                    this.failedCompile[i] = true;
                    closeCurrentTrace();
                }
            } else if (op === enums_1.OpCode.INVOKESTATIC_FAST && trace !== null) {
                var invokeJitInfo = this.makeInvokeStaticJitInfo(code, i);
                trace.addOp(i, invokeJitInfo);
                this.failedCompile[i] = true;
                closeCurrentTrace();
            } else if ((op === enums_1.OpCode.INVOKEVIRTUAL_FAST || op === enums_1.OpCode.INVOKEINTERFACE_FAST) && trace !== null) {
                var invokeJitInfo = this.makeInvokeVirtualJitInfo(code, i);
                trace.addOp(i, invokeJitInfo);
                this.failedCompile[i] = true;
                closeCurrentTrace();
            } else if (op === enums_1.OpCode.INVOKENONVIRTUAL_FAST && trace !== null) {
                var invokeJitInfo = this.makeInvokeNonVirtualJitInfo(code, i);
                trace.addOp(i, invokeJitInfo);
                this.failedCompile[i] = true;
                closeCurrentTrace();
            } else {
                if (!RELEASE) {
                    if (trace !== null) {
                        statTraceCloser[op]++;
                    }
                }
                this.failedCompile[i] = true;
                if (trace) {
                    trace.emitEndPC(i);
                }
                closeCurrentTrace();
            }
            i += opcodeSize[enums_1.OpcodeLayouts[op]];
        }
        return _this.compiledFunctions[startPC];
    };
    Method.prototype.getNativeFunction = function () {
        assert_1['default'](this.accessFlags.isNative() && typeof this.code === 'function');
        return this.code;
    };
    Method.prototype._resolveReferencedClasses = function (thread, cb) {
        var toResolve = this.parameterTypes.concat(this.returnType), code = this.code, exceptionAttribute = this.getAttribute('Exceptions');
        if (!this.accessFlags.isNative() && !this.accessFlags.isAbstract() && code.exceptionHandlers.length > 0) {
            toResolve.push('Ljava/lang/Throwable;');
            toResolve = toResolve.concat(code.exceptionHandlers.filter(function (handler) {
                return handler.catchType !== '<any>';
            }).map(function (handler) {
                return handler.catchType;
            }));
        }
        if (exceptionAttribute !== null) {
            toResolve = toResolve.concat(exceptionAttribute.exceptions);
        }
        this.cls.getLoader().resolveClasses(thread, toResolve, function (classes) {
            thread.getBsCl().resolveClasses(thread, [
                'Ljava/lang/reflect/Method;',
                'Ljava/lang/reflect/Constructor;'
            ], function (classes2) {
                if (classes === null || classes2 === null) {
                    cb(null);
                } else {
                    classes['Ljava/lang/reflect/Method;'] = classes2['Ljava/lang/reflect/Method;'];
                    classes['Ljava/lang/reflect/Constructor;'] = classes2['Ljava/lang/reflect/Constructor;'];
                    cb(classes);
                }
            });
        });
    };
    Method.prototype.reflector = function (thread, cb) {
        var _this = this;
        var bsCl = thread.getBsCl(), clazzArray = bsCl.getInitializedClass(thread, '[Ljava/lang/Class;').getConstructor(thread), jvm = thread.getJVM(), signatureAttr = this.getAttribute('Signature'), exceptionAttr = this.getAttribute('Exceptions');
        this._resolveReferencedClasses(thread, function (classes) {
            if (classes === null) {
                return cb(null);
            }
            var clazz = _this.cls.getClassObject(thread), name = jvm.internString(_this.name), parameterTypes = new clazzArray(thread, 0), returnType = classes[_this.returnType].getClassObject(thread), exceptionTypes = new clazzArray(thread, 0), modifiers = _this.accessFlags.getRawByte(), signature = signatureAttr !== null ? jvm.internString(signatureAttr.sig) : null;
            parameterTypes.array = _this.parameterTypes.map(function (ptype) {
                return classes[ptype].getClassObject(thread);
            });
            if (exceptionAttr !== null) {
                exceptionTypes.array = exceptionAttr.exceptions.map(function (eType) {
                    return classes[eType].getClassObject(thread);
                });
            }
            if (_this.name === '<init>') {
                var consCons = classes['Ljava/lang/reflect/Constructor;'].getConstructor(thread), consObj = new consCons(thread);
                consObj['java/lang/reflect/Constructor/clazz'] = clazz;
                consObj['java/lang/reflect/Constructor/parameterTypes'] = parameterTypes;
                consObj['java/lang/reflect/Constructor/exceptionTypes'] = exceptionTypes;
                consObj['java/lang/reflect/Constructor/modifiers'] = modifiers;
                consObj['java/lang/reflect/Constructor/slot'] = _this.slot;
                consObj['java/lang/reflect/Constructor/signature'] = signature;
                consObj['java/lang/reflect/Constructor/annotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleAnnotations');
                consObj['java/lang/reflect/Constructor/parameterAnnotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleParameterAnnotations');
                cb(consObj);
            } else {
                var methodCons = classes['Ljava/lang/reflect/Method;'].getConstructor(thread), methodObj = new methodCons(thread);
                methodObj['java/lang/reflect/Method/clazz'] = clazz;
                methodObj['java/lang/reflect/Method/name'] = name;
                methodObj['java/lang/reflect/Method/parameterTypes'] = parameterTypes;
                methodObj['java/lang/reflect/Method/returnType'] = returnType;
                methodObj['java/lang/reflect/Method/exceptionTypes'] = exceptionTypes;
                methodObj['java/lang/reflect/Method/modifiers'] = modifiers;
                methodObj['java/lang/reflect/Method/slot'] = _this.slot;
                methodObj['java/lang/reflect/Method/signature'] = signature;
                methodObj['java/lang/reflect/Method/annotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleAnnotations');
                methodObj['java/lang/reflect/Method/annotationDefault'] = _this.getAnnotationType(thread, 'AnnotationDefault');
                methodObj['java/lang/reflect/Method/parameterAnnotations'] = _this.getAnnotationType(thread, 'RuntimeVisibleParameterAnnotations');
                cb(methodObj);
            }
        });
    };
    Method.prototype.convertArgs = function (thread, params) {
        if (this.isSignaturePolymorphic()) {
            params.unshift(thread);
            return params;
        }
        var convertedArgs = [thread], argIdx = 0, i;
        if (!this.accessFlags.isStatic()) {
            convertedArgs.push(params[0]);
            argIdx = 1;
        }
        for (i = 0; i < this.parameterTypes.length; i++) {
            var p = this.parameterTypes[i];
            convertedArgs.push(params[argIdx]);
            argIdx += p === 'J' || p === 'D' ? 2 : 1;
        }
        return convertedArgs;
    };
    Method.prototype.methodLock = function (thread, frame) {
        if (this.accessFlags.isStatic()) {
            return this.cls.getClassObject(thread).getMonitor();
        } else {
            return frame.locals[0].getMonitor();
        }
    };
    Method.prototype.isSignaturePolymorphic = function () {
        return this.cls.getInternalName() === 'Ljava/lang/invoke/MethodHandle;' && this.accessFlags.isNative() && this.accessFlags.isVarArgs() && this.rawDescriptor === '([Ljava/lang/Object;)Ljava/lang/Object;';
    };
    Method.prototype.getVMTargetBridgeMethod = function (thread, refKind) {
        var outStream = new StringOutputStream_1['default'](), virtualDispatch = !(refKind === enums_1.MethodHandleReferenceKind.INVOKESTATIC || refKind === enums_1.MethodHandleReferenceKind.INVOKESPECIAL);
        if (this.accessFlags.isStatic()) {
            assert_1['default'](!virtualDispatch, 'Can\'t have static virtual dispatch.');
            outStream.write('var jsCons = cls.getConstructor(thread);\n');
        }
        outStream.write('function bridgeMethod(thread, descriptor, args, cb) {\n');
        if (!this.accessFlags.isStatic()) {
            outStream.write('  var obj = args.shift();\n');
            outStream.write('  if (obj === null) { return thread.throwNewException(\'Ljava/lang/NullPointerException;\', \'\'); }\n');
            outStream.write('  obj["' + util_1.reescapeJVMName(virtualDispatch ? this.signature : this.fullSignature) + '"](thread, ');
        } else {
            outStream.write('  jsCons["' + util_1.reescapeJVMName(this.fullSignature) + '"](thread, ');
        }
        outStream.write('args');
        outStream.write(', cb);\n  }\n  return bridgeMethod;');
        var evalText = outStream.flush();
        if (!RELEASE && thread !== null && thread.getJVM().shouldDumpCompiledCode()) {
            thread.getJVM().dumpBridgeMethod(this.fullSignature, evalText);
        }
        return new Function('thread', 'cls', 'util', evalText)(thread, this.cls, util);
    };
    Method.prototype.outputJavaScriptFunction = function (jsConsName, outStream, nonVirtualOnly) {
        if (nonVirtualOnly === void 0) {
            nonVirtualOnly = false;
        }
        var i;
        if (this.accessFlags.isStatic()) {
            outStream.write(jsConsName + '["' + util_1.reescapeJVMName(this.fullSignature) + '"] = ' + jsConsName + '["' + util_1.reescapeJVMName(this.signature) + '"] = ');
        } else {
            if (!nonVirtualOnly) {
                outStream.write(jsConsName + '.prototype["' + util_1.reescapeJVMName(this.signature) + '"] = ');
            }
            outStream.write(jsConsName + '.prototype["' + util_1.reescapeJVMName(this.fullSignature) + '"] = ');
        }
        outStream.write('(function(method) {\n  return function(thread, args, cb) {\n    if (typeof cb === \'function\') {\n      thread.stack.push(new InternalStackFrame(cb));\n    }\n    thread.stack.push(new ' + (this.accessFlags.isNative() ? 'NativeStackFrame' : 'BytecodeStackFrame') + '(method, ');
        if (!this.accessFlags.isStatic()) {
            outStream.write('[this');
            for (i = 0; i < this.parameterWords; i++) {
                outStream.write(', args[' + i + ']');
            }
            outStream.write(']');
        } else {
            if (this.parameterWords > 0) {
                outStream.write('args');
            } else {
                outStream.write('[]');
            }
        }
        outStream.write('));\n    thread.setStatus(' + enums_1.ThreadStatus.RUNNABLE + ');\n  };\n})(cls.getSpecificMethod("' + util_1.reescapeJVMName(this.cls.getInternalName()) + '", "' + util_1.reescapeJVMName(this.signature) + '"));\n');
    };
    return Method;
}(AbstractMethodField);
exports.Method = Method;
function makeOnError(onErrorPushes) {
    return onErrorPushes.length > 0 ? 'f.opStack.pushAll(' + onErrorPushes.join(',') + ');' : '';
}
var statTraceCloser = new Array(256);
if (!RELEASE) {
    for (var i = 0; i < 256; i++) {
        statTraceCloser[i] = 0;
    }
}
function dumpStats() {
    var range = new Array(256);
    for (var i = 0; i < 256; i++) {
        range[i] = i;
    }
    range.sort(function (x, y) {
        return statTraceCloser[y] - statTraceCloser[x];
    });
    var top = range.slice(0, 24);
    console.log('Opcodes that closed a trace (number of times encountered):');
    for (var i = 0; i < top.length; i++) {
        var op = top[i];
        if (statTraceCloser[op] > 0) {
            console.log(enums_1.OpCode[op], statTraceCloser[op]);
        }
    }
}
exports.dumpStats = dumpStats;
//# sourceMappingURL=methods.js.map