'use strict';
var gLong_1 = require('./gLong');
var util_1 = require('./util');
var enums_1 = require('./enums');
var assert_1 = require('./assert');
function isNull(thread, frame, obj) {
    if (obj == null) {
        throwException(thread, frame, 'Ljava/lang/NullPointerException;', '');
        return true;
    }
    return false;
}
exports.isNull = isNull;
function pop2(opStack) {
    opStack.pop();
    return opStack.pop();
}
exports.pop2 = pop2;
function resolveCPItem(thread, frame, cpItem) {
    thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
    cpItem.resolve(thread, frame.getLoader(), frame.method.cls, function (status) {
        if (status) {
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
        }
    }, false);
    frame.returnToThreadLoop = true;
}
exports.resolveCPItem = resolveCPItem;
function initializeClassFromClass(thread, frame, cls) {
    thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
    cls.initialize(thread, function (cdata) {
        if (cdata != null) {
            thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
        }
    }, false);
    frame.returnToThreadLoop = true;
}
exports.initializeClassFromClass = initializeClassFromClass;
function initializeClass(thread, frame, clsRef) {
    thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
    function initialize(cls) {
        cls.initialize(thread, function (cdata) {
            if (cdata != null) {
                thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            }
        });
    }
    if (!clsRef.isResolved()) {
        clsRef.resolve(thread, frame.getLoader(), frame.method.cls, function (status) {
            if (status) {
                initialize(clsRef.cls);
            }
        }, false);
    } else {
        initialize(clsRef.cls);
    }
    frame.returnToThreadLoop = true;
}
exports.initializeClass = initializeClass;
function throwException(thread, frame, clsName, msg) {
    thread.throwNewException(clsName, msg);
    frame.returnToThreadLoop = true;
}
exports.throwException = throwException;
exports.ArrayTypes = {
    4: 'Z',
    5: 'C',
    6: 'F',
    7: 'D',
    8: 'B',
    9: 'S',
    10: 'I',
    11: 'J'
};
var Opcodes = function () {
    function Opcodes() {
    }
    Opcodes._aload_32 = function (thread, frame) {
        var opStack = frame.opStack, idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + ' not in length ' + len + ' array of type ' + obj.getClass().getInternalName());
            } else {
                opStack.push(obj.array[idx]);
                frame.pc++;
            }
        }
    };
    Opcodes._aload_64 = function (thread, frame) {
        var opStack = frame.opStack, idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + ' not in length ' + len + ' array of type ' + obj.getClass().getInternalName());
            } else {
                opStack.push(obj.array[idx]);
                opStack.push(null);
                frame.pc++;
            }
        }
    };
    Opcodes._astore_32 = function (thread, frame) {
        var opStack = frame.opStack, value = opStack.pop(), idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + ' not in length ' + len + ' array of type ' + obj.getClass().getInternalName());
            } else {
                obj.array[idx] = value;
                frame.pc++;
            }
        }
    };
    Opcodes._astore_64 = function (thread, frame) {
        var opStack = frame.opStack, value = opStack.pop2(), idx = opStack.pop(), obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            var len = obj.array.length;
            if (idx < 0 || idx >= len) {
                throwException(thread, frame, 'Ljava/lang/ArrayIndexOutOfBoundsException;', idx + ' not in length ' + len + ' array of type ' + obj.getClass().getInternalName());
            } else {
                obj.array[idx] = value;
                frame.pc++;
            }
        }
    };
    Opcodes.aconst_null = function (thread, frame) {
        frame.opStack.push(null);
        frame.pc++;
    };
    Opcodes._const_0_32 = function (thread, frame) {
        frame.opStack.push(0);
        frame.pc++;
    };
    Opcodes._const_1_32 = function (thread, frame) {
        frame.opStack.push(1);
        frame.pc++;
    };
    Opcodes._const_2_32 = function (thread, frame) {
        frame.opStack.push(2);
        frame.pc++;
    };
    Opcodes.iconst_m1 = function (thread, frame) {
        frame.opStack.push(-1);
        frame.pc++;
    };
    Opcodes.iconst_3 = function (thread, frame) {
        frame.opStack.push(3);
        frame.pc++;
    };
    Opcodes.iconst_4 = function (thread, frame) {
        frame.opStack.push(4);
        frame.pc++;
    };
    Opcodes.iconst_5 = function (thread, frame) {
        frame.opStack.push(5);
        frame.pc++;
    };
    Opcodes.lconst_0 = function (thread, frame) {
        frame.opStack.pushWithNull(gLong_1['default'].ZERO);
        frame.pc++;
    };
    Opcodes.lconst_1 = function (thread, frame) {
        frame.opStack.pushWithNull(gLong_1['default'].ONE);
        frame.pc++;
    };
    Opcodes.dconst_0 = function (thread, frame) {
        frame.opStack.pushWithNull(0);
        frame.pc++;
    };
    Opcodes.dconst_1 = function (thread, frame) {
        frame.opStack.pushWithNull(1);
        frame.pc++;
    };
    Opcodes._load_32 = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(frame.locals[code[pc + 1]]);
        frame.pc += 2;
    };
    Opcodes._load_0_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[0]);
        frame.pc++;
    };
    Opcodes._load_1_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[1]);
        frame.pc++;
    };
    Opcodes._load_2_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[2]);
        frame.pc++;
    };
    Opcodes._load_3_32 = function (thread, frame) {
        frame.opStack.push(frame.locals[3]);
        frame.pc++;
    };
    Opcodes._load_64 = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.pushWithNull(frame.locals[code[pc + 1]]);
        frame.pc += 2;
    };
    Opcodes._load_0_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[0]);
        frame.pc++;
    };
    Opcodes._load_1_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[1]);
        frame.pc++;
    };
    Opcodes._load_2_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[2]);
        frame.pc++;
    };
    Opcodes._load_3_64 = function (thread, frame) {
        frame.opStack.pushWithNull(frame.locals[3]);
        frame.pc++;
    };
    Opcodes._store_32 = function (thread, frame, code) {
        var pc = frame.pc;
        frame.locals[code[pc + 1]] = frame.opStack.pop();
        frame.pc += 2;
    };
    Opcodes._store_0_32 = function (thread, frame) {
        frame.locals[0] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_1_32 = function (thread, frame) {
        frame.locals[1] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_2_32 = function (thread, frame) {
        frame.locals[2] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_3_32 = function (thread, frame) {
        frame.locals[3] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_64 = function (thread, frame, code) {
        var pc = frame.pc;
        var offset = code[pc + 1];
        frame.locals[offset + 1] = frame.opStack.pop();
        frame.locals[offset] = frame.opStack.pop();
        frame.pc += 2;
    };
    Opcodes._store_0_64 = function (thread, frame) {
        frame.locals[1] = frame.opStack.pop();
        frame.locals[0] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_1_64 = function (thread, frame) {
        frame.locals[2] = frame.opStack.pop();
        frame.locals[1] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_2_64 = function (thread, frame) {
        frame.locals[3] = frame.opStack.pop();
        frame.locals[2] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes._store_3_64 = function (thread, frame) {
        frame.locals[4] = frame.opStack.pop();
        frame.locals[3] = frame.opStack.pop();
        frame.pc++;
    };
    Opcodes.sipush = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(code.readInt16BE(pc + 1));
        frame.pc += 3;
    };
    Opcodes.bipush = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(code.readInt8(pc + 1));
        frame.pc += 2;
    };
    Opcodes.pop = function (thread, frame) {
        frame.opStack.dropFromTop(1);
        frame.pc++;
    };
    Opcodes.pop2 = function (thread, frame) {
        frame.opStack.dropFromTop(2);
        frame.pc++;
    };
    Opcodes.dup = function (thread, frame) {
        frame.opStack.dup();
        frame.pc++;
    };
    Opcodes.dup_x1 = function (thread, frame) {
        frame.opStack.dup_x1();
        frame.pc++;
    };
    Opcodes.dup_x2 = function (thread, frame) {
        frame.opStack.dup_x2();
        frame.pc++;
    };
    Opcodes.dup2 = function (thread, frame) {
        frame.opStack.dup2();
        frame.pc++;
    };
    Opcodes.dup2_x1 = function (thread, frame) {
        frame.opStack.dup2_x1();
        frame.pc++;
    };
    Opcodes.dup2_x2 = function (thread, frame) {
        var opStack = frame.opStack, v1 = opStack.pop(), v2 = opStack.pop(), v3 = opStack.pop(), v4 = opStack.pop();
        opStack.push6(v2, v1, v4, v3, v2, v1);
        frame.pc++;
    };
    Opcodes.swap = function (thread, frame) {
        frame.opStack.swap();
        frame.pc++;
    };
    Opcodes.iadd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() + opStack.pop() | 0);
        frame.pc++;
    };
    Opcodes.ladd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().add(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.fadd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.wrapFloat(opStack.pop() + opStack.pop()));
        frame.pc++;
    };
    Opcodes.dadd = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2() + opStack.pop2());
        frame.pc++;
    };
    Opcodes.isub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(-opStack.pop() + opStack.pop() | 0);
        frame.pc++;
    };
    Opcodes.fsub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.wrapFloat(-opStack.pop() + opStack.pop()));
        frame.pc++;
    };
    Opcodes.dsub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(-opStack.pop2() + opStack.pop2());
        frame.pc++;
    };
    Opcodes.lsub = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().negate().add(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.imul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(Math.imul(opStack.pop(), opStack.pop()));
        frame.pc++;
    };
    Opcodes.lmul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().multiply(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.fmul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.wrapFloat(opStack.pop() * opStack.pop()));
        frame.pc++;
    };
    Opcodes.dmul = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2() * opStack.pop2());
        frame.pc++;
    };
    Opcodes.idiv = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop(), a = opStack.pop();
        if (b === 0) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        } else {
            if (a === enums_1.Constants.INT_MIN && b === -1) {
                opStack.push(a);
            } else {
                opStack.push(a / b | 0);
            }
            frame.pc++;
        }
    };
    Opcodes.ldiv = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop2(), a = opStack.pop2();
        if (b.isZero()) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        } else {
            opStack.pushWithNull(a.div(b));
            frame.pc++;
        }
    };
    Opcodes.fdiv = function (thread, frame) {
        var opStack = frame.opStack, a = opStack.pop();
        opStack.push(util_1.wrapFloat(opStack.pop() / a));
        frame.pc++;
    };
    Opcodes.ddiv = function (thread, frame) {
        var opStack = frame.opStack, v = opStack.pop2();
        opStack.pushWithNull(opStack.pop2() / v);
        frame.pc++;
    };
    Opcodes.irem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop(), a = opStack.pop();
        if (b === 0) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        } else {
            opStack.push(a % b);
            frame.pc++;
        }
    };
    Opcodes.lrem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop2(), a = opStack.pop2();
        if (b.isZero()) {
            throwException(thread, frame, 'Ljava/lang/ArithmeticException;', '/ by zero');
        } else {
            opStack.pushWithNull(a.modulo(b));
            frame.pc++;
        }
    };
    Opcodes.frem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop();
        opStack.push(opStack.pop() % b);
        frame.pc++;
    };
    Opcodes.drem = function (thread, frame) {
        var opStack = frame.opStack, b = opStack.pop2();
        opStack.pushWithNull(opStack.pop2() % b);
        frame.pc++;
    };
    Opcodes.ineg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(-opStack.pop() | 0);
        frame.pc++;
    };
    Opcodes.lneg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().negate());
        frame.pc++;
    };
    Opcodes.fneg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(-opStack.pop());
        frame.pc++;
    };
    Opcodes.dneg = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(-opStack.pop2());
        frame.pc++;
    };
    Opcodes.ishl = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.push(opStack.pop() << s);
        frame.pc++;
    };
    Opcodes.lshl = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.pushWithNull(opStack.pop2().shiftLeft(gLong_1['default'].fromInt(s)));
        frame.pc++;
    };
    Opcodes.ishr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.push(opStack.pop() >> s);
        frame.pc++;
    };
    Opcodes.lshr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.pushWithNull(opStack.pop2().shiftRight(gLong_1['default'].fromInt(s)));
        frame.pc++;
    };
    Opcodes.iushr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.push(opStack.pop() >>> s | 0);
        frame.pc++;
    };
    Opcodes.lushr = function (thread, frame) {
        var opStack = frame.opStack, s = opStack.pop();
        opStack.pushWithNull(opStack.pop2().shiftRightUnsigned(gLong_1['default'].fromInt(s)));
        frame.pc++;
    };
    Opcodes.iand = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() & opStack.pop());
        frame.pc++;
    };
    Opcodes.land = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().and(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.ior = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() | opStack.pop());
        frame.pc++;
    };
    Opcodes.lor = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().or(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.ixor = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() ^ opStack.pop());
        frame.pc++;
    };
    Opcodes.lxor = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().xor(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.iinc = function (thread, frame, code) {
        var pc = frame.pc;
        var idx = code[pc + 1], val = code.readInt8(pc + 2);
        frame.locals[idx] = frame.locals[idx] + val | 0;
        frame.pc += 3;
    };
    Opcodes.i2l = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(gLong_1['default'].fromInt(opStack.pop()));
        frame.pc++;
    };
    Opcodes.i2f = function (thread, frame) {
        frame.pc++;
    };
    Opcodes.i2d = function (thread, frame) {
        frame.opStack.push(null);
        frame.pc++;
    };
    Opcodes.l2i = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop2().toInt());
        frame.pc++;
    };
    Opcodes.l2f = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop2().toNumber());
        frame.pc++;
    };
    Opcodes.l2d = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(opStack.pop2().toNumber());
        frame.pc++;
    };
    Opcodes.f2i = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.float2int(opStack.pop()));
        frame.pc++;
    };
    Opcodes.f2l = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pushWithNull(gLong_1['default'].fromNumber(opStack.pop()));
        frame.pc++;
    };
    Opcodes.f2d = function (thread, frame) {
        frame.opStack.push(null);
        frame.pc++;
    };
    Opcodes.d2i = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(util_1.float2int(opStack.pop2()));
        frame.pc++;
    };
    Opcodes.d2l = function (thread, frame) {
        var opStack = frame.opStack, d_val = opStack.pop2();
        if (d_val === Number.POSITIVE_INFINITY) {
            opStack.pushWithNull(gLong_1['default'].MAX_VALUE);
        } else if (d_val === Number.NEGATIVE_INFINITY) {
            opStack.pushWithNull(gLong_1['default'].MIN_VALUE);
        } else {
            opStack.pushWithNull(gLong_1['default'].fromNumber(d_val));
        }
        frame.pc++;
    };
    Opcodes.d2f = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.pop();
        opStack.push(util_1.wrapFloat(opStack.pop()));
        frame.pc++;
    };
    Opcodes.i2b = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() << 24 >> 24);
        frame.pc++;
    };
    Opcodes.i2c = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() & 65535);
        frame.pc++;
    };
    Opcodes.i2s = function (thread, frame) {
        var opStack = frame.opStack;
        opStack.push(opStack.pop() << 16 >> 16);
        frame.pc++;
    };
    Opcodes.lcmp = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop2();
        opStack.push(opStack.pop2().compare(v2));
        frame.pc++;
    };
    Opcodes.fcmpl = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop(), v1 = opStack.pop();
        if (v1 === v2) {
            opStack.push(0);
        } else if (v1 > v2) {
            opStack.push(1);
        } else {
            opStack.push(-1);
        }
        frame.pc++;
    };
    Opcodes.fcmpg = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop(), v1 = opStack.pop();
        if (v1 === v2) {
            opStack.push(0);
        } else if (v1 < v2) {
            opStack.push(-1);
        } else {
            opStack.push(1);
        }
        frame.pc++;
    };
    Opcodes.dcmpl = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop2(), v1 = opStack.pop2();
        if (v1 === v2) {
            opStack.push(0);
        } else if (v1 > v2) {
            opStack.push(1);
        } else {
            opStack.push(-1);
        }
        frame.pc++;
    };
    Opcodes.dcmpg = function (thread, frame) {
        var opStack = frame.opStack, v2 = opStack.pop2(), v1 = opStack.pop2();
        if (v1 === v2) {
            opStack.push(0);
        } else if (v1 < v2) {
            opStack.push(-1);
        } else {
            opStack.push(1);
        }
        frame.pc++;
    };
    Opcodes.ifeq = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() === 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.ifne = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() !== 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.iflt = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() < 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.ifge = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() >= 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.ifgt = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() > 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.ifle = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() <= 0) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpeq = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 === v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpne = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 !== v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmplt = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 < v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpge = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 >= v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmpgt = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 > v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_icmple = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 <= v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_acmpeq = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 === v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.if_acmpne = function (thread, frame, code) {
        var pc = frame.pc;
        var v2 = frame.opStack.pop();
        var v1 = frame.opStack.pop();
        if (v1 !== v2) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.goto = function (thread, frame, code) {
        var pc = frame.pc;
        var offset = code.readInt16BE(pc + 1);
        frame.pc += offset;
        if (offset < 0) {
            frame.method.incrBBEntries();
        }
    };
    Opcodes.jsr = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(pc + 3);
        var offset = code.readInt16BE(pc + 1);
        frame.pc += offset;
        if (offset < 0) {
            frame.method.incrBBEntries();
        }
    };
    Opcodes.ret = function (thread, frame, code) {
        var pc = frame.pc;
        frame.pc = frame.locals[code[pc + 1]];
    };
    Opcodes.tableswitch = function (thread, frame, code) {
        var pc = frame.pc;
        pc += (4 - (pc + 1) % 4) % 4 + 1;
        var defaultOffset = code.readInt32BE(pc), low = code.readInt32BE(pc + 4), high = code.readInt32BE(pc + 8), offset = frame.opStack.pop();
        if (offset >= low && offset <= high) {
            frame.pc += code.readInt32BE(pc + 12 + (offset - low) * 4);
        } else {
            frame.pc += defaultOffset;
        }
    };
    Opcodes.lookupswitch = function (thread, frame, code) {
        var pc = frame.pc;
        pc += (4 - (pc + 1) % 4) % 4 + 1;
        var defaultOffset = code.readInt32BE(pc), nPairs = code.readInt32BE(pc + 4), i, v = frame.opStack.pop();
        pc += 8;
        for (i = 0; i < nPairs; i++) {
            if (code.readInt32BE(pc) === v) {
                var offset = code.readInt32BE(pc + 4);
                frame.pc += offset;
                if (offset < 0) {
                    frame.method.incrBBEntries();
                }
                return;
            }
            pc += 8;
        }
        frame.pc += defaultOffset;
    };
    Opcodes.return = function (thread, frame) {
        frame.returnToThreadLoop = true;
        if (frame.method.accessFlags.isSynchronized()) {
            if (!frame.method.methodLock(thread, frame).exit(thread)) {
                return;
            }
        }
        thread.asyncReturn();
    };
    Opcodes._return_32 = function (thread, frame) {
        frame.returnToThreadLoop = true;
        if (frame.method.accessFlags.isSynchronized()) {
            if (!frame.method.methodLock(thread, frame).exit(thread)) {
                return;
            }
        }
        thread.asyncReturn(frame.opStack.bottom());
    };
    Opcodes._return_64 = function (thread, frame) {
        frame.returnToThreadLoop = true;
        if (frame.method.accessFlags.isSynchronized()) {
            if (!frame.method.methodLock(thread, frame).exit(thread)) {
                return;
            }
        }
        thread.asyncReturn(frame.opStack.bottom(), null);
    };
    Opcodes.getstatic = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        assert_1['default'](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        if (fieldInfo.isResolved()) {
            var fieldOwnerCls = fieldInfo.field.cls;
            if (fieldOwnerCls.isInitialized(thread)) {
                if (fieldInfo.nameAndTypeInfo.descriptor === 'J' || fieldInfo.nameAndTypeInfo.descriptor === 'D') {
                    code[pc] = enums_1.OpCode.GETSTATIC_FAST64;
                } else {
                    code[pc] = enums_1.OpCode.GETSTATIC_FAST32;
                }
                fieldInfo.fieldOwnerConstructor = fieldOwnerCls.getConstructor(thread);
            } else {
                initializeClassFromClass(thread, frame, fieldOwnerCls);
            }
        } else {
            resolveCPItem(thread, frame, fieldInfo);
        }
    };
    Opcodes.getstatic_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        frame.opStack.push(fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName]);
        frame.pc += 3;
    };
    Opcodes.getstatic_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        frame.opStack.pushWithNull(fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName]);
        frame.pc += 3;
    };
    Opcodes.putstatic = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        assert_1['default'](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        if (fieldInfo.isResolved()) {
            var fieldOwnerCls = fieldInfo.field.cls;
            if (fieldOwnerCls.isInitialized(thread)) {
                if (fieldInfo.nameAndTypeInfo.descriptor === 'J' || fieldInfo.nameAndTypeInfo.descriptor === 'D') {
                    code[pc] = enums_1.OpCode.PUTSTATIC_FAST64;
                } else {
                    code[pc] = enums_1.OpCode.PUTSTATIC_FAST32;
                }
                fieldInfo.fieldOwnerConstructor = fieldOwnerCls.getConstructor(thread);
            } else {
                initializeClassFromClass(thread, frame, fieldOwnerCls);
            }
        } else {
            resolveCPItem(thread, frame, fieldInfo);
        }
    };
    Opcodes.putstatic_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName] = frame.opStack.pop();
        frame.pc += 3;
    };
    Opcodes.putstatic_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        fieldInfo.fieldOwnerConstructor[fieldInfo.fullFieldName] = frame.opStack.pop2();
        frame.pc += 3;
    };
    Opcodes.getfield = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), loader = frame.getLoader(), obj = frame.opStack.top();
        assert_1['default'](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        if (!isNull(thread, frame, obj)) {
            if (fieldInfo.isResolved()) {
                var field = fieldInfo.field;
                if (field.rawDescriptor == 'J' || field.rawDescriptor == 'D') {
                    code[pc] = enums_1.OpCode.GETFIELD_FAST64;
                } else {
                    code[pc] = enums_1.OpCode.GETFIELD_FAST32;
                }
            } else {
                resolveCPItem(thread, frame, fieldInfo);
            }
        }
    };
    Opcodes.getfield_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            opStack.push(obj[fieldInfo.fullFieldName]);
            frame.pc += 3;
        }
    };
    Opcodes.getfield_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            opStack.pushWithNull(obj[fieldInfo.fullFieldName]);
            frame.pc += 3;
        }
    };
    Opcodes.putfield = function (thread, frame, code) {
        var pc = frame.pc;
        var fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), loader = frame.getLoader(), isLong = fieldInfo.nameAndTypeInfo.descriptor == 'J' || fieldInfo.nameAndTypeInfo.descriptor == 'D', obj = frame.opStack.fromTop(isLong ? 2 : 1);
        assert_1['default'](fieldInfo.getType() === enums_1.ConstantPoolItemType.FIELDREF);
        if (!isNull(thread, frame, obj)) {
            if (fieldInfo.isResolved()) {
                var field = fieldInfo.field;
                if (isLong) {
                    code[pc] = enums_1.OpCode.PUTFIELD_FAST64;
                } else {
                    code[pc] = enums_1.OpCode.PUTFIELD_FAST32;
                }
                fieldInfo.fullFieldName = util_1.descriptor2typestr(field.cls.getInternalName()) + '/' + fieldInfo.nameAndTypeInfo.name;
            } else {
                resolveCPItem(thread, frame, fieldInfo);
            }
        }
    };
    Opcodes.putfield_fast32 = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, val = opStack.pop(), obj = opStack.pop(), fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (!isNull(thread, frame, obj)) {
            obj[fieldInfo.fullFieldName] = val;
            frame.pc += 3;
        }
    };
    Opcodes.putfield_fast64 = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, val = opStack.pop2(), obj = opStack.pop(), fieldInfo = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (!isNull(thread, frame, obj)) {
            obj[fieldInfo.fullFieldName] = val;
            frame.pc += 3;
        }
    };
    Opcodes.invokevirtual = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            var m = methodReference.method;
            if (m.isSignaturePolymorphic()) {
                switch (m.name) {
                case 'invokeBasic':
                    code[pc] = enums_1.OpCode.INVOKEBASIC;
                    break;
                case 'invoke':
                case 'invokeExact':
                    code[pc] = enums_1.OpCode.INVOKEHANDLE;
                    break;
                default:
                    throwException(thread, frame, 'Ljava/lang/AbstractMethodError;', 'Invalid signature polymorphic method: ' + m.cls.getExternalName() + '.' + m.name);
                    break;
                }
            } else {
                code[pc] = enums_1.OpCode.INVOKEVIRTUAL_FAST;
            }
        } else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokeinterface = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            if (methodReference.method.cls.isInitialized(thread)) {
                code[pc] = enums_1.OpCode.INVOKEINTERFACE_FAST;
            } else {
                initializeClass(thread, frame, methodReference.classInfo);
            }
        } else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokedynamic = function (thread, frame, code) {
        var pc = frame.pc;
        var callSiteSpecifier = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
        callSiteSpecifier.constructCallSiteObject(thread, frame.getLoader(), frame.method.cls, pc, function (status) {
            if (status) {
                assert_1['default'](typeof callSiteSpecifier.getCallSiteObject(pc)[0].vmtarget === 'function', 'MethodName should be resolved...');
                code[pc] = enums_1.OpCode.INVOKEDYNAMIC_FAST;
                thread.setStatus(enums_1.ThreadStatus.RUNNABLE);
            }
        });
        frame.returnToThreadLoop = true;
    };
    Opcodes.invokespecial = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            code[pc] = enums_1.OpCode.INVOKENONVIRTUAL_FAST;
        } else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokestatic = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (methodReference.isResolved()) {
            var m = methodReference.method;
            if (m.cls.isInitialized(thread)) {
                var newOpcode = enums_1.OpCode.INVOKESTATIC_FAST;
                if (methodReference.method.isSignaturePolymorphic()) {
                    switch (methodReference.method.name) {
                    case 'linkToInterface':
                    case 'linkToVirtual':
                        newOpcode = enums_1.OpCode.LINKTOVIRTUAL;
                        break;
                    case 'linkToStatic':
                    case 'linkToSpecial':
                        newOpcode = enums_1.OpCode.LINKTOSPECIAL;
                        break;
                    default:
                        assert_1['default'](false, 'Should be impossible.');
                        break;
                    }
                }
                code[pc] = newOpcode;
            } else {
                initializeClassFromClass(thread, frame, m.cls);
            }
        } else {
            resolveCPItem(thread, frame, methodReference);
        }
    };
    Opcodes.invokenonvirtual_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, paramSize = methodReference.paramWordSize, obj = opStack.fromTop(paramSize);
        if (!isNull(thread, frame, obj)) {
            var args = opStack.sliceFromTop(paramSize);
            opStack.dropFromTop(paramSize + 1);
            assert_1['default'](typeof obj[methodReference.fullSignature] === 'function', 'Resolved method ' + methodReference.fullSignature + ' isn\'t defined?!', thread);
            obj[methodReference.fullSignature](thread, args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.invokestatic_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, paramSize = methodReference.paramWordSize, args = opStack.sliceAndDropFromTop(paramSize);
        assert_1['default'](methodReference.jsConstructor != null, 'jsConstructor is missing?!');
        assert_1['default'](typeof methodReference.jsConstructor[methodReference.fullSignature] === 'function', 'Resolved method isn\'t defined?!');
        methodReference.jsConstructor[methodReference.fullSignature](thread, args);
        frame.returnToThreadLoop = true;
    };
    Opcodes.invokevirtual_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), count = methodReference.paramWordSize, opStack = frame.opStack, obj = opStack.fromTop(count);
        if (!isNull(thread, frame, obj)) {
            assert_1['default'](typeof obj[methodReference.signature] === 'function', 'Resolved method ' + methodReference.signature + ' isn\'t defined?!');
            obj[methodReference.signature](thread, opStack.sliceFromTop(count));
            opStack.dropFromTop(count + 1);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.invokedynamic_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var callSiteSpecifier = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), cso = callSiteSpecifier.getCallSiteObject(pc), appendix = cso[1], fcn = cso[0].vmtarget, opStack = frame.opStack, paramSize = callSiteSpecifier.paramWordSize, args = opStack.sliceAndDropFromTop(paramSize);
        if (appendix !== null) {
            args.push(appendix);
        }
        fcn(thread, null, args);
        frame.returnToThreadLoop = true;
    };
    Opcodes.invokehandle = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, fcn = methodReference.memberName.vmtarget, paramSize = methodReference.paramWordSize + 1, appendix = methodReference.appendix, args = opStack.sliceFromTop(paramSize);
        if (appendix !== null) {
            args.push(appendix);
        }
        if (!isNull(thread, frame, args[0])) {
            opStack.dropFromTop(paramSize);
            fcn(thread, null, args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.invokebasic = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), paramSize = methodReference.getParamWordSize(), opStack = frame.opStack, obj = opStack.fromTop(paramSize), args = opStack.sliceFromTop(paramSize + 1), lmbdaForm, mn, m;
        if (!isNull(thread, frame, obj)) {
            opStack.dropFromTop(paramSize + 1);
            lmbdaForm = obj['java/lang/invoke/MethodHandle/form'];
            mn = lmbdaForm['java/lang/invoke/LambdaForm/vmentry'];
            assert_1['default'](mn.vmtarget !== null && mn.vmtarget !== undefined, 'vmtarget must be defined');
            mn.vmtarget(thread, methodReference.nameAndTypeInfo.descriptor, args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.linktospecial = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, paramSize = methodReference.paramWordSize, args = opStack.sliceFromTop(paramSize), memberName = args.pop(), desc = methodReference.nameAndTypeInfo.descriptor;
        if (!isNull(thread, frame, memberName)) {
            opStack.dropFromTop(paramSize);
            assert_1['default'](memberName.getClass().getInternalName() === 'Ljava/lang/invoke/MemberName;');
            memberName.vmtarget(thread, desc.replace('Ljava/lang/invoke/MemberName;)', ')'), args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.linktovirtual = function (thread, frame, code) {
        var pc = frame.pc;
        var methodReference = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), paramSize = methodReference.paramWordSize, opStack = frame.opStack, args = opStack.sliceFromTop(paramSize), memberName = args.pop(), desc = methodReference.nameAndTypeInfo.descriptor;
        if (!isNull(thread, frame, memberName)) {
            opStack.dropFromTop(paramSize);
            assert_1['default'](memberName.getClass().getInternalName() === 'Ljava/lang/invoke/MemberName;');
            memberName.vmtarget(thread, desc.replace('Ljava/lang/invoke/MemberName;)', ')'), args);
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.breakpoint = function (thread, frame) {
        throwException(thread, frame, 'Ljava/lang/Error;', 'breakpoint not implemented.');
    };
    Opcodes.new = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            var cls = classRef.cls;
            if (cls.isInitialized(thread)) {
                code[pc] = enums_1.OpCode.NEW_FAST;
            } else {
                initializeClassFromClass(thread, frame, cls);
            }
        } else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.new_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        frame.opStack.push(new classRef.clsConstructor(thread));
        frame.pc += 3;
    };
    Opcodes.newarray = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, type = '[' + exports.ArrayTypes[code[pc + 1]], cls = frame.getLoader().getInitializedClass(thread, type), length = opStack.pop();
        if (length >= 0) {
            opStack.push(new (cls.getConstructor(thread))(thread, length));
            frame.pc += 2;
        } else {
            throwException(thread, frame, 'Ljava/lang/NegativeArraySizeException;', 'Tried to init ' + type + ' array with length ' + length);
        }
    };
    Opcodes.anewarray = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            code[pc] = enums_1.OpCode.ANEWARRAY_FAST;
            classRef.arrayClass = frame.getLoader().getInitializedClass(thread, '[' + classRef.cls.getInternalName());
            classRef.arrayClassConstructor = classRef.arrayClass.getConstructor(thread);
        } else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.anewarray_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var opStack = frame.opStack, classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), length = opStack.pop();
        if (length >= 0) {
            opStack.push(new classRef.arrayClassConstructor(thread, length));
            frame.pc += 3;
        } else {
            throwException(thread, frame, 'Ljava/lang/NegativeArraySizeException;', 'Tried to init ' + classRef.arrayClass.getInternalName() + ' array with length ' + length);
        }
    };
    Opcodes.arraylength = function (thread, frame) {
        var opStack = frame.opStack, obj = opStack.pop();
        if (!isNull(thread, frame, obj)) {
            opStack.push(obj.array.length);
            frame.pc++;
        }
    };
    Opcodes.athrow = function (thread, frame) {
        thread.throwException(frame.opStack.pop());
        frame.returnToThreadLoop = true;
    };
    Opcodes.checkcast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            code[pc] = enums_1.OpCode.CHECKCAST_FAST;
        } else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.checkcast_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), cls = classRef.cls, opStack = frame.opStack, o = opStack.top();
        if (o != null && !o.getClass().isCastable(cls)) {
            var targetClass = cls.getExternalName();
            var candidateClass = o.getClass().getExternalName();
            throwException(thread, frame, 'Ljava/lang/ClassCastException;', candidateClass + ' cannot be cast to ' + targetClass);
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.instanceof = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            code[pc] = enums_1.OpCode.INSTANCEOF_FAST;
        } else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.instanceof_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), cls = classRef.cls, opStack = frame.opStack, o = opStack.pop();
        opStack.push(o !== null ? o.getClass().isCastable(cls) ? 1 : 0 : 0);
        frame.pc += 3;
    };
    Opcodes.monitorenter = function (thread, frame) {
        var opStack = frame.opStack, monitorObj = opStack.pop(), monitorEntered = function () {
                frame.pc++;
            };
        if (!monitorObj.getMonitor().enter(thread, monitorEntered)) {
            frame.returnToThreadLoop = true;
        } else {
            monitorEntered();
        }
    };
    Opcodes.monitorexit = function (thread, frame) {
        var monitorObj = frame.opStack.pop();
        if (monitorObj.getMonitor().exit(thread)) {
            frame.pc++;
        } else {
            frame.returnToThreadLoop = true;
        }
    };
    Opcodes.multianewarray = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (classRef.isResolved()) {
            code[pc] = enums_1.OpCode.MULTIANEWARRAY_FAST;
        } else {
            resolveCPItem(thread, frame, classRef);
        }
    };
    Opcodes.multianewarray_fast = function (thread, frame, code) {
        var pc = frame.pc;
        var classRef = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1)), opStack = frame.opStack, dim = code[pc + 3], i, args = new Array(dim), dimSize;
        for (i = 0; i < dim; i++) {
            dimSize = opStack.pop();
            args[dim - i - 1] = dimSize;
            if (dimSize < 0) {
                throwException(thread, frame, 'Ljava/lang/NegativeArraySizeException;', 'Tried to init ' + classRef.cls.getInternalName() + ' array with a dimension of length ' + dimSize);
                return;
            }
        }
        opStack.push(new (classRef.cls.getConstructor(thread))(thread, args));
        frame.pc += 4;
    };
    Opcodes.ifnull = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() == null) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.ifnonnull = function (thread, frame, code) {
        var pc = frame.pc;
        if (frame.opStack.pop() != null) {
            var offset = code.readInt16BE(pc + 1);
            frame.pc += offset;
            if (offset < 0) {
                frame.method.incrBBEntries();
            }
        } else {
            frame.pc += 3;
        }
    };
    Opcodes.goto_w = function (thread, frame, code) {
        var pc = frame.pc;
        var offset = code.readInt32BE(pc + 1);
        frame.pc += offset;
        if (offset < 0) {
            frame.method.incrBBEntries();
        }
    };
    Opcodes.jsr_w = function (thread, frame, code) {
        var pc = frame.pc;
        frame.opStack.push(frame.pc + 5);
        frame.pc += code.readInt32BE(pc + 1);
    };
    Opcodes.nop = function (thread, frame) {
        frame.pc += 1;
    };
    Opcodes.ldc = function (thread, frame, code) {
        var pc = frame.pc;
        var constant = frame.method.cls.constantPool.get(code[pc + 1]);
        if (constant.isResolved()) {
            assert_1['default'](function () {
                switch (constant.getType()) {
                case enums_1.ConstantPoolItemType.STRING:
                case enums_1.ConstantPoolItemType.CLASS:
                case enums_1.ConstantPoolItemType.METHOD_HANDLE:
                case enums_1.ConstantPoolItemType.METHOD_TYPE:
                case enums_1.ConstantPoolItemType.INTEGER:
                case enums_1.ConstantPoolItemType.FLOAT:
                    return true;
                default:
                    return false;
                }
            }(), 'Constant pool item ' + enums_1.ConstantPoolItemType[constant.getType()] + ' is not appropriate for LDC.');
            frame.opStack.push(constant.getConstant(thread));
            frame.pc += 2;
        } else {
            resolveCPItem(thread, frame, constant);
        }
    };
    Opcodes.ldc_w = function (thread, frame, code) {
        var pc = frame.pc;
        var constant = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        if (constant.isResolved()) {
            assert_1['default'](function () {
                switch (constant.getType()) {
                case enums_1.ConstantPoolItemType.STRING:
                case enums_1.ConstantPoolItemType.CLASS:
                case enums_1.ConstantPoolItemType.METHOD_HANDLE:
                case enums_1.ConstantPoolItemType.METHOD_TYPE:
                case enums_1.ConstantPoolItemType.INTEGER:
                case enums_1.ConstantPoolItemType.FLOAT:
                    return true;
                default:
                    return false;
                }
            }(), 'Constant pool item ' + enums_1.ConstantPoolItemType[constant.getType()] + ' is not appropriate for LDC_W.');
            frame.opStack.push(constant.getConstant(thread));
            frame.pc += 3;
        } else {
            resolveCPItem(thread, frame, constant);
        }
    };
    Opcodes.ldc2_w = function (thread, frame, code) {
        var pc = frame.pc;
        var constant = frame.method.cls.constantPool.get(code.readUInt16BE(pc + 1));
        assert_1['default'](constant.getType() === enums_1.ConstantPoolItemType.LONG || constant.getType() === enums_1.ConstantPoolItemType.DOUBLE, 'Invalid ldc_w constant pool type: ' + enums_1.ConstantPoolItemType[constant.getType()]);
        frame.opStack.pushWithNull(constant.value);
        frame.pc += 3;
    };
    Opcodes.wide = function (thread, frame, code) {
        var pc = frame.pc;
        var index = code.readUInt16BE(pc + 2);
        frame.pc += 4;
        switch (code[pc + 1]) {
        case enums_1.OpCode.ILOAD:
        case enums_1.OpCode.FLOAD:
        case enums_1.OpCode.ALOAD:
            frame.opStack.push(frame.locals[index]);
            break;
        case enums_1.OpCode.LLOAD:
        case enums_1.OpCode.DLOAD:
            frame.opStack.pushWithNull(frame.locals[index]);
            break;
        case enums_1.OpCode.ISTORE:
        case enums_1.OpCode.FSTORE:
        case enums_1.OpCode.ASTORE:
            frame.locals[index] = frame.opStack.pop();
            break;
        case enums_1.OpCode.LSTORE:
        case enums_1.OpCode.DSTORE:
            frame.locals[index + 1] = frame.opStack.pop();
            frame.locals[index] = frame.opStack.pop();
            break;
        case enums_1.OpCode.RET:
            frame.pc = frame.locals[index];
            break;
        case enums_1.OpCode.IINC:
            var value = code.readInt16BE(pc + 4);
            frame.locals[index] = frame.locals[index] + value | 0;
            frame.pc += 2;
            break;
        default:
            assert_1['default'](false, 'Unknown wide opcode: ' + code[pc + 1]);
            break;
        }
    };
    Opcodes.iaload = Opcodes._aload_32;
    Opcodes.faload = Opcodes._aload_32;
    Opcodes.aaload = Opcodes._aload_32;
    Opcodes.baload = Opcodes._aload_32;
    Opcodes.caload = Opcodes._aload_32;
    Opcodes.saload = Opcodes._aload_32;
    Opcodes.daload = Opcodes._aload_64;
    Opcodes.laload = Opcodes._aload_64;
    Opcodes.iastore = Opcodes._astore_32;
    Opcodes.fastore = Opcodes._astore_32;
    Opcodes.aastore = Opcodes._astore_32;
    Opcodes.bastore = Opcodes._astore_32;
    Opcodes.castore = Opcodes._astore_32;
    Opcodes.sastore = Opcodes._astore_32;
    Opcodes.lastore = Opcodes._astore_64;
    Opcodes.dastore = Opcodes._astore_64;
    Opcodes.iconst_0 = Opcodes._const_0_32;
    Opcodes.iconst_1 = Opcodes._const_1_32;
    Opcodes.iconst_2 = Opcodes._const_2_32;
    Opcodes.fconst_0 = Opcodes._const_0_32;
    Opcodes.fconst_1 = Opcodes._const_1_32;
    Opcodes.fconst_2 = Opcodes._const_2_32;
    Opcodes.iload = Opcodes._load_32;
    Opcodes.iload_0 = Opcodes._load_0_32;
    Opcodes.iload_1 = Opcodes._load_1_32;
    Opcodes.iload_2 = Opcodes._load_2_32;
    Opcodes.iload_3 = Opcodes._load_3_32;
    Opcodes.fload = Opcodes._load_32;
    Opcodes.fload_0 = Opcodes._load_0_32;
    Opcodes.fload_1 = Opcodes._load_1_32;
    Opcodes.fload_2 = Opcodes._load_2_32;
    Opcodes.fload_3 = Opcodes._load_3_32;
    Opcodes.aload = Opcodes._load_32;
    Opcodes.aload_0 = Opcodes._load_0_32;
    Opcodes.aload_1 = Opcodes._load_1_32;
    Opcodes.aload_2 = Opcodes._load_2_32;
    Opcodes.aload_3 = Opcodes._load_3_32;
    Opcodes.lload = Opcodes._load_64;
    Opcodes.lload_0 = Opcodes._load_0_64;
    Opcodes.lload_1 = Opcodes._load_1_64;
    Opcodes.lload_2 = Opcodes._load_2_64;
    Opcodes.lload_3 = Opcodes._load_3_64;
    Opcodes.dload = Opcodes._load_64;
    Opcodes.dload_0 = Opcodes._load_0_64;
    Opcodes.dload_1 = Opcodes._load_1_64;
    Opcodes.dload_2 = Opcodes._load_2_64;
    Opcodes.dload_3 = Opcodes._load_3_64;
    Opcodes.istore = Opcodes._store_32;
    Opcodes.istore_0 = Opcodes._store_0_32;
    Opcodes.istore_1 = Opcodes._store_1_32;
    Opcodes.istore_2 = Opcodes._store_2_32;
    Opcodes.istore_3 = Opcodes._store_3_32;
    Opcodes.fstore = Opcodes._store_32;
    Opcodes.fstore_0 = Opcodes._store_0_32;
    Opcodes.fstore_1 = Opcodes._store_1_32;
    Opcodes.fstore_2 = Opcodes._store_2_32;
    Opcodes.fstore_3 = Opcodes._store_3_32;
    Opcodes.astore = Opcodes._store_32;
    Opcodes.astore_0 = Opcodes._store_0_32;
    Opcodes.astore_1 = Opcodes._store_1_32;
    Opcodes.astore_2 = Opcodes._store_2_32;
    Opcodes.astore_3 = Opcodes._store_3_32;
    Opcodes.lstore = Opcodes._store_64;
    Opcodes.lstore_0 = Opcodes._store_0_64;
    Opcodes.lstore_1 = Opcodes._store_1_64;
    Opcodes.lstore_2 = Opcodes._store_2_64;
    Opcodes.lstore_3 = Opcodes._store_3_64;
    Opcodes.dstore = Opcodes._store_64;
    Opcodes.dstore_0 = Opcodes._store_0_64;
    Opcodes.dstore_1 = Opcodes._store_1_64;
    Opcodes.dstore_2 = Opcodes._store_2_64;
    Opcodes.dstore_3 = Opcodes._store_3_64;
    Opcodes.ireturn = Opcodes._return_32;
    Opcodes.freturn = Opcodes._return_32;
    Opcodes.areturn = Opcodes._return_32;
    Opcodes.lreturn = Opcodes._return_64;
    Opcodes.dreturn = Opcodes._return_64;
    Opcodes.invokeinterface_fast = Opcodes.invokevirtual_fast;
    return Opcodes;
}();
exports.Opcodes = Opcodes;
exports.LookupTable = new Array(255);
(function () {
    for (var i = 0; i < 255; i++) {
        if (enums_1.OpCode.hasOwnProperty('' + i)) {
            exports.LookupTable[i] = Opcodes[enums_1.OpCode[i].toLowerCase()];
            assert_1['default'](exports.LookupTable[i] != null, 'Missing implementation of opcode ' + enums_1.OpCode[i]);
        }
    }
}());
//# sourceMappingURL=opcodes.js.map