'use strict';
var PrefixParseResult = function () {
    function PrefixParseResult(result, unparsedArgs) {
        if (unparsedArgs === void 0) {
            unparsedArgs = [];
        }
        this._result = result;
        this._unparsedArgs = unparsedArgs;
    }
    PrefixParseResult.prototype.unparsedArgs = function () {
        return this._unparsedArgs;
    };
    PrefixParseResult.prototype.flag = function (name, defaultVal) {
        var val = this._result[name];
        if (typeof val === 'boolean') {
            return val;
        }
        return defaultVal;
    };
    PrefixParseResult.prototype.stringOption = function (name, defaultVal) {
        var val = this._result[name];
        if (typeof val === 'string') {
            return val;
        }
        return defaultVal;
    };
    PrefixParseResult.prototype.mapOption = function (name) {
        var val = this._result[name];
        if (typeof val === 'object') {
            return val;
        }
        return {};
    };
    return PrefixParseResult;
}();
exports.PrefixParseResult = PrefixParseResult;
function getOptName(prefix, name) {
    return prefix !== 'default' ? '' + prefix + name : name;
}
var OptionParser = function () {
    function OptionParser(desc) {
        var _this = this;
        this._parseMap = {};
        this._prefixes = [];
        this._mapArgs = [];
        this._rawDesc = desc;
        this._prefixes = Object.keys(desc);
        this._prefixes.forEach(function (prefix) {
            var opts = desc[prefix];
            var optNames = Object.keys(opts);
            optNames.slice(0).forEach(function (optName) {
                var option = opts[optName];
                if (option.enabled === false) {
                    delete desc[prefix][optName];
                    return;
                }
                if (!option.type) {
                    option.type = 0;
                }
                if (option.type === 4) {
                    _this._mapArgs.push(optName);
                }
                option.prefix = prefix;
                option.name = optName;
                _this._parseMap[getOptName(prefix, optName)] = option;
                if (option.alias) {
                    optNames.push(option.alias);
                    _this._parseMap[getOptName(prefix, option.alias)] = option;
                }
            });
        });
    }
    OptionParser.prototype.parse = function (argv) {
        var _this = this;
        var result = {}, ptr = 0, len;
        this._prefixes.forEach(function (prefix) {
            return result[prefix] = {};
        });
        argv = argv.map(function (arg) {
            return arg.trim();
        }).filter(function (arg) {
            return arg !== '';
        });
        len = argv.length;
        while (ptr < len) {
            var arg = argv[ptr];
            if (arg[0] === '-') {
                arg = arg.slice(1);
                var opt;
                if (opt = this._parseMap[arg]) {
                    switch (opt.type) {
                    case 0:
                    case 2:
                        result[opt.prefix][opt.name] = true;
                        break;
                    case 3:
                    case 1:
                        ptr++;
                        if (ptr < len) {
                            result[opt.prefix][opt.name] = argv[ptr];
                        } else {
                            throw new Error('-' + arg + ' requires an argument.');
                        }
                        break;
                    case 4:
                        break;
                    default:
                        throw new Error('INTERNAL ERROR: Invalid parse type for -' + arg + '.');
                    }
                } else if (this._mapArgs.filter(function (mapArg) {
                        if (arg.slice(0, mapArg.length) === mapArg) {
                            opt = _this._parseMap[mapArg];
                            return true;
                        }
                        return false;
                    }).length > 0) {
                    var mapping = arg.slice(opt.name.length), map = result[opt.prefix][opt.name];
                    if (!map) {
                        map = result[opt.prefix][opt.name] = {};
                    }
                    var eqIdx = mapping.indexOf('=');
                    if (eqIdx !== -1) {
                        map[mapping.slice(0, eqIdx)] = mapping.slice(eqIdx + 1);
                    } else {
                        map[mapping] = '';
                    }
                } else if (arg.indexOf(':') !== -1 && (opt = this._parseMap[arg.slice(0, arg.indexOf(':'))])) {
                    if (opt.type === 1 || opt.type === 2) {
                        result[opt.prefix][opt.name] = arg.slice(arg.indexOf(':') + 1);
                    } else {
                        throw new Error('Unrecognized option: -' + arg);
                    }
                } else {
                    throw new Error('Unrecognized option: -' + arg);
                }
                if (opt.stopParsing) {
                    ptr++;
                    break;
                }
            } else {
                break;
            }
            ptr++;
        }
        var unparsedArgs = argv.slice(ptr), rv = {};
        Object.keys(result).forEach(function (prefix) {
            rv[prefix] = new PrefixParseResult(result[prefix], unparsedArgs);
        });
        return rv;
    };
    OptionParser.prototype.help = function (prefix) {
        return _showHelp(this._rawDesc[prefix], prefix === 'default' ? '' : prefix);
    };
    return OptionParser;
}();
exports.OptionParser = OptionParser;
function printCol(value, width) {
    var rv = value;
    var padding = width - value.length;
    while (padding-- > 0) {
        rv += ' ';
    }
    return rv;
}
function _showHelp(category, prefix) {
    var combinedKeys = {};
    var keyColWidth = 13;
    Object.keys(category).forEach(function (key) {
        var opt = category[key];
        if (opt.stopParsing) {
            return;
        }
        var keys = [key];
        if (opt.alias != null) {
            keys.push(opt.alias);
        }
        var ckey;
        if (opt.optDesc) {
            ckey = keys.map(function (key) {
                return '-' + prefix + key + opt.optDesc;
            }).join('\n');
        } else {
            ckey = keys.map(function (key) {
                return '-' + prefix + key;
            }).join(' | ');
        }
        combinedKeys[ckey] = opt;
    });
    return Object.keys(combinedKeys).map(function (key) {
        var option = combinedKeys[key];
        if (option.optDesc) {
            var cols = key.split('\n');
            var rv = cols.map(function (row) {
                return '    ' + row;
            });
            return rv.join('\n') + '\n                  ' + option.desc;
        } else {
            var colText = printCol(key, keyColWidth);
            if (colText.length === keyColWidth) {
                return '    ' + colText + ' ' + option.desc;
            } else {
                return '    ' + colText + '\n                  ' + option.desc;
            }
        }
    }).join('\n') + '\n';
}
//# sourceMappingURL=option_parser.js.map