'use strict';
var jvm_1 = require('./jvm');
var util_1 = require('./util');
var difflib_1 = require('./difflib');
var path = require('path');
var fs = require('fs');
function makeTestingError(msg, origErr, fatal) {
    var err = new Error(msg);
    err.originalError = origErr;
    err.fatal = fatal;
    return err;
}
var OutputCapturer = function () {
    function OutputCapturer() {
        this._stdoutWrite = process.stdout.write;
        this._stderrWrite = process.stderr.write;
        this._data = '';
        this._isCapturing = false;
    }
    OutputCapturer.prototype.debugWrite = function (str) {
        this._stdoutWrite.apply(process.stdout, [
            str,
            'utf8'
        ]);
    };
    OutputCapturer.prototype.start = function (clear) {
        var _this = this;
        if (this._isCapturing) {
            throw new Error('Already capturing.');
        }
        this._isCapturing = true;
        if (clear) {
            this._data = '';
        }
        process.stderr.write = process.stdout.write = function (data, arg2, arg3) {
            if (typeof data !== 'string') {
                data = data.toString();
            }
            _this._data += data;
            return true;
        };
    };
    OutputCapturer.prototype.stop = function () {
        if (!this._isCapturing) {
            return;
        }
        this._isCapturing = false;
        process.stderr.write = this._stderrWrite;
        process.stdout.write = this._stdoutWrite;
    };
    OutputCapturer.prototype.getOutput = function (clear) {
        var data = this._data;
        if (clear) {
            this._data = '';
        }
        return data;
    };
    return OutputCapturer;
}();
var DoppioTest = function () {
    function DoppioTest(opts, cls) {
        this.outputCapturer = new OutputCapturer();
        this.opts = opts;
        if (cls.indexOf('.') !== -1) {
            cls = util_1.descriptor2typestr(util_1.int_classname(cls));
        }
        this.cls = cls;
        this.outFile = path.resolve(opts.doppioHomePath, cls) + '.runout';
    }
    DoppioTest.prototype.constructJVM = function (cb) {
        new jvm_1['default'](util_1.merge(jvm_1['default'].getDefaultOptions(this.opts.doppioHomePath), this.opts, {
            classpath: [this.opts.doppioHomePath],
            enableAssertions: true,
            enableSystemAssertions: true
        }), cb);
    };
    DoppioTest.prototype.run = function (registerGlobalErrorTrap, cb) {
        var _this = this;
        var outputCapturer = this.outputCapturer, _jvm = null, terminated = false, jvmConstructHasFinished = false, hasFinished = false;
        registerGlobalErrorTrap(function (err) {
            if (_jvm) {
                try {
                    _jvm.halt(1);
                } catch (e) {
                    err.message += '\n\nAdditionally, test runner received the following error while trying to halt the JVM: ' + e + (e.stack ? '\n\n' + e.stack : '') + '\n\nOriginal error\'s stack trace:';
                }
            }
            outputCapturer.stop();
            cb(makeTestingError('Uncaught error. Aborting further tests.\n\t' + err + (err.stack ? '\n\n' + err.stack : ''), err, true));
        });
        this.constructJVM(function (err, jvm) {
            _jvm = jvm;
            if (terminated) {
                return;
            }
            if (jvmConstructHasFinished) {
                return cb(makeTestingError('constructJVM returned twice. Aborting further tests.', null, true));
            }
            jvmConstructHasFinished = true;
            if (err) {
                cb(makeTestingError('Could not construct JVM:\n' + err, err));
            } else {
                outputCapturer.start(true);
                jvm.runClass(_this.cls, [], function (status) {
                    if (terminated) {
                        return;
                    }
                    outputCapturer.stop();
                    if (hasFinished) {
                        return cb(makeTestingError('JVM triggered completion callback twice. Aborting further tests.', null, true));
                    }
                    hasFinished = true;
                    var actual = outputCapturer.getOutput(true);
                    fs.readFile(_this.outFile, { encoding: 'utf8' }, function (err, expected) {
                        if (err) {
                            cb(makeTestingError('Could not read runout file:\n' + err, err));
                        } else {
                            var diffText = diff(actual, expected), errMsg = null;
                            if (diffText !== null) {
                                errMsg = 'Output does not match native JVM.';
                            }
                            cb(errMsg ? makeTestingError(errMsg) : null, actual, expected, diffText);
                        }
                    });
                });
            }
        });
    };
    return DoppioTest;
}();
exports.DoppioTest = DoppioTest;
function findTestClasses(doppioDir, cb) {
    var testDir = path.resolve(doppioDir, path.join('classes', 'test'));
    fs.readdir(testDir, function (err, files) {
        if (err) {
            cb([]);
        } else {
            cb(files.filter(function (file) {
                return path.extname(file) === '.java';
            }).map(function (file) {
                return path.join('classes', 'test', path.basename(file, '.java'));
            }));
        }
    });
}
function getTests(opts, cb) {
    var testClasses = opts.testClasses, tests;
    if (testClasses == null || testClasses.length === 0) {
        findTestClasses(opts.doppioHomePath, function (testClasses) {
            opts.testClasses = testClasses;
            getTests(opts, cb);
        });
    } else {
        cb(testClasses.map(function (testClass) {
            return new DoppioTest(opts, testClass);
        }));
    }
}
exports.getTests = getTests;
function diff(doppioOut, nativeOut) {
    var doppioLines = doppioOut.split(/\n/), jvmLines = nativeOut.split(/\n/), diff = difflib_1.text_diff(doppioLines, jvmLines, 2);
    if (diff.length > 0) {
        return 'Doppio | Java\n' + diff.join('\n');
    }
    return null;
}
exports.diff = diff;
function runTests(opts, quiet, continueAfterFailure, hideDiffs, registerGlobalErrorTrap, cb) {
    function print(str) {
        if (!quiet) {
            process.stdout.write(str);
        }
    }
    getTests(opts, function (tests) {
        util_1.asyncForEach(tests, function (test, nextTest) {
            var hasFinished = false;
            print('[' + test.cls + ']: Running... ');
            test.run(registerGlobalErrorTrap, function (err, actual, expected, diff) {
                if (err && !hideDiffs && diff) {
                    err.message += '\n' + diff;
                }
                if (err) {
                    print('fail.\n\t' + err.message + '\n');
                    if (err.originalError && err.originalError.stack) {
                        print(err.stack + '\n');
                    }
                    if (!continueAfterFailure || err['fatal']) {
                        err.message = 'Failed ' + test.cls + ': ' + err.message;
                        nextTest(err);
                    } else {
                        nextTest();
                    }
                } else {
                    print('pass.\n');
                    nextTest();
                }
            });
        }, cb);
    });
}
exports.runTests = runTests;
//# sourceMappingURL=testing.js.map