'use strict';
var enums_1 = require('./enums');
var assert_1 = require('./assert');
var gLong_1 = require('./gLong');
var opcodes_1 = require('./opcodes');
var logging_1 = require('./logging');
var logging = require('./logging');
var util_1 = require('./util');
var debug = logging.debug;
var vtrace = logging.vtrace;
var trace = logging.trace;
var maxMethodResumes = 10000, methodResumesLeft = maxMethodResumes, numSamples = 1;
var PreAllocatedStack = function () {
    function PreAllocatedStack(initialSize) {
        this.curr = 0;
        this.store = new Array(initialSize);
    }
    PreAllocatedStack.prototype.push = function (x) {
        this.store[this.curr++] = x;
    };
    PreAllocatedStack.prototype.pushAll = function () {
        var n = arguments.length;
        for (var i = 0; i < n; i++) {
            this.store[this.curr++] = arguments[i];
        }
    };
    PreAllocatedStack.prototype.pushWithNull = function (x) {
        this.store[this.curr] = x;
        this.curr += 2;
    };
    PreAllocatedStack.prototype.push6 = function (x, y, z, z1, z2, z3) {
        this.store[this.curr++] = x;
        this.store[this.curr++] = y;
        this.store[this.curr++] = z;
        this.store[this.curr++] = z1;
        this.store[this.curr++] = z2;
        this.store[this.curr++] = z3;
    };
    PreAllocatedStack.prototype.swap = function () {
        var tmp = this.store[this.curr - 1];
        this.store[this.curr - 1] = this.store[this.curr - 2];
        this.store[this.curr - 2] = tmp;
    };
    PreAllocatedStack.prototype.dup = function () {
        this.store[this.curr] = this.store[this.curr - 1];
        this.curr++;
    };
    PreAllocatedStack.prototype.dup2 = function () {
        this.store[this.curr] = this.store[this.curr - 2];
        this.store[this.curr + 1] = this.store[this.curr - 1];
        this.curr += 2;
    };
    PreAllocatedStack.prototype.dup_x1 = function () {
        var v1 = this.store[this.curr - 1];
        this.store[this.curr - 1] = this.store[this.curr - 2];
        this.store[this.curr] = v1;
        this.store[this.curr - 2] = v1;
        this.curr++;
    };
    PreAllocatedStack.prototype.dup_x2 = function () {
        var v1 = this.store[this.curr - 1];
        this.store[this.curr - 1] = this.store[this.curr - 2];
        this.store[this.curr - 2] = this.store[this.curr - 3];
        this.store[this.curr] = v1;
        this.store[this.curr - 3] = v1;
        this.curr++;
    };
    PreAllocatedStack.prototype.dup2_x1 = function () {
        var v1 = this.store[this.curr - 1];
        var v2 = this.store[this.curr - 2];
        this.store[this.curr] = v2;
        this.store[this.curr + 1] = v1;
        this.store[this.curr - 1] = this.store[this.curr - 3];
        this.store[this.curr - 2] = v1;
        this.store[this.curr - 3] = v2;
        this.curr += 2;
    };
    PreAllocatedStack.prototype.pop = function () {
        return this.store[--this.curr];
    };
    PreAllocatedStack.prototype.pop2 = function () {
        this.curr -= 2;
        return this.store[this.curr];
    };
    PreAllocatedStack.prototype.bottom = function () {
        return this.store[0];
    };
    PreAllocatedStack.prototype.top = function () {
        return this.store[this.curr - 1];
    };
    PreAllocatedStack.prototype.fromTop = function (n) {
        return this.store[this.curr - (n + 1)];
    };
    PreAllocatedStack.prototype.sliceFromBottom = function (n) {
        return this.store.slice(n, this.curr);
    };
    PreAllocatedStack.prototype.sliceFromTop = function (n) {
        return this.store.slice(this.curr - n, this.curr);
    };
    PreAllocatedStack.prototype.dropFromTop = function (n) {
        this.curr -= n;
    };
    PreAllocatedStack.prototype.sliceAndDropFromTop = function (n) {
        var curr = this.curr;
        this.curr -= n;
        return this.store.slice(curr - n, curr);
    };
    PreAllocatedStack.prototype.getRaw = function () {
        return this.store.slice(0, this.curr);
    };
    PreAllocatedStack.prototype.clear = function () {
        this.curr = 0;
    };
    return PreAllocatedStack;
}();
exports.PreAllocatedStack = PreAllocatedStack;
var jitUtil = {
    isNull: opcodes_1.isNull,
    resolveCPItem: opcodes_1.resolveCPItem,
    throwException: opcodes_1.throwException,
    gLong: gLong_1['default'],
    float2int: util_1.float2int,
    wrapFloat: util_1.wrapFloat,
    Constants: enums_1.Constants
};
var BytecodeStackFrame = function () {
    function BytecodeStackFrame(method, args) {
        this.pc = 0;
        this.returnToThreadLoop = false;
        this.lockedMethodLock = false;
        this.type = enums_1.StackFrameType.BYTECODE;
        this.method = method;
        method.incrBBEntries();
        assert_1['default'](!method.accessFlags.isNative(), 'Cannot run a native method using a BytecodeStackFrame.');
        assert_1['default'](!method.accessFlags.isAbstract(), 'Cannot run an abstract method!');
        this.locals = args;
        this.opStack = new PreAllocatedStack(method.getCodeAttribute().getMaxStack());
    }
    BytecodeStackFrame.prototype.run = function (thread) {
        var _this = this;
        var method = this.method, code = this.method.getCodeAttribute().getCode(), opcodeTable = opcodes_1.LookupTable;
        if (!RELEASE && logging_1.logLevel >= logging_1.LogLevel.TRACE) {
            if (this.pc === 0) {
                ;
            } else {
                ;
            }
            ;
        }
        if (method.accessFlags.isSynchronized() && !this.lockedMethodLock) {
            this.lockedMethodLock = method.methodLock(thread, this).enter(thread, function () {
                _this.lockedMethodLock = true;
            });
            if (!this.lockedMethodLock) {
                assert_1['default'](thread.getStatus() === enums_1.ThreadStatus.BLOCKED, 'Failed to enter a monitor. Thread must be BLOCKED.');
                return;
            }
        }
        this.returnToThreadLoop = false;
        if (thread.getJVM().isJITDisabled()) {
            while (!this.returnToThreadLoop) {
                var opCode = code[this.pc];
                if (!RELEASE && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                    ;
                }
                opcodeTable[opCode](thread, this, code);
                if (!RELEASE && !this.returnToThreadLoop && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                    ;
                }
            }
        } else {
            while (!this.returnToThreadLoop) {
                var op = method.getOp(this.pc, code, thread);
                if (typeof op === 'function') {
                    if (!RELEASE && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                        ;
                    }
                    op(this, thread, jitUtil);
                } else {
                    if (!RELEASE && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                        ;
                    }
                    opcodeTable[op](thread, this, code);
                }
                if (!RELEASE && !this.returnToThreadLoop && logging_1.logLevel === logging_1.LogLevel.VTRACE) {
                    ;
                }
            }
        }
    };
    BytecodeStackFrame.prototype.scheduleResume = function (thread, rv, rv2) {
        var prevOp = this.method.getCodeAttribute().getCode()[this.pc];
        switch (prevOp) {
        case enums_1.OpCode.INVOKEINTERFACE:
        case enums_1.OpCode.INVOKEINTERFACE_FAST:
            this.pc += 5;
            break;
        case enums_1.OpCode.INVOKESPECIAL:
        case enums_1.OpCode.INVOKESTATIC:
        case enums_1.OpCode.INVOKEVIRTUAL:
        case enums_1.OpCode.INVOKESTATIC_FAST:
        case enums_1.OpCode.INVOKENONVIRTUAL_FAST:
        case enums_1.OpCode.INVOKEVIRTUAL_FAST:
        case enums_1.OpCode.INVOKEHANDLE:
        case enums_1.OpCode.INVOKEBASIC:
        case enums_1.OpCode.LINKTOSPECIAL:
        case enums_1.OpCode.LINKTOVIRTUAL:
        case enums_1.OpCode.INVOKEDYNAMIC:
        case enums_1.OpCode.INVOKEDYNAMIC_FAST:
            this.pc += 3;
            break;
        default:
            assert_1['default'](false, 'Resuming from a non-invoke opcode! Opcode: ' + enums_1.OpCode[prevOp] + ' [' + prevOp + ']');
            break;
        }
        if (rv !== undefined) {
            this.opStack.push(rv);
        }
        if (rv2 !== undefined) {
            this.opStack.push(rv2);
        }
    };
    BytecodeStackFrame.prototype.scheduleException = function (thread, e) {
        var codeAttr = this.method.getCodeAttribute(), pc = this.pc, method = this.method, exceptionHandlers = codeAttr.exceptionHandlers, ecls = e.getClass(), handler;
        for (var i = 0; i < exceptionHandlers.length; i++) {
            var eh = exceptionHandlers[i];
            if (eh.startPC <= pc && pc < eh.endPC) {
                if (eh.catchType === '<any>') {
                    handler = eh;
                    break;
                } else {
                    var resolvedCatchType = method.cls.getLoader().getResolvedClass(eh.catchType);
                    if (resolvedCatchType != null) {
                        if (ecls.isCastable(resolvedCatchType)) {
                            handler = eh;
                            break;
                        }
                    } else {
                        ;
                        var handlerClasses = [];
                        for (var i_1 = 0; i_1 < exceptionHandlers.length; i_1++) {
                            var handler_1 = exceptionHandlers[i_1];
                            if (handler_1.catchType !== '<any>') {
                                handlerClasses.push(handler_1.catchType);
                            }
                        }
                        ;
                        thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
                        method.cls.getLoader().resolveClasses(thread, handlerClasses, function (classes) {
                            if (classes !== null) {
                                ;
                                thread.throwException(e);
                            }
                        });
                        return true;
                    }
                }
            }
        }
        if (handler != null) {
            ;
            this.opStack.clear();
            this.opStack.push(e);
            this.pc = handler.handlerPC;
            return true;
        } else {
            ;
            if (method.accessFlags.isSynchronized()) {
                method.methodLock(thread, this).exit(thread);
            }
            return false;
        }
    };
    BytecodeStackFrame.prototype.getLoader = function () {
        return this.method.cls.getLoader();
    };
    BytecodeStackFrame.prototype.getStackTraceFrame = function () {
        return {
            method: this.method,
            pc: this.pc,
            stack: this.opStack.sliceFromBottom(0),
            locals: this.locals.slice(0)
        };
    };
    return BytecodeStackFrame;
}();
exports.BytecodeStackFrame = BytecodeStackFrame;
var NativeStackFrame = function () {
    function NativeStackFrame(method, args) {
        this.type = enums_1.StackFrameType.NATIVE;
        this.method = method;
        this.args = args;
        assert_1['default'](method.accessFlags.isNative());
        this.nativeMethod = method.getNativeFunction();
    }
    NativeStackFrame.prototype.run = function (thread) {
        ;
        var rv = this.nativeMethod.apply(null, this.method.convertArgs(thread, this.args));
        if (thread.getStatus() === enums_1.ThreadStatus.RUNNABLE && thread.currentMethod() === this.method) {
            var returnType = this.method.returnType;
            switch (returnType) {
            case 'J':
            case 'D':
                thread.asyncReturn(rv, null);
                break;
            case 'Z':
                thread.asyncReturn(rv ? 1 : 0);
                break;
            default:
                thread.asyncReturn(rv);
                break;
            }
        }
    };
    NativeStackFrame.prototype.scheduleResume = function (thread, rv, rv2) {
    };
    NativeStackFrame.prototype.scheduleException = function (thread, e) {
        return false;
    };
    NativeStackFrame.prototype.getStackTraceFrame = function () {
        return {
            method: this.method,
            pc: -1,
            stack: [],
            locals: []
        };
    };
    NativeStackFrame.prototype.getLoader = function () {
        return this.method.cls.getLoader();
    };
    return NativeStackFrame;
}();
exports.NativeStackFrame = NativeStackFrame;
var InternalStackFrame = function () {
    function InternalStackFrame(cb) {
        this.isException = false;
        this.type = enums_1.StackFrameType.INTERNAL;
        this.cb = cb;
    }
    InternalStackFrame.prototype.run = function (thread) {
        thread.framePop();
        thread.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
        if (this.isException) {
            this.cb(this.val);
        } else {
            this.cb(null, this.val);
        }
    };
    InternalStackFrame.prototype.scheduleResume = function (thread, rv) {
        this.isException = false;
        this.val = rv;
    };
    InternalStackFrame.prototype.scheduleException = function (thread, e) {
        this.isException = true;
        this.val = e;
        return true;
    };
    InternalStackFrame.prototype.getStackTraceFrame = function () {
        return null;
    };
    InternalStackFrame.prototype.getLoader = function () {
        throw new Error('Internal stack frames have no loader.');
    };
    return InternalStackFrame;
}();
exports.InternalStackFrame = InternalStackFrame;
var JVMThread = function () {
    function JVMThread(jvm, tpool, threadObj) {
        this.status = enums_1.ThreadStatus.NEW;
        this.stack = [];
        this.interrupted = false;
        this.monitor = null;
        this.jvm = jvm;
        this.bsCl = jvm.getBootstrapClassLoader();
        this.tpool = tpool;
        this.jvmThreadObj = threadObj;
    }
    JVMThread.prototype.getJVMObject = function () {
        return this.jvmThreadObj;
    };
    JVMThread.prototype.isDaemon = function () {
        return this.jvmThreadObj['java/lang/Thread/daemon'] !== 0;
    };
    JVMThread.prototype.getPriority = function () {
        return this.jvmThreadObj['java/lang/Thread/priority'];
    };
    JVMThread.prototype.setJVMObject = function (obj) {
        obj['java/lang/Thread/threadStatus'] = this.jvmThreadObj['java/lang/Thread/threadStatus'];
        this.jvmThreadObj = obj;
    };
    JVMThread.prototype.getRef = function () {
        return this.jvmThreadObj.ref;
    };
    JVMThread.prototype.isInterrupted = function () {
        return this.interrupted;
    };
    JVMThread.prototype.currentMethod = function () {
        var stack = this.stack, idx = stack.length, method;
        while (--idx >= 0) {
            method = stack[idx].getStackTraceFrame().method;
            if (method !== null) {
                return method;
            }
        }
        return null;
    };
    JVMThread.prototype.setInterrupted = function (interrupted) {
        this.interrupted = interrupted;
    };
    JVMThread.prototype.getBsCl = function () {
        return this.bsCl;
    };
    JVMThread.prototype.getLoader = function () {
        var loader = this.stack[this.stack.length - 1].getLoader();
        if (loader) {
            return loader;
        } else {
            var len = this.stack.length;
            for (var i = 2; i <= len; i++) {
                loader = this.stack[len - i].getLoader();
                if (loader) {
                    return loader;
                }
            }
            throw new Error('Unable to find loader.');
        }
    };
    JVMThread.prototype.import = function (names, cb, explicit) {
        var _this = this;
        if (explicit === void 0) {
            explicit = true;
        }
        var loader = this.getLoader();
        this.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
        if (Array.isArray(names)) {
            var rv_1 = [];
            util_1.asyncForEach(names, function (name, nextItem) {
                _this._import(name, loader, function (cons) {
                    rv_1.push(cons);
                    nextItem();
                }, explicit);
            }, function (e) {
                cb(rv_1);
            });
        } else {
            this._import(names, loader, cb, explicit);
        }
    };
    JVMThread.prototype._import = function (name, loader, cb, explicit) {
        var _this = this;
        var cls = loader.getInitializedClass(this, name);
        if (cls) {
            setImmediate(function () {
                return cb(cls.getConstructor(_this));
            });
        } else {
            loader.initializeClass(this, name, function (cdata) {
                if (cdata) {
                    cb(cdata.getConstructor(_this));
                }
            }, explicit);
        }
    };
    JVMThread.prototype.getJVM = function () {
        return this.jvm;
    };
    JVMThread.prototype.getThreadPool = function () {
        return this.tpool;
    };
    JVMThread.prototype.getStackTrace = function () {
        var trace = [], i, frame;
        for (i = 0; i < this.stack.length; i++) {
            frame = this.stack[i].getStackTraceFrame();
            if (frame != null) {
                trace.push(frame);
            }
        }
        return trace;
    };
    JVMThread.prototype.getPrintableStackTrace = function () {
        var rv = '';
        this.getStackTrace().reverse().forEach(function (trace) {
            rv += '\tat ' + util_1.ext_classname(trace.method.cls.getInternalName()) + '::' + trace.method.name + '(';
            if (trace.pc >= 0) {
                var code = trace.method.getCodeAttribute();
                var table = code.getAttribute('LineNumberTable');
                var srcAttr = trace.method.cls.getAttribute('SourceFile');
                if (srcAttr != null) {
                    rv += srcAttr.filename;
                } else {
                    rv += 'unknown';
                }
                if (table != null) {
                    var lineNumber = table.getLineNumber(trace.pc);
                    rv += ':' + lineNumber;
                    rv += ' Bytecode offset: ' + trace.pc;
                }
            } else {
                rv += 'native';
            }
            rv += ')\n';
        });
        return rv;
    };
    JVMThread.prototype.run = function () {
        var stack = this.stack, startTime = new Date().getTime();
        methodResumesLeft = maxMethodResumes;
        while (this.status === enums_1.ThreadStatus.RUNNABLE && stack.length > 0) {
            var sf = stack[stack.length - 1];
            if (!RELEASE) {
                if (sf.type === enums_1.StackFrameType.BYTECODE && this.jvm.shouldVtrace(sf.method.fullSignature)) {
                    var oldLevel = logging_1.logLevel;
                    logging_1.setLogLevel(logging_1.LogLevel.VTRACE);
                    sf.run(this);
                    logging_1.setLogLevel(oldLevel);
                } else {
                    sf.run(this);
                }
            } else {
                sf.run(this);
            }
            if (--methodResumesLeft === 0) {
                var endTime = new Date().getTime();
                var duration = endTime - startTime;
                var estMaxMethodResumes = maxMethodResumes / duration * this.jvm.getResponsiveness() | 0;
                maxMethodResumes = (estMaxMethodResumes + numSamples * maxMethodResumes) / (numSamples + 1) | 0;
                if (maxMethodResumes <= 0) {
                    maxMethodResumes = 10;
                }
                ;
                numSamples++;
                this.tpool.quantumOver(this);
                break;
            }
        }
        if (stack.length === 0) {
            this.setStatus(enums_1.ThreadStatus.TERMINATED);
        }
    };
    JVMThread.prototype.sanityCheck = function () {
        switch (this.status) {
        case enums_1.ThreadStatus.NEW:
            return true;
        case enums_1.ThreadStatus.RUNNABLE:
            assert_1['default'](this.stack.length > 0, 'A runnable thread must not have an empty stack.');
            return true;
        case enums_1.ThreadStatus.TIMED_WAITING:
            assert_1['default'](this.monitor != null && this.monitor.isTimedWaiting(this), 'A timed waiting thread must be waiting on a monitor.');
            return true;
        case enums_1.ThreadStatus.WAITING:
            assert_1['default'](this.monitor != null && this.monitor.isWaiting(this), 'A waiting thread must be waiting on a monitor.');
            return true;
        case enums_1.ThreadStatus.BLOCKED:
        case enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED:
            assert_1['default'](this.monitor != null && this.monitor.isBlocked(this), 'A blocked thread must be blocked on a monitor');
            return true;
        case enums_1.ThreadStatus.ASYNC_WAITING:
            return true;
        case enums_1.ThreadStatus.TERMINATED:
            assert_1['default'](this.stack.length === 0, 'A terminated thread must have an empty stack.');
            return true;
        case enums_1.ThreadStatus.PARKED:
            assert_1['default'](this.jvm.getParker().isParked(this), 'A parked thread must be parked.');
            return true;
        default:
            return false;
        }
    };
    JVMThread.prototype.rawSetStatus = function (newStatus) {
        var jvmNewStatus = 0, oldStatus = this.status;
        if (logging_1.logLevel === logging_1.LogLevel.VTRACE) {
            ;
        }
        assert_1['default'](validateThreadTransition(oldStatus, newStatus), 'Invalid thread transition: ' + enums_1.ThreadStatus[oldStatus] + ' => ' + enums_1.ThreadStatus[newStatus]);
        this.status = newStatus;
        switch (newStatus) {
        case enums_1.ThreadStatus.NEW:
            jvmNewStatus |= enums_1.JVMTIThreadState.ALIVE;
            break;
        case enums_1.ThreadStatus.RUNNABLE:
            jvmNewStatus |= enums_1.JVMTIThreadState.RUNNABLE;
            break;
        case enums_1.ThreadStatus.BLOCKED:
        case enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED:
            jvmNewStatus |= enums_1.JVMTIThreadState.BLOCKED_ON_MONITOR_ENTER;
            break;
        case enums_1.ThreadStatus.WAITING:
        case enums_1.ThreadStatus.ASYNC_WAITING:
        case enums_1.ThreadStatus.PARKED:
            jvmNewStatus |= enums_1.JVMTIThreadState.WAITING_INDEFINITELY;
            break;
        case enums_1.ThreadStatus.TIMED_WAITING:
            jvmNewStatus |= enums_1.JVMTIThreadState.WAITING_WITH_TIMEOUT;
            break;
        case enums_1.ThreadStatus.TERMINATED:
            jvmNewStatus |= enums_1.JVMTIThreadState.TERMINATED;
            break;
        default:
            jvmNewStatus = enums_1.JVMTIThreadState.RUNNABLE;
            break;
        }
        this.jvmThreadObj['java/lang/Thread/threadStatus'] = jvmNewStatus;
        this.tpool.statusChange(this, oldStatus, this.status);
    };
    JVMThread.prototype.setStatus = function (status, monitor) {
        if (monitor === void 0) {
            monitor = null;
        }
        if (this.status !== status) {
            var oldStatus = this.status;
            this.monitor = monitor;
            if (status !== enums_1.ThreadStatus.TERMINATED) {
                this.rawSetStatus(status);
            } else {
                this.exit();
            }
            assert_1['default'](this.sanityCheck(), 'Invalid thread status.');
        }
    };
    JVMThread.prototype.exit = function () {
        var _this = this;
        var monitor = this.jvmThreadObj.getMonitor();
        if (monitor.isBlocked(this) || monitor.getOwner() === this || this.status === enums_1.ThreadStatus.TERMINATED) {
            return;
        }
        if (this.stack.length === 0) {
            this.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
            if (this.jvm.hasVMBooted()) {
                ;
                var phase2 = function () {
                    ;
                    _this.jvmThreadObj['exit()V'](_this, null, function (e) {
                        monitor.notifyAll(_this);
                        monitor.exit(_this);
                        ;
                        _this.rawSetStatus(enums_1.ThreadStatus.TERMINATED);
                    });
                };
                if (monitor.enter(this, phase2)) {
                    phase2();
                }
            } else {
                ;
            }
        } else {
            while (this.stack.length > 0) {
                this.stack.pop();
            }
            ;
            this.rawSetStatus(enums_1.ThreadStatus.TERMINATED);
        }
    };
    JVMThread.prototype.signalPriorityChange = function () {
        this.tpool.priorityChange(this);
    };
    JVMThread.prototype.getMonitorBlock = function () {
        return this.monitor;
    };
    JVMThread.prototype.getStatus = function () {
        return this.status;
    };
    JVMThread.prototype.asyncReturn = function (rv, rv2) {
        var stack = this.stack;
        assert_1['default'](this.status === enums_1.ThreadStatus.RUNNABLE || this.status === enums_1.ThreadStatus.ASYNC_WAITING);
        assert_1['default'](typeof rv !== 'boolean' && rv2 == null);
        var frame = stack.pop();
        if (frame.type != enums_1.StackFrameType.INTERNAL) {
            var frameCast = frame;
            if (frame.type === enums_1.StackFrameType.BYTECODE) {
                ;
            }
            ;
            assert_1['default'](validateReturnValue(this, frameCast.method, frameCast.method.returnType, this.bsCl, frameCast.method.cls.getLoader(), rv, rv2), 'Invalid return value for method ' + frameCast.method.getFullSignature());
        }
        var idx = stack.length - 1;
        if (idx >= 0) {
            stack[idx].scheduleResume(this, rv, rv2);
        }
        this.setStatus(enums_1.ThreadStatus.RUNNABLE);
    };
    JVMThread.prototype.framePop = function () {
        this.stack.pop();
    };
    JVMThread.prototype.throwException = function (exception) {
        assert_1['default'](this.status === enums_1.ThreadStatus.RUNNABLE || this.status === enums_1.ThreadStatus.ASYNC_WAITING, 'Tried to throw exception while thread was in state ' + enums_1.ThreadStatus[this.status]);
        var stack = this.stack, idx = stack.length - 1;
        if (idx >= 0) {
            if (stack[idx].type === enums_1.StackFrameType.INTERNAL) {
                stack.pop();
                idx--;
            }
            this.setStatus(enums_1.ThreadStatus.RUNNABLE);
            while (stack.length > 0 && !stack[idx].scheduleException(this, exception)) {
                stack.pop();
                idx--;
            }
        }
        if (stack.length === 0) {
            this.handleUncaughtException(exception);
        }
    };
    JVMThread.prototype.throwNewException = function (clsName, msg) {
        var _this = this;
        var cls = this.getLoader().getInitializedClass(this, clsName), throwException = function () {
                var eCons = cls.getConstructor(_this), e = new eCons(_this);
                e['<init>(Ljava/lang/String;)V'](_this, [util_1.initString(_this.bsCl, msg)], function (err) {
                    if (err) {
                        _this.throwException(err);
                    } else {
                        _this.throwException(e);
                    }
                });
            };
        if (cls != null) {
            throwException();
        } else {
            this.setStatus(enums_1.ThreadStatus.ASYNC_WAITING);
            this.getLoader().initializeClass(this, clsName, function (cdata) {
                if (cdata != null) {
                    cls = cdata;
                    throwException();
                }
            }, false);
        }
    };
    JVMThread.prototype.handleUncaughtException = function (exception) {
        this.jvmThreadObj['dispatchUncaughtException(Ljava/lang/Throwable;)V'](this, [exception]);
    };
    JVMThread.prototype.close = function () {
        this.jvm = null;
    };
    return JVMThread;
}();
exports.JVMThread = JVMThread;
exports.validTransitions = {};
exports.validTransitions[enums_1.ThreadStatus.NEW] = {};
exports.validTransitions[enums_1.ThreadStatus.NEW][enums_1.ThreadStatus.RUNNABLE] = 'RunMethod invoked on new thread';
exports.validTransitions[enums_1.ThreadStatus.NEW][enums_1.ThreadStatus.ASYNC_WAITING] = '[JVM bootup only] Internal operation occurs on new thread';
exports.validTransitions[enums_1.ThreadStatus.NEW][enums_1.ThreadStatus.TERMINATED] = '[JVM halt0 only] When the JVM shuts down, it terminates all threads, including those that have never been run.';
exports.validTransitions[enums_1.ThreadStatus.ASYNC_WAITING] = {};
exports.validTransitions[enums_1.ThreadStatus.ASYNC_WAITING][enums_1.ThreadStatus.RUNNABLE] = 'Async operation completes';
exports.validTransitions[enums_1.ThreadStatus.ASYNC_WAITING][enums_1.ThreadStatus.TERMINATED] = 'RunMethod completes and callstack is empty';
exports.validTransitions[enums_1.ThreadStatus.BLOCKED] = {};
exports.validTransitions[enums_1.ThreadStatus.BLOCKED][enums_1.ThreadStatus.RUNNABLE] = 'Acquires monitor, or is interrupted';
exports.validTransitions[enums_1.ThreadStatus.BLOCKED][enums_1.ThreadStatus.TERMINATED] = 'Thread is terminated whilst blocked.';
exports.validTransitions[enums_1.ThreadStatus.PARKED] = {};
exports.validTransitions[enums_1.ThreadStatus.PARKED][enums_1.ThreadStatus.ASYNC_WAITING] = 'Balancing unpark, or is interrupted';
exports.validTransitions[enums_1.ThreadStatus.PARKED][enums_1.ThreadStatus.TERMINATED] = 'Thread is terminated whilst parked.';
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE] = {};
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.ASYNC_WAITING] = 'Thread performs an asynchronous JavaScript operation';
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.TERMINATED] = 'Callstack is empty';
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.BLOCKED] = 'Thread waits to acquire monitor';
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.WAITING] = 'Thread waits on monitor (Object.wait)';
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.TIMED_WAITING] = 'Thread waits on monitor with timeout (Object.wait)';
exports.validTransitions[enums_1.ThreadStatus.RUNNABLE][enums_1.ThreadStatus.PARKED] = 'Thread parks itself';
exports.validTransitions[enums_1.ThreadStatus.TERMINATED] = {};
exports.validTransitions[enums_1.ThreadStatus.TERMINATED][enums_1.ThreadStatus.NEW] = 'Thread is resurrected for re-use';
exports.validTransitions[enums_1.ThreadStatus.TERMINATED][enums_1.ThreadStatus.RUNNABLE] = 'Thread is resurrected for re-use';
exports.validTransitions[enums_1.ThreadStatus.TERMINATED][enums_1.ThreadStatus.ASYNC_WAITING] = '[JVM Bootup] Thread is resurrected for internal operation';
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING] = {};
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING][enums_1.ThreadStatus.RUNNABLE] = 'Timer expires, or thread is interrupted, and thread immediately acquires lock';
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING][enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED] = 'Thread is interrupted or notified, or timer expires, and lock already owned';
exports.validTransitions[enums_1.ThreadStatus.TIMED_WAITING][enums_1.ThreadStatus.TERMINATED] = 'Thread is terminated whilst waiting.';
exports.validTransitions[enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED] = {};
exports.validTransitions[enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED][enums_1.ThreadStatus.RUNNABLE] = 'Thread acquires monitor';
exports.validTransitions[enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED][enums_1.ThreadStatus.TERMINATED] = 'Thread is terminated whilst blocked.';
exports.validTransitions[enums_1.ThreadStatus.WAITING] = {};
exports.validTransitions[enums_1.ThreadStatus.WAITING][enums_1.ThreadStatus.RUNNABLE] = 'Thread is interrupted, and immediately acquires lock';
exports.validTransitions[enums_1.ThreadStatus.WAITING][enums_1.ThreadStatus.UNINTERRUPTABLY_BLOCKED] = 'Thread is notified or interrupted, and does not immediately acquire lock';
exports.validTransitions[enums_1.ThreadStatus.WAITING][enums_1.ThreadStatus.TERMINATED] = 'Thread is terminated whilst waiting.';
function validateThreadTransition(oldStatus, newStatus) {
    var rv = exports.validTransitions.hasOwnProperty('' + oldStatus) && exports.validTransitions[oldStatus].hasOwnProperty('' + newStatus);
    return rv;
}
function validateReturnValue(thread, method, returnType, bsCl, cl, rv1, rv2) {
    if (method.fullSignature === 'java/lang/invoke/MethodHandle/invokeBasic([Ljava/lang/Object;)Ljava/lang/Object;') {
        return true;
    }
    var cls;
    if (util_1.is_primitive_type(returnType)) {
        switch (returnType) {
        case 'Z':
            assert_1['default'](rv2 === undefined, 'Second return value must be undefined for Boolean type.');
            assert_1['default'](rv1 === 1 || rv1 === 0, 'Booleans must be 0 or 1.');
            break;
        case 'B':
            assert_1['default'](rv2 === undefined, 'Second return value must be undefined for Byte type.');
            assert_1['default'](rv1 <= 127 && rv1 >= -128, 'Byte value for method ' + method.name + ' is out of bounds: ' + rv1);
            break;
        case 'C':
            assert_1['default'](rv2 === undefined, 'Second return value must be undefined for Character type.');
            assert_1['default'](rv1 <= 65535 && rv1 >= 0, 'Character value is out of bounds: ' + rv1);
            break;
        case 'S':
            assert_1['default'](rv2 === undefined, 'Second return value must be undefined for Short type.');
            assert_1['default'](rv1 <= 32767 && rv1 >= -32768, 'Short value is out of bounds: ' + rv1);
            break;
        case 'I':
            assert_1['default'](rv2 === undefined, 'Second return value must be undefined for Int type.');
            assert_1['default'](rv1 <= 2147483647 && rv1 >= -2147483648, 'Int value is out of bounds: ' + rv1);
            break;
        case 'J':
            assert_1['default'](rv2 === null, 'Second return value must be NULL for Long type.');
            assert_1['default'](rv1.lessThanOrEqual(gLong_1['default'].MAX_VALUE) && rv1.greaterThanOrEqual(gLong_1['default'].MIN_VALUE), 'Long value is out of bounds: ' + rv1);
            break;
        case 'F':
            assert_1['default'](rv2 === undefined, 'Second return value must be undefined for Float type.');
            assert_1['default'](util_1.wrapFloat(rv1) === rv1 || isNaN(rv1) && isNaN(util_1.wrapFloat(rv1)), 'Float value is out of bounds: ' + rv1);
            break;
        case 'D':
            assert_1['default'](rv2 === null, 'Second return value must be NULL for Double type.');
            assert_1['default'](typeof rv1 === 'number', 'Invalid double value: ' + rv1);
            break;
        case 'V':
            assert_1['default'](rv1 === undefined && rv2 === undefined, 'Return values must be undefined for Void type');
            break;
        }
    } else if (util_1.is_array_type(returnType)) {
        assert_1['default'](rv2 === undefined, 'Second return value must be undefined for array type.');
        assert_1['default'](rv1 === null || typeof rv1 === 'object' && typeof rv1['getClass'] === 'function', 'Invalid array object: ' + rv1);
        if (rv1 != null) {
            cls = assertClassInitializedOrResolved(thread, cl, returnType, true);
            assert_1['default'](rv1.getClass().isCastable(cls), 'Return value of type ' + rv1.getClass().getInternalName() + ' unable to be cast to return type ' + returnType + '.');
        }
    } else {
        assert_1['default'](util_1.is_reference_type(returnType), 'Invalid reference type: ' + returnType);
        assert_1['default'](rv2 === undefined, 'Second return value must be undefined for reference type.');
        assert_1['default'](rv1 === null || rv1 instanceof bsCl.getInitializedClass(thread, 'Ljava/lang/Object;').getConstructor(thread), 'Reference return type must be an instance of Object; value: ' + rv1);
        if (rv1 != null) {
            cls = assertClassInitializedOrResolved(thread, cl, returnType, false);
            if (!cls.accessFlags.isInterface()) {
                assertClassInitializedOrResolved(thread, cl, returnType, true);
            }
            assert_1['default'](rv1.getClass().isCastable(cls), 'Unable to cast ' + rv1.getClass().getInternalName() + ' to ' + returnType + '.');
        }
    }
    return true;
}
function assertClassInitializedOrResolved(thread, cl, type, initialized) {
    var cls = null;
    while (cls === null) {
        cls = initialized ? cl.getInitializedClass(thread, type) : cl.getResolvedClass(type);
        if (cl.getLoaderObject() !== null) {
            if (cl.getLoaderObject()['java/lang/ClassLoader/parent'] === null) {
                cl = thread.getBsCl();
            } else {
                cl = cl.getLoaderObject()['java/lang/ClassLoader/parent'].$loader;
            }
        } else {
            assert_1['default'](cls !== null, 'Unable to get initialized class for type ' + type + '.');
        }
    }
    return cls;
}
function printConstantPoolItem(cpi) {
    switch (cpi.getType()) {
    case enums_1.ConstantPoolItemType.METHODREF:
        var cpiMR = cpi;
        return util_1.ext_classname(cpiMR.classInfo.name) + '.' + cpiMR.signature;
    case enums_1.ConstantPoolItemType.INTERFACE_METHODREF:
        var cpiIM = cpi;
        return util_1.ext_classname(cpiIM.classInfo.name) + '.' + cpiIM.signature;
    case enums_1.ConstantPoolItemType.FIELDREF:
        var cpiFR = cpi;
        return util_1.ext_classname(cpiFR.classInfo.name) + '.' + cpiFR.nameAndTypeInfo.name + ':' + util_1.ext_classname(cpiFR.nameAndTypeInfo.descriptor);
    case enums_1.ConstantPoolItemType.NAME_AND_TYPE:
        var cpiNAT = cpi;
        return cpiNAT.name + ':' + cpiNAT.descriptor;
    case enums_1.ConstantPoolItemType.CLASS:
        var cpiClass = cpi;
        return util_1.ext_classname(cpiClass.name);
    default:
        return logging_1.debug_var(cpi.value);
    }
}
exports.OpcodeLayoutPrinters = {};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.OPCODE_ONLY] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase();
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.CONSTANT_POOL] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + printConstantPoolItem(method.cls.constantPool.get(code.readUInt16BE(pc + 1)));
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.CONSTANT_POOL_UINT8] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + printConstantPoolItem(method.cls.constantPool.get(code[pc + 1]));
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.CONSTANT_POOL_AND_UINT8_VALUE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + printConstantPoolItem(method.cls.constantPool.get(code.readUInt16BE(pc + 1))) + ' ' + code[pc + 3];
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.UINT8_VALUE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + code[pc + 1];
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.UINT8_AND_INT8_VALUE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + code[pc + 1] + ' ' + code[pc + 2];
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.INT8_VALUE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + code.readInt8(pc + 1);
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.INT16_VALUE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + code.readInt16BE(pc + 1);
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.INT32_VALUE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + code.readInt32BE(pc + 1);
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.ARRAY_TYPE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase() + ' ' + opcodes_1.ArrayTypes[code[pc + 1]];
};
exports.OpcodeLayoutPrinters[enums_1.OpcodeLayoutType.WIDE] = function (method, code, pc) {
    return enums_1.OpCode[code[pc]].toLowerCase();
};
function annotateOpcode(op, method, code, pc) {
    return exports.OpcodeLayoutPrinters[enums_1.OpcodeLayouts[op]](method, code, pc);
}
exports.annotateOpcode = annotateOpcode;
//# sourceMappingURL=threading.js.map