'use strict';
var enums_1 = require('./enums');
var assert_1 = require('./assert');
function isRunnable(status) {
    return status === enums_1.ThreadStatus.RUNNABLE;
}
var WeightedRoundRobinScheduler = function () {
    function WeightedRoundRobinScheduler() {
        this._count = 0;
        this._queue = [];
        this._threadScheduled = false;
    }
    WeightedRoundRobinScheduler.prototype.scheduleThread = function (thread) {
        this._queue.push(thread);
        if (this._queue.length === 1) {
            this.runThread();
        }
    };
    WeightedRoundRobinScheduler.prototype.runThread = function () {
        var _this = this;
        if (this._threadScheduled) {
            return;
        }
        this._threadScheduled = true;
        setImmediate(function () {
            var queue = _this._queue;
            _this._threadScheduled = false;
            if (queue.length > 0) {
                var thread = _this._queue[0];
                assert_1['default'](thread.getStatus() === enums_1.ThreadStatus.RUNNABLE, 'Attempted to run non-runnable thread.');
                thread.run();
            }
        });
    };
    WeightedRoundRobinScheduler.prototype.unscheduleThread = function (thread) {
        var queue = this._queue;
        var isRunningThread = queue[0] === thread;
        assert_1['default'](queue.indexOf(thread) > -1, 'Tried to unschedule thread that was not scheduled.');
        if (isRunningThread) {
            queue.shift();
            this._count = 0;
            this.runThread();
        } else {
            queue.splice(queue.indexOf(thread), 1);
        }
    };
    WeightedRoundRobinScheduler.prototype.getRunningThread = function () {
        var queue = this._queue;
        if (queue.length > 0) {
            return queue[0];
        } else {
            return null;
        }
    };
    WeightedRoundRobinScheduler.prototype.priorityChange = function (thread) {
    };
    WeightedRoundRobinScheduler.prototype.quantumOver = function (thread) {
        assert_1['default'](this._queue[0] === thread, 'A non-running thread has an expired quantum?');
        this._count++;
        if (this._count >= thread.getPriority() || thread.getStatus() !== enums_1.ThreadStatus.RUNNABLE) {
            this._count = 0;
            this._queue.push(this._queue.shift());
        }
        this.runThread();
    };
    return WeightedRoundRobinScheduler;
}();
var ThreadPool = function () {
    function ThreadPool(emptyCallback) {
        this.threads = [];
        this.scheduler = new WeightedRoundRobinScheduler();
        this.emptyCallback = emptyCallback;
    }
    ThreadPool.prototype.getThreads = function () {
        return this.threads.slice(0);
    };
    ThreadPool.prototype.anyNonDaemonicThreads = function () {
        for (var i = 0; i < this.threads.length; i++) {
            var t = this.threads[i];
            if (t.isDaemon()) {
                continue;
            }
            var status_1 = t.getStatus();
            if (status_1 !== enums_1.ThreadStatus.NEW && status_1 !== enums_1.ThreadStatus.TERMINATED) {
                return true;
            }
        }
        return false;
    };
    ThreadPool.prototype.threadTerminated = function (thread) {
        var idx = this.threads.indexOf(thread);
        assert_1['default'](idx >= 0);
        this.threads.splice(idx, 1);
        if (!this.anyNonDaemonicThreads()) {
            var close_1 = this.emptyCallback();
            if (close_1) {
                this.emptyCallback = null;
            }
        }
    };
    ThreadPool.prototype.statusChange = function (thread, oldStatus, newStatus) {
        var wasRunnable = isRunnable(oldStatus), nowRunnable = isRunnable(newStatus);
        if (oldStatus === enums_1.ThreadStatus.NEW || oldStatus === enums_1.ThreadStatus.TERMINATED) {
            if (this.threads.indexOf(thread) === -1) {
                this.threads.push(thread);
            }
        }
        if (wasRunnable !== nowRunnable) {
            if (wasRunnable) {
                this.scheduler.unscheduleThread(thread);
            } else {
                this.scheduler.scheduleThread(thread);
            }
        }
        if (newStatus === enums_1.ThreadStatus.TERMINATED) {
            this.threadTerminated(thread);
        }
    };
    ThreadPool.prototype.priorityChange = function (thread) {
        this.scheduler.priorityChange(thread);
    };
    ThreadPool.prototype.quantumOver = function (thread) {
        this.scheduler.quantumOver(thread);
    };
    return ThreadPool;
}();
exports.__esModule = true;
exports['default'] = ThreadPool;
//# sourceMappingURL=threadpool.js.map