import _ from 'lodash';
import moment from 'moment';
import help from './helpers';
import BBPromise from 'bluebird';

/**
 * Base class for other serializers
 *
 * @class Serializer
 */
class Serializer {

  /**
   * Serialize an item.
   *
   * @protected
   * @param  {Object} blueprint
   * @param  {Object} item
   * @return {Object} The serialized item
   */
  serializeItem(blueprint, item) {
    item = _.clone(item);
    var column;
    var columns = blueprint.columns;
    
    Object.keys(item).forEach((key) => {
      if (columns[key]) {
        column = columns[key];

        // if the given value is null, use default value when present
        if ((item[key] === null || item[key] === undefined) && column.hasOwnProperty('default')) {
          item[key] = column.default;
        }

        if (item[key] === null || item[key] === undefined) {
          return null;
        }

        var method = 'serialize' + help.capitalizeFirstLetter(column.type).replace('[]', 'Array');
        if (this[method]) {
          item[key] = this[method](item[key]);
        }
      }
    });

    return this.sortKeys(item);
  }

  /**
   * Serialize a string.
   *
   * @protected
   * @param  {string} value
   */
  serializeString(value) {
    if (value === '') {
      return null;
    }
    return value;
  }

  /**
   * Serialize a string array.
   *
   * @protected
   * @param  {string} value
   */
  serializeStringArray(value) {
    return _.map(value, this.serializeString);
  }

  /**
   * Serialize a text.
   *
   * @protected
   * @param  {string} value
   */
  serializeText(value) {
    if (value === '') {
      return null;
    }
    return value;
  }

  /**
   * Serialize a text array.
   *
   * @protected
   * @param  {string} value
   */
  serializeTextArray(value) {
    return _.map(value, this.serializeText);
  }

  /**
   * Serialize a string.
   *
   * @protected
   * @param  {string} value
   */
  serializeUuid(value) {
    if (value === '') {
      return null;
    }
    return value;
  }

  /**
   * Serialize a string.
   *
   * @protected
   * @param  {string} value
   */
  serializeUuidArray(value) {
    return _.map(value, this.serializeUuid);
  }

  /**
   * Serialize an integer.
   *
   * @protected
   * @param  {string} value
   */
  serializeInteger(value) {
    return value;
  }

  /**
   * Serialize an integer array.
   *
   * @protected
   * @param  {string} value
   */
  serializeIntegerArray(value) {
    return _.map(value, this.serializeInteger, this);
  }

  /**
   * Serialize a float.
   *
   * @protected
   * @param  {string} value
   */
  serializeFloat(value) {
    return value;
  }

  /**
   * Serialize an float array.
   *
   * @protected
   * @param  {string} value
   */
  serializeFloatArray(value) {
    return _.map(value, this.serializeFloat, this);
  }

  /**
   * Convert a date string to a date object.
   *
   * @protected
   * @param  {string} value
   */
  dateStringToMoment(value) {
    return moment(new Date(value));
  }

  /**
   * Serialize a datetime string to a datetime.
   *
   * @protected
   * @param  {string} value
   */
  serializeDatetime(value) {
    return this.dateStringToMoment(value)
      .format('YYYY-MM-DD HH:mm:ss');
  }

  /**
   * Serialize a datetime array string to a datetime.
   *
   * @protected
   * @param  {string} value
   */
  serializeDatetimeArray(value) {
    return _.map(value, this.serializeDatetime, this);
  }

  /**
   * Serialize a date string to a date.
   *
   * @protected
   * @param  {string} value
   */
  serializeDate(value) {
    return this.dateStringToMoment(value)
      .format('YYYY-MM-DD');
  }

  /**
   * Serialize a date array string to a array of dates.
   *
   * @protected
   * @param  {string} value
   */
  serializeDateArray(value) {
    return _.map(value, this.serializeDate, this);
  }

  /**
   * Serialize to boolean.
   *
   * @protected
   * @param  {string} value
   */
  serializeBoolean(value) {
    return !!value;
  }

  /**
   * Serialize to boolean array.
   *
   * @protected
   * @param  {string} value
   */
  serializeBooleanArray(value) {
    return _.map(value, this.serializeBoolean, this);
  }

  /**
   * Serialize to json.
   *
   * @protected
   * @param  {string} value
   */
  serializeJson(value) {
    return value;
  }

  /**
   * Serialize to array of json.
   *
   * @protected
   * @param  {string} value
   */
  serializeJsonArray(value) {
    return _.map(value, this.serializeJson, this);
  }

  /**
   * Serialize a point.
   *
   * @protected
   * @param  {string} value
   */
  serializePoint(value) {
    return this.serializeJson(value);
  }

  /**
   * Serialize a point array.
   *
   * @protected
   * @param  {string} value
   */
  serializePointArray(value) {
    return this.serializeJson(value);
  }

  /**
   * Serialize a linestring.
   *
   * @protected
   * @param  {string} value
   */
  serializeLinestring(value) {
    return this.serializeJson(value);
  }

  /**
   * Serialize a linestring array.
   *
   * @protected
   * @param  {string} value
   */
  serializeLinestringArray(value) {
    return this.serializeJson(value);
  }

  /**
   * Serialize a polygon.
   *
   * @protected
   * @param  {string} value
   */
  serializePolygon(value) {
    return this.serializeJson(value);
  }

  /**
   * Serialize a polygon array.
   *
   * @protected
   * @param  {string} value
   */
  serializePolygonArray(value) {
    return this.serializeJson(value);
  }

  /**
   * Sort keys of an object.
   *
   * @protected
   * @param  {Object} item
   */
  sortKeys(item) {
    var value
      , sortedItem = {}
      , keys = _.keys(item)
      , sortedKeys = _.sortBy(keys, (key) => {
        return key;
      });

    _.each(sortedKeys, (key) => {
      value = item[key];
      sortedItem[key] = value;
    });

    return sortedItem;
  }

}

export default Serializer;
