import _ from 'lodash';

/**
 * Transformer
 * 
 * @class Transformer
 */
class Transformer {

  transform(changes, blueprintId, item, fromProjectVersion, toProjectVersion) {
    var clonedItem = _.clone(item);
    var filteredChanges = this.filterChangesBetween(changes, fromProjectVersion, toProjectVersion);
    return this.applyChanges(filteredChanges, blueprintId, clonedItem);
  }

  applyChanges(changes, blueprintId, item) {
    _.each(changes, (change) => {
      if (change.blueprint !== blueprintId) {
        return;
      }

      switch (change.type) {
        case 'blueprint.remove':
          item = null;
          break;
        case 'column.create':
          item[change.column] = null;
          break;
        case 'column.rename':
          item[change.value] = item[change.column] === undefined ? null : item[change.column];
          delete item[change.column];
          break;
        case 'column.update':
          if (change.key === 'type') {
            item[change.column] = null;
          }
          break;
        case 'column.remove':
          delete item[change.column];
          break;
      }
    });
    
    return item;
  }

  filterChangesBetween(changes, fromVersion, toVersion) {
    if (toVersion === undefined) {
      toVersion = Number.POSITIVE_INFINITY;
    }

    if (fromVersion === toVersion) {
      return [];
    }

    var lastVersion = 0;
    var filteredChanges = _.filter(changes, (change) => {
      if (change.type === 'project.tag') {
        lastVersion = change.value;
      }
      return lastVersion >= fromVersion && lastVersion <= toVersion || lastVersion >= toVersion && lastVersion <= fromVersion;
    });

    if (fromVersion > toVersion) {
      return this.invertChanges(filteredChanges);
    }
    return filteredChanges;
  }

  invertChanges(changes) {
    changes = changes.reverse();
    return _.filter(_.map(changes, this.invertChange), (change) => {
      return change !== undefined;
    });
  }

  invertChange (change) {
    switch (change.type) {
      case 'project.tag':
        return change;
      case 'blueprint.create':
        return {
          type: 'blueprint.remove',
          blueprint: change.blueprint
        };
      case 'blueprint.update':
        return {
          type: 'blueprint.update',
          blueprint: change.blueprint,
          key: change.key,
          value: change.oldValue,
          oldValue: change.value
        };
      case 'blueprint.remove':
        return {
          type: 'blueprint.create',
          blueprint: change.blueprint,
          value: change.oldValue
        };
      case 'column.create':
        return {
          type: 'column.remove',
          blueprint: change.blueprint,
          column: change.column
        };
      case 'column.rename':
        return {
          type: 'column.rename',
          blueprint: change.blueprint,
          column: change.value,
          value: change.column
        };
      case 'column.update':
        return {
          type: 'column.update',
          blueprint: change.blueprint,
          column: change.column,
          key: change.key,
          oldValue: change.value,
          value: change.oldValue
        };
      case 'column.remove':
        return {
          type: 'column.create',
          blueprint: change.blueprint,
          column: change.column,
          value: change.oldValue
        };
    }
  }

}

export default Transformer;
