import _ from 'lodash';
import BBPromise from 'bluebird';
import { helpers } from 'dstore-helpers';

/**
 * LeveldbMigrator
 * 
 * @class LeveldbMigrator
 */
class LeveldbMigrator {
  /**
   * @param {Client} client
   */
  constructor(client) {
    this.client = client;
  }

  /**
   * Create a new elasticsearch index and put the mapping.
   *
   * @param  {String} projectId
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  migrate(projectId, projectVersion, blueprints) {
    var opts = {
      index: projectId + 'v' + projectVersion,
      projectVersion: projectVersion,
      projectId: projectId,
      blueprints: blueprints
    };

    return this.createIndex(opts)
      .then(() => {
        return this.putMapping(opts);
      })
      .then(() => {
        return this.updateAlias(opts);
      });
  }

  /**
   * Create a new elasticsearch index.
   *
   * @protected
   * @param opts
   * @param opts.index The name of the index to create
   *
   * @return {BBPromise}
   */
  createIndex(opts) {
    return this.client.indices.exists({
      index: opts.index
    })
    .then((exists) => {
      if (!exists) {
        return this.client.indices.create({
          index: opts.index
        });
      }
    });
  }

  /**
   * Create and put a mapping.
   *
   * @protected
   * @param opts
   * @param opts.blueprints  The blueprints to turn into a mapping
   * @param opts.index       The index to put the mapping to
   *
   * @return {BBPromise.<undefined>}
   */
  putMapping(opts) {
    return BBPromise.map(_.values(opts.blueprints), (blueprint) => {
      var mapping = {};

      var properties = _.object(_.map(blueprint.columns, (column, columnKey) => {
        var property = this['define' + helpers.capitalizeFirstLetter(column.type).replace('[]', '')](column);

        // if extra options are given for elasticsearch, use those instead
        if (column.elasticsearch) {
          property = column.elasticsearch;
        }

        return [columnKey, property];
      }));

      mapping[blueprint.elasticsearch.type] = {
        _id: {path: 'id'},
        properties: properties
      };

      return this.client.indices.putMapping({
        index: opts.index,
        type: blueprint.elasticsearch.type,
        body: mapping
      });
    });
  };

  updateAlias(opts) {
    return BBPromise.resolve()
      .then(() => {
        if (opts.projectVersion > 1) {
          return this.client.indices.deleteAlias({
            index: opts.projectId + 'v' + (opts.projectVersion - 1),
            name: opts.projectId
          });
        }
      })
      .then(() => {
        return this.client.indices.putAlias({
          index: opts.index,
          name: opts.projectId
        });
      });
  }

  /**
   * Define a uuid mapping.
   *
   * @protected
   */
  defineUuid() {
    return this.defineString();
  }

  /**
   * Define a text mapping.
   *
   * @protected
   */
  defineText() {
    return this.defineString();
  }

  /**
   * Define a string mapping.
   *
   * @protected
   */
  defineString() {
    return {
      type: 'string'
    };
  }

  /**
   * Define a point mapping.
   *
   * @protected
   */
  definePoint() {
    return {
      type: 'geo_point'
    };
  }

  /**
   * Define a linestring mapping.
   *
   * @protected
   */
  defineLinestring() {
    return this.defineShape();
  }

  /**
   * Define a polygon mapping.
   *
   * @protected
   */
  definePolygon() {
    return this.defineShape();
  }

  /**
   * Define a shape mapping.
   *
   * @protected
   */
  defineShape() {
    return {
      type: 'geo_shape',
      tree: 'quadtree',
      precision: '10m'
    };
  }

  /**
   * Define a date mapping.
   *
   * @protected
   */
  defineDate() {
    return {
      type: 'date',
      format: 'yyyy-MM-dd'
    };
  }

  /**
   * Define a datetime mapping.
   *
   * @protected
   */
  defineDatetime() {
    return {
      type: 'date',
      format: 'yyyy-MM-dd HH:mm:ss'
    };
  }

  /**
   * Define a float mapping.
   *
   * @protected
   */
  defineFloat() {
    return {
      type: 'float'
    };
  }

  /**
   * Define a integer mapping.
   *
   * @protected
   */
  defineInteger() {
    return {
      type: 'integer'
    };
  }

  /**
   * Define a boolean mapping.
   *
   * @protected
   */
  defineBoolean() {
    return {
      type: 'boolean'
    };
  }

  /**
   * Define a json mapping.
   *
   * @protected
   */
  defineJson() {
    return {
      type: 'object',
      enabled: false
    };
  };

}

export default LeveldbMigrator;
