import _  from 'lodash';
import BBPromise from 'bluebird';

/**
 * LeveldbRepository
 * 
 * @class LeveldbReposirory
 */
class LeveldbRepository {
  /**
   * @param {LeveldbSerializer} serializer
   * @param {LeveldbAdapter}     adapter
   */
  constructor(serializer, adapter) {
    this.serializer = serializer;
    this.adapter = adapter;
  }

  /**
   * Retrieve item
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {String} id
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  get(projectId, blueprint, id, projectVersion) {
    var client = this.adapter.getClient(projectId, projectVersion);
    var itemByIdDb = client.sublevel('item-by-id');
    return BBPromise.promisify(itemByIdDb.get)(id)
      .then((item) => {
        if ( ! item) {
          return null;
        }

        item = JSON.parse(item);
        _.each(blueprint.columns, (column, key) => {
          if (_.contains(['linestring', 'linestring[]', 'point', 'point[]', 'polygon', 'polygon[]'], column.type)) {
            if (item[key] && item[key].crs) {
              delete item[key].crs;
            }
          }
        });
        return item;
      });
  }

  /**
   * Retrieve all items
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  getAll(projectId, blueprint, projectVersion) {
    return BBPromise.reject('not implemented');
  }

  /**
   * Insert / update item.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {String} id
   * @param  {Object} item
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  put(projectId, blueprint, id, item, projectVersion) {
    // serialize the item
    var serializedItem = this.serializer.serialize(blueprint, item);
      
    // index the item
    var value = JSON.stringify(serializedItem);
    var client = this.adapter.getClient(projectId, projectVersion);
    var itemByIdDb = client.sublevel('item-by-id');
    var itemByTypeDb = client.sublevel('item-by-type');
    var table = itemByTypeDb.sublevel(blueprint.leveldb.type);

    return BBPromise.join(
      new BBPromise(function (resolve, reject) {
        itemByIdDb.put(id, value, function (err) {
          if (err) {
            return reject(err);
          }
          resolve();
        });
      }),
      new BBPromise(function (resolve, reject) {
        table.put(id, value, function (err) {
          if (err) {
            return reject(err);
          }
          resolve();
        });
      })
    );
  }

  /**
   * Batch insert / update items.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Object} items
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  putMany(projectId, blueprint, items, projectVersion) {
    return BBPromise.reduce(items, (memo, item) => {
      return BBPromise.join(
        item.id,  
        this.put(projectId, blueprint, item.id, item, projectVersion)
      );
    }, [])
    .then((results) => {
      return _.object(results);
    });
  }

  /**
   * Delete item.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Object} id
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  del(projectId, blueprint, id, projectVersion) {
    var client = this.adapter.getClient(projectId, projectVersion);

    var itemByIdDb = client.sublevel('item-by-id');
    var itemByTypeDb = client.sublevel('item-by-type');
    var table = itemByTypeDb.sublevel(blueprint.leveldb.type);

    return BBPromise.promisify(itemByIdDb.del)(id)
      .then(function () {
        return BBPromise.promisify(table.del)(id);
      });
  }

}

export default LeveldbRepository;
