import _ from 'lodash';
import es from 'event-stream';
import BBPromise from 'bluebird';
import { Transformer } from 'dstore-helpers';
import LevelWriteStream from 'level-writestream';
import highland from 'highland';
import { PassThrough } from 'stream';

class LeveldbSyncer {

  constructor(adapter) {
    this.adapter = adapter;
  }

  /**
   * Sync data from the old index to the new one,
   * transforming items to the new version along the way when needed.
   *
   * @param  {String} projectId
   * @param  {Number} toProjectVersion
   * @param  {Object} fromBlueprints
   * @param  {Object} toBlueprints
   * @param  {Array} log
   *
   * @return {BBPromise}
   */
  async sync(projectId, toProjectVersion, fromBlueprints, toBlueprints, log) {
    if (toProjectVersion === 1) {
      return BBPromise.resolve();
    }

    var transformer = new Transformer();

    var fromClient = this.adapter.getClient(projectId, toProjectVersion - 1);
    var toClient = this.adapter.getClient(projectId, toProjectVersion);

    var keyedBlueprints = _.map(toBlueprints, (toBlueprint, blueprintId) => {
      toBlueprint.key = blueprintId;
      return toBlueprint;
    });

    return BBPromise.reduce(keyedBlueprints, (memo, toBlueprint) => {
      var fromBlueprint = fromBlueprints[toBlueprint.key];
      if ( ! fromBlueprint) {
        return BBPromise.resolve();
      }

      var fromItemByTypeDb = fromClient.sublevel('item-by-type');
      var fromTable = fromItemByTypeDb.sublevel(fromBlueprint.leveldb.type);
      var tableStream = fromTable.createReadStream();

      var toItemByTypeDb = toClient.sublevel('item-by-type');
      var toTable = toItemByTypeDb.sublevel(toBlueprint.leveldb.type);
      var toItemByIdDb = toClient.sublevel('item-by-id');
      LevelWriteStream(toTable);
      LevelWriteStream(toItemByIdDb);
      var toTableWriteStream = toTable.createWriteStream();
      var toItemByIdWriteStream = toItemByIdDb.createWriteStream();

      var transform = es.through(function (data) {
        var item = JSON.parse(data.value);
        var transformedItem = transformer.transform(log, fromBlueprint.leveldb.type, item, toProjectVersion - 1, toProjectVersion);
        if (transformedItem !== null) {
          this.emit('data', {
            key: data.key,
            value: JSON.stringify(transformedItem)
          });
        }
      });

      return new BBPromise((resolve) => {
        var stream = tableStream
          .pipe(transform);

        var second = highland(tableStream).fork();
        second
          .pipe(toTableWriteStream)
          .on('finish', resolve);
        
        stream.pipe(toItemByIdWriteStream);
      });
    }, []);
  }

}

export default LeveldbSyncer;
