import BBPromise from 'bluebird';

/**
 * PostgresqlFacade
 * 
 * @class PostgresqlFacade
 *
 */
class PostgresqlFacade {

  /**
   * @param {Repository} repository
   * @param {Migrator}   migrator
   * @param {Dropper}    dropper
   * @param {Syncer}     syncer
   */
  constructor(repository, migrator, dropper, syncer) {
    this.repository = repository;
    this.migrator = migrator;
    this.dropper = dropper;
    this.syncer = syncer;
  }

  /**
   * Retrieve item
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {String} id
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  get(projectId, blueprint, id, projectVersion) {
    return this.repository.get(projectId, blueprint, id, projectVersion);
  }

  /**
   * Retrieve all items
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  getAll(projectId, blueprint, projectVersion) {
    return this.repository.getAll(projectId, blueprint, projectVersion);
  }

  /**
   * Insert / update item.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {String} id
   * @param  {Object} item
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  put(projectId, blueprint, id, item, projectVersion) {
    return this.repository.put(projectId, blueprint, id, item, projectVersion);
  }

  /**
   * Batch insert / update items.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Object} items
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  putMany(projectId, blueprint, items, projectVersion) {
    return this.repository.putMany(projectId, blueprint, items, projectVersion);
  }

  /**
   * Delete item.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Object} id
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  del(projectId, blueprint, id, projectVersion) {
    return this.repository.del(projectId, blueprint, id, projectVersion);
  }

  /**
   * Sync data from the old index to the new one,
   * transforming items to the new version along the way when needed.
   *
   * @param  {String} projectId
   * @param  {Number} toProjectVersion
   * @param  {Object} fromBlueprints
   * @param  {Object} toBlueprints
   * @param  {Object} log
   *
   * @return {BBPromise}
   */
  sync(projectId, toProjectVersion, fromBlueprints, toBlueprints, log) {
    return this.syncer.sync(projectId, toProjectVersion, fromBlueprints, toBlueprints, log);
  }

  /**
   * Create a new postgresql index and put the mapping.
   *
   * @param  {String} projectId
   * @param  {Number} projectVersion
   * @param  {Object} blueprints
   *
   * @return {BBPromise}
   */
  migrate(projectId, projectVersion, blueprints) {
    return this.migrator.migrate(projectId, projectVersion, blueprints);
  }

  /**
   * Drop postgresql index.
   *
   * @param  {String} projectId
   * @param  {Number} projectVersion
   * 
   * @return {BBPromise}
   */
  drop(projectId, projectVersion) {
    return this.dropper.drop(projectId, projectVersion);
  }

  /**
   * Serialize an item to make it compatible with postgresql.
   *
   * @param  {Object} blueprint
   * @param  {Object} item
   * 
   * @return {Object} The serialized item
   */
  serialize(blueprint, item) {
    return this.serializer.serializer(blueprint, item);
  }

}

export default PostgresqlFacade;
