import _ from 'lodash';
import BBPromise from 'bluebird';
import { helpers } from 'dstore-helpers';
import pg from 'pg';

BBPromise.promisifyAll(pg);

/**
 * PostgresqlMigrator
 *
 * @class PostgresqlMigrator
 */
class PostgresqlMigrator {
  /**
   * @param {PostgresqlAdapter} adapter
   */
  constructor(adapter, config) {
    this.adapter = adapter;
    this.config = config;
  }

  /**
   * Create / migrate database.
   *
   * @param  {String} projectId
   * @param  {Number} projectVersion
   *
   * @return {BBPromise.<undefined>}
   */
  async migrate(projectId, projectVersion, blueprints) {
    var connectionString = 'postgresql://' + this.config.username + ':' + this.config.password + '@' + this.config.host + ':' + this.config.port + '/';

    await this.checkExtensions(connectionString);
    await this.createDatabase(connectionString, projectId, projectVersion);
    let client = await this.getDatabaseClient(projectId, projectVersion);
    await this.createTables(client, blueprints);

    return {
      success: true
    };
  }

  checkExtensions(connectionString) {
    return pg.connectAsync(connectionString + 'template_postgis')
      .spread((postgisTemplateConnection, done) => {
        return postgisTemplateConnection.queryAsync('SELECT extname, extversion from pg_extension')
          .then((result) => {
            var promise = BBPromise.resolve();

            var extensions = _.pluck(result.rows, 'extname');
            if ( ! _.contains(extensions, 'postgis')) {
              console.log('"postgis" extension missing, installing it.');
              promise = promise.then(() => {
                return postgisTemplateConnection.queryAsync('CREATE EXTENSION postgis');
              });
            }

            if ( ! _.contains(extensions, 'uuid-ossp')) {
              console.log('"uuid-ossp" extension missing, installing it.');
              promise = promise.then(() => {
                return postgisTemplateConnection.queryAsync('CREATE EXTENSION "uuid-ossp"');
              });
            }

            return promise;
          })
          .finally(() => {
            return done();
          });
      });
  }

  /**
   * Creates a new database for given projectId and project version.
   *
   * @param  {String} connectionString The postgresql connectionString
   * @param  {String} projectId        The project's identifier
   * @param  {Number} projectVersion   The project version
   */
  createDatabase(connectionString, projectId, projectVersion) {
    return pg.connectAsync(connectionString + 'template_postgis')
      .spread((postgresConnection, done) => {
        return postgresConnection.queryAsync('CREATE DATABASE ' + projectId + 'v' + projectVersion + ' TEMPLATE=template_postgis')
          .catch((err) => {
            console.error('Error while creating database', err);
            throw err;
          })
          .finally(() => {
            return done();
          });
      });
  }

  /**
   * Retrieves the database client for given projectId and project version.
   *
   * @param  {String} projectId          The project's identifier
   * @param  {Number} projectVersion     The project version
   */
  getDatabaseClient(projectId, projectVersion) {
    return this.adapter.getClient(projectId, projectVersion);
  }

  /**
   * Creates tables in database based on the given blueprints.
   *
   * @param  {PostgresqlClient}  client
   * @param  {Object}            blueprints The blueprints
   */
  createTables(client, blueprints) {
    var promises = _.map(blueprints, (blueprint) => {
      return client.schema.createTable(blueprint.postgresql.table, (table) => {
        _.each(blueprint.columns, (column, columnKey) => {
          var chain = this['define' + helpers.capitalizeFirstLetter(column.type).replace('[]', 'Array')](table, columnKey);

          if (column.index) {
            if (column.index_type) {
              chain.index(null, column.index_type);
            } else {
              chain.index();
            }
          }

          if (column.primary) {
            chain.primary();
          }

          if (column.unique) {
            chain.unique();
          }

          if (column.references) {
            chain.inTable(column.references.table)
              .references(column.references.column);
          }

          if (column.unsigned) {
            chain.unsigned();
          }

          if (column.nullable === false) {
            chain.notNullable();
          }

          if (column.comment) {
            chain.comment(column.comment);
          }
        });
      })
      // triger knex.js's promise strategy
      .then((res) => {
        return res;
      }, (err) => {
        console.error('error while creating tables', err);
      });
    });

    return BBPromise.all(promises);
  }

  /**
   * Defines an INTEGER column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineInteger(table, columnKey) {
    return table.integer(columnKey);
  }

  /**
   * Defines an INTEGER[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineIntegerArray(table, columnKey) {
    return table.specificType(columnKey, 'INTEGER[]');
  }

  /**
   * Defines an UUID column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineUuid(table, columnKey) {
    return table.uuid(columnKey);
  }

  /**
   * Defines an UUID[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineUuidArray(table, columnKey) {
    return table.specificType(columnKey, 'UUID[]');
  }

  /**
   * Defines an STRING column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineString(table, columnKey) {
    return table.string(columnKey);
  }

  /**
   * Defines an STRING[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineStringArray(table, columnKey) {
    return table.specificType(columnKey, 'VARCHAR[]');
  }

  /**
   * Defines an TEXT column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineText(table, columnKey) {
    return table.text(columnKey);
  }

  /**
   * Defines an TEXT[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineTextArray(table, columnKey) {
    return table.specificType(columnKey, 'TEXT[]');
  }

  /**
   * Defines an DATETIME column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineDatetime(table, columnKey) {
    return table.dateTime(columnKey);
  }

  /**
   * Defines an DATETIME[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineDatetimeArray(table, columnKey) {
    return table.specificType(columnKey, 'TIMESTAMP WITH TIME ZONE[]');
  }

  /**
   * Defines an DATE column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineDate(table, columnKey) {
    return table.date(columnKey);
  }

  /**
   * Defines an DATE[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineDateArray(table, columnKey) {
    return table.specificType(columnKey, 'DATE[]');
  }

  /**
   * Defines an REAL column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineFloat(table, columnKey) {
    return table.float(columnKey);
  }

  /**
   * Defines an REAL[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineFloatArray(table, columnKey) {
    return table.specificType(columnKey, 'REAL[]');
  }

  /**
   * Defines an GEOMETRY column of type Point on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  definePoint(table, columnKey) {
    return table.specificType(columnKey, 'geometry(Point,4326)');
  }

  /**
   * Defines an GEOMETRY column of type MultiPoint on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  definePointArray(table, columnKey) {
    return table.specificType(columnKey, 'geometry(MultiPoint,4326)');
  }

  /**
   * Defines an GEOMETRY column of type LineString on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineLinestring(table, columnKey) {
    return table.specificType(columnKey, 'geometry(LineString,4326)');
  }

  /**
   * Defines an GEOMETRY column of type MultiLineString on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineLinestringArray(table, columnKey) {
    return table.specificType(columnKey, 'geometry(MultiLineString,4326)');
  }

  /**
   * Defines an GEOMETRY column of type Polygon on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  definePolygon(table, columnKey) {
    return table.specificType(columnKey, 'geometry(Polygon,4326)');
  }

  /**
   * Defines an GEOMETRY column of type MultiPolygon on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  definePolygonArray(table, columnKey) {
    return table.specificType(columnKey, 'geometry(MultiPolygon,4326)');
  }

  /**
   * Defines an BOOLEAN column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineBoolean(table, columnKey) {
    return table.boolean(columnKey);
  }

  /**
   * Defines an BOOLEAN[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineBooleanArray(table, columnKey) {
    return table.specificType(columnKey, 'BOOLEAN[]');
  }

  /**
   * Defines a JSON column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineJson(table, columnKey) {
    return table.json(columnKey);
  }

  /**
   * Defines a JSON[] column on given table.
   *
   * @protected
   * @param  {Object} table     Knex.js table object (given through createTable's callback)
   * @param  {String} columnKey Name of the column
   */
  defineJsonArray(table, columnKey) {
    return table.specificType(columnKey, 'JSON[]');
  }

}

export default PostgresqlMigrator;
