import _  from 'lodash';
import BBPromise from 'bluebird';
var knex = require('knex')({dialect: 'postgres'});

/**
 * PostgresqlRepository
 * 
 * @class PostgresqlReposirory
 */
class PostgresqlRepository {
  /**
   * @param {PostgresqlSerializer} serializer
   * @param {PostgresqlAdapter}     adapter
   */
  constructor(serializer, adapter) {
    this.serializer = serializer;
    this.adapter = adapter;
  }

  /**
   * Retrieve item
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {String} id
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  get(projectId, blueprint, id, projectVersion) {
    var client = this.adapter.getClient(projectId, projectVersion);
    var selects = this.blueprintToSelects(blueprint);
    return client.table(blueprint.postgresql.table)
      .select(selects)
      .where({
        id: id
      })
      .first(selects)
      // triggers knex.js's promise strategy
      .then((res) => {
        _.each(blueprint.columns, (column, key) => {
          if (_.contains(['linestring', 'linestring[]', 'point', 'point[]', 'polygon', 'polygon[]'], column.type)) {
            if (res[key] !== null) {
              res[key] = JSON.parse(res[key]);
            }
          }
        });
        
        return res;
      });
  }

  /**
   * Retrieve all items
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  getAll(projectId, blueprint, projectVersion) {
    return BBPromise.reject(new Error('not implemented'));
  }

  /**
   * Insert / update item.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {String} id
   * @param  {Object} item
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  async put(projectId, blueprint, id, item, projectVersion) {
    // serialize the item
    let serializedItem = this.serializer.serialize(blueprint, item);

    // retrieve the knex.js client for the given version
    let client = this.adapter.getClient(projectId, projectVersion);

    // check if the item exists
    let exists = await this.itemExists(client, blueprint, id);
    if (exists) {
      // update the item
      return await this.updateItem(client, blueprint, id, serializedItem);
    } else {
      // insert the item
      return await this.insertItem(client, blueprint, serializedItem);
    }
  }

  /**
   * Batch insert / update items.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Object} items
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  putMany(projectId, blueprint, items, projectVersion) {
    // serialize the item
    return BBPromise.map(items, (item) => {
      return this.put(projectId, blueprint, item.id, item, projectVersion);
    });
  }

  /**
   * Delete item.
   *
   * @param  {String} projectId
   * @param  {Object} blueprint
   * @param  {Object} id
   * @param  {Number} projectVersion
   *
   * @return {BBPromise}
   */
  del(projectId, blueprint, id, projectVersion) {
    // retrieve the knex.js client for the given version
    let client = this.adapter.getClient(projectId, projectVersion);
    return client.table(blueprint.postgresql.table)
      .where({
        id: id
      })
      .del()
      // triggers knex.js's promise strategy
      .then((res) => res);
  }

  /**
   * Checks if an item already exists in the database.
   * 
   * @param  {PostgresqlClient} client      The knex client
   * @param  {Object}           blueprint   Blueprint options
   * @param  {Object}           id          The id of the item
   * 
   * @return {Promise} Promise resolving true or false (exists or doesn't exist).
   */
  itemExists(client, blueprint, id) {
    return client.table(blueprint.postgresql.table)
      .first()
      .where({
        id: id
      })
      .then((row) => !!row);
  }

  /**
   * Updates an item in the database.
   *
   * @param  {PostgresqlClient} client            The knex client
   * @param  {Object}           blueprint         Blueprint options
   * @param  {Object}           id                The id of the item
   * @param  {Object}           serializedItem    The serialized item
   * 
   * @return {Promise}
   */
  updateItem(client, blueprint, id, serializedItem) {
    return client.table(blueprint.postgresql.table)
      .where({
        id: id
      })
      .update(serializedItem)
      // triggers knex.js's promise strategy
      .then((res) => res);
  }

  /**
   * Inserts a new item into the database.
   * 
   * @param  {PostgresqlClient} client          The knex client
   * @param  {Object}           blueprint       Blueprint options
   * @param  {Object}           compatibleItem  The serialized item
   * 
   * @return {Promise}
   */
  insertItem(client, blueprint, serializedItem) {
    return client.table(blueprint.postgresql.table)
      .insert(serializedItem)
      // triggers knex.js's promise strategy
      .then((res) => res);
  }

  blueprintToSelects(blueprint) {
    var selects = ['*'];
    
    _.each(blueprint.columns, (column, key) => {
      if (_.contains(['linestring', 'linestring[]', 'point', 'point[]', 'polygon', 'polygon[]'], column.type)) {
        selects.push(knex.raw('ST_AsGeoJson(' + key + ') as ' + key));
      }

      if (column.type === 'datetime') {
        selects.push(knex.raw('to_char(' + key + ', \'yyyy-MM-dd hh:mi:ss\') as ' + key));
      }

      if (column.type === 'date') {
        selects.push(knex.raw('to_char(' + key + ', \'yyyy-MM-dd\') as ' + key));
      }
    });

    return selects;
  }

}

export default PostgresqlRepository;
