import pg from 'pg';
import _ from 'lodash';
import es from 'event-stream';
import BBPromise from 'bluebird';
import { Transformer } from 'dstore-helpers';
import QueryStream from 'pg-query-stream';

BBPromise.promisifyAll(pg);

class PostgresqlSyncer {

  constructor(config, adapter, serializer) {
    this.config = config;
    this.adapter = adapter;
    this.serializer = serializer;
  }

  /**
   * Sync data from the old index to the new one,
   * transforming items to the new version along the way when needed.
   *
   * @param  {String} projectId
   * @param  {Number} toProjectVersion
   * @param  {Object} fromBlueprints
   * @param  {Object} toBlueprints
   * @param  {Array} log
   *
   * @return {BBPromise}
   */
  async sync(projectId, toProjectVersion, fromBlueprints, toBlueprints, log) {
    var self = this;

    var transformer = new Transformer();
    let fromConnectionString = 'postgresql://' + this.config.username + ':' + this.config.password + '@' + this.config.host + ':' + this.config.port + '/' + projectId + 'v' + (toProjectVersion - 1);
    var fromClient = await this.getConnection(fromConnectionString);
    fromClient.connection.stream.setMaxListeners(100);
    // retrieve the knex.js client for the new project version
    var toClient = this.adapter.getClient(projectId, toProjectVersion);
    var keyedBlueprints = _.map(toBlueprints, (blueprint, blueprintId) => {
      blueprint.id = blueprintId;
      return blueprint;
    });
    return BBPromise.reduce(keyedBlueprints, (memo, blueprint) => {
      let fromBlueprint = fromBlueprints[blueprint.id];
      if ( ! fromBlueprint) {
        return BBPromise.resolve();
      }
      let transformItem = es.through(function (item) {
        let transformedItem = transformer.transform(log, blueprint.id, item, toProjectVersion - 1, toProjectVersion);
        _.each(blueprint.columns, (column, columnKey) => {
          if (self.isGeoField(column.type)) {
            if (transformedItem[columnKey]) {
              transformedItem[columnKey] = JSON.parse(transformedItem[columnKey]);
            }
          }
        });
        if (transformedItem !== null) {
          this.emit('data', transformedItem);
        }
      });

      var toStream = es.through(function (transformedItem) {
        var streamSelf = this;
        let serializedItem = self.serializer.serialize(blueprint, transformedItem);
        streamSelf.pause();
        toClient.table(blueprint.postgresql.table)
          .insert(serializedItem)
          // triggers knex.js's promise strategy
          .then((res) => {
            streamSelf.resume();
            return res;
          })
          .catch((err) => {
            throw err;
          });
      });

      return new BBPromise((resolve, reject) => {
        var query = new QueryStream('SELECT ' + this.blueprintToSelects(fromBlueprint) + ' FROM ' + fromBlueprint.postgresql.table);
        var fromStream = fromClient.query(query)
        fromStream
          .pipe(transformItem)
          .pipe(toStream)
          .on('end', resolve)
          .on('error', (err) => {
            throw err;
          });
      });
    }, []);
  }

  getConnection(connectionString) {
    return pg.connectAsync(connectionString)
      .spread((client, done) => {
        this.connectionCloser = done;
        return client;
      });
  }


  blueprintToSelects(blueprint) {
    var selects = ['*'];

    _.each(blueprint.columns, (column, key) => {
      if (this.isGeoField(column.type)) {
        selects.push('ST_AsGeoJson(' + key + ') as ' + key);
      }

      if (column.type === 'datetime') {
        selects.push('to_char(' + key + ', \'yyyy-MM-dd hh:mi:ss\') as ' + key);
      }

      if (column.type === 'date') {
        selects.push('to_char(' + key + ', \'yyyy-MM-dd\') as ' + key);
      }
    });

    return selects.join(',');
  }

  isGeoField(type) {
    return _.contains(['linestring', 'linestring[]', 'point', 'point[]', 'polygon', 'polygon[]'], type);
  }

}

export default PostgresqlSyncer;
